#!perl
#
# Documentation, copyright and license is at the end of this file.
#
package  STD::Tester;

use 5.001;
use strict;
use warnings;
use warnings::register;

use Test;
use Data::Dumper;
use STD::TestUtil;

use vars qw($VERSION $DATE);
$VERSION = '1.02';
$DATE = '2003/06/03';

use vars qw(@ISA);
@ISA = qw(STD::TestUtil);

$Test::TestLevel = 1;

####
# Using an object to pass localized object data
# between functions. Makes the functions reentrant
# where out right globals can be clobbered when
# used with different threads (processes??)
#
sub new
{
    my ($class, $test_log) = @_;
    $class = ref($class) if ref($class);

    ###########
    # $self->[0]  Keep and restore $Test::TESTOUT
    # $self->[1]  test log file
    # $self->[2]  skip rest of the tests

    my @self = ('','','','','');
    my $self = bless \@self, $class;

    $self[1] = $test_log if $test_log;
    if($self[1]) {
        $self[0] = $Test::TESTOUT;
        unless ( open($Test::TESTOUT, ">>$self[1]") ) {
            warn( "Cannot open $self[1]\n" );
            $self->skip_rest();
            return undef
        }
        binmode $Test::TESTOUT; # make the test friendly for more platforms
    }

    $self
}

sub finish # end a test
{
   my ($self)=@_;
   if( $self->[1] ) {
       $self->[1]= '';
       unless (close( $Test::TESTOUT )) {
           warn( "Cannot close $self->[1]\n" );
       }
       $Test::TESTOUT = $self->[0];
   }
   1
}


sub skip_rest()
{
   my $self=shift @_;
   $self->[2] = 1;
}


sub work_breakdown  # open a file
{
   my $self=shift @_;
   plan( @_ );
   1
}




sub test
{
   my ($self, $actual_p, $expected_p, $name) = @_;
   print $Test::TESTOUT "# $name\n" if $name;
   if($self->[2]) {  # skip rest of tests switch
       print $Test::TESTOUT "# Test invalid because of previous failure.\n";
       skip( 1, 0, '');
       return 1; 
   }
   my $actual = Dumper(@$actual_p);
   my $expected = Dumper(@$expected_p);
   ok($actual, $expected, '');
}



sub verify  # store expected array for later use
{
   my ($self, $mod, $actual_p, $expected_p, $name) = @_;

   print $Test::TESTOUT "# $name\n" if $name;
   my $test_ok;
   if($self->[2]) {  # skip rest of tests switch
       print $Test::TESTOUT "# Test invalid because of previous failure.\n";
       skip( 1, 0, '');
       return 1; 
   }
  
   my $actual = Dumper(@$actual_p);
   my $expected = Dumper(@$expected_p);
   $test_ok = skip($mod, $actual, $expected, '');
   $test_ok = 1 if $mod;  # make sure do not stop 
   $test_ok

}


######
# Actual data
#
sub demo
{
   my ($self, $quoted_expression, @expression_results) = @_;

   #######
   # A demo trys to simulate someone typing expresssions
   # at a console.
   #

   #########
   # Print quoted expression so that see the non-executed
   # expression. The extra space is so when pasted into
   # a POD, the POD will process the line as code.
   #
   $quoted_expression =~ s/(\n+)/$1 => /g;
   print $Test::TESTOUT ' => ' . $quoted_expression . "\n";   

   ########
   # @data is the result of the script executing the 
   # quoted expression.
   #
   # The demo output most likely will end up in a pod. 
   # The the process of running the generated script
   # will execute the setup. Thus the input is the
   # actual results. Putting a space in front of it
   # tells the POD that it is code.
   #
   return unless @expression_results;
  
   $Data::Dumper::Terse = 1;
   my $data = Dumper(@expression_results);
   $data =~ s/(\n+)/$1 /g;
   $data =~ s/\\\\/\\/g;
   $data =~ s/\\'/'/g;

   print $Test::TESTOUT ' ' . $data . "\n" ;

}



1

__END__


=head1 NAME
  
STD::Tester - functions that support scripts generated by STD::TestGen

=head1 SYNOPSIS

  use STD::Tester

  $T = new Test:Tester;
  $success = $T->work_breakdown(@args);
  $success = $T->start($file_name);
  $success = $T->end( );
  $test_ok = $T->test(\@actual_results, \@expected_results);
  $test_ok = $T->verify(test, \@actual_results,  \@expected_results);
  $success = $T->skip_rest();

  $success = $T->demo( $quoted_expression, @expression_results );

=head1 DESCRIPTION

The C<STD::Testit> functions are designed to be used in test scripts
generated by C<STD::Testgen> (L<Testgen>) module.
The C<Teset::Testit> functions shall[1] isolated the actual and expected Perl
statements so they may be easily index for use in
automatically generated test documents 
and hide the intermediate variables used
to pass the acutal and expected values to the verification
routine. The C<Teset::Testit> functions shall[2] 
use the functions from the Test 
(L<Test>) module so that the results of the test scripts
generated by the STD::Testgen module may be used
with the currently developed Perl test tools.

=head1 QUALITY

The STD::Tester functions are a integral part of
the STD::Testgen module. 
The tests performed for the C<STD::Testgen>
module, (L<Testgen/QUALITY>), (L<testgen_p) by F<testgen_t.pl>
verifies all the requirements for the STD::Tester module.

=head1 AUTHOR

The holder of the copyright and maintainer is

E<lt>support@SoftwareDiamonds.comE<gt>

=head2 COPYRIGHT NOTICE

Copyrighted (c) 2002 Software Diamonds

All Rights Reserved

=head2 BINDING REQUIREMENTS NOTICE

Binding requirements are indexed with the
pharse 'shall[dd]' where dd is an unique number
for each header section.
This conforms to standard federal
government practices, 490A (L<STD490A/3.2.3.6>).
In accordance with the License, Software Diamonds
is not liable for any requirement, binding or otherwise.

=head2 LICENSE

Software Diamonds permits the redistribution
and use in source and binary forms, with or
without modification, provided that the 
following conditions are met: 

=over 4

=item 1

Redistributions of source code must retain
the above copyright notice, this list of
conditions and the following disclaimer. 

=item 2

Redistributions in binary form must 
reproduce the above copyright notice,
this list of conditions and the following 
disclaimer in the documentation and/or
other materials provided with the
distribution.

=back

SOFTWARE DIAMONDS, http::www.softwarediamonds.com,
PROVIDES THIS SOFTWARE 
'AS IS' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL SOFTWARE DIAMONDS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL,EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE,DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING USE OF THIS SOFTWARE, EVEN IF
ADVISED OF NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE POSSIBILITY OF SUCH DAMAGE. 

=head1 SEE ALSO

L<Test> L<Testgen/QUALITY> L<testgen_p> L<testgen_t>

=for html
<p><br>
<!-- BLK ID="NOTICE" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="OPT-IN" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="EMAIL" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="COPYRIGHT" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="LOG_CGI" -->
<!-- /BLK -->
<p><br>

=cut

### end of file ###