package Mojolicious::Plugin::DomIdHelper;

use Mojo::Base 'Mojolicious::Plugin';
use Mojo::Util qw{xml_escape};
use Scalar::Util qw{blessed};

our $VERSION = '0.1';

# Method used to retrieve the object's PK
my $METHOD = 'id';

# Character used to delimitthe package name from object's PK
my $DELIMITER = '_';

# Keep the full package name when generating the DOM ID, false = strip.
my $KEEP_NAMESPACE = 0;

sub register 
{
    my ($self, $app, $defaults) = @_;

    $defaults ||= {};
    $defaults->{method} ||= $METHOD;
    $defaults->{delimiter} ||= $DELIMITER;
    $defaults->{keep_namespace} ||= $KEEP_NAMESPACE;

    $app->helper( 
	dom_id => sub {
	    my $c      = shift;
	    my $obj    = shift;
	    my %config = (%$defaults, @_);

	    $self->_generate_dom_id($obj, %config);
	});
}

sub _generate_dom_id
{
    my ($self, $obj, %config) = @_;
    my $methods   = $config{method}; 
    my $delimiter = $config{delimiter};
    my $dom_id    = ref $obj;	# Provide support for all references (?)    

    return unless $dom_id;
    $dom_id = $self->_normalize_namespace($dom_id, $delimiter, $config{keep_namespace});		

    if(ref($methods) ne 'ARRAY') {
	$methods = [$methods];
    }

    # Append the ID suffix to blessed() refs only, others can't receive methods calls.
    if(blessed($obj)) {       
	my @suffix;
	for my $method (@$methods) {
	    push @suffix, $obj->$method;
	}
	
	local $_;
	@suffix = grep defined, @suffix;

	if(@suffix) {
	    $dom_id .= $delimiter;
	    $dom_id .= join '', @suffix;
	}
    }

    xml_escape($dom_id);
    $dom_id;
}

# Possibly dom_class()
sub _normalize_namespace
{
    my $self = shift;
    my ($name, $delimiter, $keep_namespace) = @_;
    my @namespace = split /\b::\b/, $name; 

    # Do we want to strip the prefix from the package name
    if(!$keep_namespace) {
      @namespace = pop @namespace;
    }
 
    # Split the package name on camelcase bounderies
    local $_;    
    $name = join $delimiter, map {
      s/([^A-Z])([A-Z])/$1$delimiter$2/g;
      s/([A-Z])([A-Z][^A-Z])/$1$delimiter$2/g;
      lc;
    } @namespace;    

    $name;
}

1;

__DATA__
=head1 NAME

Mojolicious::Plugin::DomIdHelper - Mojolicious plugin to generate DOM IDs from your ORM objects

=head1 SYNOPSIS

  # Mojolicious
  $self->plugin('dom_id_helper');

  # Or, your defaults
  $self->plugin('dom_id_helper', delimiter => '-')

  # Mojolicious::Lite   
  plugin 'dom_id_helper';

  # Set defaults 
  plugin 'dom_id_helper', delimiter => '-'

  # Your view
  <div id="<%= dom_id($object) %>">
    ...
  </div>

  <div id="<%= dom_id($object, method => 'name') ) %>">
    ...
  </div>

=head1 DESCRIPTION

DOM IDs are generated by joining an object's package name and its primary key with the character specified by the
L</delimiter> option. It's assumed that the primary key can be retrieved via a method named C<id>. 
This can be modified, see L</OPTIONS>.

For example, given an instance of C<DB::Package::User> with an ID of C<1>:

  dom_id($user)

will generate:

  user_1 

If the primary key is undefined only the package name is returned.

Multi-column primary keys are not separated by the L</delimiter> option, they are concatenated.

=head1 ORMs

The aim is to be ORM agnostic. Just set the L</method> option to the name of the method used to 
retrieve your object's primary key.

Multi-column primary keys returned as array references will cause problems (for now). 

=head1 OPTIONS

=head2 C<delimiter>

  plugin 'dom_id_helper', delimiter => '-'

The character used to delimit the object's package name from its primary key. Defaults to C<'_'>.

=head2 C<method>

  plugin 'dom_id_helper', method => 'name'
  plugin 'dom_id_helper', method => [qw{first_name last_name}]

The method used to retrieve the object's primary key. Defaults to C<'id'>.

=head2 C<keep_namespace>

  plugin 'dom_id_helper', keep_namespace => 1

Keep the full package name when generating the DOM ID. Defaults to C<0> (false).

=head1 AUTHOR

Skye Shaw <sshaw AT lucas.cis.temple.edu>
