package App::Framework;

=head1 NAME

App::Framework::Base - Base application object

=head1 SYNOPSIS

  use App::Framework ;

or 

  use App::Framework qw/Script/ ;

  App::Framework->new()->go() ;
  

=head1 DESCRIPTION

Base class for applications. Expected to be derived from by an implementable class (like App:Script).

Base provides (and handles) the following options:

	debug			Set the debug level value
	h|"help"		Show brief help message then exit
	man				Show full man page then exit
	pod				Show full man page as pod then exit

Once the object has been created it can then be run by calling the 'go()' method. go() calls in turn:

	* pre_run()		Gets & processes the options, then calls the pre_run_fn if set
	* run()			Handles any special options (-man etc), then calls the run_fn if set
	* post_run()	Calls post_run_fn if set
	* exit()		Called with exit value 0 if execution reaches this far

The pre_run_fn, run_fn, and post_run_fn fields of the object can either be set directly as part of the new() call,
or the prefered approach is to define pre_run, run, and post_run subroutines in the calling namespace. These subroutines
are detected by App::Framework::Base and automatically set on the object.

Similarly, the __DATA__ area of the calling namespace if the prefered area to use for object set up of common
fields (rather than as part of the new() call). Example __DATA__ contents:

	__DATA__
	
	[HISTORY]
	
	30-May-08	SDP		Re-written to use App::Framework::Script 
	28-May-08   SDP		New
	
	[SUMMARY]
	
	List (and repair) any faulty rrd database files
	
	[SYNOPSIS]
	
	$name [options] <rrd file(s)>
	
	[OPTIONS]
	
	-d|'dir'=s	temp directory	[default=/tmp]
	
	Specify the directory in which to store the xml output files (created by dumping the rrd databases)
	
	-repair 	Enable rrd repair
	
	When this option is specified, causes the script to repair any faulty rrd files
	
	
	[DESCRIPTION]
	
	Scans the specified rrd directory and lists any files which are 'faulty'. 
	
	Optionally this script can also repair the fault by setting the value to NaN.
	
	An export RRD database in XML file is of the form:
	
	  <!-- Round Robin Database Dump --><rrd>	<version> 0003 </version>
		<step> 300 </step> <!-- Seconds -->
		<lastupdate> 1211355308 </lastupdate> <!-- 2008-05-21 08:35:08 BST -->
 
This example sets the fields: history, summary, synopsis, options, and description. This information is also used to 
automatically create the application pod, man, and usage pages.

Similarly, the $VERSION variable in the calling namespace is detected and used to set the application version number.

=head2 FIELDS

The following fields should be defined either in the call to 'new()' or as part of the application configuration in the __DATA__ section:

 * name = Program name (default is name of program)
 * summary = Program summary text
 * synopsis = Synopsis text (default is program name and usage)
 * description = Program description text
 * history = Release history information
 * version = Program version (default is value of 'our $VERSION')
 * options = Definition of program options (see below)
 * nameargs = Definition of the program arguments and their intended usage (see below)
 * sql = Definition of sql database connection & queries (see below)
 
 * pre_run_fn = Function called before run() function (default is application-defined 'pre_run' subroutine if available)
 * run_fn = Function called to execute program (default is application-defined 'run' subroutine if available)
 * post_run_fn = Function called after run() function (default is application-defined 'post_run' subroutine if available)
 * usage_fn = Function called to display usage information (default is application-defined 'usage' subroutine if available)

During program execution, the following values can be accessed:

 * arglist = Array of the program arguments, in the order they were specified
 * arghash = Hash of the program arguments, named by the 'nameargs' field
 * package = Name of the application package (usually main::)
 * filename = Full filename path to the application (after following any links)
 * progname = Name of the program (without path or extension)
 * progpath = Pathname to program
 * progext = Extension of program
 * runobj = L<App::Framework::Base::Run> object
 


=head2 Loaded modules

App::Framework::Base pre-loads the user namespace with the following modules: 

	'Cwd', 
	'File::Basename',
	'File::Path',
	'File::Temp',
	'File::Spec',
	'File::Find',
	'File::Copy',
	"File::DosGlob 'glob'",
	
	'Pod::Usage',
	'Date::Manip',
	'Getopt::Long qw(:config no_ignore_case)',
	
=head2 MySql support

The 'sql' field may be specified as either a HASH ref or an ARRAY ref (where each ARRAY entry is a HASH ref). The HASH ref must
contain sufficient information to create a L<App::Framework::Base::Sql> object. 

Calling to the sql() method with no parameter will return the first created L<App::Framework::Base::Sql> object. Calling the sql() method with a string will
return the L<App::Framework::Base::Sql> object created for the named database (i.e. sql objects are named by their database). The sql object can then be
used as defined in L<App::Framework::Base::Sql>

=head2 HISTORY

I started this package ....


=cut

use strict ;
use Carp ;

our $VERSION = "0.01" ;


#============================================================================================
# OBJECT HIERARCHY
#============================================================================================
our @ISA ; 

#============================================================================================
# GLOBALS
#============================================================================================

# Keep track of import info
my %imports ;


#============================================================================================

=head2 CONSTRUCTOR

=over 4

=cut


#============================================================================================

# Set up module import
sub import 
{
    my $pkg     = shift;
    my $callpkg = caller(0);
    my $pattern = ( $callpkg eq 'main' ) ? '^:::' : "^$callpkg\$";
    
    
    my $verbose = 0;
    my $item;
    my $file;

    for $item (@_) 
    {
    	print " + item: $item\n" ;
    	$imports{$callpkg} = $item ;
    }
    
}

#----------------------------------------------------------------------------------------------

=item C<< new([%args]) >>

Create a new object.

The %args are specified as they would be in the B<set> method, for example:

	'adapter_num' => 0

The full list of possible arguments are as described in the L</FIELDS> section

=cut

sub new
{
	my ($obj, %args) = @_ ;

	my $class = ref($obj) || $obj ;
    my $callpkg = caller(0);
	
	# Get name of requested personality
	my $personality = ($imports{$callpkg} || 'Script' ) ;
	my $module =  "App::Framework::Modules::$personality" ; 
	eval "require $module;" ;
	croak "Sorry, App:Framework does not support personality \"$personality\"" if $@ ;

	## Create ourself as if we're an object of the required type	
	@ISA = ( $module ) ;

	# Create object
	my $this = $class->SUPER::new(
		%args, 
	) ;
	$this->set(
		'usage_fn' 	=> sub {$this->script_usage(@_);}, 
	) ;

	return($this) ;
}


=back

=head1 AUTHOR

Steve Price, C<< <sdprice at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-app-framework at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=App-Framework>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc App::Framework


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=App-Framework>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/App-Framework>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/App-Framework>

=item * Search CPAN

L<http://search.cpan.org/dist/App-Framework/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 COPYRIGHT & LICENSE

Copyright 2009 Steve Price, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.


=cut

# ============================================================================================
# END OF PACKAGE
1;

__END__


