package Perinci::Examples::Stream;

our $DATE = '2015-06-16'; # DATE
our $VERSION = '0.53'; # VERSION

use 5.010;
use strict;
use warnings;

our %SPEC;

$SPEC{':package'} = {
    v => 1.1,
    summary => 'Examples for streaming input/output',
    description => <<'_',

This package contains functions that demonstrate streaming input/output.

_
};

$SPEC{nat} = {
    v => 1.1,
    summary => 'This function produces a stream of natural numbers',
    args => {
        num => {
            schema => 'int*',
            cmdline_aliases => {n=>{}},
        },
    },
    result => {
        stream => 1,
        schema => 'int*',
    },
};
sub nat {
    my %args = @_;
    my $i = 1;
    my $num = $args{num};
    [200, "OK", sub {
         return undef if defined($num) && $i > $num;
         $i++;
     }];
}

$SPEC{hash_stream} = {
    v => 1.1,
    summary => 'This function produces a stream of hashes',
    args => {
        num => {
            schema => 'int*',
            cmdline_aliases => {n=>{}},
        },
    },
    result => {
        stream => 1,
        schema => 'hash*',
    },
};
sub hash_stream {
    my %args = @_;
    my $num = $args{num};

    my $i = 1;
    [200, "OK", sub {
         return undef if defined($num) && $i > $num;
         {num=>$i++};
     }];
}

$SPEC{square_input} = {
    v => 1.1,
    summary => 'This function squares its stream input',
    args => {
        input => {
            req => 1,
            stream => 1,
            schema => 'float*',
            cmdline_src => 'stdin_or_files',
        },
    },
    result => {
        stream => 1,
        schema => 'float*',
    },
};
sub square_input {
    my %args = @_;
    my $input = $args{input};

    [200, "OK", sub {
         my $n = $input->();
         return undef unless defined $n;
         $n*$n;
     }];
}

$SPEC{wc} = {
    v => 1.1,
    summary => 'Count the number of lines/words/characters of input, like the "wc" command',
    args => {
        input => {
            req => 1,
            stream => 1,
            schema => 'str*',
            cmdline_src => 'stdin_or_files',
        },
    },
    result => {
        schema => 'str*',
    },
};
sub wc {
    my %args = @_;
    my $input = $args{input};

    my ($lines, $words, $chars) = (0,0,0);
    while (defined( my $line = $input->())) {
        $lines++;
        $words++ for $line =~ /(\S+)/g;
        $chars += length($line);
    }
    [200, "OK", {lines=>$lines, words=>$words, chars=>$chars}];
}

$SPEC{wc_keys} = {
    v => 1.1,
    summary => 'Count the number of keys of each hash',
    description => <<'_',

This is a simple demonstration of accepting a stream of hashes. In command-line
application this will translate to JSON stream.

_
    args => {
        input => {
            req => 1,
            stream => 1,
            schema => 'hash*',
            cmdline_src => 'stdin_or_files',
        },
    },
    result => {
        schema => 'str*',
    },
};
sub wc_keys {
    my %args = @_;
    my $input = $args{input};

    my ($keys) = (0);
    while (defined(my $hash = $input->())) {
        $keys += keys %$hash;
    }
    [200, "OK", {keys=>$keys}];
}

1;
# ABSTRACT: Examples for streaming input/output

__END__

=pod

=encoding UTF-8

=head1 NAME

Perinci::Examples::Stream - Examples for streaming input/output

=head1 VERSION

This document describes version 0.53 of Perinci::Examples::Stream (from Perl distribution Perinci-Examples), released on 2015-06-16.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Perinci-Examples>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Perinci-Examples>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Perinci-Examples>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 DESCRIPTION


This package contains functions that demonstrate streaming input/output.

=head1 FUNCTIONS


=head2 hash_stream(%args) -> [status, msg, result, meta]

This function produces a stream of hashes.

Arguments ('*' denotes required arguments):

=over 4

=item * B<num> => I<int>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (hash)


=head2 nat(%args) -> [status, msg, result, meta]

This function produces a stream of natural numbers.

Arguments ('*' denotes required arguments):

=over 4

=item * B<num> => I<int>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (int)


=head2 square_input(%args) -> [status, msg, result, meta]

This function squares its stream input.

Arguments ('*' denotes required arguments):

=over 4

=item * B<input>* => I<float>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (float)


=head2 wc(%args) -> [status, msg, result, meta]

Count the number of lines/words/characters of input, like the "wc" command.

Arguments ('*' denotes required arguments):

=over 4

=item * B<input>* => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (str)


=head2 wc_keys(%args) -> [status, msg, result, meta]

Count the number of keys of each hash.

This is a simple demonstration of accepting a stream of hashes. In command-line
application this will translate to JSON stream.

Arguments ('*' denotes required arguments):

=over 4

=item * B<input>* => I<hash>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (str)

=cut
