package Dist::Zilla::Plugin::ShareDir::Clean;

our $DATE = '2015-04-06'; # DATE
our $VERSION = '0.03'; # VERSION

use 5.010001;
use strict;
use warnings;
use utf8;

use Moose;
use namespace::autoclean;

use File::Find;
use IPC::System::Options qw(backtick);
use List::Util qw(first);

with (
    'Dist::Zilla::Role::InstallTool',
);

has dir => (is=>'rw', default=>'share');
has clean_tarball => (is=>'rw', default=>1);

sub setup_installer {
  my ($self) = @_;

  my $build_script = first { $_->name eq 'Makefile.PL' }
      @{ $self->zilla->files };
  $self->log_fatal('No Makefile.PL found. Using [MakeMaker] is required')
      unless $build_script;

  my $content = $build_script->content;

  no strict 'refs';
  my $header = "# modify generated Makefile to remove current per-dist share dir before installation. this piece is generated by " .
      __PACKAGE__ . " version " . (${__PACKAGE__ ."::VERSION"} // 'dev').".\n";

  my $body = <<'_';
CLEAN_SHARE_DIR:
{
    #require Perl::osnames;
    #last unless Perl::osnames::is_posix();

    print "Modifying Makefile to remove current per-dist share dir before installation\n";
    open my($fh), "<", "Makefile" or die "Can't open generated Makefile: $!";
    my $content = do { local $/; ~~<$fh> };

    $content =~ s/^(install ::) (.+)/$1 clean_sharedir $2/m
        or die "Can't find pattern in Makefile (1)";

    $content .= qq|\nclean_sharedir :\n\t| .
        q|$(RM_RF) {{dir}}| .
        qq|\n\n|;

    open $fh, ">", "Makefile" or die "Can't write modified Makefile: $!";
    print $fh $content;
}
_


  my $dist_name = $self->zilla->name;
  my $dir = qq{"\$(SITELIBEXP)/auto/share/dist/$dist_name"};

  $body =~ s/\{\{dir\}\}/$dir/;
  $content .= $header . $body;
  $self->log_debug(["adding section in Makefile.PL to clean share dir before installation"]);
  return $build_script->content($content);
}

no Moose;
1;
# ABSTRACT: Delete current per-dist share dir before install

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::ShareDir::Clean - Delete current per-dist share dir before install

=head1 VERSION

This document describes version 0.03 of Dist::Zilla::Plugin::ShareDir::Clean (from Perl distribution Dist-Zilla-Plugin-ShareDir-Clean), released on 2015-04-06.

=head1 SYNOPSIS

In your dist.ini:

 [ShareDir]

 [ShareDir::Clean]
 ;dir=share

=head1 DESCRIPTION

B<WARNING: This plugin will add a code in Makefile.PL to *DELETE* (rm -rf) a
directory in the installation target, which might have severe consequences.
Don't use this plugin unless you know exactly what you're doing.>

This plugin is an alternative to using L<Dist::Zilla::Plugin::ShareDir::Tarball>
(please read the documentation of that module first for the rationale). With
this plugin, you can keep using L<Dist::Zilla::Plugin::ShareDir>, but eliminate
the problem of lingering old files.

What this plugin does is delete the current per-dist share directory in the
installation target before installation.

Some caveats/current limitations:

=over

=item * Only Makefile.PL is currently supported.

=item * Windows (or other non-POSIX) installation target system is not yet tested.

=back

All in all, I think Yanick's ShareDir::Tarball plugin is a slightly more elegant
hack and I use it. But when it's more appropriate, I use this plugin too.

=for Pod::Coverage ^(setup_installer)$

=head1 CONFIGURATION

=head2 dir => str (default: share)

Name of shared directory.

=head1 SEE ALSO

L<Dist::Zilla::Plugin::ShareDir>

L<Dist::Zilla::Plugin::ShareDir::Tarball>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Dist-Zilla-Plugin-ShareDir-Clean>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Dist-Zilla-Plugin-ShareDir-Clean>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Dist-Zilla-Plugin-ShareDir-Clean>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
