package Class::GenSource;

our $DATE = '2015-06-10'; # DATE
our $VERSION = '0.02'; # VERSION

use 5.010001;
use strict;
use warnings;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(gen_class_source_code);

our %SPEC;

my $re_ident = qr/\A[A-Za-z_][A-Za-z0-9_]*(::[A-Za-z_][A-Za-z0-9_]*)*\z/;

$SPEC{gen_class_source_code} = {
    v => 1.1,
    summary => 'Generate Perl source code to declare a class',
    description => <<'_',

_
    args => {
        name => {
            schema  => ['str*', match=>$re_ident],
            req => 1,
        },
        parent => {
            schema  => ['str*', match=>$re_ident],
        },
        attributes => {
            schema  => ['hash*', match=>$re_ident],
            default => {},
        },
        variant => {
            schema => ['str*', in=>[qw/classic Mo Moo Moose/]],
            default => 'classic',
        },
    },
    result_naked => 1,
};
sub gen_class_source_code {
    my %args = @_;

    # XXX schema
    my $variant = $args{variant} // 'classic';

    my @res;

    push @res, "package $args{name};\n";
    if ($variant =~ /^Mo/) {
        push @res, "use $variant;\n";
    }

    if ($args{parent}) {
        if ($variant =~ /^Mo/) {
            push @res, "extends '$args{parent}';\n";
        } else {
            push @res, "use parent qw(", $args{parent}, ");\n";
        }
    }

    if ($variant eq 'classic') {
        push @res, q[sub new { my $class = shift; bless {@_}, $class; }], "\n";
    }

    my $attrs = $args{attributes} // {};
    for my $name (sort keys %$attrs) {
        if ($variant =~ /^Mo/) {
            push @res, "has $name => (is=>'rw');\n";
        } else {
            push @res, "sub $name {}\n";
        }
    }

    join("", @res);
}

1;
# ABSTRACT: Generate Perl source code to declare a class

__END__

=pod

=encoding UTF-8

=head1 NAME

Class::GenSource - Generate Perl source code to declare a class

=head1 VERSION

This document describes version 0.02 of Class::GenSource (from Perl distribution Class-GenSource), released on 2015-06-10.

=head1 SYNOPSIS

 use Class::GenSource qw(gen_class_source_code);

 say gen_class_source_code(
     name => 'My::Class',
     attributes => {
         foo => {},
         bar => {},
         baz => {},
     },
 );

Will print something like:

 package My::Class;

 sub new { my $class = shift; bless {@_}, $class }
 sub foo {}
 sub bar {}
 sub baz {}

Another example (generating L<Moo>-based class):

 say gen_class_source_code(
     name => 'My::Class',
     attributes => {
         foo => {},
         bar => {},
         baz => {},
     },
     variant => 'Moo',
 );

will print something like:

 package My::Class;
 use Moo;
 has foo => (is=>'rw');
 has bar => (is=>'rw');
 has baz => (is=>'rw');

=head1 DESCRIPTION

=head1 FUNCTIONS


=head2 gen_class_source_code(%args) -> any

Generate Perl source code to declare a class.

Arguments ('*' denotes required arguments):

=over 4

=item * B<attributes> => I<hash> (default: {})

=item * B<name>* => I<str>

=item * B<parent> => I<str>

=item * B<variant> => I<str> (default: "classic")

=back

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Class-GenSource>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Class-GenSource>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Class-GenSource>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
