package CPAN::Module::FromURL;

our $DATE = '2016-02-21'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.010001;
use strict;
use warnings;

use Exporter qw(import);
our @EXPORT_OK = qw(extract_module_from_cpan_url);

our %SPEC;

$SPEC{extract_module_from_cpan_url} = {
    v => 1.1,
    summary => 'Detect and extract module name from some CPAN-related URL',
    args => {
        url => {
            schema => 'str*',
            req => 1,
            pos => 0,
        },
    },
    args_as => 'array',
    result => {
        schema => 'str',
    },
    result_naked => 1,
    examples => [
        {
            args => {url=>'https://metacpan.org/pod/Foo::Bar'},
            result => 'Foo::Bar',
            test => 0, # TMP: there's still a bug in testri?
        },
        {
            args => {url=>'https://metacpan.org/pod/release/SMONF/Dependencies-Searcher-0.066_001/lib/Dependencies/Searcher.pm'},
            result => 'Dependencies::Searcher',
            test => 0, # TMP: there's still a bug in testri?
        },
        {
            args => {url=>'https://www.google.com/'},
            result => undef,
            test => 0, # TMP: there's still a bug in testri?
        },
    ],
};
sub extract_module_from_cpan_url {
    my $url = shift;

    if ($url =~ m!\Ahttps?://metacpan\.org/pod/(\w+(?:::\w+)*)\z!i) {
        return $1;
    }

    if ($url =~ m!\Ahttps?://metacpan\.org/pod/release/[^/]+/[^/]+/lib/((?:[^/]+/)*\w+)\.pm\z!i) {
        my $mod = $1;
        $mod =~ s!/!::!g;
        return $mod;
    }

    if ($url =~ m!\Ahttps?://search\.cpan\.org/~[^/]+/[^/]+/lib/((?:[^/]+/)*\w+).pm\z!i) {
        my $mod = $1;
        $mod =~ s!/!::!g;
        return $mod;
    }

    undef;
}

1;
# ABSTRACT: Detect and extract module name from some CPAN-related URL

__END__

=pod

=encoding UTF-8

=head1 NAME

CPAN::Module::FromURL - Detect and extract module name from some CPAN-related URL

=head1 VERSION

This document describes version 0.01 of CPAN::Module::FromURL (from Perl distribution CPAN-Module-FromURL), released on 2016-02-21.

=head1 FUNCTIONS


=head2 extract_module_from_cpan_url($url) -> str

Detect and extract module name from some CPAN-related URL.

Examples:

=over

=item * Example #1:

 extract_module_from_cpan_url( url => "https://metacpan.org/pod/Foo::Bar"); # -> "Foo::Bar"

=item * Example #2:

 extract_module_from_cpan_url(  url => "https://metacpan.org/pod/release/SMONF/Dependencies-Searcher-0.066_001/lib/Dependencies/Searcher.pm");

Result:

 "Dependencies::Searcher"

=item * Example #3:

 extract_module_from_cpan_url( url => "https://www.google.com/"); # -> undef

=back

This function is not exported by default, but exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<url>* => I<str>

=back

Return value:  (str)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/CPAN-Module-FromURL>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-CPAN-Module-FromURL>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=CPAN-Module-FromURL>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
