package Bencher::Scenario::HumanDateParsingModules::Parsing;

our $DATE = '2016-07-17'; # DATE
our $VERSION = '0.005'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.',
    modules => {
        'DateTime::Format::Alami::EN' => {version => 0.13},
        'DateTime::Format::Alami::ID' => {version => 0.13},
    },
    participants => [
        {
            module=>'DateTime::Format::Alami::EN',
            code_template => 'state $parser = DateTime::Format::Alami::EN->new; $parser->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Alami::ID',
            code_template => 'state $parser = DateTime::Format::Alami::ID->new; $parser->parse_datetime(<text>)',
            tags => ['lang:id'],
        },
        {
            module=>'Date::Extract',
            code_template => 'state $parser = Date::Extract->new; $parser->extract(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Natural',
            code_template => 'state $parser = DateTime::Format::Natural->new; $parser->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Flexible',
            code_template => 'DateTime::Format::Flexible->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
    ],
    datasets => [
        {args => {text => '18 feb'}},
        {args => {text => '18 feb 2011'}},
        {args => {text => '18 feb 2011 06:30:45'}},
        {args => {text => 'today'}, include_participant_tags => ['lang:en']},
        {args => {text => 'hari ini'}, include_participant_tags => ['lang:id']},
    ],
};

1;
# ABSTRACT: Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::HumanDateParsingModules::Parsing - Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.

=head1 VERSION

This document describes version 0.005 of Bencher::Scenario::HumanDateParsingModules::Parsing (from Perl distribution Bencher-Scenarios-HumanDateParsingModules), released on 2016-07-17.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m HumanDateParsingModules::Parsing

To run module startup overhead benchmark:

 % bencher --module-startup -m HumanDateParsingModules::Parsing

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Date::Extract> 0.05.01

L<DateTime::Format::Alami::EN> 0.13

L<DateTime::Format::Alami::ID> 0.13

L<DateTime::Format::Flexible> 0.26

L<DateTime::Format::Natural> 1.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * DateTime::Format::Alami::EN (perl_code) [lang:en]

Code template:

 state $parser = DateTime::Format::Alami::EN->new; $parser->parse_datetime(<text>)



=item * DateTime::Format::Alami::ID (perl_code) [lang:id]

Code template:

 state $parser = DateTime::Format::Alami::ID->new; $parser->parse_datetime(<text>)



=item * Date::Extract (perl_code) [lang:en]

Code template:

 state $parser = Date::Extract->new; $parser->extract(<text>)



=item * DateTime::Format::Natural (perl_code) [lang:en]

Code template:

 state $parser = DateTime::Format::Natural->new; $parser->parse_datetime(<text>)



=item * DateTime::Format::Flexible (perl_code) [lang:en]

Code template:

 DateTime::Format::Flexible->parse_datetime(<text>)



=back

=head1 BENCHMARK DATASETS

=over

=item * 18 feb

=item * 18 feb 2011

=item * 18 feb 2011 06:30:45

=item * today

=item * hari ini

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m HumanDateParsingModules::Parsing >>):

 #table1#
 {dataset=>"18 feb"}
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | participant                 | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Flexible  |       540 |     1.9   |       1    | 1.1e-05 |      21 |
 | Date::Extract               |      1120 |     0.89  |       2.08 | 4.3e-07 |      20 |
 | DateTime::Format::Alami::EN |      1410 |     0.707 |       2.62 | 6.4e-07 |      20 |
 | DateTime::Format::Natural   |      1790 |     0.559 |       3.31 | 2.1e-07 |      20 |
 | DateTime::Format::Alami::ID |      2120 |     0.472 |       3.93 | 3.8e-07 |      25 |
 +-----------------------------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table2#
 {dataset=>"18 feb 2011"}
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | participant                 | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Alami::EN |       564 |     1.77  |       1    | 9.1e-07 |      20 |
 | DateTime::Format::Flexible  |       560 |     1.8   |       1    | 1.2e-05 |      20 |
 | Date::Extract               |       773 |     1.29  |       1.37 | 4.8e-07 |      20 |
 | DateTime::Format::Alami::ID |       957 |     1.05  |       1.7  | 4.3e-07 |      20 |
 | DateTime::Format::Natural   |      1060 |     0.947 |       1.87 | 6.5e-07 |      23 |
 +-----------------------------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table3#
 {dataset=>"18 feb 2011 06:30:45"}
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | participant                 | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Flexible  |       550 |     1.8   |       1    | 6.6e-06 |      21 |
 | Date::Extract               |       774 |     1.29  |       1.41 | 4.3e-07 |      20 |
 | DateTime::Format::Natural   |       984 |     1.02  |       1.79 | 4.8e-07 |      20 |
 | DateTime::Format::Alami::EN |      2080 |     0.482 |       3.78 | 4.2e-07 |      21 |
 | DateTime::Format::Alami::ID |      2110 |     0.474 |       3.85 | 4.7e-07 |      21 |
 +-----------------------------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table4#
 {dataset=>"hari ini"}
 +-----------------------------+----------+------+-----------+-----------+------------+---------+---------+
 | participant                 | dataset  | perl | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-----------------------------+----------+------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Alami::ID | hari ini | perl |      5100 |       190 |          1 | 4.1e-07 |      22 |
 +-----------------------------+----------+------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table5#
 {dataset=>"today"}
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | participant                 | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Flexible  |       550 |     1.8   |       1    | 9.6e-06 |      20 |
 | Date::Extract               |      1550 |     0.646 |       2.83 | 2.7e-07 |      20 |
 | DateTime::Format::Natural   |      3160 |     0.317 |       5.77 | 2.5e-07 |      23 |
 | DateTime::Format::Alami::EN |      5100 |     0.2   |       9.2  | 3.8e-07 |      25 |
 +-----------------------------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


Benchmark module startup overhead (C<< bencher -m HumanDateParsingModules::Parsing --module-startup >>):

 #table6#
 +-----------------------------+-----------+------------------------+------------+-----------+---------+
 | participant                 | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors   | samples |
 +-----------------------------+-----------+------------------------+------------+-----------+---------+
 | DateTime::Format::Flexible  |      74   |                   71.4 |        1   |   0.00028 |      25 |
 | Date::Extract               |      61   |                   58.4 |        1.2 |   0.00018 |      21 |
 | DateTime::Format::Natural   |      60   |                   57.4 |        1.2 |   0.00027 |      21 |
 | DateTime::Format::Alami::ID |      15   |                   12.4 |        4.9 | 5.7e-05   |      20 |
 | DateTime::Format::Alami::EN |      15   |                   12.4 |        4.9 | 2.6e-05   |      20 |
 | perl -e1 (baseline)         |       2.6 |                    0   |       29   | 5.3e-06   |      20 |
 +-----------------------------+-----------+------------------------+------------+-----------+---------+


=begin html

<img src="data:image/png;base64,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" />

=end html


=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-HumanDateParsingModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-DateTimeFormatAlami>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-HumanDateParsingModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
