package Bencher::Scenario::IODParsingModules;

our $DATE = '2016-06-26'; # DATE
our $VERSION = '0.04'; # VERSION

use 5.010001;
use strict;
use warnings;

use File::ShareDir::Tarball qw(dist_dir);

our $scenario = {
    summary => 'Benchmark IOD/INI parsing modules',
    participants => [
        {
            module => 'Config::IOD::Reader',
            code_template => 'state $iod = Config::IOD::Reader->new; $iod->read_file(<filename>)',
        },
        {
            module => 'Config::IOD',
            code_template => 'state $iod = Config::IOD->new; $iod->read_file(<filename>)',
        },

        {
            module => 'Config::INI::Reader',
            code_template => 'Config::INI::Reader->read_file(<filename>)',
            tags => ['ini'],
        },
        {
            module => 'Config::IniFiles',
            code_template => 'Config::IniFiles->new(-file => <filename>)',
            tags => ['ini'],
        },
        {
            module => 'Config::Simple::Conf',
            code_template => 'Config::Simple::Conf->new(<filename>)',
            tags => ['ini'],
        },
    ],

    datasets => [
    ],
};

my $dir = dist_dir('IOD-Examples')
    or die "Can't find share dir for IOD-Examples";
for my $filename (glob "$dir/examples/extra-bench-*.iod") {
    my $basename = $filename; $basename =~ s!.+/!!;
    push @{ $scenario->{datasets} }, {
        name => $basename,
        args => {filename => $filename},
        ( exclude_participant_tags => ['ini'] ) x ($basename =~ /basic\.iod/ ? 1:0), # these files are not parseable by INI parsers
    };
}

1;
# ABSTRACT: Benchmark IOD/INI parsing modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::IODParsingModules - Benchmark IOD/INI parsing modules

=head1 VERSION

This document describes version 0.04 of Bencher::Scenario::IODParsingModules (from Perl distribution Bencher-Scenario-IODParsingModules), released on 2016-06-26.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m IODParsingModules

To run module startup overhead benchmark:

 % bencher --module-startup -m IODParsingModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Config::IOD::Reader> 0.24

L<Config::IOD> 0.29

L<Config::INI::Reader> 0.025

L<Config::IniFiles> 2.92

L<Config::Simple::Conf> 2.002

=head1 BENCHMARK PARTICIPANTS

=over

=item * Config::IOD::Reader (perl_code)

Code template:

 state $iod = Config::IOD::Reader->new; $iod->read_file(<filename>)



=item * Config::IOD (perl_code)

Code template:

 state $iod = Config::IOD->new; $iod->read_file(<filename>)



=item * Config::INI::Reader (perl_code) [ini]

Code template:

 Config::INI::Reader->read_file(<filename>)



=item * Config::IniFiles (perl_code) [ini]

Code template:

 Config::IniFiles->new(-file => <filename>)



=item * Config::Simple::Conf (perl_code) [ini]

Code template:

 Config::Simple::Conf->new(<filename>)



=back

=head1 BENCHMARK DATASETS

=over

=item * extra-bench-basic-compat.iod

=item * extra-bench-basic.iod

=item * extra-bench-typical1.iod

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m IODParsingModules >>):

 +----------------------+------------------------------+-----------+-----------+------------+---------+---------+
 | participant          | dataset                      | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +----------------------+------------------------------+-----------+-----------+------------+---------+---------+
 | Config::IniFiles     | extra-bench-basic-compat.iod |       300 |     3.4   |       1    | 1.5e-05 |      20 |
 | Config::INI::Reader  | extra-bench-basic-compat.iod |      1200 |     0.836 |       4.04 | 4.8e-07 |      20 |
 | Config::IniFiles     | extra-bench-typical1.iod     |      1430 |     0.7   |       4.83 | 6.9e-07 |      20 |
 | Config::IOD          | extra-bench-basic-compat.iod |      1510 |     0.663 |       5.1  | 4.8e-07 |      20 |
 | Config::IOD          | extra-bench-basic.iod        |      1500 |     0.65  |       5.2  | 1.3e-06 |      20 |
 | Config::IOD::Reader  | extra-bench-basic-compat.iod |      1900 |     0.526 |       6.43 | 2.7e-07 |      20 |
 | Config::IOD::Reader  | extra-bench-basic.iod        |      1930 |     0.518 |       6.52 | 2.7e-07 |      20 |
 | Config::Simple::Conf | extra-bench-basic-compat.iod |      2140 |     0.466 |       7.25 | 4.5e-07 |      23 |
 | Config::INI::Reader  | extra-bench-typical1.iod     |      6200 |     0.16  |      21    | 2.1e-07 |      21 |
 | Config::IOD          | extra-bench-typical1.iod     |      6800 |     0.15  |      23    | 1.9e-07 |      26 |
 | Config::Simple::Conf | extra-bench-typical1.iod     |      8900 |     0.11  |      30    | 2.2e-07 |      29 |
 | Config::IOD::Reader  | extra-bench-typical1.iod     |     10000 |     0.098 |      35    | 1.1e-07 |      32 |
 +----------------------+------------------------------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m IODParsingModules --module-startup >>):

 +----------------------+-----------+------------------------+------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +----------------------+-----------+------------------------+------------+---------+---------+
 | Config::INI::Reader  |      26   |                   23.4 |        1   | 5.6e-05 |      21 |
 | Config::IniFiles     |      26   |                   23.4 |        1   | 8.3e-05 |      21 |
 | Config::IOD::Reader  |       8.3 |                    5.7 |        3.2 | 2.4e-05 |      21 |
 | Config::IOD          |       8.2 |                    5.6 |        3.2 | 5.5e-05 |      20 |
 | Config::Simple::Conf |       4.2 |                    1.6 |        6.3 | 4.1e-05 |      22 |
 | perl -e1 (baseline)  |       2.6 |                    0   |       10   | 1.2e-05 |      21 |
 +----------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-IODParsingModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-IODParsingModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-IODParsingModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
