package App::ZodiacUtils;

our $DATE = '2015-11-28'; # DATE
our $VERSION = '0.04'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

$SPEC{zodiac_of} = {
    v => 1.1,
    summary => 'Show zodiac for a date',
    args => {
        dates => {
            summary => 'Dates',
            'x.name.is_plural' => 1,
            #schema => ['array*', of=>'date*', min_len=>1],
            schema => ['array*', of=>'str*', min_len=>1],
            #'x.perl.coerce_to' => 'int(epoch)',
            req => 1,
            pos => 0,
            greedy => 1,
        },
    },
    result_naked => 1,
    examples => [
        {
            args => {dates=>['2015-06-15']},
            result => 'gemini',
            test => 0, # at the time of this writing, Test::Rinci hasn't used Perinci::Sub::CoerceArgs
        },
    ],
};
sub zodiac_of {
    require Zodiac::Tiny;
    my %args = @_;

    my $dates = $args{dates};

    my $res = [];
    for my $date (@$dates) {
        unless ($date =~ /\A\d\d\d\d-\d\d-\d\d\z/) {
            warn "Invalid date '$date'\n";
            next;
        }
        my $z = Zodiac::Tiny::zodiac_of($date);
        push @$res, @$dates > 1 ? [$date, $z] : $z;
    }
    $res = $res->[0] if @$res == 1;
    $res;
}

$SPEC{chinese_zodiac_of} = {
    v => 1.1,
    summary => 'Show Chinese zodiac for a date',
    args => {
        dates => {
            summary => 'Dates',
            'x.name.is_plural' => 1,
            #schema => ['array*', of=>'date*', min_len=>1],
            schema => ['array*', of=>'str*', min_len=>1],
            #'x.perl.coerce_to' => 'int(epoch)',
            req => 1,
            pos => 0,
            greedy => 1,
        },
    },
    result_naked => 1,
    examples => [
        {
            args => {dates=>['1980-02-17']},
            result => 'monkey (metal)',
        },
    ],

};
sub chinese_zodiac_of {
    require Zodiac::Chinese::Table;
    my %args = @_;

    my $dates = $args{dates};

    my $res = [];
    for my $date (@$dates) {
        unless ($date =~ /\A\d\d\d\d-\d\d-\d\d\z/) {
            warn "Invalid date '$date'\n";
            next;
        }
        my $czres = Zodiac::Chinese::Table::chinese_zodiac($date);
        my $z = $czres ? "$czres->[7] ($czres->[3])" : undef;
        push @$res, @$dates > 1 ? [$date, $z] : $z;
    }
    $res = $res->[0] if @$res == 1;
    $res;
}

1;
# ABSTRACT: CLI utilities related to zodiac

__END__

=pod

=encoding UTF-8

=head1 NAME

App::ZodiacUtils - CLI utilities related to zodiac

=head1 VERSION

This document describes version 0.04 of App::ZodiacUtils (from Perl distribution App-ZodiacUtils), released on 2015-11-28.

=head1 DESCRIPTION

This distribution includes the following CLI utilities:

=over

=item * L<chinese-zodiac-of>

=item * L<zodiac-of>

=back

=head1 FUNCTIONS


=head2 chinese_zodiac_of(%args) -> [status, msg, result, meta]

Show Chinese zodiac for a date.

Examples:

 chinese_zodiac_of( dates => ["1980-02-17"]); # -> "monkey (metal)"
This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

Dates.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 zodiac_of(%args) -> [status, msg, result, meta]

Show zodiac for a date.

Examples:

 zodiac_of( dates => ["2015-06-15"]); # -> "gemini"
This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

Dates.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 SEE ALSO

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-ZodiacUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-ZodiacUtils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-ZodiacUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
