package Net::DNS::RR::SMIMEA;

#
# $Id: SMIMEA.pm 1447 2016-01-28 15:51:54Z willem $
#
use vars qw($VERSION);
$VERSION = (qw$LastChangedRevision: 1447 $)[1];


use strict;
use base qw(Net::DNS::RR);

=head1 NAME

Net::DNS::RR::SMIMEA - DNS SMIMEA resource record

=cut


use integer;

use constant BABBLE => defined eval { require Digest::BubbleBabble; };


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $next = $offset + $self->{rdlength};

	@{$self}{qw(usage selector matchingtype)} = unpack "\@$offset C3", $$data;
	$offset += 3;
	$self->{certbin} = substr $$data, $offset, $next - $offset;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	return '' unless defined $self->{certbin};
	return pack 'C3 a*', @{$self}{qw(usage selector matchingtype certbin)};
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	return '' unless defined $self->{certbin};
	my @babble = BABBLE ? ( join '', ';', $self->babble, "\n" ) : ();
	my @cert = split /(\S{64})/, $self->cert;
	my @rdata = ( $self->usage, $self->selector, $self->matchingtype, @cert, @babble );
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	$self->usage(shift);
	$self->selector(shift);
	$self->matchingtype(shift);
	$self->cert(@_);
}


sub usage {
	my $self = shift;

	$self->{usage} = 0 + shift if scalar @_;
	$self->{usage} || 0;
}


sub selector {
	my $self = shift;

	$self->{selector} = 0 + shift if scalar @_;
	$self->{selector} || 0;
}


sub matchingtype {
	my $self = shift;

	$self->{matchingtype} = 0 + shift if scalar @_;
	$self->{matchingtype} || 0;
}


sub cert {
	my $self = shift;

	$self->certbin( pack "H*", map { die "!hex!" if m/[^0-9A-Fa-f]/; $_ } join "", @_ ) if scalar @_;
	unpack "H*", $self->certbin() if defined wantarray;
}


sub certbin {
	my $self = shift;

	$self->{certbin} = shift if scalar @_;
	$self->{certbin} || "";
}


sub certificate { &cert; }


sub babble {
	return BABBLE ? Digest::BubbleBabble::bubblebabble( Digest => shift->certbin ) : '';
}


1;
__END__


=head1 SYNOPSIS

    use Net::DNS;
    $rr = new Net::DNS::RR('name SMIMEA usage selector matchingtype certificate');

=head1 DESCRIPTION

The SMIMEA DNS resource record (RR) is used to associate an end
entity certificate or public key with the associated email address,
thus forming a "SMIMEA certificate association".
The semantics of how the SMIMEA RR is interpreted are described in
RFC6698.

=head1 METHODS

The available methods are those inherited from the base class augmented
by the type-specific methods defined in this package.

Use of undocumented package features or direct access to internal data
structures is discouraged and could result in program termination or
other unpredictable behaviour.


=head2 usage

    $usage = $rr->usage;
    $rr->usage( $usage );

8-bit integer value which specifies the provided association that
will be used to match the certificate.

=head2 selector

    $selector = $rr->selector;
    $rr->selector( $selector );

8-bit integer value which specifies which part of the certificate
presented by the server will be matched against the association data.

=head2 matchingtype

    $matchingtype = $rr->matchingtype;
    $rr->matchingtype( $matchingtype );

8-bit integer value which specifies how the certificate association
is presented.

=head2 certificate

=head2 cert

    $cert = $rr->cert;
    $rr->cert( $cert );

Hexadecimal representation of the certificate data.

=head2 certbin

    $certbin = $rr->certbin;
    $rr->certbin( $certbin );

Binary representation of the certificate data.

=head2 babble

    print $rr->babble;

The babble() method returns the 'BubbleBabble' representation of the
digest if the Digest::BubbleBabble package is available, otherwise
an empty string is returned.

BubbleBabble represents a message digest as a string of plausible
words, to make the digest easier to verify.  The "words" are not
necessarily real words, but they look more like words than a string
of hex characters.

The 'BubbleBabble' string is appended as a comment to the RDATA when
the string method is called.


=head1 COPYRIGHT

Copyright (c)2016 Dick Franks.

All rights reserved.

Package template (c)2009,2012 O.M.Kolkman and R.W.Franks.


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<perl>, L<Net::DNS>, L<Net::DNS::RR>, draft-ietf-dane-smimea,
RFC6698

=cut
