package App::bif::show::topic;
use strict;
use warnings;
use Bif::Mo;
use DBIx::ThinSQL qw/bv sum case coalesce concat qv sq/;

our $VERSION = '0.1.5_7';
extends 'App::bif::show';

sub run {
    my $self = shift;
    my $opts = $self->opts;
    my $db   = $self->db;

    $opts->{id} = $self->uuid2id( $opts->{id} );

    my $info = $self->get_node( $opts->{id}, 'topic' );
    my $now = $self->now;
    my @data;

    my $ref = $db->xhashref(
        select => [
            'SUBSTR(n.uuid,1,8) as uuid',
            'i.title',
            'n.ctime',
            'n.ctimetz',
            'n.ctimetzhm AS ctimetzhm',
            "$now - n.ctime AS ctime_age",
            'n.mtime',
            'n.mtimetz',
            'n.mtimetzhm AS mtimetzhm',
            "$now - n.mtime AS mtime_age",
            'e1.name as creator',
            'e2.name as updator',
        ],
        from       => 'nodes n',
        inner_join => 'changes c1',
        on         => 'c1.id = n.first_change_id',
        inner_join => 'entities e1',
        on         => 'e2.id = c2.identity_id',
        inner_join => 'changes c2',
        on         => 'c2.id = n.last_change_id',
        inner_join => 'entities e2',
        on         => 'e1.id = c1.identity_id',
        inner_join => 'topics i',
        on         => 'i.id = n.id',
        where      => { 'n.id' => $info->{id} },
    );

    push( @data, $self->header( '  ID', $info->{id}, $ref->{uuid} ), );
    push( @data,
        $self->header( '  Created',    $self->ctime_ago($ref) ),
        $self->header( '  Created-By', $ref->{creator} ),
    );
    push( @data,
        $self->header( '  Updated',    $self->mtime_ago($ref) ),
        $self->header( '  Updated-By', $ref->{updator} ),
    ) unless $ref->{mtime} == $ref->{ctime};

    my @refs = $db->xhashrefs(
        select => [
            't.id AS id',
            'p.path',
            'ts.status',
            'c.mtime AS mtime',
            'c.mtimetz AS mtimetz',
            'c.mtimetzhm AS mtimetzhm',
            "$now - c.mtime AS mtime_age",
        ],
        from       => 'topics t2',
        inner_join => 'topics t',
        on         => 't.link_topic_id = t2.link_topic_id',
        inner_join => 'nodes n',
        on         => 'n.id = t.id',
        inner_join => 'nodes p',
        on         => 'p.id = t.project_id',
        inner_join => 'topic_status ts',
        on         => 'ts.id = t.topic_status_id',
        inner_join => 'changes c',
        on         => 'c.id = n.first_change_id',
        where      => { 't2.id' => $info->{id} },
        order_by   => 'p.path',
    );

    my %seen;
    my $count = @refs;
    my $i     = 1;
    foreach my $ref (@refs) {
        my @ago = $self->mtime_ago($ref);

        push(
            @data,
            $self->header(
                '  Status', "$ref->{status} ($ref->{path}/$ref->{id})",
                $ago[1]
            ),
        );
    }

    $self->start_pager;
    print $self->render_table( 'l  l',
        [ ucfirst( $info->{tkind} ), $ref->{title} ],
        \@data, 1 );

    print "\n";
    $self->dispatch( 'App::bif::log::topic', opts => { id => $opts->{id} } )
      if $opts->{log};

    $self->ok( 'Show' . ucfirst( $info->{tkind} ), \@data );
}

1;
__END__

=head1 NAME

=for bif-doc #show

bif-show-topic - display an topic's current status

=head1 VERSION

0.1.5_7 (2015-11-25)

=head1 SYNOPSIS

    bif show topic ID [OPTIONS...]

=head1 DESCRIPTION

The B<bif-show-topic> command displays the characteristics of an topic.

=head1 ARGUMENTS & OPTIONS

=over

=item ID

An topic ID. Required.

=item --full, -f

Display a more verbose version of the current status.

=item --uuid, -U

Lookup the node using ID as a UUID string instead of a node integer.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2014-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

