package App::bif::pull::hub;
use strict;
use warnings;
use AnyEvent;
use Bif::Sync::Client;
use Bif::Mo;
use Coro;
use DBIx::ThinSQL qw/qv/;
use Log::Any '$log';
use Path::Tiny;

our $VERSION = '0.1.5_7';
extends 'App::bif';

sub run {
    my $self = shift;
    my $opts = $self->opts;

    # Do this early so that we complain about missing repo before we
    # complain about a missing hub
    my $dbw = $self->dbw;    # Consider upping PRAGMA cache_size?
                             #Or handle that in Bif::Role::Sync?

    if ( $opts->{location} =~ m!^(ssh://)?(.*)@(.*)! ) {
        $opts->{name} = $2;
    }
    elsif ( -d $opts->{location} ) {
        $opts->{location} = path( $opts->{location} )->realpath;
        $opts->{name}     = $opts->{location}->basename;
    }
    else {
        return $self->err( 'HubNotFound', 'hub not found: %s',
            $opts->{location} );
    }

    $log->debug("pull hub: $opts->{location}");
    if ( my $hub = $self->dbw->get_hub( $self->uuid2id( $opts->{location} ) ) )
    {
        warn "warning: hub already exists: $hub->{name}\n";
    }

    $|++;    # no buffering
    my $error;
    my $cv = AE::cv;

    my $client = Bif::Sync::Client->new(
        name          => $opts->{name},
        db            => $dbw,
        location      => $opts->{location},
        debug         => $opts->{debug},
        debug_bifsync => $opts->{debug_bifsync},
        on_update     => sub {
            my $client = shift;
            $self->lprint( $client->name . ': ' . $_[0] );
        },
        on_error => sub {
            $error = shift;
            $cv->send;
        },
    );

    my $fh   = select;
    my $coro = async {
        select $fh;

        eval {
            $dbw->txn(
                sub {
                    my ( $status, $ref ) = $client->pull_hub;

                    print "\n";

                    $client->disconnect;

                    if ( $status ne 'RepoImported' ) {
                        $error = $status;
                        $dbw->rollback;
                        return $status;
                    }

                    $dbw->xdo(
                        insert_or_replace_into =>
                          [ 'bifkv', qw/key change_id change_id2/ ],
                        select =>
                          [ qv('last_sync'), 'MAX(c.id)', 'MAX(c.id)', ],
                        from => 'changes c',
                    );

                    return $status;
                }
            );
        };

        if ($@) {
            $error = $@;
            print "\n";
        }

        return $cv->send;
    };

    $cv->recv;
    $client->disconnect;
    return $self->err( 'Unknown', $error ) if $error;
    return $self->ok('PullHub');

}

1;
__END__

=head1 NAME

=for bif-doc #sync

bif-pull-hub -  import project lists from a remote repository

=head1 VERSION

0.1.5_7 (2015-11-25)

=head1 SYNOPSIS

    bif pull hub LOCATION [OPTIONS...]

=head1 DESCRIPTION

The B<bif-pull-hub> command connects to a hub repository to obtain the
list of projects hosted there.  A hub has a name (use the
L<bif-list-hubs> command to display it) which is useable afterwards
with all other hub-aware commands to save typing the full address.

The retrieved project list is stored locally and is used by the
L<bif-pull-project> and L<bif-push-issue> commands, and changed by the
L<bif-sync> command.

=head1 ARGUMENTS & OPTIONS

=over

=item LOCATION

The location of a remote repository.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2013-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

