CREATE TABLE project_changes (
    project_id INTEGER NOT NULL,
    -- Collate below for LIKE/index support
    prefix VARCHAR NOT NULL COLLATE NOCASE CHECK (LENGTH(prefix) > 0),
    change_id INTEGER, -- Leaf nodes only
    -- real_project_id? Consider for per-project changes?
    hash VARCHAR NOT NULL DEFAULT '*not-yet-defined*',
    num_changes INTEGER NOT NULL DEFAULT 1,
    PRIMARY KEY(project_id,prefix),
    FOREIGN KEY(project_id) REFERENCES projects(id) ON DELETE CASCADE,
    FOREIGN KEY(change_id) REFERENCES changes(id) ON DELETE CASCADE
);


-- If current prefix exists then insert one level down
CREATE TRIGGER
    project_changes_bi_1
BEFORE INSERT ON
    project_changes
FOR EACH ROW WHEN
    EXISTS (SELECT
        1
    FROM
        project_changes pc
    WHERE
        pc.project_id = NEW.project_id AND
        pc.prefix = NEW.prefix AND
        pc.change_id IS NULL
    ) AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.project_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Insert next level down
    INSERT INTO
        project_changes(
            project_id,
            prefix,
            change_id,
            num_changes
        )
    SELECT
        NEW.project_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        NEW.change_id,
        1
    FROM
        changes c
    WHERE
        c.id = NEW.change_id 
    ;

    SELECT RAISE(IGNORE);
END;

-- conflict, reparent current node
CREATE TRIGGER
    project_changes_bi_2
BEFORE INSERT ON
    project_changes
FOR EACH ROW WHEN
    EXISTS (SELECT
        1
    FROM
        project_changes pc
    WHERE
        pc.project_id = NEW.project_id AND
        pc.prefix = NEW.prefix AND
        pc.change_id != NEW.change_id AND
        pc.num_changes = 1
    ) AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.project_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Push existing node one level down
    INSERT INTO
        project_changes(
            project_id,
            prefix,
            change_id,
            hash,
            num_changes
        )
    SELECT
        pc.project_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        c.id,
        c.uuid,
        1
    FROM
        project_changes pc
    INNER JOIN
        changes c
    ON
        c.id = pc.change_id
    WHERE
        pc.project_id = NEW.project_id AND
        pc.prefix = NEW.prefix
    ;

    -- Insert this node next level down
    INSERT INTO
        project_changes(
            project_id,
            prefix,
            change_id,
            hash,
            num_changes
        )
    SELECT
        NEW.project_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        NEW.change_id,
        c.uuid,
        1
    FROM
        changes c
    WHERE
        c.id = NEW.change_id 
    ;

    SELECT RAISE(IGNORE);
END;

-- Inserted, now work back up the tree
CREATE TRIGGER
    project_changes_ai_1
AFTER INSERT ON
    project_changes
FOR EACH ROW WHEN
    NEW.change_id IS NOT NULL AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.project_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Special case for prefix length 1
    UPDATE
        project_changes
    SET
        hash = (
            SELECT
                c.uuid
            FROM
                changes c
            WHERE
                c.id = NEW.change_id
        )
    WHERE
        project_id = NEW.project_id AND
        prefix = NEW.prefix
    ;

    UPDATE
        project_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(pc.hash, pc.hash)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(pc.num_changes)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        )
    WHERE
        project_id = NEW.project_id AND
        prefix = SUBSTR(NEW.prefix,1,LENGTH(NEW.prefix) - 1)
    ;
END;

-- intermediate node update, so update the parent
CREATE TRIGGER
    project_changes_au_1
AFTER UPDATE OF
    num_changes
ON
    project_changes
FOR EACH ROW WHEN
    NEW.change_id IS NULL
BEGIN
    SELECT debug(
        NEW.project_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- If no children remove ourself
    DELETE FROM
        project_changes
    WHERE
        NEW.num_changes = 0 AND
        project_id = NEW.project_id AND
        prefix = NEW.prefix
    ;

    -- If one child replace ourself with it
    UPDATE
        project_changes
    SET
        change_id = (
            SELECT
                pc.change_id
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE NEW.prefix || '_'
        ),
        hash = (
            SELECT
                pc.hash
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE NEW.prefix || '_'
        )
    WHERE
        NEW.num_changes = 1 AND OLD.num_changes > 1 AND
        project_id = NEW.project_id AND
        prefix = NEW.prefix
    ;

    DELETE FROM
        project_changes
    WHERE
        NEW.num_changes = 1 AND OLD.num_changes > 1 AND
        project_id = NEW.project_id AND
        prefix LIKE NEW.prefix || '_'
    ;

    -- Update our parent
    UPDATE
        project_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(pc.hash, pc.hash)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(pc.num_changes)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND
                pc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        )
    WHERE
        project_id = NEW.project_id AND
        prefix = SUBSTR(NEW.prefix,1,LENGTH(NEW.prefix) - 1)
    ;

END;


-- Housekeeping - calculate topics.hash
CREATE TRIGGER
    project_changes_au_2
AFTER UPDATE OF
    hash
ON
    project_changes
FOR EACH ROW WHEN
    1 = LENGTH(NEW.prefix)
BEGIN
    UPDATE
        projects
    SET
        hash = (
            SELECT
                agg_sha1_hex(pc.hash, pc.hash)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND pc.prefix LIKE '_'
        ),
        num_changes = (
            SELECT
                SUM(pc.num_changes)
            FROM
                project_changes pc
            WHERE
                pc.project_id = NEW.project_id AND pc.prefix LIKE '_'
        )
    WHERE
        id = NEW.project_id
    ;
END;


CREATE TRIGGER
    project_changes_ad_1
AFTER DELETE ON
    project_changes
FOR EACH ROW WHEN
    OLD.change_id IS NOT NULL
BEGIN
    SELECT debug(
        OLD.project_id,
        OLD.prefix,
        OLD.change_id,
        OLD.hash,
        OLD.num_changes
    );

    UPDATE
        project_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(pc.hash, pc.hash)
            FROM
                project_changes pc
            WHERE
                pc.project_id = OLD.project_id AND
                pc.prefix LIKE
                    SUBSTR(OLD.prefix, 1, LENGTH(OLD.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(pc.num_changes)
            FROM
                project_changes pc
            WHERE
                pc.project_id = OLD.project_id AND
                pc.prefix LIKE
                    SUBSTR(OLD.prefix, 1, LENGTH(OLD.prefix) - 1) || '_'
        )
    WHERE
        change_id IS NULL AND -- Don't update when this is a move
        project_id = OLD.project_id AND
        prefix = SUBSTR(OLD.prefix,1,LENGTH(OLD.prefix) - 1)
    ;

END;
