#!/usr/bin/perl -w

use Carp;

=head1 NAME

Neural_groups.c.PL -- generation of GANN group types

=head1 DESCRIPTION

This simple script takes as its input a special description
of groups (in the distribution, the file C<grouptypes>)
and outputs xsub code that implements the group types described.
The idea is to eliminate as much common code between the groups
as possible and make it easy to change the framework where
the groups are actually used.

Also, this makes it easy to define groups with different
perceptive fields for different neurons without spending
too much effort on each type of group. For example,
the relatively complex XYConvolution1 group requires only 60 lines
of code in this format instead of the over 100 lines of code
and 30 lines of declarations in several different files without
the compiler. Also, barring changes to the group compiler,
the actual perl bindings of the groups can be changed without
affecting the group code.

The group type description file is actually an executable
perl script that calls the functions provided by 
C<Neural_groups.c.PL> to generate different groups.

The groups are declared by calling the C<GANN_DECLARE_GROUP>
function. This function takes two arguments, first the name
of the group and then a hash containing some parameters
(hereafter called group parameters).
The exact contents of this hash are in a flux. 
Currently,
they are

=over 8

=item n_igroups

If defined, the number of expected input groups / sections.
0 = same as not defined, zero inputs. >0  = exactly this number of groups.
-1 = any number of groups larger than zero.

=item n_ilattices

If defined, this group will also get this many lattices as input.

=item n_neurons

Similar to n_igroups but for number of neurons.
Alternately, this can be set to the string value "LATTICE3D"
which means that the user should specify the x, y and z dimensions
for a lattice. The value of the NNEURONS C macro will work
also in this case.

=item activation

If defined as nonzero, this group has an activation function.

=item datas

A hash containing zero or more group data elements (for example
number of neurons ...) NOT YET IMPLEMENTED

=item nweights

A code snippet for calculating the number of neurons.
The result should be stored in C<gann_nwts>.

=item gradient

What type of groups this generally is. The string value BACKPROP
is all that is currently implemented. With BACKPROP,
you must define the parameter C<backprop_activation_loop>.

=item backprop_activation_loop

This is a code snippet which is actually executed
twice in different contexts: it should contain
for each neuron in the group three commands,
INIT_BACKPROP, STD_BACKPROP(wtind), 
PAUSE_BACKPROP, CONTINUE_BACKPROP and FINISH_BACKPROP.
See the C<grouptypes> file for examples.


=back

As noted, some of these parameters contain C code.
In order to avoid clashes in variable names, all names beginning
with C<gann_> are considered reserved. 
The index I<i> is defined at the beginning of each function
and is not used by the compiler itself.
The macros defined by the
group compiler are

=over 8

=item UOUT(i)

the I<i>th unit output vector element.

=item UIN(i)

the I<i>th unit inputs element.

=item NINGROUPS, NNINGROUP(i), IGROUPELEM(i,e)

These are defined only if C<n_igroups> group parameter
is defined and not equal to zero. NINGROUPS is the number of input groups,
NNINGROUP(i) is the number of neurons in the input group I<i> and
IGROUPELEM(i,e) is the I<e>th element in the input group I<i>.

=item NNEURONS

The number of neurons. (taken from C<n_neurons> group parameter
which is evaluated at group creation time)

=back

=head1 BUGS

The current implementation is inefficient: by forcing the
use of IGROUPELEM vectorization and compiler optimization is hindered.
I have not yet discovered a good fix for all this.

The whole thing should be modularized: the lattices as currently
implemented are ugly hacks. An object oriented approach would
be preferable.

=cut

use FileHandle;

$VERSION=0.01;

if ($#_ == -1) {$fn = "BasicGroups"; $sn = "grouptypes";}
else {
	$#_ == 1 or die "Need 0 or 2 args\n";
	$fn = $_[0]; $sn = $_[1];
}

($c_file = new FileHandle ">$fn.c") or
		die "Couldn't open $fn.c for output\n";

($h_file = new FileHandle ">$fn.h") or
		die "Couldn't open $fn.h for output\n";

($t_file = new FileHandle ">$fn.typ") or
		die "Couldn't open $fn.typ for output\n";

($x_file = new FileHandle ">xs_$fn.xs") or
		die "Couldn't open xs_$fn.xs for output\n";

($p_file = new FileHandle ">$fn.pm") or
		die "Couldn't open $fn.pm for output\n";


$generated = <<"FOO";
/* Generated using grouptypecompiler v$VERSION */
/* DO NOT EDIT THIS FILE! CHANGES WILL BE LOST! */
/* Generated from the file $sn */
FOO

print $c_file $generated;
print $h_file $generated;
print $x_file $generated;

print $p_file <<"FOO";
# Perl module for gann_groups
# Automatically generated by grouptypecompiler v$VERSION 
# DO NOT EDIT THIS FILE! CHANGES WILL BE LOST! 
# Generated from the file $sn
FOO


print $c_file <<FOO;
extern "C" {
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
}
#include "DVector.h"
#include "gann_gen.h"
#include "$fn.h"

#define UOUT(i) (UnitOutputs->elem(m_ostart + (i)))
#define NINGROUPS (m_nins)
#define NNINGROUP(which) (m_inouts.i[which])
#define NNEURONS (m_noutputs)
// for unstandard C++ compilers:
#define FOR_MY_NEURONS for(gann_neuron=0; gann_neuron<m_noutputs; gann_neuron++)
#define FOR_ALL_INPUT_NEURONS \\
	for(gann_igroup=0; gann_igroup < m_nins; gann_igroup++) \\
	 for(gann_ineur=0; gann_ineur < m_inouts.i[gann_igroup]; gann_ineur++)
#define THIS_INPUT_NEURON (m_iouts.i[gann_igroup]+gann_ineur)
#define FOR_ALL_INPUT_GROUPS \\
	for(gann_igroup=0; gann_igroup < m_nins; gann_igroup++)
#define THIS_INPUT_GROUP \\
	gann_igroup
#define ILATTICE_Z(x) (m_ilattice->z())
#define ILATTICE_INSIDE(ax,ay,az) (m_ilattice->inside(ax,ay,az))
#define ILATTICE_NEURON(ax,ay,az) ((*m_ilattice)(ax,ay,az))

#define THIS_NEURON (m_ostart + gann_neuron)

/* An ugly hack to get our gann_neuron assignment done at the beginning
 of each loop */
#define FOR_MY_LATTICE \\
	for(gann_myx=0; gann_myx < m_noutx; gann_myx ++)  \\
	 for(gann_myy=0; gann_myy < m_nouty; gann_myy ++)  \\
	  for(gann_myz=0; gann_myz < m_noutz; gann_myz ++)  \\
	   if (  (gann_neuron = (*m_outlatt)(gann_myx,gann_myy,gann_myz) \\
	   		- m_ostart) \\
	   	&& 0 ) {;} else 
#define MYLATTICE_X gann_myx
#define MYLATTICE_Y gann_myy
#define MYLATTICE_ZSIZE (m_outlatt->z())

FOO

print $x_file <<FOO;
extern "C" {
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
}
#include "DVector.h"
#include "gann_gen.h"
#include "$fn.h"

FOO

print $x_file `cat gann_gen.xsprefix`;

$t_typemap_lines =  "";
$t_output_lines =  "";
$t_input_lines =  "";

sub GANN_DECLARE_GROUP
{
	my ($name,$args) = @_;
	print "D $name \n";
	my $gname = "GANN_$name"."Group";
	my $pname = "Neural::$name"."Group";
	my $activation = delete $args->{activation};
	my $n_igroups = delete $args->{n_igroups};
	my $n_neurons = delete $args->{n_neurons};
	my $n_ilattices = delete $args->{n_ilattices};
	my $datas = delete $args->{datas};
	my $nweights = delete $args->{nweights};
	my $gradient = delete $args->{gradient};
	my $backprop_activation_loop = delete $args->{backprop_activation_loop};
	my $activate = delete $args->{activate};
	my $backactivate = delete $args->{backactivate};
	my @k = keys %$args;
	$#k == -1 or die "Elements @k left in parameters\n";

#
# 1. CONSTRUCT A PARAMETER LIST FOR THE CONSTRUCTOR
# 
# Parlist :
# 1. name   2. Type  3. (optional) xsubtype.
# xsubtype  = $p[-1] and type = $p[1] so don't add anything
# without a thought about this
	my @parlist = (
	);
# Constructor body
	$body = "";
# Destructor body
	$dbody = "";
# Data members
	$datamems = "";
	my $stdvars="\tint i;int gann_igroup; int gann_ineur; int gann_neuron;\n";

	if($activation) {
		push @parlist, (
			["act","int","Activation"]
		);
	}

	if($n_igroups) {
		push @parlist, (
			["nins","int","N_InputGroups"],
			["inouts","GANN_iarr","N_InputGNeurons"],
			["iouts","GANN_iarr","InputGOutStarts"],
		);
		$body .= "
/* Check that we have the same numbers of input groups */
			if(m_inouts.n != nins || m_iouts.n != nins) 
				croak(\"Group $gname constructor: unequal nins (%d,%d,%d)\\n\",
					nins,m_inouts.n,m_iouts.n);
";
		$dbody .= "
			delete[] m_inouts.i; delete[] m_iouts.i;";
	}
	if($n_ilattices && $n_ilattices != 1) {die "ilattices must be 1";}
	if($n_ilattices) {
		push @parlist, (
			["ilattice","GANN_Lattice *","InputLattice"],
		);
	}
	if( $n_neurons eq "LATTICE3D" ) {
		push @parlist, (
			["noutx","int","Output_x"],
			["nouty","int","Output_y"],
			["noutz","int","Output_z"],
		);
		$datamems .= "
			GANN_Lattice *m_outlatt;
";
		$body .= "
/* Take care of the output lattice */
			m_outlatt = new GANN_Lattice(noutx,nouty,noutz,*ostart);
			m_noutputs = noutx * nouty * noutz;";
		$dbody .= "
/* Take care of the output lattice */
			delete m_outlatt;";
		$stdvars .= "\tint gann_myx,gann_myy,gann_myz;\n";

	} elsif($n_neurons < 0) {
		push @parlist, (
			["nneurons","int","N_Neurons"]
		);
		$body .= "	m_noutputs = m_nneurons;\n";
	} else {
		$body .= " 	m_noutputs = $n_neurons;\n";
	}

	if($nweights =~ /^[0-9]+$/) {
		$body .= "	m_nweights = $nweights;\n";
	} else {
		$body .= "	int gann_nwts;\n";
		$body .= $nweights;
		$body .= "\n	m_nweights = gann_nwts;\n";
	}
#
#  Add datas
# 
# XX Conditions not used

	push @parlist, (map {[$_->[0],$_->[1],$_->[4]]} @$datas);

	$plist = join '',(map {",".$_->[1]." ".$_->[0]} @parlist);

#
# 2. Create the group declaration
#

print $h_file "
class $gname: public GANN_Group {
public:
	$gname(int *wstart, int *ostart", $plist ,");
	~$gname();
	virtual int activate(GANN_Vector *UnitInputs,
		GANN_Vector *UnitOutputs,
		GANN_Vector *Weights);
	virtual int backactivate(GANN_Vector *UnitInputs,
		GANN_Vector *UnitOutputs,
		GANN_Vector *DUnitOutputs,
		GANN_Vector *Weights,
		GANN_Vector *DWeights);
protected:
	",(join "",(map {$_->[1]." m_".$_->[0].";\n\t"} @parlist)),"
	$datamems
};";

#
# 3. Create the group constructor and destructor
#

print $c_file "
${gname}::$gname(int *wstart, int *ostart",$plist, ") :
	GANN_Group(*wstart,*ostart)
{
	$stdvars
/* Grab in parameters */
	",(join "",(map {"m_".$_->[0]." = ".$_->[0].";\n\t"} @parlist)),"
	$body
	*wstart += m_nweights;
	*ostart += m_noutputs;
}

${gname}::~$gname() 
{
	$dbody
}
";
	
#
# 4. Create the group activation and backactivation functions.
#

	if(!defined $activate) {
		if($gradient ne "BACKPROP") {
			croak "Only backprop activate can be defined\n";
		}
		$activate = "
#define INIT_BACKPROP double sum=0;
#define STD_BACKPROP(neur,wt) (sum += Weights->elem(m_wstart+(wt)) * \\
				UnitOutputs->elem(neur))
#define STD_BACKPROP_CONST(value,wt) (sum += Weights->elem(m_wstart+(wt)) * (value))
#define PAUSE_BACKPROP ((UnitInputs->elem(THIS_NEURON) = sum))
#define CONTINUE_BACKPROP double sum = UnitInputs->elem(THIS_NEURON);
#define FINISH_BACKPROP (PAUSE_BACKPROP, \\
	(UnitOutputs->elem(THIS_NEURON) = GANN_Activations[m_act].act(sum)))
$backprop_activation_loop
#undef INIT_BACKPROP
#undef STD_BACKPROP
#undef STD_BACKPROP_CONST
#undef PAUSE_BACKPROP
#undef CONTINUE_BACKPROP
#undef FINISH_BACKPROP
";
	}

	if (!defined $backactivate) {
		if($gradient ne "BACKPROP") {
			croak "Only backprop backactivate can be defined\n";
		}
		$backactivate = "
#define INIT_BACKPROP  \\
		ideriv = GANN_Activations[m_act].      \\
			dact(UnitInputs->elem(THIS_NEURON)) *  \\
			DUnitOutputs->elem(THIS_NEURON);
#define STD_BACKPROP(neur,wt) ( (DWeights->elem(m_wstart+(wt)) += \\
					UnitOutputs->elem(neur) * ideriv), \\
				(DUnitOutputs->elem(neur) += \\
					Weights->elem(m_wstart+(wt)) * ideriv) \\
			      )
#define STD_BACKPROP_CONST(value,wt) (DWeights->elem(m_wstart+(wt)) += \\
					(value) * ideriv)
#define PAUSE_BACKPROP 
#define CONTINUE_BACKPROP INIT_BACKPROP
#define FINISH_BACKPROP 
$backprop_activation_loop
#undef INIT_BACKPROP
#undef STD_BACKPROP
#undef STD_BACKPROP_CONST
#undef PAUSE_BACKPROP
#undef CONTINUE_BACKPROP
#undef FINISH_BACKPROP
";
	}

print $c_file "
int ${gname}::activate(GANN_Vector *UnitInputs,
	GANN_Vector *UnitOutputs,
	GANN_Vector *Weights)
{
	$stdvars
	$activate
	return 0;
}

int ${gname}::backactivate(GANN_Vector *UnitInputs,
	GANN_Vector *UnitOutputs,
	GANN_Vector *DUnitOutputs,
	GANN_Vector *Weights,
	GANN_Vector *DWeights)
{
	double ideriv;
	$stdvars
	$backactivate
	return 0;
}

";

#
# 	TYPEMAP ENTRY
#
	$t_typemap_lines .= "$gname *		O_$gname \n";
	$t_output_lines .= 
"O_$gname
	sv_setref_pv( \$arg, \"${pname}\", (void*)\$var );
";

	$t_input_lines .= 
"O_$gname".'
	if( sv_isobject($arg) && (SvTYPE(SvRV($arg)) == SVt_PVMG) )
	$var = ($type)SvIV((SV*)SvRV( $arg ));
	else{
	warn( \"${Package}::$func_name() -- $var is not a blessed SV reference\" );
	XSRETURN_UNDEF;
	}
';
#
# 	XSUB ENTRIES
#
	my $pnamelist = join '',map {",".$_->[0]} @parlist;
	my $pdefs = join '',map {"\t".$_->[1]." ".$_->[0]."\n"} @parlist;
	print $x_file <<"FOO";

MODULE = Neural::Basic PACKAGE = $pname

$gname *
${gname}::new(wstart,ostart $pnamelist)
	int &wstart
	int &ostart
$pdefs
	OUTPUT:
	RETVAL
	wstart
	ostart

void
${gname}::DESTROY()

int
${gname}::get_noutputs()

int
${gname}::get_ostart()

int
${gname}::activate(unitin,unitout,wei)
	GANN_Vector *unitin
	GANN_Vector *unitout
	GANN_Vector *wei;

double
${gname}::backactivate(unitin,unitout,wei,dwei,dunitout)
	GANN_Vector *unitin
	GANN_Vector *unitout
	GANN_Vector *wei
	GANN_Vector *dwei
	GANN_Vector *dunitout
FOO

#
#	Pm: argument parsing and checking
#

	print $p_file "
package $pname;
use Carp;

sub New {
	my (\$class,\$wstart,\$ostart,\%args) = \@_;
	my \@arr=();
	for ((",(join ',',map {"\"".$_->[2]."\""} @parlist),")) {
		if (!defined \$args{\$_}) {
			carp(\"Parameter \$_ not defined!\n\");
		}
		push \@arr, \$args{\$_};
	}
	my \$ob = new \$class \$wstart,\$ostart,\@arr;
	\$_[1]=\$wstart; \$_[2]=\$ostart;
	return \$ob;
}
";

}


die "PROBLEM EXECING FILE" unless do $sn;

print $t_file <<"FOO";
TYPEMAP
$t_typemap_lines

OUTPUT
$t_output_lines

INPUT
$t_input_lines
FOO

print $p_file <<"FOO";
package Neural::BasicGroups;
1; # To make the loading have a happy ending.
FOO
