#!perl
use warnings;
use strict;

=head1 Synopsis

Packages for Testing Stringification Overloading
(supporting library for IPC::Run3::Shell tests)

=head1 Author, Copyright, and License

Copyright (c) 2015 Hauke Daempfling (haukex@zero-g.net).

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl 5 itself.

For more information see the L<Perl Artistic License|perlartistic>,
which should have been distributed with your copy of Perl.
Try the command "C<perldoc perlartistic>" or see
L<http://perldoc.perl.org/perlartistic.html>.

=cut

## no critic (ProhibitMultiplePackages, RequireArgUnpacking)

# Tests for these classes are currently in 16_overload.t

{
	# FakeStringFallback: Overloads stringification, with full fallback enabled
	# so it should numify, add, etc. and mostly act like a normal string
	package FakeStringFallback;
	use overload '""'=>\&str, fallback=>1;
	sub new { return bless \"$_[1]9", $_[0] }
	sub str { return "1".${shift()} }
}

{
	# FakeStringDefault: Overloads stringification, with magic autogeneration (no fallback)
	# *only* numification is autogenerated, so eq, ==, etc. and even addition will *not* work
	package FakeStringDefault;
	use overload '""'=>\&str, fallback=>undef;
	sub new { return bless \"$_[1]8", $_[0] }
	sub str { return "2".${shift()} }
}

{
	# FakeStringOnly: Overloads stringification *only*, *no* autogeneration
	# this should mean that *only* stringification and *nothing* else works
	package FakeStringOnly;
	use overload '""'=>\&str, fallback=>0;
	sub new { return bless \"$_[1]7", $_[0] }
	sub str { return "3".${shift()} }
}

{
	# FakeNumberFallback: Overloads numification, with full fallback enabled
	# so it should stringify, add, etc. and mostly act like a normal number
	package FakeNumberFallback;
	use overload '0+'=>\&num, fallback=>1;
	sub new { return bless \($_[1]+1), $_[0] }
	sub num { return ${shift()}+1 }
}

{
	# FakeNumberDefault: Overloads numification, with magic autogeneration (no fallback)
	# *only* stringification is autogenerated, so eq, ==, etc. and even addition will *not* work
	package FakeNumberDefault;
	use overload '0+'=>\&num, fallback=>undef;
	sub new { return bless \($_[1]+2), $_[0] }
	sub num { return ${shift()}+2 }
}

{
	# FakeNumberOnly: Overloads numification *only*, *no* autogeneration
	# this should mean that *only* numification and *nothing* else works, not even addition
	# (I'm not really sure if this is useful at all?)
	package FakeNumberOnly;
	use overload '0+'=>\&num, fallback=>0;
	sub new { return bless \($_[1]+3), $_[0] }
	sub num { return ${shift()}+3 }
}

{
	# NotStrOrNumish: Uses overloading, but not numification or stringification
	# stringification or numification can't be autogenerated from addition
	package NotStrOrNumish;
	use overload '+'=>\&add, fallback=>undef;
	sub new { return bless \($_[1]+0), $_[0] }
	sub add { return ${shift()}+shift()-1 }
}

{
	# NotOverloaded: no overloading at all
	package NotOverloaded;
	sub new { return bless \("$_[1]"), $_[0] }
}

{
	# DiesOnStringify: Just for testing, dies when stringification is attempted.
	package DiesOnStringify;
	use overload '""'=>\&str, fallback=>1;
	use Carp;
	sub new { return bless \"$_[1]0", $_[0] }
	sub str { confess "ARRRGH" }
}
{
	# DiesOnNumify: Just for testing, dies on numification (w/ fallback stringification)
	package DiesOnNumify;
	use overload '0+'=>\&num, fallback=>1;
	use Carp;
	sub new { return bless \($_[1]+1), $_[0] }
	sub num { confess "BLAMMO" }
}


1;
