package App::GitHub;

use strict;
use warnings;

# ABSTRACT: GitHub Command Tools

use Moose;
use Net::GitHub;
use Term::ReadLine;
use JSON::XS;

our $VERSION = '0.02';

has 'term' => (
    is      => 'ro',
    isa     => 'Object',
    default => sub {
        my $term = new Term::ReadLine 'github';

        my $odef = select STDERR;
        $| = 1;
        select STDOUT;
        $| = 1;
        select $odef;

        return $term;
    }
);
has 'prompt' => (
    is      => 'rw',
    isa     => 'Str',
    default => "github> ",
);

has 'github' => (
    is  => 'rw',
    isa => 'Net::GitHub',
);

has '_data' => ( is => 'rw', isa => 'HashRef', default => sub { {} } );

my $dispatch = {
    'exit' => sub { exit; },
    'quit' => sub { exit; },
    'q'    => sub { exit; },
    '?'    => \&help,
    'h'    => \&help,

    # Common
    repo  => \&set_repo,
    login => \&set_login,

    # Repo
    rshow    => \&repo_show,
    rlist    => \&repo_list,
    findrepo => sub {
        my ( $self, $word ) = @_;
        $self->run_github("repos->search('$word')");
    },
    watch       => sub { shift->run_github('repos->watch()'); },
    unwatch     => sub { shift->run_github('repos->unwatch()'); },
    fork        => sub { shift->run_github('repos->fork()'); },
    create      => \&repo_create,
    delete      => \&repo_delete,
    set_private => sub { shift->run_github('repos->set_private()'); },
    set_public  => sub { shift->run_github('repos->set_public()'); },

    # XXX? TODO, deploy_keys collaborators
    network  => sub { shift->run_github('repos->network()'); },
    tags     => sub { shift->run_github('repos->tags()'); },
    branches => sub { shift->run_github('repos->branches()'); },

    # Issues
    ilist => sub {
        my ( $self, $type ) = @_;
        $type ||= 'open';
        $self->run_github("issue->list('$type')");
    },
    iview => sub {
        my ( $self, $number ) = @_;
        $self->run_github("issue->view($number)");
    },
    iopen  => \&issue_open,
    iclose => sub {
        my ( $self, $number ) = @_;
        $self->run_github("issue->close($number)");
    },
    ireopen => sub {
        my ( $self, $number ) = @_;
        $self->run_github("issue->reopen($number)");
    },

    # XXX? TODO, add_label, edit etc

    # File/Path
    cd => sub {
        eval( "chdir " . shift );
        print $@ if $@;
    },
};

sub run {
    my $self = shift;

    print <<START;

Welcome to GitHub Command Tools! (Ver: $VERSION)
Type '?' or 'h' for help.
START

    while ( defined( my $command = $self->term->readline( $self->prompt ) ) ) {

        $command =~ s/(^\s+|\s+$)//g;
        next unless length($command);

        # check dispatch
        if ( exists $dispatch->{$command} ) {
            $dispatch->{$command}->($self);
        }
        else {

            # split the command out
            ( $command, my $args ) = split( /\s+/, $command, 2 );
            if ( $command and exists $dispatch->{$command} ) {
                $dispatch->{$command}->( $self, $args );
            }
            else {
                print "Unknown command, type '?' or 'h' for help\n";
                next unless $command;
            }
        }

        $self->term->addhistory($command) if $command =~ /\S/;
    }
}

sub help {
    print <<HELP;
 command  argument          description
 repo     :user :repo       set owner/repo, eg: 'fayland perl-app-github'
 login    :login :token     authenticated as :login
 ?,h                        help
 q,exit,quit                  exit

Repos
 rshow                      more in-depth information for the :repo in repo
 rlist                      list out all the repositories for the :user in repo
 rsearch  WORD              Search Repositories
 watch                      watch repositories (authentication required)
 unwatch                    unwatch repositories (authentication required)
 fork                       fork a repository (authentication required)
 create                     create a new repository (authentication required)
 delete                     delete a repository (authentication required)
 set_private                set a public repo private (authentication required)
 set_public                 set a private repo public (authentication required)
 network                    see all the forks of the repo
 tags                       tags on the repo
 branches                   list of remote branches

Issues
 ilist    open|closed       see a list of issues for a project
 iview    :number           get data on an individual issue by number
 iopen                      open a new issue (authentication required)
 iclose   :number           close an issue (authentication required)
 ireopen  :number           reopen an issue (authentication required)

File/Path related
 cd       PATH              chdir to PATH

Others
 show     :user :repo       more in-depth information for a repository
 list     :user             list out all the repositories for a user
HELP
}

sub set_repo {
    my ( $self, $repo ) = @_;

    # validate
    unless ( $repo =~ /^([\-\w]+)[\/\\\s]([\-\w]+)$/ ) {
        print "Wrong repo args ($repo), eg fayland/perl-app-github\n";
        return;
    }
    my ( $owner, $name ) = ( $repo =~ /^([\-\w]+)[\/\\\s]([\-\w]+)$/ );
    $self->{_data}->{owner} = $owner;
    $self->{_data}->{repo}  = $name;

    # when call 'login' before 'repo'
    my @logins = ( $self->{_data}->{login} and $self->{_data}->{token} )
      ? (
        login => $self->{_data}->{login},
        token => $self->{_data}->{token}
      )
      : ();

    $self->{github} = Net::GitHub->new(
        owner => $owner,
        repo  => $name,
        @logins,
    );
    $self->{prompt} = "$owner/$name> ";
}

sub set_login {
    my ( $self, $login ) = @_;

    ( $login, my $token ) = split( /\s+/, $login, 2 );
    unless ( $login and $token ) {
        print
"Wrong login args ($login $token), eg fayland 54b5197d7f92f52abc5c7149b313cf51\n";
        return;
    }

    # save for set_repo
    $self->{_data}->{login} = $login;
    $self->{_data}->{token} = $token;

    if ( $self->github ) {
        $self->{github} = Net::GitHub->new(
            owner => $self->{_data}->{owner},
            repo  => $self->{_data}->{repo},
            login => $self->{_data}->{login},
            token => $self->{_data}->{token}
        );
    }
}

sub run_github {
    my ( $self, $command ) = @_;

    unless ( $self->github ) {
        print <<'ERR';
unknown repo. try 'repo :owner/:repo' first
ERR
        return;
    }

    eval(
qq~print JSON::XS->new->utf8->pretty->encode(\$self->github->$command) . "\n"~
    );

    if ($@) {

        # custom error
        if ( $@ =~ /login and token are required/ ) {
            print <<'ERR';
authentication required. try 'login :owner :token' first
ERR
        }
        else {
            print $@;
        }
    }
}

################## Repos
sub repo_show {
    my ( $self, $args ) = @_;
    if ( $args and $args =~ /^([\-\w]+)[\/\\\s]([\-\w]+)$/ ) {
        $self->run_github("repos->show('$1', '$2')");
    }
    else {
        $self->run_github('repos->show()');
    }
}

sub repo_list {
    my ( $self, $args ) = @_;
    if ( $args and $args =~ /^[\w\-]+$/ ) {
        $self->run_github("repos->list('$args')");
    }
    else {
        $self->run_github('repos->list()');
    }
}

sub repo_create {
    my ($self) = @_;

    my %data;
    foreach my $col ( 'name', 'desc', 'homepage' ) {
        my $data = $self->term->readline( ucfirst($col) . ': ' );
        $data{$col} = $data;
    }
    unless ( length( $data{name} ) ) {
        print <<'ERR';
create repo failed. name is required
ERR
        return;
    }

    $self->run_github(
        qq~repos->create( "$data{name}", "$data{desc}", "$data{homepage}", 1 )~
    );
}

sub repo_del {
    my ($self) = @_;

    my $data = $self->term->readline('Are you sure to delete the repo? [YN]? ');
    if ( $data eq 'Y' ) {
        print "Deleting Repos ...\n";
        $self->run_github("repos->delete( { confirm => 1 } )");
    }
}

# Issues
sub issue_open {
    my ($self) = @_;

    my %data;
    foreach my $col ( 'title', 'body' ) {
        my $data = $self->term->readline( ucfirst($col) . ': ' );
        $data{$col} = $data;
    }

    $self->run_github(qq~issue->open( "$data{title}", "$data{body}" )~);
}

1;
__END__

=head1 NAME

App::GitHub - GitHub Command Tools

=head1 VERSION

version 0.02

=head1 SYNOPSIS

    github.pl

=head1 DESCRIPTION

a command line tool wrap L<Net::GitHub>

=head1 AUTHOR

  Fayland Lam <fayland@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2009 by Fayland Lam.

This is free software; you can redistribute it and/or modify it under
the same terms as perl itself.

=head1 ALPHA WARNING

L<App::GitHub> is still in its infancy. lots of TODO

feel free to fork from L<http://github.com/fayland/perl-app-github/tree/master>

=head1 SEE ALSO

L<Net::GitHub>
