=head1 NAME

CGI::WPM::Base - Perl module that defines the API for subclasses, which are
miniature applications called "web page makers", and provides them with a
hierarchical environment that handles details for obtaining program settings,
resolving file system or web site contexts, obtaining user input, and sending new
web pages to the user.

=cut

######################################################################

package CGI::WPM::Base;
require 5.004;

# Copyright (c) 1999-2000, Darren R. Duncan. All rights reserved. This module is
# free software; you can redistribute it and/or modify it under the same terms as
# Perl itself.  However, I do request that this copyright information remain
# attached to the file.  If you modify this module and redistribute a changed
# version then please attach a note listing the modifications.

use strict;
use vars qw($VERSION);
$VERSION = '0.1b';

######################################################################

=head1 DEPENDENCIES

=head2 Perl Version

	5.004

=head2 Standard Modules

	I<none>

=head2 Nonstandard Modules

	CGI::WPM::Content
	CGI::WPM::Globals

=cut

######################################################################

use CGI::WPM::Content;
use CGI::WPM::Globals;

######################################################################
# Names of properties for objects of this class are declared here:
my $KEY_SITE_GLOBALS = 'site_globals';  # hold global site values
my $KEY_PAGE_CONTENT = 'page_content';  # hold return values
my $KEY_PAGE_ROOT_DIR = 'page_root_dir';  # root dir of support files
my $KEY_PAGE_PREFS   = 'page_prefs';    # hold our own settings
my $KEY_IS_ERROR   = 'is_error';    # holds error string, if any

# Keys for items in site global preferences:
my $GKEY_AMEND_MSG = 'amend_msg';  # personalized html appears on error page
my $GKEY_T_MAILME = 't_mailme';  # token to replace with mailme url

# Keys for items in site page preferences:
my $PKEY_BODY_ATTR    = 'body_attr';  # params for page's <BODY> tag
my $PKEY_PAGE_HEADER  = 'page_header'; # content goes above our subclass's
my $PKEY_PAGE_FOOTER  = 'page_footer'; # content goes below our subclass's

# Constant values used in this class go here:

######################################################################

sub new {
	my $class = shift( @_ );
	my $self = {};
	bless( $self, ref($class) || $class );
	$self->initialize( @_ );
	return( $self );
}

######################################################################

sub initialize {
	my $self = shift( @_ );
	%{$self} = ();
	$self->{$KEY_SITE_GLOBALS} = CGI::WPM::Globals->new();
	$self->{$KEY_PAGE_CONTENT} = CGI::WPM::Content->new();
	$self->{$KEY_PAGE_ROOT_DIR} = shift( @_ );
	$self->{$KEY_PAGE_PREFS} = $self->page_prefs( shift( @_ ) );
	$self->_initialize( @_ );
}

# subclass should have their own of these
sub _initialize {
}

######################################################################

sub dispatch_by_user {
	my $self = shift( @_ );
	$self->{$KEY_IS_ERROR} and return( 0 );
	return( $self->_dispatch_by_user( @_ ) );
}

# subclass should have their own of these
sub _dispatch_by_user {
	my $self = shift( @_ );
	my $webpage = $self->{$KEY_PAGE_CONTENT};

	$webpage->title( 'Web Page For Users' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>This web page has been generated by CGI::WPM::Base, which is 
copyright (c) 1999-2000, Darren R. Duncan.  This Perl Class 
is intended to be subclassed before it is used.</P>

<P>You are reading this message because either no subclass is in use 
or that subclass hasn't declared the _dispatch_by_user() method, 
which is required to generate the web pages that normal visitors 
would see.</P>
__endquote
}

######################################################################

sub dispatch_by_admin {
	my $self = shift( @_ );
	$self->{$KEY_IS_ERROR} and return( 0 );
	return( $self->_dispatch_by_admin( @_ ) );
}

# subclass should have their own of these
sub _dispatch_by_admin {
	my $self = shift( @_ );
	my $webpage = $self->{$KEY_PAGE_CONTENT};

	$webpage->title( 'Web Page For Administrators' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>This web page has been generated by CGI::WPM::Base, which is 
copyright (c) 1999-2000, Darren R. Duncan.  This Perl Class 
is intended to be subclassed before it is used.</P>

<P>You are reading this message because either no subclass is in use 
or that subclass hasn't declared the _dispatch_by_admin() method, 
which is required to generate the web pages that site administrators 
would use to administrate site content using their web browsers.</P>
__endquote
}

######################################################################

sub page_pref {
	my $self = shift( @_ );
	my $key = shift( @_ );
	if( defined( my $new_value = shift( @_ ) ) ) {
		$self->{$KEY_PAGE_PREFS}->{$key} = $new_value;
	}
	return( $self->{$KEY_PAGE_PREFS}->{$key} );
}

######################################################################

sub page_prefs {
	my $self = shift( @_ );
	my $globals = $self->{$KEY_SITE_GLOBALS};
	my $new_value = shift( @_ );
	if( ref( $new_value ) eq 'HASH' ) {
		$self->{$KEY_PAGE_PREFS} = {%{$new_value}};
	} elsif( defined( $new_value ) ) {
		my $root = $self->{$KEY_PAGE_ROOT_DIR};
		my $delim = $globals->system_path_delimiter();
		my $filepath = "$root$delim$new_value";
		my $result = $globals->get_hash_from_file( $filepath );
		$self->{$KEY_PAGE_PREFS} = 
			(ref($result) eq 'HASH') ? $result : {};
		$result or $self->_set_to_init_error_page( $filepath );
	}
	return( $self->{$KEY_PAGE_PREFS} );  # returns ref
}

######################################################################

sub get_page_content {   # should be called after "dispatch" methods
	my $self = shift( @_ );
	my $webpage = $self->{$KEY_PAGE_CONTENT};
	my $rh_prefs = $self->{$KEY_PAGE_PREFS};
	$webpage->body_attributes( $rh_prefs->{$PKEY_BODY_ATTR} );
	$webpage->body_prepend( $rh_prefs->{$PKEY_PAGE_HEADER} );
	$webpage->body_append( $rh_prefs->{$PKEY_PAGE_FOOTER} );
	return( $webpage );  # returns ref to original
}

######################################################################

sub get_page_string {   # should be called after "dispatch" methods
	my $self = shift( @_ );
	return( $self->get_page_content()->to_string() );
}

######################################################################

sub is_error {
	my $self = shift( @_ );
	return( $self->{$KEY_IS_ERROR} );
}

######################################################################

sub _set_to_init_error_page {
	my $self = shift( @_ );
	$self->_make_filesystem_error( shift( @_ ), "execute" );
	my $webpage = $self->{$KEY_PAGE_CONTENT};

	$webpage->title( 'Error Initializing Page Maker' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>I'm sorry, but an error has occurred while trying to initialize 
a required program module, "@{[ref( $self )]}".  The file that 
contains its preferences couldn't be opened.</P>

@{[$self->_get_amendment_message()]}

<P>Details: $self->{$KEY_IS_ERROR}</P>
__endquote
}

######################################################################

sub _get_amendment_message {
	my $self = shift( @_ );
	my $globals = $self->{$KEY_SITE_GLOBALS};
	my $ams = $globals->site_pref( $GKEY_AMEND_MSG ) || <<__endquote;
<P>This should be temporary, the result of a server glitch
or a site update being performed at the moment.  Click 
<A HREF="@{[$globals->self_url()]}">here</A> to automatically try again.  
If the problem persists, please try again later, or send an
<A HREF="@{[$globals->site_pref( $GKEY_T_MAILME )]}">e-mail</A>
message about the problem, so it can be fixed.</P>
__endquote
	return( $ams );
}

######################################################################

sub _prepend_path {
	my ($self, $filename) = @_;
	my $dir = $self->{$KEY_PAGE_ROOT_DIR};
	my $delim = $self->{$KEY_SITE_GLOBALS}->system_path_delimiter();
	return( "$dir$delim$filename" );
}

######################################################################

sub _make_filesystem_error {
	my ($self, $filename, $unique_part) = @_;
	return( $self->{$KEY_IS_ERROR} = 
		"can't $unique_part file '$filename': $!" );
}

######################################################################

1;
__END__

=head1 AUTHOR

Copyright (c) 1999-2000, Darren R. Duncan. All rights reserved. This module is
free software; you can redistribute it and/or modify it under the same terms as
Perl itself.  However, I do request that this copyright information remain
attached to the file.  If you modify this module and redistribute a changed
version then please attach a note listing the modifications.

I am always interested in knowing how my work helps others, so if you put this
module to use in any of your own code then please send me the URL.  Also, if you
make modifications to the module because it doesn't work the way you need, please
send me a copy so that I can roll desirable changes into the main release.

Address comments, suggestions, and bug reports to B<perl@DarrenDuncan.net>.

=head1 SEE ALSO

perl(1).

=cut
