# NAME

CXC::Exporter::Util - Tagged Based Exporting

# VERSION

version 0.02

# SYNOPSIS

In the exporting code:

    package My::Exporter;
    use CXC::Exporter::Util ':all';

    use parent 'Exporter' # or Exporter::Tiny

    # install sets of constants, with automatically generated
    # enumerating functions
    install_CONSTANTS( {
          DETECTORS => {
              ACIS => 'ACIS',
              HRC  => 'HRC',
          },

          AGGREGATES => {
              ALL  => 'all',
              NONE => 'none',
              ANY  => 'any',
          },
      } );

    # install some functions
    install_EXPORTS(
              { fruit => [ 'tomato', 'apple' ],
                nut   => [ 'almond', 'walnut' ],
              } );

In importing code:

    # import all of the fruit functions, all of the DETECTORS constants,
    # as well as a function enumerating the DETECTORS constants
    use My::Exporter ':fruit', ':detector', 'DETECTORS';

    # print the DETECTORS constants' values;
    say $_ for DETECTORS;

# DESCRIPTION

`CXC::Exporter::Util` provides _tag-centric_ utilities for modules
which export symbols.  It doesn't provide exporting services; its sole
purpose is to manipulate the data structures used by exporting modules
which follow the API provided by Perl's core [Exporter](https://metacpan.org/pod/Exporter) module
(e.g. [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny)).

In particular, it treats `%EXPORT_TAGS` as the definitive source for
information about exportable symbols and uses it to generate
`@EXPORT_OK` and `@EXPORT`.  Consolidation of symbol information in
one place avoids errors of omission.

## Exporting Symbols

At it simplest, the exporting module calls ["install\_EXPORTS"](#install_exports) with a
hash specifying tags and their symbols sets, e.g.,

    package My::Exporter;
    use CXC::Exporter::Util;

    use parent 'Exporter'; # or your favorite compatible exporter

    install_EXPORTS(
              { fruit => [ 'tomato', 'apple' ],
                nut   => [ 'almond', 'walnut' ],
              } );

    sub tomato {...}
    sub apple  {...}
    sub almond {...}
    sub walnut {...}

An importing module could use this via

    use My::ExportingModule ':fruit'; # import tomato, apple
    use My::ExportingModule ':nut';   # import almond, walnut
    use My::ExportingModule ':all';   # import tomato, apple,
                                      #        almond, walnut,

For more complicated setups, `%EXPORT_TAGS` may be specified first:

    package My::ExportingModule;
    use CXC::Exporter::Util;

    use parent 'Exporter';
    our %EXPORT_TAGS = ( tag => [ 'Symbol1', 'Symbol2' ] );
    install_EXPORTS;

`install_EXPORTS` may be called multiple times

## Exporting Constants

[CXC::Exporter::Util](https://metacpan.org/pod/CXC%3A%3AExporter%3A%3AUtil) provides additional support for creating,
organizing and installing constants via ["install\_CONSTANTS"](#install_constants).
Constants are created via Perl's [constant](https://metacpan.org/pod/constant) pragma.

["install\_CONSTANTS"](#install_constants) is passed one or more hashes containing sets of
constants grouped by tags, e.g.:

    install_CONSTANTS( {
          DETECTORS => {
              ACIS => 'ACIS',
              HRC  => 'HRC',
          },

          AGGREGATES => {
              ALL  => 'all',
              NONE => 'none',
              ANY  => 'any',
          },
     });

     # A call to install_EXPORTS (with or without arguments) must follow
     # install_CONSTANTS;
     install_EXPORTS;

For each set an enumerating function is created which returns the set's values.

In the above example, constant functions `ACIS`, `HRC`, `ALL`,
`NONE`, `ANY`, tags `detectors` and `aggregates`, and constant
functions `DETECTORS` and `AGGREGATES` are created.

The enumerating functions are useful for generating enumerated types
via e.g. [Type::Tiny](https://metacpan.org/pod/Type%3A%3ATiny):

    Enum[ DETECTORS ]

or iterating:

    say $_ for DETECTORS;

If the constants are used later in the module for other purposes, constant definition
should be done in a [BEGIN](https://metacpan.org/pod/BEGIN) block:

    our %CONSTANTS;
    BEGIN { %CONSTANTS = (...);
            install_CONSTANTS( \%CONSTANTS );
    }

`install_CONSTANTS` may be called multiple times.

For more complex situations, the lower level ["install\_constant\_tag"](#install_constant_tag)
and ["install\_constant\_func"](#install_constant_func) routines may be useful.

# SUBROUTINES

## install\_EXPORTS

    install_EXPORTS( [\%export_tags], [$package], [\%options]  );

Populate `$package`'s `@EXPORT` and `@EXPORT_OK` arrays based upon
`%EXPORT_TAGS` and `%export_tags`.

If not specified,  `$package` defaults to the caller's package.

Available Options:

- overwrite => \[Boolean\]

    If the `overwrite` option is true, the contents of `%export_tags`
    will overwrite `%EXPORT_TAGS` in `$package`, otherwise
    `%export_tags` is merged into `%EXPORT_TAGS`.

    Note that overwriting will wipe out tags and symbols installed by a
    previous call to ["install\_CONSTANTS"](#install_constants).

    This defaults to false.

- package => \[Package Name\]

    This provides another means of indicating which package to install into.
    Setting this overrides the optional `$package` argument.

- all => \[Boolean | 'auto' \]

    This determines whether ["install\_EXPORTS"](#install_exports) creates an `all` tag
    based on the contents of `%EXPORT_TAGS` in `$package`.  Some exporters, such as
    [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny) and [Sub::Exporter](https://metacpan.org/pod/Sub%3A%3AExporter) automatically handle the `all`
    tag, but Perl's default [Exporter](https://metacpan.org/pod/Exporter) does not.

    If set to `auto` (the default), it will install the `all` tag if
    `$package` is _not_ a subclass of [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny).

    (At present I don't know how to determine if [Sub::Exporter](https://metacpan.org/pod/Sub%3A%3AExporter) is used).

This routine does the following in `$package` based upon
`%EXPORT_TAGS` in `$package`:

- Install the symbols specified via the `$EXPORT_TAGS{default}` tag into `@EXPORT`.
- Install all of the symbols in `%EXPORT_TAGS` into `@EXPORT_OK`.

## install\_CONSTANTS

    install_CONSTANTS( @specs, ?$package  );

Create sets of constants and make them available for export in
`$package`.

If not specified,  `$package` defaults to the caller's package.

The passed `@specs` arguments are either hashrefs or arrayrefs and
contain one or more constant set specifications.  A constant set
specification consists of a unique identifier and a list of name-value
pairs, specified either has a hash or an array.  For example,

    @spec = ( { $id1 => \%set1, $id2 => \@set2 },
              [ $id3 => \%set3, $id4 => \@set4 ],
            );

The `$id?` identifiers are used to create a export tag for the set as
well as an enumerating function which returns the set's values.

The individual `$id`, `$set` pairs are passed to [install\_constant\_tag](https://metacpan.org/pod/install_constant_tag);
see that function for more information.

["install\_CONSTANTS"](#install_constants) may be called multiple times, and may be used
to updated existing sets of constants.

Note that a call to [install\_EXPORTS](https://metacpan.org/pod/install_EXPORTS) _must_ be made after the last
call to `install_CONSTANTS` or the constants won't be added to the
exports.

## install\_constant\_tag

Create and install constant functions for a set of constants.  Called either
as

    install_constant_tag( [ $tag, $fname], $constants, [$package] )

or as

    install_constant_tag( $string, $constants, [$package] )

in which case

    ( $tag, $fname ) = ( lc($string), $string );

`$constants` specifies the constants' names and values, and may be
either a hashref or an arrayref containing _name_ - _value_ pairs.

`$package` is the name of the package (the eventual exporter) into
which the constants will be installed. It defaults to the package of
the caller.

["install\_constant\_tag"](#install_constant_tag) will

1. use Perl's [constant](https://metacpan.org/pod/constant) pragma to create a function named _name_ returning _value_
for each _name_-_value_ pair in `$constants`.

    The functions are installed in `$package` and their names added to
    the symbols in `%EXPORT_TAGS` with export tag `$tag`.

2. use Perl's [constant](https://metacpan.org/pod/constant) pragma to create an enumerating function named
`$fname` which returns a list of the _values_ in `$constants`.  If
`$constants` is an arrayref, the values will be returned in the order
specified.

    If the function `$fname` already exists, a new function is created
    (named `$fname`) which returns the values returned by the original
    function and the new constant values added by this call to
    ["install\_constant\_tag"](#install_constant_tag).

    `$fname` is added to the symbols in `%EXPORT_TAGS` tagged
    with `contant_funcs`.

For example, after

    $id = 'AGGREGATES';
    $constants = { ALL => 'all', NONE => 'none', ANY => 'any' };
    install_constant_tag( $id, $constants );

1. The constant functions, `ALL`, `NONE`, `ANY` will be created and
installed in the calling package.

    A new element will be added to `%EXPORT_TAGS` with an export tag of `aggregates`.

        $EXPORT_TAGS{aggregates} = [ 'ALL', 'NONE', 'ANY ];

2. A function named `AGGREGATES` will be created and installed in the
calling package. `AGGREGATES` will return the values

        'all', 'none', 'any'

    (in a random order, as `$constants` is a hashref).

    `AGGREGATES` will be added to the symbols tagged by `constant_funcs` in `%EXPORT_TAGS`

After this, a package importing from `$package` can

- import the constant functions `ALL`, `NONE`, `ANY` via the `aggregate` tag:

        use Package ':aggregate';

- import the enumerating function `AGGREGATES` directly, via

        use Package 'AGGREGATES';

- import `AGGREGATES` via the `constant_funcs` tag:

        use Package ':constant_funcs';

As mentioned above, if the first argument to ["install\_constant\_tag"](#install_constant_tag) is an
arrayref, `$tag` and `$fname` may be specified directly. For example,

    $id = [ 'Critters', 'Animals' ];
    $constants = { HORSE => 'horse', GOAT   => 'goat' };
    install_constant_tag( $id, $constants );

will create the export tag `Critters` for the `GOAT` and `HORSE`
constant functions and an enumerating function called `Animals`.

`install_constant_tag` uses ["install\_constant\_func"](#install_constant_func) to create and install
the constant functions which return the constant values.

**Caveat!** Because of how enumerating functions are created,
specifying new values and a new enumerating function name for an
existing tag will not cause the new values to be added to the previous
enumerating function. For example,

    install_constant_tag( 'TAG', { HIGH => 'high' }  );
    install_constant_tag( [ 'TAG', 'ATAG' ], { LOW => 'low' } );

Will create a function `TAG` which returns `high` and a function
`ATAG` which returns `low`.  Don't depend upon this behavior.

## install\_constant\_func( $name, \\@values, $caller )

This routine does the following in `$package`, which defaults to the
caller's package.

1. Create a constant subroutine named `$name` which returns `@values`;
2. Adds `$name` to the `constant_funcs` tag in `%EXPORT_TAGS`.

For example, after calling

    install_constant_func( 'AGGREGATES', [ 'all', 'none', 'any' ]  );

1. The function `AGGREGATES` will return `all`, `none`, `any`.
2. A package importing from `$package` can import the `AGGREGATE`
constant function via the `constant_funcs` tag:

        use Package ':constant_funcs';

    or directly

        use Package 'AGGREGATES';

# BUGS

No attempt is made to complain if enumerating functions' names clash
with constant function names.

# EXAMPLES

- Alternate constant generation modules.

    To use an alternate constant generation function bypass
    [install\_CONSTANTS](https://metacpan.org/pod/install_CONSTANTS) and load things manually.

    For example,  using [enum](https://metacpan.org/pod/enum):

        package My::Exporter;

        use CXC::Exporter::Util ':all';

        our @DaysOfWeek;
        BEGIN{ @DaysOfWeek = qw( Sun Mon Tue Wed Thu Fri Sat ) }
        use enum @DaysOfWeek;
        use constant DaysOfWeek => map { &$_ } @DaysOfWeek;
        install_EXPORTS( { days_of_week => \@DaysOfWeek,
                           constant_funcs => [ 'DaysOfWeek' ],
                          });

    and then

        use My::Exporter -days_of_week;

        say Sun | Mon;

# SUPPORT

## Bugs

Please report any bugs or feature requests to bug-cxc-exporter-util@rt.cpan.org  or through the web interface at: https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-Exporter-Util

## Source

Source is available at

    https://gitlab.com/djerius/cxc-exporter-util

and may be cloned from

    https://gitlab.com/djerius/cxc-exporter-util.git

# SEE ALSO

Please see those modules/websites for more information related to this module.

- [Exporter](https://metacpan.org/pod/Exporter)

# AUTHOR

Diab Jerius <djerius@cpan.org>

# COPYRIGHT AND LICENSE

This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

    The GNU General Public License, Version 3, June 2007
