package Magic::ConversionChain;

use 5.006;
use strict;
use warnings FATAL => 'all';

use Data::Dumper;

=head1 NAME

Magic::ConversionChain - Create permanent conversion chains

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 SYNOPSIS

NOTE I created this module to assist me with another and found it so useful and different I decided to release it, I am still actively working on it and input is welcome, but be warned it is no where near complete.

Magic::ConversionChain Allows you to create data manipulation 'objects' for when you are working in a situation where you need to shift one type of data into another or adjust text in specific ways, it was mainly created because the owner hates pack, but it can be very useful for padding data binary/bytes/hex conversions and more.

Here is a code snippet for converting a standard number into an equivlent that would be generated by a signed int32 in perl, the output will be returned in hex.

   use Magic::ConversionChain;

   my $padTest = Convert::Permanent->new({
    conversion => [ qw( i32s hex ) ],
   });

   my $result = $n2h->proc('34243');

   # result = c3850000

I will continue expanding this module including sha, md5 and the entire of pack's data types usable in what I class to be a far more friendly way.

=head1 SUBROUTINES/METHODS

=head2 new

Create a new basic object.

=cut

sub new {
 my $class = shift;
 my $conv = shift;

 if ( (!$conv) || (ref($conv) ne 'HASH') ) { $conv = { input => '', conversion => [], output => '' } };

 if (!$conv->{input}) { $conv->{input} = '' }
 if (!$conv->{output}) { $conv->{output} = '' }
 if (!$conv->{conversion}) { $conv->{conversion} = [] }

 if (! ref($conv->{conversion}) ) {
  # Singular conversion or last in chain, convert into an ARRAY
  $conv->{conversion} = [$conv->{conversion}];
 }

 my $self = {
  config => {
   input => $conv->{input},
   output => $conv->{output},
   conversion => $conv->{conversion},
  },
 };

 if (ref($conv->{conversion}) eq 'ARRAY') {
  # Ok this is a conversion chain... so lets create another object
  my $head = shift @{ $conv->{conversion} };

  # Check for parameterized arguments 
  my $cmdbase = $head;
  if ($head =~ m#^(.*?)\(#) { $cmdbase = $1 } 

  # Keep the integrity of datatype incase of special conversions
  if ( defined Magic::ConversionChain->cmddat($self,$cmdbase) ) { 
   if ( my $datatype = Magic::ConversionChain->cmddat($self,$cmdbase,'type') ) { $self->{config}->{datatype} = $datatype }
   else { $self->{config}->{datatype} = $conv->{input} }
  }
  else { $self->{config}->{datatype} = $head }

  $self->{config}->{conversion} = $head;
  if (scalar @{ $conv->{conversion} } > 0) { 
   $self->{config}->{output} = Magic::ConversionChain->new({ 
    input => $self->{config}->{datatype}, 
    conversion => $conv->{conversion}, 
    output => $conv->{output} 
   });
  } else {
   $self->{config}->{output} = $self->{config}->{datatype};
  }
 }

 bless $self, $class;
}

=head2 proc

Feed data into the filter and get the conversion returned.

=cut

sub proc {
 my $self = shift;
 my $data = shift;

 my $conversion = $self->{config}->{conversion};

 my $cmd = $conversion;
 my @args = ();
 my $proc = "";

 if ($conversion =~ m#^(.*?)\((.*?)\)#) {
  $cmd = $1;
  @args = split(/,/,$2);
 }
  
 $proc = &{ $self->cmddat($cmd,'sub') }($self,$data,@args);

 return $self->output($proc);
# warn "Convert to: $newdata";
}

=head2 input

Configure an input filter for the object, if left unset it will be assumed the object is an 'input' for the conversion chain, you will be able to submit data for conversion via:

 $conv->proc($data);

=cut 

sub input {
 my $self = shift;
 my $data = shift;
}

=head2 i32s

Convert an object into the equivalent of an signed int32.

Requires: 1 argument

=cut

sub i32s {
 my $self = shift;
 my $data = shift;

 return pack('l',$data);
}

=head2 i32u

Convert an object into the equivalent of an unsigned int32.

Requires: 1 argument

=cut

sub i32u {
 my $self = shift;
 my $data = shift;

 return pack('V',$data);
}

=head2 i32u_bigendian

Convert an object into the equivalent of an unsigned int32 (Big Endian).

Requires: 1 argument

=cut

sub i32u_bigendian {
 my $self = shift;
 my $data = shift;

 return pack('N',$data);
}


=head2 output 

=cut 

sub output {
 my $self = shift;
 my $data = shift;

 if ( ref($self->{config}->{output}) eq 'Magic::ConversionChain' ) {
  return $self->{config}->{output}->proc($data);
 }

 return $data;  
}

=head1 Internal conversion methods

=head2 hex 

Convert data into hex.

=cut

sub hex {
 my $self = shift;
 my ($data) = @_;

 return unpack('H*',$data);
}

=head2 pad_right

Required arguments: 2

Syntax: pad_right(LENGTH,PAD_CHARACTER)

Pad the right hand side of the supplied data with ARG1 upto a length of ARG0.

=cut 

sub pad_right {
 my $self = shift;
 my ($data,$length,$char) = @_;

 while(length($data) < $length) { $data .= $char }

 return $data;
}

=head2 pad_left

Required arguments: 2

Syntax: pad_left(LENGTH,PAD_CHARACTER)

Pad the left hand side of the supplied data with ARG1 upto a length of ARG0.

=cut

sub pad_left {
 my $self = shift;
 my ($data,$length,$char) = @_;

 while(length($data) < $length) { $data = $char.$data; }

 return $data;
}

=head2 truncate_right

Truncate text treating the right hand side of the data block as the start point.

Required arguments: 1

Syntax: truncate_right(10)

=cut

sub truncate_right {
 my $self = shift;
 my ($data,$length) = @_;

 return substr $data, (length($data) - $length);
}

=head2 truncate_left 

Truncate text treating the left hand side of the data block as the start point.

=cut 

sub truncate_left {
 my $self = shift;
 my ($data,$length) = @_;

 return substr $data, 0, $length;
}

=head1 Internal callable functions

These are used internally by the module, do not call them directly unless you have a very good reason!

=head2 cmddat 

Receive information back or sub references for data types.

=cut

sub cmddat {
 my $self = shift;
 my ($cmd,$type) = @_;

 my $cmdset = {
  pad_right => {
   sub => \&pad_right,
   type => "",
  },
  pad_left => {
   sub => \&pad_left,
   type => "",
  },
  truncate_left => {
   sub => \&truncate_left,
   type => "",
  },
  truncate_right => {
   sub => \&truncate_right,
   type => "",
  },
  md5 => {
   sub => \&md5,
   type => 'bytes',
  },
  hex => {
   sub => \&hex,
   type => 'hex',
  },
  i32s => {
   sub => \&i32s,
   type => 'bytes'
  },
  i32u => {
   sub => \&i32u,
   type => 'bytes',
  },
  i32u_bigendian => {
   sub => \&i32u_bigendian,
   type => 'bytes',
  },
 };

 return 1 if ( (!$type) && ($cmdset->{$cmd}) );
 return 0 if ( (!$type) && (!$cmdset->{$cmd}) );

 return $cmdset->{$cmd}->{$type};
}


=head1 AUTHOR

Paul G Webster, C<< <daemon at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-convert-permanent at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Convert-Permanent>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Magic::ConversionChain


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Convert-Permanent>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Convert-Permanent>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Convert-Permanent>

=item * Search CPAN

L<http://search.cpan.org/dist/Convert-Permanent/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2014 Paul G Webster.

This program is distributed under the (Revised) BSD License:
L<http://www.opensource.org/licenses/BSD-3-Clause>

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

* Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

* Neither the name of Paul G Webster's Organization
nor the names of its contributors may be used to endorse or promote
products derived from this software without specific prior written
permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Magic::ConversionChain
