package OpenInteract::Handler::Classified;

# $Id: Classified.pm,v 1.8 2001/10/14 23:19:33 lachoy Exp $

use strict;
use Data::Dumper  qw( Dumper );
use OpenInteract::CommonHandler;
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::Classified::ISA     = qw( OpenInteract::CommonHandler SPOPS::Secure );
$OpenInteract::Handler::Classified::VERSION = sprintf("%d.%02d", q$Revision: 1.8 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Classified::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Classified::default_method    = 'search_form';
@OpenInteract::Handler::Classified::forbidden_methods = qw();
%OpenInteract::Handler::Classified::security          = (
 DEFAULT_SECURITY_KEY() => SEC_LEVEL_NONE,
 edit    => SEC_LEVEL_READ,
 remove  => SEC_LEVEL_WRITE,
);

use constant DEFAULT_EXPIRE => 60 * 60 * 24 * 7 * 12;     # 12 weeks

sub MY_PACKAGE                 { return 'classified' }
sub MY_HANDLER_PATH            { return '/Classified' }
sub MY_OBJECT_TYPE             { return 'classified' }
sub MY_OBJECT_CLASS            { return OpenInteract::Request->instance->classified }
sub MY_SEARCH_FIELDS           { return qw( title organization location description ) }
sub MY_SEARCH_FORM_TITLE       { return 'Search for Classified Ads' }
sub MY_SEARCH_FORM_TEMPLATE    { return 'classified_search_form' }
sub MY_SEARCH_RESULTS_TITLE    { return 'Classified Search Results' }
sub MY_SEARCH_RESULTS_TEMPLATE { return 'classified_search_results' }
sub MY_OBJECT_FORM_TITLE       { return 'Classified Detail' }

sub MY_OBJECT_FORM_TEMPLATE {
    my ( $class, $params ) = @_;
    return ( $params->{do_edit} ) ? 'classified_form' : 'classified_detail';
}

sub MY_EDIT_RETURN_URL         { return '/Classified/search_form/' }

sub MY_EDIT_FIELDS {
    return qw( title organization description location
               required_skills recommended_skills
               application_info contact phone fax email url );
}

sub MY_EDIT_FIELDS_DATE        { return qw( deadline ) }
sub MY_ALLOW_SEARCH_FORM       { return 1 }
sub MY_ALLOW_SEARCH            { return 1 }
sub MY_ALLOW_SHOW              { return 1 }
sub MY_ALLOW_EDIT              { return 1 }
sub MY_ALLOW_REMOVE            { return 1 }
sub MY_ALLOW_WIZARD            { return undef }


# Read in the 'keyword' search field and set both title and
# description; also read in the 'posted_after' date and find dates
# after that

sub _search_build_where_customize {
    my ( $class, $tables, $where, $value ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr        = $R->apache;

    my $keyword    = $class->_read_field( $apr, 'keyword' );
    if ( $keyword ) {
        push @{ $where }, '( classified.title LIKE ? OR classified.description LIKE ? )';
        push @{ $value }, "%$keyword%", "%$keyword%";
    }

    my $post_after = $class->_read_field_date( $apr, 'posted_after' );
    if ( $post_after ) {
        push @{ $where }, 'posted_on >= ?';
        push @{ $value }, $post_after;
    }
}



# We present dates to the user in three separate fields

sub _read_field_date {
     my ( $class, $apr, $field ) = @_;
     return join( '-', $apr->param( $field . '_year' ),
                       $apr->param( $field . '_month' ),
                       $apr->param( $field . '_day' ) );
}


# If the user has WRITE access to the object, then he/she is an admin
# and can set active, expires_on and active_on

# All new objects are NOT active until approved by an admin
# (unless an admin is the one doing the editing...)

# Set 'expires_on' if not set


sub _edit_customize {
    my ( $class, $classified, $old_data ) = @_;
    my $R = OpenInteract::Request->instance;

    my $now = $classified->now;

    # Set the 'posted_by' and 'posted_on' if a new item

    unless ( $classified->is_saved ) {
        $classified->{posted_by} = $R->{auth}{user}->id;
        $classified->{posted_on} = $now;
        $classified->{active}    = 'no';
    }

    # User is defined as an administrator if they are a member of the
    # 'site admin' group. This could obviously be changed...

    my $is_admin = 0;
    my $site_admin_id = $R->CONFIG->{default_objects}{site_admin_group};
    foreach my $group ( @{ $R->{auth}{group} } ) {
       $is_admin++ if ( $group->id == $site_admin_id ); 
    }

    if ( $is_admin ) {
        my $apr = $R->apache;
        $classified->{active} = $class->_read_field( $apr, 'active' );
        $classified->{active_on}  ||= $now;
        unless ( $classified->{expires_on} ) {
            my $expire_time = $R->{time} + DEFAULT_EXPIRE;
            my @t = localtime( $expire_time );
            $classified->{expires_on} = join '-', $t[5] + 1900, $t[4] + 1, $t[3];
        }
    }
}


sub notify {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my @classified_id  = $R->apache->param( 'classified_id' );
    my $email          = $R->apache->param( 'email' );
    unless ( $email and scalar @classified_id ) {
        return '<h2 align="center">Error</h2>' .
               '<p>Error: Cannot notify anyone about an object when no ID/email is given.</p>';
    }
    my @classified_list = ();
    foreach my $notify_id ( @classified_id ) {
        my $classified = $R->classified->fetch( $notify_id );
        push @classified_list, $classified   if ( $classified );
    }
    my $rv =  $R->classified->notify({ email   => $email,
                                       subject => 'Classified notification',
                                       object  => \@classified_list,
                                       type    => 'classified' });
    if ( $rv ) {
        return '<h2 align="center">Success!</h2>' .
               '<p>Notification sent properly!</p>';
    }
    return '<h2 align="center">Error</h2>' .
           '<p>Error sending email. Please check error logs!</p>';
}


1;
