package OpenInteract::Handler::BasicPage;

# $Id: BasicPage.pm,v 1.12 2001/07/13 15:23:27 lachoy Exp $

use strict;
use SPOPS::Secure  qw( :level );
use Date::Calc     qw( Date_to_Days Parse_Date );
use Data::Dumper   qw( Dumper );
use File::Basename qw();

@OpenInteract::Handler::BasicPage::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure  );
$OpenInteract::Handler::BasicPage::VERSION = sprintf("%d.%02d", q$Revision: 1.12 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::BasicPage::author            = 'chris@cwinters.com';
$OpenInteract::Handler::BasicPage::default_method    = 'listing';
@OpenInteract::Handler::BasicPage::forbidden_methods = ();
%OpenInteract::Handler::BasicPage::security          = ( 
   listing => SEC_LEVEL_WRITE, show   => SEC_LEVEL_NONE, 
   edit    => SEC_LEVEL_WRITE, remove => SEC_LEVEL_WRITE,
   notify  => SEC_LEVEL_READ 
);


use constant MAIN_SCRIPT => '/BasicPage';

# Use this to mark the beginning and end of the "good" content in a
# page in the filesystem; this allows you to use an HTML editor to
# create the content and to save a full html page to the filesystem

my $BODY_DEMARCATION = '<!-- OI BODY -->';

# Use this to check whether the file retrieved is displayable in the
# browser; others (pdf, ps, mov, etc.) get sent to the user directly

my %DISPLAY_TYPES = map { $_ => 1 } ( '', 'html', 'txt', 'xml', 'htm' );

# Use this to separate your single document into multiple pages

my $PAGE_SEPARATOR = '<!--PAGE-->';

# 52 weeks -- default expiration for page

my $DEFAULT_EXPIRE = 60 * 60 * 24 * 7 * 52;


# Overrides entry in OpenInteract::Handler::GenericDispatcher

sub _get_task {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance; 
    return 'show'  if ( $R->{path}->{full}->[0] ne 'BasicPage' );
    return lc shift @{ $R->{path}->{current} } 
                || $OpenInteract::Handler::BasicPage::default_method;
}



# Retrieve all directories, expanding the one we were asked to (if at
# all). Note that these are just the objects in the database.

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $selected_dir = $R->apache->param( 'selected_dir' );
    my $select_all = ( $selected_dir eq 'EXPANDALL' );
    my $params = { main_script  => MAIN_SCRIPT, 
                   selected_dir => $selected_dir, 
                   error_msg    => $p->{error_msg} };
    $params->{dir_list} = eval { $R->static_page->list_directories() };
    if ( $@ ) {
        OpenIntereact::Error->set( SPOPS::Error->get );
        $R->throw( { code => 403 } );
        $params->{dir_list} = [];
    }

    # Store the pages found using the directory as a key pointing to a
    # listref of files it contains

    if ( $selected_dir ) {
        $params->{children_files} = $R->static_page->fetch_iterator({ 
                                               column_group => 'listing',
                                               where        => 'directory = ?',
                                               value        => [ $selected_dir ] });
    }

    $R->{page}->{title} = 'Listing of Documents';
    return $R->template->handler( {}, $params, 
                                  { db      => 'static_dir_listing', 
                                    package => 'static_page' } );
}


sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT, 
                   error_msg   => $p->{error_msg} };

    my $page    = $p->{static_page};
    my $do_edit = ( $R->apache->param( 'edit' ) and 
                    $p->{level} >= SEC_LEVEL_WRITE );

    # If we are not passed a page, we need to try and grab the
    # location from anywhere we can and create the page; if we cannot
    # create the page, we return an error message unless we've said we
    # want to edit it.

    my $location = $p->{location} || 
                   $R->apache->param( 'location' ) || 
                   $R->{path}->{original}; 

    my $storage = $R->CONFIG->{action}->{basicpage}->{storage};

    my ( $page_info );
    if ( $page ) {
        $page_info = $class->_wrap_object( $page );
    }
    else {

        # Trap errors from any type of page retrieval

        eval {
            if ( $storage eq 'object' or $storage eq 'both-object' ) {
                $page = $class->_retrieve_object( $location );
                $page_info = $class->_wrap_object( $page );
            }
            elsif ( $storage eq 'filesystem' or $storage eq 'both-filesystem' ) {
                $page_info = $class->_retrieve_file( $location );
            }
            if ( ! $page_info and $storage =~ /^both/ ) {
                if ( $storage eq 'both-filesystem' ) {
                    $page_info = $class->_retrieve_object( $location );
                    $page_info = $class->_wrap_object( $page );
                }
                elsif ( $storage eq 'both-object' ) {
                    $page_info = $class->_retrieve_file( $location );
                }
            }
        };
        my $error_type = $@;

        unless ( $page_info or $do_edit ) {
            $R->DEBUG && $R->scrib( 1, "Creating new page or empty page info. Error type: ($error_type)" );

            # We have to force the content-type here because the user
            # might have requested a file that actually exists in the
            # filesystem and which Apache has already mapped a
            # content-type. You'll know when this happens because
            # you'll be prompted to d/l the file or a plugin (like
            # Acrobat Reader) will try to display it, but the *actual*
            # content will be plain old HTML...

            $R->{page}->{content_type} = 'text/html';

            if ( $error_type =~ /^security/ ) {
                return "<h2>Access Forbidden</h2>" .
                       "<p>You do not have access rights to view this page. " .
                       "Please e-mail an administrator if you feel you have received this message in error.</p>";
            }
            elsif ( $error_type =~ /^access/ ) {
                return "<h2>Cannot Access</h2>" .
                       "<p>Failure accessing page.</p>";
            }
            return "<h2>Page Not Found</h2>" .
                   "<p>Could not find page with location: (<tt>$location</tt>). Error logged.</p>";
        }
    }

    # If we specified that we're going to send a separate file to the
    # user (usually not HTML, text, etc.) then set the information and
    # quit processing

    if ( $page_info->{send_file} ) {
        $R->{page}->{send_file} = $page_info->{send_file};
        $R->DEBUG && $R->scrib( 1, "File being retrieved is not displayable. Set 'send_file' to $page_info->{send_file}" );
        return undef;
    }

    $params->{static_page} = $page_info;

    # Now we have a page; just check to see if we were instructed to
    # display the editable form for this page, and if so ensure this
    # user can do so.

    my $text_params = {};

    if ( $do_edit ) {
        $page ||= $R->static_page->new;
        $params->{static_page} = $page;   
        $text_params = { db      => 'static_page_form', 
                         package => 'static_page' };
        $R->{page}->{title}   = 'Edit a Document';
        my $update_items = eval { $page->fetch_updates( 5 ) } || [];
        foreach my $update_info ( @{ $update_items } ) {
            my $user = eval { $R->user->fetch( $update_info->[0] ) };
            my $username = ( $user ) ? $user->{login_name} : 'administrator';
            push @{ $params->{update_list} }, { login_name => $username, date => $update_info->[1] };
        }
    }
    else {
    
        # Ensure the page is viewable right now
    
        unless ( my $is_active = $class->_check_active_date( $page_info ) ) {
            $R->DEBUG && $R->scrib( 1, "Page has been marked as non-active ($is_active); returning error message." );
            $R->{page}->{title} = 'Page not yet active';
            return '<h2 align="center">Not Active</h2>' . 
                   '<p>Sorry, this page is not yet active.</p>';
        }

        $R->{page}->{title}   = $page_info->{title};
        $R->{page}->{script} .= $page_info->{script};

        # Allows the page to define the main template it will use; if
        # the page doesn't define one then the main UI module will use
        # the default

        $R->{page}->{_template_name_} = $page_info->{main_template};
    
        # You can split your page into multiple viewable pages -- see
        # _split_pages() for more info

        $text_params = $class->_split_pages( $page_info );

        if ( $page_info->{_storage} eq 'object' ) {
            $class->_add_object_boxes( $page_info, $p );
        }
    }
    return $R->template->handler( {}, $params, $text_params );
}


# Make changes to a document

sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}->{return_url} = '/BasicPage/listing/';

    my $apr = $R->apache;
    my $location = $apr->param( 'old_location' );

    # If an 'old_location' wasn't set in the html page for editing,
    # then this is a new page

    my $is_new = ( ! $location );
  
    $R->DEBUG && $R->scrib( 1, "Trying to modify location <<$location>>" );
    my $page = $p->{page};
    if ( $location ) {
        $page = eval { $R->static_page->fetch( $location ) };
        if ( $@ ) {
            OpenInteract::Error->set( SPOPS::Error->get );
            $R->throw( { code => 404 } );
        }
    }

    # If we didn't retrieve an object, assume it's new and that our
    # default security for this object is WRITE; if the user isn't
    # supposed to be creating document objects at all, he/she should
    # have a READ permission only for the module

    my $obj_level = ( $page ) ? $page->{tmp_security_level} : SEC_LEVEL_WRITE;
    if ( $obj_level < SEC_LEVEL_WRITE ) {
        my $user_msg = 'Sorry, you do not have access to modify this document. Returning to listing.';
        return $class->show( { page => $page, error_msg => $user_msg } );   
    }

    # Again, create a new object if one wasn't created before.

    $page ||= $R->static_page->new;
    my %no_set = ( expires_on => 1, active_on => 1 );
    foreach my $field ( @{ $R->static_page->field_list } ) {
        next if ( $no_set{ $field } );
        $R->DEBUG && $R->scrib( 1, "Find value for field <<$field>>: ", $apr->param( $field ) );
        $page->{ $field } = $apr->param( $field );
    }
  
    $page->{active_on}  = $class->date_read( 'active_on' ) || $page->now();
    $page->{expires_on} = $class->date_read( 'expires_on' );
    unless ( $page->{expires_on} ) {
        $R->DEBUG && $R->scrib( 1, "Setting page's expiration time to the",
                                   "default, which is $DEFAULT_EXPIRE seconds after now." );
        my $expire_time = $R->{time} + $DEFAULT_EXPIRE;
        my @time_info = localtime( $expire_time );
        $page->{expires_on} = join( '-', $time_info[5] + 1900, $time_info[4] + 1, $time_info[3] );
    }
  
    # If the user changed the location, be sure to send the
    # old ID value to the update
  
    my $update_id = undef;
    unless ( $is_new or $page->{location} eq $location ) {
        $update_id = $location;
    }
  
    $R->DEBUG && $R->scrib( 1, "Contents of Page object before saving:\n", Dumper( $page ) );
    eval { $page->save( { is_add => $is_new, use_id => $update_id } ) };
    if ( $@ ) {
        my $ei = OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 407 } );
        $R->scrib( 0, "Error trying to save page: $@\n", Dumper( $ei ) );
        return '<h2>Error!</h2>' .
               'Cannot save document object! See error log for details.'
           }

    $R->DEBUG && $R->scrib( 1, "Document saved ok (New location: $page->{location});", 
                               "returning to view of document." );
    $p->{static_page} = $page;
    return $class->show( $p ); 
}


sub remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    if ( my $location = $R->apache->param( 'location' ) ) {
        my $page = eval { $R->static_page->fetch( $location ) };
        if ( $@ or ! $page ) {
            return $class->listing( { error_msg => 'Cannot remove document -- object not created properly.' } );
        }
        if ( $page->{tmp_security_level} < SEC_LEVEL_WRITE ) {
            my $user_msg = 'Sorry, you do not have access to remove this document. Returning to listing.';
            return $class->listing( { error_msg => $user_msg } );   
        }

        eval { $page->remove };
        if ( $@ ) {
            OpenInteract::Error->set( SPOPS::Error->get );
            $R->throw( { code => 405 } );
            $p->{error_msg} = "Cannot remove document! See error log.";
        }
    }
    return $class->listing( $p );
}


sub _retrieve_object {
    my ( $class, $location ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Trying to retrieve OBJECT with location ($location)" );
  
    # Chop off any query strings and put in a case-consistent format
  
    $location    = $class->_remove_query_string( $location );
    $location    = lc $location;
    my $home_name = $R->CONFIG->{action}->{basicpage}->{object_home_name};
  
    my @locations = ( $location );
    if ( $location =~ m|/$| ) {
        $R->DEBUG && $R->scrib( 1, "Trying to grab a directory; adding $home_name" );
        push @locations, "$location$home_name";
    }
    else {
        $R->DEBUG && $R->scrib( 1, "Check to see if this is a directory request by adding /$home_name" );
        push @locations, "$location/$home_name";
    }

    if ( $location =~ /\.\w+$/ ) {
      my ( $sans_extension );
      ( $sans_extension = $location ) =~ s/\.\w+$//;
      $R->DEBUG && $R->scrib( 1, "Check location without the extension using $sans_extension" );
      push @locations, $sans_extension;
  }

    my $page = eval { $class->_fetch_object( @locations ) };
    if ( $@ ) {
        my $ei = SPOPS::Error->get();
        warn " -- Found error when fetching object: ", Dumper( $ei ), "\n";
        die 'security' if ( $ei->{type} eq 'security' );
    }
    return $page;
}


# We might want to make this less sensitive -- we currently don't wrap
# the 'fetch()' in an eval so errors bubble up, but maybe we should
# catch the errors here and if all the tested locations result in
# errors we bubble *that* up. For instance, what if requests 1 and 2
# result in a security error but 3 doesn't? (Well... perhaps that
# shouldn't happen in the first place :-)

sub _fetch_object {
    my ( $class, @loc ) = @_;
    my $R = OpenInteract::Request->instance;
    my ( $page );
    foreach my $location ( @loc ) {
        $page = $R->static_page->fetch( $location );
        return $page if ( $page );
    }
    return undef;
}


sub _retrieve_file {
    my ( $class, $location ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Trying to retrieve FILE with location ($location)" );
    $location = $class->_remove_query_string( $location );
    my $home_name = $R->CONFIG->{action}->{basicpage}->{filesystem_home_name};

    if ( $location =~ m|/$| ) {
        $R->DEBUG && $R->scrib( 1, "Trying to grab a directory; adding $home_name" );
        $location .= $home_name;
    }
    my $page_info = $class->_fetch_file( $location, "$location/$home_name" );
    return undef unless ( $page_info );
    $page_info->{_storage} = 'filesystem';
    return $page_info; 
}


sub _fetch_file {
    my ( $class, @loc ) = @_;
    my $R = OpenInteract::Request->instance;
    my ( $page_info );

    # All the files are stored under the 'html' directory -- no
    # exceptions (security considerations, to say the least.)

    my $html_dir = $R->CONFIG->get_dir( 'html' );
    $html_dir =~ s|/$||;
  
    foreach my $location ( @loc ) {

        # Security -- remove all '.' from the beginning of the location
        # requested so people don't try to go up the directory tree. Also
        # remove any two-dot sequence.
        #
        # In the future we might flag these as bad requests and simply
        # bail with a stern warning.

        $location =~ s/^\.+//; 
        $location =~ s/\.\./_/;
    
        my $full_location = join( '', $html_dir, $location );
        next unless ( -f $full_location );
        $R->DEBUG && $R->scrib( 1, "Filesystem location ($full_location) ",
                                   "exists. Trying to read file." );

        # First check to see if this page is protected by security -- if
        # so, check it. If not, we set a default security level of 'WRITE'
        # which will allow viewing

        my $level = SEC_LEVEL_WRITE;
        if ( $R->basicpage and $R->basicpage->isa( 'SPOPS::Secure' ) ) {
            $R->DEBUG && $R->scrib( 1, "StaticPage object is protected by security. ",
                                       "Check security in filesystem." );
            $level = eval { SPOPS::Secure::Hierarchy->check_security({
                                 security_object_class => $R->security_object,
                                 class                 => $R->basicpage, 
                                 object_id             => $location,
		                         user                  => $R->{auth}->{user}, 
                                 group                 => $R->{auth}->{group},
                                 hierarchy_separator   => '/' }) };
            $R->scrib( 0, "Error found checking security: $@" )  if ( $@ );
        }
        $R->DEBUG && $R->scrib( 1, "Security found for ($location): ($level)" );

        die 'security' if ( $level < SEC_LEVEL_READ );

        # If the file extension isn't one we would normally display, treat
        # it as a file we need to send to the user directly.

        my ( $file_extension ) = $location =~ /\.(\w+)$/;
        $R->DEBUG && $R->scrib( 1, "File extension is ($file_extension)" );
        unless ( $DISPLAY_TYPES{ lc $file_extension } ) {
            $R->DEBUG && $R->scrib( 1, "File requested seems to NOT be one to display. ",
                                       "Hopefully Apache will know what kind of file ",
                                       "this should be." );
            $page_info->{send_file} = $full_location;
            return $page_info;
        }

        $R->DEBUG && $R->scrib( 1, "File requested can be displayed in the browser. ",
                                   "Reading in the file..." );
    
        open( STATIC, $full_location ) || die "access";
        local $/ = undef;
        $page_info->{pagetext} = <STATIC>;

        # Pull out the title, main template, author, extra boxes (if
        # they exist)

        if ( $page_info->{pagetext} =~ s|<title>(.*)?</title>|| ) {
            $page_info->{title}    = $1;
        }
        if ( $page_info->{pagetext} =~ s|<template>(.*)?</template>|| ) {
            $page_info->{main_template} = $1;
        }
        if ( $page_info->{pagetext} =~ s|<author>(.*)?</author>|| ) {
            $page_info->{author}   = $1;
        }
        if ( $page_info->{pagetext} =~ s|<boxes>(.*)?</boxes>|| ) {
            $page_info->{boxes}    = $1;
        }

        # Only use the information between the $BODY_DEMARCATION tags
        # (if they exist)

        $page_info->{pagetext} =~ s/$BODY_DEMARCATION(.*)?$BODY_DEMARCATION/$1/;

        # If the page still has <body> tags, only use the information
        # between them

        $page_info->{pagetext} =~ s|<body>(.*)?</body>|$1|i;
    
        $page_info->{location} = $location;
        $page_info->{_filesystem} = $full_location;
        $R->DEBUG && $R->scrib( 2, "Page information being returned: ", Dumper( $page_info ) );
        return $page_info;
    }
    return undef;
}


# Deref the object into a plain old hashref, add a reference back to
# the object and referring to the method of storage

sub _wrap_object {
    my ( $class, $page ) = @_;
    return undef unless ( $page );
    return { %{ $page }, '_storage' => 'object', '_object' => $page };
}


sub _remove_query_string {
    my ( $class, $text ) = @_;
    $text =~  s|^(.*)\?.*$|$1|;
    return $text;
}


sub _split_pages {
    my ( $class, $page_info ) = @_;
    my $R = OpenInteract::Request->instance;

    # Split the page into separate pages -- first check and see if the
    # document IS paged, then do the splitting and other contortions

    if ( $page_info->{pagetext} =~ /$PAGE_SEPARATOR/ ) {
        my @text_pages = split /$PAGE_SEPARATOR/, $page_info->{pagetext};
        my $page_num = $R->apache->param( 'pagenum' ) || 1;
        my $this_page =  $text_pages[ $page_num - 1 ];
        my $total_pages = scalar @text_pages;
        my $current_pagenum = $page_num;
        $this_page .= <<PCOUNT;
     <p align="right"><font size="-1">
     [% comp( 'page_count', total_pages     = $total_pages, 
                            url             = '$page_info->{location}',
                            current_pagenum = $current_pagenum ) %]
     </font></p>
PCOUNT
       return { text => $this_page };
    }
    return { text => $page_info->{pagetext} };
}


sub _add_object_boxes {
    my ( $class, $page_info, $p ) = @_;
    my $R = OpenInteract::Request->instance;

  # Do the related items
#   push @{ $R->{boxes} }, { name => 'relateditems', params => { object => $page } };
 
  # If this page has specified any boxes, push them onto the stack
  # unless they start with a '-', in which case set that name aside
  # so we can go through all of the boxes and remove it.
  
    my %box_remove = ();
    if ( $page_info->{boxes} ) {
        my @boxes = split /\s+/, $page_info->{boxes};
        foreach my $box_name ( @boxes ) {
            if ( $box_name =~ s/^\-// ) {
                $box_remove{ $box_name }++;
            }
            else {
                push @{ $R->{boxes} }, $box_name;
            }
        }
    
        my @box_keep = ();
        foreach my $box_info ( @{ $R->{boxes} } ) {
            next if ( ref $box_info and $box_remove{ $box_info->{name} } );
            next if ( ! ref $box_info and $box_remove{ $box_info } );
            push @box_keep, $box_info;
        }
        $R->{boxes} = \@box_keep;
    }

    # If this person has WRITE access to the module, give them a box
    # so they can edit/remove this document

    if ( $p->{level} >= SEC_LEVEL_WRITE ) { 
        push @{ $R->{boxes} }, { name => 'edit_document_box', params => { location => $page_info->{location} } }, 
                               { name => 'objectmodbox',      params => { object => $page_info->{_object} } };
    }
    return undef;
}


sub _check_active_date {
    my ( $class, $page_info ) = @_;
    my $R = OpenInteract::Request->instance;

    return 1 unless ( $page_info->{active_on} );
    my ( $active_days, $now_days, $expire_days );
    eval {
        $active_days = Date_to_Days( split '-', $page_info->{active_on} );
        $now_days    = Date_to_Days( split '-', SPOPS::Utility->now( { format => '%Y-%m-%d' } ) );
        $expire_days = Date_to_Days( split '-', $page_info->{expires_on} );
    };
    if ( $@ ) {
        $R->scrib( 0, "Problem while trying to evaluate days! Error: $@" );
        return 1;
    }
  
    $R->DEBUG && $R->scrib( 1, "Active on: $page_info->{active_on}; Expires on: $page_info->{expires_on}\n",
                               "Active: $active_days; Now: $now_days; Expire: $expire_days" );
    return undef if ( $now_days < $active_days );
    return undef if ( $now_days > $expire_days );
    $R->DEBUG && $R->scrib( 1, "Page is active since $active_days < $now_days < $expire_days." );
    return 1;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::BasicPage - Edit an html document

=head1 SYNOPSIS

=head1 DESCRIPTION

Displays a 'static' page from information in the database. The URL to
the page looks like a normal page rather than a database call or
something.

=head1 METHODS

=head2 PUBLIC

B<listing>

List the page objects in the database.

B<show>

Display an individual page. Pass a positive value for the 'edit'
parameter to be put into edit view for objects from the database.

B<edit>

Enter changes from objects being edited -- only to objects from the
database.

B<remove>

Removes a page object from the database.

=head2 PRIVATE

B<_get_task>

Override the C<_get_task()> method from
L<OpenInteract::Handler::GenericDispatcher> so that this module can
handle just about anything thrown at it and try to translate it as a
page request.

B<_retrieve_object>

See if we can retrieve an object. If the request has a '/' at the end
we append the 'home' name to the end. (The specific name is set in
your action configuration.) If the request has a file extension we
also test for the request without the file extension.

B<_fetch_object>

Try to fetch an object matching one of many locations.

B<_retrieve_file>

See if we can retrieve a file.

B<_fetch_file>

Try to open a file matchine one of many locations.

B<_wrap_object>

Wrap an the database object info so both the file and object appear
the same once we have fetched them.

B<_remove_query_string>

Just strip the query string off the end.

B<_split_pages>

If the document is 'paged' (split into multiple documents using the
character sequence $PAGE_SEPARATOR) then make the paging actually
happen.

B<_add_object_boxes>

Add boxes from this page.

B<_check_active_date>

Ensure this page is active.

=head1 SEE ALSO

Please see the documentation for this package for the big picture,
including TO DO items and other things. (Should be in
C<../../../doc/package.pod>.)

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
