#!/usr/bin/perl

# $Id: create_object_security.pl,v 1.1 2001/06/08 21:42:43 lachoy Exp $

use strict;
use Getopt::Long qw( GetOptions );
use OpenInteract::Startup;
use SPOPS::Secure qw( :level :scope );

# create_object_security.pl

#   Take a class name and scope information and create security
#   settings for all objects in the class

{
  my ( $OPT_scope, $OPT_scope_id, $OPT_level, $OPT_class,
       $OPT_website_dir, $OPT_help );
  GetOptions( 'scope=s'       => \$OPT_scope,
              'scope_id=s'    => \$OPT_scope_id,
              'level=s'       => \$OPT_level,
              'class=s'       => \$OPT_class,
              'website_dir=s' => \$OPT_website_dir,
              'help'          => \$OPT_help );

  if ( $OPT_help ) { return usage(); exit(0); }

  unless ( $OPT_scope =~ /^(world|group|user)$/ ) {
    die "--scope must be set to 'world', 'group', or 'user'\n";
  }
  if ( $OPT_scope ne 'world' and ! $OPT_scope_id ) {
    die "--scope_id must be set if the --scope is set to 'group' or 'user'\n";   
  }
  unless ( $OPT_level =~ /^(none|read|write)$/ ) {
    die "--level must be set to 'none', 'read' or 'write'\n";
  }
  unless ( $OPT_class ) {
    die "--class must be set to the class of the object you want to\n",
        "set security for (e.g., 'MySite::News')\n";
  }
  $OPT_website_dir ||= $ENV{OIWEBSITE};
  unless ( -d $OPT_website_dir ) {
    die "--website_dir or \$ENV{OIWEBSITE} must be set to an existing\n",
        "OpenInteract website directory\n";
  }

  my $R = OpenInteract::Startup->setup_static_environment( $OPT_website_dir );
  unless ( $R ) {
    die "OpenInteract environment cannot be setup! Is ($OPT_website_dir)\n",
        "a valid OpenInteract website directory?\n";
  }

  unless ( $OPT_class->can( 'fetch' ) ) {
    die "It doesn't appear that ($OPT_class) is a valid class for an\n",
        "SPOPS class defined in your OpenInteract setup.\n";
  }

  my $scope = _find_valid_scope( $OPT_scope );
  my $level = _find_valid_level( $OPT_level );

  print "Objects for which security was successfully set:\n";
  my $object_list = $OPT_class->fetch_group({ skip_security => 1 });
  foreach my $object ( @{ $object_list } ) {
    my $oid = $object->id;
    $OPT_class->set_security({ class          => $OPT_class,
                               object_id      => $oid,
                               scope          => $scope,
                               scope_id       => $OPT_scope_id,
                               security_level => $level });
    if ( $@ ) {
      print "  FAIL: $oid ($@) (", SPOPS::Error->get->{system_msg}, ")\n";
    }
    else {
      print "  OK $oid\n";
    }
    
  }
  print "All done!\n";
}

sub _find_valid_scope {
  my ( $scope ) = @_;
  $scope = lc $scope;
  return SEC_SCOPE_WORLD if ( $scope eq 'world' );
  return SEC_SCOPE_USER  if ( $scope eq 'user' );
  return SEC_SCOPE_GROUP if ( $scope eq 'group' );
  die "Invalid scope given! (Scope: $scope)\n";
}

sub _find_valid_level {
  my ( $level ) = @_;
  $level = lc $level;
  return SEC_LEVEL_NONE  if ( $level eq 'none' );
  return SEC_LEVEL_READ  if ( $level eq 'read' );
  return SEC_LEVEL_WRITE if ( $level eq 'write' );
  die "Invalid level given! (Level: $level)\n";
}

sub usage {
  return <<USAGE;
create_object_security.pl - Create security settings for all objects
of a particular class

Usage:

 create_object_security.pl [options]

Options:

 --class        class of object (e.g., 'MySite::News')

 --level        Security level to set (none|read|write)

 --scope        Scope to set security for (world|group|user)

 --scope_id     ID of user/group -- ignored for 'world'

 --website_dir  Path to OI website (or set \$ENV{OIWEBSITE})

 --help         Display this message

Example:

Create security setting so that all members of group with ID 3 get
write permission for all 'MySite::News' objects.

 perl create_object_security.pl --class=MySite::News \
                                --level=write
                                --scope=group \
                                --scope_id=3 \
                                --website_dir=/home/httpd/mysite.com
USAGE
}

__END__

=pod

=head1 NAME

create_object_security.pl - Batch-create security settings for objects

=head1 SYNOPSIS

 # Create security setting so that all members of group with ID 3 get
 # write permission for all 'MySite::News' objects.

 perl create_object_security.pl --class=MySite::News \
                                --level=write
                                --scope=group \
                                --scope_id=3 \
                                --website_dir=/home/httpd/mysite.com

=head1 DESCRIPTION

Creates security settings for all objects in a class. Run with
'--help' to get syntax and more info.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
