#!/usr/bin/perl

# $Id: add_objects_to_index.pl,v 1.2 2001/02/04 23:06:33 lachoy Exp $

# add_objects_to_index.pl -- Re-index a particular class.

use strict;
use OpenInteract::Startup;
use Data::Dumper  qw( Dumper );
use Getopt::Long  qw( GetOptions );

{
 my $usage = "Usage: $0 --website_dir=/path/to/my_site object-tag";
 
 # Get the options

 my ( $OPT_website_dir );
 GetOptions( 'website_dir=s' => \$OPT_website_dir );

 if ( ! $OPT_website_dir and $ENV{OIWEBSITE} ) {
   print "[oi_manage]: Using ($ENV{OIWEBSITE}) for 'website_dir'.\n";
   $OPT_website_dir = $ENV{OIWEBSITE};
 }

 unless ( -d $OPT_website_dir ) {
   die "$usage\n Parameter 'website_dir' must refer to an OpenInteract website directory!\n";
 }

 my $object_tag = shift;
 die "$usage\n" unless ( $object_tag );

 # Setup the OI environment

 my $R = OpenInteract::Startup->setup_static_environment( 
                                    $OPT_website_dir );

 # Try to get the class corresponding to the object tag passed in

 my $obj_class = eval { $R->$object_tag() };
 if ( $@ or ! $obj_class ) {
   my $error_msg = $@ || 'none returned';
   die "Cannot retrieve objects without a class -- no match for ",
       "$object_tag. (Error: $error_msg)\n";
 }

 # Ensure the object class is currently being indexed

 unless ( $obj_class->isa( 'OpenInteract::FullText' ) ) {
   die "Failed! The class ($obj_class) corresponding to tag \n",
       "($object_tag) does not currently use the full-text indexing\n",
       "engine. Change the 'isa' tag for the object.\n";
 }

 # Retrieve all the objects

 my $obj_list = $obj_class->fetch_group( { skip_security => 1 } );
 print "Objects found in group: ", scalar @{ $obj_list }, "\n";

 # Index each object

 my $count = 0;
 foreach my $obj ( @{ $obj_list } ) {
   print " $count" if ( $count % 25 == 0 );
   print "\n"      if ( $count % 150 == 0 );
   $obj->reindex_object;
   $count++;
 }
 print "Objects indexed: $count\n";

 # Some DBD drivers complain about not explicitly disconnecting, which
 # worries people...

 $R->db->disconnect;
}
 
__END__

=pod

=head1 NAME

add_objects_to_index.pl - Reindex objects in a particular class

=head1 SYNOPSIS

 # 'object-tag' is the label you use in your 'spops.perl' file for the
 # object -- e.g., 'user' for 'OpenInteract::User' objects or
 # 'sitetemplate' for 'OpenInteract::SiteTemplate' objects.

 > add_objects_to_index.pl --website_dir=/path/to/my_website object-tag

OR (using a bash shell):

 > export OIWEBSITE=/path/to/my_website
 > add_objects_to_index.pl object-tag


=head1 DESCRIPTION

Cycles through every available object in the class given and calls
'reindex_object' on it. Pretty simple.

=head1 BUGS 

None yet!

=head1 TO DO

B<Web interface>

This same function should be implemented via a web interface.

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
