
package OpenInteract::Handler::Classified;

# $Id: Classified.pm,v 1.2 2001/02/13 12:41:21 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );

@OpenInteract::Handler::Classified::ISA     = qw( OpenInteract::Handler::GenericDispatcher  SPOPS::Secure );
$OpenInteract::Handler::Classified::VERSION = sprintf("%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Classified::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Classified::default_method    = 'listing';
@OpenInteract::Handler::Classified::forbidden_methods = qw();
%OpenInteract::Handler::Classified::security          = ( 
 listing => SEC_LEVEL_READ,  search => SEC_LEVEL_READ,
 edit    => SEC_LEVEL_WRITE, remove => SEC_LEVEL_WRITE,
 notify  => SEC_LEVEL_READ 
);


use constant DEFAULT_EXPIRE => 60 * 60 * 24 * 7 * 12;     # 12 weeks


sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => '/Classified', error_msg => $p->{error_msg} };
  my $current_date = $R->classified->now;
  my $date_fmt = q/DATE_FORMAT( deadline, '%M %e, %Y' )/;
  my $cl = eval { $R->classified->fetch_group({
                             where => "active = ? and " .
                                      "( deadline is null or deadline > ? )",
                             value => [ 'yes', $current_date ],   
                             order => 'posted_on DESC',
                             field_alter => { deadline => $date_fmt } }) };
  $R->scrib( 2, "Classified list: ", Dumper( $cl ) );
  $params->{classified_list} = $cl;
  $R->{page}->{title} = 'Classified Listing';
  return $R->template->handler( {}, $params, { db => 'classified_list', 
                                               package => 'classified' } );
}


sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => '/Classified', error_msg => $p->{error_msg} };
  my $classified_id = $R->apache->param( 'classified_id' );
  my $classified = $p->{classified};
  my $do_edit = ( $R->apache->param( 'edit' ) and $p->{level} >= SEC_LEVEL_WRITE );
  my $date_format = {};
  unless ( $do_edit ) {
    $date_format = { deadline => "DATE_FORMAT( deadline, '%M %e, %Y' )",
                     posted_on => "DATE_FORMAT( posted_on, '%M %e, %Y' )"  };
  }
  $classified  ||= eval { $R->classified->fetch( $classified_id, 
                                                 { field_alter => $date_format } ) };

  # If the user has less than WRITE access to module and there is no
  # news_id or news object, there's no reason to continue

  if ( ! $classified and $p->{level} < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to create a new Classified object. Returning to listing.';
    return $class->listing( { error_msg => $user_msg } );
  }

  # Object security defaults to WRITE: if there is no object,
  # then you're creating it and security then depends on the application

  my $obj_level = ( $classified ) 
                   ? $classified->{tmp_security_level} 
                   : SEC_LEVEL_WRITE;

  $classified  ||= $R->classified->new;
  $params->{classified} = $classified;
  my $tmpl_name = 'classified_detail';
  if ( $do_edit ) {
    $tmpl_name = 'classified_form';
  }
  $R->{page}->{title} = 'Classified Ad Detail';
  return $R->template->handler( {}, $params, { db => $tmpl_name, package => 'classified' } ); 
}


sub edit {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  $R->{page}->{return_url} = '/Classified/';
  my $apr = $R->apache;
  my $classified_id = $apr->param( 'classified_id' );
  $R->scrib( 1, "Trying to modify ID <<$classified_id>>" );
  my $classified = eval { $R->classified->fetch( $classified_id ) };
  my $obj_level = ( $classified ) ? eval { $classified->check_security } : SEC_LEVEL_WRITE;
  if ( $obj_level < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to modify this classified object. Returning to listing.';
    return $class->listing( { classified => $classified, error_msg => $user_msg } );   
  }

  $classified ||= $R->classified->new;
  my %skip = map { $_ => 1 } qw( posted_by classified_id deadline expires_on );
  foreach my $field ( @{ $R->classified->field_list } ) {
    next if ( $skip{ $field } );
    $R->scrib( 1, "Find value for field <<$field>>" );
    $classified->{ $field } = $apr->param( $field );
  }
  $classified->{deadline} = $class->date_read( 'deadline' );
  unless ( $classified->{expires_on} ) {
    my $expire_time = $R->{time} + DEFAULT_EXPIRE;
    my @time_info = localtime( $expire_time );
    $classified->{expires_on} = join '-', $time_info[5] + 1900, $time_info[4] + 1, $time_info[3];
  }
  $classified->{posted_by} ||= $R->{auth}->{user}->{user_id};
  $classified->{posted_on} ||= $classified->now;
  eval { $classified->save };
  if ( $@ ) {
    $p->{error_msg} = 'Cannot save classified object! See error log for details.';
  }
  $p->{classified} = $classified;
  return $class->show( $p ); 
}


sub remove {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  if ( my $classified_id = $R->apache->param( 'classified_id' ) ) {
    my $classified = $R->classified->fetch( $classified_id );
    if ( $@ or ! $classified ) {
      return $class->listing( param => { error_msg => 'Cannot remove classified -- object not created properly.' } );
    }
    if ( SEC_LEVEL_WRITE > eval { $classified->check_security } ) {
      my $user_msg = 'Sorry, you do not have access to remove this classified object. Returning to listing.';
      return $class->listing( { error_msg => $user_msg } );   
    }
    
    eval { $classified->remove };
    if ( $@ ) {
      $p->{error_msg} = 'Cannot remove classified object! See error log for details...';
    }
  }
  return $class->listing( $p );
}


sub notify {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my @classified_id  = $R->apache->param( 'classified_id' );
  my $email    = $R->apache->param( 'email' );
  if ( ! $email or ! scalar @classified_id ) {
    return '<h2 align="center">Error</h2>' .
           '<p>Error: Cannot notify anyone about an object when no ID/email is given.</p>';
  }
  my @classified_list = ();
  foreach my $nid ( @classified_id ) {
    my $classified = $R->classified->fetch( $nid );
    push @classified_list, $classified   if ( $classified );
  }
  my $rv =  $R->classified->notify({ email => $email, 
                                     subject => 'Classified notification', 
                                     object => \@classified_list, 
                                     type => 'classified' });
  if ( $rv ) {
    return '<h2 align="center">Success!</h2>' .
           '<p>Notification sent properly!</p>';
  }
  return '<h2 align="center">Error</h2>' .
         '<p>Error sending email. Please check error logs!</p>';   
}

1;

__END__

=pod

sub search {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => '/Classified', error_msg => $p->{error_msg} };
  my $search = $R->apache->param( 'search' );
  my $current_date = $R->classified->now;
  my $where = "active = ? and ( deadline is null or deadline > ? )";
  if ( $search ) {
    $search =~ s/ /\', \'/go;
    my $in_clause = "IN ( '$search' )";
    $where .= " AND ( title $in_clause OR classified_item $in_clause ) ";
  }

 my $cl = eval { $R->classified->fetch_group( { where => $where,
                                                DEBUG => 1,
                                                value => [ 'yes', $current_date ],   
                                                order => 'posted_on DESC',
                                                field_alter => { 
                                                    deadline => q/DATE_FORMAT( deadline, '%M %e, %Y' )/ } 
                                              } ) };
 $R->scrib( 1, "Classified list: ", Dumper( $cl ) );

 $params->{classified_list} = $cl;
 my $tmpl_name = 'classified_list'; 
 $R->{page}->{title} = 'IAF Classified Search';

 return $R->template->handler( {}, $params, { db => $tmpl_name } );
}

=cut
