package OpenInteract::Handler::Group;

# $Id: Group.pm,v 1.9 2001/02/01 05:24:42 cwinters Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );

@OpenInteract::Handler::Group::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Group::VERSION = sprintf("%d.%02d", q$Revision: 1.9 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Group::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Group::default_method    = 'listing';
@OpenInteract::Handler::Group::forbidden_methods = ();
%OpenInteract::Handler::Group::security          = ( 
 listing => SEC_LEVEL_READ,  edit => SEC_LEVEL_WRITE,
 show    => SEC_LEVEL_READ,  remove => SEC_LEVEL_WRITE,
);

use constant DEBUG => 0;

my $MAIN_SCRIPT  = '/Group';
my $MEMBER_FIELD = 'group_members';


# Simple: Just retrieve a list of groups. Currently, the only limit is
# based on what the user can 'see' (security-wise)

sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  $params->{group_list} = eval { $R->group->fetch_group( { order => 'name' } ); }; 
  if ( $@ ) {
    $params->{error_msg} .= "\n\nCannot retrieve group listing";
    OpenInteract::Error->set( SPOPS::Error->get );
    $OpenInteract::Error::user_msg = 'Cannot retrieve groups for listing.';
    $R->throw( { code => 403 } );
  }
  $R->{page}->{title} = 'Group Listing';
  return $R->template->handler( {}, $params, { db => 'group_list', package => 'base_group' } );
}

sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance; 
  my $params = { error_msg => $p->{error_msg}, main_script => $MAIN_SCRIPT };
  my $group_id = $R->apache->param( 'group_id' ) || $R->apache->param( 'gid' );
  my $group = $p->{group} || eval { $R->group->fetch( $group_id ) };
  if ( $@ ) {
    my $ei = OpenInteract::Error->set( SPOPS::Error->get );
    if ( $ei->{type} eq 'security' ) {
      $params->{error_msg} .= "You do not have access to view this group. Permission denied -- returning to listing";
    }
    else {
      $R->throw( { code => 404 } );
      $params->{error_msg} .= "Error -- most likely a database error -- retrieving group object." .
                              "Error has been logged. Returning you to listing.";
    }
    return $class->listing( $params );
  }
 
 # If the user has less than WRITE access to module and there is no
 # group object, there's no reason to continue

  if ( ! $group and $p->{level} < SEC_LEVEL_WRITE ) {
    $params->{error_msg} = 'Sorry, you do not have access to create a new group object. Returning to listing.';
    return $class->listing( $params );
  }

  # Object security defaults to WRITE: if there is no object,
  # then you're creating it and security then depends on the application
  # 
  # Why do we check that both the object and the ID exist? Picture a situation
  # where the user fails to enter necessary information and edit() bounces the 
  # error back to us, including the actual object in our parameter list. Since it
  # does indeed exist, we'll try and run check_security() on it here, which 
  # will bomb entirely, and we'll be left with an object that needs to be
  # edited, that doesn't actually exist -- but it's on a read-only screen :) 
  
  my $obj_level = ( $group ) ? $group->{tmp_security_level} : SEC_LEVEL_WRITE;
  $group  ||= $R->group->new;

  # Retrieve the member users

  my $mem_users = eval { $group->user } || [];
  $params->{error_msg} .= "Failed to retrieve user listing: $@->{error}"   if ( $@ );
  my @mem_user_list = map { { id => $_->{user_id}, name => $_->{login_name} } } @{ $mem_users };
 
  $params->{group} = $group;
  $params->{mem_user_list} = \@mem_user_list;

  my $tmpl_name = 'group_detail';

  # If the user has WRITE access to tool **plus** WRITE access
  # to the object, change the template and retrieve some additional info

  if ( $p->{level} == SEC_LEVEL_WRITE and $obj_level == SEC_LEVEL_WRITE ) {
    my $all_user_list = eval { $R->user->fetch_group; };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 403 } );
      $params->{error_msg} = "Cannot retrieve all users. Error logged.";
    }
    $all_user_list ||= [];
    my %all_nonmem_users = map { $_->{user_id} => $_->{login_name} } 
                               @{ $all_user_list };
    foreach ( @mem_user_list ) { delete $all_nonmem_users{ $_->{id} } }
    my @all_nonmem_users = map { { id => $_, name => $all_nonmem_users{ $_ } } }
                               sort { $all_nonmem_users{ $a } cmp $all_nonmem_users{ $b } } 
                               keys %all_nonmem_users;
    $params->{member_field}  = $MEMBER_FIELD;
    $params->{all_user_list} = \@all_nonmem_users;
    $tmpl_name = 'group_form';
  }

  $R->{page}->{title}  = 'Group Detail';
  return $R->template->handler( {}, $params, { db => $tmpl_name } );
}

sub edit {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { error_msg => $p->{error_msg}, main_script => $MAIN_SCRIPT };
  $R->{page}->{return_url} = '/Group/'; 

  # First get the gid from the user and try to fetch

  my $apr = $R->apache;
  my $group_id = $apr->param( 'group_id' ); 
  my $group = eval { $R->group->fetch( $group_id ); };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 404 } );
    $params->{error_msg} .= "Cannot retrieve group object for editing -- no changes made. Returning to listing";
    return $class->listing( $p );
  }

  # See what our security level for this object is; bail on failure or
  # if our level is less than WRITE

  my $obj_level = ( $group ) ? $group->{tmp_security_level} : SEC_LEVEL_WRITE;
  if ( $obj_level < SEC_LEVEL_WRITE ) {
    $params->{error_msg} .= 'Sorry, you do not have access to modify this group object. Returning to listing.';
    return $class->listing( $params );
  }

  # If we don't have a group yet, create one; then read in
  # all relevant form values for the group (new or old)

  $group ||= $R->group->new;
  foreach my $field ( @{ $R->group->field_list } ) {
    $group->{ $field } = $apr->param( $field );
  }

  # Try to save; if we can't pass back an error.

  eval { $group->save };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $OpenInteract::Error::user_msg = 'Error saving group object.';
    $R->throw( { code => 407 } );   
    $params->{error_msg} = "Cannot save group object. Error logged.";
    return $class->show( { group => $group, %{ $params } } );
  }

  # Now save the members that were specified
  # 
  # First get the existing members, then split apart the members
  # specified in the form. Give both pieces of information to
  # the list_process method to separate them out into removals,
  # additions and keepers.

  my $existing_user_list = eval { $group->user } || []; 
  my @existing_uid = map { $_->{user_id} } @{ $existing_user_list };
  my @member_uid   = split ';', $apr->param( $MEMBER_FIELD );
  $R->scrib( 1, "User IDs retrieved: ", join( '/', @member_uid ) );
  my $mem_info = $group->list_process( \@existing_uid, \@member_uid );
  $R->scrib( 1, "After processing: ", Dumper( $mem_info ) );

  my $removed = eval { $group->user_remove( $mem_info->{remove} ); };
  if ( $@ ) { $R->scrib( 0, "Error removing users from group: $@" ) }
  $R->scrib( 1, "Removed <$removed> of <", scalar @{ $mem_info->{remove} }, ">" );
  
  my $added   = eval { $group->user_add( $mem_info->{add} ); };
  if ( $@ ) { $R->scrib( 0, "Error adding users to group: $@" ) }
  $R->scrib( 1, "Added <$added> of <", scalar @{ $mem_info->{add} }, ">" );

  # always display this group: if we had an error, then they can
  # edit the information again; if they didn't have an error,
  # then they'll see that something happened.

  $params->{group} = $group;
  return $class->show( $params ); 
}

sub remove {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $group_id = $R->apache->param( 'group_id' );
  my $group = undef;
  if ( $group_id ) {
    my $group = eval { $R->group->fetch( $group_id ) };
    if ( $@ ) {
      my $error_msg = undef;
      my $ei = OpenInteract::Error->set( SPOPS::Error->get );
      if ( $ei->{type} eq 'security' ) {
        $error_msg = "Cannot remove group: you do not have access to view this group.";
      }
      else {
        $error_msg = 'Cannot remove Group -- object not created properly. Error logged.';
      }
      return $class->listing( { error_msg => $error_msg } );
    }
  }
  unless ( $group ) {
    return $class->listing( { error_msg => 'Cannot create group for removal: bad ID for object.' } );
  }
  if ( $group->{tmp_security_level} < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to remove this group object. Returning to listing.';
    return $class->listing( { error_msg => $user_msg } );   
  }
  
  eval { $group->remove };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 405 } );
    $p->{error_msg} = 'Error removing group object! Error logged.';
  }
  return $class->listing( $p );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Group - Manipulate group records

=head1 SYNOPSIS

=head1 DESCRIPTION

The group records modified by this handler are fairly rudimentary. If
you add information and/or actions to the group object, you might want
to use this as a base handler so you do not have to start from scratch.

=head1 METHODS

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
