{
# GENERATE RECURSIVE DESCENT PARSER OBJECTS FROM A GRAMMAR

use 5.006;
use strict;

package Parse::RecDescent::_Runtime;

use Text::Balanced qw ( extract_codeblock extract_bracketed extract_quotelike extract_delimited );

use vars qw ( $skip );

   *defskip  = \ '\s*'; # DEFAULT SEPARATOR IS OPTIONAL WHITESPACE
   $skip  = '\s*';      # UNIVERSAL SEPARATOR IS OPTIONAL WHITESPACE
my $MAXREP  = 100_000_000;  # REPETITIONS MATCH AT MOST 100,000,000 TIMES



package Parse::RecDescent::_Runtime::LineCounter;


sub TIESCALAR   # ($classname, \$text, $thisparser, $prevflag)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    my $cache = $parser->{linecounter_cache};
    my $from = $parser->{fulltextlen}-length(${$_[0]->{text}})-$_[0]->{prev}
;

    unless (exists $cache->{$from})
    {
        $parser->{lastlinenum} = $parser->{offsetlinenum}
          - Parse::RecDescent::_Runtime::_linecount(substr($parser->{fulltext},$from))
          + 1;
        $cache->{$from} = $parser->{lastlinenum};
    }
    return $cache->{$from};
}

sub STORE
{
    my $parser = $_[0]->{parser};
    $parser->{offsetlinenum} -= $parser->{lastlinenum} - $_[1];
    return undef;
}

sub resync   # ($linecounter)
{
    my $self = tied($_[0]);
    die "Tried to alter something other than a LineCounter\n"
        unless $self =~ /Parse::RecDescent::_Runtime::LineCounter/;

    my $parser = $self->{parser};
    my $apparently = $parser->{offsetlinenum}
             - Parse::RecDescent::_Runtime::_linecount(${$self->{text}})
             + 1;

    $parser->{offsetlinenum} += $parser->{lastlinenum} - $apparently;
    return 1;
}

package Parse::RecDescent::_Runtime::ColCounter;

sub TIESCALAR   # ($classname, \$text, $thisparser, $prevflag)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    my $missing = $parser->{fulltextlen}-length(${$_[0]->{text}})-$_[0]->{prev}+1;
    substr($parser->{fulltext},0,$missing) =~ m/^(.*)\Z/m;
    return length($1);
}

sub STORE
{
    die "Can't set column number via \$thiscolumn\n";
}


package Parse::RecDescent::_Runtime::OffsetCounter;

sub TIESCALAR   # ($classname, \$text, $thisparser, $prev)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?-1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    return $parser->{fulltextlen}-length(${$_[0]->{text}})+$_[0]->{prev};
}

sub STORE
{
    die "Can't set current offset via \$thisoffset or \$prevoffset\n";
}



package Parse::RecDescent::_Runtime::Rule;

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    my $name  = $_[1];
    my $owner = $_[2];
    my $line  = $_[3];
    my $replace = $_[4];

    if (defined $owner->{"rules"}{$name})
    {
        my $self = $owner->{"rules"}{$name};
        if ($replace && !$self->{"changed"})
        {
            $self->reset;
        }
        return $self;
    }
    else
    {
        return $owner->{"rules"}{$name} =
            bless
            {
                "name"     => $name,
                "prods"    => [],
                "calls"    => [],
                "changed"  => 0,
                "line"     => $line,
                "impcount" => 0,
                "opcount"  => 0,
                "vars"     => "",
            }, $class;
    }
}

sub reset($)
{
    @{$_[0]->{"prods"}} = ();
    @{$_[0]->{"calls"}} = ();
    $_[0]->{"changed"}  = 0;
    $_[0]->{"impcount"}  = 0;
    $_[0]->{"opcount"}  = 0;
    $_[0]->{"vars"}  = "";
}

sub DESTROY {}

sub hasleftmost($$)
{
    my ($self, $ref) = @_;

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        return 1 if $prod->hasleftmost($ref);
    }

    return 0;
}

sub leftmostsubrules($)
{
    my $self = shift;
    my @subrules = ();

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        push @subrules, $prod->leftmostsubrule();
    }

    return @subrules;
}

sub expected($)
{
    my $self = shift;
    my @expected = ();

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        my $next = $prod->expected();
        unless (! $next or _contains($next,@expected) )
        {
            push @expected, $next;
        }
    }

    return join ', or ', @expected;
}

sub _contains($@)
{
    my $target = shift;
    my $item;
    foreach $item ( @_ ) { return 1 if $target eq $item; }
    return 0;
}

sub addcall($$)
{
    my ( $self, $subrule ) = @_;
    unless ( _contains($subrule, @{$self->{"calls"}}) )
    {
        push @{$self->{"calls"}}, $subrule;
    }
}

sub addprod($$)
{
    my ( $self, $prod ) = @_;
    push @{$self->{"prods"}}, $prod;
    $self->{"changed"} = 1;
    $self->{"impcount"} = 0;
    $self->{"opcount"} = 0;
    $prod->{"number"} = $#{$self->{"prods"}};
    return $prod;
}

sub addvar
{
    my ( $self, $var, $parser ) = @_;
    if ($var =~ /\A\s*local\s+([%@\$]\w+)/)
    {
        $parser->{localvars} .= " $1";
        $self->{"vars"} .= "$var;\n" }
    else
        { $self->{"vars"} .= "my $var;\n" }
    $self->{"changed"} = 1;
    return 1;
}

sub addautoscore
{
    my ( $self, $code ) = @_;
    $self->{"autoscore"} = $code;
    $self->{"changed"} = 1;
    return 1;
}

sub nextoperator($)
{
    my $self = shift;
    my $prodcount = scalar @{$self->{"prods"}};
    my $opcount = ++$self->{"opcount"};
    return "_operator_${opcount}_of_production_${prodcount}_of_rule_$self->{name}";
}

sub nextimplicit($)
{
    my $self = shift;
    my $prodcount = scalar @{$self->{"prods"}};
    my $impcount = ++$self->{"impcount"};
    return "_alternation_${impcount}_of_production_${prodcount}_of_rule_$self->{name}";
}


sub code
{
    my ($self, $namespace, $parser, $check) = @_;

eval 'undef &' . $namespace . '::' . $self->{"name"} unless $parser->{saving};

    my $code =
'
# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub ' . $namespace . '::' . $self->{"name"} .  '
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"' . $self->{"name"} . '"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [' . $self->{"name"} . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{' . $self->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;

    ' . ($parser->{deferrable}
        ? 'my $def_at = @{$thisparser->{deferred}};'
        : '') .
    '
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{' . $self->expected() . '});
    $expectation->at($_[1]);
    '. ($parser->{_check}{thisoffset}?'
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    ':'') . ($parser->{_check}{prevoffset}?'
    my $prevoffset;
    tie $prevoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser, 1;
    ':'') . ($parser->{_check}{thiscolumn}?'
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    ':'') . ($parser->{_check}{prevcolumn}?'
    my $prevcolumn;
    tie $prevcolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser, 1;
    ':'') . ($parser->{_check}{prevline}?'
    my $prevline;
    tie $prevline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser, 1;
    ':'') . '
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    '. $self->{vars} .'
';

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        $prod->addscore($self->{autoscore},0,0) if $self->{autoscore};
        next unless $prod->checkleftmost();
        $code .= $prod->code($namespace,$self,$parser);

        $code .= $parser->{deferrable}
                ? '     splice
                @{$thisparser->{deferred}}, $def_at unless $_matched;
                  '
                : '';
    }

    $code .=
'
    unless ( $_matched || defined($score) )
    {
        ' .($parser->{deferrable}
            ? '     splice @{$thisparser->{deferred}}, $def_at;
              '
            : '') . '

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{' . $self->{"name"} .'},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{' . $self->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' rule<< (return value: [} .
                      $return . q{])}, "",
                      q{' . $self->{"name"} .'},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{' . $self->{"name"} .'},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}
';

    return $code;
}

my @left;
sub isleftrec($$)
{
    my ($self, $rules) = @_;
    my $root = $self->{"name"};
    @left = $self->leftmostsubrules();
    my $next;
    foreach $next ( @left )
    {
        next unless defined $rules->{$next}; # SKIP NON-EXISTENT RULES
        return 1 if $next eq $root;
        my $child;
        foreach $child ( $rules->{$next}->leftmostsubrules() )
        {
            push(@left, $child)
            if ! _contains($child, @left) ;
        }
    }
    return 0;
}

package Parse::RecDescent::_Runtime::Production;

sub describe ($;$)
{
    return join ' ', map { $_->describe($_[1]) or () } @{$_[0]->{items}};
}

sub new ($$;$$)
{
    my ($self, $line, $uncommit, $error) = @_;
    my $class = ref($self) || $self;

    bless
    {
        "items"    => [],
        "uncommit" => $uncommit,
        "error"    => $error,
        "line"     => $line,
        strcount   => 0,
        patcount   => 0,
        dircount   => 0,
        actcount   => 0,
    }, $class;
}

sub expected ($)
{
    my $itemcount = scalar @{$_[0]->{"items"}};
    return ($itemcount) ? $_[0]->{"items"}[0]->describe(1) : '';
}

sub hasleftmost ($$)
{
    my ($self, $ref) = @_;
    return ${$self->{"items"}}[0] eq $ref  if scalar @{$self->{"items"}};
    return 0;
}

sub isempty($)
{
    my $self = shift;
    return 0 == @{$self->{"items"}};
}

sub leftmostsubrule($)
{
    my $self = shift;

    if ( $#{$self->{"items"}} >= 0 )
    {
        my $subrule = $self->{"items"}[0]->issubrule();
        return $subrule if defined $subrule;
    }

    return ();
}

sub checkleftmost($)
{
    my @items = @{$_[0]->{"items"}};
    if (@items==1 && ref($items[0]) =~ /\AParse::RecDescent::_Runtime::Error/
        && $items[0]->{commitonly} )
    {
        Parse::RecDescent::_Runtime::_warn(2,"Lone <error?> in production treated
                        as <error?> <reject>");
        Parse::RecDescent::_Runtime::_hint("A production consisting of a single
                      conditional <error?> directive would
                      normally succeed (with the value zero) if the
                      rule is not 'commited' when it is
                      tried. Since you almost certainly wanted
                      '<error?> <reject>' Parse::RecDescent::_Runtime
                      supplied it for you.");
        push @{$_[0]->{items}},
            Parse::RecDescent::_Runtime::UncondReject->new(0,0,'<reject>');
    }
    elsif (@items==1 && ($items[0]->describe||"") =~ /<rulevar|<autoscore/)
    {
        # Do nothing
    }
    elsif (@items &&
        ( ref($items[0]) =~ /\AParse::RecDescent::_Runtime::UncondReject/
        || ($items[0]->describe||"") =~ /<autoscore/
        ))
    {
        Parse::RecDescent::_Runtime::_warn(1,"Optimizing away production: [". $_[0]->describe ."]");
        my $what = $items[0]->describe =~ /<rulevar/
                ? "a <rulevar> (which acts like an unconditional <reject> during parsing)"
             : $items[0]->describe =~ /<autoscore/
                ? "an <autoscore> (which acts like an unconditional <reject> during parsing)"
                : "an unconditional <reject>";
        my $caveat = $items[0]->describe =~ /<rulevar/
                ? " after the specified variable was set up"
                : "";
        my $advice = @items > 1
                ? "However, there were also other (useless) items after the leading "
                  . $items[0]->describe
                  . ", so you may have been expecting some other behaviour."
                : "You can safely ignore this message.";
        Parse::RecDescent::_Runtime::_hint("The production starts with $what. That means that the
                      production can never successfully match, so it was
                      optimized out of the final parser$caveat. $advice");
        return 0;
    }
    return 1;
}

sub changesskip($)
{
    my $item;
    foreach $item (@{$_[0]->{"items"}})
    {
        if (ref($item) =~ /Parse::RecDescent::_Runtime::(Action|Directive)/)
        {
            return 1 if $item->{code} =~ /\$skip\s*=/;
        }
    }
    return 0;
}

sub adddirective
{
    my ( $self, $whichop, $line, $name ) = @_;
    push @{$self->{op}},
        { type=>$whichop, line=>$line, name=>$name,
          offset=> scalar(@{$self->{items}}) };
}

sub addscore
{
    my ( $self, $code, $lookahead, $line ) = @_;
    $self->additem(Parse::RecDescent::_Runtime::Directive->new(
                  "local \$^W;
                   my \$thisscore = do { $code } + 0;
                   if (!defined(\$score) || \$thisscore>\$score)
                    { \$score=\$thisscore; \$score_return=\$item[-1]; }
                   undef;", $lookahead, $line,"<score: $code>") )
        unless $self->{items}[-1]->describe =~ /<score/;
    return 1;
}

sub check_pending
{
    my ( $self, $line ) = @_;
    if ($self->{op})
    {
        while (my $next = pop @{$self->{op}})
        {
        Parse::RecDescent::_Runtime::_error("Incomplete <$next->{type}op:...>.", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The current production ended without completing the
             <$next->{type}op:...> directive that started near line
             $next->{line}. Did you forget the closing '>'?");
        }
    }
    return 1;
}

sub enddirective
{
    my ( $self, $line, $minrep, $maxrep ) = @_;
    unless ($self->{op})
    {
        Parse::RecDescent::_Runtime::_error("Unmatched > found.", $line);
        Parse::RecDescent::_Runtime::_hint(
            "A '>' angle bracket was encountered, which typically
             indicates the end of a directive. However no suitable
             preceding directive was encountered. Typically this
             indicates either a extra '>' in the grammar, or a
             problem inside the previous directive.");
        return;
    }
    my $op = pop @{$self->{op}};
    my $span = @{$self->{items}} - $op->{offset};
    if ($op->{type} =~ /left|right/)
    {
        if ($span != 3)
        {
        Parse::RecDescent::_Runtime::_error(
            "Incorrect <$op->{type}op:...> specification:
             expected 3 args, but found $span instead", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The <$op->{type}op:...> directive requires a
             sequence of exactly three elements. For example:
             <$op->{type}op:leftarg /op/ rightarg>");
        }
        else
        {
        push @{$self->{items}},
            Parse::RecDescent::_Runtime::Operator->new(
                $op->{type}, $minrep, $maxrep, splice(@{$self->{"items"}}, -3));
        $self->{items}[-1]->sethashname($self);
        $self->{items}[-1]{name} = $op->{name};
        }
    }
}

sub prevwasreturn
{
    my ( $self, $line ) = @_;
    unless (@{$self->{items}})
    {
        Parse::RecDescent::_Runtime::_error(
            "Incorrect <return:...> specification:
            expected item missing", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The <return:...> directive requires a
            sequence of at least one item. For example:
            <return: list>");
        return;
    }
    push @{$self->{items}},
        Parse::RecDescent::_Runtime::Result->new();
}

sub additem
{
    my ( $self, $item ) = @_;
    $item->sethashname($self);
    push @{$self->{"items"}}, $item;
    return $item;
}

sub _duplicate_itempos
{
    my ($src) = @_;
    my $dst = {};

    foreach (keys %$src)
    {
        %{$dst->{$_}} = %{$src->{$_}};
    }
    $dst;
}

sub _update_itempos
{
    my ($dst, $src, $typekeys, $poskeys) = @_;

    my @typekeys = 'ARRAY' eq ref $typekeys ?
      @$typekeys :
      keys %$src;

    foreach my $k (keys %$src)
    {
        if ('ARRAY' eq ref $poskeys)
        {
            @{$dst->{$k}}{@$poskeys} = @{$src->{$k}}{@$poskeys};
        }
        else
        {
            %{$dst->{$k}} = %{$src->{$k}};
        }
    }
}

sub preitempos
{
    return q
    {
        push @itempos, {'offset' => {'from'=>$thisoffset, 'to'=>undef},
                        'line'   => {'from'=>$thisline,   'to'=>undef},
                        'column' => {'from'=>$thiscolumn, 'to'=>undef} };
    }
}

sub incitempos
{
    return q
    {
        $itempos[$#itempos]{'offset'}{'from'} += length($lastsep);
        $itempos[$#itempos]{'line'}{'from'}   = $thisline;
        $itempos[$#itempos]{'column'}{'from'} = $thiscolumn;
    }
}

sub unincitempos
{
    # the next incitempos will properly set these two fields, but
    # {'offset'}{'from'} needs to be decreased by length($lastsep)
    # $itempos[$#itempos]{'line'}{'from'}
    # $itempos[$#itempos]{'column'}{'from'}
    return q
    {
        $itempos[$#itempos]{'offset'}{'from'} -= length($lastsep) if defined $lastsep;
    }
}

sub postitempos
{
    return q
    {
        $itempos[$#itempos]{'offset'}{'to'} = $prevoffset;
        $itempos[$#itempos]{'line'}{'to'}   = $prevline;
        $itempos[$#itempos]{'column'}{'to'} = $prevcolumn;
    }
}

sub code($$$$)
{
    my ($self,$namespace,$rule,$parser) = @_;
    my $code =
'
    while (!$_matched'
    . (defined $self->{"uncommit"} ? '' : ' && !$commit')
    . ')
    {
        ' .
        ($self->changesskip()
            ? 'local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;'
            : '') .'
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['
                      . $self->describe . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{' . $rule ->{name}. '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[' . $self->{"number"} . '];
        ' . (defined $self->{"error"} ? '' : '$text = $_[1];' ) . '
        my $_savetext;
        @item = (q{' . $rule->{"name"} . '});
        %item = (__RULE__ => q{' . $rule->{"name"} . '});
        my $repcount = 0;

';
    $code .=
'        my @itempos = ({});
'           if $parser->{_check}{itempos};

    my $item;
    my $i;

    for ($i = 0; $i < @{$self->{"items"}}; $i++)
    {
        $item = ${$self->{items}}[$i];

        $code .= preitempos() if $parser->{_check}{itempos};

        $code .= $item->code($namespace,$rule,$parser->{_check});

        $code .= postitempos() if $parser->{_check}{itempos};

    }

    if ($parser->{_AUTOACTION} && defined($item) && !$item->isa("Parse::RecDescent::_Runtime::Action"))
    {
        $code .= $parser->{_AUTOACTION}->code($namespace,$rule);
        Parse::RecDescent::_Runtime::_warn(1,"Autogenerating action in rule
                       \"$rule->{name}\":
                        $parser->{_AUTOACTION}{code}")
        and
        Parse::RecDescent::_Runtime::_hint("The \$::RD_AUTOACTION was defined,
                      so any production not ending in an
                      explicit action has the specified
                      \"auto-action\" automatically
                      appended.");
    }
    elsif ($parser->{_AUTOTREE} && defined($item) && !$item->isa("Parse::RecDescent::_Runtime::Action"))
    {
        if ($i==1 && $item->isterminal)
        {
            $code .= $parser->{_AUTOTREE}{TERMINAL}->code($namespace,$rule);
        }
        else
        {
            $code .= $parser->{_AUTOTREE}{NODE}->code($namespace,$rule);
        }
        Parse::RecDescent::_Runtime::_warn(1,"Autogenerating tree-building action in rule
                       \"$rule->{name}\"")
        and
        Parse::RecDescent::_Runtime::_hint("The directive <autotree> was specified,
                      so any production not ending
                      in an explicit action has
                      some parse-tree building code
                      automatically appended.");
    }

    $code .=
'
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' production: ['
                      . $self->describe . ']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;

' . ( $parser->{_check}{itempos} ? '
        if ( defined($_itempos) )
        {
            Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itempos[ 1], undef, [qw(from)]);
            Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itempos[-1], undef, [qw(to)]);
        }
' : '' ) . '

        $_matched = 1;
        last;
    }

';
    return $code;
}

1;

package Parse::RecDescent::_Runtime::Action;

sub describe { undef }

sub sethashname { $_[0]->{hashname} = '__ACTION' . ++$_[1]->{actcount} .'__'; }

sub new
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "code"      => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
    }, $class;
}

sub issubrule { undef }
sub isterminal { 0 }

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        $_tok = ($_noactions) ? 0 : do ' . $self->{"code"} . ';
        ' . ($self->{"lookahead"}<0?'if':'unless') . ' (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        ' . ($self->{line}>=0 ? '$item{'. $self->{hashname} .'}=$_tok;' : '' ) .'
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
'
}


1;

package Parse::RecDescent::_Runtime::Directive;

sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{name} }

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "code"      => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
        "name"      => $_[4],
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: ['
                    . $self->describe . ']},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE; ' .'
        $_tok = do { ' . $self->{"code"} . ' };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        ' . ($self->{"lookahead"} ? '$text = $_savetext and ' : '' ) .'
        last '
        . ($self->{"lookahead"}<0?'if':'unless') . ' defined $_tok;
        push @item, $item{'.$self->{hashname}.'}=$_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
'
}

1;

package Parse::RecDescent::_Runtime::UncondReject;

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{name} }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub new ($$$;$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "lookahead" => $_[1],
        "line"      => $_[2],
        "name"      => $_[3],
    }, $class;
}

# MARK, YOU MAY WANT TO OPTIMIZE THIS.


sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found '
                     . $self->describe . ')},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        $_tok = undef;
        ' . ($self->{"lookahead"} ? '$text = $_savetext and ' : '' ) .'
        last '
        . ($self->{"lookahead"}<0?'if':'unless') . ' defined $_tok;
'
}

1;

package Parse::RecDescent::_Runtime::Error;

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{commitonly} ? '<error?:...>' : '<error...>' }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "msg"        => $_[1],
        "lookahead"  => $_[2],
        "commitonly" => $_[3],
        "line"       => $_[4],
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

    my $action = '';

    if ($self->{"msg"})  # ERROR MESSAGE SUPPLIED
    {
        #WAS: $action .= "Parse::RecDescent::_Runtime::_error(qq{$self->{msg}}" .  ',$thisline);';
        $action .= 'push @{$thisparser->{errors}}, [qq{'.$self->{msg}.'},$thisline];';

    }
    else      # GENERATE ERROR MESSAGE DURING PARSE
    {
        $action .= '
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        ';
    }

    my $dir =
          new Parse::RecDescent::_Runtime::Directive('if (' .
        ($self->{"commitonly"} ? '$commit' : '1') .
        ") { do {$action} unless ".' $_noactions; undef } else {0}',
                    $self->{"lookahead"},0,$self->describe);
    $dir->{hashname} = $self->{hashname};
    return $dir->code($namespace, $rule, 0);
}

1;

package Parse::RecDescent::_Runtime::Token;

sub sethashname { $_[0]->{hashname} = '__PATTERN' . ++$_[1]->{patcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'}}


# ARGS ARE: $self, $pattern, $left_delim, $modifiers, $lookahead, $linenum
sub new ($$$$$$)
{
    my $class = ref($_[0]) || $_[0];
    my $pattern = $_[1];
    my $pat = $_[1];
    my $ldel = $_[2];
    my $rdel = $ldel;
    $rdel =~ tr/{[(</}])>/;

    my $mod = $_[3];

    my $desc;

    if ($ldel eq '/') { $desc = "$ldel$pattern$rdel$mod" }
    else          { $desc = "m$ldel$pattern$rdel$mod" }
    $desc =~ s/\\/\\\\/g;
    $desc =~ s/\$$/\\\$/g;
    $desc =~ s/}/\\}/g;
    $desc =~ s/{/\\{/g;

    if (!eval "no strict;
           local \$SIG{__WARN__} = sub {0};
           '' =~ m$ldel$pattern$rdel$mod" and $@)
    {
        Parse::RecDescent::_Runtime::_warn(3, "Token pattern \"m$ldel$pattern$rdel$mod\"
                         may not be a valid regular expression",
                       $_[5]);
        $@ =~ s/ at \(eval.*/./;
        Parse::RecDescent::_Runtime::_hint($@);
    }

    # QUIETLY PREVENT (WELL-INTENTIONED) CALAMITY
    $mod =~ s/[gc]//g;
    $pattern =~ s/(\A|[^\\])\\G/$1/g;

    bless
    {
        "pattern"   => $pattern,
        "ldelim"      => $ldel,
        "rdelim"      => $rdel,
        "mod"         => $mod,
        "lookahead"   => $_[4],
        "line"        => $_[5],
        "description" => $desc,
    }, $class;
}


sub code($$$$$)
{
    my ($self, $namespace, $rule, $check) = @_;
    my $ldel = $self->{"ldelim"};
    my $rdel = $self->{"rdelim"};
    my $sdel = $ldel;
    my $mod  = $self->{"mod"};

    $sdel =~ s/[[{(<]/{}/;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  $text =~ m' . $ldel . '\A(?:' . $self->{"pattern"} . ')' . $rdel . $mod . ')
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;') .
            ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$current_match;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::Literal;

sub sethashname { $_[0]->{hashname} = '__STRING' . ++$_[1]->{strcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'} }

sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];

    my $pattern = $_[1];

    my $desc = $pattern;
    $desc=~s/\\/\\\\/g;
    $desc=~s/}/\\}/g;
    $desc=~s/{/\\{/g;

    bless
    {
        "pattern"     => $pattern,
        "lookahead"   => $_[2],
        "line"        => $_[3],
        "description" => "'$desc'",
    }, $class;
}


sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  $text =~ m/\A' . quotemeta($self->{"pattern"}) . '/)
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;').'
            '. ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$current_match;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::InterpLit;

sub sethashname { $_[0]->{hashname} = '__STRING' . ++$_[1]->{strcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'} }

sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];

    my $pattern = $_[1];
    $pattern =~ s#/#\\/#g;

    my $desc = $pattern;
    $desc=~s/\\/\\\\/g;
    $desc=~s/}/\\}/g;
    $desc=~s/{/\\{/g;

    bless
    {
        "pattern"   => $pattern,
        "lookahead" => $_[2],
        "line"      => $_[3],
        "description" => "'$desc'",
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  do { $_tok = "' . $self->{"pattern"} . '"; 1 } and
             substr($text,0,length($_tok)) eq $_tok and
             do { substr($text,0,length($_tok)) = ""; 1; }
        )
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;').'
            '. ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $_tok . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::Subrule;

sub issubrule ($) { return $_[0]->{"subrule"} }
sub isterminal { 0 }
sub sethashname {}

sub describe ($)
{
    my $desc = $_[0]->{"implicit"} || $_[0]->{"subrule"};
    $desc = "<matchrule:$desc>" if $_[0]->{"matchrule"};
    return $desc;
}

sub callsyntax($$)
{
    if ($_[0]->{"matchrule"})
    {
        return "&{'$_[1]'.qq{$_[0]->{subrule}}}";
    }
    else
    {
        return $_[1].$_[0]->{"subrule"};
    }
}

sub new ($$$$;$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "subrule"   => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
        "implicit"  => $_[4] || undef,
        "matchrule" => $_[5],
        "argcode"   => $_[6] || undef,
    }, $class;
}


sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [' . $self->{"subrule"} . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' )
        . ($self->{"lookahead"}<0?'if':'unless')
        . ' (defined ($_tok = '
        . $self->callsyntax($namespace.'::')
        . '($thisparser,$text,$repeating,'
        . ($self->{"lookahead"}?'1':'$_noactions')
        . ($self->{argcode} ? ",sub { return $self->{argcode} }"
                   : ',sub { \\@arg }')
        . ($check->{"itempos"}?',$itempos[$#itempos]':',undef')
        . ')))
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '').'
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' subrule: ['
            . $self->{subrule} . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' subrule: ['
                    . $self->{subrule} . ']<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{' . $self->{subrule} . '}} = $_tok;
        push @item, $_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
        }
'
}

package Parse::RecDescent::_Runtime::Repetition;

sub issubrule ($) { return $_[0]->{"subrule"} }
sub isterminal { 0 }
sub sethashname {  }

sub describe ($)
{
    my $desc = $_[0]->{"expected"} || $_[0]->{"subrule"};
    $desc = "<matchrule:$desc>" if $_[0]->{"matchrule"};
    return $desc;
}

sub callsyntax($$)
{
    if ($_[0]->{matchrule})
        { return "sub { goto &{''.qq{$_[1]$_[0]->{subrule}}} }"; }
    else
        { return "\\&$_[1]$_[0]->{subrule}"; }
}

sub new ($$$$$$$$$$)
{
    my ($self, $subrule, $repspec, $min, $max, $lookahead, $line, $parser, $matchrule, $argcode) = @_;
    my $class = ref($self) || $self;
    ($max, $min) = ( $min, $max) if ($max<$min);

    my $desc;
    if ($subrule=~/\A_alternation_\d+_of_production_\d+_of_rule/)
        { $desc = $parser->{"rules"}{$subrule}->expected }

    if ($lookahead)
    {
        if ($min>0)
        {
           return new Parse::RecDescent::_Runtime::Subrule($subrule,$lookahead,$line,$desc,$matchrule,$argcode);
        }
        else
        {
            Parse::RecDescent::_Runtime::_error("Not symbol (\"!\") before
                        \"$subrule\" doesn't make
                        sense.",$line);
            Parse::RecDescent::_Runtime::_hint("Lookahead for negated optional
                       repetitions (such as
                       \"!$subrule($repspec)\" can never
                       succeed, since optional items always
                       match (zero times at worst).
                       Did you mean a single \"!$subrule\",
                       instead?");
        }
    }
    bless
    {
        "subrule"   => $subrule,
        "repspec"   => $repspec,
        "min"       => $min,
        "max"       => $max,
        "lookahead" => $lookahead,
        "line"      => $line,
        "expected"  => $desc,
        "argcode"   => $argcode || undef,
        "matchrule" => $matchrule,
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

    my ($subrule, $repspec, $min, $max, $lookahead) =
        @{$self}{ qw{subrule repspec min max lookahead} };

'
        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [' . $self->describe . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'
        unless (defined ($_tok = $thisparser->_parserepeat($text, '
        . $self->callsyntax($namespace.'::')
        . ', ' . $min . ', ' . $max . ', '
        . ($self->{"lookahead"}?'1':'$_noactions')
        . ',$expectation,'
        . ($self->{argcode} ? "sub { return $self->{argcode} }"
                        : 'sub { \\@arg }')
        . ($check->{"itempos"}?',$itempos[$#itempos]':',undef')
        . ')))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' repeated subrule: ['
            . $self->describe . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' repeated subrule: ['
                    . $self->{subrule} . ']<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{' . "$self->{subrule}($self->{repspec})" . '}} = $_tok;
        push @item, $_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'

'
}

package Parse::RecDescent::_Runtime::Result;

sub issubrule { 0 }
sub isterminal { 0 }
sub describe { '' }

sub new
{
    my ($class, $pos) = @_;

    bless {}, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

    '
        $return = $item[-1];
    ';
}

package Parse::RecDescent::_Runtime::Operator;

my @opertype = ( " non-optional", "n optional" );

sub issubrule { 0 }
sub isterminal { 0 }

sub describe { $_[0]->{"expected"} }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }


sub new
{
    my ($class, $type, $minrep, $maxrep, $leftarg, $op, $rightarg) = @_;

    bless
    {
        "type"      => "${type}op",
        "leftarg"   => $leftarg,
        "op"        => $op,
        "min"       => $minrep,
        "max"       => $maxrep,
        "rightarg"  => $rightarg,
        "expected"  => "<${type}op: ".$leftarg->describe." ".$op->describe." ".$rightarg->describe.">",
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

    my @codeargs = @_[1..$#_];

    my ($leftarg, $op, $rightarg) =
        @{$self}{ qw{leftarg op rightarg} };

    my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [' . $self->describe . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;
';

    $code .= '
          my  $_itempos = $itempos[-1];
          my  $itemposfirst;
' if $check->{itempos};

    if ($self->{type} eq "leftop" )
    {
        $code .= '
          # MATCH LEFTARG
          ' . $leftarg->code(@codeargs) . '

';

        $code .= '
          if (defined($_itempos) and !defined($itemposfirst))
          {
              $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
          }
' if $check->{itempos};

        $code .= '
          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < ' . $self->{max} . ')
          {
            $backtrack = 0;
            ' . $op->code(@codeargs) . '
            ' . ($op->isterminal() ? 'pop @item;' : '$backtrack=1;' ) . '
            ' . (ref($op) eq 'Parse::RecDescent::_Runtime::Token'
                ? 'if (defined $1) {push @item, $item{'.($self->{name}||$self->{hashname}).'}=$1; $backtrack=1;}'
                : "" ) . '
            ' . $rightarg->code(@codeargs) . '
            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          ';
    }
    else
    {
        $code .= '
          my $savetext = $text;
          my $backtrack;
          # MATCH (LEFTARG OP)(s)
          while ($repcount < ' . $self->{max} . ')
          {
            $backtrack = 0;
            ' . $leftarg->code(@codeargs) . '
';
        $code .= '
            if (defined($_itempos) and !defined($itemposfirst))
            {
                $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
            }
' if $check->{itempos};

        $code .= '
            $repcount++;
            $backtrack = 1;
            ' . $op->code(@codeargs) . '
            $savetext = $text;
            ' . ($op->isterminal() ? 'pop @item;' : "" ) . '
            ' . (ref($op) eq 'Parse::RecDescent::_Runtime::Token' ? 'do { push @item, $item{'.($self->{name}||$self->{hashname}).'}=$1; } if defined $1;' : "" ) . '
          }
          $text = $savetext;
          pop @item if $backtrack;

          # MATCH RIGHTARG
          ' . $rightarg->code(@codeargs) . '
          $repcount++;
          ';
    }

    $code .= 'unless (@item) { undef $_tok; last }' unless $self->{min}==0;

    $code .= '
          $_tok = [ @item ];
';


    $code .= '
          if (defined $itemposfirst)
          {
              Parse::RecDescent::_Runtime::Production::_update_itempos(
                  $_itempos, $itemposfirst, undef, [qw(from)]);
          }
' if $check->{itempos};

    $code .= '
          last;
        } # end of OPLOOP
';

    $code .= '
        unless ($repcount>='.$self->{min}.')
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' operator: ['
                          . $self->describe
                          . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' operator: ['
                      . $self->describe
                      . ']<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{'.($self->{name}||$self->{hashname}).'}=$_tok||[];
';

    return $code;
}


package Parse::RecDescent::_Runtime::Expectation;

sub new ($)
{
    bless {
        "failed"      => 0,
        "expected"    => "",
        "unexpected"      => "",
        "lastexpected"    => "",
        "lastunexpected"  => "",
        "defexpected"     => $_[1],
          };
}

sub is ($$)
{
    $_[0]->{lastexpected} = $_[1]; return $_[0];
}

sub at ($$)
{
    $_[0]->{lastunexpected} = $_[1]; return $_[0];
}

sub failed ($)
{
    return unless $_[0]->{lastexpected};
    $_[0]->{expected}   = $_[0]->{lastexpected}   unless $_[0]->{failed};
    $_[0]->{unexpected} = $_[0]->{lastunexpected} unless $_[0]->{failed};
    $_[0]->{failed} = 1;
}

sub message ($)
{
    my ($self) = @_;
    $self->{expected} = $self->{defexpected} unless $self->{expected};
    $self->{expected} =~ s/_/ /g;
    if (!$self->{unexpected} || $self->{unexpected} =~ /\A\s*\Z/s)
    {
        return "Was expecting $self->{expected}";
    }
    else
    {
        $self->{unexpected} =~ /\s*(.*)/;
        return "Was expecting $self->{expected} but found \"$1\" instead";
    }
}

1;

package Parse::RecDescent::_Runtime;

use Carp;
use vars qw ( $AUTOLOAD $VERSION $_FILENAME);

my $ERRORS = 0;

our $VERSION = '1.967009';
$VERSION = eval $VERSION;
$_FILENAME=__FILE__;

# BUILDING A PARSER

my $nextnamespace = "namespace000001";

sub _nextnamespace()
{
    return "Parse::RecDescent::_Runtime::" . $nextnamespace++;
}

# ARGS ARE: $class, $grammar, $compiling, $namespace
sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];
    local $Parse::RecDescent::_Runtime::compiling = $_[2];
    my $name_space_name = defined $_[3]
        ? "Parse::RecDescent::_Runtime::".$_[3]
        : _nextnamespace();
    my $self =
    {
        "rules"     => {},
        "namespace" => $name_space_name,
        "startcode" => '',
        "localvars" => '',
        "_AUTOACTION" => undef,
        "_AUTOTREE"   => undef,

        # Precompiled parsers used to set _precompiled, but that
        # wasn't present in some versions of Parse::RecDescent::_Runtime used to
        # build precompiled parsers.  Instead, set a new
        # _not_precompiled flag, which is remove from future
        # Precompiled parsers at build time.
        "_not_precompiled" => 1,
    };


    if ($::RD_AUTOACTION) {
        my $sourcecode = $::RD_AUTOACTION;
        $sourcecode = "{ $sourcecode }"
            unless $sourcecode =~ /\A\s*\{.*\}\s*\Z/;
        $self->{_check}{itempos} =
            $sourcecode =~ /\@itempos\b|\$itempos\s*\[/;
        $self->{_AUTOACTION}
            = new Parse::RecDescent::_Runtime::Action($sourcecode,0,-1)
    }

    bless $self, $class;
    return $self->Replace($_[1])
}

sub Compile($$$$) {
    die "Compilation of Parse::RecDescent::_Runtime grammars not yet implemented\n";
}

sub DESTROY {
    my ($self) = @_;
    my $namespace = $self->{namespace};
    $namespace =~ s/Parse::RecDescent::_Runtime:://;
    if ($self->{_not_precompiled}) {
        # BEGIN WORKAROUND
        # Perl has a bug that creates a circular reference between
        # @ISA and that variable's stash:
        #   https://rt.perl.org/rt3/Ticket/Display.html?id=92708
        # Emptying the array before deleting the stash seems to
        # prevent the leak.  Once the ticket above has been resolved,
        # these two lines can be removed.
        no strict 'refs';
        @{$self->{namespace} . '::ISA'} = ();
        # END WORKAROUND

        # Some grammars may contain circular references between rules,
        # such as:
        #   a: 'ID' | b
        #   b: '(' a ')'
        # Unless these references are broken, the subs stay around on
        # stash deletion below.  Iterate through the stash entries and
        # for each defined code reference, set it to reference sub {}
        # instead.
        {
            local $^W; # avoid 'sub redefined' warnings.
            my $blank_sub = sub {};
            while (my ($name, $glob) = each %{"Parse::RecDescent::_Runtime::$namespace\::"}) {
                *$glob = $blank_sub if defined &$glob;
            }
        }

        # Delete the namespace's stash
        delete $Parse::RecDescent::_Runtime::{$namespace.'::'};
    }
}

# BUILDING A GRAMMAR....

# ARGS ARE: $self, $grammar, $isimplicit, $isleftop
sub Replace ($$)
{
    # set $replace = 1 for _generate
    splice(@_, 2, 0, 1);

    return _generate(@_);
}

# ARGS ARE: $self, $grammar, $isimplicit, $isleftop
sub Extend ($$)
{
    # set $replace = 0 for _generate
    splice(@_, 2, 0, 0);

    return _generate(@_);
}

sub _no_rule ($$;$)
{
    _error("Ruleless $_[0] at start of grammar.",$_[1]);
    my $desc = $_[2] ? "\"$_[2]\"" : "";
    _hint("You need to define a rule for the $_[0] $desc
           to be part of.");
}

my $NEGLOOKAHEAD    = '\G(\s*\.\.\.\!)';
my $POSLOOKAHEAD    = '\G(\s*\.\.\.)';
my $RULE        = '\G\s*(\w+)[ \t]*:';
my $PROD        = '\G\s*([|])';
my $TOKEN       = q{\G\s*/((\\\\/|\\\\\\\\|[^/])*)/([cgimsox]*)};
my $MTOKEN      = q{\G\s*(m\s*[^\w\s])};
my $LITERAL     = q{\G\s*'((\\\\['\\\\]|[^'])*)'};
my $INTERPLIT       = q{\G\s*"((\\\\["\\\\]|[^"])*)"};
my $SUBRULE     = '\G\s*(\w+)';
my $MATCHRULE       = '\G(\s*<matchrule:)';
my $SIMPLEPAT       = '((\\s+/[^/\\\\]*(?:\\\\.[^/\\\\]*)*/)?)';
my $OPTIONAL        = '\G\((\?)'.$SIMPLEPAT.'\)';
my $ANY         = '\G\((s\?)'.$SIMPLEPAT.'\)';
my $MANY        = '\G\((s|\.\.)'.$SIMPLEPAT.'\)';
my $EXACTLY     = '\G\(([1-9]\d*)'.$SIMPLEPAT.'\)';
my $BETWEEN     = '\G\((\d+)\.\.([1-9]\d*)'.$SIMPLEPAT.'\)';
my $ATLEAST     = '\G\((\d+)\.\.'.$SIMPLEPAT.'\)';
my $ATMOST      = '\G\(\.\.([1-9]\d*)'.$SIMPLEPAT.'\)';
my $BADREP      = '\G\((-?\d+)?\.\.(-?\d+)?'.$SIMPLEPAT.'\)';
my $ACTION      = '\G\s*\{';
my $IMPLICITSUBRULE = '\G\s*\(';
my $COMMENT     = '\G\s*(#.*)';
my $COMMITMK        = '\G\s*<commit>';
my $UNCOMMITMK      = '\G\s*<uncommit>';
my $QUOTELIKEMK     = '\G\s*<perl_quotelike>';
my $CODEBLOCKMK     = '\G\s*<perl_codeblock(?:\s+([][()<>{}]+))?>';
my $VARIABLEMK      = '\G\s*<perl_variable>';
my $NOCHECKMK       = '\G\s*<nocheck>';
my $AUTOACTIONPATMK = '\G\s*<autoaction:';
my $AUTOTREEMK      = '\G\s*<autotree(?::\s*([\w:]+)\s*)?>';
my $AUTOSTUBMK      = '\G\s*<autostub>';
my $AUTORULEMK      = '\G\s*<autorule:(.*?)>';
my $REJECTMK        = '\G\s*<reject>';
my $CONDREJECTMK    = '\G\s*<reject:';
my $SCOREMK     = '\G\s*<score:';
my $AUTOSCOREMK     = '\G\s*<autoscore:';
my $SKIPMK      = '\G\s*<skip:';
my $OPMK        = '\G\s*<(left|right)op(?:=(\'.*?\'))?:';
my $ENDDIRECTIVEMK  = '\G\s*>';
my $RESYNCMK        = '\G\s*<resync>';
my $RESYNCPATMK     = '\G\s*<resync:';
my $RULEVARPATMK    = '\G\s*<rulevar:';
my $DEFERPATMK      = '\G\s*<defer:';
my $TOKENPATMK      = '\G\s*<token:';
my $AUTOERRORMK     = '\G\s*<error(\??)>';
my $MSGERRORMK      = '\G\s*<error(\??):';
my $NOCHECK     = '\G\s*<nocheck>';
my $WARNMK      = '\G\s*<warn((?::\s*(\d+)\s*)?)>';
my $HINTMK      = '\G\s*<hint>';
my $TRACEBUILDMK    = '\G\s*<trace_build((?::\s*(\d+)\s*)?)>';
my $TRACEPARSEMK    = '\G\s*<trace_parse((?::\s*(\d+)\s*)?)>';
my $UNCOMMITPROD    = $PROD.'\s*<uncommit';
my $ERRORPROD       = $PROD.'\s*<error';
my $LONECOLON       = '\G\s*:';
my $OTHER       = '\G\s*([^\s]+)';

my @lines = 0;

sub _generate
{
    my ($self, $grammar, $replace, $isimplicit, $isleftop) = (@_, 0);

    my $aftererror = 0;
    my $lookahead = 0;
    my $lookaheadspec = "";
    my $must_pop_lines;
    if (! $lines[-1]) {
        push @lines, _linecount($grammar) ;
        $must_pop_lines = 1;
    }
    $self->{_check}{itempos} = ($grammar =~ /\@itempos\b|\$itempos\s*\[/)
        unless $self->{_check}{itempos};
    for (qw(thisoffset thiscolumn prevline prevoffset prevcolumn))
    {
        $self->{_check}{$_} =
            ($grammar =~ /\$$_/) || $self->{_check}{itempos}
                unless $self->{_check}{$_};
    }
    my $line;

    my $rule = undef;
    my $prod = undef;
    my $item = undef;
    my $lastgreedy = '';
    pos $grammar = 0;
    study $grammar;

    local $::RD_HINT  = $::RD_HINT;
    local $::RD_WARN  = $::RD_WARN;
    local $::RD_TRACE = $::RD_TRACE;
    local $::RD_CHECK = $::RD_CHECK;

    while (pos $grammar < length $grammar)
    {
        $line = $lines[-1] - _linecount($grammar) + 1;
        my $commitonly;
        my $code = "";
        my @components = ();
        if ($grammar =~ m/$COMMENT/gco)
        {
            _parse("a comment",0,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            next;
        }
        elsif ($grammar =~ m/$NEGLOOKAHEAD/gco)
        {
            _parse("a negative lookahead",$aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $lookahead = $lookahead ? -$lookahead : -1;
            $lookaheadspec .= $1;
            next;   # SKIP LOOKAHEAD RESET AT END OF while LOOP
        }
        elsif ($grammar =~ m/$POSLOOKAHEAD/gco)
        {
            _parse("a positive lookahead",$aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $lookahead = $lookahead ? $lookahead : 1;
            $lookaheadspec .= $1;
            next;   # SKIP LOOKAHEAD RESET AT END OF while LOOP
        }
        elsif ($grammar =~ m/(?=$ACTION)/gco
            and do { ($code) = extract_codeblock($grammar); $code })
        {
            _parse("an action", $aftererror, $line, $code);
            $item = new Parse::RecDescent::_Runtime::Action($code,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  $self->_addstartcode($code);
        }
        elsif ($grammar =~ m/(?=$IMPLICITSUBRULE)/gco
            and do { ($code) = extract_codeblock($grammar,'{([',undef,'(',1);
                $code })
        {
            $code =~ s/\A\s*\(|\)\Z//g;
            _parse("an implicit subrule", $aftererror, $line,
                "( $code )");
            my $implicit = $rule->nextimplicit;
            return undef
                if !$self->_generate("$implicit : $code",$replace,1);
            my $pos = pos $grammar;
            substr($grammar,$pos,0,$implicit);
            pos $grammar = $pos;;
        }
        elsif ($grammar =~ m/$ENDDIRECTIVEMK/gco)
        {

        # EXTRACT TRAILING REPETITION SPECIFIER (IF ANY)

            my ($minrep,$maxrep) = (1,$MAXREP);
            if ($grammar =~ m/\G[(]/gc)
            {
                pos($grammar)--;

                if ($grammar =~ m/$OPTIONAL/gco)
                    { ($minrep, $maxrep) = (0,1) }
                elsif ($grammar =~ m/$ANY/gco)
                    { $minrep = 0 }
                elsif ($grammar =~ m/$EXACTLY/gco)
                    { ($minrep, $maxrep) = ($1,$1) }
                elsif ($grammar =~ m/$BETWEEN/gco)
                    { ($minrep, $maxrep) = ($1,$2) }
                elsif ($grammar =~ m/$ATLEAST/gco)
                    { $minrep = $1 }
                elsif ($grammar =~ m/$ATMOST/gco)
                    { $maxrep = $1 }
                elsif ($grammar =~ m/$MANY/gco)
                    { }
                elsif ($grammar =~ m/$BADREP/gco)
                {
                    _parse("an invalid repetition specifier", 0,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                    _error("Incorrect specification of a repeated directive",
                           $line);
                    _hint("Repeated directives cannot have
                           a maximum repetition of zero, nor can they have
                           negative components in their ranges.");
                }
            }

            $prod && $prod->enddirective($line,$minrep,$maxrep);
        }
        elsif ($grammar =~ m/\G\s*<[^m]/gc)
        {
            pos($grammar)-=2;

            if ($grammar =~ m/$OPMK/gco)
            {
                # $DB::single=1;
                _parse("a $1-associative operator directive", $aftererror, $line, "<$1op:...>");
                $prod->adddirective($1, $line,$2||'');
            }
            elsif ($grammar =~ m/$UNCOMMITMK/gco)
            {
                _parse("an uncommit marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive('$commit=0;1',
                                  $lookahead,$line,"<uncommit>");
                $prod and $prod->additem($item)
                      or  _no_rule("<uncommit>",$line);
            }
            elsif ($grammar =~ m/$QUOTELIKEMK/gco)
            {
                _parse("an perl quotelike marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'my ($match,@res);
                     ($match,$text,undef,@res) =
                          Text::Balanced::extract_quotelike($text,$skip);
                      $match ? \@res : undef;
                    ', $lookahead,$line,"<perl_quotelike>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_quotelike>",$line);
            }
            elsif ($grammar =~ m/$CODEBLOCKMK/gco)
            {
                my $outer = $1||"{}";
                _parse("an perl codeblock marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'Text::Balanced::extract_codeblock($text,undef,$skip,\''.$outer.'\');
                    ', $lookahead,$line,"<perl_codeblock>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_codeblock>",$line);
            }
            elsif ($grammar =~ m/$VARIABLEMK/gco)
            {
                _parse("an perl variable marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'Text::Balanced::extract_variable($text,$skip);
                    ', $lookahead,$line,"<perl_variable>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_variable>",$line);
            }
            elsif ($grammar =~ m/$NOCHECKMK/gco)
            {
                _parse("a disable checking marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                if ($rule)
                {
                    _error("<nocheck> directive not at start of grammar", $line);
                    _hint("The <nocheck> directive can only
                           be specified at the start of a
                           grammar (before the first rule
                           is defined.");
                }
                else
                {
                    local $::RD_CHECK = 1;
                }
            }
            elsif ($grammar =~ m/$AUTOSTUBMK/gco)
            {
                _parse("an autostub marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $::RD_AUTOSTUB = "";
            }
            elsif ($grammar =~ m/$AUTORULEMK/gco)
            {
                _parse("an autorule marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $::RD_AUTOSTUB = $1;
            }
            elsif ($grammar =~ m/$AUTOTREEMK/gco)
            {
                my $base = defined($1) ? $1 : "";
                my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
                $base .= "::" if $base && $base !~ /::$/;
                _parse("an autotree marker", $aftererror,$line, $current_match);
                if ($rule)
                {
                    _error("<autotree> directive not at start of grammar", $line);
                    _hint("The <autotree> directive can only
                           be specified at the start of a
                           grammar (before the first rule
                           is defined.");
                }
                else
                {
                    undef $self->{_AUTOACTION};
                    $self->{_AUTOTREE}{NODE}
                        = new Parse::RecDescent::_Runtime::Action(q({bless \%item, ').$base.q('.$item[0]}),0,-1);
                    $self->{_AUTOTREE}{TERMINAL}
                        = new Parse::RecDescent::_Runtime::Action(q({bless {__VALUE__=>$item[1]}, ').$base.q('.$item[0]}),0,-1);
                }
            }

            elsif ($grammar =~ m/$REJECTMK/gco)
            {
                _parse("an reject marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,"<reject>");
                $prod and $prod->additem($item)
                      or  _no_rule("<reject>",$line);
            }
            elsif ($grammar =~ m/(?=$CONDREJECTMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                      $code })
            {
                _parse("a (conditional) reject marker", $aftererror,$line, $code );
                $code =~ /\A\s*<reject:(.*)>\Z/s;
                my $cond = $1;
                $item = new Parse::RecDescent::_Runtime::Directive(
                          "($1) ? undef : 1", $lookahead,$line,"<reject:$cond>");
                $prod and $prod->additem($item)
                      or  _no_rule("<reject:$cond>",$line);
            }
            elsif ($grammar =~ m/(?=$SCOREMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                      $code })
            {
                _parse("a score marker", $aftererror,$line, $code );
                $code =~ /\A\s*<score:(.*)>\Z/s;
                $prod and $prod->addscore($1, $lookahead, $line)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$AUTOSCOREMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("an autoscore specifier", $aftererror,$line,$code);
                $code =~ /\A\s*<autoscore:(.*)>\Z/s;

                $rule and $rule->addautoscore($1,$self)
                      or  _no_rule($code,$line);

                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/$RESYNCMK/gco)
            {
                _parse("a resync to newline marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                          'if ($text =~ s/(\A[^\n]*\n)//) { $return = 0; $1; } else { undef }',
                          $lookahead,$line,"<resync>");
                $prod and $prod->additem($item)
                      or  _no_rule("<resync>",$line);
            }
            elsif ($grammar =~ m/(?=$RESYNCPATMK)/gco
                and do { ($code) = extract_bracketed($grammar,'<');
                      $code })
            {
                _parse("a resync with pattern marker", $aftererror,$line, $code );
                $code =~ /\A\s*<resync:(.*)>\Z/s;
                $item = new Parse::RecDescent::_Runtime::Directive(
                          'if ($text =~ s/(\A'.$1.')//) { $return = 0; $1; } else { undef }',
                          $lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$SKIPMK)/gco
                and do { ($code) = extract_codeblock($grammar,'<');
                      $code })
            {
                _parse("a skip marker", $aftererror,$line, $code );
                $code =~ /\A\s*<skip:(.*)>\Z/s;
                if ($rule) {
                    $item = new Parse::RecDescent::_Runtime::Directive(
                        'my $oldskip = $skip; $skip='.$1.'; $oldskip',
                        $lookahead,$line,$code);
                    $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
                } else {
                    #global <skip> directive
                    $self->{skip} = $1;
                }
            }
            elsif ($grammar =~ m/(?=$RULEVARPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a rule variable specifier", $aftererror,$line,$code);
                $code =~ /\A\s*<rulevar:(.*)>\Z/s;

                $rule and $rule->addvar($1,$self)
                      or  _no_rule($code,$line);

                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$AUTOACTIONPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("an autoaction specifier", $aftererror,$line,$code);
                $code =~ s/\A\s*<autoaction:(.*)>\Z/$1/s;
                if ($code =~ /\A\s*[^{]|[^}]\s*\Z/) {
                    $code = "{ $code }"
                }
        $self->{_check}{itempos} =
            $code =~ /\@itempos\b|\$itempos\s*\[/;
        $self->{_AUTOACTION}
            = new Parse::RecDescent::_Runtime::Action($code,0,-$line)
            }
            elsif ($grammar =~ m/(?=$DEFERPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a deferred action specifier", $aftererror,$line,$code);
                $code =~ s/\A\s*<defer:(.*)>\Z/$1/s;
                if ($code =~ /\A\s*[^{]|[^}]\s*\Z/)
                {
                    $code = "{ $code }"
                }

                $item = new Parse::RecDescent::_Runtime::Directive(
                          "push \@{\$thisparser->{deferred}}, sub $code;",
                          $lookahead,$line,"<defer:$code>");
                $prod and $prod->additem($item)
                      or  _no_rule("<defer:$code>",$line);

                $self->{deferrable} = 1;
            }
            elsif ($grammar =~ m/(?=$TOKENPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a token constructor", $aftererror,$line,$code);
                $code =~ s/\A\s*<token:(.*)>\Z/$1/s;

                my $types = eval 'no strict; local $SIG{__WARN__} = sub {0}; my @arr=('.$code.'); @arr' || ();
                if (!$types)
                {
                    _error("Incorrect token specification: \"$@\"", $line);
                    _hint("The <token:...> directive requires a list
                           of one or more strings representing possible
                           types of the specified token. For example:
                           <token:NOUN,VERB>");
                }
                else
                {
                    $item = new Parse::RecDescent::_Runtime::Directive(
                              'no strict;
                               $return = { text => $item[-1] };
                               @{$return->{type}}{'.$code.'} = (1..'.$types.');',
                              $lookahead,$line,"<token:$code>");
                    $prod and $prod->additem($item)
                          or  _no_rule("<token:$code>",$line);
                }
            }
            elsif ($grammar =~ m/$COMMITMK/gco)
            {
                _parse("an commit marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive('$commit = 1',
                                  $lookahead,$line,"<commit>");
                $prod and $prod->additem($item)
                      or  _no_rule("<commit>",$line);
            }
            elsif ($grammar =~ m/$NOCHECKMK/gco) {
                _parse("an hint request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_CHECK = 0;
        }
            elsif ($grammar =~ m/$HINTMK/gco) {
                _parse("an hint request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_HINT = $self->{__HINT__} = 1;
        }
            elsif ($grammar =~ m/$WARNMK/gco) {
                _parse("an warning request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_WARN = $self->{__WARN__} = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$TRACEBUILDMK/gco) {
                _parse("an grammar build trace request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_TRACE = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$TRACEPARSEMK/gco) {
                _parse("an parse trace request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $self->{__TRACE__} = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$AUTOERRORMK/gco)
            {
                $commitonly = $1;
                _parse("an error marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Error('',$lookahead,$1,$line);
                $prod and $prod->additem($item)
                      or  _no_rule("<error>",$line);
                $aftererror = !$commitonly;
            }
            elsif ($grammar =~ m/(?=$MSGERRORMK)/gco
                and do { $commitonly = $1;
                     ($code) = extract_bracketed($grammar,'<');
                    $code })
            {
                _parse("an error marker", $aftererror,$line,$code);
                $code =~ /\A\s*<error\??:(.*)>\Z/s;
                $item = new Parse::RecDescent::_Runtime::Error($1,$lookahead,$commitonly,$line);
                $prod and $prod->additem($item)
                      or  _no_rule("$code",$line);
                $aftererror = !$commitonly;
            }
            elsif (do { $commitonly = $1;
                     ($code) = extract_bracketed($grammar,'<');
                    $code })
            {
                if ($code =~ /^<[A-Z_]+>$/)
                {
                    _error("Token items are not yet
                    supported: \"$code\"",
                           $line);
                    _hint("Items like $code that consist of angle
                    brackets enclosing a sequence of
                    uppercase characters will eventually
                    be used to specify pre-lexed tokens
                    in a grammar. That functionality is not
                    yet implemented. Or did you misspell
                    \"$code\"?");
                }
                else
                {
                    _error("Untranslatable item encountered: \"$code\"",
                           $line);
                    _hint("Did you misspell \"$code\"
                           or forget to comment it out?");
                }
            }
        }
        elsif ($grammar =~ m/$RULE/gco)
        {
            _parseunneg("a rule declaration", 0,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            my $rulename = $1;
            if ($rulename =~ /Replace|Extend|Precompile|Save/ )
            {
                _warn(2,"Rule \"$rulename\" hidden by method
                       Parse::RecDescent::_Runtime::$rulename",$line)
                and
                _hint("The rule named \"$rulename\" cannot be directly
                       called through the Parse::RecDescent::_Runtime object
                       for this grammar (although it may still
                       be used as a subrule of other rules).
                       It can't be directly called because
                       Parse::RecDescent::_Runtime::$rulename is already defined (it
                       is the standard method of all
                       parsers).");
            }
            $rule = new Parse::RecDescent::_Runtime::Rule($rulename,$self,$line,$replace);
            $prod->check_pending($line) if $prod;
            $prod = $rule->addprod( new Parse::RecDescent::_Runtime::Production );
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$UNCOMMITPROD/gco)
        {
            pos($grammar)-=9;
            _parseunneg("a new (uncommitted) production",
                    0, $lookahead, $line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;

            $prod->check_pending($line) if $prod;
            $prod = new Parse::RecDescent::_Runtime::Production($line,1);
            $rule and $rule->addprod($prod)
                  or  _no_rule("<uncommit>",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$ERRORPROD/gco)
        {
            pos($grammar)-=6;
            _parseunneg("a new (error) production", $aftererror,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            $prod->check_pending($line) if $prod;
            $prod = new Parse::RecDescent::_Runtime::Production($line,0,1);
            $rule and $rule->addprod($prod)
                  or  _no_rule("<error>",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$PROD/gco)
        {
            _parseunneg("a new production", 0,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            $rule
              and (!$prod || $prod->check_pending($line))
              and $prod = $rule->addprod(new Parse::RecDescent::_Runtime::Production($line))
            or  _no_rule("production",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$LITERAL/gco)
        {
            my $literal = $1;
            ($code = $literal) =~ s/\\\\/\\/g;
            _parse("a literal terminal", $aftererror,$line,$literal);
            $item = new Parse::RecDescent::_Runtime::Literal($code,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("literal terminal",$line,"'$literal'");
        }
        elsif ($grammar =~ m/$INTERPLIT/gco)
        {
            _parse("an interpolated literal terminal", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $item = new Parse::RecDescent::_Runtime::InterpLit($1,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("interpolated literal terminal",$line,"'$1'");
        }
        elsif ($grammar =~ m/$TOKEN/gco)
        {
            _parse("a /../ pattern terminal", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $item = new Parse::RecDescent::_Runtime::Token($1,'/',$3?$3:'',$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("pattern terminal",$line,"/$1/");
        }
        elsif ($grammar =~ m/(?=$MTOKEN)/gco
            and do { ($code, undef, @components)
                    = extract_quotelike($grammar);
                 $code }
              )

        {
            _parse("an m/../ pattern terminal", $aftererror,$line,$code);
            $item = new Parse::RecDescent::_Runtime::Token(@components[3,2,8],
                                 $lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("pattern terminal",$line,$code);
        }
        elsif ($grammar =~ m/(?=$MATCHRULE)/gco
                and do { ($code) = extract_bracketed($grammar,'<');
                     $code
                       }
               or $grammar =~ m/$SUBRULE/gco
                and $code = $1)
        {
            my $name = $code;
            my $matchrule = 0;
            if (substr($name,0,1) eq '<')
            {
                $name =~ s/$MATCHRULE\s*//;
                $name =~ s/\s*>\Z//;
                $matchrule = 1;
            }

        # EXTRACT TRAILING ARG LIST (IF ANY)

            my ($argcode) = extract_codeblock($grammar, "[]",'') || '';

        # EXTRACT TRAILING REPETITION SPECIFIER (IF ANY)

            if ($grammar =~ m/\G[(]/gc)
            {
                pos($grammar)--;

                if ($grammar =~ m/$OPTIONAL/gco)
                {
                    _parse("an zero-or-one subrule match", $aftererror,$line,"$code$argcode($1)");
                    $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,0,1,
                                       $lookahead,$line,
                                       $self,
                                       $matchrule,
                                       $argcode);
                    $prod and $prod->additem($item)
                          or  _no_rule("repetition",$line,"$code$argcode($1)");

                    !$matchrule and $rule and $rule->addcall($name);
                }
                elsif ($grammar =~ m/$ANY/gco)
                {
                    _parse("a zero-or-more subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(s?)': $name $2 $name>(s?) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,0,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);

                        _check_insatiable($name,$1,$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$MANY/gco)
                {
                    _parse("a one-or-more subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        # $DB::single=1;
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(s)': $name $2 $name> ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,1,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);

                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);

                        _check_insatiable($name,$1,$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$EXACTLY/gco)
                {
                    _parse("an exactly-$1-times subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1)': $name $2 $name>($1) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,$1,$1,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$BETWEEN/gco)
                {
                    _parse("a $1-to-$2 subrule match", $aftererror,$line,"$code$argcode($1..$2)");
                    if ($3)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1..$2)': $name $3 $name>($1..$2) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"$1..$2",$1,$2,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1..$2)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$ATLEAST/gco)
                {
                    _parse("a $1-or-more subrule match", $aftererror,$line,"$code$argcode($1..)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1..)': $name $2 $name>($1..) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"$1..",$1,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1..)");

                        !$matchrule and $rule and $rule->addcall($name);
                        _check_insatiable($name,"$1..",$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$ATMOST/gco)
                {
                    _parse("a one-to-$1 subrule match", $aftererror,$line,"$code$argcode(..$1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(..$1)': $name $2 $name>(..$1) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"..$1",1,$1,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode(..$1)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$BADREP/gco)
                {
                    my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
                    _parse("an subrule match with invalid repetition specifier", 0,$line, $current_match);
                    _error("Incorrect specification of a repeated subrule",
                           $line);
                    _hint("Repeated subrules like \"$code$argcode$current_match\" cannot have
                           a maximum repetition of zero, nor can they have
                           negative components in their ranges.");
                }
            }
            else
            {
                _parse("a subrule match", $aftererror,$line,$code);
                my $desc;
                if ($name=~/\A_alternation_\d+_of_production_\d+_of_rule/)
                    { $desc = $self->{"rules"}{$name}->expected }
                $item = new Parse::RecDescent::_Runtime::Subrule($name,
                                       $lookahead,
                                       $line,
                                       $desc,
                                       $matchrule,
                                       $argcode);

                $prod and $prod->additem($item)
                      or  _no_rule("(sub)rule",$line,$name);

                !$matchrule and $rule and $rule->addcall($name);
            }
        }
        elsif ($grammar =~ m/$LONECOLON/gco   )
        {
            _error("Unexpected colon encountered", $line);
            _hint("Did you mean \"|\" (to start a new production)?
                   Or perhaps you forgot that the colon
                   in a rule definition must be
                   on the same line as the rule name?");
        }
        elsif ($grammar =~ m/$ACTION/gco   ) # BAD ACTION, ALREADY FAILED
        {
            _error("Malformed action encountered",
                   $line);
            _hint("Did you forget the closing curly bracket
                   or is there a syntax error in the action?");
        }
        elsif ($grammar =~ m/$OTHER/gco   )
        {
            _error("Untranslatable item encountered: \"$1\"",
                   $line);
            _hint("Did you misspell \"$1\"
                   or forget to comment it out?");
        }

        if ($lookaheadspec =~ tr /././ > 3)
        {
            $lookaheadspec =~ s/\A\s+//;
            $lookahead = $lookahead<0
                    ? 'a negative lookahead ("...!")'
                    : 'a positive lookahead ("...")' ;
            _warn(1,"Found two or more lookahead specifiers in a
                   row.",$line)
            and
            _hint("Multiple positive and/or negative lookaheads
                   are simply multiplied together to produce a
                   single positive or negative lookahead
                   specification. In this case the sequence
                   \"$lookaheadspec\" was reduced to $lookahead.
                   Was this your intention?");
        }
        $lookahead = 0;
        $lookaheadspec = "";

        $grammar =~ m/\G\s+/gc;
    }

    if ($must_pop_lines) {
        pop @lines;
    }

    unless ($ERRORS or $isimplicit or !$::RD_CHECK)
    {
        $self->_check_grammar();
    }

    unless ($ERRORS or $isimplicit or $Parse::RecDescent::_Runtime::compiling)
    {
        my $code = $self->_code();
        if (defined $::RD_TRACE)
        {
            my $mode = ($nextnamespace eq "namespace000002") ? '>' : '>>';
            print STDERR "printing code (", length($code),") to RD_TRACE\n";
            local *TRACE_FILE;
            open TRACE_FILE, $mode, "RD_TRACE"
            and print TRACE_FILE "my \$ERRORS;\n$code"
            and close TRACE_FILE;
        }

        unless ( eval "$code 1" )
        {
            _error("Internal error in generated parser code!");
            $@ =~ s/at grammar/in grammar at/;
            _hint($@);
        }
    }

    if ($ERRORS and !_verbosity("HINT"))
    {
        local $::RD_HINT = defined $::RD_HINT ? $::RD_HINT : 1;
        _hint('Set $::RD_HINT (or -RD_HINT if you\'re using "perl -s")
               for hints on fixing these problems.  Use $::RD_HINT = 0
               to disable this message.');
    }
    if ($ERRORS) { $ERRORS=0; return }
    return $self;
}


sub _addstartcode($$)
{
    my ($self, $code) = @_;
    $code =~ s/\A\s*\{(.*)\}\Z/$1/s;

    $self->{"startcode"} .= "$code;\n";
}

# CHECK FOR GRAMMAR PROBLEMS....

sub _check_insatiable($$$$)
{
    my ($subrule,$repspec,$grammar,$line) = @_;
    pos($grammar)=pos($_[2]);
    return if $grammar =~ m/$OPTIONAL/gco || $grammar =~ m/$ANY/gco;
    my $min = 1;
    if ( $grammar =~ m/$MANY/gco
      || $grammar =~ m/$EXACTLY/gco
      || $grammar =~ m/$ATMOST/gco
      || $grammar =~ m/$BETWEEN/gco && do { $min=$2; 1 }
      || $grammar =~ m/$ATLEAST/gco && do { $min=$2; 1 }
      || $grammar =~ m/$SUBRULE(?!\s*:)/gco
       )
    {
        return unless $1 eq $subrule && $min > 0;
        my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
        _warn(3,"Subrule sequence \"$subrule($repspec) $current_match\" will
               (almost certainly) fail.",$line)
        and
        _hint("Unless subrule \"$subrule\" performs some cunning
               lookahead, the repetition \"$subrule($repspec)\" will
               insatiably consume as many matches of \"$subrule\" as it
               can, leaving none to match the \"$current_match\" that follows.");
    }
}

sub _check_grammar ($)
{
    my $self = shift;
    my $rules = $self->{"rules"};
    my $rule;
    foreach $rule ( values %$rules )
    {
        next if ! $rule->{"changed"};

    # CHECK FOR UNDEFINED RULES

        my $call;
        foreach $call ( @{$rule->{"calls"}} )
        {
            if (!defined ${$rules}{$call}
              &&!defined &{"Parse::RecDescent::_Runtime::$call"})
            {
                if (!defined $::RD_AUTOSTUB)
                {
                    _warn(3,"Undefined (sub)rule \"$call\"
                          used in a production.")
                    and
                    _hint("Will you be providing this rule
                           later, or did you perhaps
                           misspell \"$call\"? Otherwise
                           it will be treated as an
                           immediate <reject>.");
                    eval "sub $self->{namespace}::$call {undef}";
                }
                else    # EXPERIMENTAL
                {
                    my $rule = qq{'$call'};
                    if ($::RD_AUTOSTUB and $::RD_AUTOSTUB ne "1") {
                        $rule = $::RD_AUTOSTUB;
                    }
                    _warn(1,"Autogenerating rule: $call")
                    and
                    _hint("A call was made to a subrule
                           named \"$call\", but no such
                           rule was specified. However,
                           since \$::RD_AUTOSTUB
                           was defined, a rule stub
                           ($call : $rule) was
                           automatically created.");

                    $self->_generate("$call: $rule",0,1);
                }
            }
        }

    # CHECK FOR LEFT RECURSION

        if ($rule->isleftrec($rules))
        {
            _error("Rule \"$rule->{name}\" is left-recursive.");
            _hint("Redesign the grammar so it's not left-recursive.
                   That will probably mean you need to re-implement
                   repetitions using the '(s)' notation.
                   For example: \"$rule->{name}(s)\".");
            next;
        }

    # CHECK FOR PRODUCTIONS FOLLOWING EMPTY PRODUCTIONS
      {
          my $hasempty;
          my $prod;
          foreach $prod ( @{$rule->{"prods"}} ) {
              if ($hasempty) {
                  _error("Production " . $prod->describe . " for \"$rule->{name}\"
                         will never be reached (preceding empty production will
                         always match first).");
                  _hint("Reorder the grammar so that the empty production
                         is last in the list or productions.");
                  last;
              }
              $hasempty ||= $prod->isempty();
          }
      }
    }
}

# GENERATE ACTUAL PARSER CODE

sub _code($)
{
    my $self = shift;
    my $initial_skip = defined($self->{skip}) ? $self->{skip} : $skip;

    my $code = qq!
package $self->{namespace};
use strict;
use vars qw(\$skip \$AUTOLOAD $self->{localvars} );
\@$self->{namespace}\::ISA = ();
\$skip = '$initial_skip';
$self->{startcode}

{
local \$SIG{__WARN__} = sub {0};
# PRETEND TO BE IN Parse::RecDescent::_Runtime NAMESPACE
*$self->{namespace}::AUTOLOAD   = sub
{
    no strict 'refs';
!
# This generated code uses ${"AUTOLOAD"} rather than $AUTOLOAD in
# order to avoid the circular reference documented here:
#    https://rt.perl.org/rt3/Public/Bug/Display.html?id=110248
# As a result of the investigation of
#    https://rt.cpan.org/Ticket/Display.html?id=53710
. qq!
    \${"AUTOLOAD"} =~ s/^$self->{namespace}/Parse::RecDescent::_Runtime/;
    goto &{\${"AUTOLOAD"}};
}
}

!;
    $code .= "push \@$self->{namespace}\::ISA, 'Parse::RecDescent::_Runtime';";
    $self->{"startcode"} = '';

    my $rule;
    foreach $rule ( values %{$self->{"rules"}} )
    {
        if ($rule->{"changed"})
        {
            $code .= $rule->code($self->{"namespace"},$self);
            $rule->{"changed"} = 0;
        }
    }

    return $code;
}


# EXECUTING A PARSE....

sub AUTOLOAD    # ($parser, $text; $linenum, @args)
{
    croak "Could not find method: $AUTOLOAD\n" unless ref $_[0];
    my $class = ref($_[0]) || $_[0];
    my $text = ref($_[1]) eq 'SCALAR' ? ${$_[1]} : "$_[1]";
    $_[0]->{lastlinenum} = _linecount($text);
    $_[0]->{lastlinenum} += ($_[2]||0) if @_ > 2;
    $_[0]->{offsetlinenum} = $_[0]->{lastlinenum};
    $_[0]->{fulltext} = $text;
    $_[0]->{fulltextlen} = length $text;
    $_[0]->{linecounter_cache} = {};
    $_[0]->{deferred} = [];
    $_[0]->{errors} = [];
    my @args = @_[3..$#_];
    my $args = sub { [ @args ] };

    $AUTOLOAD =~ s/$class/$_[0]->{namespace}/;
    no strict "refs";

    local $::RD_WARN  = $::RD_WARN  || $_[0]->{__WARN__};
    local $::RD_HINT  = $::RD_HINT  || $_[0]->{__HINT__};
    local $::RD_TRACE = $::RD_TRACE || $_[0]->{__TRACE__};

    croak "Unknown starting rule ($AUTOLOAD) called\n"
        unless defined &$AUTOLOAD;
    my $retval = &{$AUTOLOAD}(
        $_[0], # $parser
        $text, # $text
        undef, # $repeating
        undef, # $_noactions
        $args, # \@args
        undef, # $_itempos
    );


    if (defined $retval)
    {
        foreach ( @{$_[0]->{deferred}} ) { &$_; }
    }
    else
    {
        foreach ( @{$_[0]->{errors}} ) { _error(@$_); }
    }

    if (ref $_[1] eq 'SCALAR') { ${$_[1]} = $text }

    $ERRORS = 0;
    return $retval;
}

sub _parserepeat($$$$$$$$$)    # RETURNS A REF TO AN ARRAY OF MATCHES
{
    my ($parser, $text, $prod, $min, $max, $_noactions, $expectation, $argcode, $_itempos) = @_;
    my @tokens = ();

    my $itemposfirst;
    my $reps;
    for ($reps=0; $reps<$max;)
    {
        $expectation->at($text);
        my $_savetext = $text;
        my $prevtextlen = length $text;
        my $_tok;
        if (! defined ($_tok = &$prod($parser,$text,1,$_noactions,$argcode,$_itempos)))
        {
            $text = $_savetext;
            last;
        }

        if (defined($_itempos) and !defined($itemposfirst))
        {
            $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
        }

        push @tokens, $_tok if defined $_tok;
        last if ++$reps >= $min and $prevtextlen == length $text;
    }

    do { $expectation->failed(); return undef} if $reps<$min;

    if (defined $itemposfirst)
    {
        Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itemposfirst, undef, [qw(from)]);
    }

    $_[1] = $text;
    return [@tokens];
}

sub set_autoflush {
    my $orig_selected = select $_[0];
    $| = 1;
    select $orig_selected;
    return;
}

# ERROR REPORTING....

sub _write_ERROR {
    my ($errorprefix, $errortext) = @_;
    return if $errortext !~ /\S/;
    $errorprefix =~ s/\s+\Z//;
    local $^A = q{};

    formline(<<'END_FORMAT', $errorprefix, $errortext);
@>>>>>>>>>>>>>>>>>>>>: ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT
    formline(<<'END_FORMAT', $errortext);
~~                     ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT
    print {*STDERR} $^A;
}

# TRACING

my $TRACE_FORMAT = <<'END_FORMAT';
@>|@|||||||||@^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|
  | ~~       |^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|
END_FORMAT

my $TRACECONTEXT_FORMAT = <<'END_FORMAT';
@>|@|||||||||@                                      |^<<<<<<<<<<<<<<<<<<<<<<<<<<<
  | ~~       |                                      |^<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT

sub _write_TRACE {
    my ($tracelevel, $tracerulename, $tracemsg) = @_;
    return if $tracemsg !~ /\S/;
    $tracemsg =~ s/\s*\Z//;
    local $^A = q{};
    my $bar = '|';
    formline($TRACE_FORMAT, $tracelevel, $tracerulename, $bar, $tracemsg, $tracemsg);
    print {*STDERR} $^A;
}

sub _write_TRACECONTEXT {
    my ($tracelevel, $tracerulename, $tracecontext) = @_;
    return if $tracecontext !~ /\S/;
    $tracecontext =~ s/\s*\Z//;
    local $^A = q{};
    my $bar = '|';
    formline($TRACECONTEXT_FORMAT, $tracelevel, $tracerulename, $bar, $tracecontext, $tracecontext);
    print {*STDERR} $^A;
}

sub _verbosity($)
{
       defined $::RD_TRACE
    or defined $::RD_HINT    and  $::RD_HINT   and $_[0] =~ /ERRORS|WARN|HINT/
    or defined $::RD_WARN    and  $::RD_WARN   and $_[0] =~ /ERRORS|WARN/
    or defined $::RD_ERRORS  and  $::RD_ERRORS and $_[0] =~ /ERRORS/
}

sub _error($;$)
{
    $ERRORS++;
    return 0 if ! _verbosity("ERRORS");
    my $errortext   = $_[0];
    my $errorprefix = "ERROR" .  ($_[1] ? " (line $_[1])" : "");
    $errortext =~ s/\s+/ /g;
    print {*STDERR} "\n" if _verbosity("WARN");
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _warn($$;$)
{
    return 0 unless _verbosity("WARN") && ($::RD_HINT || $_[0] >= ($::RD_WARN||1));
    my $errortext   = $_[1];
    my $errorprefix = "Warning" .  ($_[2] ? " (line $_[2])" : "");
    print {*STDERR} "\n" if _verbosity("HINT");
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _hint($)
{
    return 0 unless $::RD_HINT;
    my $errortext = $_[0];
    my $errorprefix = "Hint" .  ($_[1] ? " (line $_[1])" : "");
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _tracemax($)
{
    if (defined $::RD_TRACE
        && $::RD_TRACE =~ /\d+/
        && $::RD_TRACE>1
        && $::RD_TRACE+10<length($_[0]))
    {
        my $count = length($_[0]) - $::RD_TRACE;
        return substr($_[0],0,$::RD_TRACE/2)
            . "...<$count>..."
            . substr($_[0],-$::RD_TRACE/2);
    }
    else
    {
        return substr($_[0],0,500);
    }
}

sub _tracefirst($)
{
    if (defined $::RD_TRACE
        && $::RD_TRACE =~ /\d+/
        && $::RD_TRACE>1
        && $::RD_TRACE+10<length($_[0]))
    {
        my $count = length($_[0]) - $::RD_TRACE;
        return substr($_[0],0,$::RD_TRACE) . "...<+$count>";
    }
    else
    {
        return substr($_[0],0,500);
    }
}

my $lastcontext = '';
my $lastrulename = '';
my $lastlevel = '';

sub _trace($;$$$)
{
    my $tracemsg      = $_[0];
    my $tracecontext  = $_[1]||$lastcontext;
    my $tracerulename = $_[2]||$lastrulename;
    my $tracelevel    = $_[3]||$lastlevel;
    if ($tracerulename) { $lastrulename = $tracerulename }
    if ($tracelevel)    { $lastlevel = $tracelevel }

    $tracecontext =~ s/\n/\\n/g;
    $tracecontext =~ s/\s+/ /g;
    $tracerulename = qq{$tracerulename};
    _write_TRACE($tracelevel, $tracerulename, $tracemsg);
    if ($tracecontext ne $lastcontext)
    {
        if ($tracecontext)
        {
            $lastcontext = _tracefirst($tracecontext);
            $tracecontext = qq{"$tracecontext"};
        }
        else
        {
            $tracecontext = qq{<NO TEXT LEFT>};
        }
        _write_TRACECONTEXT($tracelevel, $tracerulename, $tracecontext);
    }
}

sub _matchtracemessage
{
    my ($self, $reject) = @_;

    my $prefix = '';
    my $postfix = '';
    my $matched = not $reject;
    my @t = ("Matched", "Didn't match");
    if (exists $self->{lookahead} and $self->{lookahead})
    {
        $postfix = $reject ? "(reject)" : "(keep)";
        $prefix = "...";
        if ($self->{lookahead} < 0)
        {
            $prefix .= '!';
            $matched = not $matched;
        }
    }
    $prefix . ($matched ? $t[0] : $t[1]) . $postfix;
}

sub _parseunneg($$$$$)
{
    _parse($_[0],$_[1],$_[3],$_[4]);
    if ($_[2]<0)
    {
        _error("Can't negate \"$_[4]\".",$_[3]);
        _hint("You can't negate $_[0]. Remove the \"...!\" before
               \"$_[4]\".");
        return 0;
    }
    return 1;
}

sub _parse($$$$)
{
    my $what = $_[3];
       $what =~ s/^\s+//;
    if ($_[1])
    {
        _warn(3,"Found $_[0] ($what) after an unconditional <error>",$_[2])
        and
        _hint("An unconditional <error> always causes the
               production containing it to immediately fail.
               \u$_[0] that follows an <error>
               will never be reached.  Did you mean to use
               <error?> instead?");
    }

    return if ! _verbosity("TRACE");
    my $errortext = "Treating \"$what\" as $_[0]";
    my $errorprefix = "Parse::RecDescent::_Runtime";
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
}

sub _linecount($) {
    scalar substr($_[0], pos $_[0]||0) =~ tr/\n//
}


package main;

use vars qw ( $RD_ERRORS $RD_WARN $RD_HINT $RD_TRACE $RD_CHECK );
$::RD_CHECK = 1;
$::RD_ERRORS = 1;
$::RD_WARN = 3;

1;

}
package XML::XSH2::Parser;
{ my $ERRORS;


package Parse::RecDescent::_Runtime::XML::XSH2::Parser;
use strict;
use vars qw($skip $AUTOLOAD  );
@Parse::RecDescent::_Runtime::XML::XSH2::Parser::ISA = ();
$skip = '(\s|\n|#[^\n]*)*';


{
local $SIG{__WARN__} = sub {0};
# PRETEND TO BE IN Parse::RecDescent::_Runtime NAMESPACE
*Parse::RecDescent::_Runtime::XML::XSH2::Parser::AUTOLOAD   = sub
{
    no strict 'refs';

    ${"AUTOLOAD"} =~ s/^Parse::RecDescent::_Runtime::XML::XSH2::Parser/Parse::RecDescent::_Runtime/;
    goto &{${"AUTOLOAD"}};
}
}

push @Parse::RecDescent::_Runtime::XML::XSH2::Parser::ISA, 'Parse::RecDescent::_Runtime';
# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xps
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xps"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xps]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xps},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?: [^\\$\\[\\]()'"\};]+ |
              \\$(?!\\\{) |
              \\$\\\{ (?:\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\\{.*?\\\} |
                       \\(.+?\\)
                   )
               \\\} |
              '(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} | 
                  \\$(?!\\\{) | [^\\$'] 
               )*' |
              "(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} |
                  \\$(?!\\\{) | [^\\$"]
               )*"
          )*/x});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?: [^\\$\\[\\]()'"\};]+ |
              \\$(?!\\\{) |
              \\$\\\{ (?:\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\\{.*?\\\} |
                       \\(.+?\\)
                   )
               \\\} |
              '(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} | 
                  \\$(?!\\\{) | [^\\$'] 
               )*' |
              "(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} |
                  \\$(?!\\\{) | [^\\$"]
               )*"
          )*/x]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xps},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xps});
        %item = (__RULE__ => q{xps});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?: [^\\$\\[\\]()'"\};]+ |
              \\$(?!\\\{) |
              \\$\\\{ (?:\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\\{.*?\\\} |
                       \\(.+?\\)
                   )
               \\\} |
              '(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} | 
                  \\$(?!\\\{) | [^\\$'] 
               )*' |
              "(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} |
                  \\$(?!\\\{) | [^\\$"]
               )*"
          )*/x]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xps},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?: [^\$\[\]()'"};]+ |
              \$(?!\{) |
              \$\{ (?:\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \{.*?\} |
                       \(.+?\)
                   )
               \} |
              '(?:\$\{ (?: \$?[a-zA-Z_][a-zA-Z0-9_]* | \{.*?\} | \(.+?\)) \} | 
                  \$(?!\{) | [^\$'] 
               )*' |
              "(?:\$\{ (?: \$?[a-zA-Z_][a-zA-Z0-9_]* | \{.*?\} | \(.+?\)) \} |
                  \$(?!\{) | [^\$"]
               )*"
          )*)/x)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?: [^\\$\\[\\]()'"\};]+ |
              \\$(?!\\\{) |
              \\$\\\{ (?:\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\\{.*?\\\} |
                       \\(.+?\\)
                   )
               \\\} |
              '(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} | 
                  \\$(?!\\\{) | [^\\$'] 
               )*' |
              "(?:\\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\)) \\\} |
                  \\$(?!\\\{) | [^\\$"]
               )*"
          )*/x]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xps},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xps},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xps},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xps},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xps},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--input-file|:f/ filename]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--input-file|:f/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--input-file|:f)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [filename]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{filename})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::filename($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [filename]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [filename]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filename}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'string/input-file' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--input-file|:f/ filename]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--input-pipe|:p/ filename]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--input-pipe|:p/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--input-pipe|:p)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [filename]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{filename})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::filename($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [filename]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [filename]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filename}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'string/input-pipe' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--input-pipe|:p/ filename]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--input-string|:s/ exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--input-string|:s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--input-string|:s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'exp/input-string' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--input-string|:s/ exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--output-file|:F/ filename]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--output-file|:F/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--output-file|:F)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [filename]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{filename})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::filename($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [filename]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [filename]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filename}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'string/output-file' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--output-file|:F/ filename]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--output-encoding|:E/ enc_string]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--output-encoding|:E/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--output-encoding|:E)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [enc_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{enc_string})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::enc_string($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [enc_string]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [enc_string]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{enc_string}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'string/output-encoding' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--output-encoding|:E/ enc_string]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--output-pipe|:P/ filename]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--output-pipe|:P/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--output-pipe|:P)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [filename]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{filename})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::filename($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [filename]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [filename]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filename}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'string/output-pipe' => $item[2] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--output-pipe|:P/ filename]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--output-string|:S/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/--output-string|:S/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:--output-string|:S)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ '/output-string' => 1 ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--output-string|:S/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::loose_exp
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"loose_exp"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [loose_exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{loose_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/^(?=\{)/, or '&', or /^(?=<<)/, or xpinter});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^(?=\{)/ perl_block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{loose_exp});
        %item = (__RULE__ => q{loose_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^(?=\{)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^(?={))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [perl_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{loose_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{perl_block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::perl_block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [perl_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{loose_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [perl_block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{perl_block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^(?=\{)/ perl_block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['&' block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{loose_exp});
        %item = (__RULE__ => q{loose_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['&']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\&/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{loose_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{loose_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['&' block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^(?=<<)/ inline_doc]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{loose_exp});
        %item = (__RULE__ => q{loose_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^(?=<<)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^(?=<<))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [inline_doc]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{loose_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{inline_doc})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::inline_doc($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [inline_doc]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{loose_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [inline_doc]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{inline_doc}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^(?=<<)/ inline_doc]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpinter]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{loose_exp});
        %item = (__RULE__ => q{loose_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpinter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{loose_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpinter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpinter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{loose_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpinter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpinter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpinter]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{loose_exp},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{loose_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{loose_exp},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{loose_exp},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::trail
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"trail"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [trail]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{trail},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\s*[\};]|\\s*\\Z)/, or '|>', or '|'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\s*[\};]|\\s*\\Z)/ <commit> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{trail});
        %item = (__RULE__ => q{trail});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\s*[\};]|\\s*\\Z)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\s*[};]|\s*\Z))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\s*[\};]|\\s*\\Z)/ <commit> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['|>' <commit> variable]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{trail});
        %item = (__RULE__ => q{trail});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['|>']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\|\>/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{trail},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{variable})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{trail},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [variable]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['var',$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['|>' <commit> variable]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['|' <commit> shline]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{trail});
        %item = (__RULE__ => q{trail});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['|']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\|/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shline]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{trail},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{shline})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shline]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{trail},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shline]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['pipe',$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['|' <commit> shline]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{trail},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{trail},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{trail},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{trail},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbrackets
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpbrackets"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpbrackets]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpbrackets},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\()/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\()/ xpbracket <skip:""> xpfilters]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpbrackets});
        %item = (__RULE__ => q{xpbrackets});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\()/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\())/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpbracket]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpbrackets},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpbracket})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbracket($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpbracket]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpbrackets},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpbracket]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpbracket}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpfilters]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpbrackets},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpfilters})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilters, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpfilters]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpbrackets},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpfilters]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpfilters(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join "",$item[2],@{$item[4]} };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\()/ xpbracket <skip:""> xpfilters]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpbrackets},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpbrackets},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpbrackets},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpbrackets},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpintercont
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpintercont"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpintercont]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xpfilters, or xpbrackets});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpfilters, or xpbrackets <skip:""> xpinter]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpintercont});
        %item = (__RULE__ => q{xpintercont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_1_of_rule_xpintercont]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_xpintercont($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_1_of_rule_xpintercont]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpintercont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_1_of_rule_xpintercont]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_xpintercont}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpinter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpinter})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpinter, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpinter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpintercont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpinter]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpinter(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join("",$item[1],@{$item[3]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpfilters, or xpbrackets <skip:""> xpinter]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpintercont},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpintercont},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpintercont},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::inline_doc_string
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"inline_doc_string"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [inline_doc_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{inline_doc_string},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/'[a-zA-Z_][a-zA-Z0-9_]*'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\([a-zA-Z_][a-zA-Z0-9_]*\\)|\\\{[a-zA-Z_][a-zA-Z0-9_]*\\\}|[a-zA-Z_][a-zA-Z0-9_]*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/'[a-zA-Z_][a-zA-Z0-9_]*'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\([a-zA-Z_][a-zA-Z0-9_]*\\)|\\\{[a-zA-Z_][a-zA-Z0-9_]*\\\}|[a-zA-Z_][a-zA-Z0-9_]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{inline_doc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{inline_doc_string});
        %item = (__RULE__ => q{inline_doc_string});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/'[a-zA-Z_][a-zA-Z0-9_]*'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\([a-zA-Z_][a-zA-Z0-9_]*\\)|\\\{[a-zA-Z_][a-zA-Z0-9_]*\\\}|[a-zA-Z_][a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:'[a-zA-Z_][a-zA-Z0-9_]*'|"[a-zA-Z_][a-zA-Z0-9_]*"|\([a-zA-Z_][a-zA-Z0-9_]*\)|\{[a-zA-Z_][a-zA-Z0-9_]*\}|[a-zA-Z_][a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [($item[1]=~/^(['"({])?(.*?)(['")}])?$/)] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/'[a-zA-Z_][a-zA-Z0-9_]*'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\([a-zA-Z_][a-zA-Z0-9_]*\\)|\\\{[a-zA-Z_][a-zA-Z0-9_]*\\\}|[a-zA-Z_][a-zA-Z0-9_]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{inline_doc_string},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{inline_doc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{inline_doc_string},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{inline_doc_string},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_nosc
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"shline_nosc"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [shline_nosc]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{shline_nosc},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/([^;()\\\\"'\\|]|\\|[^>]|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/([^;()\\\\"'\\|]|\\|[^>]|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shline_nosc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{shline_nosc});
        %item = (__RULE__ => q{shline_nosc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/([^;()\\\\"'\\|]|\\|[^>]|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_nosc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:([^;()\\"'\|]|\|[^>]|\\.|\"([^\"\\]|\\.)*\"|\'([^\'\\]|\\\'|\\\\|\\[^\'\\])*\')*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/([^;()\\\\"'\\|]|\\|[^>]|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_nosc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{shline_nosc},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{shline_nosc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{shline_nosc},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{shline_nosc},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpinter
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpinter"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpinter]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpinter},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xps});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xps <skip:""> xpintercont]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpinter});
        %item = (__RULE__ => q{xpinter});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xps]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpinter},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xps($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xps]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpinter},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xps]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xps}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpintercont]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpinter},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpintercont})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpintercont, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpintercont]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpinter},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpintercont]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpintercont(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join("",$item[1],@{$item[3]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xps <skip:""> xpintercont]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpinter},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpinter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpinter},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpinter},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::inline_doc
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"inline_doc"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [inline_doc]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{inline_doc},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/<</});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/<</ inline_doc_string <skip:""> /.*\\n/ /(.|\\n)*?\\n$item[2][1]\\s*(\\n|$)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{inline_doc});
        %item = (__RULE__ => q{inline_doc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/<</]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:<<)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [inline_doc_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{inline_doc},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{inline_doc_string})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::inline_doc_string($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [inline_doc_string]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{inline_doc},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [inline_doc_string]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{inline_doc_string}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/.*\\n/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/.*\\n/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:.*\n)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(.|\\n)*?\\n$item[2][1]\\s*(\\n|$)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(.|\\n)*?\\n$item[2][1]\\s*(\\n|$)/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(.|\n)*?\n$item[2][1]\s*(\n|$))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  $text=$item[4].$text;
	  local $_=$item[5]; s/\n$item[2][1]\s*$//;
	  my $paren = $item[2][0];
	  $paren = '"' if $paren eq "";
	  if ($paren eq "'") {
	      $paren = '"';
	      s{(\\)(.|\n)|(\$)}{ (defined($3) and $3 eq '$') ? "\\\$"
	        : ((defined($2) and $2 eq "\\")
	        ? "\\\\" : ((defined($2) and $2 eq "'") ? "'" : (
	        (defined($2) and $2 eq '$') ? "\\\\\\$2" :
	        "\\\\$2"))) }eg;
	  } "<<".$paren.$_;
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/<</ inline_doc_string <skip:""> /.*\\n/ /(.|\\n)*?\\n$item[2][1]\\s*(\\n|$)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{inline_doc},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{inline_doc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{inline_doc},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{inline_doc},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::statement
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"statement"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [statement]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\s*[\}\{;])/, or /(if)\\b/, or /(unless)\\b/, or /(while)\\b/, or /(foreach|for)\\b/, or /(try)\\b/, or /(iterate)\\b/, or /(def|define)\\b/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\s*[\}\{;])/ <commit> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\s*[\}\{;])/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\s*[}{;]))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\s*[\}\{;])/ <commit> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(if)\\b/ <commit> exp block elsif_block else_block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(if)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(if)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [elsif_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{elsif_block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::elsif_block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [elsif_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [elsif_block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{elsif_block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [else_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{else_block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::else_block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [else_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [else_block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{else_block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'if_statement',[$item[3],$item[4]],@{$item[5]},@{$item[6]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(if)\\b/ <commit> exp block elsif_block else_block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(unless)\\b/ <commit> exp block else_block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(unless)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(unless)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [else_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{else_block})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::else_block, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [else_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [else_block]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{else_block(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'unless_statement',$item[3],$item[4],@{$item[5]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(unless)\\b/ <commit> exp block else_block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(while)\\b/ <commit> exp block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(while)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(while)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'while_statement',$item[3],$item[4]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(while)\\b/ <commit> exp block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(foreach|for)\\b/ <commit> local_var_in exp block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(foreach|for)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(foreach|for)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [local_var_in]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{local_var_in})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var_in, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [local_var_in]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [local_var_in]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{local_var_in(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'foreach_statement',@item[4,5],@{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(foreach|for)\\b/ <commit> local_var_in exp block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(try)\\b/ <commit> block 'catch' local_var block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(try)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(try)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['catch']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'catch'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Acatch/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [local_var]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{local_var})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [local_var]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [local_var]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{local_var(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'try_catch',$item[3],$item[6],@{$item[5]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(try)\\b/ <commit> block 'catch' local_var block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(iterate)\\b/ <commit> xpstep block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(iterate)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(iterate)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpstep]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpstep})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpstep($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpstep]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpstep]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpstep}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'iterate',$item[4],@{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(iterate)\\b/ <commit> xpstep block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(def|define)\\b/ <commit> ID variable block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[7];
        $text = $_[1];
        my $_savetext;
        @item = (q{statement});
        %item = (__RULE__ => q{statement});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(def|define)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(def|define)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [ID]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{ID})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::ID($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [ID]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [ID]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{ID}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { XML::XSH2::Functions::is_command($item[3])?undef:1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{variable})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable, 0, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [variable]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable(s?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{statement},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{statement},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'def',$item[3],$item[6],$item[5]]
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(def|define)\\b/ <commit> ID variable block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{statement},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{statement},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{statement},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{statement},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_xpintercont
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_xpintercont"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_xpintercont]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xpfilters, or xpbrackets});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpfilters]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_xpintercont});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_xpintercont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpfilters]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilters($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpfilters]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_xpintercont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpfilters]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpfilters}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpfilters]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpbrackets]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_xpintercont});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_xpintercont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpbrackets]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_xpintercont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbrackets($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpbrackets]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_xpintercont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpbrackets]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpbrackets}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpbrackets]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_xpintercont},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_xpintercont},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xplocationstep
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xplocationstep"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xplocationstep]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xplocationstep},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xpaxis});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpaxis xpnodetest]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xplocationstep});
        %item = (__RULE__ => q{xplocationstep});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpaxis]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xplocationstep},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpaxis, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpaxis]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xplocationstep},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpaxis]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpaxis(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpnodetest]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xplocationstep},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpnodetest})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpnodetest($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpnodetest]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xplocationstep},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpnodetest]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpnodetest}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ (@{$item[1]} ? $item[1][0] : 'child::'),$item[3] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpaxis xpnodetest]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xplocationstep},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xplocationstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xplocationstep},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xplocationstep},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var_in
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"local_var_in"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [local_var_in]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{local_var_in},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{local_var});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [local_var 'in']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{local_var_in});
        %item = (__RULE__ => q{local_var_in});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [local_var]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{local_var_in},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [local_var]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{local_var_in},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [local_var]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{local_var}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['in']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'in'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Ain/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [local_var 'in']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{local_var_in},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{local_var_in},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{local_var_in},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{local_var_in},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::param
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"param"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [param]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{param},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/[^=\\s]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[^=\\s]+/ '=' exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{param});
        %item = (__RULE__ => q{param});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[^=\\s]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[^=\s]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['=']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'='})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\=/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{param},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{param},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[1],$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[^=\\s]+/ '=' exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{param},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{param},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{param},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{param},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::startrule
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"startrule"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [startrule]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{startrule},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{shell, or complex_command});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [shell <commit> eof]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{startrule});
        %item = (__RULE__ => q{startrule});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shell]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{startrule},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shell($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shell]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{startrule},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shell]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shell}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [eof]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{startrule},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{eof})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::eof($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [eof]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{startrule},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [eof]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{eof}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [shell <commit> eof]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [complex_command <commit> eof]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{startrule});
        %item = (__RULE__ => q{startrule});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [complex_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{startrule},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::complex_command, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [complex_command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{startrule},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [complex_command]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{complex_command(s)}} = $_tok;
        push @item, $_tok;
        


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [eof]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{startrule},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{eof})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::eof($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [eof]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{startrule},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [eof]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{eof}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [complex_command <commit> eof]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{startrule},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{startrule},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{startrule},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{startrule},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"exp"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{exp},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/^(?=\{)/, or '&', or /^(?=<<)/, or xpath});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^(?=\{)/ perl_block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp});
        %item = (__RULE__ => q{exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^(?=\{)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^(?={))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [perl_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{perl_block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::perl_block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [perl_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [perl_block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{perl_block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^(?=\{)/ perl_block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['&' block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp});
        %item = (__RULE__ => q{exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['&']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\&/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['&' block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^(?=<<)/ inline_doc]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp});
        %item = (__RULE__ => q{exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^(?=<<)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^(?=<<))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [inline_doc]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{inline_doc})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::inline_doc($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [inline_doc]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [inline_doc]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{inline_doc}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^(?=<<)/ inline_doc]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpath]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp});
        %item = (__RULE__ => q{exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpath]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpath($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpath]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpath]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpath}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpath]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{exp},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{exp},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{exp},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpstring
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpstring"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpstring]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpstring},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/'(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$'])*' |
              "(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$"])*"/x});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/'(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$'])*' |
              "(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$"])*"/x]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpstring},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpstring});
        %item = (__RULE__ => q{xpstring});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/'(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$'])*' |
              "(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$"])*"/x]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstring},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:'(?:\$\{(?:\$?[a-zA-Z_][a-zA-Z0-9_]*|\{.*?\}|\(.+?\))\}|\$(?!\{)|[^\$'])*' |
              "(?:\$\{(?:\$?[a-zA-Z_][a-zA-Z0-9_]*|\{.*?\}|\(.+?\))\}|\$(?!\{)|[^\$"])*")/x)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/'(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$'])*' |
              "(?:\\$\\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\{.*?\\\}|\\(.+?\\))\\\}|\\$(?!\\\{)|[^\\$"])*"/x]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstring},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpstring},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpstring},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpstring},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpstring},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::nodename
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"nodename"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [nodename]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{nodename},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{exp});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{nodename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{nodename});
        %item = (__RULE__ => q{nodename});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{nodename},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{nodename},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nodename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nodename},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{nodename},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{nodename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{nodename},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{nodename},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::undef
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"undef"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [undef]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{undef},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(undef|undefine)\\b/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(undef|undefine)\\b/ <commit> /\\$?[a-zA-Z_][a-zA-Z0-9_]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{undef});
        %item = (__RULE__ => q{undef});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(undef|undefine)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(undef|undefine)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\$?[a-zA-Z_][a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\$?[a-zA-Z_][a-zA-Z0-9_]*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\$?[a-zA-Z_][a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'undefine',$item[3]];
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(undef|undefine)\\b/ <commit> /\\$?[a-zA-Z_][a-zA-Z0-9_]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{undef},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{undef},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{undef},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{undef},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilters
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpfilters"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpfilters]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpfilters},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\[)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\[)/ xpfilter]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpfilters});
        %item = (__RULE__ => q{xpfilters});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\[)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\[))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpfilter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpfilters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpfilter})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilter, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpfilter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpfilters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpfilter]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpfilter(s)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join("",@{$item[2]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\[)/ xpfilter]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpfilters},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpfilters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpfilters},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpfilters},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_local_var
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_local_var"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_local_var]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_local_var},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(local|my)\\s/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(local|my)\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_local_var});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_local_var});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(local|my)\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(local|my)\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(local|my)\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_local_var},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_local_var},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::loc
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"loc"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [loc]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{loc},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/after\\s/, or /before\\s/, or /(in)?to\\s/, or /(append(ing)?|as\\s+(a\\s+)child(\\s+of)?)\\s/, or /(prepend(ing)?|(as\\s+)(the\\s+)first(\\s+child(\\s+of)?)?)\\s/, or /(replace|instead( of)?)\\s/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/after\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/after\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:after\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "after" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/after\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/before\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/before\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:before\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "before" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/before\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(in)?to\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(in)?to\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(in)?to\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "into" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(in)?to\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(append(ing)?|as\\s+(a\\s+)child(\\s+of)?)\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(append(ing)?|as\\s+(a\\s+)child(\\s+of)?)\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(append(ing)?|as\s+(a\s+)child(\s+of)?)\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "append" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(append(ing)?|as\\s+(a\\s+)child(\\s+of)?)\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(prepend(ing)?|(as\\s+)(the\\s+)first(\\s+child(\\s+of)?)?)\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(prepend(ing)?|(as\\s+)(the\\s+)first(\\s+child(\\s+of)?)?)\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(prepend(ing)?|(as\s+)(the\s+)first(\s+child(\s+of)?)?)\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "prepend" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(prepend(ing)?|(as\\s+)(the\\s+)first(\\s+child(\\s+of)?)?)\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(replace|instead( of)?)\\s/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{loc});
        %item = (__RULE__ => q{loc});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(replace|instead( of)?)\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(replace|instead( of)?)\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "replace" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(replace|instead( of)?)\\s/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{loc},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{loc},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{loc},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{loc},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::eof
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"eof"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [eof]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{eof},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/\\Z/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\Z/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{eof},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{eof});
        %item = (__RULE__ => q{eof});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\Z/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{eof},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\Z)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{eof},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 1; };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\Z/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{eof},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{eof},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{eof},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{eof},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{eof},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::nodetype
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"nodetype"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [nodetype]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{nodetype},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/element|attribute|attributes|text|cdata|pi|comment|chunk|entity_reference/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/element|attribute|attributes|text|cdata|pi|comment|chunk|entity_reference/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{nodetype},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{nodetype});
        %item = (__RULE__ => q{nodetype});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/element|attribute|attributes|text|cdata|pi|comment|chunk|entity_reference/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nodetype},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:element|attribute|attributes|text|cdata|pi|comment|chunk|entity_reference)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/element|attribute|attributes|text|cdata|pi|comment|chunk|entity_reference/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nodetype},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{nodetype},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{nodetype},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{nodetype},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{nodetype},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpcont
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpcont"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpcont]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xpfilters, or xpbrackets, or xpath});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpfilters, or xpbrackets <skip:""> xpath]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpcont});
        %item = (__RULE__ => q{xpcont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_1_of_rule_xpcont]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_xpcont($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_1_of_rule_xpcont]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpcont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_1_of_rule_xpcont]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_xpcont}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpath]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpath})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpath, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpath]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpcont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpath]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpath(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1].join("",@{$item[3]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpfilters, or xpbrackets <skip:""> xpath]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpath]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpcont});
        %item = (__RULE__ => q{xpcont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpath]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpath($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpath]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpcont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpath]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpath}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpath]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpcont},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpcont},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpcont},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::stream_select
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"stream_select"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [stream_select]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{stream_select},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/select\\s/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/select\\s/ xpath block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{stream_select});
        %item = (__RULE__ => q{stream_select});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/select\\s/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:select\s)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpath]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{stream_select},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpath})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpath($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpath]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{stream_select},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpath]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpath}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{stream_select},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{stream_select},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[2],$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/select\\s/ xpath block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{stream_select},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{stream_select},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{stream_select},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{stream_select},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::ID
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"ID"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [ID]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{ID},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/[a-zA-Z_][a-zA-Z0-9_]*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[a-zA-Z_][a-zA-Z0-9_]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{ID},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{ID});
        %item = (__RULE__ => q{ID});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[a-zA-Z_][a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ID},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[a-zA-Z_][a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[a-zA-Z_][a-zA-Z0-9_]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ID},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{ID},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{ID},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{ID},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{ID},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"variable"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{variable},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/\\$[a-zA-Z_][a-zA-Z0-9_]*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\$[a-zA-Z_][a-zA-Z0-9_]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{variable},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{variable});
        %item = (__RULE__ => q{variable});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\$[a-zA-Z_][a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{variable},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\$[a-zA-Z_][a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\$[a-zA-Z_][a-zA-Z0-9_]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{variable},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{variable},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{variable},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{variable},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{variable},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpstep
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpstep"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpstep]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpstep},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xplocationstep});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xplocationstep <skip:""> xpfilter]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpstep});
        %item = (__RULE__ => q{xpstep});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xplocationstep]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpstep},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xplocationstep($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xplocationstep]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpstep},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xplocationstep]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xplocationstep}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpfilter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpstep},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpfilter})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilter, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpfilter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpstep},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpfilter]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpfilter(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ @{$item[1]}, @{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xplocationstep <skip:""> xpfilter]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpstep},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpstep},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpstep},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpstep},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpaxis
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpaxis"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpaxis]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpaxis},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/[-a-z]+::/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[-a-z]+::/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpaxis},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpaxis});
        %item = (__RULE__ => q{xpaxis});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[-a-z]+::/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpaxis},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[-a-z]+::)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[-a-z]+::/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpaxis},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpaxis},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpaxis},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpaxis},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpaxis},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"local_var"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [local_var]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{local_var},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(local|my)\\s/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(local|my)\\s/ variable]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{local_var});
        %item = (__RULE__ => q{local_var});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/(local|my)\\s/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{local_var},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_local_var, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/(local|my)\\s/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{local_var},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_local_var]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_local_var(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{local_var},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{variable})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{local_var},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [variable]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[2],@{$item[1]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(local|my)\\s/ variable]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{local_var},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{local_var},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{local_var},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{local_var},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbracket
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpbracket"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpbracket]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpbracket},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'('});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['(' <skip:""> xpinter ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpbracket});
        %item = (__RULE__ => q{xpbracket});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpinter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpbracket},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpinter})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpinter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpinter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpbracket},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpinter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpinter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "($item[3])" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['(' <skip:""> xpinter ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpbracket},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpbracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpbracket},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpbracket},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::complex_command
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"complex_command"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [complex_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{complex_command},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\s*[\{\}]|\\s*\\Z)/, or /\\s*;/, or /(?=(?:foreach|for|if|unless|iterate|while|try|def|define)\\b)/, or command});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\s*[\{\}]|\\s*\\Z)/ <commit> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{complex_command});
        %item = (__RULE__ => q{complex_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\s*[\{\}]|\\s*\\Z)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\s*[{}]|\s*\Z))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\s*[\{\}]|\\s*\\Z)/ <commit> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\s*;/ <commit>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{complex_command});
        %item = (__RULE__ => q{complex_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*;/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*;)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\s*;/ <commit>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=(?:foreach|for|if|unless|iterate|while|try|def|define)\\b)/ statement <commit> trail]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{complex_command});
        %item = (__RULE__ => q{complex_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=(?:foreach|for|if|unless|iterate|while|try|def|define)\\b)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=(?:foreach|for|if|unless|iterate|while|try|def|define)\b))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [statement]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{complex_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{statement})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::statement($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [statement]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{complex_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [statement]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{statement}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [trail]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{complex_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{trail})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::trail, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [trail]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{complex_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [trail]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{trail(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  if (scalar(@{$item[4]})) {
	    if ($item[4][0][0] eq 'pipe') {
  	      $return=[$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'pipe_command',[$item[2]],$item[4][0][1]]
	    } else {
   	      $return=[$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'string_pipe_command',[$item[2]],$item[4][0][1]]
	    }
          } else {
            $return=$item[2]
          }
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=(?:foreach|for|if|unless|iterate|while|try|def|define)\\b)/ statement <commit> trail]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [command <commit> trail /\\s*;|(?=\\s*\}|\\s*\\Z)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{complex_command});
        %item = (__RULE__ => q{complex_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{complex_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{complex_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [trail]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{complex_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{trail})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::trail, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [trail]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{complex_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [trail]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{trail(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*;|(?=\\s*\}|\\s*\\Z)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*;|(?=\\s*\}|\\s*\\Z)/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*;|(?=\s*}|\s*\Z))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  if (scalar(@{$item[3]})) {
	    if ($item[3][0][0] eq 'pipe') {
  	      $return=[$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'pipe_command',[$item[1]],$item[3][0][1]]
	    } else {
   	      $return=[$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'string_pipe_command',[$item[1]],$item[3][0][1]]
	    }
          } else {
            $return=$item[1]
          }
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [command <commit> trail /\\s*;|(?=\\s*\}|\\s*\\Z)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        
        my $_savetext;
        @item = (q{complex_command});
        %item = (__RULE__ => q{complex_command});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {push @{$thisparser->{errors}}, [qq{Parse error near: "}.substr($text,0,40).qq{ ..."},$thisline];} unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{complex_command},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{complex_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{complex_command},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{complex_command},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::enc_string
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"enc_string"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [enc_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{enc_string},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{exp});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{enc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{enc_string});
        %item = (__RULE__ => q{enc_string});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{enc_string},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{enc_string},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{enc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{enc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{enc_string},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{enc_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{enc_string},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{enc_string},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpsimple
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpsimple"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpsimple]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpsimple},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?: 
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^-:\\$\\[\\]\{\}|"'\\ \\s();] |
              -[^-\\[\\]\{\}|"'\\ \\s();]
          )
          (?:
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^\\[\\]\\$\{\}|"'\\ \\s();]
          )*/x, or xpbrackets});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?: 
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^-:\\$\\[\\]\{\}|"'\\ \\s();] |
              -[^-\\[\\]\{\}|"'\\ \\s();]
          )
          (?:
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^\\[\\]\\$\{\}|"'\\ \\s();]
          )*/x]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpsimple});
        %item = (__RULE__ => q{xpsimple});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?: 
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^-:\\$\\[\\]\{\}|"'\\ \\s();] |
              -[^-\\[\\]\{\}|"'\\ \\s();]
          )
          (?:
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^\\[\\]\\$\{\}|"'\\ \\s();]
          )*/x]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?: 
              \$\{ (?: \$?[a-zA-Z_][a-zA-Z0-9_]* | \{.*?\} | \(.+?\) ) \} |
              \$(?!\{) |
              [^-:\$\[\]{}|"'\ \s();] |
              -[^-\[\]{}|"'\ \s();]
          )
          (?:
              \$\{ (?: \$?[a-zA-Z_][a-zA-Z0-9_]* | \{.*?\} | \(.+?\) ) \} |
              \$(?!\{) |
              [^\[\]\${}|"'\ \s();]
          )*)/x)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?: 
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^-:\\$\\[\\]\{\}|"'\\ \\s();] |
              -[^-\\[\\]\{\}|"'\\ \\s();]
          )
          (?:
              \\$\\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\{.*?\\\} | \\(.+?\\) ) \\\} |
              \\$(?!\\\{) |
              [^\\[\\]\\$\{\}|"'\\ \\s();]
          )*/x]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpbrackets]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpsimple});
        %item = (__RULE__ => q{xpsimple});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpbrackets]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpsimple},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbrackets($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpbrackets]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpsimple},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpbrackets]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpbrackets}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpbrackets]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpsimple},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpsimple},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpsimple},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpsimple},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::shell
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"shell"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [shell]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{shell},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/!\\s*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/!\\s*/ <commit> /.*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{shell});
        %item = (__RULE__ => q{shell});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/!\\s*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:!\s*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/.*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/.*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:.*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [[$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'sh_noev',$item[3]]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/!\\s*/ <commit> /.*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{shell});
        %item = (__RULE__ => q{shell});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {push @{$thisparser->{errors}}, [qq{Parse error near: "! }.substr($text,0,40).qq{ ..."},$thisline];} unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{shell},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{shell},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{shell},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{shell},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_2_of_rule_command
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_2_of_rule_command"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_2_of_rule_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_2_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/, or /\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/ <commit> loose_exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_2_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_2_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:[\-\+\*\/%x.]|\|\||\&\&)?=)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [loose_exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_2_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{loose_exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::loose_exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [loose_exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_2_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [loose_exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{loose_exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['xpath_assign',$item[3],$item[1]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/ <commit> loose_exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/ command]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_2_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_2_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?:[\-\+\*\/%x.]|\|\||\&\&)?:=)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_2_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{command})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_2_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['command_assign',$item[2],$item[1]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/ command]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_2_of_rule_command},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_2_of_rule_command},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::option
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"option"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [option]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{option},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/:[[:alnum:]]|--[-_[:alnum:]]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/:[[:alnum:]]|--[-_[:alnum:]]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{option},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{option});
        %item = (__RULE__ => q{option});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/:[[:alnum:]]|--[-_[:alnum:]]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{option},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?::[[:alnum:]]|--[-_[:alnum:]]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/:[[:alnum:]]|--[-_[:alnum:]]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{option},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{option},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{option},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{option},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{option},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_inter
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"shline_inter"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [shline_inter]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{shline_inter},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/([^()\\\\"']|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/([^()\\\\"']|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shline_inter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{shline_inter});
        %item = (__RULE__ => q{shline_inter});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/([^()\\\\"']|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_inter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:([^()\\"']|\\.|\"([^\"\\]|\\.)*\"|\'([^\'\\]|\\\'|\\\\|\\[^\'\\])*\')*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/([^()\\\\"']|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\'([^\\'\\\\]|\\\\\\'|\\\\\\\\|\\\\[^\\'\\\\])*\\')*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_inter},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{shline_inter},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{shline_inter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{shline_inter},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{shline_inter},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::perl_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"perl_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [perl_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{perl_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{exp});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{perl_expression});
        %item = (__RULE__ => q{perl_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $main::myline = $thisline; };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{perl_expression});
        %item = (__RULE__ => q{perl_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{perl_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{perl_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { {local $^W=0; "\n# line $main::myline \"$XML::XSH2::Functions::SCRIPT\"\n".$item[1]} };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{perl_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{perl_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{perl_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{perl_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::command
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"command"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\s*[\}\{;]|\\s*\\Z)/, or /assign\\b|(?:local\\b|my\\b)?(?=\\s*\\$[a-zA-Z_][a-zA-Z0-9_]*\\s*\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:?=)/, or /(my)\\b/, or /(local)\\b/, or /(do)\\b/, or /(if)\\b/, or /(unless)\\b/, or /(while)\\b/, or /(foreach|for)\\b/, or /(stream)\\b/, or /(undef|undefine)\\b/, or /(use)\\b/, or /(test-mode|test_mode)/, or /(run-mode|run_mode)/, or /(?!(?:iterate|try|def|define)\\b)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\s*[\}\{;]|\\s*\\Z)/ <commit> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\s*[\}\{;]|\\s*\\Z)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\s*[}{;]|\s*\Z))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\s*[\}\{;]|\\s*\\Z)/ <commit> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/assign\\b|(?:local\\b|my\\b)?(?=\\s*\\$[a-zA-Z_][a-zA-Z0-9_]*\\s*\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:?=)/ <commit> variable /(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/, or /\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/assign\\b|(?:local\\b|my\\b)?(?=\\s*\\$[a-zA-Z_][a-zA-Z0-9_]*\\s*\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:?=)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:assign\b|(?:local\b|my\b)?(?=\s*\$[a-zA-Z_][a-zA-Z0-9_]*\s*\s*(?:[\-\+\*\/%x.]|\|\||\&\&)?:?=))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{variable})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [variable]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_2_of_rule_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{/(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/, or /\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_2_of_rule_command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_2_of_rule_command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_2_of_rule_command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_2_of_rule_command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,@{$item[4]},$item[1],$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/assign\\b|(?:local\\b|my\\b)?(?=\\s*\\$[a-zA-Z_][a-zA-Z0-9_]*\\s*\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:?=)/ <commit> variable /(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?=/, or /\\s*(?:[\\-\\+\\*\\/%x.]|\\|\\||\\&\\&)?:=/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(my)\\b/ variable]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(my)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(my)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{variable})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [variable]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable(s)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'store_lex_variables',0,@{$item[2]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(my)\\b/ variable]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(local)\\b/ variable]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(local)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(local)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [variable]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{variable})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::variable, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [variable]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [variable]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{variable(s)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'make_local',@{$item[2]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(local)\\b/ variable]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(do)\\b/ <commit> block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(do)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(do)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'run_commands',$item[3],0] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(do)\\b/ <commit> block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(if)\\b/ <commit> exp command]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(if)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(if)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{command})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'if_statement',[$item[3],[$item[4]]]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(if)\\b/ <commit> exp command]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(unless)\\b/ <commit> exp command]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(unless)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(unless)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{command})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'unless_statement',$item[3],[$item[4]]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(unless)\\b/ <commit> exp command]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(while)\\b/ <commit> exp command]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[7];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(while)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(while)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{command})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'while_statement',$item[3],[$item[4]]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(while)\\b/ <commit> exp command]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(foreach|for)\\b/ <commit> local_var_in exp command]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[8];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(foreach|for)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(foreach|for)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [local_var_in]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{local_var_in})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::local_var_in, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [local_var_in]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [local_var_in]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{local_var_in(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{command})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'foreach_statement',$item[4],[$item[5]],@{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(foreach|for)\\b/ <commit> local_var_in exp command]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(stream)\\b/ <commit> /--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/ stream_select]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[9];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(stream)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(stream)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_10_of_rule_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_10_of_rule_command($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_10_of_rule_command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_10_of_rule_command]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_10_of_rule_command}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [stream_select]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{stream_select})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::stream_select, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [stream_select]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [stream_select]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{stream_select(s)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'stream_process',$item[3],$item[4]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(stream)\\b/ <commit> /--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/ stream_select]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(undef|undefine)\\b/ <commit> /\\$?[a-zA-Z_][a-zA-Z0-9_]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[10];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(undef|undefine)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(undef|undefine)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\$?[a-zA-Z_][a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\$?[a-zA-Z_][a-zA-Z0-9_]*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\$?[a-zA-Z_][a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,'undefine',$item[3]];
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(undef|undefine)\\b/ <commit> /\\$?[a-zA-Z_][a-zA-Z0-9_]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(use)\\b/ <commit> /XML::XSH2::(?:Inline|Compile)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[11];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(use)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(use)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/XML::XSH2::(?:Inline|Compile)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/XML::XSH2::(?:Inline|Compile)/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:XML::XSH2::(?:Inline|Compile))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(use)\\b/ <commit> /XML::XSH2::(?:Inline|Compile)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(test-mode|test_mode)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[12];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(test-mode|test_mode)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(test-mode|test_mode))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,"test-mode"] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(test-mode|test_mode)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(run-mode|run_mode)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[13];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(run-mode|run_mode)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(run-mode|run_mode))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,"run-mode"] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(run-mode|run_mode)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?!(?:iterate|try|def|define)\\b)/ /\\.|[a-zA-Z_][-a-zA-Z0-9_]*/ exp_or_opt]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[14];
        $text = $_[1];
        my $_savetext;
        @item = (q{command});
        %item = (__RULE__ => q{command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!(?:iterate|try|def|define)\\b)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!(?:iterate|try|def|define)\b))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\.|[a-zA-Z_][-a-zA-Z0-9_]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\.|[a-zA-Z_][-a-zA-Z0-9_]*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\.|[a-zA-Z_][-a-zA-Z0-9_]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [exp_or_opt]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{exp_or_opt})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp_or_opt, 0, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [exp_or_opt]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [exp_or_opt]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp_or_opt(s?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	  bless
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,$item[2],@{$item[3]}],
	  'XML::XSH2::Command'
	 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?!(?:iterate|try|def|define)\\b)/ /\\.|[a-zA-Z_][-a-zA-Z0-9_]*/ exp_or_opt]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{command},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{command},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{command},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::else_block
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"else_block"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [else_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{else_block},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(else)\\b/, or /(?!else)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(else)\\b/ <commit> block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{else_block});
        %item = (__RULE__ => q{else_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(else)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(else)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{else_block},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{else_block},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [[undef,$item[3]]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(else)\\b/ <commit> block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?!else)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{else_block});
        %item = (__RULE__ => q{else_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!else)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!else))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?!else)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<uncommit> <error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{else_block});
        %item = (__RULE__ => q{else_block});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<uncommit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit=0;1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {push @{$thisparser->{errors}}, [qq{Parse error near keyword else: "}.substr($text,0,40).qq{ ..."},$thisline];} unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<uncommit> <error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{else_block},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{else_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{else_block},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{else_block},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilter
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpfilter"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpfilter]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpfilter},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'['});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['[' xpinter ']']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpfilter});
        %item = (__RULE__ => q{xpfilter});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpinter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpfilter},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpinter})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpinter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpinter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpfilter},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpinter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpinter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [']']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{']'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\]/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { "[$item[2]]" };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['[' xpinter ']']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpfilter},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpfilter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpfilter},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpfilter},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_10_of_rule_command
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_10_of_rule_command"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_10_of_rule_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_10_of_rule_command});
        %item = (__RULE__ => q{_alternation_1_of_production_10_of_rule_command});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_10_of_rule_command},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command, 0, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_10_of_rule_command},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command(s?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ map { @$_ } @{$item[1]} ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/--input-file|:f/, or /--input-pipe|:p/, or /--input-string|:s/, or /--output-file|:F/, or /--output-encoding|:E/, or /--output-pipe|:P/, or /--output-string|:S/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_10_of_rule_command},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_10_of_rule_command},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpath
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpath"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpath]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpath},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=['"])/, or xpsimple});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=['"])/ <commit> xpstring]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpath});
        %item = (__RULE__ => q{xpath});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=['"])/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=['"]))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpstring]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpath},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{xpstring})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpstring($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpstring]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpath},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpstring]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpstring}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[3] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=['"])/ <commit> xpstring]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpsimple <skip:""> xpcont]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpath});
        %item = (__RULE__ => q{xpath});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpsimple]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpath},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpsimple($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpsimple]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpath},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpsimple]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpsimple}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:"">]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=""; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [xpcont]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{xpath},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{xpcont})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpcont, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [xpcont]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{xpath},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [xpcont]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpcont(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1].join("",@{$item[3]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpsimple <skip:""> xpcont]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        
        my $_savetext;
        @item = (q{xpath});
        %item = (__RULE__ => q{xpath});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {push @{$thisparser->{errors}}, [qq{expected XPath, but got "}.substr($text,0,40).qq{ ..."},$thisline];} unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpath},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpath},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpath},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpath},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpnodetest
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"xpnodetest"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [xpnodetest]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{xpnodetest},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/node\\(\\)|text\\(\\)|comment\\(\\)|processing-instruction\\(\\s*(?:"[^"]*"\\s*|'[^'*]'\\s*)?\\)|[^\\(\\[\\/\\"\\'\\&\\;\\s]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/node\\(\\)|text\\(\\)|comment\\(\\)|processing-instruction\\(\\s*(?:"[^"]*"\\s*|'[^'*]'\\s*)?\\)|[^\\(\\[\\/\\"\\'\\&\\;\\s]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{xpnodetest},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{xpnodetest});
        %item = (__RULE__ => q{xpnodetest});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/node\\(\\)|text\\(\\)|comment\\(\\)|processing-instruction\\(\\s*(?:"[^"]*"\\s*|'[^'*]'\\s*)?\\)|[^\\(\\[\\/\\"\\'\\&\\;\\s]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpnodetest},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:node\(\)|text\(\)|comment\(\)|processing-instruction\(\s*(?:"[^"]*"\s*|'[^'*]'\s*)?\)|[^\(\[\/\"\'\&\;\s]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/node\\(\\)|text\\(\\)|comment\\(\\)|processing-instruction\\(\\s*(?:"[^"]*"\\s*|'[^'*]'\\s*)?\\)|[^\\(\\[\\/\\"\\'\\&\\;\\s]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{xpnodetest},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{xpnodetest},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{xpnodetest},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{xpnodetest},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{xpnodetest},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::filename
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"filename"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [filename]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{filename},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{exp});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{filename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{filename});
        %item = (__RULE__ => q{filename});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{filename},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{filename},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filename},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{filename},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{filename},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{filename},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{filename},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::_alternation_1_of_production_1_of_rule_xpcont
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_xpcont"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_xpcont]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{xpfilters, or xpbrackets});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpfilters]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_xpcont});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_xpcont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpfilters]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpfilters($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpfilters]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_xpcont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpfilters]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpfilters}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpfilters]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [xpbrackets]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_xpcont});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_xpcont});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [xpbrackets]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_xpcont},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::xpbrackets($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [xpbrackets]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_xpcont},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [xpbrackets]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{xpbrackets}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [xpbrackets]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_xpcont},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_xpcont},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::elsif_block
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"elsif_block"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [elsif_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{elsif_block},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(elsif)\\b/, or /(?!elsif)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(elsif)\\b/ <commit> exp block elsif_block]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{elsif_block});
        %item = (__RULE__ => q{elsif_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(elsif)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(elsif)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{elsif_block},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{elsif_block},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{elsif_block},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{elsif_block},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [elsif_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{elsif_block},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{elsif_block})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::elsif_block($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [elsif_block]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{elsif_block},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [elsif_block]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{elsif_block}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [[$item[3],$item[4]],@{$item[5]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(elsif)\\b/ <commit> exp block elsif_block]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?!elsif)/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{elsif_block});
        %item = (__RULE__ => q{elsif_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!elsif)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!elsif))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?!elsif)/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<uncommit> <error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{elsif_block});
        %item = (__RULE__ => q{elsif_block});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<uncommit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit=0;1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {push @{$thisparser->{errors}}, [qq{Parse error near keyword elsif: "}.substr($text,0,40).qq{ ..."},$thisline];} unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<uncommit> <error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{elsif_block},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{elsif_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{elsif_block},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{elsif_block},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"shline"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [shline]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{shline},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{shline_nosc});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [shline_nosc shline_bracket shline_nosc]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{shline});
        %item = (__RULE__ => q{shline});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shline_nosc]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_nosc($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shline_nosc]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shline_nosc]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_nosc}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [shline_bracket]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{shline_bracket})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_bracket, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [shline_bracket]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [shline_bracket]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_bracket(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shline_nosc]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{shline_nosc})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_nosc($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shline_nosc]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shline_nosc]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_nosc}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join("",$item[1],@{$item[2]},$item[3]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [shline_nosc shline_bracket shline_nosc]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{shline},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{shline},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{shline},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{shline},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::block
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"block"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [block]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{block},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'\{'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['\{' <commit> complex_command '\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{block});
        %item = (__RULE__ => q{block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [complex_command]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{block},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{complex_command})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::complex_command, 1, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [complex_command]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{block},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [complex_command]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{complex_command(s)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [grep ref,@{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['\{' <commit> complex_command '\}']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{block},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{block},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{block},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::perl_block
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"perl_block"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [perl_block]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{perl_block},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{perl_block});
        %item = (__RULE__ => q{perl_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $main::myline = $thisline; };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{perl_block});
        %item = (__RULE__ => q{perl_block});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $main::myline = $thisline; };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<perl_codeblock>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{perl_block});
        %item = (__RULE__ => q{perl_block});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<perl_codeblock>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { Text::Balanced::extract_codeblock($text,undef,$skip,'{}');
                     };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { {
	  $return=$item[1];
	  {
  	    local $^W = 0; # don't warn about undefined contants
	    my $pos="# line $main::myline \"$XML::XSH2::Functions::SCRIPT\"\n";
	    $return=~s/^\{/\{\n$pos/;
          }
	  } };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<perl_codeblock>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{perl_block},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{perl_block},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{perl_block},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{perl_block},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_bracket
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"shline_bracket"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [shline_bracket]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{shline_bracket},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'('});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['(' shline_inter shline_bracket shline_inter ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{shline_bracket});
        %item = (__RULE__ => q{shline_bracket});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shline_inter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline_bracket},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{shline_inter})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_inter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shline_inter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline_bracket},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shline_inter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_inter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [shline_bracket]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline_bracket},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{shline_bracket})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_bracket, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [shline_bracket]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline_bracket},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [shline_bracket]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_bracket(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [shline_inter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{shline_bracket},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{shline_inter})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::shline_inter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [shline_inter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{shline_bracket},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [shline_inter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{shline_inter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { join("",'(',$item[2],@{$item[3]},$item[4],')') };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['(' shline_inter shline_bracket shline_inter ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{shline_bracket},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{shline_bracket},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{shline_bracket},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{shline_bracket},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp_or_opt
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"exp_or_opt"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [exp_or_opt]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{exp_or_opt},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?=\\s*[\};]|\\s*\\Z)/, or option, or exp});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?=\\s*[\};]|\\s*\\Z)/ <commit> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp_or_opt});
        %item = (__RULE__ => q{exp_or_opt});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?=\\s*[\};]|\\s*\\Z)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?=\s*[};]|\s*\Z))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?=\\s*[\};]|\\s*\\Z)/ <commit> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [option]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp_or_opt});
        %item = (__RULE__ => q{exp_or_opt});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [option]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp_or_opt},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::option($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [option]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp_or_opt},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [option]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{option}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [option]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [exp]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{exp_or_opt});
        %item = (__RULE__ => q{exp_or_opt});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{exp_or_opt},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::XML::XSH2::Parser::exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{exp_or_opt},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{exp}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [exp]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{exp_or_opt},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{exp_or_opt},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{exp_or_opt},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{exp_or_opt},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}
}
package XML::XSH2::Parser; sub new { my $self = bless( {
                 'localvars' => '',
                 'startcode' => '',
                 'namespace' => 'Parse::RecDescent::_Runtime::XML::XSH2::Parser',
                 'rules' => {
                              'xps' => bless( {
                                                'impcount' => 0,
                                                'calls' => [],
                                                'opcount' => 0,
                                                'prods' => [
                                                             bless( {
                                                                      'number' => 0,
                                                                      'strcount' => 0,
                                                                      'dircount' => 0,
                                                                      'uncommit' => undef,
                                                                      'error' => undef,
                                                                      'patcount' => 1,
                                                                      'actcount' => 0,
                                                                      'items' => [
                                                                                   bless( {
                                                                                            'description' => '/(?: [^\\\\$\\\\[\\\\]()\'"\\};]+ |
              \\\\$(?!\\\\\\{) |
              \\\\$\\\\\\{ (?:\\\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\\\\\{.*?\\\\\\} |
                       \\\\(.+?\\\\)
                   )
               \\\\\\} |
              \'(?:\\\\$\\\\\\{ (?: \\\\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\\\\{.*?\\\\\\} | \\\\(.+?\\\\)) \\\\\\} | 
                  \\\\$(?!\\\\\\{) | [^\\\\$\'] 
               )*\' |
              "(?:\\\\$\\\\\\{ (?: \\\\$?[a-zA-Z_][a-zA-Z0-9_]* | \\\\\\{.*?\\\\\\} | \\\\(.+?\\\\)) \\\\\\} |
                  \\\\$(?!\\\\\\{) | [^\\\\$"]
               )*"
          )*/x',
                                                                                            'rdelim' => '/',
                                                                                            'pattern' => '(?: [^\\$\\[\\]()\'"};]+ |
              \\$(?!\\{) |
              \\$\\{ (?:\\$?[a-zA-Z_][a-zA-Z0-9_]* |
                       \\{.*?\\} |
                       \\(.+?\\)
                   )
               \\} |
              \'(?:\\$\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\{.*?\\} | \\(.+?\\)) \\} | 
                  \\$(?!\\{) | [^\\$\'] 
               )*\' |
              "(?:\\$\\{ (?: \\$?[a-zA-Z_][a-zA-Z0-9_]* | \\{.*?\\} | \\(.+?\\)) \\} |
                  \\$(?!\\{) | [^\\$"]
               )*"
          )*',
                                                                                            'hashname' => '__PATTERN1__',
                                                                                            'lookahead' => 0,
                                                                                            'ldelim' => '/',
                                                                                            'mod' => 'x',
                                                                                            'line' => 244
                                                                                          }, 'Parse::RecDescent::_Runtime::Token' )
                                                                                 ],
                                                                      'line' => undef
                                                                    }, 'Parse::RecDescent::_Runtime::Production' )
                                                           ],
                                                'name' => 'xps',
                                                'vars' => '',
                                                'changed' => 0,
                                                'line' => 243
                                              }, 'Parse::RecDescent::_Runtime::Rule' ),
                              '_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command' => bless( {
                                                                                                                                   'impcount' => 0,
                                                                                                                                   'calls' => [
                                                                                                                                                'filename',
                                                                                                                                                'exp',
                                                                                                                                                'enc_string'
                                                                                                                                              ],
                                                                                                                                   'opcount' => 0,
                                                                                                                                   'prods' => [
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 0,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--input-file|:f/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--input-file|:f',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 440
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'filename',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 440
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 440,
                                                                                                                                                                               'code' => '{ [ \'string/input-file\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => undef
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 1,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--input-pipe|:p/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--input-pipe|:p',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 441
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'filename',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 441
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 441,
                                                                                                                                                                               'code' => '{ [ \'string/input-pipe\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 441
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 2,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--input-string|:s/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--input-string|:s',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 442
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'exp',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 442
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 442,
                                                                                                                                                                               'code' => '{ [ \'exp/input-string\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 442
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 3,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--output-file|:F/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--output-file|:F',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 443
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'filename',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 443
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 443,
                                                                                                                                                                               'code' => '{ [ \'string/output-file\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 443
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 4,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--output-encoding|:E/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--output-encoding|:E',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 444
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'enc_string',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 444
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 444,
                                                                                                                                                                               'code' => '{ [ \'string/output-encoding\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 444
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 5,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--output-pipe|:P/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--output-pipe|:P',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 445
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'subrule' => 'filename',
                                                                                                                                                                               'matchrule' => 0,
                                                                                                                                                                               'implicit' => undef,
                                                                                                                                                                               'argcode' => undef,
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 445
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 445,
                                                                                                                                                                               'code' => '{ [ \'string/output-pipe\' => $item[2] ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 445
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                                                                bless( {
                                                                                                                                                         'number' => 6,
                                                                                                                                                         'strcount' => 0,
                                                                                                                                                         'dircount' => 0,
                                                                                                                                                         'uncommit' => undef,
                                                                                                                                                         'error' => undef,
                                                                                                                                                         'patcount' => 1,
                                                                                                                                                         'actcount' => 1,
                                                                                                                                                         'items' => [
                                                                                                                                                                      bless( {
                                                                                                                                                                               'description' => '/--output-string|:S/',
                                                                                                                                                                               'rdelim' => '/',
                                                                                                                                                                               'pattern' => '--output-string|:S',
                                                                                                                                                                               'hashname' => '__PATTERN1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'ldelim' => '/',
                                                                                                                                                                               'mod' => '',
                                                                                                                                                                               'line' => 446
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                                                                                      bless( {
                                                                                                                                                                               'hashname' => '__ACTION1__',
                                                                                                                                                                               'lookahead' => 0,
                                                                                                                                                                               'line' => 446,
                                                                                                                                                                               'code' => '{ [ \'/output-string\' => 1 ] }'
                                                                                                                                                                             }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                                                                                                    ],
                                                                                                                                                         'line' => 446
                                                                                                                                                       }, 'Parse::RecDescent::_Runtime::Production' )
                                                                                                                                              ],
                                                                                                                                   'name' => '_alternation_1_of_production_1_of_rule__alternation_1_of_production_10_of_rule_command',
                                                                                                                                   'vars' => '',
                                                                                                                                   'changed' => 0,
                                                                                                                                   'line' => 440
                                                                                                                                 }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'loose_exp' => bless( {
                                                      'impcount' => 0,
                                                      'calls' => [
                                                                   'perl_block',
                                                                   'block',
                                                                   'inline_doc',
                                                                   'xpinter'
                                                                 ],
                                                      'opcount' => 0,
                                                      'prods' => [
                                                                   bless( {
                                                                            'number' => 0,
                                                                            'strcount' => 0,
                                                                            'dircount' => 0,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/^(?=\\{)/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '^(?={)',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 290
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'perl_block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 290
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 291,
                                                                                                  'code' => '{ $item[2] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => undef
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 1,
                                                                            'strcount' => 1,
                                                                            'dircount' => 0,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 0,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'pattern' => '&',
                                                                                                  'hashname' => '__STRING1__',
                                                                                                  'description' => '\'&\'',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 293
                                                                                                }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 293
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 294,
                                                                                                  'code' => '{ $item[2] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 293
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 2,
                                                                            'strcount' => 0,
                                                                            'dircount' => 0,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/^(?=<<)/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '^(?=<<)',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 296
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'inline_doc',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 296
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 297,
                                                                                                  'code' => '{ $item[2] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 296
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 3,
                                                                            'strcount' => 0,
                                                                            'dircount' => 0,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 0,
                                                                            'actcount' => 0,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'subrule' => 'xpinter',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 299
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' )
                                                                                       ],
                                                                            'line' => 299
                                                                          }, 'Parse::RecDescent::_Runtime::Production' )
                                                                 ],
                                                      'name' => 'loose_exp',
                                                      'vars' => '',
                                                      'changed' => 0,
                                                      'line' => 289
                                                    }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'trail' => bless( {
                                                  'impcount' => 0,
                                                  'calls' => [
                                                               'variable',
                                                               'shline'
                                                             ],
                                                  'opcount' => 0,
                                                  'prods' => [
                                                               bless( {
                                                                        'number' => 0,
                                                                        'strcount' => 0,
                                                                        'dircount' => 2,
                                                                        'uncommit' => undef,
                                                                        'error' => undef,
                                                                        'patcount' => 1,
                                                                        'actcount' => 0,
                                                                        'items' => [
                                                                                     bless( {
                                                                                              'description' => '/(?=\\\\s*[\\};]|\\\\s*\\\\Z)/',
                                                                                              'rdelim' => '/',
                                                                                              'pattern' => '(?=\\s*[};]|\\s*\\Z)',
                                                                                              'hashname' => '__PATTERN1__',
                                                                                              'lookahead' => 0,
                                                                                              'ldelim' => '/',
                                                                                              'mod' => '',
                                                                                              'line' => 330
                                                                                            }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                     bless( {
                                                                                              'hashname' => '__DIRECTIVE1__',
                                                                                              'name' => '<commit>',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 330,
                                                                                              'code' => '$commit = 1'
                                                                                            }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                     bless( {
                                                                                              'hashname' => '__DIRECTIVE2__',
                                                                                              'name' => '<reject>',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 330
                                                                                            }, 'Parse::RecDescent::_Runtime::UncondReject' )
                                                                                   ],
                                                                        'line' => undef
                                                                      }, 'Parse::RecDescent::_Runtime::Production' ),
                                                               bless( {
                                                                        'number' => 1,
                                                                        'strcount' => 1,
                                                                        'dircount' => 1,
                                                                        'uncommit' => undef,
                                                                        'error' => undef,
                                                                        'patcount' => 0,
                                                                        'actcount' => 1,
                                                                        'items' => [
                                                                                     bless( {
                                                                                              'pattern' => '|>',
                                                                                              'hashname' => '__STRING1__',
                                                                                              'description' => '\'|>\'',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 331
                                                                                            }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                     bless( {
                                                                                              'hashname' => '__DIRECTIVE1__',
                                                                                              'name' => '<commit>',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 331,
                                                                                              'code' => '$commit = 1'
                                                                                            }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                     bless( {
                                                                                              'subrule' => 'variable',
                                                                                              'matchrule' => 0,
                                                                                              'implicit' => undef,
                                                                                              'argcode' => undef,
                                                                                              'lookahead' => 0,
                                                                                              'line' => 331
                                                                                            }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                     bless( {
                                                                                              'hashname' => '__ACTION1__',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 332,
                                                                                              'code' => '{ [\'var\',$item[3]] }'
                                                                                            }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                   ],
                                                                        'line' => 331
                                                                      }, 'Parse::RecDescent::_Runtime::Production' ),
                                                               bless( {
                                                                        'number' => 2,
                                                                        'strcount' => 1,
                                                                        'dircount' => 1,
                                                                        'uncommit' => undef,
                                                                        'error' => undef,
                                                                        'patcount' => 0,
                                                                        'actcount' => 1,
                                                                        'items' => [
                                                                                     bless( {
                                                                                              'pattern' => '|',
                                                                                              'hashname' => '__STRING1__',
                                                                                              'description' => '\'|\'',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 334
                                                                                            }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                     bless( {
                                                                                              'hashname' => '__DIRECTIVE1__',
                                                                                              'name' => '<commit>',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 334,
                                                                                              'code' => '$commit = 1'
                                                                                            }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                     bless( {
                                                                                              'subrule' => 'shline',
                                                                                              'matchrule' => 0,
                                                                                              'implicit' => undef,
                                                                                              'argcode' => undef,
                                                                                              'lookahead' => 0,
                                                                                              'line' => 334
                                                                                            }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                     bless( {
                                                                                              'hashname' => '__ACTION1__',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 335,
                                                                                              'code' => '{ [\'pipe\',$item[3]] }'
                                                                                            }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                   ],
                                                                        'line' => 334
                                                                      }, 'Parse::RecDescent::_Runtime::Production' )
                                                             ],
                                                  'name' => 'trail',
                                                  'vars' => '',
                                                  'changed' => 0,
                                                  'line' => 329
                                                }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'xpbrackets' => bless( {
                                                       'impcount' => 0,
                                                       'calls' => [
                                                                    'xpbracket',
                                                                    'xpfilters'
                                                                  ],
                                                       'opcount' => 0,
                                                       'prods' => [
                                                                    bless( {
                                                                             'number' => 0,
                                                                             'strcount' => 0,
                                                                             'dircount' => 1,
                                                                             'uncommit' => undef,
                                                                             'error' => undef,
                                                                             'patcount' => 1,
                                                                             'actcount' => 1,
                                                                             'items' => [
                                                                                          bless( {
                                                                                                   'description' => '/(?=\\\\()/',
                                                                                                   'rdelim' => '/',
                                                                                                   'pattern' => '(?=\\()',
                                                                                                   'hashname' => '__PATTERN1__',
                                                                                                   'lookahead' => 0,
                                                                                                   'ldelim' => '/',
                                                                                                   'mod' => '',
                                                                                                   'line' => 227
                                                                                                 }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                          bless( {
                                                                                                   'subrule' => 'xpbracket',
                                                                                                   'matchrule' => 0,
                                                                                                   'implicit' => undef,
                                                                                                   'argcode' => undef,
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 227
                                                                                                 }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                          bless( {
                                                                                                   'hashname' => '__DIRECTIVE1__',
                                                                                                   'name' => '<skip:"">',
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 227,
                                                                                                   'code' => 'my $oldskip = $skip; $skip=""; $oldskip'
                                                                                                 }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                          bless( {
                                                                                                   'subrule' => 'xpfilters',
                                                                                                   'expected' => undef,
                                                                                                   'min' => 0,
                                                                                                   'argcode' => undef,
                                                                                                   'max' => 1,
                                                                                                   'matchrule' => 0,
                                                                                                   'repspec' => '?',
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 227
                                                                                                 }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                          bless( {
                                                                                                   'hashname' => '__ACTION1__',
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 228,
                                                                                                   'code' => '{ join "",$item[2],@{$item[4]} }'
                                                                                                 }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                        ],
                                                                             'line' => undef
                                                                           }, 'Parse::RecDescent::_Runtime::Production' )
                                                                  ],
                                                       'name' => 'xpbrackets',
                                                       'vars' => '',
                                                       'changed' => 0,
                                                       'line' => 226
                                                     }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'xpintercont' => bless( {
                                                        'impcount' => 1,
                                                        'calls' => [
                                                                     '_alternation_1_of_production_1_of_rule_xpintercont',
                                                                     'xpinter'
                                                                   ],
                                                        'opcount' => 0,
                                                        'prods' => [
                                                                     bless( {
                                                                              'number' => 0,
                                                                              'strcount' => 0,
                                                                              'dircount' => 1,
                                                                              'uncommit' => undef,
                                                                              'error' => undef,
                                                                              'patcount' => 0,
                                                                              'actcount' => 1,
                                                                              'items' => [
                                                                                           bless( {
                                                                                                    'subrule' => '_alternation_1_of_production_1_of_rule_xpintercont',
                                                                                                    'matchrule' => 0,
                                                                                                    'implicit' => 'xpfilters, or xpbrackets',
                                                                                                    'argcode' => undef,
                                                                                                    'lookahead' => 0,
                                                                                                    'line' => 239
                                                                                                  }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                           bless( {
                                                                                                    'hashname' => '__DIRECTIVE1__',
                                                                                                    'name' => '<skip:"">',
                                                                                                    'lookahead' => 0,
                                                                                                    'line' => 239,
                                                                                                    'code' => 'my $oldskip = $skip; $skip=""; $oldskip'
                                                                                                  }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                           bless( {
                                                                                                    'subrule' => 'xpinter',
                                                                                                    'expected' => undef,
                                                                                                    'min' => 0,
                                                                                                    'argcode' => undef,
                                                                                                    'max' => 1,
                                                                                                    'matchrule' => 0,
                                                                                                    'repspec' => '?',
                                                                                                    'lookahead' => 0,
                                                                                                    'line' => 239
                                                                                                  }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                           bless( {
                                                                                                    'hashname' => '__ACTION1__',
                                                                                                    'lookahead' => 0,
                                                                                                    'line' => 240,
                                                                                                    'code' => '{ join("",$item[1],@{$item[3]}) }'
                                                                                                  }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                         ],
                                                                              'line' => undef
                                                                            }, 'Parse::RecDescent::_Runtime::Production' )
                                                                   ],
                                                        'name' => 'xpintercont',
                                                        'vars' => '',
                                                        'changed' => 0,
                                                        'line' => 236
                                                      }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'inline_doc_string' => bless( {
                                                              'impcount' => 0,
                                                              'calls' => [],
                                                              'opcount' => 0,
                                                              'prods' => [
                                                                           bless( {
                                                                                    'number' => 0,
                                                                                    'strcount' => 0,
                                                                                    'dircount' => 0,
                                                                                    'uncommit' => undef,
                                                                                    'error' => undef,
                                                                                    'patcount' => 1,
                                                                                    'actcount' => 1,
                                                                                    'items' => [
                                                                                                 bless( {
                                                                                                          'description' => '/\'[a-zA-Z_][a-zA-Z0-9_]*\'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\\\([a-zA-Z_][a-zA-Z0-9_]*\\\\)|\\\\\\{[a-zA-Z_][a-zA-Z0-9_]*\\\\\\}|[a-zA-Z_][a-zA-Z0-9_]*/',
                                                                                                          'rdelim' => '/',
                                                                                                          'pattern' => '\'[a-zA-Z_][a-zA-Z0-9_]*\'|"[a-zA-Z_][a-zA-Z0-9_]*"|\\([a-zA-Z_][a-zA-Z0-9_]*\\)|\\{[a-zA-Z_][a-zA-Z0-9_]*\\}|[a-zA-Z_][a-zA-Z0-9_]*',
                                                                                                          'hashname' => '__PATTERN1__',
                                                                                                          'lookahead' => 0,
                                                                                                          'ldelim' => '/',
                                                                                                          'mod' => '',
                                                                                                          'line' => 144
                                                                                                        }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                                 bless( {
                                                                                                          'hashname' => '__ACTION1__',
                                                                                                          'lookahead' => 0,
                                                                                                          'line' => 145,
                                                                                                          'code' => '{ [($item[1]=~/^([\'"({])?(.*?)([\'")}])?$/)] }'
                                                                                                        }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                               ],
                                                                                    'line' => undef
                                                                                  }, 'Parse::RecDescent::_Runtime::Production' )
                                                                         ],
                                                              'name' => 'inline_doc_string',
                                                              'vars' => '',
                                                              'changed' => 0,
                                                              'line' => 143
                                                            }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'shline_nosc' => bless( {
                                                        'impcount' => 0,
                                                        'calls' => [],
                                                        'opcount' => 0,
                                                        'prods' => [
                                                                     bless( {
                                                                              'number' => 0,
                                                                              'strcount' => 0,
                                                                              'dircount' => 0,
                                                                              'uncommit' => undef,
                                                                              'error' => undef,
                                                                              'patcount' => 1,
                                                                              'actcount' => 0,
                                                                              'items' => [
                                                                                           bless( {
                                                                                                    'description' => '/([^;()\\\\\\\\"\'\\\\|]|\\\\|[^>]|\\\\\\\\.|\\\\"([^\\\\"\\\\\\\\]|\\\\\\\\.)*\\\\"|\\\\\'([^\\\\\'\\\\\\\\]|\\\\\\\\\\\\\'|\\\\\\\\\\\\\\\\|\\\\\\\\[^\\\\\'\\\\\\\\])*\\\\\')*/',
                                                                                                    'rdelim' => '/',
                                                                                                    'pattern' => '([^;()\\\\"\'\\|]|\\|[^>]|\\\\.|\\"([^\\"\\\\]|\\\\.)*\\"|\\\'([^\\\'\\\\]|\\\\\\\'|\\\\\\\\|\\\\[^\\\'\\\\])*\\\')*',
                                                                                                    'hashname' => '__PATTERN1__',
                                                                                                    'lookahead' => 0,
                                                                                                    'ldelim' => '/',
                                                                                                    'mod' => '',
                                                                                                    'line' => 339
                                                                                                  }, 'Parse::RecDescent::_Runtime::Token' )
                                                                                         ],
                                                                              'line' => undef
                                                                            }, 'Parse::RecDescent::_Runtime::Production' )
                                                                   ],
                                                        'name' => 'shline_nosc',
                                                        'vars' => '',
                                                        'changed' => 0,
                                                        'line' => 338
                                                      }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'xpinter' => bless( {
                                                    'impcount' => 0,
                                                    'calls' => [
                                                                 'xps',
                                                                 'xpintercont'
                                                               ],
                                                    'opcount' => 0,
                                                    'prods' => [
                                                                 bless( {
                                                                          'number' => 0,
                                                                          'strcount' => 0,
                                                                          'dircount' => 1,
                                                                          'uncommit' => undef,
                                                                          'error' => undef,
                                                                          'patcount' => 0,
                                                                          'actcount' => 1,
                                                                          'items' => [
                                                                                       bless( {
                                                                                                'subrule' => 'xps',
                                                                                                'matchrule' => 0,
                                                                                                'implicit' => undef,
                                                                                                'argcode' => undef,
                                                                                                'lookahead' => 0,
                                                                                                'line' => 232
                                                                                              }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                       bless( {
                                                                                                'hashname' => '__DIRECTIVE1__',
                                                                                                'name' => '<skip:"">',
                                                                                                'lookahead' => 0,
                                                                                                'line' => 232,
                                                                                                'code' => 'my $oldskip = $skip; $skip=""; $oldskip'
                                                                                              }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                       bless( {
                                                                                                'subrule' => 'xpintercont',
                                                                                                'expected' => undef,
                                                                                                'min' => 0,
                                                                                                'argcode' => undef,
                                                                                                'max' => 1,
                                                                                                'matchrule' => 0,
                                                                                                'repspec' => '?',
                                                                                                'lookahead' => 0,
                                                                                                'line' => 232
                                                                                              }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                       bless( {
                                                                                                'hashname' => '__ACTION1__',
                                                                                                'lookahead' => 0,
                                                                                                'line' => 233,
                                                                                                'code' => '{ join("",$item[1],@{$item[3]}) }'
                                                                                              }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                     ],
                                                                          'line' => undef
                                                                        }, 'Parse::RecDescent::_Runtime::Production' )
                                                               ],
                                                    'name' => 'xpinter',
                                                    'vars' => '',
                                                    'changed' => 0,
                                                    'line' => 231
                                                  }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'inline_doc' => bless( {
                                                       'impcount' => 0,
                                                       'calls' => [
                                                                    'inline_doc_string'
                                                                  ],
                                                       'opcount' => 0,
                                                       'prods' => [
                                                                    bless( {
                                                                             'number' => 0,
                                                                             'strcount' => 0,
                                                                             'dircount' => 1,
                                                                             'uncommit' => undef,
                                                                             'error' => undef,
                                                                             'patcount' => 3,
                                                                             'actcount' => 1,
                                                                             'items' => [
                                                                                          bless( {
                                                                                                   'description' => '/<</',
                                                                                                   'rdelim' => '/',
                                                                                                   'pattern' => '<<',
                                                                                                   'hashname' => '__PATTERN1__',
                                                                                                   'lookahead' => 0,
                                                                                                   'ldelim' => '/',
                                                                                                   'mod' => '',
                                                                                                   'line' => 149
                                                                                                 }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                          bless( {
                                                                                                   'subrule' => 'inline_doc_string',
                                                                                                   'matchrule' => 0,
                                                                                                   'implicit' => undef,
                                                                                                   'argcode' => undef,
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 149
                                                                                                 }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                          bless( {
                                                                                                   'hashname' => '__DIRECTIVE1__',
                                                                                                   'name' => '<skip:"">',
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 149,
                                                                                                   'code' => 'my $oldskip = $skip; $skip=""; $oldskip'
                                                                                                 }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                          bless( {
                                                                                                   'description' => '/.*\\\\n/',
                                                                                                   'rdelim' => '/',
                                                                                                   'pattern' => '.*\\n',
                                                                                                   'hashname' => '__PATTERN2__',
                                                                                                   'lookahead' => 0,
                                                                                                   'ldelim' => '/',
                                                                                                   'mod' => '',
                                                                                                   'line' => 149
                                                                                                 }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                          bless( {
                                                                                                   'description' => '/(.|\\\\n)*?\\\\n$item[2][1]\\\\s*(\\\\n|$)/',
                                                                                                   'rdelim' => '/',
                                                                                                   'pattern' => '(.|\\n)*?\\n$item[2][1]\\s*(\\n|$)',
                                                                                                   'hashname' => '__PATTERN3__',
                                                                                                   'lookahead' => 0,
                                                                                                   'ldelim' => '/',
                                                                                                   'mod' => '',
                                                                                                   'line' => 149
                                                                                                 }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                          bless( {
                                                                                                   'hashname' => '__ACTION1__',
                                                                                                   'lookahead' => 0,
                                                                                                   'line' => 150,
                                                                                                   'code' => '{ 
	  $text=$item[4].$text;
	  local $_=$item[5]; s/\\n$item[2][1]\\s*$//;
	  my $paren = $item[2][0];
	  $paren = \'"\' if $paren eq "";
	  if ($paren eq "\'") {
	      $paren = \'"\';
	      s{(\\\\)(.|\\n)|(\\$)}{ (defined($3) and $3 eq \'$\') ? "\\\\\\$"
	        : ((defined($2) and $2 eq "\\\\")
	        ? "\\\\\\\\" : ((defined($2) and $2 eq "\'") ? "\'" : (
	        (defined($2) and $2 eq \'$\') ? "\\\\\\\\\\\\$2" :
	        "\\\\\\\\$2"))) }eg;
	  } "<<".$paren.$_;
	 }'
                                                                                                 }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                        ],
                                                                             'line' => undef
                                                                           }, 'Parse::RecDescent::_Runtime::Production' )
                                                                  ],
                                                       'name' => 'inline_doc',
                                                       'vars' => '',
                                                       'changed' => 0,
                                                       'line' => 148
                                                     }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'statement' => bless( {
                                                      'impcount' => 0,
                                                      'calls' => [
                                                                   'exp',
                                                                   'block',
                                                                   'elsif_block',
                                                                   'else_block',
                                                                   'local_var_in',
                                                                   'local_var',
                                                                   'xpstep',
                                                                   'ID',
                                                                   'variable'
                                                                 ],
                                                      'opcount' => 0,
                                                      'prods' => [
                                                                   bless( {
                                                                            'number' => 0,
                                                                            'strcount' => 0,
                                                                            'dircount' => 2,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 0,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(?=\\\\s*[\\}\\{;])/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(?=\\s*[}{;])',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 72
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 72,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE2__',
                                                                                                  'name' => '<reject>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 72
                                                                                                }, 'Parse::RecDescent::_Runtime::UncondReject' )
                                                                                       ],
                                                                            'line' => undef
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 1,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(if)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(if)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 73
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 73,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'exp',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 73
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 73
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'elsif_block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 73
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'else_block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 73
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 74,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'if_statement\',[$item[3],$item[4]],@{$item[5]},@{$item[6]}] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 73
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 2,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(unless)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(unless)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 76
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 76,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'exp',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 76
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 76
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'else_block',
                                                                                                  'expected' => undef,
                                                                                                  'min' => 0,
                                                                                                  'argcode' => undef,
                                                                                                  'max' => 1,
                                                                                                  'matchrule' => 0,
                                                                                                  'repspec' => '?',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 76
                                                                                                }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 77,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'unless_statement\',$item[3],$item[4],@{$item[5]}] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 76
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 3,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(while)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(while)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 79
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 79,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'exp',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 79
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 79
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 80,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'while_statement\',$item[3],$item[4]] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 79
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 4,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(foreach|for)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(foreach|for)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 82
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 82,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'local_var_in',
                                                                                                  'expected' => undef,
                                                                                                  'min' => 0,
                                                                                                  'argcode' => undef,
                                                                                                  'max' => 1,
                                                                                                  'matchrule' => 0,
                                                                                                  'repspec' => '?',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 82
                                                                                                }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'exp',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 82
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 82
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 83,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'foreach_statement\',@item[4,5],@{$item[3]}] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 82
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 5,
                                                                            'strcount' => 1,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(try)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(try)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 85
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 85,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 85
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'pattern' => 'catch',
                                                                                                  'hashname' => '__STRING1__',
                                                                                                  'description' => '\'catch\'',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 85
                                                                                                }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'local_var',
                                                                                                  'expected' => undef,
                                                                                                  'min' => 0,
                                                                                                  'argcode' => undef,
                                                                                                  'max' => 1,
                                                                                                  'matchrule' => 0,
                                                                                                  'repspec' => '?',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 85
                                                                                                }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 85
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 86,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'try_catch\',$item[3],$item[6],@{$item[5]}] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 85
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 6,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(iterate)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(iterate)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 88
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 88,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'xpstep',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 88
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 88
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 89,
                                                                                                  'code' => '{ [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'iterate\',$item[4],@{$item[3]}] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 88
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 7,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 1,
                                                                            'actcount' => 2,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'description' => '/(def|define)\\\\b/',
                                                                                                  'rdelim' => '/',
                                                                                                  'pattern' => '(def|define)\\b',
                                                                                                  'hashname' => '__PATTERN1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'ldelim' => '/',
                                                                                                  'mod' => '',
                                                                                                  'line' => 91
                                                                                                }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 91,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'ID',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 91
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 92,
                                                                                                  'code' => '{ XML::XSH2::Functions::is_command($item[3])?undef:1 }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'variable',
                                                                                                  'expected' => undef,
                                                                                                  'min' => 0,
                                                                                                  'argcode' => undef,
                                                                                                  'max' => 100000000,
                                                                                                  'matchrule' => 0,
                                                                                                  'repspec' => 's?',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 93
                                                                                                }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'block',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 93
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION2__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 94,
                                                                                                  'code' => '{ 
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'def\',$item[3],$item[6],$item[5]]
	 }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 91
                                                                          }, 'Parse::RecDescent::_Runtime::Production' )
                                                                 ],
                                                      'name' => 'statement',
                                                      'vars' => '',
                                                      'changed' => 0,
                                                      'line' => 71
                                                    }, 'Parse::RecDescent::_Runtime::Rule' ),
                              '_alternation_1_of_production_1_of_rule_xpintercont' => bless( {
                                                                                               'impcount' => 0,
                                                                                               'calls' => [
                                                                                                            'xpfilters',
                                                                                                            'xpbrackets'
                                                                                                          ],
                                                                                               'opcount' => 0,
                                                                                               'prods' => [
                                                                                                            bless( {
                                                                                                                     'number' => 0,
                                                                                                                     'strcount' => 0,
                                                                                                                     'dircount' => 0,
                                                                                                                     'uncommit' => undef,
                                                                                                                     'error' => undef,
                                                                                                                     'patcount' => 0,
                                                                                                                     'actcount' => 0,
                                                                                                                     'items' => [
                                                                                                                                  bless( {
                                                                                                                                           'subrule' => 'xpfilters',
                                                                                                                                           'matchrule' => 0,
                                                                                                                                           'implicit' => undef,
                                                                                                                                           'argcode' => undef,
                                                                                                                                           'lookahead' => 0,
                                                                                                                                           'line' => 445
                                                                                                                                         }, 'Parse::RecDescent::_Runtime::Subrule' )
                                                                                                                                ],
                                                                                                                     'line' => undef
                                                                                                                   }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                                                            bless( {
                                                                                                                     'number' => 1,
                                                                                                                     'strcount' => 0,
                                                                                                                     'dircount' => 0,
                                                                                                                     'uncommit' => undef,
                                                                                                                     'error' => undef,
                                                                                                                     'patcount' => 0,
                                                                                                                     'actcount' => 0,
                                                                                                                     'items' => [
                                                                                                                                  bless( {
                                                                                                                                           'subrule' => 'xpbrackets',
                                                                                                                                           'matchrule' => 0,
                                                                                                                                           'implicit' => undef,
                                                                                                                                           'argcode' => undef,
                                                                                                                                           'lookahead' => 0,
                                                                                                                                           'line' => 446
                                                                                                                                         }, 'Parse::RecDescent::_Runtime::Subrule' )
                                                                                                                                ],
                                                                                                                     'line' => 446
                                                                                                                   }, 'Parse::RecDescent::_Runtime::Production' )
                                                                                                          ],
                                                                                               'name' => '_alternation_1_of_production_1_of_rule_xpintercont',
                                                                                               'vars' => '',
                                                                                               'changed' => 0,
                                                                                               'line' => 445
                                                                                             }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'xplocationstep' => bless( {
                                                           'impcount' => 0,
                                                           'calls' => [
                                                                        'xpaxis',
                                                                        'xpnodetest'
                                                                      ],
                                                           'opcount' => 0,
                                                           'prods' => [
                                                                        bless( {
                                                                                 'number' => 0,
                                                                                 'strcount' => 0,
                                                                                 'dircount' => 0,
                                                                                 'uncommit' => undef,
                                                                                 'error' => undef,
                                                                                 'patcount' => 0,
                                                                                 'actcount' => 1,
                                                                                 'items' => [
                                                                                              bless( {
                                                                                                       'subrule' => 'xpaxis',
                                                                                                       'expected' => undef,
                                                                                                       'min' => 0,
                                                                                                       'argcode' => undef,
                                                                                                       'max' => 1,
                                                                                                       'matchrule' => 0,
                                                                                                       'repspec' => '?',
                                                                                                       'lookahead' => 0,
                                                                                                       'line' => 431
                                                                                                     }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                              bless( {
                                                                                                       'subrule' => 'xpnodetest',
                                                                                                       'matchrule' => 0,
                                                                                                       'implicit' => undef,
                                                                                                       'argcode' => undef,
                                                                                                       'lookahead' => 0,
                                                                                                       'line' => 431
                                                                                                     }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                              bless( {
                                                                                                       'hashname' => '__ACTION1__',
                                                                                                       'lookahead' => 0,
                                                                                                       'line' => 432,
                                                                                                       'code' => '{ [ (@{$item[1]} ? $item[1][0] : \'child::\'),$item[3] ] }'
                                                                                                     }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                            ],
                                                                                 'line' => undef
                                                                               }, 'Parse::RecDescent::_Runtime::Production' )
                                                                      ],
                                                           'name' => 'xplocationstep',
                                                           'vars' => '',
                                                           'changed' => 0,
                                                           'line' => 430
                                                         }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'local_var_in' => bless( {
                                                         'impcount' => 0,
                                                         'calls' => [
                                                                      'local_var'
                                                                    ],
                                                         'opcount' => 0,
                                                         'prods' => [
                                                                      bless( {
                                                                               'number' => 0,
                                                                               'strcount' => 1,
                                                                               'dircount' => 0,
                                                                               'uncommit' => undef,
                                                                               'error' => undef,
                                                                               'patcount' => 0,
                                                                               'actcount' => 1,
                                                                               'items' => [
                                                                                            bless( {
                                                                                                     'subrule' => 'local_var',
                                                                                                     'matchrule' => 0,
                                                                                                     'implicit' => undef,
                                                                                                     'argcode' => undef,
                                                                                                     'lookahead' => 0,
                                                                                                     'line' => 385
                                                                                                   }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                            bless( {
                                                                                                     'pattern' => 'in',
                                                                                                     'hashname' => '__STRING1__',
                                                                                                     'description' => '\'in\'',
                                                                                                     'lookahead' => 0,
                                                                                                     'line' => 385
                                                                                                   }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                            bless( {
                                                                                                     'hashname' => '__ACTION1__',
                                                                                                     'lookahead' => 0,
                                                                                                     'line' => 386,
                                                                                                     'code' => '{ $item[1] }'
                                                                                                   }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                          ],
                                                                               'line' => undef
                                                                             }, 'Parse::RecDescent::_Runtime::Production' )
                                                                    ],
                                                         'name' => 'local_var_in',
                                                         'vars' => '',
                                                         'changed' => 0,
                                                         'line' => 384
                                                       }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'param' => bless( {
                                                  'impcount' => 0,
                                                  'calls' => [
                                                               'exp'
                                                             ],
                                                  'opcount' => 0,
                                                  'prods' => [
                                                               bless( {
                                                                        'number' => 0,
                                                                        'strcount' => 1,
                                                                        'dircount' => 0,
                                                                        'uncommit' => undef,
                                                                        'error' => undef,
                                                                        'patcount' => 1,
                                                                        'actcount' => 1,
                                                                        'items' => [
                                                                                     bless( {
                                                                                              'description' => '/[^=\\\\s]+/',
                                                                                              'rdelim' => '/',
                                                                                              'pattern' => '[^=\\s]+',
                                                                                              'hashname' => '__PATTERN1__',
                                                                                              'lookahead' => 0,
                                                                                              'ldelim' => '/',
                                                                                              'mod' => '',
                                                                                              'line' => 397
                                                                                            }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                     bless( {
                                                                                              'pattern' => '=',
                                                                                              'hashname' => '__STRING1__',
                                                                                              'description' => '\'=\'',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 397
                                                                                            }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                     bless( {
                                                                                              'subrule' => 'exp',
                                                                                              'matchrule' => 0,
                                                                                              'implicit' => undef,
                                                                                              'argcode' => undef,
                                                                                              'lookahead' => 0,
                                                                                              'line' => 397
                                                                                            }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                     bless( {
                                                                                              'hashname' => '__ACTION1__',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 398,
                                                                                              'code' => '{ [$item[1],$item[3]] }'
                                                                                            }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                   ],
                                                                        'line' => undef
                                                                      }, 'Parse::RecDescent::_Runtime::Production' )
                                                             ],
                                                  'name' => 'param',
                                                  'vars' => '',
                                                  'changed' => 0,
                                                  'line' => 396
                                                }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'startrule' => bless( {
                                                      'impcount' => 0,
                                                      'calls' => [
                                                                   'shell',
                                                                   'eof',
                                                                   'complex_command'
                                                                 ],
                                                      'opcount' => 0,
                                                      'prods' => [
                                                                   bless( {
                                                                            'number' => 0,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 0,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'subrule' => 'shell',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 322
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 322,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'eof',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 322
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 323,
                                                                                                  'code' => '{ $item[1] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => undef
                                                                          }, 'Parse::RecDescent::_Runtime::Production' ),
                                                                   bless( {
                                                                            'number' => 1,
                                                                            'strcount' => 0,
                                                                            'dircount' => 1,
                                                                            'uncommit' => undef,
                                                                            'error' => undef,
                                                                            'patcount' => 0,
                                                                            'actcount' => 1,
                                                                            'items' => [
                                                                                         bless( {
                                                                                                  'subrule' => 'complex_command',
                                                                                                  'expected' => undef,
                                                                                                  'min' => 1,
                                                                                                  'argcode' => undef,
                                                                                                  'max' => 100000000,
                                                                                                  'matchrule' => 0,
                                                                                                  'repspec' => 's',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 325
                                                                                                }, 'Parse::RecDescent::_Runtime::Repetition' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__DIRECTIVE1__',
                                                                                                  'name' => '<commit>',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 325,
                                                                                                  'code' => '$commit = 1'
                                                                                                }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                         bless( {
                                                                                                  'subrule' => 'eof',
                                                                                                  'matchrule' => 0,
                                                                                                  'implicit' => undef,
                                                                                                  'argcode' => undef,
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 325
                                                                                                }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                         bless( {
                                                                                                  'hashname' => '__ACTION1__',
                                                                                                  'lookahead' => 0,
                                                                                                  'line' => 326,
                                                                                                  'code' => '{ $item[1] }'
                                                                                                }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                       ],
                                                                            'line' => 325
                                                                          }, 'Parse::RecDescent::_Runtime::Production' )
                                                                 ],
                                                      'name' => 'startrule',
                                                      'vars' => '',
                                                      'changed' => 0,
                                                      'line' => 321
                                                    }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'exp' => bless( {
                                                'impcount' => 0,
                                                'calls' => [
                                                             'perl_block',
                                                             'block',
                                                             'inline_doc',
                                                             'xpath'
                                                           ],
                                                'opcount' => 0,
                                                'prods' => [
                                                             bless( {
                                                                      'number' => 0,
                                                                      'strcount' => 0,
                                                                      'dircount' => 0,
                                                                      'uncommit' => undef,
                                                                      'error' => undef,
                                                                      'patcount' => 1,
                                                                      'actcount' => 1,
                                                                      'items' => [
                                                                                   bless( {
                                                                                            'description' => '/^(?=\\{)/',
                                                                                            'rdelim' => '/',
                                                                                            'pattern' => '^(?={)',
                                                                                            'hashname' => '__PATTERN1__',
                                                                                            'lookahead' => 0,
                                                                                            'ldelim' => '/',
                                                                                            'mod' => '',
                                                                                            'line' => 302
                                                                                          }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                   bless( {
                                                                                            'subrule' => 'perl_block',
                                                                                            'matchrule' => 0,
                                                                                            'implicit' => undef,
                                                                                            'argcode' => undef,
                                                                                            'lookahead' => 0,
                                                                                            'line' => 302
                                                                                          }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                   bless( {
                                                                                            'hashname' => '__ACTION1__',
                                                                                            'lookahead' => 0,
                                                                                            'line' => 303,
                                                                                            'code' => '{ $item[2] }'
                                                                                          }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                 ],
                                                                      'line' => undef
                                                                    }, 'Parse::RecDescent::_Runtime::Production' ),
                                                             bless( {
                                                                      'number' => 1,
                                                                      'strcount' => 1,
                                                                      'dircount' => 0,
                                                                      'uncommit' => undef,
                                                                      'error' => undef,
                                                                      'patcount' => 0,
                                                                      'actcount' => 1,
                                                                      'items' => [
                                                                                   bless( {
                                                                                            'pattern' => '&',
                                                                                            'hashname' => '__STRING1__',
                                                                                            'description' => '\'&\'',
                                                                                            'lookahead' => 0,
                                                                                            'line' => 305
                                                                                          }, 'Parse::RecDescent::_Runtime::Literal' ),
                                                                                   bless( {
                                                                                            'subrule' => 'block',
                                                                                            'matchrule' => 0,
                                                                                            'implicit' => undef,
                                                                                            'argcode' => undef,
                                                                                            'lookahead' => 0,
                                                                                            'line' => 305
                                                                                          }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                   bless( {
                                                                                            'hashname' => '__ACTION1__',
                                                                                            'lookahead' => 0,
                                                                                            'line' => 306,
                                                                                            'code' => '{ $item[2] }'
                                                                                          }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                 ],
                                                                      'line' => 305
                                                                    }, 'Parse::RecDescent::_Runtime::Production' ),
                                                             bless( {
                                                                      'number' => 2,
                                                                      'strcount' => 0,
                                                                      'dircount' => 0,
                                                                      'uncommit' => undef,
                                                                      'error' => undef,
                                                                      'patcount' => 1,
                                                                      'actcount' => 1,
                                                                      'items' => [
                                                                                   bless( {
                                                                                            'description' => '/^(?=<<)/',
                                                                                            'rdelim' => '/',
                                                                                            'pattern' => '^(?=<<)',
                                                                                            'hashname' => '__PATTERN1__',
                                                                                            'lookahead' => 0,
                                                                                            'ldelim' => '/',
                                                                                            'mod' => '',
                                                                                            'line' => 308
                                                                                          }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                   bless( {
                                                                                            'subrule' => 'inline_doc',
                                                                                            'matchrule' => 0,
                                                                                            'implicit' => undef,
                                                                                            'argcode' => undef,
                                                                                            'lookahead' => 0,
                                                                                            'line' => 308
                                                                                          }, 'Parse::RecDescent::_Runtime::Subrule' ),
                                                                                   bless( {
                                                                                            'hashname' => '__ACTION1__',
                                                                                            'lookahead' => 0,
                                                                                            'line' => 309,
                                                                                            'code' => '{ $item[2] }'
                                                                                          }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                 ],
                                                                      'line' => 308
                                                                    }, 'Parse::RecDescent::_Runtime::Production' ),
                                                             bless( {
                                                                      'number' => 3,
                                                                      'strcount' => 0,
                                                                      'dircount' => 0,
                                                                      'uncommit' => undef,
                                                                      'error' => undef,
                                                                      'patcount' => 0,
                                                                      'actcount' => 0,
                                                                      'items' => [
                                                                                   bless( {
                                                                                            'subrule' => 'xpath',
                                                                                            'matchrule' => 0,
                                                                                            'implicit' => undef,
                                                                                            'argcode' => undef,
                                                                                            'lookahead' => 0,
                                                                                            'line' => 311
                                                                                          }, 'Parse::RecDescent::_Runtime::Subrule' )
                                                                                 ],
                                                                      'line' => 311
                                                                    }, 'Parse::RecDescent::_Runtime::Production' )
                                                           ],
                                                'name' => 'exp',
                                                'vars' => '',
                                                'changed' => 0,
                                                'line' => 301
                                              }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'xpstring' => bless( {
                                                     'impcount' => 0,
                                                     'calls' => [],
                                                     'opcount' => 0,
                                                     'prods' => [
                                                                  bless( {
                                                                           'number' => 0,
                                                                           'strcount' => 0,
                                                                           'dircount' => 0,
                                                                           'uncommit' => undef,
                                                                           'error' => undef,
                                                                           'patcount' => 1,
                                                                           'actcount' => 0,
                                                                           'items' => [
                                                                                        bless( {
                                                                                                 'description' => '/\'(?:\\\\$\\\\\\{(?:\\\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\\\\{.*?\\\\\\}|\\\\(.+?\\\\))\\\\\\}|\\\\$(?!\\\\\\{)|[^\\\\$\'])*\' |
              "(?:\\\\$\\\\\\{(?:\\\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\\\\\{.*?\\\\\\}|\\\\(.+?\\\\))\\\\\\}|\\\\$(?!\\\\\\{)|[^\\\\$"])*"/x',
                                                                                                 'rdelim' => '/',
                                                                                                 'pattern' => '\'(?:\\$\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\{.*?\\}|\\(.+?\\))\\}|\\$(?!\\{)|[^\\$\'])*\' |
              "(?:\\$\\{(?:\\$?[a-zA-Z_][a-zA-Z0-9_]*|\\{.*?\\}|\\(.+?\\))\\}|\\$(?!\\{)|[^\\$"])*"',
                                                                                                 'hashname' => '__PATTERN1__',
                                                                                                 'lookahead' => 0,
                                                                                                 'ldelim' => '/',
                                                                                                 'mod' => 'x',
                                                                                                 'line' => 260
                                                                                               }, 'Parse::RecDescent::_Runtime::Token' )
                                                                                      ],
                                                                           'line' => undef
                                                                         }, 'Parse::RecDescent::_Runtime::Production' )
                                                                ],
                                                     'name' => 'xpstring',
                                                     'vars' => '',
                                                     'changed' => 0,
                                                     'line' => 259
                                                   }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'nodename' => bless( {
                                                     'impcount' => 0,
                                                     'calls' => [
                                                                  'exp'
                                                                ],
                                                     'opcount' => 0,
                                                     'prods' => [
                                                                  bless( {
                                                                           'number' => 0,
                                                                           'strcount' => 0,
                                                                           'dircount' => 0,
                                                                           'uncommit' => undef,
                                                                           'error' => undef,
                                                                           'patcount' => 0,
                                                                           'actcount' => 0,
                                                                           'items' => [
                                                                                        bless( {
                                                                                                 'subrule' => 'exp',
                                                                                                 'matchrule' => 0,
                                                                                                 'implicit' => undef,
                                                                                                 'argcode' => undef,
                                                                                                 'lookahead' => 0,
                                                                                                 'line' => 176
                                                                                               }, 'Parse::RecDescent::_Runtime::Subrule' )
                                                                                      ],
                                                                           'line' => undef
                                                                         }, 'Parse::RecDescent::_Runtime::Production' )
                                                                ],
                                                     'name' => 'nodename',
                                                     'vars' => '',
                                                     'changed' => 0,
                                                     'line' => 175
                                                   }, 'Parse::RecDescent::_Runtime::Rule' ),
                              'undef' => bless( {
                                                  'impcount' => 0,
                                                  'calls' => [],
                                                  'opcount' => 0,
                                                  'prods' => [
                                                               bless( {
                                                                        'number' => 0,
                                                                        'strcount' => 0,
                                                                        'dircount' => 1,
                                                                        'uncommit' => undef,
                                                                        'error' => undef,
                                                                        'patcount' => 2,
                                                                        'actcount' => 1,
                                                                        'items' => [
                                                                                     bless( {
                                                                                              'description' => '/(undef|undefine)\\\\b/',
                                                                                              'rdelim' => '/',
                                                                                              'pattern' => '(undef|undefine)\\b',
                                                                                              'hashname' => '__PATTERN1__',
                                                                                              'lookahead' => 0,
                                                                                              'ldelim' => '/',
                                                                                              'mod' => '',
                                                                                              'line' => 390
                                                                                            }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                     bless( {
                                                                                              'hashname' => '__DIRECTIVE1__',
                                                                                              'name' => '<commit>',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 390,
                                                                                              'code' => '$commit = 1'
                                                                                            }, 'Parse::RecDescent::_Runtime::Directive' ),
                                                                                     bless( {
                                                                                              'description' => '/\\\\$?[a-zA-Z_][a-zA-Z0-9_]*/',
                                                                                              'rdelim' => '/',
                                                                                              'pattern' => '\\$?[a-zA-Z_][a-zA-Z0-9_]*',
                                                                                              'hashname' => '__PATTERN2__',
                                                                                              'lookahead' => 0,
                                                                                              'ldelim' => '/',
                                                                                              'mod' => '',
                                                                                              'line' => 390
                                                                                            }, 'Parse::RecDescent::_Runtime::Token' ),
                                                                                     bless( {
                                                                                              'hashname' => '__ACTION1__',
                                                                                              'lookahead' => 0,
                                                                                              'line' => 391,
                                                                                              'code' => '{ 
	  [$thisline,$thiscolumn,$thisoffset,$XML::XSH2::Functions::SCRIPT,\'undefine\',$item[3]];
	 }'
                                                                                            }, 'Parse::RecDescent::_Runtime::Action' )
                                                                                   ],
 