# ABSTRACT: Data::Seek Search Result Class
package Data::Seek::Search::Result;

use Data::Seek::Data;
use Data::Seek::Exception;
use Data::Seek::Search;

use Data::Object::Class;

our $VERSION = '0.08'; # VERSION

has 'datasets' => (
    is      => 'ro',
    default => sub { shift->search->perform },
    lazy    => 1
);

has 'search' => (
    is      => 'ro',
    default => sub { Data::Seek::Search->new },
    lazy    => 1
);

sub data {
    my $self = shift;
    my $sets = $self->datasets;
    my $data = {};

    for my $set (@$sets) {
        for my $node (@{$$set{nodes}}) {
            $$data{$node} = $$set{dataset}{$node};
        }
    }

    $data = Data::Seek::Data->new($data);
    return $data->decode;
}

sub nodes {
    my $self = shift;
    my $sets = $self->datasets;
    my $keys = [];

    for my $set (@$sets) {
        push @$keys, sort @{$$set{nodes}};
    }

    return $keys;
}

sub values {
    my $self = shift;
    my $sets = $self->datasets;
    my $vals = [];

    for my $set (@$sets) {
        push @$vals, $$set{dataset}{$_} for sort @{$$set{nodes}};
    }

    return $vals;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Data::Seek::Search::Result - Data::Seek Search Result Class

=head1 VERSION

version 0.08

=head1 SYNOPSIS

    use Data::Seek::Search::Result;

=head1 DESCRIPTION

Data::Seek::Search::Result is a class within L<Data::Seek> which provides access
to the search results produced by L<Data::Seek::Search>.

=head1 ATTRIBUTES

=head2 datasets

    my $dataset = $result->dataset;

Perform the search and introspection using the search object,
L<Data::Seek::Search>, and cache the resulting data set.

=head2 search

    my $search = $result->search;

Reference the search object, L<Data::Seek::Search>, which the resulting data set
is derived from.

=head1 METHODS

=head2 data

    my $data = $result->data;

Produce a data structure, i.e. a hash reference, comprised of only the nodes
matching the criteria used in the search.

=head2 nodes

    my $nodes = $result->nodes;

Produce a data structure, i.e. an array reference, comprised of only the node
keys/paths matching the criteria used in the search.

=head2 values

    my $values = $result->values;

Produce a data structure, i.e. an array reference, comprised of only the values
matching the criteria used in the search.

=head1 AUTHOR

Al Newkirk <anewkirk@ana.io>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Al Newkirk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
