# Copyright (C) 2011-2018 A S Lewis
#
# This program is free software: you can redistribute it and/or modify it under the terms of the GNU
# General Public License as published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
# even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with this program. If not,
# see <http://www.gnu.org/licenses/>.
#
#
# Games::Axmud::Session
# The code that handles a single session (each tab in the 'main' window has its own session,
#   representing a single connection to a world - even if we're not currently connected to the
#   world)

{ package Games::Axmud::Session;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(Games::Axmud);

    ##################
    # Constructors

    sub new {

        # Called by GA::Client->startSession (only)
        #
        # Expected arguments
        #   $number         - A unique number for the session (matches a key in
        #                       GA::Client->sessionHash)
        #   $world          - The world's name (matches a world profile name)
        #
        # Optional arguments
        #   $host           - The world's host address (if 'undef', default host address used)
        #   $port           - The world's port (if 'undef', default host port used)
        #   $char           - A character name (matches a character profile name (if 'undef', no
        #                       character profile used)
        #   $pass           - The corresponding password (if 'undef', the world profile is consulted
        #                       to provide the password, if possible)
        #   $account        - The character's associated account name, for worlds that use both
        #                       (if 'undef', no account name used)
        #   $protocol       - If set to 'telnet', 'ssh' or 'ssl', that protocol is used; if 'undef'
        #                       or an unrecognised value, the world profile's ->protocol is used
        #   $loginMode      - Set when called by $self->connectBlind, when a new world profile is to
        #                       be created, and the user has specified what type of ->loginMode this
        #                       world uses; otherwise set to 'undef'
        #   $offlineFlag    - If TRUE, the session doesn't actually connect to the world, but still
        #                       loads all data and makes some client commands available. If FALSE
        #                       (or 'undef'), the session tries to connect to the world
        #   $tempFlag       - If set to TRUE, the world profile is a temporary world profile,
        #                       created because the user didn't specify a world name. File saving
        #                       in the session will be disabled. Otherwise set to FALSE (or
        #                       'undef')
        #
        # Return values
        #   'undef' on improper arguments
        #   Blessed reference to the newly-created object on success

        my (
            $class, $number, $world, $host, $port, $char, $pass, $account, $protocol, $loginMode,
            $offlineFlag, $tempFlag, $check,
        ) = @_;

        # Check for improper arguments
        if (! defined $class || ! defined $number || ! defined $world || defined $check) {

            return $axmud::CLIENT->writeImproper($class . '->new', @_);
        }

        # Setup
        my $self = {
            _objName                    => 'session',
            _objClass                   => $class,
            _parentFile                 => undef,       # No parent file object
            _parentWorld                => undef,       # No parent file object
            _privFlag                   => TRUE,        # All IVs are private

            # Perl object components
            # ----------------------

            # A copy of $self, for methods inherited by many objects that need a ->session IV
            session                     => undef,       # Set below
            # A unique number for the session (matches a key in GA::Client->sessionHash)
            number                      => $number,

            # Blessed reference to this session's 'main' window, a GA::Win::Internal object whose
            #   ->winType is 'main'
            # If GA::CLIENT->shareMainWinFlag = TRUE, there is a single 'main' window is shared by
            #   all sessions. If GA::CLIENT->shareMainWinFlag = FALSE, each session has its own
            #   'main' window
            mainWin                     => undef,
            # In this session's 'main' window, there is a default pane object (GA::Table::Pane)
            #   which contains a default tab object (GA::Obj::Tab) which in turn contains a default
            #   textview object (GA::Obj::TextView) for this session
            # This session uses its default tab to display most text received from the world. In
            #   addition, when the default tab is closed manually by the user, the session
            #   terminates
            # The default tab object for this session (once set, it does not change)
            defaultTabObj               => undef,
            # The tab object currently being used to display text received from the world (for
            #   example, MXP may want multiple panes, and to switch between them frequently)
            currentTabObj               => undef,

            # Blessed reference to the Automapper window (GA::Win::Map) for this session (only one
            #   can be opened per session, set to 'undef' when it's not open)
            mapWin                      => undef,       # Set by $self->set_mapWin
            # Blessed reference to the automapper object (GA::Obj::Map) for this session (always one
            #   per session)
            mapObj                      => undef,
            # Blessed reference to the world model object (GA::Obj::WorldModel) for the current
            #   world profile (always one per session)
            worldModelObj               => undef,       # Saved in file [worldmodel]
            # Blessed reference to the GUI window (GA::OtherWin::Gui) for this session (only one can
            #   be opened per session, set to 'undef' when it's not open)
            guiWin                      => undef,       # Set by $self->set_guiWin
            # Blessed reference to any 'wiz' window (inherited from GA::Generic::WizWin) opened by
            #   this session (only one 'wiz' window can be opened per session, set to 'undef' when
            #   no 'wiz' window is open)
            wizWin                      => undef,       # Set by $self->set_wizWin

            # IVs for this session
            # --------------------

            # Flag set to TRUE once $self->start has finished...
            startCompleteFlag           => FALSE,
            # ...any text received from the world is stored here, until that moment
            initialTextBuffer           => '',
            # The GA::Obj::ConnectHistory object for this connection (if one was created, otherwise
            #   it remains 'undef')
            connectHistoryObj           => undef,
            # When the object exists, its IVs are updated once a second. The next time (matches
            #   $self->sessionTime) to update it
            historyCheckTIme            => undef,

            # The character set to use for this session. $self->spinIncomingLoop encodes text
            #   received from the world using this charset
            # If the world profile's ->worldCharSet is defined, use that; otherwise use
            #   GA::Client->charSet. If the latter isn't available for some reason, this IV remains
            #   set to 'null', and no encoding of text received from the world takes place
            sessionCharSet              => 'null',

            # Automatic logins
            # ----------------

            # Flag set to TRUE once the user has logged into the world (or until they use the client
            #  command ';login'). Set back to FALSE temporarily during an MXP crosslinking operation
            loginFlag                   => FALSE,
            # For login modes 'lp', 'tiny', 'world_cmd', 'telnet' and 'mission', this IV is set to
            #   the mode; otherwise set to 'none' ($self->spinMaintainLoop watches for prompts -
            #   text received from the world without a newline character - and responds to them)
            loginPromptsMode            => 'none',
            # For login modes 'lp', 'tiny', 'world_cmd' and 'telnet', once a response to the
            #   prompt(s) has been sent, a list of login success patterns to watch out for (imported
            #   from GA::Profile::World->loginSuccessPatternList). When the list is empty, we're not
            #   looking out for these patterns
            loginSuccessPatternList     => [],
            # For login mode 'telnet', the regexes used on prompts. As each regex matches a prompt,
            #   it is removed from the list. After sending the username / password, the list is
            #   empty
            loginPromptPatternList      => [
                # Used in a m/.../i expression
                # Use regexes similar to those used by GA::Net::Telnet, but abbreviate 'username'
                #   to 'name', allow it to end with a question mark as well as a colon, and also
                #   allow it to end with 'wish', so that 'telnet' mode logins works on the default
                #   Dead Souls mudlib
                '(login|name|wish)[:? ]*$',
                'password[: ]*$',
            ],
            # For login mode 'tiny' and 'world_cmd', a list of patterns matching lines which confirm
            #   the world is ready to receive the login (set to
            #   GA::Profile::World->loginConnectPatternList)
            loginConnectPatternList     => [],
            # For login mode 'tiny', when ->processLineSegment spots a line matching one of the
            #   patterns in $self->loginConnectPatternList (contains the pattern 'connect' by
            #   default), sets this flag (so that $self->processLineSegment can call ->doLogin)
            # For login mode 'world_cmd', the same applies if ->loginConnectPatternList contains one
            #   or more patterns. If ->loginConnectPatternList is empty, we wait for the first
            #   prompt, instead
            loginConnectFoundFlag       => FALSE,
            # For login mode 'mission', a local copy of GA::Profile::World->loginSpecialList. If
            #   set, the session checks every incoming line until it finds one matching both a
            #   character name (insensitively) and one of these patterns; when it finds one, it
            #   sends the corresponding world commands, and then stops checking received lines, by
            #   emptying this local copy of the list (and does not call ->doLogin - that's the
            #   responsibility of the login mission)
            loginSpecialList            => [],
            # For all login modes (even 'none'), the time (matches $self->sessionTime) at which to
            #   display a reminder to use the ';login' command (default is 30 seconds). Set to
            #   'undef' after the warning is displayed, after a disconnection or after the character
            #   is marked as logged in
            loginWarningTime            => undef,
            # When $self->doLogin completes an automatic login, the confirmation message is not
            #   displayed immediately (which might interrupt a block of text received from the
            #   world), but stored here temporarily, and displayed at the end of the call to
            #   $self->spinIncomingLoop
            loginConfirmText            => undef,

            # When $self->processLineSegment notices some text that looks like a prompt, it
            #   updates these IVs.
            # The number of prompts received during this session and processed by
            #   $self->processPrompt (the total won't include prompts that were processed by
            #   ->dispatchCmd, if the user types a world command before ->processPrompt can act)
            promptCount                 => 0,
            # The actual line of text received
            promptLine                  => undef,
            # The line of text, stripped of escape sequences
            promptStripLine             => undef,
            # When something that looks like a prompt is received, we wait a short time
            #   (GA::Client->promptWaitTime seconds); if nothing else is received in the meantime,
            #   treat it as a prompt. This value matches $self->sessionTime; if the time passes and
            #   no more text is received, it's a prompt
            promptCheckTime             => undef,
            # Flag set to TRUE when ->processIncomingData deals with a packet of text that ends in a
            #   recognised command prompt (matches one of the patterns in
            #   GA::Profile::World->cmdPromptPatternList). Set back to FALSE when the function
            #   receives the next packet of text
            cmdPromptFlag               => FALSE,

            # Flag set to TRUE when ->processIncomingData processes a newline token in a packet of
            #   text. Set back to FALSE when the function finds any other token in a packet of text.
            #   (Used with MSP, to make sure MSP sound triggers are at the start of a line. The
            #   initial value is TRUE because, at the start of a session, the first text received
            #   follows an imaginary newline character)
            nlTokenFlag                 => TRUE,
            # Flag set to TRUE when $self->dispatchPassword sends a password to the world which has
            #   turned oFf ECHO (i.e. $self->echoMode is 'client_agree'). The world is supposed to
            #   supply its own newline character after the password, but many worlds don't, meaning
            #   that the next line is joined to the previous one
            # When TRUE, ->processIncomingData inserts an extra newline character at the start of
            #   the next packet of received text (but only if it doesn't already begin with a
            #   newline character), and then resets the flag back to FALSE
            nlEchoFlag                  => FALSE,
            # IV used to cope with <CR><LF>, or <LF><CR>, or even (occasionally, e.g. at TorilMUD)
            #   <CR><ANSI esc sequence><LR>
            # <CR> and <LF> are processed as separate tokens, setting this IV to 'cr' or 'lf'. Any
            #   text token sets the IV to an empty string. Other non-printing tokens (such as ANSI
            #   escape sequences) don't affect the setting of the IV
            # If this IV is set to 'cr' and a line feed token (\n) is processed, or if this IV is
            #   set to 'lf' and a carriage return (\r) is encountered, the IV is set back to an
            #   empty string, regarding the two tokens (whether sequential or not) as a single
            #   newline token
            crlfMode                    => '',

            # File objects
            # ------------

            # Most file objects are stored in GA::Client's registry
            # This GA::Session's registry contains file objects of the type 'worldprof',
            #   'otherprof' and 'worldmodel'. Only those files relating to the current world
            #   profile have their file objects stored here
            # The 'worldprof' file object will appear both here, and in the GA::Client registry
            #   File type    Standard directory                               Unique name    Stored
            #   'worldprof'  <SCRIPT_DIR>/data/worlds/<WORLD>/worldprof.axm   WORLD          Both
            #   'otherprof'  <SCRIPT_DIR>/data/worlds/<WOPLD>/otherprof.axm   otherprof      Session
            #   'worldmodel' <SCRIPT_DIR>/data/worlds/<WORLD>/worldmodel.axm  worldmodel     Session
            #
            # Registry hash of file objects, in the form
            #   $sessionFileObjHash{unique_name} = blessed_reference_to_file_object
            sessionFileObjHash          => {},
            # Flag set to TRUE by ';qquit', ';xxit' (etc) - stops $self->reactDisconnect from
            #   saving any files (including the 'config' file) as the connection closes. Otherwise
            #   set to FALSE
            disconnectNoSaveFlag        => FALSE,
            #
            # When auto-saves are turned on (i.e. GA::Client->autoSaveFlag is TRUE), the time
            #   (matches $self->sessionTime) at which the next auto-save should take place. When
            #   auto-saves are turned off, set to 0
            autoSaveCheckTime           => 0,
            # The last time (matches $self->sessionTime) at which an auto-save took place. Set
            #   to 0 if autosaves are turned off, or if no autosaves have been performed yet
            autoSaveLastTime            => 0,

            # Profiles
            # --------

            # Customisable registry list of profile priorities (which can include the standard
            #   categories of profile - 'char', 'race', 'guild' and 'world' - as well as any custom
            #   categories created by the user)
            profPriorityList            => [],      # [otherprof] Set below
            # (Profiles are explained in the comments for GA::Client->new. Each profile has a unique
            #   name, max 16 characters)
            # Registry hash of all profile templates (one entry for each new kind of custom
            #   profile). Hash in the form
            #   $templateHash{unique_string_name} = blessed_reference_to_profile_template_object
            templateHash                => {},      # [otherprof]
            #
            # Registry hash of profile objects. Includes the profile for the current world, as well
            #   as all profiles associated with that world. (The current world profile is also
            #   stored in GA::Client->worldProfHash). Hash in the form
            #   $profHash{unique_string_name} = blessed_reference_to_profile_object
            profHash                    => {},      # [otherprof]
            # Registry hash of current profile objects - a subset of $self->profHash containing only
            #   those profiles which are 'current' - no more than one for each category of profile.
            #   Categories include 'world', 'guild', 'race', 'char' and any other kind of category
            #   created by the user. Hash in the form
            #   $currentProfHash{category} = blessed_reference_to_profile_object
            currentProfHash             => {},
            # Four shortcuts to the current profile objects for the four 'standard' categories. Set
            #   to 'undef' when there isn't a current profile for that category
            currentWorld                => undef,
            currentGuild                => undef,
            currentRace                 => undef,
            currentChar                 => undef,
            # When one of the current profiles for this session changes, this flag is set to TRUE
            currentProfChangeFlag       => FALSE,
            #
            # The calling function supplies the world name, host IP address, port, character name,
            #   password and flags. They are stored here until required
            initWorld                   => $world,
            initHost                    => $host,
            initPort                    => $port,
            initChar                    => $char,
            initPass                    => $pass,
            initAccount                 => $account,
            initProtocol                => $protocol,
            initLoginMode               => $loginMode,
            initOfflineFlag             => $offlineFlag,
            initTempFlag                => $tempFlag,

            # Cages
            # -----

            # Cages are discussed in the comments for GA::Client->new. Each cage has a unique name
            #   in the form type_profileCategory_profileName, e.g 'alias_world_deathmud', so the
            #   built-in maximum length is 42 characters (8_16_16 - type has maximum length of 8
            #   characters, profiles, 16 characters)
            # Registry hash of all cages associated with this world and its profiles. Hash in the
            #   form
            #   $cageHash{unique_name} = blessed_reference_to_cage_object
            cageHash                    => {},      # [otherprof]
            # Registry hash of all cages associated with current profiles - a subset of
            #   $self->cageHash
            currentCageHash             => {},
            # Registry connecting each cage in ->cageHash to its inferior cage. Hash in the form
            #   $inferiorCageHash{cage_name} = blessed_reference_to_inferior_cage
            #   $inferiorCageHash{cage_name} = undef (if there is no inferior cage)
            inferiorCageHash            => {},

            # Dictionaries
            # ------------

            # Blessed reference of the current dictionary
            currentDict                 => undef,

            # Interfaces
            # ----------

            # Interfaces are triggers, aliases, macros, timers and hooks
            # 'Active' interfaces are those that are in operation right now (although, if their
            #   ->enabledFlag is FALSE, they won't do anything). Active interfaces are
            #   GA::Interface::Active objects that inherit most of their values from a parent
            #   GA::Interface::Trigger (etc) object
            # (Interface names max 32 chars)
            #
            # Registry hash of active interface objects, in the form
            #   $interfaceHash{unique_name} = blessed_reference_to_active_interface_object
            interfaceHash               => {},
            # Parallel registry hash of active interface objects, containing the same list of
            #   objects, in the form
            #       $interfaceNumHash{number} = blessed_reference_to_active_interface_object
            interfaceNumHash            => {},
            # How many interfaces have been created during this session
            interfaceCount              => 0,
            # Whenever a dependent interface fires, some part of the Axmud code is called; that
            #   code may want to delete the interface. However, it's not a good idea to start
            #   tampering with the interface registries while some other part of the code is
            #   halfway through checking them
            # This list contains a list of interfaces that have been marked for deletion by
            #   $self->deleteInterface. The list is checked whenever the task loop spins, and the
            #   interfaces are deleted.
            deleteInterfaceList         => [],

            # Registry hash of active trigger interface objects, in the form
            #   $triggerHash{number} = undef
            triggerHash                 => {},
            # Registry list of active trigger interface numbers. Contains all the keys in
            #   ->triggerHash, but in order of creation, e.g. (2, 3, 5, 10, 19)
            # This makes sure that triggers fire in a predictable order
            triggerOrderList            => [],
            # Registry hash of active alias interface objects, in the form
            #   $aliasHash{number} = undef
            aliasHash                   => {},
            # Registry list of active alias interface numbers. Contains all the keys in
            #   ->aliasHash, but in order of creation, e.g. (2, 3, 5, 10, 19)
            # This makes sure that aliases fire in a predictable order
            aliasOrderList              => [],
            # Registry hash of active macro interface objects, in the form
            #   $macroHash{number} = keycode
            # ...where keycode is an Axmud standard keycode (or keycode string)
            macroHash                   => {},
            # Registry list of active macro interface numbers. Contains all the keys in
            #   ->macroHash, but in order of creation, e.g. (2, 3, 5, 10, 19)
            # This makes sure that macros fire in a predictable order
            macroOrderList              => [],
            # Registry hash of active timer interface objects, in the form
            #   $timerHash{number} = next_fire_time
            # ...where next_fire_time matches $self->sessionTime
            timerHash                   => {},
            # Registry list of active timer interface numbers. Contains all the keys in
            #   ->timerHash, but in order of creation, e.g. (2, 3, 5, 10, 19)
            # This makes sure that timers fire in a predictable order
            timerOrderList              => [],
            # Registry hash of active hook interface objects, in the form
            #   $hookHash{number} = hook_event
            hookHash                    => {},
            # Registry list of active hook interface numbers. Contains all the keys in
            #   ->hookHash, but in order of creation, e.g. (2, 3, 5, 10, 19)
            # This makes sure that hooks fire in a predictable order
            hookOrderList               => [],

            # Interface responses that are instructions (world commands, forced world commands
            #   beginning with ',,', client commands beginning ';', echo commands beginning '"',
            #   Perl commands beginning '/', script commands beginning '&' and multi commands
            #   beginning ':') may need to access information about the event that caused the
            #   interface to fire
            # When an interface fires, the information is stored in this hash. When the response
            #   has been completed, the information is removed
            # The keys of the hash are a variable available to the programmes executed by Perl
            #   commands (those starting with '/'); e.g. the value of the key '_hookVar' is
            #   available to those programes as the variable $_hookVar
            perlCmdDataHash             => {
                # All interfaces
                '_interface'            => undef,
                # Trigger interfaces
                '_line'                 => undef,
                '_stripLine'            => undef,
                '_modLine'              => undef,
                # Alias interfaces
                '_originalCmd'          => undef,
                # Macro response
                '_keycode'              => undef,
                # Timer response
                '_timerExpect'          => undef,
                '_timerTime'            => undef,
                # Hook response
                '_hookEvent'            => undef,
                '_hookVar'              => undef,
                '_hookVal'              => undef,
            },
            # Parallel hash used by trigger interfaces which stores escape sequences (converted into
            #   Axmud colour/style tags) and their positions in a line of received text (see the
            #   comments in $self->checkAliases)
            perlCmdTagHash              => {},
            # Parallel list used by trigger interfaces which stores the portions of a line of
            #   received text which match the backreferences of the regex used as the trigger's
            #   ->stimulus (see the comments in $self->checkAliases)
            perlCmdBackRefList          => [],

            # Tasks
            # -----

            # Tasks are mini-scripts which interact with the world, and with the data Axmud stores
            #   about the world. Axmud comes with several built-in tasks - some of which are more
            #   useful than others - and provides the possibility of the user writing new ones. (It
            #   is usually quicker to write scripts in Axmud's own scripting language, Axbasic;
            #   writing your own tasks allows you to customise Axmud's behaviour, but it's not
            #   possible to modify them which Axmud is running.)
            #
            # Actually, there are two categories of task - 'process' tasks and 'activity' tasks
            # 'Process' tasks are usually called once per task loop. They are good for performing
            #   actions in small chunks. called 'stages'. A process task might have the following
            #   chunks: (1) Go e;e;e;s;e to the bank (2) Wait for the character to arrive (3)
            #   Withdraw money (4) Check the inventory, to see if your character has enough (5) Go
            #   w;n;w;w;w to the town centre (6) Wait for the arrival (7) End the task
            # 'Activity' tasks are event-driven: after initial setup, nothing happens until they
            #   are prompted, usually as the result of a trigger, alias, macro, timer or hook
            # All tasks - both activities and processes - should inherit from GA::Generic::Task
            # Axmud maintains several tasklists. The most important one is the 'current tasklist',
            #   containing all the tasks that are running now.
            # There is also a 'global initial tasklist'. containing tasks which start as soon as the
            #   user connects to any world. Each profile also has its own initial tasklist,
            #   containing tasks which start when the user connects to a particular world, or with a
            #   particular character, and so on
            # Finally, there is also a 'custom tasklist', which wait in the background until the
            #   user starts them. These tasks are usually customised by the user to perform a
            #   particular function.
            #
            # Some tasks are 'jealous' - only one copy of them can be run at any time.
            # In any case, when a new task is created, it's given a unique name (the task object's
            #   name followed by a number)
            #
            # Task name IVs have size limits. If they are breached, the task can't initialise:
            #   ->name          16  e.g. 'status_task'
            #   ->prettyName    32  e.g. 'Status Task'
            #   ->uniqueName    24  e.g. 'status_task_15'
            #   ->stage         5   e.g. '1'
            #
            # Registry hash of tasks - the 'current tasklist'
            #   ->currentTaskHash{unique_task_name} = blessed_reference_to_task_object
            #       e.g. ->currentTaskHash{'status_task_57'}
            currentTaskHash             => {},
            # The same hash, but with multiple copies of the same task eliminated. Each key is
            #   associated with the most recently created task of that type (e.g. if there are two
            #   TaskList tasks running, only the second is included)
            #   ->currentTaskNameHash{task_name} = blessed_reference_to_task_object
            #       e.g. ->currentTaskNameHash{'status_task'}
            currentTaskNameHash         => {},
            # Any tasks paused with the ';pausetask' command have their blessed references stored in
            #   this array; then, the ';resumetask' command can un-pause all of these tasks, without
            #   affecting tasks that were paused for some other reason
            pausedTaskList              => [],
            #
            # Most of Axmud's built-in tasks are 'jealous' (only one copy can be run at a time). The
            #   Script task is an exception, and can have any number of instances running
            #   simultaneously
            # Convenient shortcuts for the built-in 'jealous' tasks (set to 'undef' when the task
            #   isn't running)
            advanceTask                 => undef,
            attackTask                  => undef,
            chatTask                    => undef,   # Only stored the 'lead' Chat task
            compassTask                 => undef,
            conditionTask               => undef,
            debuggerTask                => undef,
            divertTask                  => undef,
            inventoryTask               => undef,
            launchTask                  => undef,
            locatorTask                 => undef,
            notepadTask                 => undef,
            rawTextTask                 => undef,
            rawTokenTask                => undef,
            statusTask                  => undef,
            taskListTask                => undef,
            watchTask                   => undef,

            # Axbasic scripts
            # ---------------

            # A list of Language::Axbasic::Script objects which are paused, and ready to resume
            #   (which happens at the beginning of every task loop). The list can contain multiple
            #   copies of the same LA::Script object, since the task loop checks that the script is
            #   paused, before unpausing it
            scriptResumeList            => [],

            # Recordings, missions, routes and quests
            # ---------------------------------------

            # Recordings - the user can record all commands sent to the world for a period; these
            #   recordings can then be saved as missions or routes. Only one recording at a time is
            #   possible (per session)
            # Flag is set to TRUE if a recording is currently in progress, FALSE if not
            recordingFlag               => FALSE,
            # Flag is set to TRUE if a recording is currently in progress, but is paused (FALSE if
            #   not)
            recordingPausedFlag         => FALSE,
            # The current recording, a list of strings starting with an initial character which
            #   identifies the type of command, optionally followed by whitespace, followed by the
            #   command itself (empty when no recording in progress)
            # Valid initial characters are:
            #   '>' identifies the string as a world command
            #   ';' identifies the string as a client command
            #   '#' identifies the string as a comment to be displayed in the 'main' window
            #   '@' identifies the string as a break (anything following the '@' is ignored)
            # There are three special kinds of break (see the help for ';startmission'):
            #   't' identifies a trigger break; it is followed by a string used as the stimulus
            #       in a trigger. When the trigger fires, the mission continues
            #   'p' identifies a pause break; it is followed by a number, the time (in seconds) to
            #       wait
            #   'l' identifies a Locator break; it waits for the Locator task to decide it's not
            #       expecting any more room descriptions (anything following the 'l' is ignored)
            recordingList               => [],
            # The user can specify that, while making a recording, new commands can be inserted at
            #   an arbitrary position (rather than added to the end, as happens by default).
            # Set to the line number at which the next command should be inserted, or 'undef' if
            #   new commands should just be added to the end of ->recordingList
            recordingPosn               => undef,

            # Missions are stored in the world profile. Only one mission can be carried out at a
            #   time. When the user starts a mission with ';startmission', the GA::Obj::Mission
            #   stored in the world profile is cloned, and the cloned object is stored here
            currentMission              => undef,

            # Logging
            # -------

            # Axmud can log to several different logfiles, all of which are in the /logs
            #   directory and its sub-directories (in the Axmud data directory, $DATA_DIR)
            # The 'sleep', 'passout' and 'dead' logfiles are written by the Status task, which
            #   attempts to write lines of text received by the world before and after the event
            # After one of these events, the line until which these logfiles should be written, if
            #   allowed (matches $self->displayBufferCount). Set to 'undef' when those logfiles
            #   aren't being written because the event hasn't happened recently)
            logAsleepUntilLine          => undef,
            logPassedOutUntilLine       => undef,
            logDeadUntilLine            => undef,

            # System loops
            # ------------

            # The session time (used by various parts of the code that need a time that stays
            #   consistent for some measured period, but which is updated frequently). Set by
            #   $self->spinTimerLoop, whenever the session loop spins
            # (See also the corresponding client IV, GA::Client->clientTime)
            sessionTime                 => 0,
            # For the benefit of GA::Client->spinClientLoop, an IV set to $self->sessionTime rounded
            #   down to the nearest second, every time $self->getTimeLabelText() is called
            # (The connection info strip object, GA::Strip::ConnectInfo, displays three time counts,
            #   and it's very distracting if they don't update simultaneously)
            connectInfoCheckTime        => undef,
            # A string describing the date and time at which the connection was actually established
            #   (used by GA::Strip::ConnectInfo to show a tooltip). Initially set by
            #   $self->connectionComplete. Set to 'undef' whenever $self->status is not 'connected'
            connectedTimeString         => undef,

            # Session loop
            # ------------

            # The GA::Obj::Loop which handles the session loop
            # NB The maintenance, timer, incoming data, task and replay loops do not have their
            #   own GA::Obj::Loop object. Instead, there is a single loop object for the session
            #   loop
            # The session loop, every time it spins, checks whether it's time to call the
            #   subservient loops' ->spin functions, based on the IVs below
            sessionLoopObj              => undef,
            # Flag set to TRUE (by GA::Obj::Loop->spinLoop) when the session loop spins, set back to
            #   FALSE when the spin is complete. The TRUE setting prevents one session loop spin
            #   from taking place if another is still being processed
            sessionLoopSpinFlag         => FALSE,
            # Flag set to TRUE (by $self->spinTimerLoop, etc) when any of the session loop's
            #   subservient loops spin, set back to FALSE when the spin is complete. The TRUE
            #   setting prevents one type of subservient loop from spinning if another type of loop
            #   spin is already being processed
            childLoopSpinFlag           => FALSE,
            # The session loop default delay, in seconds (never changes once set; absolute minimum
            #   value is 0.01)
            sessionLoopDelay            => 0.05,

            # Maintenance loop
            # ----------------

            # The maintenance loop delay, in seconds (can be changed while the session is running,
            #   but should not be set lower than $self->sessionLoopDelay)
            maintainLoopDelay           => 0.1,
            # The time at which the maintenance loop should next spin (i.e. the time at which
            #   $self->spinSessionLoop should call $self->spinMaintainLoop)
            # When set to 'undef', the loop is not running at all; the first time it spins, set to 0
            maintainLoopCheckTime       => undef,

            # If the world profile defines slowwalking settings (a maximum number of world commands
            #   to send per time period), excess world commands are stored here until it's time to
            #   send them to the world
            # This IV is updated after the time interval specified by
            #   GA::Profile::World->excessCmdDelay, and any excess stored commands can be sent. Set
            #   by $self->spinMaintainLoop to the current value of $self->sessionTime
            lastExcessCmdTime           => 0,
            # How many world commands have been sent during that second
            excessCmdCount              => 0,
            # A list of stored commands that can't be sent yet; a list in the form
            #   (command, time, command, time)
            # ...where time is the $self->sessionTime after which it's safe to send the command
            excessCmdList               => [],

            # Crawl mode is a temporary slowwalking mode. Once enabled, the maximum number of
            #   world commands to send per time period depends on the IVs just below, not on the
            #   world profile IVs (as they usually do)
            # Once enabled, crawl mode waits up to two minutes for an excess command to be stored.
            #   If none are stored within that time, crawl mode disables itself. If at least one
            #   excesss command is stored, crawl mode will apply until the excess command list is
            #   emptied; at that point it disables itself (immediately)
            # Flag set to TRUE when crawl mode is enabled, FALSE when disabled
            crawlModeFlag               => FALSE,
            # When crawl mode is enabled, the maximum number of commands to send per second (not
            #   per time period, as with the world profile IVs)
            crawlModeCmdLimit           => undef,
            # How long to wait for the first excess command to be stored, before crawl mode disables
            #   itself (in seconds)
            crawlModeWaitTime           => 120,
            # The time at which crawl mode, if no excess commands have been stored, disables itself
            #   (set back to 'undef' as soon as the first excess command is stored; matches
            #   $self->sessionTime)
            crawlModeCheckTime          => undef,

            # v1.0.287 - it's apparently possible for $self->worldCmd to be called a second time,
            #   while the function is still processing commands from an earlier call. As a result,
            #   world commands can be processed in the wrong order.
            # To resolve this issue, the second call to $self->worldCmd temporarily stores the
            #   world commands in this list, and the function returns immediately. When the first
            #   call to ->worldCmd has completed, if there are any world commands in this emergency
            #   list, then they are processed
            emergencyCmdList            => [],
            # This flag is set to TRUE at the start of a call to ->worldCmd, and FALSE at the end
            #   of it
            worldCmdProcessFlag         => FALSE,

            # Delayed quit. If this IV is set, it is the moment in the future (matches
            #   $self->sessionTime) at which some kind of 'quit' or 'exit' client command must be
            #   performed
            delayedQuitTime             => undef,
            # When it's time to perform the delayed quit, the actual client command to use - will
            #   be one of 'quit', 'qquit', 'exit', 'xxit' (the ';' sigil is not required)
            # NB The client commands ';quitall' / ';exitall' set these IVs in every session
            delayedQuitCmd              => undef,
            # The disconnection time (a real clock time), set upon disconnection, and used to
            #   update the 'main' window's connection info host label. Set to 'undef' until
            #   $self->status changes from 'connected' to 'disconnected'
            disconnectTime              => undef,

            # Timer loop
            # ----------

            # The timer loop delay, in seconds (can be changed while the session is running, but
            #   should not be set lower than $self->sessionLoopDelay)
            timerLoopDelay              => 0.1,
            # The time at which the timer loop should next spin (i.e. the time at which
            #   $self->spinSessionLoop should call $self->spinTimerLoop)
            # When set to 'undef', the loop is not running at all; the first time it spins, set to 0
            timerLoopCheckTime          => undef,

            # Incoming data loop
            # ------------------

            # The incoming data loop delay, in seconds (can be changed while the session is running,
            #   but should not be set lower than $self->sessionLoopDelay)
            incomingLoopDelay           => 0.1,
            # The time at which the incoming data loop should next spin (i.e. the time at which
            #   $self->spinSessionLoop should call $self->spinIncomingLoop)
            # When set to 'undef', the loop is not running at all; the first time it spins, set to 0
            incomingLoopCheckTime       => undef,

            # An emergency buffer used for invalid escape sequences which are probably the result of
            #   a valid escape sequence split over two packets, the second of which hasn't been
            #   received yet. Set and reset by $self->processIncomingData
            emergencyBuffer             => undef,

            # The world's host address and port (the ones actually used, not the ones supplied by
            #   the calling function - just in case they are different)
            host                        => undef,
            port                        => undef,
            # Which connection protocol this session is using: 'telnet', 'ssh' or 'ssl' ('undef'
            #   when not connected)
            protocol                    => undef,
            # The GA::Net::Telnet handling the connection ('undef' when not connected)
            connectObj                  => undef,
            # For SSH connections, the Net::OpenSSH and Perl pty (an IO::Tty filehandle) objects
            #   ('undef' for telnet/SSL connections and when not connected)
            sshObj                      => undef,
            ptyObj                      => undef,
            # For SSL connections, the IO::Socket::SSL object ('undef' for telnet/SSH connections
            #   and when not connected)
            sslObj                      => undef,
            # The number of packets received (i.e. the number of times $self->processIncomingData
            #   has been called) during this session
            packetCount                 => 0,
            # The current connection status:
            #   'waiting'       - The first connection hasn't been attempted yet
            #   'connecting'    - Attempting to connect
            #   'connected'     - Connected to the remote host
            #   'offline'       - Session opened in 'connect offline' mode
            #   'disconnected'  - Disconnected from the remote host (or connection failed, or an
            #                       'offline' mode session has finished, or an MXP crosslinking
            #                       operation is in progress)
            status                      => 'waiting',
            # On disconnection, $self->reactDisconnect is called from several places in the session
            #   code. In rare circumstances (such as the GA::Net::Telnet object returning TRUE to
            #   an ->eof() call), it might be called more than once
            # On the first call, this flag is set to TRUE. On any subsequent calls, nothing happens
            #   if this flag is TRUE
            reactDisconnectFlag         => FALSE,

            # When $self->processIncomingData receives text from the world, it converts it into
            #   tokens
            # Text tokens are stored in this IV, each new token being added to the last one (until
            #   they are collectively processed by $self->processLinePortion)
            recvLineText                => '',
            # The current length of $self->recvLineText
            recvLineLength              => 0,
            # Non-text tokens are removed from the received line. Some of them are converted into
            #   Axmud colour/style tags, and those tags are stored here. Hash in the form
            #       $recvLineHash{offset} = reference_to_list_of_Axmud_colour_and_style_tags
            #   ...where 'offset' is a position in $self->recvLineText at which this list of colour/
            #       style tags applies
            # NB To keep the code simple, the hash always contains an entry corresponding to the
            #   start of the string in $self->recvLineText
            recvLineHash                => {
                0                       => [],
            },
            # If a whole line is received (i.e. a line that ends in a newline character), the
            #   contents of the previous IVs are copied into these IVs, before
            #   $self->processLinePortion is called
            # If a partial line is received (i.e. a line that doesn't end in a newline character),
            #   they are again copied. The next time ->processIncomingData is called, the NEW
            #   contents of the above IVs are added to the existing contents of the IVs below,
            #   so that both the whole line, as well as the portion of the line being processed, is
            #   available to the code
            recvUsedText                => '',
            recvUsedLength              => 0,
            recvUsedHash                => {
                0                       => [],
            },
            # An IV similar to ->recvLineText, in that text tokens are stored in this IV, each new
            #   token being added to the last one. However, this IV stores the whole of a single
            #   line, and is reset to an empty string every time a newline is processed
            # (Used by Pueblo to reduce multiple whitespace characters to a single one)
            recvWholeLineText           => '',
            # When text is added ->recvLineText, it's also added to this IV. When an image is
            #   processed, some text is added to this IV (but nothing is added to ->recvLineText).
            #   This IV is then used to write the 'receive' logfile
            recvImgLineText             => '',

            # Task loop
            # ---------

            # The task loop delay, in seconds (can be changed while the session is running, but
            #   should not be set lower than $self->sessionLoopDelay)
            taskLoopDelay               => 0.1,
            # The time at which the task loop should next spin (i.e. the time at which
            #   $self->spinSessionLoop should call $self->spinTaskLoop)
            # When set to 'undef', the loop is not running at all; the first time it spins, set to 0
            taskLoopCheckTime           => undef,

            # Flag set to TRUE if the task loop should be frozen - nothing happens when
            #   $self->taskLoop is called
            # NB A frozen task loop doesn't prevent other code from calling a task's method
            freezeTaskLoopFlag          => FALSE,
            # Set to TRUE when the first task loop completes (required by $self->startTask)
            firstTaskLoopCompleteFlag   => FALSE,
            # The time (matches $self->sessionTime) at which each task should next be called.
            #   ->taskCallHash{task_unique_name} = next_call_time
            # Set to '0' if the task should be called on the next task loop, if it should be called
            #   every task loop, or if the task is an activity task
            taskCallHash                => {},

            # Replay loop
            # -----------

            # The replay loop delay, in seconds (can be changed while the session is running, but
            #   should not be set lower than $self->sessionLoopDelay)
            replayLoopDelay             => 0.05,     # Same as $self->sessionLoopDelay, please
            # The time at which the replay loop should next spin (i.e. the time at which
            #   $self->spinSessionLoop should call $self->spinReplayLoop)
            # When set to 'undef', the loop is not running at all; the first time it spins, set to 0
            replayLoopCheckTime         => undef,

            # The time at which the replay loop first spun (system time, in seconds)
            replayLoopStartTime         => undef,
            # The time at which the replay loop last spun (seconds after ->replayLoopStartTime)
            replayLoopTime              => undef,
            # The time at which the replay loop is due to halt (matches ->sessionTime)
            replayLoopStopTime          => undef,

            # Flag set to TRUE if the text buffer should be replayed, FALSE (or 'undef') if not
            replayLoopTextFlag          => undef,
            # Flag set to TRUE if the command buffer should be replayed, FALSE (or 'undef') if not
            replayLoopCmdFlag           => undef,
            # The line number of the next buffer line that can be 'replayed'
            replayLoopNextText          => undef,
            replayLoopNextCmd           => undef,

            # The replay text buffer hash (modelled on the display buffer hash)
            replayDisplayBufferHash     => {},
            replayDisplayBufferCount    => 0,
            replayDisplayBufferFirst    => undef,
            replayDisplayBufferLast     => undef,

            # The replay command buffer hash (modelled on the command buffer hash)
            replayCmdBufferHash         => {},
            replayCmdBufferCount        => 0,
            replayCmdBufferFirst        => undef,
            replayCmdBufferLast         => undef,

            # Buffers
            # -------

            # The display buffer for this session. Every line of text received from the world and
            #   display in the default tab object is assigned its own GA::Buffer:Display object in
            #   this session's registry (there is no equivalent registry in GA::Client)
            # Text which is redirected to another tab (for example, because MXP has created internal
            #   frames) is not stored in this hash
            # Text which is inserted in the Gtk2::TextBuffer at an insertion point other than the
            #   one at the end of the buffer is added to the end of this hash, on a new line.
            #   Earlier lines in the hash are not modified)
            # When the buffer is full, adding a line to the buffer also deletes the oldest existing
            #   one
            # The hash might not be the same size as the default tab's Gtk2::TextBuffer (although
            #   they are, by default); so it could happen that text visible in the textview is no
            #   longer stored in this hash, or that text still stored in this hash is no longer
            #   visible in the textview
            # Hash in the form
            #   $displayBufferHash{number} = blessed_reference_to_buffer_object
            displayBufferHash           => {},
            # How many lines have been added to the buffer altogether (used to give every
            #   GA::Buffer::Display object a unique number)
            displayBufferCount          => 0,
            # The ->number of the earliest surviving GA::Buffer::Display object in the buffer,
            #   deleted when the buffer is full and a new object is added
            displayBufferFirst          => undef,
            # The ->number of the most recently-created GA::Buffer::Display object in the buffer
            #   (when set, always one less than $self->displayBufferCount)
            displayBufferLast           => undef,

            # The instruction buffer for this session. Every instruction processed by this session
            #   is assigned its own GA::Buffer::Instruct object in this session's registry, as well
            #   as a separate GA::Buffer:Instruct object in the Axmud client's registry
            # Instructions includes client commands like ';setworld deathmud', Perl commands, echo
            #   commands as well as world commands
            # In addition, if the user types 'north;kill troll;eat corpse', that chain of world
            #   commands is stored as a single GA::Buffer::Instruct (and also as three separate
            #   GA::Buffer::Cmd objects in the world command buffer)
            # When the buffer is full, adding an instruction to the buffer also deletes the oldest
            #   existing one
            # Hash in the form
            #   $instructBufferHash{number} = blessed_reference_to_buffer_object
            instructBufferHash          => {},
            # How many instructions have been processed altogether (used to give every
            #   GA::Buffer::Instruct object a unique number)
            instructBufferCount         => 0,
            # The ->number of the earliest surviving GA::Buffer::Instruct object in the buffer,
            #   deleted when the buffer is full and a new object is added
            instructBufferFirst         => undef,
            # The ->number of the most recently-created GA::Buffer::Instruct object in the buffer
            #   (when set, always one less than $self->instructBufferCount)
            instructBufferLast          => undef,
            # When the user navigates through instructions in an 'internal' window's command entry
            #   box by pressing the up/down arrow keys, the number of the buffer object whose
            #   instruction is currently used (matches a key in ->instructBufferHash)
            # Set on the first such key press when this is the current session. Reset back to
            #   'undef' when another session becomes the current session, or an instruction is
            #   processed (even if it's a $self->pseudoCmd call)
            instructBufferPosn          => undef,

            # The world command buffer for this session. Every world command processed by this
            #   session is assigned its own GA::Buffer::Cmd object in this session's registry, as
            #   well as a separate GA::Buffer:Cmd object in the Axmud client's registry
            # If the user types 'north;kill troll;eat corpse', three world commands are processed
            #   and three GA::Buffer:Cmd objects are added
            # When the buffer is full, adding an instruction to the buffer also deletes the oldest
            #   existing one
            # Hash in the form
            #   $cmdBufferHash{number} = blessed_reference_to_buffer_object
            cmdBufferHash               => {},
            # How many world commands have been processed altogether (used to give every
            #   GA::Buffer::Cmd object a unique number)
            cmdBufferCount              => 0,
            # The ->number of the earliest surviving GA::Buffer::Cmd object in the buffer, deleted
            #   when the buffer is full and a new object is added
            cmdBufferFirst              => undef,
            # The ->number of the most recently-created GA::Buffer::Cmd object in the buffer (when
            #   set, always one less than $self->cmdBufferCount)
            cmdBufferLast               => undef,
            # When the user navigates through world commands in an 'internal' window's command entry
            #   box by pressing the up/down arrow keys, the number of the buffer object whose
            #   world command is currently used (matches a key in ->cmdBufferHash)
            # Set on the first such key press when this is the current session. Reset back to
            #   'undef' when another session becomes the current session, or an instruction is
            #   processed (even if it's a $self->pseudoCmd call)
            cmdBufferPosn               => undef,

            # When the Locator task is running and a new command is sent to the world,
            #   GA::Buffer::Cmd->interpretCmd is called to decide whethere it's a movement command
            #   or not (unless it's already been processed as a redirect mode command or an assisted
            #   move)
            # This IV is usually set to 'unknown', which tells ->interpretCmd that it can decide for
            #   itself whether the sent command is a movement command, or not
            # $self->moveCmd briefly sets the IV to 'is_move', meaning that it is definitely a
            #   movement command
            # $self->relayCmd briefly sets the IV to 'not_move', meaning that it is definitely NOT a
            #   movement command
            moveMode                    => 'unknown',
            # Flag set temporarily to TRUE by $self->checkAssistedMove, when it cancels a movement
            #   command in protected moves mode (when the world models' ->superProtectedMovesFlag
            #   is also TRUE).
            # $self->worldCmd and ->spinMaintainLoop both process world commands by calling
            #   $self->processWorldCmd. After the call is returned, they both check the value of
            #   this flag and, if it's TRUE, they both stop processing world commands;
            #   ->spinMaintainLoop deletes all excess commands that might be stored. Then they
            #   restore this flag's value to FALSE
            overruleMoveFlag            => FALSE,

            # Three IVs set so any code can work out how long the user has been idle (in this
            #   session). To work out the user's idle time, subtract these values from
            #   $self->sessionTime
            # The time at which text was last received from the world and displayed in the default
            #   textview object ('undef' if no text has been received yet)
            #  Text received and displayed in a different
            #   textview object does not count. Matches $self->sessionTime, set by
            #   $self->updateDisplayBuffer
            lastDisplayTime             => undef,
            # The time at which the last instruction was processed ('undef' if no instructions have
            #   been processed yet)
            # Includes client commands, world commands, Perl commands and echo commands. Matches
            #   $self->sessionTime, set by $self->updateInstructBuffer
            lastInstructTime            => undef,
            # The time at which the last world command was processed ('undef' if no world commands
            #   have been processed yet)
            # Matches $self->sessionTime, set by $self->updateCmdBuffer
            lastCmdTime                 => undef,
            # How many seconds should we wait before the 'user_idle' and 'world_idle' hook events
            #   take place (i.e. how many seconds after $self->lastCmdTime or
            #   $self->lastDisplayTime)
            # NB Setting the literal value to 0 would cause these hook events to never happen
            constHookIdleTime           => 60,
            # Flags set to TRUE when one of these hook events happens; set back to FALSE the next
            #   time $self->lastCmdTime or $self->lastDisplayTime are updated
            disableUserIdleFlag         => FALSE,
            disableWorldIdleFlag        => FALSE,

            # For text-to-speech, we try to process the whole packet of received text, before
            #   converting it to speech. Processed line segments are added to the existing text
            #   in this IV by $self->processLineSegment; the IV is then reset by
            #   $self->incomingDataLoop
            ttsBuffer                   => '',
            # What type of message was last converted to speech by a call from this session to
            #   GA::Client->tts: 'receive' for text received from the world, 'system', 'error' for
            #   system messages, 'command' or 'cmd' for a world command, 'dialogue' for a 'dialogue'
            #   window or 'task' for a task window message or 'other' for something else (used for
            #   multi-line system messages, so the TTS engine doesn't have to read out 'system
            #   message' at the beginning of every consecutive line)
            ttsLastType                 => undef,
            # Flag that can be set to TRUE by anything, that wants to temporarily disable TTS for
            #   system messages (but not system error messages). Used mainly by $self->start, so
            #   that all the copyright (etc) messages don't get read out (so that the first thing
            #   the user hears is 'Connecting to...' )
            ttsTempDisableFlag          => FALSE,

            # Tabs and textviews
            # ------------------

            # When there are no file objects in $self->sessionFileObjHash whose ->modifyFlag is set
            #   to TRUE (meaning that none of them need to be saved), this flag is set to FALSE
            # When the first file object has its ->modifyFlag set to TRUE, $self->setModifyFlag
            #   calls GA::Table::Pane->setTabLabel for each session using the same world profile as
            #   this one
            # That function displays an asterisk in the label for the session's default tab. In
            #   addition, this flag gets set to TRUE
            # The flag is set back to FALSE by $self->checkTabLabels when all file objects are saved
            showModFlag                 => FALSE,
            # Flag set to TRUE whenever text is received from the world while this session isn't the
            #   current session. Set back to FALSE when this session becomes the current session
            showNewTextFlag             => FALSE,
            # Which colour to use in the default tab's label
            #   'normal'        - Show the normal colour
            #   'active'        - Show red (meaning that it's not the current session, and that some
            #                       text has been received by the world, which hasn't yet been seen
            #                       by the user)

            #   'disconnected'  - Show blue (the session is disconnected)
            #   'offline'       - Show magenta (the session is running in 'connect offline' mode)
            showTabColourMode           => 'normal',
            # The xterm title sent by the world (even if it isn't displayed in the tab label);
            #   'undef' if no xterm title has been received
            xTermTitle                  => undef,
            # Flag set to TRUE when an xterm title is received (but only when
            #   GA::Client->xTermTitleFlag is TRUE), set back to FALSE by $self->checkTabLabels once
            #   the tab label has been updated
            showXTermTitleFlag          => FALSE,
            # Some MUDs might use the OSC colour palette to adjust the colour of the basic 16 ANSI
            #   colours. If an escape sequence 'ESC]Pxxxxxxx' is received (where x is a hexadecimal
            #   character in the range 0-9, A-F), this hash is updated which causes
            #   $self->processEscSequence to use the adjusted colour instead of the Axmud standard
            #   colour tag. Hash in the form
            #       $oscColourHash{standard_colour_tag} = rgb_colour_tag
            # If the MUD hasn't tried to adjust the basic 16 ANSI colours, then this hash will be
            #   empty
            oscColourHash               => {},

            # Blinkers
            # --------

            # 'Internal' windows (and especially 'main' windows) can include a strip object
            #   (GA::Strip::ConnectInfo) that displays blinkers - little blobs of colour which are
            #   lit up (briefly) when data is sent to and forth from the world
            # This version of Axmud implements the following blinker numbers:
            #   0   - blinker turned on when data is received from the world
            #   1   - blinker turned on when telnet option/protocol data (invisible to users) is
            #           received from the world
            #   2   - blinker turned on when a world command is sent
            # In each window, only the window object's ->visibleSession can light up blinkers; when
            #   the visible session changes, all blinkers are reset
            # Hash of blinkers and their current status, for when this session is the visible
            #   session. Hash in the form
            #       $blinkerStatusHash{blinker_number} = state
            # ...where 'state' represents the blinker's state: when the blinker is turned off, set
            #   to 'undef', when the blinker is turned on, set to a time (matches
            #   GA::Client->clientTime) when the blinker should be turned off again
            blinkerStateHash            => {
                    0                   => undef,
                    1                   => undef,
                    2                   => undef,
            },

            # Telnet options
            # --------------

            # ECHO mode:
            #   'no_invite' - The server has not suggested ECHO yet
            #   'client_agree' - The server has suggested that the client stop ECHOing, and the
            #       client has agreed
            #   'client_refuse' - The server has suggested that the client stop ECHOing, and the
            #       client has refused
            #   'server_stop' - The server has suggested that the client resume ECHOing, and the
            #       client has agreed
            echoMode                    => 'no_invite',
            # SGA mode:
            #   'no_invite' - The server has not suggested SGA yet
            #   'client_agree' - The server has suggested that the client SGA, and the client has
            #       agreed
            #   'client_refuse' - The server has suggested that the client SGA, and the client has
            #       refused
            #   'server_stop' - The server has suggested that the client stopped SGA, and the client
            #       has agreed
            sgaMode                     => 'no_invite',
            # List of TTYPE terminal types to send (items are removed from the front of the list,
            #   one by one, until one item remains, which should be 'unknown', and then this item
            #   is sent again, if another server request is received)
            sendTTypeList               => [],
            # The first terminal type sent is supposed to be the client's preferred one. It is
            #   stored here, in case anything needs it later
            specifiedTType              => undef,
            # EOR mode:
            #   'no_invite' - The server has not suggested EOR yet, but the client is willing
            #   'client_agree' - The server has suggested EOR, and the client has agreed
            #   'client_refuse' - The client refuses to use EOR
            eorMode                     => 'no_invite',
            # NAWS mode:
            #   'no_invite' - The server has not suggested NAWS yet, but the client is willing
            #   'client_agree' - The server has suggested NAWS, and the client has agreed
            #   'client_refuse' - The client refuses to use NAWS
            nawsMode                    => 'no_invite',
            # The size of the 'main' window's default tab textview for this session, in characters
            textViewWidthChars          => undef,
            textViewHeightChars         => undef,

            # IVs for MUD protocols
            # ---------------------

            # MSDP mode:
            #   'no_invite' - The server has not suggested MSDP yet, but the client is willing
            #   'client_agree' - The server has suggested MSDP, and the client has agreed
            #   'client_refuse' - The client refuses to use MSDP
            msdpMode                    => 'no_invite',
            # Hash of generic MSDP commands supported by the server (returned after LIST COMMANDS)
            msdpGenericCmdHash          => {},          # Set below
            # Hash of custom MSDP commands supported by the server (in the same form as above, but
            #   only supported commands are included, so for all key-value pairs, the value is TRUE)
            msdpCustomCmdHash           => {},
            # Hash of generic MSDP lists supported by the server (returned after LIST LISTS)
            msdpGenericListHash         => {},          # Set below
            # Hash of custom MSDP lists supported by the server (in the same form as above, but only
            #   supported lists are included, so for all key-value pairs, the value is TRUE)
            msdpCustomListHash          => {},
            # During the initial negotiation, Axmud asks the server to send its supported lists,
            #   waiting for a response before sending the next one. This IV keeps track of where we
            #   are in the process.
            #   'none_sent' - No LIST command sent yet
            #   'sent_commands' - Sent LIST COMMANDS, waiting response
            #   'sent_lists' - Sent LIST LISTS, waiting response
            #   'sent_other' - Sent LIST <something else>, waiting for a response
            #   'ready' - All responses received (the client might repeat any of them at any
            #       point in the future, but ->mdspInitMode remains set to 4)
            msdpInitMode                => 'none_sent',
            # During ->msdpInitMode 'sent_other', a list of supported lists to request. As each
            #   request is sent, the first item in this list is removed until it is empty
            msdpRequestList             => [],
            # Hash of generic MSDP configurable variables supported by the server (returned after
            #   LIST CONFIGURABLE_VARIABLES
            msdpGenericConfigFlagHash   => {},          # Set below
            # Hash of custom MSDP configurable variables (in the same form as above, but only
            #   supported configurable variables are included, so for all key-value pairs, the value
            #   is TRUE)
            msdpCustomConfigFlagHash    => {},
            # Hash of generic MSDP configurable variables whose values have been configured (sent to
            #   the server), in the form
            #       $hash{variable} = value
            # ...where 'variable' is one of the keys in $self->msdpGenericConfigFlagHash and 'value'
            #   is its corresponding value
            msdpGenericConfigValHash    => {},
            # Hash of custom MSDP configurable variables whose values have been configured (in the
            #   same form)
            msdpCustomConfigValHash     => {},
            # Hash of generic MSDP reportable variables supported by the server (returned after
            #   LIST REPORTABLE_VARIABLES
            msdpGenericReportableFlagHash
                                        => {},          # Set below
            # Hash of custom MSDP reportable variables (in the same form as above, but only
            #   supported reportable variables are included, so for all key-value pairs, the value
            #   is TRUE)
            msdpCustomReportableFlagHash
                                        => {},
            # Hash of generic MSDP reported variables supported by the server (returned after
            #   LIST REPORTED_VARIABLES; but is also updated after a call to
            #   $self->optSendMsdpReport or ->optSendMsdpUnreport)
            msdpGenericReportedFlagHash => {},          # Set below
            # Hash of custom MSDP reported variables (in the same form as above; key-value pairs can
            #   have a TRUE or FALSE value) (updated after LIST REPORTED VARIABLES; but is also
            #   updated after a call to $self->optSendMsdpReport or ->optSendMsdpUnreport)
            msdpCustomReportedFlagHash  => {},
            # Hash of generic MSDP sendable variables supported by the server (returned after
            #   LIST SENDABLE_VARIABLES)
            msdpGenericSendableFlagHash => {},          # Set below
            # Hash of custom MSDP sendable variables (in the same form as above, but only
            #   supported sendable variables are included, so for all key-value pairs, the value
            #   is TRUE)
            msdpCustomSendableFlagHash  => {},
            # Hash of generic MSDP reportable variables whose values have been reported (sent by
            #   the server), in the form
            #       $hash{variable} = value
            # ...where 'variable' is one of the keys in $self->msdpGenericReportableFlagHash and
            #   'value' is its corresponding value
            msdpGenericValueHash        => {},
            # Hash of custom MSDP reportable variables whose values have been reported (in the same
            #   form)
            msdpCustomValueHash         => {},
            # If an MSDP sanity check fails, this IV is set with a 3-digit error code representing
            #   the point at which the failure occured
            msdpSanityNum               => undef,

            # MSSP mode:
            #   'no_invite' - The server has not suggested MSSP yet, but the client is willing
            #   'client_agree' - The server has suggested MSSP, and the client has agreed
            #   'client_refuse - The client refuses to use MSSP
            msspMode                    => 'no_invite',

            # MCCP mode (both MCCP1 and MCCP2 are implemented):
            #   'no_invite' - The server has not suggested MCCP yet, but the client is willing
            #   'client_agree' - The server has suggested MCCP, and the client has agreed
            #   'client_refuse' - The client refuses to use MCCP
            #   'compress_start' - The server has suggested MCCP, the client has agreed, and the
            #       server has signalled that it has started compression
            #   'compress_error' - During mode 'compress_start' there was a compression error, so
            #       the client has signalled the server to stop using MCCP
            #   'compress_stop' - During mode 'compress_start' the client sent Z_FINISH, terminating
            #       the compression stream
            mccpMode                    => 'no_invite',

            # MSP mode:
            #   'no_invite' - The server has not suggested MSP yet, but the client is willing
            #   'client_agree' - The server has suggested MSP, and the client has agreed
            #   'client_refuse' - The client refuses to use MSP
            #   'client_simulate' - The server did not negotiate MSP, but Axmud is responding to MSP
            #       sound/music triggers
            mspMode                     => 'no_invite',
            # A registry of GA::Obj::Sound objects. A new one is created for every sound played by
            #   GA::Client->playSoundFile (usually initiated by MSP), but not Axmud sound effects
            #   played by GA::Client->playSound (which are independent of MSP)
            # (Only sounds played by this session are stored in this session's hash)
            # Hash in the form
            #   $soundHarnessHash{unique_number} = $sound_object
            soundHarnessHash            => {},
            # The number of GA::Obj::Sound objects created (used to give each object a unique number
            soundHarnessCount           => 0,
            # The default URL to use for MSP downloads (if allowed)
            mspDefaultURL               => undef,

            # MXP mode:
            #   'no_invite' - The server has not suggested MXP yet, but the client is willing
            #   'client_agree' - The server has suggested MXP, and the client has agreed
            #   'client_refuse - The client refuses to use MXP
            mxpMode                     => 'no_invite',
            # MXP line mode, set by $self->extractEscapeSequences as each MXP escape sequence is
            #   processed, and reset by the end of each line
            #   0   - Open line
            #           'Only MXP commands in the "open" category are allowed. When a newline is
            #           received from the MUD, the mode reverts back to the Default mode.  OPEN MODE
            #           starts as the Default mode until changes with one of the "lock mode" tags
            #           listed below.'
            #   1   - Secure line
            #           'All tags and commands in MXP are allowed within the line. When a newline is
            #           received from the MUD, the mode reverts back to the Default mode.'
            #   2   - Locked line
            #           'No MXP or HTML commands are allowed in the line. The line is not parsed for
            #           any tags at all. This is useful for "verbatim" text output from the MUD.
            #           When a newline is received from the MUD, the mode reverts back to the
            #           Default mode.'
            # NB Set to 'undef' when $self->mxpMode is not 'client_agree', or when it is
            #   'client_agree' but no MXP escape sequences have been processed yet. Immediately
            #   after that, the default value for ->mxpLineMode is 0 (Open line)
            mxpLineMode                 => undef,
            # MXP default line mode, also set by $self->extractEscapeSequences as MXP escape
            #   sequences are processed
            #   0   - No lock mode
            #           ->mxpLineMode resets to the default line mode, 0, after every line
            #   5   - Lock open mode
            #           'Set open mode. Mode remains in effect until changed. OPEN mode becomes the
            #           new default mode.'
            #   6   - Lock secure mode
            #           'Set secure mode. Mode remains in effect until changed. Secure mode becomes
            #           the new default mode.'
            #   7   - Lock locked mode
            #           'Set locked mode. Mode remains in effect until changed. Locked mode becomes
            #           the new default mode.'
            # NB Set to 'undef' when $self->mxpMode is not 'client_agree', or when it is
            #   'client_agree' but no MXP escape sequences have been processed yet. Immediately
            #   after that, the default value for ->mxpDefaultMode is 0 (No lock mode)
            mxpDefaultMode              => undef,
            # MXP temporary secure mode. When '<ESC>[4z', the escape sequence for 'temp secure mode'
            #   is received, this IV is set to the value of $self->mxpLineMode. The escape sequence
            #   must be followed by an MXP tag <...>; as soon as it's been processed, the value of
            #   ->mxpMode is restored, and the value of this IV is set back to 'undef'
            mxpTempMode                 => undef,
            # IV set in $self->processIncomingData just before the call to $self->processMxpElement,
            #   representing all the text that's been processed by $self->processIncomingData, but
            #   which hasn't been displayed in the textview object yet; reset after the call to
            #   ->processMxpElement is complete (or when a newline is processed)
            # Used to display the undisplayed text just before a <FRAME> or <DEST> tag is processed,
            #   so it's displayed in the right textview
            # Used by <A> and <SEND> tags to get the positioning of link objects (GA::Obj::Link)
            #   right, as the link objects created by those tags aren't applied to the textview
            #   until ->processIncomingData finishes processing the packet of text
            mxpOrigText                 => undef,
            # When the link object (GA::Obj::Link) is created by the <A> and <SEND> tags, it can't
            #   be applied to the current textview immediately, because there may be some
            #   displayable text before the link which hasn't been displayed yet
            # If so, $self->processMxpLinkElement and ->processMxpSendElement store the link object
            #   in this list, temporarily, so that ->processIncomingData can apply the link(s) to
            #   the textview as soo as it's ready
            mxpTempLinkList             => [],
            # A hash of current user-defined MXP elements, each stored in a GA::Mxp::Element
            #   object. Since element names are case insensitive, Axmud stores user-defined elements
            #   with lower-case names. Hash in the form
            #       $mxpElementHash{name} = blessed_reference_of_mxp_element_object
            mxpElementHash              => {},
            # A hash of current user-defined MXP entities, each stored in a GA::Mxp::Entity
            #   object. Entity names are case-sensitive. Hash in the form
            #       $mxpEntityHash{name} = blessed_reference_of_mxp_entity_object
            # NB There is a constant hash of entity names in GA::Client->constMxpEntityHash. If the
            #   world chooses to redefine one of these standard entity names, a GA::Mxp::Entity
            #   object is created for it and added to $self->mxpEntityHash; but standard entity
            #   names that haven't been redefined don't have their own GA::Mxp::Entity object
            # NB Axmud also recognises entities in the form &#nnn; - but they don't exist in
            #   GA::Client->constMxpEntityHash and are never added to this hash
            mxpEntityHash               => {},
            # A hash of file filters specified by the world, each stored in a GA::Mxp::Filter
            #   object. Hash in the form
            #       $mxpFilterHash{source_extension} = blessed_reference_of_mxp_filter_object
            #   ...where 'source_extension' is the file extension of the world's own image/sound
            #       format, e.g. 'gff' (which the specified plugin will convert to 'gif'
            mxpFilterHash               => {},
            # A hash of current MXP frames, each stored in a a GA::Mxp::Frame object (the MXP spec
            #   doesn't say if the frame name is case-sensitive or not, so Axmud will treat them as
            #   case-sensitive). Used for both external and internal frames
            # The first frame added always has the name '_top'. The name '_previous' refers to
            #   one of the frames in this hash, so there is never a frame object called '_previous'
            # Hash in the form
            #   $mxpFrameHash{frame_name} = blessed_reference_to_frame_object
            mxpFrameHash                => {},
            # The name of the frame currently being used to display text received from the world,
            #   and the name of the frame previously used to display text received from the world
            mxpCurrentFrame             => undef,
            mxpPrevFrame                => undef,
            # For internal frames, the size of the default tab's pane object on the 60x60 grid when
            #   the first frame (besides the default '_top' frame) is added
            mxpFrameWidth               => undef,
            mxpFrameHeight              => undef,
            # For internal frames, the size of new frames is a factor of the original size of the
            #   default tab's pane object
            mxpFrameXFactor             => 0.33,        # Width is 33% of default pane
            mxpFrameYFactor             => 0.25,        # Height is 25% of default pane
            # When a <DEST>...</DEST> construction is being processed, the opening <DEST> tag causes
            #   the creation on a GA::Mxp::Dest object. When the matching </DEST> tag is processed,
            #   the text received between the tags is sent to the specified frame. Meanwhile, any
            #   text tokens encountered are added to the object's ->text IV
            mxpCurrentDest              => undef,
            # When a <V>...</V> construction is being processed, the opening <V> tag causes the
            #   creation of a GA::Mxp::Var object. When the matching </V> tag is processed, the
            #   object's properties are transferred to the corresponding entity. Meanwhile, any
            #   text tokens encountered are added to the object's ->value IV
            # The current GA::Mxp::Var object, if we're between matching <V> and </V> tags
            mxpCurrentVar               => undef,
            # When an <A>...</A> construction is being processed, the opening <A> tag causes the
            #   creation of a temporary GA::Obj::Link object whose ->number is set to -1. When the
            #   matching </A> tag is processed, the object is added to the current textview and
            #   given a unique ->number within that textview
            # The temporary (unfinished) GA::Obj::Link object, if we're between matching <A> and
            #   </A> tags
            mxpCurrentLink              => undef,
            # When a <SEND>...</SEND> construction is being processed, the opening <SEND> tag causes
            #   the creation of a GA::Obj::Link object whose ->number is set to -1. When the
            #   matching </SEND> tag is processed, the object is added to the current textview and
            #   given a unique ->number within that textview
            # The temporary (unfinished) GA::Obj::Link object, if we're between matching <SEND>
            #   and </SEND> tags
            mxpCurrentSend              => undef,
            # Hash used to store the text between two matching custom elements, when those elements
            #   have a FLAG=... argument
            # e.g. from the MXP spec, <!ELEMENT RName FLAG="RoomName"> ... <RName>Temple</RName>
            # When the closing tag is received, any text between the matching tags is stored in the
            #   display buffer object for the line containing the closing tag
            # If there were newline characters between matching tags, they are ignored (and a space
            #   character is inserted, if necessary); so the stored text is always a single string
            #   without newline characters
            # The MXP spec defineds six standard tag properties. 'RoomName', 'RoomDesc', 'RoomExit'
            #   and 'RoomNum' are handled by the Locator task. 'Prompt' is handled by
            #   $self->processLineSegment. 'Set xxx' is handled by $self->popMxpStack. Non-standard
            #   tag properties are simply stored in the display buffer, as described above
            # Hash in the form
            #   $mxpFlagTextHash{tag_property} = string
            mxpFlagTextHash             => {},
            # As soon as the closing tag is received, a key value pair is removed from
            #   ->mxpFlagTextHash and added to this hash, which is used by $self->processLineSegment
            #   to update the display at the next available opportunity
            # Hash in the form
            #   $mxpFlagTextStoreHash{tag_property} = string
            mxpFlagTextStoreHash        => {},
            # MXP can send a style sheet number, indicating "the current version of the optional
            #   style sheet" which seems to be something to do with the subset of elements and
            #   entities currently in use. The MXP spec only requires us to store this number, and
            #   to return it in response to a VERSION tag
            mxpStyleSheetNum            => undef,
            # Two flags used to handle the MXP tags <NOBR>, <P> and </P>, and set/reset by
            #   $self->processMxpSpacingTag
            mxpIgnoreNewLineFlag        => FALSE,
            mxpParagraphFlag            => FALSE,
            # Equvalent flag used to handle the MXP tags <H1>...</H1> to <H6>...</H6>
            mxpHeadingFlag              => FALSE,
            # If MXP has created a 'main' window gauge box, GA::Strip::GaugeBox object which handles
            #   gauges in that 'main' window
            mxpGaugeStripObj            => undef,
            # If MXP has created a 'main' window gauge box, the unique number of the
            #   GA::Obj::GaugeLevel object used to draw MXP gauges
            mxpGaugeLevel               => undef,
            # A hash of GA::Obj::Gauge objects, one for each MXP gauge drawn. Hash in the form
            #   $mxpGaugeHash{entity_name} = gauge_object
            # ...where 'entity_name' matches a key in $self->mxpEntityHash
            mxpGaugeHash                => {},
            # Whenever an entity is created or updated, it is added to this hash so that
            #   $self->updateMxpGauges (called by $self->spinMaintainLoop) knows to update the
            #   corresponding gauges. Hash in the form
            #       $mxpGaugeUpdateHash{entity_name} = undef
            mxpGaugeUpdateHash          => {},
            # MXP server crosslinking mode, used in response to a <RELOCATE>...</RELOCATE> sequence
            #   'none' - No crosslinking operation is in progress, or a crosslinking operation has
            #       been completed (and the character has been marked 'logged in')
            #   'wait_start' - The <RELOCATE> tag has been received, and the crosslinking operation
            #       is waiting to start
            #   'started' - The crosslinking operation has started, and the old connection has been
            #       terminated
            #   'wait_login' - The new connection has been initiated, and we're waiting to log in
            #       (manually or automatically)
            mxpRelocateMode             => 'none',
            # The hostname and port specified by the <RELOCATE> tag
            mxpRelocateHost             => undef,
            mxpRelocatePort             => undef,
            # Flag set to TRUE when the <QUIET> tag is received, set back to FALSE when the
            #   </RELOCATE> tag is received
            mxpRelocateQuietFlag        => FALSE,
            # Flag set to TRUE when an MXP escape sequence (<esc>!19z) is received, which is
            #   equivalent to a <QUIET> tag for a single line
            mxpRelocateQuietLineFlag    => FALSE,
            # Flag set to TRUE when an external frame (implemented as a Frame task window) is
            #   close by the user; when TRUE, frames are disable for the rest of the session
            mxpDisableFrameFlag         => FALSE,
            # MXP login mode, used to react when the world sends the <USER> and <PASSWORD> tags
            #   umprompted (i.e., not during a server crosslinking operation). In case either tag is
            #   ever sent twice during a session, any part of the code can reset this IV back to 0,
            #   ready to receive the next <USER> tag
            #   'no_tag' - Neither tag has been received
            #   'user_tag' - The <USER> tag has been received,
            #   'pwd_tag' - The <PASSWORD> tag has been received (after <USER> was received)
            mxpLoginMode                => 'no_tag',
            # List of MXP/Pueblo debug messages created when processing a token. A message is added
            #   to the list by a call to $self->mxpDebug or $self->puebloDebug. When
            #   $self->processIncomingData is ready, any debug messages are displayed. (Doing it
            #   this way saves us from some very ugly Gtk2 errors)
            # List in groups of 4, in the form
            #   (protocol, token, num, message...)
            # ...where 'protocol' is the string 'mxp' or 'pueblo', 'token' is the original MXP or
            #   Pueblo token (e.g. '<H1>'), 'num' is an identifying 4-digit error number and
            #   'message' is the error message
            # NBH Currently, MXP errors use the range 1000-4999, Pueblo errors use the range
            #   6000-8999, mixed MXP/Pueblo errors use the range 6000-6999 and any other code
            #   (e.g. temporary eror messages) can use the range 9000-9999
            mxpPuebloDebugList          => [],

            # Pueblo mode:
            #   'no_invite' - The server has not suggested Pueblo
            #   'client_agree' - The server has suggested Pueblo, and the client has agreed
            #   'client_refuse - The server has suggested Pueblo, and the client has refused
            # (NB In line with other major MUD clients, Axmud offers only partial Pueblo support)
            puebloMode                  => 'no_invite',
            # The Pueblo version detected (e.g. 1.10). Axmud responds with 'PUEBLOCLIENT 2.01', just
            #   as zMud does
            puebloVersion               => undef,
            # When the </XCH_MUDTEXT> tag is received, Axmud begins interpreting text received from
            #   the world as HTML. and this flag is set to TRUE. When the corresponding
            #   <XCH_MUDTEXT> tag is received, Axmud resumes interpreting text as normal, and this
            #   flag is set to FALSE
            puebloActiveFlag            => FALSE,
            # The default base font size (in the range 1-7, default 3), used by <FONT> tags and set
            #   by <BASEFONT>
            puebloBaseFontSize          => 3,
            # A list of GA::Pueblo::List objects currently being processed, each one representing
            #   an unordered list (<UL>...</UL>) or and ordered list <OL>...</OL>)
            # If the list contains more than one object, the lists are embedded within each other.
            #   The outermost object is the first one in the list, and the currently processed
            #   object is the last one in the list
            puebloStackList             => [],
            # For list items (<LI>), the number of space characterss per list. A simple list will
            #   have three spaces before each list item. If there are two lists, one embedded within
            #   the other, each list item will have six spaces. If there are three lists, nine
            #   spaces. (This roughly matches the format of help files)
            puebloColumnSize            => 3,
            # For the tags </UL>, </OL> and <LI>, $self->processPuebloListElement and
            #   $self->processPuebloListItemElement create a string that should be inserted into
            #   the text being processed by $self->processIncomingData; that string is stored here
            #   until being used
            puebloInsertString          => '',
            # Flag set to TRUE inside a <P>...</P> construction, and FALSE outside it
            puebloParagraphFlag         => FALSE,
            # Flag set to TRUE inside a <CODE>...</CODE> construction, and FALSE outside it
            # (Axmud currently implements <PRE>...</PRE> in the same way, so that construction also
            #   sets the flag to TRUE)
            puebloLiteralFlag           => FALSE,
            # Flag set to TRUE inside a <SAMP>...</SAMP> construction, and FALSE outside it
            #   (Axmud implements <SAMP> in the same wasy <CODE> except that whitespace is not
            #   reduced inside <SAMP>...</SAMP> constructions)
            puebloLiteralSampFlag       => FALSE,
            # After a tag like </CENTER>, the Axmud style tag 'justify_default' must be applied to
            #   the beginning of the next line (or, at least, the beginning of the next line that
            #   might contain characters. Various functions set this IV to these values:
            #       'normal' - normal operations
            #       'wait_newline' - </CENTER> tag processed, now waiting for newline character
            #       'wait_loop' - newline character processed, now waiting for next loop inside
            #               $self->processIncomingData
            puebloJustifyMode           => 'normal',

            # ATCP mode:
            #   'no_invite' - The server has not suggested ATCP yet, but the client is willing
            #   'client_agree' - The server has suggested ATCP, and the client has agreed
            #   'client_refuse' - The client refuses to use ATCP
            atcpMode                    => 'no_invite',
            # Hash of ATCP data received. Keys are strings representing
            #   'Package[.SubPackage][.Message]'; the corresponding values are GA::Obj::Atcp objects
            # NB ATCP package names are case-insensitive; Axmud always converts them to lower case
            #   before storing them in this hash
            atcpDataHash                => {},

            # GMCP mode:
            #   'no_invite' - The server has not suggested GMCP yet, but the client is willing
            #   'client_agree' - The server has suggested GMCP, and the client has agreed
            #   'client_refuse' - The client refuses to use GMCP
            gmcpMode                    => 'no_invite',
            # Hash of GMCP data received. Keys are strings representing
            #   'Package[.SubPackage][.Message]'; the corresponding values are GA::Obj::Gmcp objects
            # NB GMCP package names are case-insensitive; Axmud always converts them to lower case
            #   before storing them in this hash
            gmcpDataHash                => {},

            # Other IVs
            # ---------

            # Flag used by ';simulatecommand' to temporarily block any world commands from being
            #   sent to the world (by $self->dispatchCmd); however, everything else - such as
            #   the call to $self->checkAliases and ->updateCmdBuffer - happen as normal, as if the
            #   commands had been sent
            # Set to TRUE if all world commands should be blocked. Set to FALSE otherwise
            disableWorldCmdFlag         => FALSE,

            # Integer (0 or above) used by generic client command functions
            spelunkerMode               => 0,

            # Redirect mode (which transforms all world commands that are (custom) primary and/or
            #   secondary directions, e.g. 'north', 'n' or 'out', into a world command like
            #   'sail north', 'sail n' or 'sail out'
            # The redirect mode string. Every occurence of the character @ is replaced by the
            #   original command, e.g. 'sail @' becomes 'sail north', 'sail n' or 'sail out'. Set to
            #   'undef' when redirect mode  is turned off
            redirectString              => undef,
            # Redirect mode can operate in one of three states:
            #   'primary_only' - redirect primary directions
            #   'primary_secondary' - redirect primary and secondary directions
            #   'all_exits' - redirect primary and secondary directions, plus any command matching
            #       an exit in the current room (if set; actually the automapper's ->ghostRoom)
            # NB Redirect mode takes priority over assisted moves, if they are turned on
            redirectMode                => 'all_exits',

            # The timeout (in seconds) used by $self->doConnect when attempting to open a telnet,
            #   SSH or SSL connection
            connectTimeOut              => 60,
            # The current list of GA::Obj::Repeat objects (created by ';repeatinterval'), which
            #   store world commands to be sent several times, at fixed intervals.
            repeatObjList               => [],
            # How client commands show standard messages (produced by calls to
            #   GA::Generic::Cmd->complete, ->error or ->improper. Does not affect calls to
            #   Games::Axmud->writeText, ->writeDebug etc)
            #  'show_all' - (default) show all standard messages produced by the command (with calls
            #       to GA::Generic::Cmd->complete, ->error and ->improper)
            #  'hide_complete' - suppress messages produced by a call to GA::Generic::Cmd->complete
            #       (on the successful execution of a command), but display error messages
            #  'hide_system' -  suppress all standard messages produced by the command (with calls
            #       to GA::Generic::Cmd->complete, ->error and ->improper)
            #  'win_error' - show messages produced by a call to GA::Generic::Cmd->complete (on the
            #       successful execution of a command) in the 'main' window, but show error message
            #       calls to ->error and ->improper in a 'dialogue' window
            #  'win_only' - suppress all messages produced by a call to GA::Generic::Cmd->complete
            #       (on the successful execution of a command), but show error message calls to
            #       ->error and ->improper in a 'dialogue' window
            cmdMode                     => 'show_all',

            # For the benefit of the ';peek' command, the last string that was used in a ';poke' or
            #   a ';peek' command ('undef' if neither used during this session)
            prevPokeString              => undef,

            # When the user imports a world model via the Automapper window's 'Import/load world
            #   model' menu item, this flag gets set to TRUE. It allows ;importfiles to transfer a
            #    world model whose ->_parentWorld is not the current world, into the current world's
            #   file structure, after a prompt to the user. Once the operation is complete, the
            #   Automapper window loads the file, sets its ->_parentWorld to the rigth value, saves
            #   the file, and then resets this flag
            transferWorldModelFlag      => FALSE,

            # When rewriter triggers have their 'rewrite_global' attribute set, it would be possible
            #   to cause an infinite loop by (for example) replacing a line of '=', 20 characters or
            #   more, by a line of '=' characters exactly 20 characters in length
            # Users tend to get upset when Axmud crashes, so we'll use a maximum number of rewrites
            #   per trigger, per line
            constRewriteMax             => 16,
        };

        # Bless the object into existence
        bless $self, $class;

        # Set remaining IVs
        $self->{session}                = $self;
        $self->{profPriorityList}       = [$axmud::CLIENT->constProfPriorityList];

        $self->resetMsdpData();

        return $self;
    }

    ##################
    # Methods

    sub start {

        # Called by GA::Client->startSession
        # Sets up the session
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the session can't be started
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my (
            $sessionCount, $winObj, $mode, $string, $taskObj, $pluginString, $dialogueWin,
            @list,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->start', @_);
        }

        # Count the number of sessions that exist, besides this one
        $sessionCount = 0;
        foreach my $otherSession ($axmud::CLIENT->ivValues('sessionHash')) {

            if ($otherSession ne $self) {

                $sessionCount++;
            }
        }

        # Create a 'main' window for this session (or use the existing one, if all sessions share a
        #   'main' window)
        $winObj = $self->setMainWin($sessionCount);
        if (! $winObj) {

            return $self->writeError(
                'Could not set up a \'main\' window for this session',
                $self->_objClass . '->start',
            );

        } else {

            $self->ivPoke('mainWin', $winObj);
        }

        # Set the tab used as this session's default tab, creating it if necessary
        if (! $self->setDefaultTab()) {

            # (Don't destroy a 'main' window being used by other sessions)
            if (! $axmud::CLIENT->shareMainWinFlag || ! $sessionCount) {

                $self->mainWin->winDestroy();
            }

            return $self->writeError(
                'Could not set up the \'main\' window for this session',
                $self->_objClass . '->start',
            );
        }

        # Set up profiles for this session
        if (! $self->initTempFlag) {
            $mode = 'start';        # Create a new world profile, if it doesn't yet exist
        } else {
            $mode = 'start_temp';   # Create a new temporary world profile
        }

        if (
            ! $self->setupProfiles(
                $mode,
                $self->initWorld,
                $self->initChar,
                $self->initHost,
                $self->initPort,
                $self->initLoginMode,
            )
        ) {
            # (Error message already displayed)
            return undef;
        }

        # There has been a change in current profiles (from 'nothing' to 'something'), but there are
        #   no tasks running, so they don't need to be informed about it
        $self->ivPoke('currentProfChangeFlag', FALSE);

        # Set the character set used to display text received from the world in this session
        $self->setCharSet();

        # Temporarily disable text-to-speech for the introductory system messages (a nuisance, if
        #   the user has to hear them at the start of every session)
        $self->ivPoke('ttsTempDisableFlag', TRUE);

        # If this is the first session, display information about the client
        if ($axmud::CLIENT->sessionCount <= 1) {

            $self->writeText(
                $axmud::SCRIPT . ' v' . $axmud::VERSION . ' (' . $axmud::DATE . ') - '
                . $axmud::COPYRIGHT,
            );

            $self->writeText(
                'Type \';about\' for license information, \';help\' or \';quickhelp\' for'
                . ' assistance',
            );

            $self->writeText(' ');      # Empty line

            $self->writeText('Support and new releases available from ', 'echo');
            $self->writeText($axmud::URL, 'link');

            $self->writeText(' ');      # Empty line
        }

        if ($self->initTempFlag) {

            # Display some explanatory text, one line at a time
            @list = (
                'Because you didn\'t specify a name for this world, a temporary world profile',
                '   (called \'' . $self->currentWorld->name . '\') has been created for you.',
                'Temporary worlds can\'t be saved, so any world-related data gathered during',
                '   this session will be lost unless you set a new world profile with the',
                '   \';setworld\' command.',
                'Alternatively, you can close this session and start a new one. Other kinds of',
                '   data (initial tasks, display settings, and so on) are not affected and can',
                '   still be saved using the \';save\' command.',
                ' ',
            );

            foreach my $line (@list) {

                $self->writeText($line);
            }
        }

        if ($self->initOfflineFlag) {

            # This session is operating in 'connect offline' mode
            $self->ivPoke('status', 'offline');

            # Display some reassuring text, one line at a time
            @list = (
                'This session is running in CONNECT OFFLINE mode - data files have been loaded as',
                'usual, but ' . $axmud::SCRIPT . ' is only simulating a connection to the world',
                ' ',
                'Current world     : ' . $self->currentWorld->name,
            );

            if ($self->currentGuild) {

                push (@list, 'Current guild     : ' . $self->currentGuild->name);
            }

            if ($self->currentRace) {

                push (@list, 'Current race      : ' . $self->currentRace->name);
            }

            if ($self->currentChar) {
                $string = $self->currentChar->name;
            } else {
                $string = '<none>';
            }

            push (@list,
                'Current character : ' . $string,
                ' ',
            );

            foreach my $line (@list) {

                $self->writeText($line);
            }

            # Update this session's tab label. The TRUE argument means definitely update it.
            #   (Nothing happens if the session is using a simple tab)
            $self->checkTabLabels(TRUE);

            # Update the connection info strip object for any 'internal' windows used by this
            #   session (should only be one, at this point)
            foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

                $winObj->setHostLabel($self->getHostLabelText());
            }
        }

        # Display a list of loaded plugins
        if ($axmud::CLIENT->pluginHash) {

            $pluginString = '';

            foreach my $pluginObj (
                sort {lc($a->name) cmp lc($b->name)} ($axmud::CLIENT->ivValues('pluginHash'))
            ) {
                if (! $pluginObj->enabledFlag) {
                    $pluginString .= ' -' . $pluginObj->name;
                } else {
                    $pluginString .= ' +' . $pluginObj->name;
                }
            }

            $self->writeText('Plugins loaded:' . $pluginString);
            $self->writeText(' ');
        }

        # If a world hint message is set, display it now
        if ($self->currentWorld->worldHint) {

            if ($self->currentWorld->longName) {
                $string = uc($self->currentWorld->longName);
            } else {
                $string = uc($self->currentWorld->name);
            }

            $string .= ': ' . $self->currentWorld->worldHint;

            $self->writeText($string);
            $self->writeText(' ');

            # If this is the first connection to this world, also display the message in a
            #   'dialogue' window
            if (! $self->currentWorld->numberConnects) {

                $self->mainWin->showMsgDialogue(
                    'World hint',
                    'info',
                    $string .= "\n\nTo see this message again, type ';hint'",
                    'ok',
                );
            }
        }

        # When connecting to a world, the 'Connecting...' message will appear on this line, instead
        if ($self->initOfflineFlag) {

            $self->writeText('Session ready');
        }

        # Re-enable text-to-speech after displaying the introductory system messages
        $self->ivPoke('ttsTempDisableFlag', FALSE);
        # Inserting a Gtk2 update here allows all of the introductory messages actually to be
        #   displayed, before any text-to-speech stuff is done
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->start');

        # Start the session loop (to which the maintenance, timer, incoming data, task and replay
        #   loops are subservient)
        if (! $self->startSessionLoop()) {

            return $self->writeError(
                'Could not start the session loop',
                $self->_objClass . '->start',
            );
        }

        if (! $self->initOfflineFlag) {

            # Attempt to connect to the world
            if (! $self->doConnect($self->initHost, $self->initPort)) {

                # (The return value is only false when improper arguments supplied)
                return undef;
            }

            # If an attempted connection is immediately refused by the host, $self->status will
            #   already be set to 'disconnected'. In that case, we don't want to do most of the
            #   things usually done by the rest of this function
            $self->spinIncomingLoop();

            if ($self->status eq 'disconnected') {

                $self->stopSessionLoop();

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $axmud::CLIENT->desktopObj->restrictWidgets();

                return 1;
            }
        }

        # The session may now display received text in its 'main' window tab
        $self->ivPoke('startCompleteFlag', TRUE);
        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $axmud::CLIENT->desktopObj->restrictWidgets();

        # Handle automatic logins
        if ($self->initOfflineFlag) {

            # In 'connect offline' mode, the character is always marked as logged in immediately
            $self->doLogin();

        } else {

            # Set up the automatic login (if any), but obviously don't attempt a login if we
            #   don't know the character's name and password
            if ($self->currentWorld->loginMode ne 'none' && $self->initChar && $self->initPass) {

                if (! $self->setupLogin()) {

                    $self->writeWarning(
                        'Could not set up an automatic login (in login mode '
                        . $self->currentWorld->loginMode
                        . '); use the \';login\' command after logging in manually',
                        $self->_objClass . '->start',
                    );
                }
            }

            # Check for already-received text
            if ($self->initialTextBuffer) {

                # Some text has been received which we haven't displayed yet
                $self->processIncomingData($self->initialTextBuffer);
                # (We don't need to keep that text)
                $self->ivPoke('initialTextBuffer', '');
            }
        }

        return 1;
    }

    sub stop {

        # Called by GA::Client->stopSession and ->stopAllSessions (only)
        # Terminates the session. Any existing connection is terminated (without halting the
        #   session) by a call to $self->doDisconnect or to the callback $self->connectionError
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the session can't be terminated
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my $actualCount;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stop', @_);
        }

        # Stop the session loop (if running)
        if ($self->sessionLoopObj && ! $self->stopSessionLoop()) {

            return $self->writeError(
                'Could not stop the session loop',
                $self->_objClass . '->stop',
            );
        }

        # Terminate the connection, if connected (or connecting)
        if (! $self->doDisconnect()) {

            return $self->writeError(
                'Could not terminate the connection',
                $self->_objClass . '->stop',
            );
        }

        # Update IVs
        $self->ivPoke('status', 'disconnected');
        # Update the world's connection history object, if one was created for this session
        if ($self->connectHistoryObj) {

            $self->connectHistoryObj->set_disconnectedTime();
        }

        # Count the number of sessions that exist, besides this one. We can't rely on
        #   GA::Client->sessionCount, because it might have been updated by the calling functions
        $actualCount = 0;
        foreach my $otherSession ($axmud::CLIENT->ivValues('sessionHash')) {

            if ($otherSession ne $self) {

                $actualCount++;
            }
        }

        # Ask the 'main' window to remove the tab for this session (if allowed)
        # Don't bother if sessions don't share a 'main' window (because this session's 'main'
        #   window is about to be closed anyway)
        # (If the session has ended because the 'main' windows has been destroyed, then the call to
        #   $self->del_winObj will already have set $self->defaultTabObj to 'undef')
        if (
            $axmud::CLIENT->shareMainWinFlag
            && $self->defaultTabObj
            && ! $self->defaultTabObj->paneObj->removeTab($self)
        ) {
            return $self->writeError(
                'Could not remove the tab for a session',
                $self->_objClass . '->stop',
            );
        }

        # Close any 'free' windows produced by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionFreeWins($self)) {

            # As one 'free' window is closed, its child 'free' windows are also closed, so we have
            #   to check the window still exists, before destroying it
            if ($axmud::CLIENT->desktopObj->ivExists('freeWinHash', $winObj->number)) {

                $winObj->winDestroy();
            }
        }

        # If sessions have their own workspace grids, remove the workspace grids (which closes their
        #   'grid' windows, but not this session's 'main' window, which we'll deal with in a moment)
        # If sessions share a workspace grid, do nothing
        $axmud::CLIENT->desktopObj->removeSessionWorkspaceGrids($self);

        # Remove any temporary zonemaps for this session
        foreach my $zonemapObj ($axmud::CLIENT->ivValues('zonemapHash')) {

            if ($zonemapObj->tempFlag && $zonemapObj->tempSession eq $self) {

                $axmud::CLIENT->del_zonemap($zonemapObj);
            }
        }

        # Check if there are any remaining 'grid' windows associated with this session and, if so,
        #   close them (but still don't close the 'main' window)
        $axmud::CLIENT->desktopObj->removeSessionWindows($self);

        # If this session has any 'external' windows on this session's workspace grid, and if this
        #   wasn't the current session, those 'external' windows may be invisible/minimised. Make
        #   them visible
        $axmud::CLIENT->desktopObj->revealGridWins($self);

        # Otherwise, when sessions don't share a 'main' window, we can delete it this session's
        #   'main' window now
        if (! $axmud::CLIENT->shareMainWinFlag) {

            $self->mainWin->winDestroy();
            $self->ivUndef('mainWin');

        } elsif (! $actualCount && ! $axmud::CLIENT->shutdownFlag) {

            # Convert the single remaining 'main' window back into a spare 'main' window
            $axmud::CLIENT->desktopObj->deconvertSpareMainWin($self->mainWin);
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $axmud::CLIENT->desktopObj->restrictWidgets();

        return 1;
    }

    # Setup

    sub setMainWin {

        # Called by $self->start
        # Creates a new 'main' window or re-uses an existing one
        #
        # Expected arguments
        #   $sessionCount   - The number of sessions that exist, besides this one (so can be 0)
        #
        # Return values
        #   'undef' on improper arguments or if the 'main' window can't be created
        #   1 on success

        my ($self, $sessionCount, $check) = @_;

        # Local variables
        my (
            $winmap, $winObj, $successFlag, $thisWorkspaceObj, $thisWorkspaceGridObj, $thisZoneObj,
            %workspaceHash,
        );

        # Check for improper arguments
        if (! defined $sessionCount || defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setMainWin', @_);
        }

        # If a winmap has been marked as the default for this world, use it (otherwise the function
        #   returns 'undef', and a default winmap is used)
        $winmap = $self->checkWinmapWorlds($self->initWorld);

        # If sessions share a 'main' window, create a workspace grid on every workspace for this
        #   session
        # If sessions don't share a 'main' window, use the shared workspace grid on every
        #   workspace (or create one on every workspace, if this is the first session)
        OUTER: foreach my $workspaceObj ($axmud::CLIENT->desktopObj->listWorkspaces()) {

            my $gridObj;

            if ($axmud::CLIENT->shareMainWinFlag) {
                $gridObj = $workspaceObj->addWorkspaceGrid($self);
            } elsif (! $sessionCount) {
                $gridObj = $workspaceObj->addWorkspaceGrid();
            }

            # (Temporarily storing workspace grid object makes the following call to
            #   ->createGridWin a lot simpler)
            if ($gridObj) {

                $workspaceHash{$workspaceObj->number} = $gridObj->number;
            }
        }

        # Create a 'main' window, or use an existing one
        if (
            ($axmud::CLIENT->shareMainWinFlag && $axmud::CLIENT->mainWin)
            || (! $axmud::CLIENT->shareMainWinFlag && ! $sessionCount)
        ) {
            # Use the existing shared 'main' window
            $winObj = $axmud::CLIENT->mainWin;
        }

        if (! $sessionCount && ! $axmud::TEST_MODE_FLAG) {

            # Convert a spare 'main' window into a normal one
            if (! $axmud::CLIENT->desktopObj->convertSpareMainWin($self, $winObj, $winmap)) {

                # Could not reposition the 'main' window, for some reason. Destroy it, and allow
                #   the code below to create a new one
                $winObj->winDestroy();
                $axmud::CLIENT->reset_mainWin();
                $winObj = undef;
            }
        }

        if (! $winObj) {

            # Create a new 'main' window for this session, using the first available workspace. If
            #   >shareMainWinFlag = TRUE, we can specify the workspace grid to use, too
            OUTER: foreach my $workspaceObj ($axmud::CLIENT->desktopObj->listWorkspaces()) {

                $winObj = $workspaceObj->createGridWin(
                    'main',                                 # Window type
                    'main',                                 # Window name
                    undef,                                  # Window title set automatically
                    $winmap,                                # Winmap name
                    'Games::Axmud::Win::Internal',          # Package name
                    undef,                                  # No known Gtk2::Window
                    undef,                                  # No known Gnome2::Wnck::Window
                    $self,                                  # Owner
                    $self,                                  # Owner session
                    $workspaceHash{$workspaceObj->number},  # 'undef' if ->shareMainWinFlag = FALSE
                );

                if ($winObj) {

                    # New 'main' window created on this workspace
                    last OUTER;
                }
            }
        }

        # Operation complete; if it failed, $winObj is 'undef'
        return $winObj;
    }

    sub checkWinmapWorlds {

        # Called by $self->setMainWin
        # Some winmaps are marked for use as the 'default' winmap for a particular world. Ideally,
        #   each world should have no more than one winmap which is marked as the default for that
        #   world but, just in case, we'll check all winmaps alphabetically, using the first one we
        #   find
        #
        # Expected arguments
        #   $worldName      - A world profile name (if called by $self->setMainWin, the same as
        #                       $self->initWorld)
        #
        # Return values
        #   'undef' on improper arguments or if this world has no default winmaps
        #   Otherwise, returns the first default winmap found

        my ($self, $worldName, $check) = @_;

        # Local variables
        my @winmapList;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->checkWinmapWorlds', @_);
        }

        @winmapList = sort {lc($a->name) cmp lc($b->name)} ($axmud::CLIENT->ivValues('winmapHash'));
        foreach my $winmapObj (@winmapList) {

            if ($winmapObj->ivExists('worldHash', $worldName)) {

                # This is the default winmap for the world
                return $winmapObj->name;
            }
        }

        # No default winmap found for this world
        return undef;
    }

    sub setDefaultTab {

        # Called by $self->start
        # Sets this session's default tab, adding a tab to the 'main' window if necessary
        #
        # Return values
        #   'undef' on improper arguments or if the default pane object / textview object can't be
        #       created
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my ($paneObj, $tabLabelText, $tabObj);

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setDefaultTab', @_);
        }

        # Use the first pane object (GA::Table::Pane) found as the default pane object
        #   (GA::Obj::Workspace->getWinmap has already checked that this 'main' window has one)
        $paneObj = $self->mainWin->findTableObj('pane');

        # Set the initial text to use in the session's tab label (if one is visible)
        if ($axmud::CLIENT->sessionTabMode eq 'char') {
            $tabLabelText = '(' . ucfirst($self->initWorld) . ')';
        } else {
            $tabLabelText = ucfirst($self->initWorld);
        }

        # Add a new tab (containing a textview object)
        if (! $paneObj->notebook && ! $paneObj->tabObjHash) {
            $tabObj = $paneObj->addSimpleTab($self, TRUE, TRUE, $tabLabelText);
        } else {
            $tabObj = $paneObj->addTab($self, TRUE, TRUE, $tabLabelText);
        }

        if (! $tabObj) {

            return undef;

        } else {

            # Update IVs
            $self->ivPoke('defaultTabObj', $tabObj);
            $self->ivPoke('currentTabObj', $tabObj);

            # Don't let anything remove the pane object
            $paneObj->set_allowRemoveFlag(FALSE);

            return 1;
        }
    }

    sub setCharSet {

        # Called by $self->start or GA::Cmd::SetCharSet->do
        # Sets the character set used to display text received from the world in this session
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the character set now used (e.g. 'iso-8859-1', 'utf-8-strict')

        my ($self, $check) = @_;

        # Local variables
        my $charSet;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setCharSet', @_);
        }

        # Decide which IV to use
        if ($self->currentWorld->worldCharSet) {
            $charSet = $self->currentWorld->worldCharSet;
        } else {
            $charSet = $axmud::CLIENT->charSet;
        }

        # Check the character set is available by loading it
        if (! Encode::find_encoding($charSet)) {

            # Emergency fallback is 'null', which $self->incomingDataLoop uses to mean 'don't
            #   convert anything into a new character set'
            $charSet = 'null';
        }

        # Update the IV
        $self->ivPoke('sessionCharSet', $charSet);

        return $charSet;
    }

    sub setModifyFlag {

        # Called by anything
        # Sets a file object's ->modifyFlag.
        # First checks that the file object is stored in this session's file object registry
        #   ($self->sessionFileObjHash). If not passes the request to the GA::Client's own
        #   ->setModifyFlag function
        # (In this way, code - especially in the generic object, Games::Axmud - can set a flag
        #   without knowing in which registry the file object is stored)
        #
        # Expected arguments
        #   $objName    - The unique name of the file object, matching a key in
        #                   $self->sessionFileObjHash ('otherprof', 'worldmodel' or the name of a
        #                   profile associated with the current world profile) or in
        #                   GA::Client->fileObjHash (any other file object)
        #   $flag       - The setting for the flag (TRUE of FALSE)
        #
        # Optional arguments
        #
        #   $func       - The calling function. Ignored for now, if specified
        #
        # Return values
        #   'undef' on improper arguments
        #   $flag on success

        my ($self, $objName, $flag, $func, $check) = @_;

        # Check for improper arguments
        if (! defined $objName || ! defined $flag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setModifyFlag', @_);
        }

        # Check the file object exists
        if (! $self->ivExists('sessionFileObjHash', $objName)) {

            # Ask the client to consult its registry
            return $axmud::CLIENT->setModifyFlag($objName, $flag, $func);

        } else {

            # Consult the session's registry
            return $self->ivShow('sessionFileObjHash', $objName)->set_modifyFlag($flag);
        }
    }

    sub sessionEmergencySave {

        # Called by GA::Client->doEmergencySave
        # Performs an emergency save, saving all file objects for this session in the specified
        #   directory (which is usually different from the normal Axmud data directory)
        #
        # Expected arguments
        #   $dir    - The directory path in which to save files
        #
        # Return values
        #   'undef' on improper arguments or if any of the save attempts fail (this function
        #       continues saving file objects, even if a save attempt fails)
        #   1 on success

        my ($self, $dir, $check) = @_;

        # Local variables
        my $errorFlag;

        # Check for improper arguments
        if (! defined $dir || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sessionEmergencySave', @_);
        }

        # Create a sub-directory for this world
        $dir .= '/worlds/' . $self->currentWorld->name;

        # Save each file object in turn
        foreach my $fileObj ($self->ivValues('sessionFileObjHash')) {

            # ('worldprof' files have already been saved by the calling function)
            if ($fileObj->fileType ne 'worldprof') {

                if (
                    ! $fileObj->saveDataFile(
                        $fileObj->actualFileName,
                        $dir . '/' . $fileObj->actualFileName,
                        $dir,
                        # The TRUE flag tells the function that an emergency save is in progress
                        TRUE,
                    )
                ) {
                    $errorFlag = TRUE;
                }
            }
        }

        if ($errorFlag) {

            # At least one save attempt failed
            return undef;

        } else {

            # All save attempts succeeded
            return 1;
        }
    }

    # Login

    sub setupLogin {

        # Called by $self->start
        # If the world profile specifies an automatic login (i.e. GA::Profile::World->loginMode is
        #   not set to 'none'), initiate the login process
        # The login process is completed by $self->doLogin
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the setup process fails
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my (
            $mode, $name, $packageName, $newTask, $rawScriptObj, $path, $scriptObj, $missionObj,
            $cloneObj, $msg,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupLogin', @_);
        }

        # Import the world profile's login IVs
        $mode = $self->currentWorld->loginMode;
        $name = $self->currentWorld->loginObjName;

        # Mode 'immediate' - immediate login (character marked as 'logged in' as soon as the
        #   connection is established - not recommended, as tasks like the Locator task will start
        #   looking for room descriptions in the world's introductory text)
        # This is performed when $self->connectionComplete is called

        # Mode 'lp' - LP/Diku/AberMUD login (consecutive prompts for character/password)
        # Mode 'tiny' - TinyMUD login (send 'connect char pass' at first prompt)
        # Mode 'world_cmd' - send a sequence of world commands at the first prompt
        # Mode 'telnet' - basic telnet login (e.g. 'login:' 'password:')
        if ($mode eq 'lp' || $mode eq 'tiny' || $mode eq 'world_cmd' || $mode eq 'telnet') {

            $self->ivPoke('loginPromptsMode', $mode);
            if (
                $mode eq 'tiny'
                # (in mode 'world_cmd', we look for a login pattern if one has been set; otherwise
                #   we wait for a prompt, as usual)
                || ($mode eq 'world_cmd' && $self->currentWorld->loginConnectPatternList)
            ) {
                $self->ivPoke(
                    'loginConnectPatternList',
                    $self->currentWorld->loginConnectPatternList,
                );
            }

        # Mode 'task' - run a task (character is logged in if when the task calls
        #   GA::Session->doLogin)
        } elsif ($mode eq 'task' && $self->currentWorld->loginObjName) {

            # $self->findTaskPackageName recognises unique names of currently running tasks (e.g.
            #   'status_task_57'), so before we consult it, check that $name isn't already running
            if ($self->ivExists('currentTaskHash', $name)) {

                $msg = '\'' . $name . '\' isn\'t a valid task name or task label';

            } else {

                # Get the task's package name (e.g. 'Games::Axmud::Task::Status')
                $packageName = Games::Axmud::Generic::Cmd->findTaskPackageName($self, $name);
                if (! $packageName) {

                    $msg = 'Could not start the task \'' . $name . '\'';

                } else {

                    # Create the task object
                    $packageName->new($self, 'current');
                }
            }

        # Mode 'script_task' - run an Axbasic script from within a task (character is logged in if
        #   the script executes a LOGIN statement)
        } elsif ($mode eq 'script_task' && $self->currentWorld->loginObjName) {

            # We need to check that the file containing the script exists, because the Script task
            #   won't pass us a convenient error return value. Create a dummy raw script to do it
            $rawScriptObj = Language::Axbasic::RawScript->new($self, $name);
            if (! $rawScriptObj) {

                $msg = 'Could not run the ' . $axmud::BASIC_NAME . ' script \'' . $name . '\'';

            } else {

                # Load the script into the raw script object
                $path = $axmud::DATA_DIR . '/scripts/' . $name . '.bas';
                if (! $rawScriptObj->loadFile($path)) {

                    $msg = 'Could not load the ' . $axmud::BASIC_NAME . ' script \'' . $name . '\'';

                } else {

                    # Create the task object
                    $newTask = Games::Axmud::Task::Script->new($self, 'current');
                    if (! $newTask) {

                        $msg = 'Could not start a Script task running the ' . $axmud::BASIC_NAME
                                    . ' script \'' . $name . '\'';

                    } else {

                        # Tell it which script to execute
                        $newTask->set_scriptName($name);
                    }
                }
            }

        # Mode 'script' - run an Axbasic script (character is logged in if the script executes a
        #   LOGIN statement)
        } elsif ($mode eq 'script' && $self->currentWorld->loginObjName) {

            # Create the raw script object
            $rawScriptObj = Language::Axbasic::RawScript->new($self, $name);
            if (! $rawScriptObj) {

                $msg = 'Could not run the ' . $axmud::BASIC_NAME . ' script \'' . $name . '\'';

            } else {

                # Load the script into the raw script object
                $path = $axmud::DATA_DIR . '/scripts/' . $name . '.bas';
                if (! $rawScriptObj->loadFile($path)) {

                    $msg = 'Could not load the ' . $axmud::BASIC_NAME . ' script \'' . $name . '\'';

                } else {

                    # Create a script object, which processes the raw script, removing extraneous
                    #   whitespace, empty lines, comments, etc
                    $scriptObj = Language::Axbasic::Script->new($self, $rawScriptObj);
                    if (! $scriptObj) {

                        $msg = 'Could not run the ' . $axmud::BASIC_NAME . ' script \'' . $name
                                    . '\'';

                    } else {

                        # Execute the script
                        $scriptObj->implement();
                    }
                }
            }

        # Mode 'mission' - start a mission (character is logged in if the mission uses the ';login'
        #   client command)
        } elsif ($mode eq 'mission' && $self->currentWorld->loginObjName) {

            # If the mission exists...
            if (! $self->currentWorld->ivExists('missionHash', $name)) {

                $msg = 'The mission \'' . $name . '\' doesn\'t exist';

            } else {

                $missionObj = $self->currentWorld->ivShow('missionHash', $name);
                # The mission object stored in the current world might be executed simultaneously by
                #   more than one session, so we create a clone of it, and store the clone in
                #   $self->currentMission
                $cloneObj = $missionObj->clone($self);
                if (! $cloneObj) {

                    $msg = 'The mission \'' . $name . '\' couldn\'t be started',

                } else {

                    $self->set_currentMission($cloneObj);
                    $self->ivPoke('loginPromptsMode', $mode);

                    # If the world profile's ->loginSpecialList is set, make a local copy of the
                    #   list
                    $self->ivPoke('loginSpecialList', $self->currentWorld->loginSpecialList);

                    # Start the mission. The TRUE flag means to refrain from displaying confirmation
                    #   messages
                    if (! $cloneObj->startMission(TRUE)) {

                        $msg = 'Could not start the mission \'' . $name . '\'';

                    } else {

                        # Automatically send the first group of commands (as if ';mission' had been
                        #   typed by the user
                        $cloneObj->continueMission($self);
                    }
                }
            }
        }

        if ($msg) {

            return $self->writeWarning(
                'Automatic login failed (' . $msg . ')',
            );

        } else {

            # Setup succeeded
            return 1;
        }
    }

    sub doLogin {

        # Called by $self->start, ->setupLogin, ->processLineSegment, ->setLoginPatterns and
        #   GA::Cmd::Login->do
        # Marks the character as logged in to the world. Updates IVs and starts all initial tasks
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the character is already marked as 'logged in'
        #   1 on sucess

        my ($self, $check) = @_;

        # Local variables
        my (
            $taskCount, $scriptCount, $cmdCount, $msg,
            @reportList, @sendList,
            %cmdHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doLogin', @_);
        }

        # If the character is already marked as 'logged in', do nothing
        if ($self->loginFlag) {

            return undef;
        }

        # Otherwise, mark the character as logged in
        $self->ivPoke('loginFlag', TRUE);

        # In login mode 'mission', terminate a mission that hasn't finished yet (usually because
        #   after a sudden disconnection, some worlds require fewer keypresses to complete the login
        #   process)
        if ($self->loginPromptsMode eq 'mission' && $self->currentMission) {

            $self->reset_currentMission();
        }

        # Reset IVs used to process the login
        $self->ivPoke('loginPromptsMode', 'none');
        $self->ivEmpty('loginSuccessPatternList');
        $self->ivEmpty('loginPromptPatternList');
        $self->ivEmpty('loginConnectPatternList');
        $self->ivPoke('loginConnectFoundFlag', FALSE);
        $self->ivEmpty('loginSpecialList');
        $self->ivUndef('loginWarningTime');

        # Update the current world profile's connection variables (but not while in 'offline' mode
        #   or during an MXP crosslinking operation)
        if ($self->status ne 'offline' && $self->mxpRelocateMode eq 'none') {

            $self->currentWorld->ivPoke('lastConnectDate', $axmud::CLIENT->localDate());
            $self->currentWorld->ivPoke('lastConnectTime', $axmud::CLIENT->localClock());
            $self->currentWorld->ivIncrement('numberConnects');
            if ($self->currentChar) {

                $self->currentWorld->ivPoke('lastConnectChar', $self->currentChar->name);
            }
        }

        # If MSDP is enabled, we ask the server to REPORT any reportable variables that it supports,
        #   and to SEND any sendable variables that it supports
        if (
            $axmud::CLIENT->useMsdpFlag
            && ! $self->currentWorld->ivExists('telnetOverrideHash', 'msdp')
        ) {
            foreach my $var ($self->ivKeys('msdpGenericReportableFlagHash')) {

                if ($self->ivShow('msdpGenericReportableFlagHash', $var)) {

                    push (@reportList, $var);
                }
            }

            if (@reportList) {

                $self->optSendMsdpReport(@reportList);
            }

            foreach my $var ($self->ivKeys('msdpGenericSendableFlagHash')) {

                if ($self->ivShow('msdpGenericSendableFlagHash', $var)) {

                    push (@sendList, $var);
                }
            }

            if (@sendList) {

                $self->optSendMsdpSend(@sendList);
            }
        }

        # Send details about the preferred terminal size, if allowed
        if ($self->currentWorld->sendSizeInfoFlag && $self->status ne 'offline') {

            if ($self->currentWorld->columns) {

                $self->sendModCmd('cols', 'number', $self->currentWorld->columns);
            }

            if ($self->currentWorld->rows) {

                $self->sendModCmd('rows', 'number', $self->currentWorld->rows);
            }
        }

        # If the Automapper window is due to open automatically, open it (but not during an MXP
        #   crosslinking operation)
        if (
            $self->worldModelObj->autoOpenWinFlag
            && $self->mapObj
            && ! $self->mapWin
            && $self->mxpRelocateMode eq 'none'
        ) {
            $self->mapObj->openWin();
        }

        # Deal with initial tasks, scripts and missions (except during an MXP crosslinking
        #   operation)
        if ($self->mxpRelocateMode eq 'none') {

            # Copy tasks stored in the client's initial tasklist and each current profile's initial
            #   tasklist into the current tasklist
            $taskCount = $self->startInitTasks(TRUE);

            # Perform one spin of the task loop to allow tasks which are waiting to initialise to do
            #   so before we start sending world commands
            $self->spinSessionLoop($self->sessionLoopObj, 'task');

            # Start scripts from the client's initial scriptlist and each current profile's initial
            #   scriptlist
            $scriptCount = $self->startInitScripts(TRUE);

            # Start one (and only one) initial mission. Check current profiles in priority order,
            #   and start the first mission found
            $msg = '';
            OUTER: foreach my $category ($self->profPriorityList) {

                my $profObj;

                if ($self->ivExists('currentProfHash', $category)) {

                    $profObj = $self->ivShow('currentProfHash', $category);

                    if (
                        $profObj->initMission
                        && $self->pseudoCmd('startmission ' . $profObj->initMission)
                    ) {
                        # Only start one mission
                        $msg = ' started initial mission \'' . $profObj->initMission . '\',';
                        last OUTER;
                    }
                }
            }
        }

        # Execute initial commands like 'look', 'score', if specified. Client, echo and perl
        #   commands can also be executed at this point. Each profile carries its own list of
        #   commands; send current profiles in priority order, but don't send duplicate commands
        $cmdCount = 0;
        OUTER: foreach my $category ($self->profPriorityList) {

            my $profObj;

            if ($self->ivExists('currentProfHash', $category)) {

                $profObj = $self->ivShow('currentProfHash', $category);

                INNER: foreach my $cmd ($profObj->initCmdList) {

                    # Don't send duplicate commands
                    if (! exists $cmdHash{$cmd}) {

                        $self->doInstruct($cmd);
                        $cmdHash{$cmd} = undef;
                        $cmdCount++;
                    }
                }
            }
        }

        # Fire any hooks that are using the 'login' hook event (except during an MXP crosslinking
        #   operation)
        if ($self->mxpRelocateMode eq 'none') {

            $self->checkHooks('login');
        }

        # Prepare confirmation. For the convenience of visually-impaired readers, don't read
        #   out the initial tasks, scripts and commands if system messages are being converted to
        #   speech
        # The message itself is not displayed until the end of the call to ->spinIncomingLoop (this
        #   prevents the message interrupting a block of text received from the world)
        if (! $self->mxpRelocateQuietFlag) {

            if ($axmud::CLIENT->systemAllowTTSFlag && $axmud::CLIENT->ttsSystemFlag) {

                $self->ivPoke('loginConfirmText', 'Automatic login complete');

            } elsif (! $taskCount && ! $scriptCount && ! $cmdCount) {

                $self->ivPoke(
                    'loginConfirmText',
                    'Automatic login complete (no initial tasks, scripts or commands)',
                );

            } else {

                $self->ivPoke(
                    'loginConfirmText',
                    'Automatic login complete (initial tasks: ' . $taskCount . ', initial scripts: '
                    . $scriptCount . $msg . ', initial commands: ' . $cmdCount . ')',
                );
            }
        }

        # Execute the client command ';test', if the global flag is set (but not during an MXP
        #   crosslinking operation)
        if (
            $axmud::TEST_MODE_FLAG
            && $axmud::TEST_MODE_CMD_FLAG
            && $self->mxpRelocateMode eq 'none'
        ) {
            $self->pseudoCmd('test');
        }

        # Terminate an MXP crosslinking operation, if one is in progress
        if ($self->mxpRelocateMode ne 'none') {

            $self->ivPoke('mxpRelocateMode', 'none');
            $self->ivUndef('mxpRelocateHost');
            $self->ivUndef('mxpRelocatePort');
            $self->ivPoke('mxpRelocateQuietFlag', FALSE);
            $self->ivPoke('mxpRelocateQuietLineFlag', FALSE);
        }

        return 1;
    }

    sub startInitTasks {

        # Called by $self->doLogin, $self->setupProfiles and GA::Generic::Cmd->setProfile
        #
        # Starts the tasks specified in the initial tasklist of each profile in a list supplied by
        #   the calling function, in the right order
        # So, if two profiles are supplied and their initial tasklists both contain a Status task,
        #   the Status task already started is the one in the first profile's initial tasklist (we
        #   cannot have two Status tasks running at the same time because their jealousy flags are
        #   set to TRUE)
        # For two tasks of the same type, whose jealousy flag is set to FALSE, they will both be
        #   created (in the order of the supplied list)
        # (The profile list supplied should already be in priority order, with the highest-priority
        #   profile first)
        #
        # If the supplied list is empty, the initial tasklists from all current profiles are used
        #   in their usual priority order
        # The flag is set to TRUE when called by $self->doLogin, in which case this function first
        #   starts tasks in the GA::Client's initial tasklist (which takes priority over any initial
        #   tasks stored by profiles)
        #
        # Expected arguments
        #   $globalFlag - If set to TRUE, starts tasks in the GA::Client's initial tasklist first
        #
        # Optional arguments
        #   @profList   - A list of profile objects in priority order (highest priority first). The
        #                   list can contain 0 or 1 items without causing an error
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns the number of initial tasks created (may be 0)

        my ($self, $globalFlag, @profList) = @_;

        # Local variables
        my (
            $count, $failCount, $msg,
            @taskList,
            %taskHash,
        );

        # Check for improper arguments
        if (! defined $globalFlag) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startInitTasks', @_);
        }

        $count = 0;
        $failCount = 0;

        # If @profList is empty, populate it with all current profiles in their usual priority order
        if (! @profList) {

            foreach my $category ($self->profPriorityList) {

                if ($self->ivExists('currentProfHash', $category)) {

                    push (@profList, $self->ivShow('currentProfHash', $category));
                }
            }
        }

        # If the flag is specified, use every task in the GA::Client's initial tasklist
        if ($globalFlag) {

            foreach my $name ($axmud::CLIENT->initTaskOrderList) {

                push (@taskList, $axmud::CLIENT->ivShow('initTaskHash', $name));
            }
        }

        # Go through all the initial tasklists, loading initial tasks into a hash in the form
        #   $taskHash{task_formal_name} = blessed_reference_to_task_object
        # If an initial task already exists in the hash, what we do depends on the task's jealousy
        #   flag.
        # If it's set to TRUE, ignore that task, preserving the task already in the hash. If
        #   it's set to FALSE, add the task to @taskList (and make sure there's an entry in the
        #   hash for that type of task)
        OUTER: foreach my $profObj (@profList) {

            # If this is a current profile, and its initial tasklist exists...
            if (
                $self->ivExists('currentProfHash', $profObj->category)
                && $self->ivShow('currentProfHash', $profObj->category) eq $profObj
                && $profObj->initTaskHash
            ) {
                # Add each task to @taskList
                foreach my $name ($profObj->initTaskOrderList) {

                    my $taskObj = $profObj->ivShow('initTaskHash', $name);

                    if (! $taskObj->jealousyFlag || ! exists $taskHash{$taskObj->name}) {

                        push (@taskList, $taskObj);
                        $taskHash{$taskObj->name} = undef;  # Replaces an existing entry

                    } else {

                        # Initial task won't be started
                        $failCount++;
                    }
                }
            }
        }

        OUTER: foreach my $taskObj (@taskList) {

            # If the task code is in a plugin, but the plugin didn't load, then the blessed task
            #   object $taskObj will exist, but the code (including all of its accessors) isn't
            #   available
            # (In other words, $taskObj->{name} works, but $taskObj->name doesn't)
            # In this situation, Axmud produces a 'Can't locate object method "jealousyFlag"' error,
            #   which is a bit confusing for the user. So we'll check the task is actually available
            #   right now
            if (! $axmud::CLIENT->ivExists('taskPackageHash', $taskObj->{name})) {

                $failCount++;
                next OUTER;
            }

            # If ->jealousyFlag is set to TRUE and there's already a task of this type in the
            #   current tasklist, don't bother trying to start it
            if ($taskObj->jealousyFlag) {

                foreach my $otherTaskObj ($self->ivValues('currentTaskHash')) {

                    if ($otherTaskObj->name eq $taskObj->name) {

                        $failCount++;
                        next OUTER;
                    }
                }
            }

            # Clone the initial task, and place the cloned copy into the current tasklist
            if (! $taskObj->clone($self, 'current', $taskObj->profName, $taskObj->profCategory)) {
                $failCount++;
            } else {
                $count++;
            }
        }

        if ($globalFlag && $failCount) {

            # Called by $self->doLogin, so display the results
            if ($failCount == 1) {
                $msg = '1 initial task';
            } else {
                $msg = $failCount . ' initial tasks';
            }

            $self->writeWarning(
                'Could not copy ' . $msg . ' into the current tasklist',
                $self->_objClass . '->startInitTasks',
            );
        }

        # Return the number of tasks created (may be 0)
        return $count;
    }

    sub startInitScripts {

        # Called by $self->doLogin, $self->setupProfiles and GA::Generic::Cmd->setProfile
        #
        # Starts the scripts specified in the initial scriptlist of each profile in a list supplied
        #   by the calling function, in the right order
        # (The profile list supplied should already be in priority order, with the highest-priority
        #   profile first)
        # Each script is only started once - duplicate scripts are ignored
        #
        # If the supplied list is empty, the initial scriptlists from all current profiles are used
        #   in their usual priority order
        # The flag is set to TRUE when called by $self->doLogin, in which case this function first
        #   starts scripts in the GA::Client's initial scriptlist
        #
        # Expected arguments
        #   $globalFlag - If set to TRUE, starts tasks in the GA::Client's initial scriptlist first
        #
        # Optional arguments
        #   @profList   - A list of profile objects in priority order (highest priority first). The
        #                   list can contain 0 or 1 items without causing an error
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns the number of initial scripts created (may be 0)

        my ($self, $globalFlag, @profList) = @_;

        # Local variables
        my (
            $count, $failCount, $msg,
            @list,
        );

        # Check for improper arguments
        if (! defined $globalFlag) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startInitScripts', @_);
        }

        # If @profList is empty, populate it with all current profiles in their usual priority order
        if (! @profList) {

            foreach my $category ($self->profPriorityList) {

                if ($self->ivExists('currentProfHash', $category)) {

                    push (@profList, $self->ivShow('currentProfHash', $category));
                }
            }
        }

        # If the flag is specified, use every script in the GA::Client's initial scriptlist
        if ($globalFlag) {

            foreach my $script ($axmud::CLIENT->initScriptOrderList) {

                # @list is in the form (script_name, mode, 'undef', ...)
                push (@list, $script, $axmud::CLIENT->ivShow('initScriptHash', $script), undef);
            }
        }

        # Go through all the profiles' initial scriptlists, adding initial scripts as we find them
        OUTER: foreach my $profObj (@profList) {

            # If this is a current profile, and its initial scriptlist exists...
            if (
                $self->ivExists('currentProfHash', $profObj->category)
                && $self->ivShow('currentProfHash', $profObj->category) eq $profObj
                && $profObj->initScriptHash
            ) {
                foreach my $script ($profObj->initScriptOrderList) {

                    # @list is in the form (script_name, mode, profile_object, ...)
                    push (@list, $script, $profObj->ivShow('initScriptHash', $script), $profObj);
                }
            }
        }

        $count = 0;
        $failCount = 0;

        if (@list) {

            do {
                my ($script, $mode, $profObj, $newTask);

                $script = shift @list;
                $mode = shift @list;
                $profObj = shift @list;

                if ($mode eq 'no_task') {

                    # Mode 'no_task': Run without task
                    if (! $self->pseudoCmd('runscript ' . $script)) {
                        $failCount++;
                    } else {
                        $count++;
                    }

                } else {

                    # Mode 'run_task': Run with task
                    # Mode 'run_task_win': Run with task in 'forced window' mode

                    # Create the task object
                    if ($profObj) {

                        $newTask = Games::Axmud::Task::Script->new(
                            $self,
                            'current',
                            $profObj->name,
                            $profObj->category,
                        );

                    } else {

                        $newTask = Games::Axmud::Task::Script->new($self, 'current');
                    }

                    if (! $newTask) {

                        $failCount++;

                    } else {

                        $count++;

                        # Set the task's paramenters
                        $newTask->set_scriptName($script);
                        if ($mode eq 'run_task_win') {

                            $newTask->ivPoke('forcedWinFlag', TRUE);
                        }
                    }
                }

            } until (! @list);
        }

        if ($globalFlag && $failCount) {

            # Called by $self->doLogin, so display the results
            if ($failCount == 1) {
                $msg = '1 initial script';
            } else {
                $msg = $failCount . ' initial scripts';
            }

            $self->writeWarning(
                'Could not start ' . $msg,
                $self->_objClass . '->startInitScripts',
            );
        }

        # Return the number of scripts created (may be 0)
        return $count;
    }

    sub processCmdLoginMode {

        # Called by $self->processPrompt or ->spinMaintainLoop in login mode 'world_cmd'
        # Sends a series of login commands, substituting @account@, @name@ and @password@ in any
        #   commands that contain them
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $initChar, $initAccount, $initPass,
            @loginCmdList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processCmdLoginMode', @_);
        }

        # Import the list of commands to send
        @loginCmdList = $self->currentWorld->loginCmdList;
        # Prepare to substitute '@account@, '@name@' and '@password@' in any commands that
        #   contain them
        $initChar = $self->initChar;
        if (! $initChar) {

            # We will use '@name@' literally, if character name not set
            $initChar = '';
        }

        $initAccount = $self->initAccount;
        if (! $initAccount) {

            $initAccount = '';     # Likewise for '@account@'
        }

        $initPass = $self->initPass;
        if (! $initPass) {

            $initPass = '';     # Likewise for '@password@'
        }

        foreach my $cmd (@loginCmdList) {

            $cmd =~ s/\@name\@/$initChar/;
            $cmd =~ s/\@account\@/$initAccount/;
            $cmd =~ s/\@password\@/$initPass/;
            if ($cmd) {

                $self->worldCmd($cmd);
            }
        }

        # Wait for login success patterns (if there are any), otherwise mark the character
        #   as logged in
        $self->setLoginPatterns();

        return 1;
    }

    sub setLoginPatterns {

        # Called by $self->processPrompt after sending a username and/or password to the world (in
        #   login modes 2-5)
        # If the world profile defines any login success patterns, we have to wait for them;
        #   otherwise, we can mark the character as 'logged in' right now
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setLoginPatterns', @_);
        }

        # If there are no login success patterns, the character should now be marked as
        #   logged in
        if (! $self->currentWorld->loginSuccessPatternList) {

            $self->ivPoke('loginPromptsMode', 'none');
            $self->doLogin();

        } else {

            # Wait for one of the patterns to appear
            $self->ivPoke('loginSuccessPatternList', $self->currentWorld->loginSuccessPatternList);
        }

        return 1;
    }

    # Profiles

    sub setupProfiles {

        # Called by $self->start, GA::Cmd::SetWorld->do
        # Sets up current profiles, together with associated cages and dictionaries. For profiles
        #   that already exist, read data from files. For profiles that don't yet exist, create them
        #
        # Expected arguments
        #   $mode       - Set to one of the following values:
        #                   'start' - called by $self->start
        #                   'start_temp' - called by $self->start for a temporary world profile
        #                   'set_exist' - called by ';setworld' for an existing world profile
        #                   'set_new' - called by ';setworld' for a new world profile
        #   $worldName  - The name of the world profile to be set as the current world profile. If
        #                   the profile doesn't exist, it is created
        #
        # Optional arguments
        #   $charName   - The name of the character profile to be set as the current character
        #                   profile. If 'undef', no current character profile is set. If the
        #                   profile doesn't exist, it is created.
        #   $host, $port, $loginMode
        #               - When called by $self->start, the host/port/login mode. If the $worldName
        #                   profile has to be be created, the profile has its host, port and login
        #                   mode set to these values. Otherwise, all set to 'undef'
        #
        # Return values
        #   'undef' on improper arguments, or if there any errors reading/writing the files, or if
        #       file loading/saving isn't allowed (because of global flags), or if the user chooses
        #       the 'cancel' button when asked about saving files
        #   1 otherwise

        my ($self, $mode, $worldName, $charName, $host, $port, $loginMode, $check) = @_;

        # Local variables
        my (
            $worldProfFileObj, $otherProfFileObj, $worldModelFileObj, $dictsFileObj, $result,
            $saveMode, $worldObj, $dictName, $dictObj, $worldModelObj, $updateCageFlag,
            $otherSession, $mapObj, $basicObj, $dialogueWin, $otherProfPath, $worldModelPath,
            %newHash,
        );

        # Check for improper arguments
        if (
            ! defined $mode
            || (
                $mode ne 'start' && $mode ne 'start_temp' && $mode ne 'set_exist'
                && $mode ne 'set_new'
            )
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupProfiles', @_);
        }

        # PART 1    - deal with existing profiles
        #
        # Mode 'set_exist'/'set_new' :
        #   There is a current world, so if any data stored in the corresponding 'worldprof' file
        #       and the related 'otherprof'/'worldmodel' files have been changed, ask the user's
        #       permission to save the files
        # Reset this session's profile and cage registries
        if ($mode eq 'set_exist' || $mode eq 'set_new') {

            # Get the relevant file objects
            $worldProfFileObj = $self->ivShow('sessionFileObjHash', $self->currentWorld->name);
            $otherProfFileObj = $self->ivShow('sessionFileObjHash', 'otherprof');
            $worldModelFileObj = $self->ivShow('sessionFileObjHash', 'worldmodel');

            # Ask permission from the user to save all the profile-related data in memory (if it has
            #   been modified)
            if (
                ! $self->currentWorld->noSaveFlag
                && (
                    $worldProfFileObj->modifyFlag
                    || $otherProfFileObj->modifyFlag
                    || $worldModelFileObj->modifyFlag
                )
            ) {
                $result = $self->mainWin->showMsgDialogue(
                    'New world profile',
                    'question',
                    'Save all data related to the current world profile? (If you select \'no\','
                    . ' the changes you have made to this data will be lost)',
                    'yes-no',
                );

                if ($result eq 'yes') {

                    # Save the current world profile's file
                    if ($worldProfFileObj->modifyFlag && ! $worldProfFileObj->saveDataFile()) {

                        return $self->writeError(
                            'Could not save the current world profile to file - new current world'
                            . ' profile not set',
                            $self->_objClass . '->setupProfiles',
                        );
                    }

                    # Save its 'otherprof' and 'worldmodel' files
                    if ($otherProfFileObj->modifyFlag && ! $otherProfFileObj->saveDataFile()) {

                        return $self->writeError(
                            'Could not save profiles to file - new current world profile not set',
                            $self->_objClass . '->setupProfiles',
                        );
                    }

                    if ($worldModelFileObj->modifyFlag && ! $worldModelFileObj->saveDataFile()) {

                        return $self->writeError(
                            'Could not save world model to file - new current world profile not'
                            . ' set',
                            $self->_objClass . '->setupProfiles',
                        );
                    }
                }
            }

            # Remove any tasks in the current tasklist which were created from initial tasks/scripts
            #   belonging to one the old current profiles
            $self->haltProfileTasks();
            # Reset (remove) all interfaces from this session
            $self->resetProfileInterfaces();

            # Reset this session's registries
            $self->ivPoke('profPriorityList', $axmud::CLIENT->constProfPriorityList);
            $self->ivEmpty('templateHash');
            $self->ivEmpty('profHash');
            $self->ivEmpty('currentProfHash');
            $self->ivUndef('currentWorld');
            $self->ivUndef('currentGuild');
            $self->ivUndef('currentRace');
            $self->ivUndef('currentChar');

            $self->ivEmpty('cageHash');
            $self->ivEmpty('currentCageHash');
            $self->ivEmpty('inferiorCageHash');

            $self->ivUndef('currentDict');

            $self->ivEmpty('sessionFileObjHash');

            $self->ivUndef('worldModelObj');

            # We need to inform tasks that the current profiles have changed
            $self->ivPoke('currentProfChangeFlag', TRUE);
        }

        # PART 2   - Create a new automapper object (all modes)
        $mapObj = Games::Axmud::Obj::Map->new($self);
        if (! $mapObj) {

            $self->writeWarning(
                'Could not create new automapper object',
                $self->_objClass . '->setupProfiles',
            );

        } else {

            # Update IVs
            $self->ivPoke('mapObj', $mapObj);
        }

        # PART 3    - Create new world profile
        #
        # Mode 'start'/'start_temp'/'set_new'
        #   If the world profile corresponding to $worldName doesn't yet exist, create one (and set
        #       up its cages, etc)
        if ($mode ne 'set_exist' && ! $axmud::CLIENT->ivExists('worldProfHash', $worldName)) {

            # Create a file object for the world profile
            $worldProfFileObj = Games::Axmud::Obj::File->new('worldprof', $worldName);
            # Create two file objects for the other two files objects associated with this
            #   profile, replacing the ones that we deleted earlier in this function
            $otherProfFileObj = Games::Axmud::Obj::File->new('otherprof', $worldName, $self);
            $worldModelFileObj = Games::Axmud::Obj::File->new('worldmodel', $worldName, $self);
            if (! $worldProfFileObj || ! $otherProfFileObj || ! $worldModelFileObj) {

                # (Improper arguments error already displayed)
                return $self->writeError(
                    'Could not set up current profiles - error creating file objects',
                    $self->_objClass . '->setupProfiles',
                );

            } else {

                # Add these file objects to the file object registries
                $self->ivAdd('sessionFileObjHash', $worldName, $worldProfFileObj);
                $self->ivAdd('sessionFileObjHash', 'otherprof', $otherProfFileObj);
                $self->ivAdd('sessionFileObjHash', 'worldmodel', $worldModelFileObj);
                # The GA::Client also stores the 'worldprof' file object
                $axmud::CLIENT->add_fileObj($worldProfFileObj);
                # Some files have to be saved shortly
                $saveMode = 'save';
            }

            # Create the new world profile
            $worldObj = Games::Axmud::Profile::World->new($self, $worldName);
            if (! $worldObj) {

                # (Error already displayed)
                return $self->writeError(
                    'Could not set up current profiles - error creating new world profile',
                    $self->_objClass . '->setupProfiles',
                );

            } else {

                # Mark it as the current profile
                $self->ivAdd('profHash', $worldObj->name, $worldObj);
                $self->ivAdd('currentProfHash', $worldObj->category, $worldObj);
                $self->ivPoke('currentWorld', $worldObj);

                # Also add the world profile to the client's list of world profiles
                $axmud::CLIENT->add_worldProf($worldObj);

                # In mode 'start_temp', it's a temporary world profile. Implement this by settings
                #   its ->noSaveFlag
                if ($mode eq 'start_temp') {

                    $worldObj->ivPoke('noSaveFlag', TRUE);
                }

                # Set the host/port/login mode, if supplied as arguments
                if (defined $host) {

                    if ($axmud::CLIENT->ipv6Check($host)) {
                        $worldObj->ivPoke('ipv6', $host);
                    } elsif ($axmud::CLIENT->ipv4Check($host)) {
                        $worldObj->ivPoke('ipv4', $host);
                    } else {
                        $worldObj->ivPoke('dns', $host);
                    }
                }

                if (defined $port) {

                    $worldObj->ivPoke('port', $port);
                }

                if (defined $loginMode) {

                    $worldObj->ivPoke('loginMode', $loginMode);
                }

                # If this world profile has been created from the Connections window, using a world
                #   from the basic mudlist, use the latter's remaining IVs
                $basicObj = $axmud::CLIENT->ivShow('constBasicWorldHash', $worldObj->name);
                if ($basicObj) {

                    $worldObj->ivPoke('longName', $basicObj->longName);
                    $worldObj->ivPoke('adultFlag', $basicObj->adultFlag);
                }
            }

            # (The rest of this section is re-used in part 4; change this one, change that one)
            # Create the new world profile's associated cage
            if (! $self->createCages($worldObj, TRUE)) {

                # Some objects couldn't be created. Destroy any newly-created cages, if any
                $self->destroyCages($worldObj, TRUE);

                # Unset IVs set above
                $self->ivDelete('profHash', $worldObj->name);
                $self->ivDelete('currentProfHash', $worldObj->category);
                $self->ivUndef('currentWorld');
                $axmud::CLIENT->del_worldProf($worldObj);

                # (Improper arguments error already displayed)
                return $self->writeError(
                    'Could not set up current profiles - error creating cages',
                    $self->_objClass . '->setupProfiles',
                );
            }

            # Create a dictionary for the profile (unless a dictionary with the same name as the
            #   profile already exists). However, for temporary profiles, use (or create) a
            #   dictionary called 'temp_dict'
            if ($mode eq 'start_temp') {
                $dictName = 'temp_dict';
            } else {
                $dictName = $worldName;
            }

            if (! $axmud::CLIENT->ivExists('dictHash', $dictName)) {

                $dictObj = Games::Axmud::Obj::Dict->new($self, $dictName);
                if (! $dictObj) {

                    # (Error already displayed)
                    return $self->writeError(
                        'Could not set up current profiles - error creating dictionary',
                        $self->_objClass . '->setupProfiles',
                    );

                } else {

                    # Update IVs. Add the dictionary object to its registry
                    $axmud::CLIENT->add_dict($dictObj);
                    # Some files have to be saved shortly
                    $saveMode = 'save_dict';
                }

            } else {

                # The dictionary already exists
                $dictObj = $axmud::CLIENT->ivShow('dictHash', $dictName);
            }

            # Make this dictionary as the current dictionary
            $worldObj->ivPoke('dict', $dictName);
            $self->ivPoke('currentDict', $dictObj);

            # Create a world model for the profile
            $worldModelObj = Games::Axmud::Obj::WorldModel->new($self);
            if (! $worldModelObj) {

                # Display a warning, but don't give up
                $self->writeWarning(
                    'Could not create a world model',
                    $self->_objClass . '->setupProfiles',
                );

            } else {

                # Update IVs
                $self->ivPoke('worldModelObj', $worldModelObj);
            }

            # Create a directory in /logs especially for this world, first deleting the old
            #   directory if the global flag is set
            if ($axmud::CLIENT->deleteWorldLogsFlag) {

                $axmud::CLIENT->deleteWorldLogDir($worldName);
            }

            $axmud::CLIENT->createWorldLogDir($worldName);

            # Change the profile's flag to show it has been set up for the first time
            $worldObj->ivPoke('setupCompleteFlag', TRUE);

        } elsif ($mode ne 'set_new' && $axmud::CLIENT->ivExists('worldProfHash', $worldName)) {

            $worldObj = $axmud::CLIENT->ivShow('worldProfHash', $worldName);

            if (! $worldObj->setupCompleteFlag) {

                # PART 4    - Setup an existing world profile
                #
                # Mode 'start'/'start_temp'/'set_exist'
                #   If the world profile corresponding to $worldName exists but has never been a
                #       current profile (probably because it was created with the ';addworld'
                #       command, set it up

                # Create two file objects for the two files objects associated with this profile,
                #   replacing the ones that we deleted earlier in this function
                $otherProfFileObj = Games::Axmud::Obj::File->new('otherprof', $worldName, $self);
                $worldModelFileObj = Games::Axmud::Obj::File->new('worldmodel', $worldName, $self);
                if (! $otherProfFileObj || ! $worldModelFileObj) {

                    # (Improper arguments error already displayed)
                    return $self->writeError(
                        'Could not set up current profiles - error creating file objects',
                        $self->_objClass . '->setupProfiles',
                    );

                } else {

                    # Add these file objects to the file object registries
                    $self->ivAdd('sessionFileObjHash', 'otherprof', $otherProfFileObj);
                    $self->ivAdd('sessionFileObjHash', 'worldmodel', $worldModelFileObj);
                    # The world profile's file object must appear in the GA::Client's registry
                    #   and the GA::Session's registry too
                    $self->ivAdd(
                        'sessionFileObjHash',
                        $worldName,
                        $axmud::CLIENT->ivShow('fileObjHash', $worldName),
                    );

                    # Some files have to be saved shortly
                    $saveMode = 'save';
                }

                # Mark this world profile as the current one
                $self->ivAdd('profHash', $worldObj->name, $worldObj);
                $self->ivAdd('currentProfHash', $worldObj->category, $worldObj);
                $self->ivPoke('currentWorld', $worldObj);

                # (The rest of this section is the same as the corresponding section of part 4)
                # Create this world profile's associated cage
                if (! $self->createCages($worldObj, TRUE)) {

                    # (Improper arguments error already displayed)
                    return $self->writeError(
                        'Could not set up current profiles - error creating cages',
                        $self->_objClass . '->setupProfiles',
                    );
                }

                # Create a dictionary for the profile (unless a dictionary with the same name as the
                #   profile already exists)
                if (! $axmud::CLIENT->ivExists('dictHash', $worldName)) {

                    $dictObj = Games::Axmud::Obj::Dict->new($self, $worldName);
                    if (! $dictObj) {

                        # (Error already displayed)
                        return $self->writeError(
                            'Could not set up current profiles - error creating dictionary',
                            $self->_objClass . '->setupProfiles',
                        );

                    } else {

                        # Update IVs. Add the dictionary object to its registry
                        $axmud::CLIENT->add_dict($dictObj);
                        # Some files have to be saved shortly
                        $saveMode = 'save_dict';
                    }

                } else {

                    # The dictionary already exists
                    $dictObj = $axmud::CLIENT->ivShow('dictHash', $worldName);
                }

                # Make this dictionary as the current dictionary
                $worldObj->ivPoke('dict', $dictObj->name);
                $self->ivPoke('currentDict', $dictObj);

                # Create a world model for the profile
                $worldModelObj = Games::Axmud::Obj::WorldModel->new($self);
                if (! $worldModelObj) {

                    # Display a warning, but don't give up
                    $self->writeWarning(
                        'Could not create a world model',
                        $self->_objClass . '->setupProfiles',
                    );

                } else {

                    # Update IVs
                    $self->ivPoke('worldModelObj', $worldModelObj);
                }

                # Create a directory in /logs especially for this world, first deleting the old
                #   directory if the global flag is set
                if ($axmud::CLIENT->deleteWorldLogsFlag) {

                    $axmud::CLIENT->deleteWorldLogDir($worldName);
                }

                $axmud::CLIENT->createWorldLogDir($worldName);

                # Change the profile's flag to show it has been set up for the first time
                $worldObj->ivPoke('setupCompleteFlag', TRUE);

            # PART 5    - Use an already-loaded world profile
            #
            # Mode 'start'/'set_exist'
            #   If the world profile corresponding to $worldName is being used by a different
            #       session, this session needs to use the same Perl objects (and therefore, we
            #       don't need to load the 'worldprof', 'otherprof' and 'worldmodel' data files)
            } elsif (($otherSession) = $axmud::CLIENT->findSessions($worldObj->name, $self)) {

                # Copy the contents of the other session's registries directly into this session

                # Copy data that has already been loaded from the 'otherprof' file
                $self->ivPoke('profPriorityList', $otherSession->profPriorityList);
                $self->ivPoke('templateHash', $otherSession->templateHash);
                $self->ivPoke('profHash', $otherSession->profHash);
                $self->ivPoke('cageHash', $otherSession->cageHash);
                # (Only one current profile so far - the world profile)
                $self->ivAdd('currentProfHash', $worldObj->category, $worldObj);
                $self->ivPoke('currentWorld', $worldObj);

                # Copy data that has already been loaded from the 'worldmodel' file
                $self->ivPoke('worldModelObj', $otherSession->worldModelObj);

                # Copy the other session's file object registry
                $self->ivPoke('sessionFileObjHash', $otherSession->sessionFileObjHash);

                # Copy the other session's current dictionary
                $self->ivPoke('currentDict', $otherSession->currentDict);

            # PART 6    - Load data for an existing world profile
            #
            # Mode 'start'/'set_exist':
            #   If the world profile corresponding to $worldName exists and has been a current
            #       profile before, it already has things like cages, a world model, and so on. Load
            #       these things into memory from the 'otherprof' and 'worldmodel' files
            } else {

                # Mark this world profile as the current one
                $self->ivAdd('profHash', $worldObj->name, $worldObj);
                $self->ivAdd('currentProfHash', $worldObj->category, $worldObj);
                $self->ivPoke('currentWorld', $worldObj);

                # Create two file objects for the two files objects associated with this profile,
                #   replacing the ones that we deleted earlier in this function
                $otherProfFileObj = Games::Axmud::Obj::File->new('otherprof', $worldName, $self);
                $worldModelFileObj = Games::Axmud::Obj::File->new('worldmodel', $worldName, $self);
                if (! $otherProfFileObj || ! $worldModelFileObj) {

                    # (Improper arguments error already displayed)
                    return $self->writeError(
                        'Could not set up current profiles - error creating file objects',
                        $self->_objClass . '->setupProfiles',
                    );

                } else {

                    # Add these file objects to the file object registries
                    $self->ivAdd('sessionFileObjHash', 'otherprof', $otherProfFileObj);
                    $self->ivAdd('sessionFileObjHash', 'worldmodel', $worldModelFileObj);
                    # The world profile's file object must appear in the GA::Client's registry and
                    #   the GA::Session's registry too
                    $self->ivAdd(
                        'sessionFileObjHash',
                        $worldName,
                        $axmud::CLIENT->ivShow('fileObjHash', $worldName),
                    );

                    # Load the 'otherprof' and 'worldmodel' data files (which sets IVs such as the
                    #   cage registries). If the files are big, show a 'loading...' popup
                    $otherProfPath = $axmud::DATA_DIR . $otherProfFileObj->standardPath;
                    if (
                        (-e $otherProfPath)
                        && (-s $otherProfPath) > $axmud::CLIENT->constLargeFileSize
                    ) {
                        $dialogueWin = $self->mainWin->showBusyWin();
                    }

                    if (! $otherProfFileObj->loadDataFile()) {

                        # (Error already displayed)
                        return $self->writeError(
                            'Could not set up current profiles - error loading the \'otherprof\''
                            . ' file (file might be corrupt or missing)',
                            $self->_objClass . '->setupProfiles',
                        );
                    }

                    $worldModelPath = $axmud::DATA_DIR . $worldModelFileObj->standardPath;
                    if (
                        ! $dialogueWin
                        && (-e $worldModelPath)
                        && (-s $worldModelPath) > $axmud::CLIENT->constLargeFileSize
                    ) {
                        $dialogueWin = $self->mainWin->showBusyWin();
                    }

                    if (! $worldModelFileObj->loadDataFile()) {

                        # (Error already displayed)
                        return $self->writeError(
                            'Could not set up current profiles - error loading the'
                            . ' \'worldmodel\' file (file might be corrupt or missing)',
                            $self->_objClass . '->setupProfiles',
                        );
                    }

                    if ($dialogueWin) {

                        $self->mainWin->closeDialogueWin($dialogueWin);
                    }
                }

                # If the user has coded (in Perl) any new cages, the new current world profile
                #   doesn't know about them yet. Make sure cages are up to date
                $self->updateCages(TRUE);       # TRUE - don't display create/destroy messages
                $updateCageFlag = TRUE;

                # Set the current dictionary
                if ($worldObj->dict) {

                    if ($axmud::CLIENT->ivExists('dictHash', $worldObj->dict)) {

                        $dictObj = $axmud::CLIENT->ivShow('dictHash', $worldObj->dict);

                    } else {

                        # The dictionary is missing; this can happen when setting a pre-configured
                        #   world as the current world for the first time. Silently create a new
                        #   dictionary
                        $dictObj = Games::Axmud::Obj::Dict->new($self, $worldName);
                        if ($dictObj) {

                            # Update IVs. Add the dictionary object to its registry
                            $axmud::CLIENT->add_dict($dictObj);
                            # Some files have to be saved shortly
                            $saveMode = 'save_dict';
                        }
                    }

                } else {

                    # If there is, by any chance, a dictionary with the same name as the world, use
                    #   it
                    if ($axmud::CLIENT->ivExists('dictHash', $worldName)) {

                        $dictObj = $axmud::CLIENT->ivShow('dictHash', $worldName);
                        # Tell the profile that it has a dictionary
                        $worldObj->ivPoke('dict', $dictObj->name);
                    }
                }

                if (! $dictObj) {

                    $self->writeWarning(
                        'Could not mark a dictionary as the current dictionary - try creating one'
                        . ' with the \';setdictionary\' command',
                        $self->_objClass . '->setupProfiles',
                    );

                } else {

                    $self->ivPoke('currentDict', $dictObj);
                }

                # If loading a pre-configured world, a /logs directory won't exist for it, so
                #   create one
                $axmud::CLIENT->createWorldLogDir($worldObj->name);
            }
        }

        # PART 7    - Set up new characters/passwords
        #
        # All modes:
        #   Each world profile has a ->newPasswordHash IV. It is mainly set by the Connections
        #       window, where the user can specify new characters (with or without their passwords)
        #       and modify the passwords of existing characters. Implement those changes now
        %newHash = $self->currentWorld->newPasswordHash;
        OUTER: foreach my $char (keys %newHash) {

            my ($pass, $charObj);

            $pass = $newHash{$char};     # May be 'undef'

            # If a matching character profile doesn't already exist, create it (code adapted from
            #   GA::Generic::Cmd->addProfile)
            if (! $self->ivExists('profHash', $char)) {

                $charObj
                    = Games::Axmud::Profile::Char->new($self, $char, $self->currentWorld->name);

                if (! $charObj) {

                    $self->writeWarning(
                        'Could not add character profile \'' . $char . '\'',
                        $self->_objClass . '->setupProfiles',
                    );

                    next OUTER;
                }


                # Create the profile's associated cages
                if (! $self->createCages($charObj)) {

                    $self->writeWarning(
                        'Could not add character profile \'' . $char . '\'',
                        $self->_objClass . '->setupProfiles',
                    );

                    next OUTER;
                }

                # Update IVs
                $self->add_prof($charObj);
                # Tell the current world it's acquired a new associated definiton
                $self->currentWorld->ivAdd('profHash', $char, 'char');
            }

            # In either case, update the world profile's password hashes
            $self->currentWorld->ivAdd('passwordHash', $char, $pass);
        }

        # There is also a corresponding ->newAccountHash IV. Process it in the same way
        %newHash = $self->currentWorld->newAccountHash;
        OUTER: foreach my $char (keys %newHash) {

            my ($account, $charObj);

            $account = $newHash{$char};     # May be 'undef'

            # The matching character profile should already exist, having been created just above
            #   (if necessary), but we'll check anyway
            if (! $self->ivExists('profHash', $char)) {

                $charObj
                    = Games::Axmud::Profile::Char->new($self, $char, $self->currentWorld->name);

                if (! $charObj) {

                    $self->writeWarning(
                        'Missing character profile \'' . $char . '\', spotted when checking'
                        . ' associated account names',
                        $self->_objClass . '->setupProfiles',
                    );

                    next OUTER;
                }

            } else {

                # Character profile exists, as we expected, so update the account hash
                $self->currentWorld->ivAdd('accountHash', $char, $account);
            }
        }

        # All new characters/passwords/associated accounts processed
        $self->currentWorld->ivEmpty('newPasswordHash');
        $self->currentWorld->ivEmpty('newAccountHash');

        # PART 8    - Set up initial tasks/scripts
        #
        # Mode 'set_exist'/'set_new' :
        #   If the current world profile has any initial tasks or initial scripts, clone them into
        #       the current tasklist. The FALSE argument means 'don't consult the global initial
        #       tasklist/scriptlist'. (For modes 'start'/'start_temp', $self->doLogin calls
        #       $self->startInitTasks and $self->startInitScripts)
        if ($mode eq 'set_exist' || $mode eq 'set_new') {

            $self->startInitTasks(FALSE, $self->currentWorld);
            $self->startInitScripts(FALSE, $self->currentWorld);
        }

        # PART 9    - Set the current character (if specified) (all modes)
        if ($charName) {

            $self->pseudoCmd('setchar ' . $charName, 'hide_complete');
        }

        # PART 10    - Process cages (unless it's already been done by a call to $self->updateCages)
        #   (all modes)
        if (! $updateCageFlag) {

            foreach my $obj ($self->ivValues('currentProfHash')) {

                $self->setCurrentCages($obj->name, $obj->category);
            }

            $self->setCageInferiors();

            # Set up interfaces for this profile
            $self->setProfileInterfaces($worldName);
        }

        # PART 11   - save files, if necessary (all modes)
        if ($saveMode) {

            # Import file objects
            $worldProfFileObj = $self->ivShow('sessionFileObjHash', $worldName);
            $otherProfFileObj = $self->ivShow('sessionFileObjHash', 'otherprof');
            $worldModelFileObj = $self->ivShow('sessionFileObjHash', 'worldmodel');

            # Save files
            $axmud::CLIENT->configFileObj->saveConfigFile();
            $worldProfFileObj->saveDataFile();
            $otherProfFileObj->saveDataFile();
            $worldModelFileObj->saveDataFile();

            if ($saveMode eq 'save_dict') {

                $dictsFileObj = $axmud::CLIENT->ivShow('fileObjHash', 'dicts');
                $dictsFileObj->saveDataFile();
            }
        }

        # PART 12 - Update the automapper object (all modes)
        if ($self->mapObj) {

            # The world model has been loaded, and the automapper object requires it
            $mapObj->set_worldModelObj($self->worldModelObj);

            # Appropriate the existing Automapper window (belonging to the previous automapper
            #   object), if there is one
            if ($self->mapWin) {

                $mapObj->openWin();
            }
        }

        return 1;
    }

    sub findSuperiorList {

        # Can be called by anything
        # When supplied with a profile category (e.g. 'world', 'char', 'race', 'faction' etc),
        #   returns a list of all profile categories in $self->profPriorityList that have a higher
        #   priority
        #
        # Expected arguments
        #   $category   - The profile category to check
        #
        # Return values
        #   An empty list on improper arguments, if $category has the highest priority, or if the
        #       category isn't in the priority list
        #   Otherwise returns a list of profile categories with higher priority
        #       e.g. $self->profPriorityList = ('char', 'race', 'guild', 'world')
        #       e.g. If $category is 'guild', returns the list ('char', 'race')

        my ($self, $category, $check) = @_;

        # Local variables
        my (
            $match,
            @emptyList, @priorityList, @returnArray,
        );

        # Check for improper arguments
        if (! defined $category || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->findSuperiorList', @_);
            return @emptyList;
        }

        # Import IVs
        @priorityList = $self->profPriorityList;

        # Find $category's position in $self->profPriorityList
        OUTER: for (my $count = 0; $count < scalar @priorityList; $count++) {

            if ($category eq $priorityList[$count]) {

                $match = $count;
                last OUTER;
            }
        }

        if (defined $match) {

            # $category exists in $self->profPriorityList at index $match
            if ($match > 0) {

                for (my $count = 0; $count < $match; $count++) {

                    push @returnArray, $priorityList[$count];
                }
            }
        }

        # Return the list (which might be empty if $category isn't in the list, or if it is in the
        #   list and has the highest priority)
        return @returnArray;
    }

    sub findInferiorList {

        # Can be called by anything
        # When supplied with a profile category (e.g. 'world', 'char', 'race', 'faction' etc),
        #   returns a list of all profile categories in $self->profPriorityList that have a lower
        #   priority
        #
        # Expected arguments
        #   $category   - The profile category to check
        #
        # Return values
        #   An empty list on improper arguments, if $category has the lowest priority, or if the
        #       category isn't in the priority list
        #   Otherwise returns a list of profile categories with lower priority
        #       e.g. $self->profPriorityList = ('char', 'race', 'guild', 'world')
        #       e.g. If $category is 'race', returns the list ('guild', 'world')

        my ($self, $category, $check) = @_;

        # Local variables
        my (
            $match,
            @emptyList, @priorityList, @returnArray,
        );

        # Check for improper arguments
        if (! defined $category || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->findInferiorList', @_);
            return @emptyList;
        }

        # Import IVs
        @priorityList = $self->profPriorityList;

        # Find $category's position in $self->profPriorityList
        OUTER: for (my $count = 0; $count < scalar @priorityList; $count++) {

            if ($category eq $priorityList[$count]) {

                $match = $count;
                last OUTER;
            }
        }

        if (defined $match) {

            # $category exists in $self->profPriorityList at index $match
            if ($match < (scalar @priorityList - 1)) {

                for (my $count = ($match + 1); $count < scalar @priorityList; $count++) {

                    push @returnArray, $priorityList[$count];
                }
            }
        }

        # Return the list (which might be empty if $category isn't in the list, or if it is in the
        #   list and has the lowest priority)
        return @returnArray;
    }

    # Active interfaces

    sub setProfileInterfaces {

        # Called by $self->setupProfiles, ->createCages, ->updateCages and
        #   GA::Generic::Cmd->setProfile, GA::Cmd::SetCustomProfile->do
        # When a profile is set as the current profile, all of the (inactive) interfaces stored in
        #   the profile's cages must have active interfaces (stored in $self->interfaceHash) created
        #   for them
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $name     - The name of the current profile whose inactive interfaces must be made
        #                   active. If 'undef', all current profiles are processed
        #
        # Return values
        #   'undef' on improper arguments, if a specified profile doesn't exist or if the specified
        #       profile's category isn't in the profile priority list ($self->profPriorityList)
        #   Otherwise returns the number of active interfaces created (may be 0)

        my ($self, $name, $check) = @_;

        # Local variables
        my (
            $count,
            @interfaceList, @profList, @modList, @sortedList,
            %priorityHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setProfileInterfaces', @_);
        }

        @interfaceList = ('trigger', 'alias', 'macro', 'timer', 'hook');

        # Compile a list of current profiles to process
        if (defined $name) {

            # Check the specified profile exists
            if (! $self->ivExists('profHash', $name)) {

                return $self->writeError(
                    'Unrecognised profile \'' . $name . '\'',
                    $self->_objClass . '->setProfileInterfaces',
                );

            } else {

                push (@profList, $self->ivShow('profHash', $name));
            }

        } else {

            # Use all current profiles
            @profList = $self->ivValues('currentProfHash');
        }

        # Convert the profile priority list into a hash for quick lookup
        $count = 0;
        foreach my $category ($self->profPriorityList) {

            $count++;
            $priorityHash{$category} = $count;
        }

        # Remove any profiles of categories that don't appear on the priority list
        foreach my $obj (@profList) {

            if (exists $priorityHash{$obj->category}) {

                push (@modList, $obj);
            }
        }

        if (! @modList) {

            # The specified profile, $name, didn't survive the cut. (The priority list always
            #   includes 'world', and there is always a current world, so it can't be the case that
            #   $name wasn't specified by the calling function)
            return undef;
        }

        # Sort the list of profiles in priority order (highest priority first)
        @sortedList
            = sort {lc($priorityHash{$a->category}) cmp lc($priorityHash{$b->category})} (@modList);

        # Process each profile in turn
        $count = 0;
        OUTER: foreach my $profObj (@sortedList) {

            my (@superiorList, @inferiorList);

            # Get a list of profiles with higher priority than this one
            @superiorList = $self->findSuperiorList($profObj->category);
            # Get a list of profiles with lower priority than this one
            @inferiorList = $self->findInferiorList($profObj->category);

            MIDDLE: foreach my $type (@interfaceList) {  # 'trigger', 'alias', etc

                my (
                    $cage,
                    %intHash,
                );

                # Find the cage for this profile and interface type
                $cage = $self->findCage($type, $profObj->name);
                if ($cage) {

                    # Import the cage's list of inactive interfaces
                    %intHash = $cage->interfaceHash;

                    # For each inactive interface, if there are no active interfaces with the same
                    #   name belonging to cages associated with superior profiles, create an active
                    #   interface
                    # If there are active interfaces with the same name belonging to cages
                    #   associated with inferior profiles, destory those active interfaces
                    INNER: foreach my $interfaceName (keys %intHash) {

                        my $interfaceObj = $intHash{$interfaceName};
                        if (
                            $self->injectInterface(
                                $interfaceObj,
                                $interfaceName,
                                $profObj->name,         # Profile name
                                \@superiorList,
                                \@inferiorList,
                            )
                        ) {
                            $count++;
                        }
                    }
                }
            }
        }

        # Return the number of active interfaces created (may be 0)
        return $count;
    }

    sub resetProfileInterfaces {

        # Called by $self->setupProfiles, GA::Generic::Cmd->setProfile, ->unsetProfile,
        #   ->deleteProfile, GA::Cmd::SetCustomProfile->do, UnsetCustomProfile->do
        #
        # When a current profile is unset as a current profile, all of the active interfaces created
        #   for it must be destroyed
        # For each active interface that's destroyed, if there's an inactive interface with the same
        #   name, belonging to a cage associated with an inferior profile, an active interface must
        #   be created for it
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $name     - The name of the current profile whose active interfaces must be destroyed.
        #                   If 'undef', the active interfaces for all current profiles are destroyed
        #
        # Return values
        #   'undef' on improper arguments, if a specified profile doesn't exist or if the specified
        #       profile's category isn't in the profile priority list ($self->profPriorityList)
        #   Otherwise returns the number of active interfaces destroyed (may be 0)

        my ($self, $name, $check) = @_;

        # Local variables
        my (
            $count,
            @interfaceList, @profList, @modList, @sortedList,
            %priorityHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetProfileInterfaces', @_);
        }

        @interfaceList = ('trigger', 'alias', 'macro', 'timer', 'hook');

        # Compile a list of current profiles to process
        if (defined $name) {

            # Check the specified profile exists
            if (! $self->ivExists('profHash', $name)) {

                return $self->writeError(
                    'Unrecognised profile \'' . $name . '\'',
                    $self->_objClass . '->resetProfileInterfaces',
                );

            } else {

                push (@profList, $self->ivShow('profHash', $name));
            }

        } else {

            # Use all current profiles
            @profList = $self->ivValues('currentProfHash');
        }

        # Convert the profile priority list into a hash for quick lookup
        $count = 0;
        foreach my $category ($self->profPriorityList) {

            $count++;
            $priorityHash{$category} = $count;
        }

        # Remove any profiles of categories that don't appear on the priority list
        foreach my $obj (@profList) {

            if (exists $priorityHash{$obj->category}) {

                push (@modList, $obj);
            }
        }

        if (! @modList) {

            # The specified profile, $name, didn't survive the cut. (The priority list always
            #   includes 'world', and there is always a current world, so it can't be the case that
            #   $name wasn't specified by the calling function)
            return undef;
        }

        # Sort the list of profiles in priority order (highest priority first)
        @sortedList
            = sort {lc($priorityHash{$a->category}) cmp lc($priorityHash{$b->category})} (@modList);

        # Process each profile in turn
        $count = 0;
        OUTER: foreach my $profObj (@sortedList) {

            # Get a list of profiles with lower priority than this one
            my @inferiorList = $self->findInferiorList($profObj->category);

            MIDDLE: foreach my $type (@interfaceList) {  # 'trigger', 'alias', etc

                my (
                    $cage,
                    %intHash,
                );

                # Find the cage for this profile and interface type
                $cage = $self->findCage($type, $profObj->name);
                if ($cage) {

                    # Import the cage's list of inactive interfaces
                    %intHash = $cage->interfaceHash;

                    # For each inactive interface, if there's an active interface based on it,
                    #   destroy the active interface
                    # At the same time, if there's an inactive interface with the same name
                    #   belonging to a cage associated with an inferior profile, create an active
                    #   interface for it
                    INNER: foreach my $interfaceName (keys %intHash) {

                        my ($interfaceObj, $result);

                        $interfaceObj = $intHash{$interfaceName};
                        $result = $self->recallInterface(
                            $interfaceObj,
                            $interfaceName,
                            \@inferiorList,
                        );

                        if ($result) {

                            $count++;
                        }
                    }
                }
            }
        }

        # Return the number of active interfaces destroyed (may be 0)
        return $count;
    }

    sub injectInterface {

        # Called by $self->setProfileInterfaces or GA::Generic::Cmd->addInterface
        # Given a specified inactive interface, this function checks whether there are any
        #   inactive interfaces with the same name whose cages are associated with superior or
        #   inferior profiles
        # If there are no interfaces attached to superior cages, an active interface is created
        #   using the attributes of the specfied (inactive) interface
        # If there is an interface attached to an inferior cage, the corresponding active interface
        #   is destroyed
        # As a result of these operations, there will never be any more than one active interface
        #   with a certain name, no matter how many inactive interfaces share that name
        # NB If two or more sessions are using the current world, active interfaces are updated for
        #   all of them
        #
        # Expected arguments
        #   $interfaceObj       - Blessed reference of the inactive interface object
        #   $interfaceName      - The inactive interface's name
        #   $assocProf          - The associated profile of the cage which stores the inactive
        #                           interface
        #   $superiorListRef    - Reference to a list of profile categories superior to $assocProf's
        #                           category (may be an empty list)
        #   $inferiorListRef    - Reference to a list of profile categories inferior to $assocProf
        #                           (may be an empty list)
        #
        # Return values
        #   'undef' on improper arguments, or for any errors
        #   1 if an 'active' interface is created for the specified interface
        #   2 is the specified interface remains 'inactive'
        #   3 if an active interface corresponding to the specified interface already exists (and
        #       was not replaced)

        my (
            $self, $interfaceObj, $interfaceName, $assocProf, $superiorListRef, $inferiorListRef,
            $replaceFlag, $check,
        ) = @_;

        # Local variables
        my ($createActiveFlag, $activeObj, $match, $assocProfObj);

        # Check for improper arguments
        if (
            ! defined $interfaceObj || ! defined $interfaceName || ! defined $assocProf
            || ! defined $superiorListRef || ! defined $inferiorListRef || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->injectInterface', @_);
        }

        # If an active interface has already been created for this inactive interface, we don't
        #   need to create another
        if ($self->ivExists('interfaceHash', $interfaceName)) {

            $activeObj = $self->ivShow('interfaceHash', $interfaceName);
            if ($activeObj->parent eq $interfaceObj) {

                return 3;
            }
        }

        # Other cages with a higher priority might already have an inactive interface with the same
        #   name. If so, the existing inactive interface takes priority over $interfaceObj, and we
        #   don't create a new active interface
        OUTER: foreach my $profCategory (@$superiorListRef) {

            my $cage = $self->findCurrentCage($interfaceObj->category, $profCategory);
            if ($cage && $cage->ivExists('interfaceHash', $interfaceName)) {

                $match = $profCategory;
                last OUTER;
            }
        }

        if (! $match) {

            # There is no superior trigger, so we need to create an active interface
            $createActiveFlag = TRUE;

            # Other cages with a lower priority might already have an inactive interface with the
            #   same name. If so, $interfaceObj takes priority over it. We need to destroy the
            #   existing active interface, and create a new one modelled on $interfaceObj
            # Look for a lower-priority trigger with the same name
            OUTER: foreach my $profCategory (@$inferiorListRef) {

                my $cage = $self->findCurrentCage($interfaceObj->category, $profCategory);
                if ($cage && $cage->ivExists('interfaceHash', $interfaceName)) {

                    $match = $profCategory;
                    last OUTER;
                }

            }

            if (defined $match) {

                # An inferior cage has an inactive interface with the same name. We must destroy the
                #   corresponding active interface
                if (! $self->removeInterface($interfaceObj)) {

                    # Couldn't destroy existing active interface (error message already displayed)
                    return undef;
                }
            }
        }

        # Create a new active interface, if required
        if ($createActiveFlag) {

            $assocProfObj = $self->ivShow('profHash', $assocProf);

            $activeObj = Games::Axmud::Interface::Active->new(
                $self,
                $interfaceObj->category,
                TRUE,               # An independent active interface
                $interfaceObj,      # The inactive interface, upon which the active one is based
                $assocProf,         # The inactive interface's associated profile
                $assocProfObj->category,
                                    # The associated profile's category
            );

            if (! $activeObj) {

                # Couldn't create the active interface (error message already displayed)
                return undef;

            } else {

                # Add the new active interface to every session using the same current world as this
                #   one (this session is affected too, obviously)
                foreach my $otherSession ($axmud::CLIENT->listSessions()) {

                    if ($otherSession->currentWorld eq $self->currentWorld) {

                        $otherSession->addInterface($activeObj);
                    }
                }

                # Return 1 to show creation of an active interface
                return 1;
            }

        } else {

            # Return 2 to show $interfaceObj didn't have an active interface created for it
            return 2;
        }
    }

    sub recallInterface {

        # Called by $self->resetProfileInterfaces and GA::Generic::Cmd->deleteInterface
        # Given a specified inactive interface, this function checks whether there is a
        #   corresponding active interface. If so, the active interface is deleted
        # If the active interface is deleted, checks whether there's an inactive interface with the
        #   same name, associated with an inferior profile. If so, an active interface is creatd for
        #   it
        # As a result of these operations, there will never be any more than one active interface
        #   with a certain name, no matter how many inactive interfaces share that name
        # NB Active interfaces which aren't based on an inactive interface can be deleted with a
        #   call to ->deleteInterface
        # NB If two or more sessions are using the current world, active interfaces are updated for
        #   all of them
        #
        # Expected arguments
        #   $interfaceObj       - Blessed reference of the inactive interface object
        #   $interfaceName      - The inactive interface's name
        #   $inferiorListRef    - Reference to a list of profile categories inferior to $assocProf
        #                           (may be an empty list)
        #
        # Optional arguments
        #   $noRecurseFlag      - Set to TRUE when this function is called by the ->recallInterface
        #                           method of other sessions sharing the same current world; this
        #                           function does not call any other sessions. If set to FALSE (or
        #                           'undef'), other sessions are updated
        #
        # Return values
        #   'undef' on improper arguments, if the specified inactive interface has no corresponding
        #       active interface, or for any errors
        #   1 if an active interface exists, but there was an error deleting it
        #   2 if the active interface is deleted, and there's an inactive interface with the same
        #       name, associated with an inferior profile, but there was an error creating an active
        #       interface for it
        #   3 if the active interface is deleted, and there's an inactive interface with the same
        #       name, associated with an inferior profile, for which an active interface was created
        #       created
        #   4 if the active interface is deleted, and there's no inactive interface with the same
        #       name, associated with an inferior profile

        my ($self, $interfaceObj, $interfaceName, $inferiorListRef, $noRecurseFlag, $check) = @_;

        # Local variables
        my (
            $match, $inactiveObj, $newActiveObj,
            @sessionList,
        );

        # Check for improper arguments
        if (! defined $interfaceObj || ! defined $interfaceName || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->recallInterface', @_);
        }

        if (! $noRecurseFlag) {

            # Every session using the same current world as this one must have its active interfaces
            #   updated. Call those sessions first, before updating interfaces in this session
            OUTER: foreach my $otherSession ($axmud::CLIENT->listSessions()) {

                if (
                    $otherSession->currentWorld eq $self->currentWorld
                    && $otherSession ne $self
                ) {
                    $otherSession->recallInterface(
                        $interfaceObj,
                        $interfaceName,
                        $inferiorListRef,
                        TRUE,                   # Only need one call to each ->recallInterface
                    );
                }
            }
        }

        # Now update this session's active interfaces

        # Look for an active interface corresponding to the inactive interface $interfaceObj
        OUTER: foreach my $activeObj ($self->ivValues('interfaceHash')) {

            if (defined $activeObj->parent && $activeObj->parent eq $interfaceObj) {

                $match = $activeObj;
                last OUTER;
            }
        }

        if (! $match) {

            # No corresponding active interface to delete
            return undef;

        # Delete the active interface
        } elsif (! $self->removeInterface($match)) {

            # Return 1 to show a general error deleting the active interface
            return 1;
        }

        # Other cages with a lower priority might already have an inactive interface with the same
        #   same name - if so, we must create a corresponding active interface
        OUTER: foreach my $profCategory (@$inferiorListRef) {

            my $cage = $self->findCurrentCage($interfaceObj->category, $profCategory);
            if ($cage && $cage->ivExists('interfaceHash', $interfaceName)) {

                # The TRUE flag means 'don't consult other cages'
                $inactiveObj = $cage->ivShow('interfaceHash', $interfaceName, TRUE);
                last OUTER;
            }
        }

        if ($inactiveObj) {

            # Create an active interface corresponding to the inferior cage's inactive interface
            $newActiveObj = $self->createActiveInterface(
                TRUE,               # An independent active interface, not a dependent one
                $inactiveObj,       # Active interface based on the inactive interface $interfaceObj
            );

            if (! $newActiveObj) {

                # Return 2 to show a general error creating a new active interface
                return 2;

            } else {

                # Return 3 to show the deleted active interface has been replaced by a new one
                #   corresponding to an inactive interface from an inferior cage
                return 3;
            }

        } else {

            # Return 4 to show the active interface was deleted, and no other active interface
            #   was created
            return 4;
        }
    }

    sub updateInterfaces {

        # Called by GA::Generic::Cmd->modifyInterface in response to ;modifytrigger, and so on
        # When an inactive interface is modified, active interfaces in every session sharing the
        #   same current world might need to be updated. ->modifyInterface calls this function in
        #   every affected session
        #
        # Expected arguments
        #   $inactiveObj    - The inactive interfaces which has been modified
        #   %attribHash     - The hash of modified interface attributes created by the calling
        #                       function (should not be empty)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error modifying an active interface
        #   1 otherwise

        my ($self, $inactiveObj, %attribHash) = @_;

        # Check for improper arguments
        if (! defined $inactiveObj || ! %attribHash) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateInterfaces', @_);
        }

        foreach my $activeObj ($self->ivValues('interfaceHash')) {

            my $timerEnableFlag;

            if (defined $activeObj->parent && $activeObj->parent eq $inactiveObj) {

                # For active timer interfaces that are disabled, but are about to become
                #   enabled, some additional IVs need to be set
                if (
                    $activeObj->category eq 'timer'     # It's a timer
                    && ! $activeObj->enabledFlag        # Currently disabled
                    && exists ($attribHash{'enabled'})  # The 'enabled' attribute will be set
                ) {
                    $timerEnableFlag = TRUE;
                }

                # Modify the interface
                if (! $activeObj->modifyAttribs($self, %attribHash)) {

                    return undef;

                } else {

                    if ($timerEnableFlag && $activeObj->enabledFlag) {

                        # A disabled timer has become enabled. Set a few IVs
                        $activeObj->becomeEnabled();
                    }

                    return 1;
                }
            }
        }
    }

    sub deleteInterface {

        # Can be called by anything
        # This function deletes an active interface that's not based on an inactive interface
        #   (created by a call to $self->createIndepInterface or $self->createInterface)
        # (There is no Axmud command to delete active interfaces - but since the code can create
        #   them, we also need a way for the code to delete them)
        #
        # Actually, the active interface is not deleted immediately - if $self->checkTriggers (etc)
        #   is halfway through checking triggers, it wouldn't be a good idea to tamper with the
        #   interface registries until it's finished. Instead, we add the interface to a list of
        #   doomed interfaces, and let the task loop delete them.
        #
        # Expected arguments
        #   $interface      - Name of the active interface to delete
        #
        # Optional arguments
        #   $noMsgFlag      - Should be set to TRUE when called by tasks in their shutdown code (or
        #                       anything similar); if the interface doesn't exist (presumably
        #                       because it has already been deleted during a disconnection), don't
        #                       show an error message.
        #                   - If set to FALSE (or 'undef'), trying to delete an interface that
        #                       doesn't now exist generates an error message
        #
        # Return values
        #   'undef' on improper arguments, if the interface doesn't exist or if it is based on an
        #       inactive interface
        #   1 otherwise

        my ($self, $interface, $noMsgFlag, $check) = @_;

        # Local variables
        my ($obj);

        # Check for improper arguments
        if (! defined $interface || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->deleteInterface', @_);
        }

        # Check the interface exists
        if (! $self->ivExists('interfaceHash', $interface)) {

            if ($noMsgFlag) {

                return undef;

            } else {

                return $self->writeError(
                    'Active interface \'' . $interface . '\' doesn\'t exist',
                    $self->_objClass . '->deleteInterface',
                );
            }

        } else {

            $obj = $self->ivShow('interfaceHash', $interface);
        }

        # Check the active interface isn't based on an inactive interface
        if ($obj->assocProf) {

            return $self->writeError(
                'Active interface \'' . $interface . '\' can\'t be deleted directly because it is'
                . ' based on an inactive interface (try \';delete' . $obj->category . '\' instead)',
                $self->_objClass . '->deleteInterface',
            );
        }

        # Disable the interface (so it doesn't fire unexpectedly)
        $obj->set_enabledFlag(FALSE);
        # Mark the interface for deletion
        $self->ivPush('deleteInterfaceList', $obj);

        return 1;
    }

    sub tidyInterfaces {

        # Called by $self->spinTaskLoop every time a task is marked as 'finished', in order to
        #   destroy all the active interfaces associated with the task
        # Can be called by tasks when they want to reset themselves. Any active interfaces they
        #   created are destroyed, so the tasks can create new ones
        #
        # Expected arguments
        #   $taskObj    - Blessed reference of the task
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the number of interfaces deleted (may be 0)

        my ($self, $taskObj, $check) = @_;

        # Local variables
        my (
            $count,
            %hash,
        );

        # Check for improper arguments
        if (! defined $taskObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->tidyInterfaces', @_);
        }

        # Import the interface registry
        %hash = $self->interfaceHash;

        # Go through every entry in the interface registry, deleting any active interfaces
        #   associated with $taskObj
        $count = 0;
        foreach my $interfaceObj (values %hash) {

            # If the active interface is associated with a task, the ->callClass IV will be set to
            #   the blessed reference of that task
            if ($interfaceObj->callClass && $interfaceObj->callClass eq $taskObj) {

                $count++;

                # Delete the corresponding entries in Axmud's interface registries
                $self->removeInterface($interfaceObj);
            }
        }

        # Return the number of active interfaces deleted (may be 0)
        return $count;
    }

    sub addInterface {

        # Called by $self->injectInterface or ->setupInterface
        # Adds an active interface from this GA::Session's registries
        # Should not be called by anything else (the correct way to add an active interface is
        #   to call $self->injectInterface, ->createIndepInterface or ->createInterface)
        #
        # Expected arguments
        #   $interfaceObj    - The active interface object to add
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $interfaceObj, $check) = @_;

        # Local variables
        my ($initialDelay, $randDelay, $minDelay, $delay);

        # Check for improper arguments
        if (! defined $interfaceObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addInterface', @_);
        }

        # Give the interface a number unique for this session
        $interfaceObj->set_number($self->interfaceCount);
        $self->ivIncrement('interfaceCount');

        # Add interface to registries
        $self->ivAdd('interfaceHash', $interfaceObj->name, $interfaceObj);
        $self->ivAdd('interfaceNumHash', $interfaceObj->number, $interfaceObj);

        if ($interfaceObj->category eq 'trigger') {

            # Update the trigger registries
            $self->ivAdd('triggerHash', $interfaceObj->number, $interfaceObj->stimulus);
            $self->posnInterface('triggerOrderList', $interfaceObj);

        } elsif ($interfaceObj->category eq 'alias') {

            # Update the alias registries
            $self->ivAdd('aliasHash', $interfaceObj->number, $interfaceObj->stimulus);
            $self->posnInterface('aliasOrderList', $interfaceObj);

        } elsif ($interfaceObj->category eq 'macro') {

            # Update the macro registries
            $self->ivAdd('macroHash', $interfaceObj->number, $interfaceObj->stimulus);
            $self->posnInterface('macroOrderList', $interfaceObj);

            # The GA::Client also has a registry of keycodes that are used by active macro
            #   interfaces. Make sure it contains an entry for this macro
            $axmud::CLIENT->add_activeKeycode($interfaceObj->stimulus);

        } elsif ($interfaceObj->category eq 'timer') {

            # Get the timer's initial delay, in seconds (might be 0)
            $initialDelay = $interfaceObj->ivShow('attribHash', 'initial_delay');

            # $interfaceObj->stimulus is a time interval, in seconds. The timer loop only fires once
            #   every tenth of a second, so if the interval isn't a multiple of a tenth of a second
            #   ('1' or '5' or '0.2'), it is automatically rounded
            if (! defined $self->sessionTime) {

                # The timer loop hasn't spun yet. The timer will start counting after the timer loop
                #   starts, at which time $self->sessionTime will be set to 0
                $delay = $initialDelay;

            } elsif ($initialDelay) {

                # The timer fires for the first time after its initial delay, and thereafter after
                #   the delay stored in the 'stimulus' attribute
                $delay = $initialDelay + $self->sessionTime;

            } elsif ($interfaceObj->ivShow('attribHash', 'random_delay')) {

                # The timer fires for the first time after a random delay between 0 (or the value
                #   set by the 'random_min' attribute), and $interfaceObj->stimulus
                $minDelay = $interfaceObj->ivShow('attribHash', 'random_min');
                # Check: if the minimum random delay is greater than $interfaceObj->stimulus,
                #   ignore it
                if ($minDelay >= $interfaceObj->stimulus) {

                    $minDelay = 0;
                }

                $randDelay = rand($interfaceObj->stimulus - $minDelay) + $minDelay;
                $delay = $self->sessionTime + $randDelay;

            } else {

                # The timer fires for the first time after the delay stored in the 'stimulus'
                #   attribute ((even if the 'random_delay' attribute is set)
                $delay = $self->sessionTime + $interfaceObj->stimulus;
            }

            # Set the time at which the timer will fire for the first time by adding an entry to the
            #   active timer registry
            $self->ivAdd('timerHash', $interfaceObj->number, $delay);

            # The parallel registry, $self->timerOrderList, contains all the keys of
            #   $self->timerHash but in ascending order (so that timers always fire in a predictable
            #   order)
            $self->posnInterface('timerOrderList', $interfaceObj);

        } elsif ($interfaceObj->category eq 'hook') {

            # Update the hook registries
            $self->ivAdd('hookHash', $interfaceObj->number, $interfaceObj->stimulus);
            $self->posnInterface('hookOrderList', $interfaceObj);
        }

        return 1;
    }

    sub removeInterface {

        # Called by $self->injectInterface, ->recallInterface and ->tidyInterfaces
        # Called by $self->checkTimers (when an active timer interface expires)
        # Called by $self->deleteInterface (to delete an active interface that's not based on an
        #   inactive interface)
        #
        # Deletes an active interface from this GA::Session's registries
        # Should not be called by anything else (the correct way to remove an active interface is
        #   to call $self->recallInterface or ->deleteInterface)
        #
        # Expected arguments
        #   $interfaceObj    - The active interface object to remove
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $interfaceObj, $check) = @_;

        # Local variables
        my $index;

        # Check for improper arguments
        if (! defined $interfaceObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeInterface', @_);
        }

        # Check that the interface hasn't alredy been removed (possible, if the interface has
        #   been added to $self->deleteInterfaceList, marked to be deleted on the next spin of the
        #   task loop)
        if (! $self->ivExists('interfaceHash', $interfaceObj->name)) {

            # Interface already removed
            return 1;
        }

        $self->ivDelete('interfaceHash', $interfaceObj->name);
        $self->ivDelete('interfaceNumHash', $interfaceObj->number);

        if ($interfaceObj->category eq 'trigger') {

            # Remove the entry in the main active trigger registry
            $self->ivDelete('triggerHash', $interfaceObj->number);
            # Remove the corresponding entry in the ordered registry
            $index = $self->ivFind('triggerOrderList', $interfaceObj->number);
            $self->ivSplice('triggerOrderList', $index, 1);

        } elsif ($interfaceObj->category eq 'alias') {

            # Remove the entry in the main active alias registry
            $self->ivDelete('aliasHash', $interfaceObj->number);
            # Remove the corresponding entry in the ordered registry
            $index = $self->ivFind('aliasOrderList', $interfaceObj->number);
            $self->ivSplice('aliasOrderList', $index, 1);

        } elsif ($interfaceObj->category eq 'macro') {

            # Remove the entry in the main active macro registry
            $self->ivDelete('macroHash', $interfaceObj->number);
            # Remove the corresponding entry in the ordered registry
            $index = $self->ivFind('macroOrderList', $interfaceObj->number);
            $self->ivSplice('macroOrderList', $index, 1);

            # The GA::Client also has a registry of keycodes that are used by active macro
            #   interfaces. It would be a bad idea to modify this registry by checking every active
            #   macro in every session - especially if we're deleting several hundred macros in
            #   one go
            # Instead, set a flag for the timer loop. When the loop spins (once every tenth of a
            #   second), if this flag is TRUE, the whole client registry is updated
            $axmud::CLIENT->set_resetKeycodesFlag(TRUE);

        } elsif ($interfaceObj->category eq 'timer') {

            # Remove the entry in the main active timer registry
            $self->ivDelete('timerHash', $interfaceObj->number);
            # Remove the corresponding entry in the ordered registry
            $index = $self->ivFind('timerOrderList', $interfaceObj->number);
            $self->ivSplice('timerOrderList', $index, 1);

        } elsif ($interfaceObj->category eq 'hook') {

            # Remove the entry in the main active hook registry
            $self->ivDelete('hookHash', $interfaceObj->number);
            # Remove the corresponding entry in the ordered registry
            $index = $self->ivFind('hookOrderList', $interfaceObj->number);
            $self->ivSplice('hookOrderList', $index, 1);
        }

        return 1;
    }

    sub posnInterface {

        # Called by $self->addInterface, when a new active interface is added to one of the
        #   registries $self->triggerOrderList, ->aliasOrderList, ->macroOrderList, ->timerOrderList
        #   or ->hookOrderList
        # The specified active interface, $interfaceObj, has a corresponding inactive interface
        #   which can specify that the active interface should appear before (or after) other
        #   named interfaces
        # Check the registries. If necessary, position the specified active interface at some
        #   point in the registry before (or after) other named interfaces; otherwise, position it
        #   at the end of the registry
        #
        # Expected arguments
        #   $iv             - The registry to which $interfaceObj must be added - 'triggerOrderList'
        #                       etc
        #   $interfaceObj   - The specified active interface
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $iv, $interfaceObj, $check) = @_;

        # Local variables
        my ($inactiveObj, $first, $last);

        # Check for improper arguments
        if (! defined $iv || ! defined $interfaceObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->posnInterface', @_);
        }

        # Get the corresponding inactive interface
        $inactiveObj = $interfaceObj->parent;
        if (! $inactiveObj) {

            # The active interface, $interfaceObj, doesn't have an inactive parent (because it was
            #   created by a call to $self->createIndepInterface or $self->createInterface), so it
            #   can be positioned at the end of the registry
            $self->ivPush($iv, $interfaceObj->number);

        } elsif (
            (! $inactiveObj->beforeHash && ! $inactiveObj->afterHash)
            || ! $self->$iv
        ) {
            # The inactive interface's before/after hashes are empty; or this is the first active
            #   interface to be added to the registry - so simply add the active interface to the
            #   end of the registry
            $self->ivPush($iv, $interfaceObj->number);

        } else {

            # Find the latest index at which the interface could be inserted, in order to be before
            #   other named interfaces
            for (my $index = 0; $index < $self->ivNumber($iv); $index++) {

                my ($thisObj, $name);

                # Get the active interface at position $index
                $thisObj = $self->ivIndex($index);
                # Get the name of the corresponding inactive interface
                $name = $thisObj->parent->name;

                if ($inactiveObj->ivExists('beforeHash', $name)) {

                    # $interfaceObj must appear at this index (or earlier)
                    if (! defined $first || $first > $index) {

                        $first = $index;
                    }
                }
            }

            # Find the earliest index at which the interface could be inserted, in order to be
            #   after other named interfaces
            for (my $index = 0; $index < $self->ivNumber($iv); $index++) {

                my ($thisObj, $name);

                # Get the active interface at position $index
                $thisObj = $self->ivIndex($index);
                # Get the name of the corresponding inactive interface
                $name = $thisObj->parent->name;

                if ($inactiveObj->ivExists('afterHash', $name)) {

                    # $interfaceObj must appear later than this index
                    if (! defined $last || $last < $index) {

                        $last = $index + 1;
                    }
                }
            }

            if (! defined $first && ! defined $last) {

                # Specified interface can be placed at the end of the ordered list
                $self->ivPush($iv, $interfaceObj->number);

            } elsif (defined $first && defined $last) {

                # The 'after' hash takes priority over the 'before' hash, so use $last
                $self->ivSplice($iv, $last, 1, $interfaceObj->number);

            } elsif (defined $first) {

                $self->ivSplice($iv, $first, 1, $interfaceObj->number);

            } else {

                $self->ivSplice($iv, $last, 1, $interfaceObj->number);
            }
        }

        return 1;
    }

    sub moveInterface {

        # Quick-and-dirty alternative to ->posnInterface, can be called by any code, usually after
        #   creating active interface(s) that are not based on an inactive interface
        # Moves the active interface(s) to the beginning (or end) of their order lists
        #
        # Expected arguments
        #   $flag       - Set to TRUE if the interfaces should be moved to the beginning of their
        #                   order lists, FALSE if they should be moved to the end
        #
        # Optional arguments
        #   @list       - List of names of active interface names. If the list is empty, no
        #                   interfaces are moved
        #
        # Return values
        #   'undef' on improper arguments or if @list is empty
        #   1 otherwise

        my ($self, $flag, @list) = @_;

        # Check for improper arguments
        if (! defined $flag) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moveInterface', @_);
        }

        if (! @list) {

            # No interfaces to move
            return undef;
        }

        if ($flag) {

            # Reverse the list of interfaces, so that if it contains (for example) the triggers
            #   numbered [11, 12, 13], $self->triggerOrderList (etc) is modified so that it begins
            #   with triggers [11, 12, 13]
            @list = reverse @list;
        }

        OUTER: foreach my $name (@list) {

            my ($obj, $iv, @orderList);

            # Get the active interface object
            $obj = $self->ivShow('interfaceHash', $name);
            if (! $obj) {

                next OUTER;
            }

            # Set which IV stores this type of interface
            $iv = $obj->category . 'OrderList';
            @orderList = $self->$iv;

            # Find the position of the interface in the list
            INNER: for (my $count = 0; $count < (scalar @orderList); $count++) {

                if ($orderList[$count] == $obj->number) {

                    # Remove the interface from its position, and move it to the beginning (or end)
                    #   of the list
                    splice (@orderList, $count, 1);
                    if ($flag) {
                        unshift(@orderList, $obj->number);
                    } else {
                        push (@orderList, $obj->number);
                    }

                    last INNER;
                }
            }

            # Update the IV
            $self->ivPoke($iv, @orderList);
        }

        return 1;
    }

    sub createIndepInterface {

        # Can be called by anything to create an independent active interface whose attributes
        #   aren't copied from an existing inactive interface, stored in some cage
        # Most usually called by tasks. 'Independent' interfaces perform some action in 'response'
        #   to a 'stimulus'.
        #
        # 'Dependent' interfaces call an object method as their 'response', and are created by a
        #   call to ->createInterface. Both this function and ->createIndepInterface pass their
        #   arguments to ->setupInterface; we use three functions instead of one so it's clear, in
        #   the task code, what sort of interface is being created
        # To create an active interface using attributes copied from an existing inactive interface,
        #   stored in a cage, call ->injectInterface
        #
        # This function carries out a few basic checks before passing its arguments to
        #   ->setupInterface
        #
        # Expected arguments
        #   $type       - Which type of interface: 'trigger' ,'alias', 'macro', 'timer' or 'hook'
        #   $stimulus   - The stimulus, e.g. for a trigger interface, a regex like
        #                   'You kill the (.*) orc'
        #   $response   - The response, e.g. for a trigger interface, a string like
        #                   'You bravely attack your foe!'
        #
        # Optional arguments
        #   @args       - A list of attributes, in the form
        #                   (attribute, value, attribute, value)
        #               - Attributes can be two of the four standard attributes ('name', 'enabled')
        #                   or any of the category-specific attributes (e.g. 'gaglog' for a
        #                   trigger interface
        #               - The standard attributes 'stimulus' and 'response' can't be used in @args
        #
        # Return values
        #   'undef' on improper arguments, if $callClass isn't a blessed reference or if @args
        #       contains an odd number of elements
        #   Otherwise, returns the value of the call to $self->setupInterface

        my ($self, $type, $stimulus, $response, @args) = @_;

        # Check for improper arguments
        if (! defined $type || ! defined $stimulus || ! defined $response) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->createIndepInterface', @_);
        }

        # Carry out a few basic checks
        if (@args > 0 && @args % 2 == 1) {

            return $self->writeError(
                'Missing value in attribute/value pair',
                $self->_objClass . '->createIndepInterface',
            );

        } elsif (
            $type ne 'trigger' && $type ne 'alias' && $type ne 'macro' && $type ne 'timer'
            && $type ne 'hook'
        ) {
            return $self->writeError(
                'Unrecognised interface type \'' . $type . '\'',
                $self->_objClass . '->createIndepInterface',
            );

        } elsif ($self->status eq 'disconnected') {

            # Can't create this kind of interface after a disconnection
            return $self->writeError(
                'Can\'t create independent active interfaces after disconnection',
                $self->_objClass . '->createInterface',
            );

        } else {

            # Set up the interface
            return $self->setupInterface(
                $type,
                TRUE,      # Inependent interface
                $stimulus,
                $response,
                @args,
            );
        }
    }

    sub createInterface {

        # Can be called by anything to create a dependent active interface whose attributes
        #   aren't copied from an existing inactive interface, stored in some cage
        # Most usually called by tasks. 'Dependent' interfaces call an Axmud object method as their
        #   'response' to a 'stimulus'; in this case, a specially-created method in the task will
        #   usually be called. (Built-in tasks have methods that end with 'Seen', such as
        #   ->triggerSeen, for this sort of call)
        #
        # 'Independent' interfaces define their own 'response', and are created by a call to
        #   ->createIndepInterface. Both this function and ->createIndepInterface pass their
        #   arguments to ->setupInterface; we use three functions instead of one so it's clear, in
        #   the task code, what sort of interface is being created
        # To create an active interface using attributes copied from an existing inactive interface,
        #   stored in a cage, call ->injectInterface
        #
        # This function carries out a few basic checks before passing its arguments to
        #   ->setupInterface
        #
        # Expected arguments
        #   $type       - Which type of interface: 'trigger' ,'alias', 'macro', 'timer' or 'hook'
        #   $stimulus   - The stimulus, e.g. for a trigger interface, a string like
        #                   'You kill the (.*) orc'
        #   $callClass  - The object to call in response
        #   $callMethod - The object's method to call in response
        #
        # Optional arguments
        #   @args       - A list of attributes, in the form
        #                   (attribute, value, attribute, value)
        #               - Attributes can be two of the four standard attributes ('name', 'enabled')
        #                   or any of the category-specific attributes (e.g. 'gaglog' for a
        #                   trigger interface)
        #               - The standard attributes 'stimulus' and 'response' can't be used in @args
        #
        # Return values
        #   'undef' on improper arguments, if $callClass isn't a blessed reference, if $type isn't
        #       a valid interface type or if @args contains an odd number of elements
        #   Otherwise, returns the value of the call to $self->setupInterface

        my ($self, $type, $stimulus, $callClass, $callMethod, @args) = @_;

        # Check for improper arguments
        if (
            ! defined $type || ! defined $stimulus || ! defined $callClass || ! defined $callMethod
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->createInterface', @_);
        }

        # Carry out a few basic checks
        if (! ref $callClass) {

            return $self->writeError(
                'Unrecognised call class \'' . $callClass . '\' (must be a blessed reference)',
                $self->_objClass . '->createInterface',
            );

        } elsif (
            $type ne 'trigger' && $type ne 'alias' && $type ne 'macro' && $type ne 'timer'
            && $type ne 'hook'
        ) {
            return $self->writeError(
                'Unrecognised interface type \'' . $type . '\'',
                $self->_objClass . '->createInterface',
            );

        } elsif (@args > 0 && @args % 2 == 1) {

            return $self->writeError(
                'Missing value in attribute/value pair',
                $self->_objClass . '->createInterface',
            );

        } elsif ($self->status eq 'disconnected') {

            # Can't create this kind of interface after a disconnection
            return $self->writeError(
                'Can\'t create dependent active interfaces after disconnection',
                $self->_objClass . '->createInterface',
            );

        } else {

            # Set up the interface
            return $self->setupInterface(
                $type,
                FALSE,      # Dependent interface
                $stimulus,
                $callClass,
                $callMethod,
                @args,
            );
        }
    }

    sub setupInterface {

        # Called by $self->createInterface and ->createIndepInterface with a list of arguments that
        #   has already undergone basic checks (must not be called by any other function)
        # Creates the GA::Interface::Active object, sets its attributes, and adds the object to this
        #   GA::Session's active interface registries
        #
        # Expected arguments
        #   $type       - Which type of interface: 'trigger' ,'alias', 'macro', 'timer' or 'hook'
        #   $indepFlag  - TRUE for independent interfaces, FALSE for dependent interfaces
        #   $stimulus   - The stimulus, e.g. for a trigger interface, a regex like
        #                   'You kill the (.*) orc'
        #   @args       - The remaining arguments
        #
        # Return values
        #   'undef' on improper arguments, if any of the @args are invalid or if the object can't
        #       be created
        #   Otherwise, returns the new GA::Interface::Active object

        my ($self, $type, $indepFlag, $stimulus, @args) = @_;

        # Local variables
        my (
            $response, $callClass, $callMethod, $activeObj, $result, $interfaceModelObj,
            %attribHash,
        );

        # Check for improper arguments
        if (! defined $type || ! defined $indepFlag || ! defined $stimulus) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupInterface', @_);
        }

        # Extract some arguments from @args
        if ($indepFlag) {

            # Independent interface
            $response = shift @args;

        } else {

            # Dependent interface
            $callClass = shift @args;
            $callMethod = shift @args;
        }

        # Now, any remaining arguments are in the form (attribute, value, attribute, value...)

        # Extract the remaining @args, which will be a list in the form
        #   (attribute, value, attribute, value...)
        if (@args) {

            do {

                my ($attrib, $value);

                $attrib = shift @args;
                $value = shift @args;

                if (! defined $value) {

                    return $self->writeError(
                        'Undefined value matching attribute \'' . $attrib . '\'',
                        $self->_objClass . '->setupInterface',
                    );

                # @args shouldn't contain 'stimulus', 'response', because they have already been
                #   specified
                # For dependent interfaces, @args shouldn't contain 'name', because a name will be
                #   automatically generated by this function (but it can be specified with
                #   independent interfaces)
                } elsif (
                    $attrib eq 'stimulus'
                    || $attrib eq 'response'
                ) {
                    return $self->writeError(
                        'Recognised but forbidden attribute \'' . $attrib . '\'',
                        $self->_objClass . '->setupInterface',
                    );

                # Warn about duplicate attributes
                } elsif (exists $attribHash{$value}) {

                    # Not a fatal error
                    $self->writeWarning(
                        'Duplicate attribute \'' . $attrib . '\'',
                        $self->_objClass . '->setupInterface',
                    );

                } else {

                    # Use this attribute-value pair
                    $attribHash{$attrib} = $value;
                }

            } until (! @args);
        }

        # Set the 'name' attribute, if %attribHash doesn't already contain one
        if (! exists $attribHash{'name'}) {

            if ($indepFlag) {

                # e.g. 'indep_trigger_36'
                $attribHash{'name'} = 'indep_' . $type . '_' . ($self->interfaceCount + 1);

            } else {

                # e.g. 'status_task_trigger_36'
                $attribHash{'name'}
                    = $callClass->_objName . '_' . $type . '_' . ($self->interfaceCount + 1);
            }
        }

        # Add the stimulus and response to %attribHash
        $attribHash{'stimulus'} = $stimulus;
        if ($indepFlag) {

            # (For dependent active interfaces, the response is made up of two components, set
            #   later in this function)
            $attribHash{'response'} = $response;
        }

        # Set the 'enabled' attribute, if %attribHash doesn't already contain one
        if (! exists $attribHash{'enabled'}) {

            # Enabled by default
            $attribHash{'enabled'} = TRUE;
        }

        # Create the active interface
        $activeObj = Games::Axmud::Interface::Active->new(
            $self,
            $type,
            $indepFlag,
        );

        if (! $activeObj) {

            # Couldn't create the active interface (error message already displayed)
            return undef;
        }

        # Set the active interface's attributes (including the ->name and ->enabledFlag attributes,
        #   if they are specified in %attribHash), overwriting the default attributes created just
        #   above
        $result = $activeObj->modifyAttribs($self, %attribHash);
        if (! $result) {

            # Error message already displayed
            return undef;
        }

        # For dependent active interfaces, the response is made up of two components
        if (! $indepFlag) {

            $activeObj->set_callClass($callClass);
            $activeObj->set_callMethod($callMethod);
        }

        # Add the new active interface to this GA::Session's registries
        if (! $self->addInterface($activeObj)) {
            return undef;
        } else {
            return $activeObj;
        }
    }

    # Fire active interfaces

    sub checkTriggers {

        # Called by $self->processLineSegment
        # Checks every active trigger interface. Fires every trigger that should be fired in
        #   response to a line of text received from the world (before it is displayed in the 'main'
        #   window)
        #
        # Checking against aliases triggers with the first trigger in $self->triggerOrderList, and
        #   ends when:
        #   - the first matching trigger whose 'keep_checking' attribute is TRUE, or
        #   - the first matching trigger whose 'response' attribute is a Perl command, or
        #   - the end of the active trigger list is reached
        #
        # The specified line, $stripLine, gets modified by any matching rewriter triggers.
        # The accompanying %tagHash (which holds the escape sequences in the original $line before
        #   they were removed, and now converted into Axmud colour/style tags) is also modified by
        #   any matching rewriter triggers, preserving the positions of those tags for parts of the
        #   line that were not modified (and adding new tags if matching triggers define a style)
        # Non-rewriter triggers specify an instruction, which is executed before the line is
        #   checked against other triggers (if allowed)
        #
        # NB This function ignores any splitter triggers - these are checked in calls to
        #   $self->checkSplitLine
        #
        # Expected arguments
        #   $text        - The line of text received from the world (including any escape sequences)
        #   $stripText   - The same line, with escape sequences removed
        #   $newLineFlag - Flag set to TRUE if the line of text ended with a newline character,
        #                    FALSE if not
        #
        # Optional arguments
        #   %tagHash     - A hash containing the removed escape sequences converted into Axmud
        #                    colour/style tags, in the form
        #                    $tagHash{position} = reference_to_a_list_of_Axmud_colour_and_style_tags
        #                  ...where $position is the position in $stripText where one or more escape
        #                    sequences occured (the first character is position 0. An empty hash
        #                    if $text contains no escape sequences)
        #
        # Return values
        #   An empty list on improper arguments or if no triggers fire
        #   Otherwise, returns a list in the form
        #       ($modText, $gagFlag, $gagLogFlag, \@instructList, \@dependentCallList, %tagHash)
        #   ...where:
        #       - $modText is the modified form of $stripText
        #       - $gagFlag is TRUE if the last trigger that fired had its 'gag' attribute set,
        #           FALSE if it didn't have its 'gag' attribute set
        #       - $gagLogFlag is TRUE if the last trigger that fired had its 'gag_log' attribute
        #           set, FALSE if it didn't have its 'gag_log' attribute set
        #       - @instructList is a list of instructions (not including Perl commands, which have
        #           already been evaluated), in the order their triggers fired, for the calling
        #           function to execute
        #       - @dependentCallList contains list references, each one of which stores the
        #           arguments used when a fired dependent trigger is processed by
        #           $self->processLineSegment
        #       - %tagHash is a modified hash of Axmud colour/style tags

        my ($self, $text, $stripText, $newLineFlag, %tagHash) = @_;

        # Local variables
        my (
            $modText, $lastObj, $gagFlag, $gagLogFlag, $firstFireFlag,
            @emptyList, @instructList, @dependentCallList, @deleteList,
            %hash,
        );

        # Check for improper arguments
        if (! defined $text || ! defined $stripText || ! defined $newLineFlag) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->checkTriggers', @_);
            return @emptyList;
        }

        # Import the active trigger registry for quick lookup
        %hash = $self->triggerHash;

        # Check every active trigger interface, in the correct order, to see if it is due to fire
        $modText = $stripText;
        OUTER: foreach my $number ($self->triggerOrderList) {

            my (
                $obj, $regex, $paneName, $substitution, $instruction, $result, $ignoreFlag,
                $thisObj, $perlFlag, $class, $method, $startOffset, $stopOffset, $lengthChange,
                $mode, $start, $stop, $safeFlag, $globalFlag, $compareString, $loopCount,
                @itemList, @matchMinusList, @matchPlusList, @backRefList,
                %oldTagHash,
            );

            $obj = $self->ivShow('interfaceNumHash', $number);
            $regex = $obj->stimulus;

            # If the trigger is disabled, don't fire it
            # If the trigger is a splitter trigger, don't fire it
            # If the trigger requires a line ending with a newline character and the line doesn't
            #   end with one, don't fire it
            # If the trigger requires a login and the character isn't logged in, don't fire it
            # If the trigger requires a named pane, don't fire it
            $paneName = $obj->ivShow('attribHash', 'pane_name');
            if ($paneName eq '') {

                $paneName = undef;
            }

            if (
                ! $obj->enabledFlag
                || $obj->ivShow('attribHash', 'splitter')
                || ($obj->ivShow('attribHash', 'need_prompt') && $newLineFlag)
                || ($obj->ivShow('attribHash', 'need_login') && ! $self->loginFlag)
                || (defined $paneName && $self->currentTabObj->paneObj->name ne $paneName)
                || (
                    ! defined $paneName
                    && $obj->ivShow('attribHash', 'default_pane')
                    && $self->currentTabObj ne $self->defaultTabObj
                )
            ) {
               next OUTER;
            }

            if ($obj->indepFlag) {

                # An independent trigger

                # If the trigger ->response begins with a forward slash, checking against other
                #   triggers halts and the Perl command is evaluated
                # Otherwise, the ->response is executed as an instruction and checking against
                #   other triggers continues (if allowed)
                #
                # However, 'rewriter' triggers behave in a different way. Their ->response is used
                #   in a substitution to modify the received line of text. Checking against other
                #   triggers continues (if allowed, in which case any of them that are rewriter
                #   triggers can also modify the line of text)
                if ($obj->ivShow('attribHash', 'rewriter')) {

                    # An indendent rewriter trigger
                    $substitution = $obj->response;
                    # Perform this trigger's pattern match on the result of previous matches
                    $result = $modText;
                    # Compare the length of $result before and after each substitution with this var
                    $compareString = $result;

                    $ignoreFlag = $obj->ivShow('attribHash', 'ignore_case');
                    $globalFlag = $obj->ivShow('attribHash', 'rewrite_global');

                    # If $globalFlag is true, it would be nice to do substitutions with s//g, but
                    #   it's not possible to accurately keep track of tag positions after a
                    #   substitution that might have twice (or more) on the same line
                    # Solution is a do..until loop, which terminates when $globalFlag is FALSE. If
                    #   $globalFlag is currently false, we do one substitution; if $globalFlag is
                    #   currently TRUE, we continue doing substitutions until the regex no longer
                    #   matches the line - at that point, $globalFlag is set to FALSE, which
                    #   terminates the loop
                    # To prevent infinite loops, Axmud doesn't do more than $self->constRewriteMax
                    #   rewrite operations per line, per trigger
                    $loopCount = 0;     # Give up when this gets to 16
                    do {

                        $loopCount++;
                        if ($loopCount >= $self->constRewriteMax) {

                            $globalFlag = FALSE;   # No more rewrites on this line, for this trigger
                        }

                        if (
                            ($ignoreFlag && $result =~ m/$regex/i)
                            || (! $ignoreFlag && $result =~ m/$regex/)
                        ) {
                            # The trigger has fired
                            $thisObj = $obj;        # Marks the trigger as having fired
                            if (! $firstFireFlag && $thisObj->ivShow('attribHash', 'temporary')) {

                                # A temporary trigger, which must be deleted when it first fires
                                push (@deleteList, $thisObj);
                                # Don't add it to the list more than once
                                $firstFireFlag = TRUE;
                            }

                            # Perform the rewriting operation. If $substitution is enclosed in
                            #   double-quotes, it's safe to use the /ee modifier which will convert
                            #   $1, $2 etc into the contents of any matched backreferences
                            #   (otherwise we could end up executing arbitrary Perl code)
                            if (
                                substr($substitution, 0, 1) eq '"'
                                && substr($substitution, -1) eq '"'
                            ) {
                                $safeFlag = TRUE;
                            }

                            if ($ignoreFlag) {

                                if ($safeFlag) {
                                    $result =~ s/$regex/$substitution/iee;
                                } else {
                                    $result =~ s/$regex/$substitution/i;
                                }

                            } else {

                                if ($safeFlag) {
                                    $result =~ s/$regex/$substitution/ee;
                                } else {
                                    $result =~ s/$regex/$substitution/;
                                }
                            }

                            @matchMinusList = @-;
                            @matchPlusList = @+;

                            # We need to modify the hash of Axmud colour/style tags, %tagHash
                            #
                            # e.g.
                            #               0123456789012345678901234567890123456789012
                            #                   ...............
                            #   $modText = "the quick brown fox jumps over the lazy dog";
                            #   $modText =~ s/quick (.*) fox)/speedy $1 cat/;
                            #
                            # The offsets for the replaced text are $-[0] = 4 and $+[0] = 19
                            # Any colour/style tags which occur at offsets 0-4 or 19+ are preserved,
                            #   but all colour/style tags at offsets 5-18 are destroyed
                            # Any colour/style tags at offset 19+ must have their offset increased
                            #   (if $modText has increased in length) or decreased (if $modText has
                            #   decreased in length)
                            # (@-)
                            $startOffset = $matchMinusList[0];
                            # (@+)
                            $stopOffset = $matchPlusList[0];
                            # (Negative if shorter)
                            $lengthChange = length ($result) - length ($compareString);
                            # (Next do..until loop iteration, compare length of this iteration)
                            $compareString = $result;

                            # Modify the hash
                            %oldTagHash = %tagHash;
                            %tagHash = ();

                            foreach my $textNum (keys %oldTagHash) {

                                my ($listRef, $newLineNum);

                                $listRef = $oldTagHash{$textNum};

                                # Preserve only the escape sequences at positions (e.g.) 0-4, 19+
                                if ($textNum <= $startOffset) {

                                    $tagHash{$textNum} = $listRef;     # e.g. offsets 0-4

                                } elsif ($textNum >= $stopOffset) {

                                    $newLineNum = $textNum + $lengthChange;
                                    $tagHash{$newLineNum} = $listRef;
                                }
                            }

                        } else {

                            # The regex no longer matches the line, so stop doing substitutions
                            $globalFlag = FALSE;
                        }

                    } until (! $globalFlag);

                    # Future modifier triggers modify the modified line
                    $modText = $result;

                } else {

                    # An independent trigger (not a rewriter). Perform the pattern match
                    if ($obj->ivShow('attribHash', 'ignore_case')) {

                        $result = @backRefList = ($modText =~ m/$regex/i);
                        # Backref data is local to this code block, so store it now
                        if ($result) {

                            # (Index 0 should contain the whole matched string, index 1 should be
                            #   the first matched substring, corresponding to $1, etc)
                            unshift (@backRefList, substr($modText, $-[0], $+[0] - $-[0]));
                            @matchMinusList = @-;
                            @matchPlusList = @+;
                        }

                    } else {

                        $result = @backRefList = ($modText =~ m/$regex/);
                        # Backref data is local to this code block, so store it now
                        if ($result) {

                            unshift (@backRefList, substr($modText, $-[0], $+[0] - $-[0]));
                            @matchMinusList = @-;
                            @matchPlusList = @+;
                        }
                    }

                    if ($result) {

                        # The trigger has (potentially) fired
                        $instruction = $obj->response;
                        if ($obj->ivShow('attribHash', 'temporary')) {

                            # A temporary trigger, which must be deleted when it first fires
                            push (@deleteList, $thisObj);
                        }

                        if (
                            $axmud::CLIENT->perlSigilFlag
                            && index ($instruction, $axmud::CLIENT->constPerlSigil) == 0
                        ) {
                            # An independent trigger using a Perl command as an instruction

                            # Store data so that it's available to the Perl mini-programme
                            $self->ivAdd('perlCmdDataHash', '_line', $text);
                            $self->ivAdd('perlCmdDataHash', '_stripLine', $stripText);
                            $self->ivAdd('perlCmdDataHash', '_modLine', $modText);
                            $self->ivPoke('perlCmdTagHash', %tagHash);
                            $self->ivPoke('perlCmdBackRefList', @backRefList);
                            # (Also store the interface which fired)
                            $self->ivAdd('perlCmdDataHash', '_interface', undef);

                            # Evaluate the Perl command, getting an instruction as a return value
                            $instruction = $self->perlCmd($instruction, TRUE);

                            # The stored data is no longer needed
                            $self->ivAdd('perlCmdDataHash', '_line', undef);
                            $self->ivAdd('perlCmdDataHash', '_stripLine', undef);
                            $self->ivAdd('perlCmdDataHash', '_modLine', undef);
                            $self->ivEmpty('perlCmdTagHash');
                            $self->ivEmpty('perlCmdBackRefList');
                            $self->ivAdd('perlCmdDataHash', '_interface', undef);

                            # Don't check any other triggers
                            $perlFlag = TRUE;
                        }

                        # If the call to ->perlCmd returned 'undef' (because the Perl mini-programme
                        #   was invalid), we'll treat this trigger as not having fired
                        if ($instruction) {

                            # The trigger has (definitely) fired
                            $thisObj = $obj;
                            # Add the instruction to the list of instructions to process, when
                            #   control is returned to the calling function
                            push (@instructList, $instruction);
                        }
                    }
                }

            } else {

                # A dependent trigger. If its 'rewriter' attribute is TRUE, ignore the attribute
                #   (dependent triggers can't be rewriters). If the 'style_mode' attribute is not
                #   0, ignore the attribute (dependent triggers can't be used with styles).
                # Perform the pattern match
                if ($obj->ivShow('attribHash', 'ignore_case')) {

                    $result = @backRefList = ($modText =~ m/$regex/i);
                    # Backref data is local to this code block, so store it now
                    if ($result) {

                        # (Index 0 should contain the whole matched string, index 1 should be the
                        #   first matched substring, corresponding to $1, etc)
                        unshift (@backRefList, substr($modText, $-[0], $+[0] - $-[0]));
                        @matchMinusList = @-;
                        @matchPlusList = @+;
                    }

                } else {

                    $result = @backRefList = ($modText =~ m/$regex/);
                    # Backref data is local to this code block, so store it now
                    if ($result) {

                        unshift (@backRefList, substr($modText, $-[0], $+[0] - $-[0]));
                        @matchMinusList = @-;
                        @matchPlusList = @+;
                    }
                }

                if ($result) {

                    # The trigger has fired
                    $thisObj = $obj;
                    if ($thisObj->ivShow('attribHash', 'temporary')) {

                        # A temporary trigger, which must be deleted when it first fires
                        push (@deleteList, $thisObj);
                    }

                    $class = $obj->callClass;
                    $method = $obj->callMethod;

                    # As soon as the received line of text has been displayed,
                    #   $self->processLineSegment can call $class->$method. Store the arguments to
                    #   be used in the call until then
                    push (
                        @dependentCallList,
                        [
                            $class,
                            $method,
                            $self,
                            $number,
                            $text,
                            $stripText,
                            $modText,
                            \@backRefList,
                            \@matchMinusList,
                            \@matchPlusList,
                        ],
                    );
                 }
            }

            if ($thisObj) {

                # The trigger did fire
                $lastObj = $thisObj;        # At least one trigger has fired

                # If it's an independent trigger and the 'style_mode' attribute is set, we need to
                #   apply the style to the whole (or part of) $modText (styles can't be applied to
                #   dependent triggers)
                $mode = $thisObj->ivShow('attribHash', 'style_mode');
                if ($thisObj->indepFlag && $mode) {

                    # Set the first and last character in $modText to which the style should be
                    #   applied. $start is the offset of the first character, $stop is the offset
                    #   of the character AFTER the last oneN (which may be outside $modText)
                    if ($mode == -1) {

                        # Apply the style to the whole line
                        $start = 0;
                        $stop = length ($modText);

                    } elsif ($mode == -2) {

                        # Apply the style to the matched text
                        $start = $matchMinusList[0];        # @-
                        $stop = $matchPlusList[0];          # @+

                    } else {

                        # Apply the style to matched substring
                        $start = $matchMinusList[$mode];    # @-
                        $stop = $matchPlusList[$mode];      # @+
                    }

                    if (defined $start && defined $stop) {

                        %oldTagHash = %tagHash;

                        ($result, %tagHash) = $self->applyTriggerStyle(
                            $thisObj,
                            $modText,
                            $start,
                            $stop,
                            %tagHash,
                        );

                        if (! defined $result) {

                            # In case ->applyTriggerStyle fails, restore values
                            %tagHash = %oldTagHash;
                        }
                    }
                }

                # Should we continue checking other triggers?
                if ($perlFlag || ! $thisObj->ivShow('attribHash', 'keep_checking')) {

                    # Don't check any more triggers
                    last OUTER;
                }
            }
        }

        if (! $lastObj) {

            # No trigger fired, so $modText is unaltered
            return @emptyList;

        } else {

            # Prepare the list of return values. Use attributes from the last trigger that fired
            if ($lastObj->ivShow('attribHash', 'gag')) {
                $gagFlag = TRUE;
            } else {
                $gagFlag = FALSE;
            }

            if ($lastObj->ivShow('attribHash', 'gag_log')) {
                $gagLogFlag = TRUE;
            } else {
                $gagLogFlag = FALSE;
            }

            # Any temporary triggers which fired can now be deleted
            foreach my $obj (@deleteList) {

                $self->removeInterface($obj);
            }

            return ($modText, $gagFlag, $gagLogFlag, \@instructList, \@dependentCallList, %tagHash);
        }
    }

    sub applyTriggerStyle {

        # Called by $self->checkTriggers when a trigger, whose 'style_mode' attribute is set, fires
        # Applies the style to the line (or a portion of the line)
        #
        # Expected arguments
        #   $obj            - The GA::Interface::Trigger object which has fired
        #   $line           - The line which caused the trigger to fire
        #   $start, $stop   - Offsets describing which portion of the line are subject to the
        #                       style. $start is the offset of the first character in the portion;
        #                       $stop is the offset of the character after the last character in
        #                       the portion (which may be outside $line)
        #
        # Optional arguments
        #   %tagHash        - A hash containing Axmud colour/style tags, in the form
        #                       $tagHash{position} = reference_to_list_of_Axmud_colour_&_style_tags
        #                     ...where $position is the position in $line where one or more colour
        #                       or style tags apply (the first character is position 0. An empty
        #                       hash if no colour/style tags apply to $line, yet)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list in the form
        #       (1, %tagHash)
        #   ...where 1 shows success, and %tagHash is the (possibly modified) hash of Axmud colour
        #       and style tags

        my ($self, $obj, $line, $start, $stop, %tagHash) = @_;

        # Local variables
        my (
            $textViewObj, $listRef, $flag, $attribsOffFlag,
            @emptyList, @posnList, @styleList,
            %styleHash, %styleOffHash, %attribHash, %prevHash, %startHash, %stopHash, %flagHash,
        );

        # Check for improper arguments
        if (! defined $obj || ! defined $line || ! defined $start || ! defined $stop) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->applyTriggerStyle', @_);
            return @emptyList;
        }

        # Import the current textview object (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # Get a sorted list of positions at which an Axmud colour/style tag applies
        @posnList = sort {$a <=> $b} (keys %tagHash);
        # A list of trigger styles (which include both Axmud colour tags and style tags) processed
        #   by this function
        @styleList = (
            'text',
            'underlay',
            'italics',
            'underline',
            'blink_slow',
            'blink_fast',
            'strike',
            'link',
        );

        # A hash of trigger styles that correspond to Axmud style tags (for quick lookup)
        %styleHash = (
            'italics'       => undef,
            'underline'     => undef,
            'blink_slow'    => undef,
            'blink_fast'    => undef,
            'strike'        => undef,
            'link'          => undef,
        );

        # A hash off Axmud style tags that turn off those style tags
        %styleOffHash = (
             # This tag        turns off this one
            'italics_off'   => 'italics',
            'underline_off' => 'underline',
            'blink_off'     => 'blink_slow',        # (blink_fast is handled as an exception)
            'strike_off'    => 'strike',
            'link_off'      => 'link',
        );

        # Get the trigger's current style attributes (->attribHash contains other key-value pairs,
        #   besides the style attributes)
        foreach my $style (@styleList) {

            # ->attribHash keys are 'style_text', 'style_blink_fast' etc
            $attribHash{$style} = $obj->ivShow('attribHash', 'style_' . $style);
        }

        if (! %attribHash) {

            # No trigger styles to apply
            return (1, %tagHash);
        }

        # PART 1
        # Fetch the colour and style tags that applied at the end of the previous line
        #   - For colour tags, 'undef' if default colours being used, or if there was no previous
        #       line)
        #   - For style tags, FALSE if off, TRUE if on
        # NB ->prevColourStyleHash can also contain 'dummy' style tags like 'bold' and
        #   'reverse_off', which we don't need here
        foreach my $style (@styleList) {

            # ->colourStyleHash keys are 'text', 'blink_fast', etc
            $prevHash{$style} = $textViewObj->ivShow('prevColourStyleHash', $style);
        }

        # PART 2
        # Work out which colours/styles apply at the start of the portion. Go through every offset
        #   in %tagHash in ascending order, until we reach an offset at the start of the portion
        #   ($start), or the nearest offset before it
        %startHash = %prevHash;

        OUTER: foreach my $posn (@posnList) {

            if ($posn > $start) {

                last OUTER;
            }

            # Get the list of Axmud colour/style tags that were applied at this offset
            $listRef = $tagHash{$posn};
            if (! defined $listRef) {

                # (If there is no list of colour/style tags at this offset, so create one)
                @$listRef = ();
            }

            # Work out the colours/styles that would actually be displayed at this offset, storing
            #   it in a hash in the same format as GA::Client->constColourStyleHash
            %startHash = $self->applyColourStyleTags(
                \%startHash,
                $listRef,
            );
       }

        # PART 3
        # Work out which colours/styles apply at the end of the portion. Go through every offset
        #   in %tagHash, ignoring those up to and including $start, and those after $stop
        %stopHash = %startHash;

        OUTER: foreach my $posn (@posnList) {

            if ($posn > $stop) {

                last OUTER;

            } elsif ($posn <= $start) {

                next OUTER;     # We've already checked these offsets in PART 2
            }

            # Get the list of Axmud colour/style tags that were applied at this offset
            $listRef = $tagHash{$posn};
            if (! defined $listRef) {

                # (If there is no list of colour/style tags at this offset, so create one)
                @$listRef = ();
            }

            # Work out the colours/styles that would actually be displayed at this offset, storing
            #   it in a hash in the same format as GA::Client->constColourStyleHash
            %stopHash = $self->applyColourStyleTags(
                \%stopHash,
                $listRef,
            );
        }

        # PART 4
        # Apply style/colour tags at the start of the portion; use the existing list of tags at
        #   that offset, and add new ones to the end of the list, as appropriate
        $listRef = $tagHash{$start};
        if (! defined $listRef) {

            # (If there is no list of tags at this offset, create one)
            @$listRef = ();
        }

        foreach my $trigStyle (keys %attribHash) {

            # Text colours
            if ($trigStyle eq 'text') {

                if (
                    $attribHash{'text'}
                    && (
                        ! $startHash{'text'}
                        || $startHash{'text'} ne $attribHash{'text'}
                    )
                ) {
                    # Change the text colour by adding a new tag to the list
                    push (@$listRef, $attribHash{'text'});
                }

            # Underlay colours
            } elsif ($trigStyle eq 'underlay') {

                if (
                    $attribHash{'underlay'}
                    && (
                        ! $startHash{'underlay'}
                        || $startHash{'underlay'} ne $attribHash{'underlay'}
                    )
                ) {
                    # Change the underlay colour by adding a new tag to the list
                    push (@$listRef, $attribHash{'underlay'});
                }

            # Styles
            } else {

                # A style
                if ($attribHash{$trigStyle} && ! $startHash{$trigStyle}) {

                    # Style is off, so turn it on
                    push (@$listRef, $trigStyle);

                } elsif (! $attribHash{$trigStyle} && $startHash{$trigStyle}) {

                    # Style is on, so turn it off
                    if ($trigStyle eq 'blink_slow' || $trigStyle eq 'blink_fast') {
                        push (@$listRef, 'blink_off');
                    } else {
                        push (@$listRef, $trigStyle . '_off');
                    }
                }
            }
        }

        $tagHash{$start} = $listRef;

        # PART 5
        # Apply style/colour tags at the end of the portion; use the existing list of tags at
        #   that offset, and add new ones to the end of the list, as appropriate
        $listRef = $tagHash{$stop};
        if (! defined $listRef) {

            # (If there is no list of tags at this offset, create one)
            @$listRef = ();
        }

        foreach my $trigStyle (keys %attribHash) {

            # Text colours
            if ($trigStyle eq 'text') {

                if (
                    $attribHash{'text'}
                    && (
                        ! $stopHash{'text'}
                        || $stopHash{'text'} ne $attribHash{'text'}
                    )
                ) {
                    # Change the text colour by adding a new tag to the list
                    if ($stopHash{'text'}) {
                        push (@$listRef, $stopHash{'text'});
                    } else {
                        push (@$listRef, $self->currentTabObj->textViewObj->textColour);
                    }
                }

            # Underlay colours
            } elsif ($trigStyle eq 'underlay') {

                if (
                    $attribHash{'underlay'}
                    && (
                        ! $stopHash{'underlay'}
                        || $stopHash{'underlay'} ne $attribHash{'underlay'}
                    )
                ) {
                    # Change the underlay colour by adding a new tag to the list
                    if ($stopHash{'underlay'}) {
                        push (@$listRef, $stopHash{'underlay'});
                    } else {
                        push (@$listRef, $self->currentTabObj->textViewObj->underlayColour);
                    }
                }

            # Styles
            } else {

                # A style
                if ($attribHash{$trigStyle} && ! $stopHash{$trigStyle}) {

                    # Style is on, so turn it off
                    if ($trigStyle eq 'blink_slow' || $trigStyle eq 'blink_fast') {
                        push (@$listRef, 'blink_off');
                    } else {
                        push (@$listRef, $trigStyle . '_off');
                    }

                } elsif (! $attribHash{$trigStyle} && $stopHash{$trigStyle}) {

                    # Style is on, so turn it off
                    push (@$listRef, $trigStyle);
                }
            }
        }

        $tagHash{$stop} = $listRef;

        # Operation complete
        return (1, %tagHash);
    }

    sub checkAliases {

        # Called by $self->worldCmd
        # Checks every active alias interface. Fires every alias that should be fired in response to
        #   a world command (before it is sent to the world).
        #
        # Checking against aliases starts with the first alias in $self->aliasOrderList, and ends
        #   when:
        #   - the first matching alias whose 'keep_checking' attribute is TRUE, or
        #   - the first matching alias whose 'response' attribute evaluates as a Perl command, or
        #   - the end of the active alias list is reached
        #
        # The specified command, $originalCmd, gets converted into an instruction that should be:
        #   - the original world command, which is returned as a string
        #   - an instruction containing one or more modified world commands, e.g. 'north' or
        #       'north;eat cake;kill orc', which is returned as a string
        #   - an instruction consisting of a Perl command which is evaluated, and the return value
        #       returned to be used as a world command
        # If the return value starts with any kind of sigil, the calling function will ignore it
        #   - only world commands are actually sent to the world
        #
        # Expected arguments
        #   $originalCmd    - The original world command (always a single world command, not a
        #                       sequence of world commands)
        #
        # Return values
        #   'undef' on improper arguments or if no command is to be sent to the world
        #   Otherwise, returns the command to send to the world (which will be the same as
        #       $originalCmd, if no aliases fired)

        my ($self, $originalCmd, $check) = @_;

        # Local variables
        my (
            $cmd, $depFireFlag, $returnValue,
            @deleteList,
            %hash,
        );

        # Check for improper arguments
        if (! defined $originalCmd || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkAliases', @_);
        }

        # Import the active alias registry for quick lookup
        %hash = $self->aliasHash;

        # Check every active alias interface, in the correct order, to see if it is due to fire
        $cmd = $originalCmd;
        OUTER: foreach my $number ($self->aliasOrderList) {

            my (
                $obj, $regex, $substitution, $result, $class, $method, $safeFlag,
                @backRefList,
            );

            $obj = $self->ivShow('interfaceNumHash', $number);
            $regex = $obj->stimulus;

            # If the alias is disabled, do nothing
            if (! $obj->enabledFlag) {

               next OUTER;
            }

            if ($obj->indepFlag) {

                # An independent alias

                # If the alias ->response begins with a forward slash, checking against other
                #   aliases halts and the Perl command is evaluated (at the end of this function)
                # Otherwise, we perform a substitution. If the resulting string begins with a
                #   forward slash, checking against other aliases halts, and the Perl command is
                #   evaluated
                # Otherwise, the resulting string is checked against the next alias (if allowed).
                #   The last resulting string is used as an instruction
                if (substr ($regex, 0, 1) eq '/') {

                    # An independent alias has fired
                    $depFireFlag = FALSE;
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        # A temporary alias, which must be deleted when it first fires
                        push (@deleteList, $obj);
                    }

                    # Don't check against any more aliases
                    $cmd = $regex;
                    last OUTER;
                }

                # Otherwise, perform the substitution, using the alias's ->stimulus as a regex. If
                #   the $result is different from $cmd, we say the alias has fired
                $substitution = $obj->response;
                $result = $cmd;
                # If $substitution is enclosed in double-quotes, it's safe to use the /ee modifier
                #   which will convert $1, $2 etc into the contents of any matched backreferences
                #   (otherwise we could end up executing arbitrary Perl code)
                if (
                    substr($substitution, 0, 1) eq '"'
                    && substr($substitution, -1) eq '"'
                ) {
                    $safeFlag = TRUE;
                }

                if ($obj->ivShow('attribHash', 'ignore_case')) {

                    if ($safeFlag) {
                        $result =~ s/$regex/$substitution/iee;
                    } else {
                        $result =~ s/$regex/$substitution/i;
                    }

                } else {

                    if ($safeFlag) {
                        $result =~ s/$regex/$substitution/ee;
                    } else {
                        $result =~ s/$regex/$substitution/;
                    }
                }

                if ($result ne $cmd) {

                    # An independent alias has fired
                    $depFireFlag = FALSE;
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        # A temporary alias, which must be deleted when it first fires
                        push (@deleteList, $obj);
                    }

                    # Any further alias substitutions are performed on the result, not the original
                    #   command
                    $cmd = $result;

                    # Should we continue checking other aliases?
                    if (
                        $obj->ivShow('attribHash', 'keep_checking')
                        || substr ($result, 0, 1) eq '/'
                    ) {
                        # Don't check any more aliases
                        last OUTER;
                    }
                }

            } else {

                # A dependent alias
                # Perform the patern match, using the alias's ->stimulus as a regex. If TRUE, call
                #   some part of the Axmud code
                if ($obj->ivShow('attribHash', 'ignore_case')) {
                    @backRefList = ($cmd =~ m/$regex/i);
                } else {
                    @backRefList = ($cmd =~ m/$regex/);
                }

                if (@backRefList) {

                    # A dependent alias has fired
                    $depFireFlag = TRUE;
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        # A temporary alias, which must be deleted when it first fires
                        push (@deleteList, $obj);
                    }

                    # Call the specified function
                    $class = $obj->callClass;
                    $method = $obj->callMethod;
                    $class->$method($self, $number, @backRefList);

                    # Should we continue checking other aliases?
                    if ($obj->ivShow('attribHash', 'keep_checking')) {

                        # Don't check any more aliases
                        last OUTER;
                    }
                }
            }
        }

        # Any temporary aliases which fired can now be deleted
        foreach my $obj (@deleteList) {

            $self->removeInterface($obj);
        }

        # If at least one alias fired, and the last alias to fire was a dependent alias, we return
        #   'undef' to show that there's no command to be sent to the world
        if ($depFireFlag) {

            return undef;

        # Otherwise, if the modified $cmd begins with a forward slash, evaluate it like a Perl
        #   command and use the return value as a world command
        # (However, if the original command began with a forward slash, it's because the user
        #   typed something like ',,/create', meaning 'send everything after the ,, as a literal
        #   world command' - so don't try to execute a Perl command
        } elsif (substr ($cmd, 0, 1) eq '/' && substr($originalCmd, 0, 1) ne '/') {

            # Store data so that it's available to the Perl mini-programme
            $self->ivAdd('perlCmdDataHash', '_originalCmd', $originalCmd);
            # (Also store the interface which fired)
            $self->ivAdd('perlCmdDataHash', '_interface', undef);

            $returnValue = $self->perlCmd($cmd);

            # The stored data is no longer needed
            $self->ivAdd('perlCmdDataHash', '_originalCmd', undef);
            $self->ivAdd('perlCmdDataHash', '_interface', undef);

            return $returnValue;        # If 'undef', no command is sent to the world

        } else {

            # Return the original command (or the modified instruction, if any alias has fired)
            return $cmd;
        }
    }

    sub checkMacros {

        # Called by ->signal_connect in GA::Win::Internal->setKeyPressEvent
        # Checks every active macro interface. Fires every macro that should be fired in response to
        #   a keypress
        #
        # Expected arguments
        #   $keycode    - The standard keycode, e.g. 'F5' (or keycode string, e.g. 'ctrl shift F5')
        #                   representing the keypress
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns FALSE if no macros fire and TRUE if any macros fire

        my ($self, $keycode, $check) = @_;

        # Local variables
        my (
            $fireFlag,
            @deleteList,
            %hash,
        );

        # Check for improper arguments
        if (! defined $keycode || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkMacros', @_);
        }

        # Import the active macro registry for quick lookup
        %hash = $self->macroHash;

        # Check every active macro interface, in the correct order, to see if it is due to fire
        $fireFlag = FALSE;
        OUTER: foreach my $number ($self->macroOrderList) {

            my ($obj, $class, $method);

            if ($hash{$number} eq $keycode) {

                # The macro keycode matches. Check the interface number is valid
                if (! $self->ivExists('interfaceNumHash', $number)) {

                    $self->writeError(
                        'Invalid active macro interface #' . $number,
                        $self->_objClass . '->checkMacros',
                    );

                    # Avoid multiple error messages by stopping checking macros now
                    last OUTER;

                } else {

                    $obj = $self->ivShow('interfaceNumHash', $number);
                }

                # If the macro is disabled, do nothing
                if ($obj->enabledFlag) {

                    $fireFlag = TRUE;

                    # Store the keycode so that it's available to a macro response that starts with
                    #   '/' (meaning it's a Perl mini-programme)
                    $self->ivAdd('perlCmdDataHash', '_keycode', $keycode);
                    # (Also store the interface which fired)
                    $self->ivAdd('perlCmdDataHash', '_interface', $number);

                    # For an independent macro, perform the instruction in ->response
                    if ($obj->indepFlag) {

                        $self->doInstruct($obj->response);

                    # For a dependent macro, make a function call
                    } else {

                        $class = $obj->callClass;
                        $method = $obj->callMethod;

                        $class->$method($self, $obj->number, $keycode);
                    }

                    # $keycode is no longer needed
                    $self->ivAdd('perlCmdDataHash', '_keycode', undef);
                    $self->ivAdd('perlCmdDataHash', '_interface', undef);

                    # Mark a temporary active macro to be deleted
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        push (@deleteList, $obj);
                    }
                }
            }
        }

        # Any temporary macros which fired can now be deleted
        foreach my $obj (@deleteList) {

            $self->removeInterface($obj);
        }

        return $fireFlag;
    }

    sub checkTimers {

        # Called by $self->spinTimerLoop
        # Checks every active timer interface. Calls $self->checkTimers for every timer that's due
        #   to be fired
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns FALSE if no timers fire and TRUE if any timers fire

        my ($self, $check) = @_;

        # Local variables
        my (
            $time, $fireFlag,
            %hash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkTimers', @_);
        }

        # Import IVs for quick lookup
        $time = $self->sessionTime;
        %hash = $self->timerHash;

        # Check every active timer interface, in the correct order, to see if it is due to fire
        $fireFlag = FALSE;
        OUTER: foreach my $number ($self->timerOrderList) {

            my ($obj, $class, $method, $count, $minDelay, $randDelay);

            if ($hash{$number} < $time) {

                # The timer is due to fire. Check the number is valid
                if (! $self->ivExists('interfaceNumHash', $number)) {

                    $self->writeError(
                        'Invalid active timer interface #' . $number,
                        $self->_objClass . '->checkTimers',
                    );

                    # Avoid multiple error messages by stopping checking timers now
                    last OUTER;

                } else {

                    $obj = $self->ivShow('interfaceNumHash', $number);
                }

                # If the timer is disabled, or if the timer is waiting for the character to log in,
                #   do nothing
                if ($obj->ivShow('attribHash', 'wait_login') && ! $self->loginFlag) {

                    # Don't check again for at least another second
                    $self->ivAdd('timerHash', $number, $self->sessionTime + 1);

                    next OUTER;
                }

                if ($obj->enabledFlag) {

                    $fireFlag = TRUE;

                    # Store the times so they're available to a timer response that starts with '/'
                    #   (meaning it's a Perl mini-programme)
                    $self->ivAdd('perlCmdDataHash', '_timerExpect', $time);
                    $self->ivAdd('perlCmdDataHash', '_timerTime', $self->sessionTime);
                    # (Also store the interface which fired)
                    $self->ivAdd('perlCmdDataHash', '_interface', $number);

                    # For an independent timer, perform the instruction in ->response
                    if ($obj->indepFlag) {

                        $self->doInstruct($obj->response);

                    # For a dependent timer, make a function call
                    } else {

                        $class = $obj->callClass;
                        $method = $obj->callMethod;

                        $class->$method($self, $obj->number, $time, $self->sessionTime);
                    }

                    # Set the timer's repeat count
                    $count = $obj->ivShow('attribHash', 'count');
                    if ($count == 1) {

                        # If the 'temporary' attribute is TRUE, delete the timer
                        if ($obj->ivShow('attribHash', 'temporary')) {

                            $self->removeInterface($obj);

                        # Otherwise, just disable it
                        } else {

                            $obj->modifyAttribs(
                                $self,
                                'enabled'   => FALSE,
                                # When reenabled, the repeat count will be 'repeat forever'
                                'count'     => -1,
                            );

                            # Set the time at which the timer will fire again (if it is not now
                            #   disabled) by replacing the entry in the active timer registry
                            $self->ivAdd('timerHash', $obj->number, ($time + $obj->stimulus));
                        }

                    } else {

                        # If the repeat count is not -1 (which means repeat forever), the number of
                        #   times the timer must fire should be decreased by 1
                        if ($count > -1) {

                            $count--;
                            $obj->modifyAttribs(
                                $self,
                                'count', $count,
                            );
                        }

                        # Set the time at which the timer will fire again by replacing the entry in
                        #   the active timer registry
                        if ($obj->ivShow('attribHash', 'random_delay')) {

                            # Get the minimum delay (may be 0)
                            $minDelay = $obj->ivShow('attribHash', 'random_min');
                            # Check: if the minimum random delay is greater than $obj->stimulus,
                            #   ignore it
                            if ($minDelay >= $obj->stimulus) {

                                $minDelay = 0;
                            }

                            $randDelay = rand($obj->stimulus - $minDelay) + $minDelay;
                            $self->ivAdd('timerHash', $obj->number, ($time + int($randDelay)));

                        } else {

                            $self->ivAdd('timerHash', $obj->number, ($time + $obj->stimulus));
                        }
                    }

                    # The times are no longer needed
                    $self->ivAdd('perlCmdDataHash', '_timerExpect', undef);
                    $self->ivAdd('perlCmdDataHash', '_timerTime', undef);
                    $self->ivAdd('perlCmdDataHash', '_interface', undef);
                }
            }
        }


        return $fireFlag;
    }

    sub checkHooks {

        # Called by various functions for various hook events
        #
        # NB If the list below is modified, GA::InterfaceModel::Hook->new and
        #   GA::Cmd::SimulateHook->do must be updated, too
        #
        #   HOOK EVENT          FUNCTIONS                   HOOK DATA
        #   connect             $self->connectionComplete   (none)
        #   disconnect          $self->reactDisconnect      (none)
        #   login               $self->doLogin              (none)
        #   prompt              $self->processPrompt        Line, stripped of escape sequences
        #   receive_text        $self->processLineSegment   Line, stripped of escape sequences
        #   sending_cmd         $self->worldCmd, ->teleportCmd
        #                                                   The command being processed
        #   send_cmd            $self->dispatchCmd          The command to be sent
        #   msdp                $self->processMsdpData      The variable/value pair received
        #   mssp                $self->processMsspData      The variable/value pair received
        #   atcp                $self->processAtcpData      The ATCP package name
        #   gmcp                $self->processGmcpData      The GMCP package name
        #   current_session     GA::Client->setCurrentSession
        #                                                   (none)
        #   not_current         GA::Client->setCurrentSession
        #                                                   The new current session's ->number
        #   change_current      GA::Client->setCurrentSession
        #                                                   The new current session's ->number
        #   visible_session     GA::Win::Internal->setVisibleSession
        #                                                   (none)
        #   not_visible         GA::Win::Internal->setVisibleSession
        #                                                   The new visible session's ->number
        #   change_visible      GA::Win::Internal->setVisibleSession
        #                                                   The new visible session's ->number
        #   user_idle           $self->spinTimerLoop        $self->lastCmdTime
        #   world_idle          $self->spinTimerLoop        $self->lastDisplayTime
        #   get_focus           GA::Win::Internal->setFocusInEvent
        #                                                   (none)
        #   lose_focus          GA::Win::Internal->setFocusOutEvent
        #                                                   (none)
        #   close_disconnect    GA::Client->stop          (none)
        #
        # Checks every active hook interface. Calls $self->checkHooks for every hook that has
        #   fired in response to a specified hook event
        #
        # Expected arguments
        #   $event      - The hook event that has taken place (matches a key in
        #                   GA::InterfaceModel::Hook->hookEventHash)]
        #
        # Optional arguments
        #   $hookVar    - Some hook events (such as 'receive_text') need to pass on some data
        #                   concerning the event (in this example, the text received). If so,
        #                   it is stored in $hookVar; otherwise set to 'undef'
        #   $hookVal    - Some hook events ('msdp' and 'mssp') need to pass on a variable-value
        #                   pair. If so, they are stored in $hookVar and $hookVal; otherwise
        #                   $hookVal is set to 'undef'
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns FALSE if no hooks fire and TRUE if any hooks fire

        my ($self, $event, $hookVar, $hookVal, $check) = @_;

        # Local variables
        my (
            $fireFlag,
            @deleteList,
            %hash,
        );

        # Check for improper arguments
        if (! defined $event || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkHooks', @_);
        }

        # Import the active hook registry for quick lookup
        %hash = $self->hookHash;

        # Check every active hook interface, in the correct order, to see if responds to this hook
        #   $event
        $fireFlag = FALSE;
        OUTER: foreach my $number ($self->hookOrderList) {

            my ($obj, $class, $method);

            if ($hash{$number} eq $event) {

                # The hook event matches. Check the number is valid
                if (! $self->ivExists('interfaceNumHash', $number)) {

                    $self->writeError(
                        'Invalid active hook interface #' . $number,
                        $self->_objClass . '->checkHooks',
                    );

                    # Avoid multiple error messages by stopping checking macros now
                    last OUTER;

                } else {

                    $obj = $self->ivShow('interfaceNumHash', $number);
                }

                # If the hook is disabled, do nothing
                if ($obj->enabledFlag) {

                    $fireFlag = TRUE;

                    # If hook data was specified, store it so that it's available to a hook response
                    #   that starts with '/' (meaning it's a Perl mini-programme)
                    if ($hookVar) {

                        $self->ivAdd('perlCmdDataHash', '_hookVar', $hookVar);
                    }

                    if ($hookVal) {

                        $self->ivAdd('perlCmdDataHash', '_hookVal', $hookVal);
                    }

                    # Also store the hook event
                    $self->ivAdd('perlCmdDataHash', '_hookEvent', $obj->stimulus);
                    # (Also store the interface which fired)
                    $self->ivAdd('perlCmdDataHash', '_interface', $number);

                    # For an independent hook, perform the instruction in ->response
                    if ($obj->indepFlag) {

                        $self->doInstruct($obj->response);

                    # For a dependent hook, make a function call
                    } else {

                        $class = $obj->callClass;
                        $method = $obj->callMethod;

                        $class->$method($self, $obj->number, $hookVar, $hookVal);
                    }

                    # If hook data was specified, it's no longer needed
                    if ($hookVar) {

                        $self->ivAdd('perlCmdDataHash', '_hookVar', undef);
                    }

                    if ($hookVal) {

                        $self->ivAdd('perlCmdDataHash', '_hookVal', undef);
                    }

                    $self->ivAdd('perlCmdDataHash', '_hookEvent', undef);
                    $self->ivAdd('perlCmdDataHash', '_interface', undef);

                    # Mark a temporary active hook to be deleted
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        push (@deleteList, $obj);
                    }
                }
            }
        }

        # Any temporary hooks which fired can now be deleted
        foreach my $obj (@deleteList) {

            $self->removeInterface($obj);
        }

        return $fireFlag;
    }

    # Cages

    sub createCages {

        # Called by $self->setupProfiles
        # Also called by GA::Cmd::SetupWorld->do (etc), AddGuild->do (etc), CloneGuild->do (etc),
        #   SetupCustomProfile->do, AddCustomProfile->do, CloneCustomProfile->do
        #
        # When a new profile is created, this function creates all the associated cage objects
        # If the new profile is a current profile, marks the new objects as the current cage
        #
        # Expected arguments
        #   $profObj        - blessed reference of the new profile
        #
        # Optional arguments
        #   $currentFlag    - TRUE if this profile will be set as the current profile, FALSE (or
        #                       'undef') if not
        #
        # Return values
        #   'undef' on improper arguments
        #   1 upon success

        my ($self, $profObj, $currentFlag, $check) = @_;

        # Local variables
        my ($profName, $profCategory);

        # Check for improper arguments
        if (! defined $profObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->createCages', @_);
        }

        # Import IVs
        $profName = $profObj->name;
        $profCategory = $profObj->category;

        # Create one new cage for each cage type
        foreach my $type ($axmud::CLIENT->cageTypeList) {

            my ($package, $obj);

            if ($axmud::CLIENT->ivExists('pluginCagePackageHash', $type)) {

                # Cage added by a plugin
                $package = $axmud::CLIENT->ivShow('pluginCagePackageHash', $type);

            } else {

                # Built-in cage
                $package = 'Games::Axmud::Cage::' . ucfirst($type);
            }

            $obj = $package->new($self, $profName, $profCategory);
            if (! $obj) {

                $self->writeWarning(
                    'Failed to create the \'' . $package . '\' cage for the \'' . $profName
                    . '\' profile',
                    $self->_objClass . '->createCages',
                );

            } else {

                # Use the 'set' accessor rather than ->ivAdd so that other sessions using the same
                #   world profile are updated, too
                $self->add_cage($obj);
            }
        }

        # If this is a current profile, and there were no errors, mark the new objects as current
        #   cages and set the inferior cage for all cages (those not belonging to a current profile
        #   have their inferior cage set to 'undef')
        if ($currentFlag) {

            $self->setCurrentCages($profName, $profCategory);
            $self->setCageInferiors();

            # Create new interfaces for this profile
            $self->setProfileInterfaces($profObj->name)
        }

        return 1;
    }

    sub updateCages {

        # Called by $self->setupProfiles, GA::Client->addPluginCages are by code in any plugin
        # If the user writes a plugin which adds new cages, existing profiles each need to have one
        #   of these cages created for it
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $silenceFlag    - If set to TRUE, doesn't display a message for each cage created
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $silenceFlag, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateCages', @_);
        }

        foreach my $profObj ($self->ivValues('profHash')) {

            # Check every type of cage. If a cage associated with the profile doesn't exist, create
            #   it
            foreach my $type ($axmud::CLIENT->cageTypeList) {

                my ($uniqueName, $package, $obj);

                $uniqueName = lc($type) . '_' . $profObj->category . '_' . $profObj->name;

                if (! $self->ivExists('cageHash', $uniqueName)) {

                    # Cage doesn't already exist, so create it
                    if ($axmud::CLIENT->ivExists('pluginCagePackageHash', $type)) {

                        # Cage added by a plugin
                        $package = $axmud::CLIENT->ivShow('pluginCagePackageHash', $type);

                    } else {

                        # Built-in cage
                        $package = 'Games::Axmud::Cage::' . ucfirst($type);
                    }

                    $obj = $package->new($self, $profObj->name, $profObj->category);
                    if (! $obj) {

                        # Show a warning message, even if $silenceFlag is TRUE
                        $self->writeWarning(
                            'Failed to create the \'' . $package . '\' cage for the \''
                            . $profObj->name . '\' profile',
                            $self->_objClass . '->updateCages',
                        );

                    } else {

                        # Add the new cage to this session's registries
                        # Use the 'set' accessor rathern than ->ivAdd so that other sessions using
                        #   the same world profile are updated, too
                        $self->add_cage($obj);

                        if (! $silenceFlag) {

                            $self->writeText('Created cage \'' . $uniqueName . '\'');
                        }
                    }
                }
            }
        }

        # Set current cages for current profiles
        foreach my $profObj ($self->ivValues('currentProfHash')) {

            $self->setCurrentCages($profObj->name, $profObj->category);
        }

        # Set all the cage inferiors
        $self->setCageInferiors();

        # Set correct active interfaces for every current profile
        $self->setProfileInterfaces();

        return 1;
    }

    sub deleteCages {

        # Called by GA::Cmd::DeleteCage->do
        # Users can writes plugins which add new types of cages. $self->updateCages is called to
        #   make sure all profiles have one of these new cages
        # If the user decides to stop using a plugin, or if the user wants to delete one of the
        #   types of cage specified by a plugin, this function is called
        # All existing profiles have their copies of the specified cage type destroyed. If no type
        #   is specified, all cages of unrecognised types are destroyed
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $type           - The cage type to destroy (matches a key in
        #                       GA::Client->pluginCageHash). If 'undef', all cages of unrecognised
        #                       types are destroyed
        #   $silenceFlag    - If set to TRUE, doesn't display a message for each cage destroyed
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the number of cages destroyed (may be 0)

        my ($self, $type, $silenceFlag, $check) = @_;

        # Local variables
        my $count;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->deleteCages', @_);
        }

        # Now go through the entire cage registry, removing any cages of an unrecognised type (i.e.
        #   remove all 'mycage' cages, if the user has removed 'mycage' from
        #   GA::Client->pluginCageHash)
        $count = 0;
        foreach my $cageName ($self->ivKeys('cageHash')) {

            my (
                $thisType,
                @list,
            );

            # The cage name is in the form cageType_profCategory_profName. We're only interested in
            #   the cage type
            @list = split(m/_/, $cageName);

            if (
                (defined $type && $list[0] eq $type)
                || (
                    ! defined $type
                    && ! defined $axmud::CLIENT->ivFind('cageTypeList', $list[0])
                )
            ) {
                # Use the 'set' accessor rathern than ->ivDelete so that other sessions using the
                #   same world profile are updated, too
                $self->del_cage($cageName);
                $count++;

                if (! $silenceFlag) {

                    $self->writeText('Deleted cage \'' . $cageName . '\'');
                }
            }
        }

        # Set current cages for current profiles
        foreach my $profObj ($self->ivValues('currentProfHash')) {

            $self->setCurrentCages($profObj->name, $profObj->category);
        }

        # Set all the cage inferiors
        $self->setCageInferiors();

        # Set correct active interfaces for every current profile
        $self->setProfileInterfaces();

        return $count;
    }

    sub setCurrentCages {

        # Called by $self->createCages
        # Also called by GA::SetRace->do, SetGuild->do, SetChar->do and SetCustomProfile->do
        # Marks a set of cages associated with a profile as the current cages for this category of
        #   profile
        #
        # Expected arguments
        #   $profName       - Profile's unique name
        #   $profCategory   - Profile's category, e.g. 'world', 'race', 'faction'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $profName, $profCategory, $check) = @_;

        # Check for improper arguments
        if (! defined $profName || ! defined $profCategory || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setCurrentCages', @_);
        }

        foreach my $type ($axmud::CLIENT->cageTypeList) {

            my $uniqueName = lc($type) . '_' . $profCategory . '_' . $profName;
            if ($self->ivExists('cageHash', $uniqueName)) {

                $self->ivAdd(
                    'currentCageHash',
                    $uniqueName,
                    $self->ivShow('cageHash', $uniqueName),
                );
            }
        }

        return 1;
    }

    sub unsetCurrentCages {

        # Called by GA::Cmd::UnsetGuild->do (etc) and UnsetCustomProfile->do
        # When a current profile (that's not a world profile) is unset - as opposed to being
        #   deleted - its cages are not destroyed (as they are by $self->destroyCages), but merely
        #   marked as being no longer current
        #
        # Expected arguments
        #   $profName       - Profile's unique name
        #   $profCategory   - Profile's category, e.g. 'guild', 'race', 'faction' (but not
        #                       'world')
        #
        # Return values
        #   'undef' on improper arguments, or if the specified profile is a world profile
        #   1 otherwise

        my ($self, $profName, $profCategory, $check) = @_;

        # Check for improper arguments
        if (! defined $profName || ! defined $profCategory || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->unsetCurrentCages', @_);
        }

        # World profiles not allowed in this function
        if ($profCategory eq 'world') {

            return $self->writeError(
                'World profile specified, but can\'t be processed',
                $self->_objClass . '->unsetCurrentCages',
            );
        }

        # Unset current cages
        foreach my $type ($axmud::CLIENT->cageTypeList) {

            my $uniqueName = lc($type) . '_' . $profCategory . '_' . $profName;
            if ($self->ivExists('currentCageHash', $uniqueName)) {

                $self->ivDelete('currentCageHash', $uniqueName);
            }
        }

        return 1;
    }

    sub cloneCages {

        # Called by GA::Cmd::CloneWorld->do (etc) and GA::Generic::Cmd->cloneProfile
        #
        # When a profile is cloned with one of the above commands, this function clones all the
        #   related cage objects
        #
        # Expected arguments
        #   $profObj        - blessed reference to the original profile
        #   $clonedProfObj  - blessed reference to the cloned profile
        #
        # Optional arguments
        #   $cageHashRef   - When called by GA::Cmd::CloneWorld->do, the contents of this session's
        #                       ->cageHash IV, which has temporarily been emptied. Otherwise set to
        #                       'undef'
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns a list of blessed references to cloned cages

        my ($self, $profObj, $clonedProfObj, $cageHashRef, $check) = @_;

        # Local variables
        my (
            $profCategory,
            @returnArray,
            %cageHash,
        );

        # Check for improper arguments
        if (! defined $profObj || ! defined $clonedProfObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->cloneCages', @_);
        }

        # Import IVs
        $profCategory = $profObj->category;
        if (defined $cageHashRef) {
            %cageHash = %$cageHashRef;
        } else {
            %cageHash = $self->cageHash;
        }

        # Clone cages
        foreach my $type ($axmud::CLIENT->cageTypeList) {

            my ($uniqueName, $obj, $clonedObj);

            $uniqueName = lc($type) . '_' . $profCategory . '_' . $profObj->name;

            $obj = $cageHash{$uniqueName};
            if ($obj) {

                $clonedObj = $obj->clone($self, $clonedProfObj->name, $clonedProfObj->category);
            }

            if (! $obj || ! $clonedObj) {

                $self->writeWarning(
                    'Failed to clone the \'' . $uniqueName . '\' cage',
                    $self->_objClass . '->cloneCages',
                );

            } else {

                push (@returnArray, $clonedObj);
            }
        }

        return @returnArray;
    }

    sub destroyCages {

        # Called by $self->setupProfiles
        # Called by GA::Cmd::SetGuild->do (etc) and AddGuild->do (etc)
        # Called by GA::Cmd::SetCustomProfile->do and AddCustomProfile->do
        #
        # Deletes all cages associated with a particular profile (but not the profile itself)
        #
        # Expected arguments
        #   $profObj        - blessed reference to the profile
        #
        # Optional arguments
        #   $currentFlag    - TRUE if this profile is a current profile, FALSE (or 'undef') if not
        #
        # Return values
        #   'undef' on improper arguments
        #   1 on success

        my ($self, $profObj, $currentFlag, $check) = @_;

        # Local variables
        my ($profName, $profCategory);

        # Check for improper arguments
        if (! defined $profObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->destroyCages', @_);
        }

        # Import IVs
        $profName = $profObj->name;
        $profCategory = $profObj->category;

        # Destroy cages
        foreach my $type ($axmud::CLIENT->cageTypeList) {

            my $uniqueName = lc($type) . '_' . $profCategory . '_' . $profName;

            if ($self->ivExists('cageHash', $uniqueName)) {

                # Use the 'set' accessor rathern than ->ivDelete so that other sessions using the
                #   same world profile are updated, too
                $self->del_cage($uniqueName, $self->ivShow('cageHash', $uniqueName));
            }
        }

        if ($currentFlag) {

            # All remaining cages must have their inferior cages re-set
            $self->setCageInferiors();
        }

        return 1;
    }

    sub setCageInferiors {

        # Called by $self->createCages and $self->destroyCages
        # Also called by GA::Cmd::SetupRace->do, SetupGuild->do, SetupChar->do and
        #   SetupCustomProfile->do
        # Every time a profile is set to the current profile, every time a current profile is
        #   destroyed, and every time the profile priority list changes, all cages must have their
        #   inferior cages reset (which involves re-building $self->inferiorCageHash from scratch)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my %newHash;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setCageInferiors', @_);
        }

        # Empty this session's hash of inferior cages
        $self->ivEmpty('inferiorCageHash');

        # Rebuild the hash
        OUTER: foreach my $category ($self->profPriorityList) {

            # Check each possible inferior in turn...
            MIDDLE: foreach my $inferiorCategory ($self->findInferiorList($category)) {

                if (
                    $self->ivExists('currentProfHash', $category)
                    && $self->ivExists('currentProfHash', $inferiorCategory)
                ) {
                    INNER: foreach my $cageType ($axmud::CLIENT->cageTypeList) {

                        my ($cageName, $inferiorCageName);

                        $cageName = lc($cageType) . '_' . $category . '_'
                            . $self->ivShow('currentProfHash', $category)->name;
                        $inferiorCageName = lc($cageType) . '_' . $inferiorCategory
                            . '_' . $self->ivShow('currentProfHash', $inferiorCategory)->name;

                        # If the inferior has been created, mark it as inferior
                        if (
                            $self->ivExists('cageHash', $cageName)
                            && $self->ivExists('cageHash', $inferiorCageName)
                        ) {
                            $newHash{$cageName} = $self->ivShow('cageHash', $inferiorCageName);
                        }
                    }

                    next OUTER;
                }
            }
        }

        $self->ivPoke('inferiorCageHash', %newHash);

        return 1;
    }

    sub findCage {

        # Can be called by anything
        # Finds the cage of a specified type, associated with a specified profile
        #
        # Expected arguments
        #   $type       - one of the cage types in GA::Client->cageTypeList (e.g. 'Cmd'); $type is
        #                   treated as case-insensitive (so 'cmd' is ok)
        #   $profName   - the name of the associate profile
        #
        # Return values
        #   'undef' on improper arguments, if the cage $type doesn't exist, if the profile $profName
        #       doesn't exist, or if (for some reason) a cage associated with it doesn't exist
        #   Otherwise returns the blessed reference of the cage

        my ($self, $type, $profName, $check) = @_;

        # Local variables
        my ($match, $profile, $uniqueName);

        # Check for improper arguments
        if (! defined $type || ! defined $profName || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findCage', @_);
        }

        # Check that $type is a recognised cage type; at the same time, if $type is in lower-case,
        #   change it (e.g. convert 'cmd' to 'Cmd')
        OUTER: foreach my $item ($axmud::CLIENT->cageTypeList) {

            if (lc ($item) eq lc ($type)) {

                $match = $item;
                last OUTER;
            }
        }

        if (! $match) {

            # $type isn't a recognised cage type
            return undef;

        } else {

            $type = $match;     # Used the capitalised form
        }

        # Check that the profile $profName exists
        if (! $self->ivExists('profHash', $profName)) {

            # Doesn't exist
            return undef;

        } else {

            $profile = $self->ivShow('profHash', $profName);
        }

        # Check that the cage exists
        $uniqueName = lc($type) . '_' . $profile->category . '_' . $profName;
        if (! $self->ivExists('cageHash', $uniqueName)) {

            # Cage matching $type and $profName doesn't exist
            return undef;

        } else {

            # Matching cage exists
            return $self->ivShow('cageHash', $uniqueName)
        }
    }

    sub findCurrentCage {

        # Can be called by anything
        # Finds the cage of a specified type, associated with the current profile of a specified
        #   category
        #
        # Expected arguments
        #   $type       - one of the cage types in GA::Client->cageTypeList (e.g. 'Cmd'); $type is
        #                   treated as case-insensitive (so 'cmd' is ok)
        #   $category   - A category of profile, e.g. 'world', 'guild'
        #
        # Return values
        #   'undef' on improper arguments, if the cage $type doesn't exist, if the profile $category
        #       doesn't exist or if (for some reason) there isn't a current profile of that
        #       category, or if there isn't a current cage matching them
        #   Otherwise returns the blessed reference of the current cage

        my ($self, $type, $category, $check) = @_;

        # Local variables
        my ($match, $profile, $uniqueName);

        # Check for improper arguments
        if (! defined $type || ! defined $category || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findCurrentCage', @_);
        }

        # Check that $type is a recognised cage type; at the same time, if $type is in lower-case,
        #   change it (e.g. convert 'cmd' to 'Cmd')
        OUTER: foreach my $item ($axmud::CLIENT->cageTypeList) {

            if (lc ($item) eq lc ($type)) {

                $match = $item;
                last OUTER;
            }
        }

        if (! $match) {

            # $type isn't a recognised cage type
            return undef;

        } else {

            $type = $match;     # Used the capitalised form
        }

        # Check that the profile $category exists and that there's a current profile of that
        #   category
        if (! $self->ivExists('currentProfHash', $category)) {

            # Doesn't exist
            return undef;

        } else {

            $profile = $self->ivShow('currentProfHash', $category);
        }

        # Check that the cage exists
        $uniqueName = lc($type) . '_' . $category . '_' . $profile->name;
        if (! $self->ivExists('cageHash', $uniqueName)) {

            # Cage matching $type and $profName doesn't exist
            return undef;

        } else {

            # Matching cage exists
            return $self->ivShow('cageHash', $uniqueName)
        }
    }

    sub findHighestCage {

        # Can be called by anything
        # Finds the highest-priority existing cage of a given type
        #
        # Expected arguments
        #   $type       - One of the cage types in GA::Client->cageTypeList (e.g. 'Cmd'); $type is
        #                   treated as case-insensitive (so 'cmd' is ok)
        #
        # Return values
        #   'undef' on improper arguments, if $type is an unrecognised cage type or if no cage of
        #       that type is found matching current profiles in the profile priority list
        #   Otherwise returns the highest-priority cage of that type

        my ($self, $type, $check) = @_;

        # Local variables
        my $match;

        # Check for improper arguments
        if (! defined $type || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findHighestCage', @_);
        }

        # Check that $type is a recognised cage type; at the same time, if $type is in lower-case,
        #   change it (e.g. convert 'cmd' to 'Cmd')
        OUTER: foreach my $item ($axmud::CLIENT->cageTypeList) {

            if (lc ($item) eq lc ($type)) {

                $match = $item;
                last OUTER;
            }
        }

        if (! $match) {

            # Unrecognised cage type
            return undef;
        }

        # Get the highest priority cage of this type
        OUTER: foreach my $category ($self->profPriorityList) {

            my ($profile, $uniqueName, $cage);

            # Check that the profile $category exists and that there's a current profile of that
            #   category
            if (! $self->ivExists('currentProfHash', $category)) {

                # Doesn't exist
                next OUTER;

            } else {

                $profile = $self->ivShow('currentProfHash', $category);
            }

            # Check that the cage exists
            $uniqueName = lc($type) . '_' . $category . '_' . $profile->name;
            if (! $self->ivExists('cageHash', $uniqueName)) {

                # Cage matching $type and $profName doesn't exist
                next OUTER;

            } else {

                # Matching cage exists - this is the highest-priority cage for which we're looking
                return $self->ivShow('cageHash', $uniqueName)
            }
        }

        # No cage found
        return undef;
    }

    sub findCmd {

        # Can be called by anything
        # Given a standard command (a key in GA::Cage::Cmd->cmdHash), finds the replacement command
        #   from the highest-priority cage
        # e.g. If there's a replacement command defined for the standard command 'kill' in the cages
        #   associated with guild and race profiles, returns the replacement command from the race
        #   profile's cage (assuming that races still have priority over guilds)
        #
        # Expected arguments
        #   $cmd        - A standard command matching a key in $self->cmdHash
        #
        # Return values
        #   'undef' on improper arguments, if $cmd doesn't seem to exist in $self->cmdHash, if no
        #       command cage can be found or if no replacement command can be found
        #   Otherwise, returns the replacement command from the highest-priority cage

        my ($self, $cmd, $check) = @_;

        # Local variables
        my $cage;

        # Check for improper arguments
        if (! defined $cmd || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findCmd', @_);
        }

        # Get the highest priority command cage
        OUTER: foreach my $category ($self->profPriorityList) {

            $cage = $self->findCurrentCage('cmd', $category);
            if ($cage) {

                # Use the first available cage
                last OUTER;
            }
        }

        if (! $cage) {

            # No command cage available
            return undef;
        }

        # Check that the specified command exists in this cage's command hash
        if (! $cage->ivExists('cmdHash', $cmd)) {

            return undef;

        } else {

            # Return the highest-priority replacement command (consult inferior cage, if
            #   necessary)
            return $cage->ivShow('cmdHash', $cmd, $self);
        }
    }

    sub findWord {

        # Can be called by anything, using any command cage (e.g. make a direct call to
        #   GA::Cage::Cmd->findWord)
        # Given a standard word (a key in $self->wordHash), finds the replacement word from
        #   the highest-priority cage
        # e.g. If there's a replacement word defined for the standard word 'all' in the cages
        #   associated with guild and race profiles, returns the replacement word from the race
        #   profile's cage (assuming that races still have priority over guilds)
        #
        # Expected arguments
        #   $word       - A standard command matching a key in $self->wordHash
        #
        # Return values
        #   'undef' on improper arguments, if $word doesn't seem to exist in $self->wordHash, if no
        #       command cage can be found or if no replacement word can be found
        #   Otherwise, returns the replacement command from the highest-priority cage

        my ($self, $word, $check) = @_;

        # Local variables
        my $cage;

        # Check for improper arguments
        if (! defined $word || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findWord', @_);
        }

        # Get the highest priority command cage
        OUTER: foreach my $category ($self->profPriorityList) {

            $cage = $self->findCurrentCage('cmd', $category);
            if ($cage) {

                # Use the first available cage
                last OUTER;
            }
        }

        if (! $cage) {

            # No command cages available
            return undef;
        }

        # Check that the specified word exists in this cage's word hash
        if (! $cage->ivExists('wordHash', $word)) {

            return undef;

        } else {

            # Return the highest-priority replacement command (consult inferior cage, if necessary)
            return $cage->ivShow('wordHash', $word, $self);
        }
    }

    # Windows

    sub getHostLabelText {

        # Called by $self->start, ->doConnect, ->reactDisconnect and ->connectionComplete (and by
        #   GA::Win::Interal->setVisibleSession, when the window's visible session changes)
        # Prepares the text displayed in the 'main' window's connection info host label (it's the
        #   calling function's duty to tell the 'main' window to actually display it)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns the text to display

        my ($self, $check) = @_;

        # Local variables
        my $labelText;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->getHostLabelText', @_);
        }

        # Default label text is empty - in case something calls this function while the
        #   current connection status is 'waiting' or 'connecting'
        $labelText = '';

        if ($self->status eq 'connecting') {

            if ($self->mxpRelocateMode ne 'none') {

                $labelText = 'Relocating (via ' . $self->protocol . ') to \''
                                . $self->mxpRelocateHost . ' ' . $self->mxpRelocatePort . '\'...';

            } else {

                $labelText = 'Connecting (via ' . $self->protocol . ') to \'' . $self->initHost
                                . ' ' . $self->initPort . '\'...';
            }

        } elsif ($self->status eq 'connected') {

            $labelText = 'Connected (via ' . $self->protocol . ') to \'' . $self->initHost
                            . ' ' . $self->initPort . '\'';

        } elsif ($self->status eq 'offline') {

            $labelText = 'Connected (in offline mode) to \'' . $self->initHost . ' '
                            . $self->initPort . '\'';

        } elsif ($self->status eq 'disconnected') {

            $labelText = 'Disconnected at ' . $self->disconnectTime;
        }

        if ($labelText) {

            # Not too near the edge of the window, please
            $labelText .= ' ';
        }

        return $labelText;
    }

    sub getTimeLabelText {

        # Called by GA::Client->spinClientLoop (regularly), GA::Win::Internal->setVisibleSession
        #   and several other functions when required
        # Prepares the text displayed in a GA::Strip::ConnectInfo object's time label (it's the
        #   calling function's duty to tell the strip object to actually display the text)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns the text to display

        my ($self, $check) = @_;

        # Local variables
        my $labelText;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->getTimeLabelText', @_);
        }

        # For the benefit of the client loop, set an IV every time this function is called
        $self->ivPoke('connectInfoCheckTime', int($self->sessionTime));

        # Default label text is empty - in case something calls this function while the
        #   current connection status is 'waiting' or 'connecting'
        $labelText = '';

        if ($self->status eq 'connected') {

            # The call to ->getCounter converts a fractional number into a nice string like
            #   '4h 53m 07s'
            $labelText .= 'C ' . $axmud::CLIENT->getCounter($self->sessionTime);

            if (defined $self->delayedQuitTime) {

                $labelText .= ' | Q '
                        . $axmud::CLIENT->getCounter($self->delayedQuitTime - $self->sessionTime);
            }

            if (defined $self->lastDisplayTime) {

                $labelText .= ' | W '
                        . $axmud::CLIENT->getCounter($self->sessionTime - $self->lastDisplayTime);
            }

            if (defined $self->lastCmdTime) {

                $labelText .= ' | U '
                        . $axmud::CLIENT->getCounter($self->sessionTime - $self->lastCmdTime);
            }

        } elsif ($self->status eq 'offline') {

            $labelText = 'Session: ' . $axmud::CLIENT->getCounter($self->sessionTime);

        } elsif ($self->status eq 'disconnected') {

            # (If disconnected immediately, $self->sessionTime might be 'undef')
            if ($self->sessionTime) {
                $labelText = 'Connection length: ' . $axmud::CLIENT->getCounter($self->sessionTime);
            } else {
                $labelText = 'Connection length: n/a';
            }
        }

        if ($labelText) {

            # Not too near the strip object's blinkers, please
            $labelText .= '   ';
        }

        return $labelText;
    }

    # Session loop

    sub startSessionLoop {

        # Called by $self->start
        # Starts the session loop, which calls $self->spinSessionLoop whenever the loop spins
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the loop can't be started
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $loopObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startSessionLoop', @_);
        }

        # Create the object that handles the loop
        $loopObj = Games::Axmud::Obj::Loop->new(
            $self,
            'spinSessionLoop',
            'session',
        );

        if (! $loopObj) {

            return undef;

        } else {

            $self->ivPoke('sessionLoopObj', $loopObj);
        }

        # Start the loop
        if (! $loopObj->startLoop($self->sessionLoopDelay)) {

            return undef;
        }

        # Start some subservient loops (the replay loop is started by GA::Cmd::ReplayBuffer->do,
        #   when required)
        $self->startMaintainLoop();
        $self->startTimerLoop();
        $self->startIncomingLoop();
        $self->startTaskLoop();

        return 1;
    }

    sub stopSessionLoop {

        # Called by $self->stop
        # Stops the session loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the loop isn't running or if it can't be stopped
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $result;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopSessionLoop', @_);
        }

        if (! $self->sessionLoopObj) {

            return undef;
        }

        # Stop subservient loops
        $self->stopMaintainLoop();
        $self->stopTimerLoop();
        $self->stopIncomingLoop();
        $self->stopTaskLoop();
        $self->stopReplayLoop();

        # Stop the session loop
        $result = $self->sessionLoopObj->stopLoop();

        # Update IVs
        $self->ivUndef('sessionLoopObj');
        $self->ivPoke('sessionLoopSpinFlag', FALSE);
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return $result;
    }

    sub spinSessionLoop {

        # Called by $self->sessionLoopObj->spinLoop when the session loop spins
        # Sets the session time and spins all subservient loops
        #
        # Expected arguments
        #   $loopObj    - The GA::Obj::Loop object handling the session loop
        #
        # Optional arguments
        #   $spinType   - Any calling function which wants an extra spin of a single subservient
        #                   loop should call this function, not $self->spinMaintainLoop, etc (for
        #                   example, $self->doLogin spins the task loop just before sending world
        #                   commands). If defined, should be one of the strings 'maintain', 'timer',
        #                   'incoming', 'task' or 'replay'
        #
        # Return values
        #   'undef' on improper arguments, if the session loop isn't running or if $loopObj is the
        #       wrong loop object
        #   1 otherwise

        my ($self, $loopObj, $spinType, $check) = @_;

        # Check for improper arguments
        if (! defined $loopObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinSessionLoop', @_);
        }

        if (
            ! $self->sessionLoopObj
            || $self->sessionLoopObj ne $loopObj
            # ($self->sessionLoopSpinFlag has already been checked by the calling function)
            || $self->childLoopSpinFlag
            || $axmud::CLIENT->suspendSessionLoopFlag
        ) {
            return undef;
        }

        # Update IVs
        $self->ivPoke('sessionTime', $loopObj->spinTime);

        # Spin subservient loops, if they are running
        if (! $spinType || $spinType eq 'maintain') {

            $self->spinMaintainLoop();
        }

        if (! $spinType || $spinType eq 'timer') {

            $self->spinTimerLoop();
        }

        if (! $spinType || $spinType eq 'incoming') {

            $self->spinIncomingLoop();
        }

        if (! $spinType || $spinType eq 'task') {

            $self->spinTaskLoop();
        }

        if (! $spinType || $spinType eq 'replay') {

            $self->spinReplayLoop();
        }

        return 1;
    }

    # Maintenance loop

    sub startMaintainLoop {

        # Called by $self->startSessionLoop
        # Starts this session's maintenance loop (which is subservient to the session loop)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the maintenance loop is already running or if the
        #       parent session loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startMaintainLoop', @_);
        }

        if (defined $self->maintainLoopCheckTime || ! defined $self->sessionLoopObj) {

            # The maintenance loop is already running or the parent session loop is not running
            return undef;

        } else {

            # Do the first spin of the maintenance loop as soon as possible
            $self->ivPoke('maintainLoopCheckTime', 0);

            # Maintenance loop initialisation

            # Set the time for the first autosave (if any)
            $self->resetAutoSave();

            return 1;
        }
    }

    sub stopMaintainLoop {

        # Called by $self->stopSessionLoop
        # Stops this session's maintenance loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the maintenance loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopMaintainLoop', @_);
        }

        if (! defined $self->maintainLoopCheckTime) {

            # The maintenance loop isn't running
            return undef;

        } else {

            # Stop the maintenance loop
            $self->ivUndef('maintainLoopCheckTime');

            # Maintenance loop shutdown
            #   (nothing to do)

            return 1;
        }
    }

    sub spinMaintainLoop {

        # Called by $self->spinSessionLoop periodically, or by any other code which needs to spin
        #   this maintenance loop immediately
        # Spins the session's maintenance loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if another of the session's subservient loops is
        #       currently spinning or if the maintenance loop is not running
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my (
            $cage, $exitFlag, $cmdLimit, $cmdDelay,
            @repeatObjList, @excessCmdList, @soundObjList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinMaintainLoop', @_);
        }

        # If another of the session's subservient loops is currently spinning, or if the maintenance
        #   loop is not running, do nothing
        if ($self->childLoopSpinFlag || ! defined $self->maintainLoopCheckTime) {

            return undef;

        } else {

            # Don't let other subservient loops spin, until this loop is finished
            $self->ivPoke('childLoopSpinFlag', TRUE);
        }

        # Update the world's connection history object, if one was created for this session
        if ($self->connectHistoryObj) {

            $self->connectHistoryObj->set_currentTime();
        }

        # In login mode 3/4, if $self->processLineSegment has set the flag, we can complete the
        #   login
        if ($self->loginConnectFoundFlag) {

            # Only respond once
            $self->ivPoke('loginConnectFoundFlag', FALSE);
            $self->ivEmpty('loginConnectPatternList');

            if ($self->loginPromptsMode eq 'tiny') {

                # In mode 3, send a standard 'connect' world command, using the current command
                #   cages
                $self->sendModCmd('connect', 'name', $self->initChar, 'password', $self->initPass);

                # Wait for login success patterns (if there are any), otherwise mark the character
                #   as logged in
                $self->setLoginPatterns();

            } elsif ($self->loginPromptsMode eq 'world_cmd') {

                # In mode 4, send a list of world commands, substituting any that contain @name@,
                #   @account@ or @password@
                $self->processCmdLoginMode();
            }

        # If the character hasn't been marked as logged in yet, check whether it's time to show a
        #   warning message
        } elsif ($self->loginWarningTime && ! $self->mxpRelocateQuietFlag) {

            if ($self->status eq 'disconnected') {

                # Don't bother showing a warning message after a disconnection
                $self->ivUndef('loginWarningTime');

            } elsif ($self->loginWarningTime < $self->sessionTime) {

                $self->writeText(
                    'The character is not marked as \'logged in\' yet (use the \';login\' command'
                    . ' to override the automatic login process)',
                );

                # Only show the warning once
                $self->ivUndef('loginWarningTime');
            }
        }

        # If some text has been received that looks like a prompt, see if the waiting time has
        #   expired
        if (defined $self->promptLine && $self->promptCheckTime < $self->sessionTime) {

            # The time has expired. Process the prompt
            $self->processPrompt();
        }

        # If there are any GA::Obj::Repeat objects, ask them to send their commands to the world
        #   (if it's time to do so)
        if ($self->mxpRelocateMode eq 'none') {

            OUTER: foreach my $obj ($self->repeatObjList) {

                if ($obj->nextCheckTime < $self->sessionTime) {

                    if (! $obj->sendCmd($self)) {

                        # Object has finished sending commands
                        next OUTER;
                    }
                }

                # Object not finished sending commands
                push (@repeatObjList, $obj);
            }

            # Reset the IV, having removed any repeat objects which have finished sending commands
            $self->ivPoke('repeatObjList', @repeatObjList);

            # Deal with excess commands
            if ($self->crawlModeFlag) {

                # Crawl mode overrules the world profile's slowwalking settings, temporarily
                $cmdLimit = $self->crawlModeCmdLimit;
                $cmdDelay = 1;

            } else {

                $cmdLimit = $self->currentWorld->excessCmdLimit;
                $cmdDelay = $self->currentWorld->excessCmdDelay;
            }

            if (
                ! $self->lastExcessCmdTime
                || ($self->lastExcessCmdTime + $cmdDelay) < $self->sessionTime
            ) {
                # The minimum delay has passed
                $self->ivPoke('lastExcessCmdTime', $self->sessionTime);
                $self->ivPoke('excessCmdCount', 0);

                # If any excess commands are waiting to be sent, send them
                @excessCmdList = $self->excessCmdList;
                if (@excessCmdList) {

                    # We'll need the highest-priority command cage when we call ->dispatchCmd
                    $cage = $self->findHighestCage('cmd');

                    do {

                        my ($cmd, $time);

                        $cmd = shift @excessCmdList;
                        $time = shift @excessCmdList;

                        if ($time < $self->sessionTime) {

                            # The command is about to be sent to the world
                            $self->processWorldCmd($cmd, $cage);

                            # Don't send too many commands per second
                            if ($cmdLimit && $self->excessCmdCount >= $cmdLimit) {

                                $exitFlag = TRUE;
                            }

                        } else {

                            # Wait for another time period before trying to send the command again
                            unshift(@excessCmdList, $cmd, $time);
                            $exitFlag = TRUE;
                        }

                    } until ($exitFlag || $self->overruleMoveFlag || ! @excessCmdList);

                    if ($self->overruleMoveFlag) {

                        # A movement command has been overruled in protected moves mode, and the
                        #   world model flag instructs us to stop processing world commands, now. In
                        #   this case, all remaining excess commands are cancelled
                        $self->ivPoke('overruleMoveFlag', FALSE);
                        @excessCmdList = ();
                    }

                    $self->ivPoke('excessCmdList', @excessCmdList);      # May be an empty list

                    if ($self->crawlModeFlag && ! $self->excessCmdList) {

                        # Stored excess commands have been sent, and there are none left, so crawl
                        #   mode must disable itself
                        $self->resetCrawlMode();
                    }
                }
            }

            # In crawl mode, if no excess commands have been stored within two minutes of crawl
            #   mode being enabled, then disable it
            if (
                $self->crawlModeFlag
                && ! $self->excessCmdList
                && $self->crawlModeCheckTime < $self->sessionTime
            ) {
                $self->resetCrawlMode();
            }
        }

        # If any exits in the exit model have been modified, we may need to check and re-calculate
        #   regions paths
        if ($self->worldModelObj->updatePathHash || $self->worldModelObj->updateBoundaryHash) {

            $self->worldModelObj->updateRegionPaths($self);
        }

        # If any world model objects or exit model objects have been deleted since the last spin of
        #   the timer loop, make them available for re-use
        if ($self->worldModelObj->modelBufferList || $self->worldModelObj->exitBufferList) {

            $self->worldModelObj->updateModelBuffers();
        }

        # If model room objects have been added, moved or deleted since the last spin of the timer
        #   loop, check the regionmap IVs which recorded the highest and lowest occupied levels
        if ($self->worldModelObj->checkLevelsHash) {

            $self->worldModelObj->updateRegionLevels();
        }

        # If the Automapper window is open, allow it to update itself
        if ($self->mapWin) {

            $self->mapWin->winUpdate();
        }

        # Check if sounds played by GA::Client->playSoundFile have finished and, if so, prune the
        #   registry accordingly
        @soundObjList = $self->ivValues('soundHarnessHash');
        foreach my $soundObj (@soundObjList) {

            my $result;

            if (! $soundObj->harness->pumpable()) {

                # Sound has finished playing. Should it repeat?
                if ($soundObj->repeat > 1) {

                    # Repeat at least once more
                    $soundObj->ivDecrement('repeat');
                    $result = $axmud::CLIENT->repeatSoundFile($soundObj);

                } elsif ($soundObj->repeat == -1) {

                    # Repeat indefinitely
                    $result = $axmud::CLIENT->repeatSoundFile($soundObj);
                }

                # If the call to ->repeatSoundFile failed, if the sound should only be played once
                #   or has finished repeating, delete the GA::Obj::Sound object
                if (! $result) {

                    $soundObj->stop();
                    $self->ivDelete('soundHarnessHash', $soundObj->number);
                }
            }
        }

        # If auto-saves are turned on, see if it's time for an auto-save (but not during an MXP
        #   crosslinking operation)
        if (
            $axmud::CLIENT->autoSaveFlag
            && $self->autoSaveCheckTime
            && $self->autoSaveCheckTime < $self->sessionTime
            && $self->mxpRelocateMode eq 'none'
        ) {
            # Perform the auto-save
            $self->pseudoCmd('save');
            $self->ivPoke('autoSaveLastTime', $self->sessionTime);
            # Set the time at which the next auto-save will occur
            $self->resetAutoSave();
        }

        # Handle changes to this session's tab label (if visible)
        $self->checkTabLabels();

        # Update any MXP gauges whose entities have been modified
        $self->updateMxpGauges();

        # Perform a delayed quit, if one has been set
        if (defined $self->delayedQuitTime && $self->delayedQuitTime < $self->sessionTime) {

            $self->clientCmd($self->delayedQuitCmd);
            # (Only quit once)
            $self->ivUndef('delayedQuitTime');
            $self->ivUndef('delayedQuitCmd');
        }

        # Allow other loops to spin
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return 1;
    }

    sub resetAutoSave {

        # Called by $self->startMaintainLoop, ->spinMaintainLoop or GA::Cmd::AutoSave->do
        # If auto-saves are turned on, sets the time (matches $self->sessionTime) at which the
        #   next auto-save will occur. If auto-saves are turned off, sets the IV to 0
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 on success

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetAutoSave', @_);
        }

        if (! $axmud::CLIENT->autoSaveFlag) {

            # Auto-saves turned off
            $self->ivPoke('autoSaveCheckTime', 0);
            $self->ivPoke('autoSaveLastTime', 0);

        } else {

            # Auto-saves turned on
            $self->ivPoke(
                'autoSaveCheckTime',
                $self->sessionTime + ($axmud::CLIENT->autoSaveWaitTime * 60),
            );
        }

        return 1;
    }

    sub checkTabLabels {

        # Called by $self->spinMaintainLoop, $self->reactDisconnect, and also by $self->start in
        #   'connect offline' mode
        # This session's tab label (if visible) needs to be changed from time to time. This function
        #   checks whether it's necessary to change it and performs the operation, if so
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $changeTabFlag  - Set to TRUE when called by $self->start or ->reactDisconnect, in which
        #                       case the session's tab label (if visible) is definitely updated. Set
        #                       to FALSE (or 'undef') otherwise
        #
        # Return values
        #   'undef' on improper arguments
        #   1 on success

        my ($self, $changeTabFlag, $check) = @_;

        # Local variables
        my $exitFlag;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkTabLabels', @_);
        }

        # Decide whether this session's tab label colour needs to be changed from red to black (or
        #   vice-versa), or whether an xterm title needs to be updated
        if (
            ($self->showNewTextFlag && $self->showTabColourMode eq 'normal')
            || (! $self->showNewTextFlag && $self->showTabColourMode eq 'active')
            || ($self->showXTermTitleFlag)
        ) {
            # Need to change the tab label shortly
            $changeTabFlag = TRUE;
            # Update IVs
            $self->ivPoke('showXTermTitleFlag', FALSE);
        }

        # Check file objects for this session
        if ($self->showModFlag) {

            # The tab label contains an asterisk, meaning that some files need to be saved. If they
            #   have all been saved, we change the tab label
            OUTER: foreach my $fileObj ($self->ivValues('sessionFileObjHash')) {

                if ($fileObj->modifyFlag) {

                    $exitFlag = TRUE;
                    last OUTER;
                }
            }

            if (! $exitFlag) {

                # All files already saved, need to change the tab label shortly
                $changeTabFlag = TRUE;
                # Update IVs
                $self->ivPoke('showModFlag', FALSE);
            }

        } else {

            # The tab label doesn't contain an asterisk, meaning that no files need to be saved.
            #   If any of them now need to be saved, we change the tab label
            OUTER: foreach my $fileObj ($self->ivValues('sessionFileObjHash')) {

                if ($fileObj->modifyFlag) {

                    $exitFlag = TRUE;
                    last OUTER;
                }
            }

            if ($exitFlag) {

                # At least one file needs to be saved, need to change the tab label shortly
                $changeTabFlag = TRUE;
                # Update IVs
                $self->ivPoke('showModFlag', TRUE);
            }
        }

        # Change the tab label, if we need to
        if ($changeTabFlag) {

            $self->defaultTabObj->paneObj->setTabLabel(
                $self,
                $self->getTabLabelText(),
                $self->showModFlag,      # Show an asterisk, or not
            );
        }

        return 1;
    }

    sub getTabLabelText {

        # Called by $self->checkTabLabels
        # Gets the text to display in this session's tab label
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the text to display

        my ($self, $check) = @_;

        # Local variables
        my ($labelText, $worldName, $charName);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->getTabLabelText', @_);
        }

        # Use the world profile name, or the world's long name, depending on the GA::Client flag
        #   setting (but if the long name isn't set, don't use it)
        if ($axmud::CLIENT->longTabLabelFlag && $self->currentWorld->longName) {

            $worldName = $self->currentWorld->longName;
            # The long name is likely to be capitalised, so capitalise the character name, too (but
            #   not if the name contains ANY capital letters - we don't want to capitalise a
            #   character profile whose ->name is 'kIlLeR', for example
            if ($self->currentChar) {

                $charName = $self->currentChar->name;
                if (! ($charName =~ m/[A-Z]/)) {

                    $charName = ucfirst($charName);
                }
            }

        } else {

            $worldName = $self->currentWorld->name;
            if ($self->currentChar) {

                $charName = $self->currentChar->name;
            }
        }

        # The text to display depends on GA::Client->sessionTabMode. If there is no current
        #   character profile, in most cases we have to use a fallback format
        # However, if $self->xTermTitle is set and $axmud::CLIENT->xTermTitleFlag is TRUE, we
        #   display the xterm title, instead
        if ($axmud::CLIENT->xTermTitleFlag && $self->xTermTitle) {

            $labelText = $self->xTermTitle;

        } elsif ($axmud::CLIENT->sessionTabMode eq 'bracket') {

            # 'Deathmud (Gandalf)'
            if ($charName) {

                $labelText = $worldName . ' (' . $charName . ')';

            } else {

                # Fallback format
                $labelText = $worldName;
            }

        } elsif ($axmud::CLIENT->sessionTabMode eq 'hyphen') {

            # 'Deathmud - Gandalf'
            if ($charName) {

                $labelText = $worldName . ' - ' . $charName;

            } else {

                # Fallback format
                $labelText = $worldName;
            }

        } elsif ($axmud::CLIENT->sessionTabMode eq 'world') {

            # 'Deathmud'
            $labelText = $worldName;

        } elsif ($axmud::CLIENT->sessionTabMode eq 'char') {

            # 'Gandalf'
            if ($charName) {

                $labelText = $charName;

            } else {

                # Fallback format
                $labelText = '(' . $worldName . ')';
            }
        }

        # Change the colour, if necessary
        if ($self->status eq 'offline') {

            $labelText = '<span foreground="magenta">' . $labelText . '</span>';
            $self->ivPoke('showTabColourMode', 'offline');

        } elsif ($self->status eq 'disconnected') {

            $labelText = '<span foreground="blue">' . $labelText . '</span>';
            $self->ivPoke('showTabColourMode', 'disconnected');

        } elsif ($self->showNewTextFlag) {

            $labelText = '<span foreground="red">' . $labelText . '</span>';
            $self->ivPoke('showTabColourMode', 'active');

        } else {

            # Use the usual colour
            $self->ivPoke('showTabColourMode', 'normal');
        }

        return $labelText;
    }

    sub setCrawlMode {

        # Called by GA::Cmd::Crawl->do
        # Enables crawl mode, setting a temporary limit on the number of world commands that can be
        #   sent per second
        #
        # Expected arguments
        #   $num    - The command limit per second
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $num, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setCrawlMode', @_);
        }

        $self->ivPoke('crawlModeFlag', TRUE);
        $self->ivPoke('crawlModeCmdLimit', $num);
        $self->ivPoke('crawlModeCheckTime', $self->sessionTime + $self->crawlModeWaitTime);

        return 1;
    }

    sub resetCrawlMode {

        # Called by $self->spinMaintainLoop and GA::Cmd::Crawl->do
        # Disables crawl mode, removing a temporary limit on the number of world commands that can
        #   be sent per second. (If the current world profile specifies a limit, that limit then
        #   applies instead; otherwise there is no limit)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $noMsgFlag  - If TRUE, a system message is not displayed (because the calling function
        #                   wants to display its own one). If FALSE (or 'undef'), a short system
        #                   message is displayed
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $noMsgFlag, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setCrawlMode', @_);
        }

        $self->ivPoke('crawlModeFlag', FALSE);
        $self->ivPoke('crawlModeCmdLimit', undef);
        $self->ivPoke('crawlModeCheckTime', undef);

        if (! $noMsgFlag) {

            $self->writeText('Crawl mode disabled.');
        }

        return 1;
    }

    # Timer loop

    sub startTimerLoop {

        # Called by $self->startSessionLoop
        # Starts this session's timer loop (which is subservient to the session loop)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the timer loop is already running or if the parent
        #       session loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startTimerLoop', @_);
        }

        if (defined $self->timerLoopCheckTime || ! defined $self->sessionLoopObj) {

            # The timer loop is already running or the parent session loop is not running
            return undef;

        } else {

            # Do the first spin of the timer loop as soon as possible
            $self->ivPoke('timerLoopCheckTime', 0);

            # Timer loop initialisation
            #   (nothing to do)

            return 1;
        }
    }

    sub stopTimerLoop {

        # Called by $self->stopSessionLoop
        # Stops this session's timer loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the timer loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopTimerLoop', @_);
        }

        if (! defined $self->timerLoopCheckTime) {

            # The timer loop isn't running
            return undef;

        } else {

            # Stop the timer loop
            $self->ivUndef('timerLoopCheckTime');

            # Timer loop shutdown
            #   (nothing to do)

            return 1;
        }
    }

    sub spinTimerLoop {

        # Called by $self->spinSessionLoop periodically, or by any other code which needs to spin
        #   this timer loop immediately
        # Spins the session's timer loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if another of the session's subservient loops is
        #       currently spinning or if the timer loop is not running
        #   1 on success

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinTimerLoop', @_);
        }

        # If another of the session's subservient loops is currently spinning, or if the timer
        #   loop is not running, do nothing
        if ($self->childLoopSpinFlag || ! defined $self->timerLoopCheckTime) {

            return undef;

        } else {

            # Don't let other subservient loops spin, until this loop is finished
            $self->ivPoke('childLoopSpinFlag', TRUE);
        }

        # Check every active timer interface, to see if it is ready to fire (but not during an
        #   MXP crosslinking operation)
        if ($self->mxpRelocateMode eq 'none') {

            $self->checkTimers();
        }

        # Check to see whether it's time to fire a hook in response to the hook events 'user_idle'
        #   (user has been idle for $self->constHookIdleTime seconds) and 'world_idle' (world has
        #   sent no text for $self->constHookIdleTime seconds)
        # (Don't do it during an MXP crosslinking operation, obviously)
        if ($self->constHookIdleTime && $self->mxpRelocateMode eq 'none') {

            if (
                $self->lastCmdTime
                && ! $self->disableUserIdleFlag
                && $self->sessionTime > ($self->lastCmdTime + $self->constHookIdleTime)
            ) {
                # Fire any hooks that are using the 'user_idle' hook event
                $self->checkHooks('user_idle', $self->lastCmdTime);
                # This hook event will not fire again until $self->lastCmdTime is next set (i.e.
                #   when the next world command is sent)
                $self->ivPoke('disableUserIdleFlag', TRUE);
            }

            if (
                $self->lastDisplayTime
                && ! $self->disableWorldIdleFlag
                && $self->sessionTime > ($self->lastDisplayTime + $self->constHookIdleTime)
            ) {
                # Fire any hooks that are using the 'world_idle' hook event
                $self->checkHooks('world_idle', $self->lastDisplayTime);
                # This hook event will not fire again until $self->lastDisplayTime is next set (i.e.
                #   when text is next received from the world)
                $self->ivPoke('disableWorldIdleFlag', TRUE);
            }
        }

        # Tell GA::Client to update its registry of active keycodes, if any active macro interfaces
        #   have been deleted by this session since the loop last spun
        if ($axmud::CLIENT->resetKeycodesFlag) {

            $axmud::CLIENT->reset_activeKeycodes();
        }

        # Allow other loops to spin
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return 1;
    }

    # Task loop

    sub startTaskLoop {

        # Called by $self->startSessionLoop
        # Starts this session's task loop (which is subservient to the session loop)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the task loop is already running or if the parent
        #       session loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startTaskLoop', @_);
        }

        if (defined $self->taskLoopCheckTime || ! defined $self->sessionLoopObj) {

            # The task loop is already running or the parent session loop is not running
            return undef;

        } else {

            # Do the first spin of the task loop as soon as possible
            $self->ivPoke('taskLoopCheckTime', 0);

            # Task loop initialisation
            #   (nothing to do)

            return 1;
        }
    }

    sub stopTaskLoop {

        # Called by $self->stopSessionLoop
        # Stops this session's task loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the task loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopTaskLoop', @_);
        }

        if (! defined $self->taskLoopCheckTime) {

            # The task loop isn't running
            return undef;

        } else {

            # Stop the task loop
            $self->ivUndef('taskLoopCheckTime');

            # Task loop shutdown

            # Shut down any current tasks which are running
            foreach my $taskObj ($self->ivValues('currentTaskHash')) {

                $taskObj->set_shutdownFlag(TRUE);

                if ($taskObj->category eq 'process') {
                    $taskObj->main();
                } else {
                    $taskObj->shutdown();
                }
            }

            return 1;
        }
    }

    sub spinTaskLoop {

        # Called by $self->spinSessionLoop periodically, or by any other code which needs to spin
        #   this task loop immediately
        # Spins the session's task loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if another of the session's subservient loops is
        #       currently spinning or if the task loop is not running
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my (
            $resetWinFlag, $benchFlag, $currentTab,
            @taskList, @selectedList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinTaskLoop', @_);
        }

        # If another of the session's subservient loops is currently spinning, if the task
        #   loop is not running, if an MXP crosslinking operation is in progress or if the
        #   'freeze tasks' flag was set (by ';freezetask'), do nothing
        if (
            $self->childLoopSpinFlag
            || ! defined $self->taskLoopCheckTime
            || $self->mxpRelocateMode ne 'none'
            || $self->freezeTaskLoopFlag
        ) {
            return undef;

        } else {

            # Don't let other subservient loops spin, until this loop is finished
            $self->ivPoke('childLoopSpinFlag', TRUE);
        }

        # Delete any active interfaces (which aren't based on an inactive interface) that have
        #   marked for deletion by $self->deleteInterface
        foreach my $obj ($self->deleteInterfaceList) {

            $self->removeInterface($obj);
        }

        $self->ivEmpty('deleteInterfaceList');

        # Remove any tasks that have finished by deleting them from the current tasklist
        OUTER: foreach my $taskObj ($self->ivValues('currentTaskHash')) {

            if ($taskObj->status eq 'finished') {

                # Remove all interfaces (triggers, aliases, macros, timers and hooks) associated
                #   with this task
                $self->tidyInterfaces($taskObj);

                # Remove the task from the current tasklist
                $self->ivDelete('currentTaskHash', $taskObj->uniqueName);
                # Don't call the task again
                if ($self->ivExists('taskCallHash', $taskObj->uniqueName)) {

                    $self->ivDelete('taskCallHash', $taskObj->uniqueName);
                }

                # $self->currentTaskNameHash contains a single copy of each type of task running
                #   (e.g. the most recently created Social task). Remove this task from the registry
                #   and, if there are any other copies of the same type of task running, find the
                #   most recent one and add it to the registry
                $self->ivDelete('currentTaskNameHash', $taskObj->name);
                INNER: foreach my $otherObj ($self->ivValues('currentTaskHash')) {

                    if ($otherObj->name eq $taskObj->name) {

                        if (
                            # The first task of the same type found
                            ! $self->ivExists('currentTaskNameHash', $taskObj->name)
                            # $otherObj was created more recently than a task of the same type we
                            #   added earlier in this INNER loop
                            || $otherObj->startTime
                                < $self->ivShow('currentTaskNameHash', $taskObj->name)->startTime
                        ) {
                            $self->ivAdd('currentTaskNameHash', $taskObj->name, $otherObj);
                        }
                    }
                }

                # If the Status task is running, tell it to update its display of active tasks
                if ($self->statusTask) {

                    $self->update_statusTask();
                }

                # Every time a task stops, the 'main' window's menu items must be sensitised /
                #   desensitised, depending on current conditions (but only if this session is the
                #   current one)
                $resetWinFlag = TRUE;
            }
        }

        # Maintain tasks by checking their settings, and starting/pausing/stopping them as necessary
        @taskList = $self->compileTasks();
        OUTER: foreach my $taskObj (@taskList) {

            my $taskName = $taskObj->uniqueName;

            # Task is waiting to start, and is due to start
            if (
                $taskObj->status eq 'wait_init'
                && $taskObj->startTime < $self->sessionTime
            ) {
                $self->startTask($taskName, $taskObj);
                # Every time a task stops, the 'main' window's menu items must be sensitised /
                #   desensitised, depending on current conditions (but only if this session is the
                #   current one)
                $resetWinFlag = TRUE;

            # Task is waiting for another task to exist, before starting
            } elsif ($taskObj->status eq 'wait_task_exist') {

                # If it's time to check (which takes place once a minute)
                if ($taskObj->checkTime <= $self->sessionTime) {

                    # See if the matching task exists. First check task unique names
                    my $waitForTask = $taskObj->waitForTask;
                    if (
                        $self->ivExists('currentTaskHash', $waitForTask)
                        || $self->ivExists('currentTaskNameHash', $waitForTask)
                    ) {
                        # Match found, so initialise $taskObj
                        $self->startTask($taskName, $taskObj);
                        $resetWinFlag = TRUE;

                    # No match found, so now check task generic names
                    } else {

                        INNER: foreach my $otherTask ($self->ivValues('currentTaskHash')) {

                            if ($otherTask->name eq $waitForTask) {

                                # Match found, so initialise $taskObj
                                $self->startTask($taskName, $taskObj);
                                $resetWinFlag = TRUE;

                                last INNER;
                            }
                        }
                    }
                }

            # Task is waiting for another task to not exist, before starting
            } elsif ($taskObj->status eq 'wait_task_no_exist') {

                # If it's time to check (which takes place once a minute)
                if ($taskObj->checkTime <= $self->sessionTime) {

                    # See if the matching task exists. First check task unique names
                    my $waitForTask = $taskObj->waitForTask;
                    if (
                        ! $self->ivExists('currentTaskHash', $waitForTask)
                        && ! $self->ivExists('currentTaskNameHash', $waitForTask)
                    ) {
                        # No match found, so now task check generic names
                        my $match = 0;

                        INNER: foreach my $otherTask ($self->ivValues('currentTaskHash')) {

                            if ($otherTask->name eq $waitForTask) {

                                $match = 1;
                                last INNER;
                            }
                        }

                        if (! $match) {

                            # No match found, so initialise $taskObj
                            $self->startTask($taskName, $taskObj);
                            $resetWinFlag = TRUE;
                        }
                    }
                }

            # Task is waiting for another task to start and then stop, before starting
            } elsif ($taskObj->status eq 'wait_task_start_stop') {

                # If it's time to check (which takes place once a minute)
                if ($taskObj->checkTime <= $self->sessionTime) {

                    # See if the matching task exists. First check task unique names
                    my $waitForTask = $taskObj->waitForTask;
                    if (
                        $self->ivExists('currentTaskHash', $waitForTask)
                        || $self->ivExists('currentTaskNameHash', $waitForTask)
                    ) {
                        # Match found, so change $taskObj's status to waiting for the matched task
                        #   to stop
                        $taskObj->set_status('wait_task_no_exist');

                    # No match found, so now check task generic names
                    } else {

                        INNER: foreach my $otherTask ($self->ivValues('currentTaskHash')) {

                            if ($otherTask->name eq $waitForTask) {

                                # Match found, so change $taskObj's status to waiting for the
                                #   matched task to stop
                                $taskObj->set_status('wait_task_no_exist');
                                last INNER;
                            }
                        }
                    }
                }

            # Task is active or paused, and is due to stop
            } elsif (
                ($taskObj->status eq 'paused' || $taskObj->status eq 'running')
                && $taskObj->endTime
                && $taskObj->endTime <= $self->sessionTime
            ) {
                # Send the task a shutdown message, so it can shut down gracefully (when the
                #   process is complete, the task should set its own status to 'finished')
                $taskObj->set_shutdownFlag(TRUE);

            # Task is currently paused
            } elsif ($taskObj->status eq 'paused') {

                # Unpause the task if it's time
                if ($taskObj->checkTime && $taskObj->checkTime <= $self->sessionTime) {

                    $taskObj->set_status('running');
                }
            }
        }

        # If a current profile has been changed, some tasks need to be told to reset
        if ($self->currentProfChangeFlag) {

            # Tell the task that is should reset, if it is either active or paused, but if the task
            #   stage is 1 (meaning the task has not yet been executed for the first time), there's
            #   no need to reset it
            foreach my $taskObj ($self->ivValues('currentTaskHash')) {

                if (
                    $taskObj->profSensitivityFlag
                    && ($taskObj->status eq 'running' || $taskObj->status eq 'paused')
                    && (
                        ($taskObj->category eq 'process' && $taskObj->stage != 1)
                        || ($taskObj->category eq 'activity')
                    )
                ) {
                    $taskObj->set_status('reset');
                }
            }

            # In addition, every session's tab label must be updated (as they contain names of
            #   current profiles)
            foreach my $session ($axmud::CLIENT->listSessions()) {

                if ($session->defaultTabObj) {

                    $session->defaultTabObj->paneObj->setTabLabel(
                        $session,
                        $session->getTabLabelText(),
                        $session->showModFlag,              # Show an asterisk, or not
                    );
                }
            }

            # Resets complete
            $self->ivPoke('currentProfChangeFlag', FALSE);
        }

        # Before calling active tasks, we must resume any Axbasic scripts which have been paused
        #   as the result of an INPUT statement, and which aren't running with a parent task
        # (All other kinds of Axbasic pauses are handled by the parent task)
        # The list can contain duplicate entries, so if any script isn't paused, just ignore its
        #   entry (or entries) on the list
        do {

            my $scriptObj = $self->ivShift('scriptResumeList');

            if ($scriptObj && $scriptObj->scriptStatus eq 'paused') {

                $scriptObj->implement();
                # 'Internal' window menus must be reset when a script resumes
                $resetWinFlag = TRUE;
            }

        } until (! $self->scriptResumeList);

        # Now, call each running process task in turn. Don't call activity tasks unless their
        #   ->shutdownFlag has been set, or they must be reset; don't call process tasks if their
        #   ->delayTime hasn't expired yet
        @taskList = $self->compileTasks();
        foreach my $taskObj (@taskList) {

            # Process tasks
            if (
                $taskObj->category eq 'process'
                && ($taskObj->status eq 'running' || $taskObj->status eq 'reset')
                && (
                    ! $self->ivExists('taskCallHash', $taskObj->uniqueName)
                    || ! $self->ivShow('taskCallHash', $taskObj->uniqueName)
                    || $self->ivShow('taskCallHash', $taskObj->uniqueName) < $self->sessionTime
                )
            ) {
                # Call the task
                $taskObj->main();

                # Set the time at which the task will next be called (if it is still running)
                if ($taskObj->status eq 'running') {

                    if ($taskObj->delayTime && $taskObj->delayTime > $self->taskLoopDelay) {

                        # Don't call the task on the next task loop, but after a delay
                        $self->ivAdd(
                            'taskCallHash',
                            $taskObj->uniqueName,
                            $self->sessionTime + $taskObj->delayTime,
                        );

                    } else {

                        # Call the task on the next task loop
                        $self->ivAdd('taskCallHash', $taskObj->uniqueName, 0);
                    }
                }

            # Activity tasks
            } elsif ($taskObj->category eq 'activity') {

                if ($taskObj->shutdownFlag) {

                    # This an activity task whose shutdown flag has been set
                    $taskObj->shutdown();

                } elsif ($taskObj->status eq 'reset') {

                    # This an activity task which must be reset
                    $taskObj->reset();
                }
            }
        }

        # If the GUI is open at the tab containing the list of current tasks, re-draw the list
        if ($self->guiWin) {

            $currentTab = $self->guiWin->notebookGetTab();

            if (defined $currentTab && $currentTab eq 'Current tasklist') {

                # If there are currently any selected lines in the tab's Gtk2::Ex::Simple::List,
                #   remember them, so we can select them again as soon as the list is redrawn
                @selectedList = $self->guiWin->notebookGetSelectedLines();

                # Redraw the list
                $self->guiWin->currentTaskHeader();

                if (@selectedList) {

                    # Re-select each selected line
                    $self->guiWin->notebookSetSelectedLines(@selectedList);
                }
            }
        }

        # At least one task loop has completed
        $self->ivPoke('firstTaskLoopCompleteFlag', TRUE);

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions (if a task
        #   has started or stopped during this task loop or if a Axbasic script has resumed)
        if ($resetWinFlag) {

            $axmud::CLIENT->desktopObj->restrictWidgets();
        }

        # Allow other loops to spin
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return 1;
    }

    sub compileTasks {

        # Called by $self->spinTaskLoop at various points during its spin
        # Extracts a list of tasks from the current tasklist
        # Moves those tasks which must be processed first to the front of the list, and those tasks
        #    which must be processed last to the end of the list
        # Returns the modified list
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns the modified list

        my ($self, $check) = @_;

        # Local variables
        my (@emptyList, @taskList, @initialList, @firstList, @lastList);

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->compileTasks', @_);
            return @emptyList;
        }

        # Compile a list of active process tasks that must be called. Activity tasks must also be
        #   called if their shutdown flag has been set, or if they must be reset
        @taskList = $self->ivValues('currentTaskHash');

        # Go through the list, removing any tasks on the 'first' runlist (tasks that must be run
        #   first, before any others)
        if ($axmud::CLIENT->taskRunFirstList) {

            OUTER: foreach my $string ($axmud::CLIENT->taskRunFirstList) {

                @initialList = @taskList;
                @taskList = ();

                INNER: foreach my $taskObj (@initialList) {

                    if ($taskObj->name eq $string) {
                        push (@firstList, $taskObj);
                    } else {
                        push (@taskList, $taskObj);
                    }
                }
            }
        }

        # Now remove tasks on the 'last' runlist (tasks that must be run last, before any others)
        if ($axmud::CLIENT->taskRunLastList) {

            OUTER: foreach my $string ($axmud::CLIENT->taskRunLastList) {

                @initialList = @taskList;
                @taskList = ();

                INNER: foreach my $taskObj (@initialList) {

                    if ($taskObj->name eq $string) {
                        push (@lastList, $taskObj);
                    } else {
                        push (@taskList, $taskObj);
                    }
                }
            }
        }

        # Now put tasks at the beginning, and at the end. (@lastList contains tasks in reverse
        #   order)
        @taskList = (@firstList, @taskList, (reverse @lastList));

        return @taskList;
    }

    sub startTask {

        # Called by $self->spinTaskLoop to start a new task
        # Performs some final checks, marks the task as 'running', and sets the conditions for the
        #   task to stop
        #
        # Expected arguments
        #   $taskName   - A key in the hash $self->currentTaskHash (also matches $taskObj->name)
        #   $taskObj    - The corresponding value, set to the task object's blessed reference
        #
        # Return values
        #   'undef' if improper arguments supplied
        #   'undef' if the task can't be started because its jealousy flag is 1, and another copy of
        #       the task is already running
        #   'undef' if the task requires the Locator task, and the Locator task doesn't exist (or
        #       isn't active)
        #   'undef' if the task was due to finish before it started
        #   1 if the task is successfully started

        my ($self, $taskName, $taskObj, $check) = @_;

        # Check for improper arguments
        if (! defined $taskName || ! defined $taskObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startTask', @_);
        }

        # Check the task's jealousy flag
        if ($taskObj->jealousyFlag) {

            foreach my $otherTask ($self->ivValues('currentTaskHash')) {

                if ($otherTask ne $taskObj && $otherTask->name eq $taskObj->name) {

                    # Another copy of the $taskObj has been found; mark the task as 'finished' (it
                    #   will never be executed)
                    $taskObj->set_status('finished');
                    return $self->writeError(
                        'Cannot run more than one copy of the \'' . $taskObj->prettyName
                        . '\' task concurrently ' . '- the most recent copy has been aborted',
                        $self->_objClass . '->startTask',
                    );
                }
            }
        }

        # Check the task's dependency on the Locator task
        if (
            $taskObj->requireLocatorFlag == 1
            && $taskObj->status eq 'wait_init'
            && ! defined $self->locatorTask
        ) {
            $taskObj->set_status('wait_task_exist');
            $taskObj->set_waitForTask('locator_task');

            return undef;
        }

        # Check the size limits on the task's name variables
        if (! $axmud::CLIENT->nameCheck($taskObj->name, 16)) {

            $taskObj->set_status('finished');
            return $self->writeError(
                'Illegal task standard name \'' . $taskObj->name . '\' - task aborted',
                $self->_objClass . '->startTask',
            );

        } elsif (length ($taskObj->prettyName) > 32) {

            $taskObj->set_status('finished');
            return $self->writeError(
                'Illegal task pretty name \'' . $taskObj->prettyName . '\' - task aborted',
                $self->_objClass . '->startTask',
            );

        } elsif (! $axmud::CLIENT->nameCheck($taskObj->uniqueName, 24)) {

            $taskObj->set_status('finished');
            return $self->writeError(
                'Illegal task unique name \'' . $taskObj->uniqueName . '\' - task aborted',
                $self->_objClass . '->startTask',
            );
        }

        # Start the task
        $taskObj->set_status('running');
        $taskObj->set_startTime($self->sessionTime);

        # If it's a process task, set the task stage to 1. If it's an activity task, call the task's
        #   ->init() function to initialise it
        if ($taskObj->category eq 'process') {

            $taskObj->set_stage(1);

        } elsif ($taskObj->category eq 'activity') {

            $taskObj->init();
        }

        # Set the task's end time (if appropriate)
        if ($taskObj->endStatus eq 'run_for') {

            # Task runs for a certain number of minutes after this moment
            $taskObj->set_endTime($self->sessionTime + ($taskObj->endTime * 60));

        } elsif ($taskObj->endStatus eq 'run_until') {

            # Task runs until this time specified by $taskObj->endTime. If that time has already
            #   passed, mark the task as 'finished' (it will never be executed, and will be deleted
            #   on the next task loop)
            if ($taskObj->endTime <= $self->sessionTime) {

                $taskObj->set_status('finished');
                return undef;
            }

        } else {

            # Mark the task as running indefinitely, by setting ->endTime to zero
            $taskObj->set_endTime(0);
        }

        return 1;
    }

    sub haltProfileTasks {

        # Called by GA::Generic::Cmd->setProfile and $self->setupProfiles
        # When a current profile is unset as a current profile, all of the tasks in the current
        #   tasklist that were created from the profile's initial tasklist or initial scriptlist
        #   must be shut down
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $name     - The name of the current profile whose associated tasks must be shut down.
        #                   If 'undef', the tasks associated with all current profiles are shut down
        #
        # Return values
        #   'undef' on improper arguments, if a specified profile doesn't exist or if the specified
        #       profile's category isn't in the profile priority list ($self->profPriorityList)
        #   Otherwise returns the number of associated tasks that were shut down (may be 0)

        my ($self, $name, $check) = @_;

        # Local variables
        my (
            $count,
            @profList,
            %profHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->haltProfileTasks', @_);
        }

        # Compile a list of current profiles to process
        if (defined $name) {

            # Check the specified profile exists
            if (! $self->ivExists('profHash', $name)) {

                return $self->writeError(
                    'Unrecognised profile \'' . $name . '\'',
                    $self->_objClass . '->haltProfileTasks',
                );

            } else {

                push (@profList, $self->ivShow('profHash', $name));
            }

        } else {

            # Use all current profiles
            @profList = $self->ivValues('currentProfHash');
        }

        # Convert the list of profiles into a hash, for quick lookup
        foreach my $profObj (@profList) {

            $profHash{$profObj->name} = undef;
        }

        # Check each task in the current tasklist
        $count = 0;
        foreach my $taskObj ($self->ivValues('currentTaskHash')) {

            if ($taskObj->profName && exists ($profHash{$taskObj->profName})) {

                # This task must be shut down
                $taskObj->set_shutdownFlag(TRUE);
                $count++;
            }
        }

        # Return the number of shutting down tasks (may be 0)
        return $count;
    }

    # Replay loop

    sub startReplayLoop {

        # Called by GA::Cmd::ReplayBuffer->do
        # Starts this session's replay loop (which is subservient to the session loop)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $textFlag   - If TRUE, replay information from the replay text buffer. If FALSE (or
        #                   'undef'), don't use the replay text buffer
        #   $cmdFlag    - If TRUE, replay information from the replay command buffer. If FALSE (or
        #                   'undef'), don't use the replay command buffer. (Both $textFlag and
        #                   $cmdFlag can be TRUE, but if both are FALSE, the replay loop doesn't
        #                   start)
        #   $beginTime  - The time at which to begin (matches GA::Buffer::Display->time and
        #                   GA::Buffer::Cmd->time). If 'undef', the loop starts at the time of the
        #                   first buffer object
        #   $endTime    - The time at which to end (matches GA::Buffer::Display->time and
        #                   GA::Buffer::Cmd->time). If 'undef', the loop stops at the time of the
        #                   last buffer object
        #
        # Return values
        #   'undef' on improper arguments, if the replay loop is already running or if the parent
        #       session loop is not running
        #   1 otherwise

        my ($self, $textFlag, $cmdFlag, $beginTime, $endTime, $check) = @_;

        my ($firstText, $firstCmd);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startReplayLoop', @_);
        }

        if (
            defined $self->replayLoopCheckTime
            || ! defined $self->sessionLoopObj
            # Also don't start the replay loop if both flag arguments are FALSE...
            || (! $textFlag && ! $cmdFlag)
            # ...or when connected to a world
            || ($self->status ne 'disconnected' && $self->status ne 'offline')
        ) {
            # The replay loop is already running or the parent session loop is not running
            return undef;

        } else {

            # Do the first spin of the replay loop as soon as possible
            $self->ivPoke('replayLoopCheckTime', 0);

            # Replay loop initialisation

            # The two flag must be either TRUE or FALSE
            if ($textFlag) {
                $textFlag = TRUE;
            } else {
                $textFlag = FALSE;
            }

            if ($cmdFlag) {
                $cmdFlag = TRUE;
            } else {
                $cmdFlag = FALSE;
            }

            # Decide the time (matching the $self->sessionTime when objects were added to the
            #   buffers) at which the replay should start
            if (! defined $beginTime) {

                $beginTime = 0;
            }

            # Find the first line in each buffer to check
            if ($textFlag) {

                OUTER: for (
                    my $num = $self->replayDisplayBufferFirst;
                    $num <= $self->replayDisplayBufferLast;
                    $num++
                ) {
                    my $obj = $self->ivShow('replayDisplayBufferHash', $num);
                    if ($obj->time >= $beginTime) {

                        # This buffer line is the first one that can be used
                        $firstText = $num;
                        last OUTER;
                    }
                }

                if (! defined $firstText) {

                    # There is no buffer line to replay
                    $textFlag = FALSE;
                }
            }

            if ($cmdFlag) {

                OUTER: for (
                    my $num = $self->replayCmdBufferFirst;
                    $num <= $self->replayCmdBufferLast;
                    $num++
                ) {
                    my $obj = $self->ivShow('replayCmdBufferHash', $num);
                    if ($obj->time >= $beginTime) {

                        # This buffer line is the first one that can be used
                        $firstCmd = $num;
                        last OUTER;
                    }
                }

                if (! defined $firstCmd) {

                    # There is no buffer line to replay
                    $cmdFlag = FALSE;
                }
            }

            if (! $textFlag && ! $cmdFlag) {

                # Nothing to replay; don't allow the replay loop to start
                $self->ivUndef('replayLoopCheckTime');

                return undef;

            } else {

                # Update IVs
                $self->ivPoke('replayLoopStartTime', $axmud::CLIENT->getTime() - $beginTime);
                $self->ivPoke('replayLoopTime', $beginTime);
                $self->ivUndef('replayLoopStopTime', $endTime);         # May be 'undef'
                $self->ivPoke('replayLoopTextFlag', $textFlag);
                $self->ivPoke('replayLoopCmdFlag', $cmdFlag);
                $self->ivPoke('replayLoopNextText', $firstText);
                $self->ivPoke('replayLoopNextCmd', $firstCmd);

                return 1;
            }
        }
    }

    sub stopReplayLoop {

        # Called by $self->stopSessionLoop
        # Stops this session's replay loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the replay loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopReplayLoop', @_);
        }

        if (! defined $self->replayLoopCheckTime) {

            # The replay loop isn't running
            return undef;

        } else {

            # Stop the replay loop
            $self->ivUndef('replayLoopCheckTime');

            # Replay loop shutdown

            # Reset IVs
            $self->ivUndef('replayLoopStartTime');
            $self->ivUndef('replayLoopTime');
            $self->ivUndef('replayLoopStopTime');
            $self->ivUndef('replayLoopTextFlag');
            $self->ivUndef('replayLoopCmdFlag');
            $self->ivUndef('replayLoopNextText');
            $self->ivUndef('replayLoopNextCmd');

            return 1;
        }
    }

    sub spinReplayLoop {

        # Called by $self->spinSessionLoop periodically, or by any other code which needs to spin
        #   this replay loop immediately
        # Spins the session's replay loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if another of the session's subservient loops is
        #       currently spinning or if the replay loop is not running
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my $exitFlag;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinReplayLoop', @_);
        }

        # If another of the session's subservient loops is currently spinning, if the replay
        #   loop is not running or if an MXP crosslinking operation is in progress, do nothing
        if (
            $self->childLoopSpinFlag
            || ! defined $self->replayLoopCheckTime
            || $self->mxpRelocateMode ne 'none'
        ) {
            return undef;

        } else {

            # Don't let other subservient loops spin, until this loop is finished
            $self->ivPoke('childLoopSpinFlag', TRUE);
        }

        # NB $self->replayLoopTime is updated at the end of the function, so that if there are
        #   buffer objects whose ->time is set to 0, those buffer objects can still be used
        #   (assuming that, on the first call to this function, $self->replayLoopTime is still 0)
        do {

            my ($textNum, $cmdNum, $textObj, $cmdObj);

            # Import IVs for convenience
            $textNum = $self->replayLoopNextText;
            $cmdNum = $self->replayLoopNextCmd;

            # Exit this loop after the first spin in which no buffer object is used
            $exitFlag = TRUE;

            # Check the buffer line actually exists. If not, we stop replaying the buffer here
            if ($self->replayLoopTextFlag) {

                if (! $self->ivExists('replayDisplayBufferHash', $textNum)) {

                    $self->ivPoke('replayLoopTextFlag', FALSE);
                    $self->ivUndef('replayLoopNextText');
                }
            }

            if ($self->replayLoopCmdFlag) {

                if (! $self->ivExists('replayCmdBufferHash', $cmdNum)) {

                    $self->ivPoke('replayLoopCmdFlag', FALSE);
                    $self->ivUndef('replayLoopNextCmd');
                }
            }

            if ($self->replayLoopTextFlag || $self->replayLoopCmdFlag) {

                # Get the GA::Buffer::Display and GA::Buffer::Cmd objects
                if ($self->replayLoopTextFlag) {

                    $textObj = $self->ivShow('replayDisplayBufferHash', $textNum);

                    # Is it time to display this object?
                    if ($textObj->time > $self->replayLoopTime) {

                        # Not time yet
                        $textObj = undef;

                    # Is it time to stop replaying text buffer objects?
                    } elsif (
                        defined $self->replayLoopStopTime
                        && $textObj->time < $self->replayLoopStopTime
                    ) {
                        # Time to stop
                        $self->ivPoke('replayLoopTextFlag', FALSE);
                        $self->ivUndef('replayLoopNextText');
                        $textObj = undef;
                    }
                }

                if ($self->replayLoopCmdFlag) {

                    $cmdObj = $self->ivShow('replayCmdBufferHash', $cmdNum);
                    # Is it time to display this object?
                    if ($cmdObj->time > $self->replayLoopTime) {

                        # Not time yet
                        $cmdObj = undef;

                    # Is it time to stop replaying command buffer objects?
                    } elsif (
                        defined $self->replayLoopStopTime
                        && $cmdObj->time < $self->replayLoopStopTime
                    ) {
                        # Time to stop
                        $self->ivPoke('replayLoopCmdFlag', FALSE);
                        $self->ivUndef('replayLoopNextCmd');
                        $cmdObj = undef;
                    }
                }

                # If $textObj and $cmdObj are both still defined, display information from the
                #   earlier one
                if ($textObj && $cmdObj) {

                    if ($textObj->time < $cmdObj->time) {

                        $cmdObj = undef;

                    } elsif ($textObj->time > $cmdObj->time) {

                        $textObj = undef;

                    } else {

                        # Default - display text before command
                        $cmdObj = undef;
                    }
                }

                # Display information from the buffer
                if ($textObj) {

                    # Show the text in ->stripLine, so that interfaces can interact with the line
                    #   again. End the string with " \n" so that an empty line is displayed, and so
                    #   that each buffer line is shown on a separate line in the 'main' window
                    $self->processIncomingData($textObj->stripLine . " \n");
                    $self->ivIncrement('replayLoopNextText');
                    # Repeat this loop, so the next buffer line can be used
                    $exitFlag = FALSE;

                } elsif ($cmdObj) {

                    # Process the world command
                    $self->worldCmd($cmdObj->cmd);
                    $self->ivIncrement('replayLoopNextCmd');
                    # Repeat this loop, so the next buffer line can be used
                    $exitFlag = FALSE;
                }
            }

        } until ($exitFlag);

        if (! $self->replayLoopTextFlag && ! $self->replayLoopCmdFlag) {

            # We've finished replaying the buffer(s)
            $self->stopReplayLoop();
            $self->writeText('Buffer replay complete');

        } else {

            # Update replay loop time. The subtraction produces a system rounding error, so we need
            #   to round the value to 3dp
            $self->ivPoke(
                'replayLoopTime',
                sprintf('%.3f', $axmud::CLIENT->getTime() - $self->replayLoopStartTime),
            );
        }

        # Allow other loops to spin
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return 1;
    }

    # Incoming data loop

    sub startIncomingLoop {

        # Called by $self->startSessionLoop
        # Starts this session's incoming data loop (which is subservient to the session loop)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the incoming data loop is already running or if the
        #       parent session loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startIncomingLoop', @_);
        }

        if (defined $self->incomingLoopCheckTime || ! defined $self->sessionLoopObj) {

            # The incoming data loop is already running or the parent session loop is not running
            return undef;

        } else {

            # Do the first spin of the incoming data loop as soon as possible
            $self->ivPoke('incomingLoopCheckTime', 0);

            # Incoming data loop initialisation
            #   (nothing to do)

            return 1;
        }
    }

    sub stopIncomingLoop {

        # Called by $self->stopSessionLoop
        # Stops this session's incoming data loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the incoming data loop is not running
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopIncomingLoop', @_);
        }

        if (! defined $self->incomingLoopCheckTime) {

            # The incoming data loop isn't running
            return undef;

        } else {

            # Stop the incoming data loop
            $self->ivUndef('incomingLoopCheckTime');

            # Incoming data loop shutdown
            #   (nothing to do)

            return 1;
        }
    }

    sub spinIncomingLoop {

        # Called by $self->spinSessionLoop periodically, or by any other code which needs to spin
        #   this incoming data loop immediately
        # Spins the session's incoming data loop
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if another of the session's subservient loops is
        #       currently spinning, if the incoming data loop is not running or if the loop spin is
        #       halted prematurely
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my ($haltFlag, $text, $result);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->spinIncomingLoop', @_);
        }

        # If another of the session's subservient loops is currently spinning, or if the incoming
        #   data loop is not running, do nothing
        if ($self->childLoopSpinFlag || ! defined $self->incomingLoopCheckTime) {

            return undef;

        } else {

            # Don't let other subservient loops spin, until this loop is finished
            $self->ivPoke('childLoopSpinFlag', TRUE);
        }

        # If an MXP crosslinking operation is ready to start, start it (before continuing to
        #   process incoming data, as usual)
        # NB Crosslinking operations are only permitted when there is a real connection ('offline'
        #   connections don't count)
        if ($self->mxpRelocateMode eq 'wait_start' && $self->status eq 'connected') {

            # Check there's actually a current connection to terminate
            if (! $self->connectObj) {

                # Abandon the operation
                $self->ivPoke('mxpRelocateMode', 'none');
                $self->ivUndef('mxpRelocateHost');
                $self->ivUndef('mxpRelocatePort');
                $self->ivPoke('mxpRelocateQuietFlag', FALSE);
                $self->ivPoke('mxpRelocateQuietLineFlag', FALSE);

            } else {

                # Initiate the operation
                $self->mxpDoRelocate();
                if ($self->status eq 'disconnected') {

                    # Connection to new server failed; the incoming data loop has already halted
                    $haltFlag = TRUE;
                }
            }
        }

        # When a crosslinking operation is not in progress, check the connection is still valid
        if ($self->mxpRelocateMode eq 'none') {

            # Check connection is still valid
            if (! $self->connectObj) {

                # Connection already closed, or session is connected in offline mode
                $self->ivPoke('childLoopSpinFlag', FALSE);
                return undef;

            } elsif ($self->protocol eq 'telnet' && $self->connectObj->eof()) {

                # Telnet session has disconnected
                $self->reactDisconnect();
                $haltFlag = TRUE;
            }
        }

        if ($haltFlag) {

            # Halt this loop spin prematurely
            $self->ivPoke('childLoopSpinFlag', FALSE);
            return undef;
        }

        # Read incoming data

        # v1.0.242 - Surprisingly, ->get doesn't return all the data that has been received; this
        #   can lead to ->processIncomingData being called to process half a line, when the rest of
        #   the line has actually been received by GA::Net::Telnet (bad news for any triggers that
        #   might match the whole line). Therefore we need to continue polling GA::Net::Telnet until
        #   it returns 'undef'
        $text = '';
        do {

            $result = $self->connectObj->get(
                Errmode => sub { },                 # Do nothing on error
                Timeout => 0,
            );

            if (defined $result) {

                $text .= $result;
            }

        } until (! defined $result);

        if ($text) {

            # Check our status and amend it, if need be
            if ($self->status eq 'connecting') {

                $self->connectionComplete();
            }

            # Decode $text from the world's character set into standard Perl utf-8
            if ($self->sessionCharSet ne 'null') {

                $text = Encode::decode($self->sessionCharSet, $text);
            }

            if ($self->startCompleteFlag) {

                # Display the text in the 'main' window, if $self->start has finished its jobs...
                $self->processIncomingData($text);

            } else {

                # ...otherwise store the incoming text and wait for $self->start to finish
                $self->ivPoke('initialTextBuffer', $self->initialTextBuffer . $text);
            }

            # If this session isn't the 'main' window's visible session, set the flag which tells
            #   $self->getTabLabelText (and ->checkTabLabels) that the tab's label colour should be
            #   changed
            if ($self->mainWin->visibleSession && $self->mainWin->visibleSession ne $self) {

                $self->ivPoke('showNewTextFlag', TRUE);
            }
        }

        # Convert text to speech, if required
        if ($axmud::CLIENT->systemAllowTTSFlag && $self->ttsBuffer) {

            # Make sure the received text is visible in the 'main' window...
            $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->spinIncomingLoop');
            # ...before converting text to speech
            if (defined $self->ttsLastType && $self->ttsLastType ne 'receive') {

                # (Don't read out 'received text' again and again and again!
                $axmud::CLIENT->tts(
                    'Received text: ' . $self->ttsBuffer,
                    'receive',
                    'receive',
                    $self,
                );

            } else {

                # Last TTS conversion was something other than received text
                $axmud::CLIENT->tts($self->ttsBuffer, 'receive', 'receive', $self);
            }
        }

        # Always empty the buffer, in case ->systemAllowTTSFlag has been set in the last microsecond
        #   (or something)
        $self->ivPoke('ttsBuffer', '');

        # We can now display an automatic login confirmation message, if one has been prepared
        if ($self->loginConfirmText) {

            $self->writeText($self->loginConfirmText);
            $self->ivUndef('loginConfirmText');
        }

        # Allow other loops to spin
        $self->ivPoke('childLoopSpinFlag', FALSE);

        return 1;
    }

    sub doConnect {

        # Called by $self->start, and also by $self->mxpDoRelocate
        # Attempts to connect to the world specified by $self->host and $self->port
        #
        # Expected arguments
        #   $host       - The world's host address (default 127.0.0.1)
        #   $port       - The world's port (default 23)
        #
        # Optional arguments
        #   $protocol   - When called by $self->mxpDoRelocate, the protocol to use ('telnet', 'ssh'
        #                   or 'ssl')
        #
        # Return values
        #   'undef' on improper arguments or if the attempted connection fails
        #   1 otherwise

        my ($self, $host, $port, $protocol, $check) = @_;

        # Local variables
        my (
            $user, $pass, $capProtocol, $connectObj, $longHost, $sshObj, $ptyObj, $pid, $sslObj,
            $historyObj,
        );

        # Check for improper arguments
        if (! defined $host || ! defined $port || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doConnect', @_);
        }

        # Make sure any 'Connecting...' messages are visible immediately
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->doConnect');

        # Decide which protocol to use, if one was not specified by the calling function
        if (! $protocol) {

            if ($self->initProtocol) {

                $protocol = $self->initProtocol;

            } elsif ($self->currentWorld->protocol) {

                $protocol = $self->currentWorld->protocol;
            }
        }

        # Failsafe - default protocol is 'telnet'
        if (
            ! $protocol
            || ($protocol ne 'telnet' && $protocol ne 'ssh' && $protocol ne 'ssl')
        ) {
            $protocol = 'telnet';
        }

        # If using a temporary profile and the 'ssh' protocol, prompt the user for an SSH username/
        #   password. If the user declines to provide one, revert to the 'telnet' protocol
        # The same thing happens if the current world profile doesn't provide at least a
        #   ->sshUserName
        if ($protocol eq 'ssh') {

            if (
                ($self->initTempFlag && $self->initSshFlag)
                || ! $self->currentWorld->sshUserName
            ) {
                # Prompt the user for an SSH username/password
                ($user, $pass) = $self->mainWin->showDoubleEntryDialogue(
                    'SSH login',
                    'Enter the SSH username',
                    'Enter the SSH password',
                );

                if ($user && $pass) {

                    # Update the world profile's IVs
                    $self->currentWorld->ivPoke('protocol', 'ssh');
                    $self->currentWorld->ivPoke('sshUserName', $user);
                    if ($pass) {

                        $self->currentWorld->ivPoke('sshPassword', $pass);
                    }

                } else {

                    # Default back to telnet
                    $protocol = 'telnet';
                    $self->writeText(
                        'SSH username/password not set; reverting to a telnet connection...',
                    );

                    $self->writeText(' ');      # Blank line
                }

            } else {

                $user = $self->currentWorld->sshUserName;
                $pass = $self->currentWorld->sshPassword;
            }
        }

        # (Make sure any system messages so far are actually visible, in case the connection hangs,
        #   by calling GA::Obj::Desktop->updateWidgets
        if ($self->mxpRelocateMode eq 'none') {

            if ($protocol eq 'telnet') {
                $capProtocol = $protocol;
            } else {
                $capProtocol = uc($protocol);
            }

            $self->writeText(
                'Connecting (via ' . $capProtocol . ') to \'' . $host . ' ' . $port . '\'...',
            );
        }

        # Update some initial IVs, so that we can call $self->getHostLabelText
        $self->ivPoke('protocol', $protocol);
        $self->ivPoke('status', 'connecting');
        # Create a new connection history object, if allowed
        if ($axmud::CLIENT->connectHistoryFlag) {

            $historyObj = Games::Axmud::Obj::ConnectHistory->new($self);
            if ($historyObj) {

                # Update session IVs
                $self->ivPoke('connectHistoryObj', $historyObj);
                # Update the object's ->currentTime every second
                $self->ivPoke('historyCheckTIme', $self->sessionTime + 1);

                # Update world profile IVs
                $self->currentWorld->ivPush('connectHistoryList', $historyObj);
            }
        }

        # Update the connection info strip object for any 'internal' windows used by this
        #   session (should only be one, at this point)
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            $winObj->setHostLabel($self->getHostLabelText());
        }

        # Make sure any system messages so far are actually visible, in case the connection hangs
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->doConnect');

        # Connect to the world using the specified protocol
        if ($protocol eq 'telnet') {

            # Connect using GA::Net::Telnet
            $connectObj = Games::Axmud::Net::Telnet->new(
                Axmud_session   => $self,
                Errmode         => 'return',
                Timeout         => $self->connectTimeOut,
            );

            if (! $connectObj) {

                $self->writeError(
                    'System telnet error',
                    $self->_objClass . '->doConnect',
                );

                # React to the disconnection
                $self->reactDisconnect();

                # Return 'undef' to show failure
                return undef;
            }

        } elsif ($protocol eq 'ssh') {

            # The first argument in the call to Net::OpenSSH->new is in the form
            #   'jack@foo.bar.com'
            #   'jack:secret@foo.bar.com:10022');
            #   'jsmith@2001:db8::1428:57ab');      # IPv6
            # In addition, IPv6 addresses can be enclosed in brackets (which we will do)
            #   'jsmith@[::1]:1022'

            # Compose the first argument
            $longHost = $user;
            if ($pass) {

                $longHost .= ':' . $pass;
            }

            if ($self->currentWorld->ipv6 && $self->currentWorld->ipv6 eq $host) {
                $longHost .= '@[' . $host . ']';
            } else {
                $longHost .= '@' . $host;
            }

            if ($self->currentWorld->sshPortFlag) {

                $longHost .= ':' . $port;
            }

            # Connect using Net::OpenSSH
            $sshObj = Net::OpenSSH->new(
                $longHost,
                timeout     => $self->connectTimeOut,
                master_opts => [ -o => "StrictHostKeyChecking=no" ],
            );

            if ($sshObj) {

                ($ptyObj, $pid) = $sshObj->open2pty();

                if ($ptyObj) {

                    $connectObj = Games::Axmud::Net::Telnet->new(
                        -fhopen                     => $ptyObj,
                        Axmud_session               => $self,
                        Errmode                     => 'return',
                        Timeout                     => $self->connectTimeOut,
                    );
                }
            }

            if (! $connectObj) {

                $self->writeError(
                    'System SSH error',
                    $self->_objClass . '->doConnect',
                );

                # React to the disconnection
                $self->reactDisconnect();

                # Return 'undef' to show failure
                return undef;
            }

        } elsif ($protocol eq 'ssl') {

            # Connect using IO::Socket::SSL and GA::Net::Telnet
            $sslObj = IO::Socket::SSL->new(
                PeerAddr        => $host,
                PeerPort        => $port,
                SSL_verify_mode => 0x00,
            );

            if ($sslObj) {

                $connectObj = Games::Axmud::Net::Telnet->new(
                    -fhopen         => $sslObj,
                    Axmud_session   => $self,
                    Errmode         => 'return',
                    Timeout         => $self->connectTimeOut,
                );
            }

            if (! $connectObj) {

                $self->writeError(
                    'System SSL error',
                    $self->_objClass . '->doConnect',
                );

                # React to the disconnection
                $self->reactDisconnect();

                # Return 'undef' to show failure
                return undef;
            }
        }

        # Telnet option / sub-option negotiation
        $connectObj->option_callback(sub {

            my ($obj, $option, $isRemote, $isEnabled, $wasEnabled, $bufPosn) = @_;

            return $self->optCallback(
                $obj,
                $option,
                $isRemote,
                $isEnabled,
                $wasEnabled,
                $bufPosn,
            );
        });

        $connectObj->suboption_callback(sub {

            my ($obj, $option, $parameters) = @_;

            return $self->subOptCallback($obj, $option, $parameters);
        });

        # Use GA::Net::Telnet's option negotiation ability to write logfiles, if the GA::Client's
        #   flag is set
        if ($axmud::CLIENT->debugTelnetLogFlag) {

            $connectObj->option_log($axmud::SHARE_DIR . '/telopt.log');
        }

        # Prepare telnet options
        $self->prepareTelnetOptions($connectObj);
        # Prepare MUD protocols
        $self->prepareMudProtocols($connectObj);

        if ($protocol eq 'telnet') {

            $connectObj->open(
                Host        => $host,
                Port        => $port,
                Family      => 'any',       # Permit ipv4 or ipv6
                Errmode     => sub { return $self->connectionError(shift); },
            );

        } else {

            # For SSH, ivp4/ipv6 is already supported by the code above
            # For SSL, ipv4 and ipv6 are already enabled, due to IO::Socket::SSL being able to call
            #   on IO::Socket::INET6
            $connectObj->errmode( sub { return $self->connectionError(shift); } );
        }

        # If the connection is refused (e.g. an invalid host is specified),
        #   $self->connectionError will be called before the following lines of code can be
        #   executed.
        if ($self->status ne 'disconnected' && $self->status ne 'offline') {

            # Update IVs
            $self->ivPoke('connectObj', $connectObj);
            $self->ivPoke('sshObj', $sshObj);
            $self->ivPoke('ptyObj', $ptyObj);
            $self->ivPoke('sslObj', $sslObj);
            $self->ivPoke('host', $host);
            $self->ivPoke('port', $port);
        }

        return 1;
    }

    sub doDisconnect {

        # Called by $self->stop and also by GA::Cmd::Exit->do, XXit->do, etc
        # Terminates the connection immediately (if $self->status is 'connecting' or 'connected')
        # (Hooks using the 'disconnect' event do not fire)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $flag   - If TRUE, don't update IVs, because the calling function is about to call
        #               ->reactDisconnect to handle that. If FALSE (or 'undef'), IVs are updated
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $flag, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doDisconnect', @_);
        }

        if ($self->status eq 'connecting' || $self->status eq 'connected') {

            # Terminate the connection
            $self->connectObj->close();

            # Halt any MSP sound files that are currently playing
            foreach my $soundObj ($self->ivValues('soundHarnessHash')) {

                $soundObj->stop();
            }

            $self->ivEmpty('soundHarnessHash');

            # Show confirmation. If sessions don't share a 'main' window and one of the windows is
            #   suddenly closed, this message would be diverted to one of the other 'main' windows,
            #   which we definitely don't want, so check for that
            if ($self->defaultTabObj) {

                $self->defaultTabObj->textViewObj->showText('Disconnected from host');
            }

            # Update IVs (if allowed)
            if (! $flag) {

                $self->ivUndef('protocol');
                $self->ivUndef('connectObj');
                $self->ivUndef('sshObj');
                $self->ivUndef('ptyObj');
                $self->ivUndef('sslObj');
                $self->ivPoke('status', 'disconnected');

                $self->ivPoke('mxpRelocateMode', 'none');

                $self->ivUndef('delayedQuitTime');
                $self->ivUndef('delayedQuitCmd');
                $self->ivPoke('disconnectTime', $axmud::CLIENT->localClock);

                $self->ivEmpty('interfaceHash');
                $self->ivEmpty('interfaceNumHash');
                $self->ivPoke('interfaceCount', 0);
                $self->ivEmpty('deleteInterfaceList');

                $self->ivEmpty('triggerHash');
                $self->ivEmpty('triggerOrderList');
                $self->ivEmpty('aliasHash');
                $self->ivEmpty('aliasOrderList');
                $self->ivEmpty('macroHash');
                $self->ivEmpty('macroOrderList');
                $self->ivEmpty('timerHash');
                $self->ivEmpty('timerOrderList');
                $self->ivEmpty('hookHash');
                $self->ivEmpty('hookOrderList');

                # Update the world's connection history object, if one was created for this session
                if ($self->connectHistoryObj) {

                    $self->connectHistoryObj->set_disconnectedTime();
                }
            }

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $axmud::CLIENT->desktopObj->restrictWidgets();

            # Make sure the command entry box isn't obscured in any 'internal' windows used by this
            #   session
            foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

                my $stripObj = $winObj->ivShow('firstStripHash', 'Games::Axmud::Strip::Entry');
                if ($stripObj) {

                    $stripObj->obscureEntry(FALSE);
                }
            }
        }

        return 1;
    }

    sub doTempDisconnect {

        # Alternative to ->doDisconnect, called by $self->mxpDoRelocate
        # Disconnects the current connection, but doesn't reset all IVs, in the expectation that
        #   some of them apply to the new server, once the connection to it is completed
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doTempDisconnect', @_);
        }

        # Terminate the connection
        $self->connectObj->close();

        # (Allow MSP sound files, if any, to continue playing)

        # Show confirmation
        if (! $self->mxpRelocateQuietFlag) {

            $self->writeText(
                'Relocating (via ' . $self->protocol . ') to new server, \'' . $self->initHost
                . ' ' . $self->initPort . '\'...',
            );
        }

        # Update (some) IVs
        $self->ivUndef('connectObj');
        $self->ivUndef('sshObj');
        $self->ivUndef('ptyObj');
        $self->ivUndef('sslObj');
        $self->ivPoke('status', 'disconnected');
        $self->ivPoke('loginFlag', FALSE);

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $axmud::CLIENT->desktopObj->restrictWidgets();

        # Make sure the command entry box isn't obscured in any 'internal' windows used by this
        #   session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            my $stripObj = $winObj->ivShow('firstStripHash', 'Games::Axmud::Strip::Entry');
            if ($stripObj) {

                $stripObj->obscureEntry(FALSE);
            }
        }

        return 1;
    }

    sub reactDisconnect {

        # Called by $self->connectionError when the GA::Net::Telnet object reports an error (usually
        #   due to the host disconnecting us)
        # Also called by ->incomingDataLoop when it reads an end-of-file (usually due to the host
        #   disconnecting us)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $flag   - If TRUE, a confirmation message has already been displayed (by a call to
        #               $self->doDisconnect). If FALSE (or 'undef'), this function must display a
        #               confirmation message
        #
        # Return values
        #   'undef' on improper arguments or if this function has already been called
        #   1 otherwise

        my ($self, $flag, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->reactDisconnect', @_);
        }

        # On disconnection, this function is called from several places in the session code. In
        #   rare circumstances (such as the GA::Net::Telnet object returning TRUE to an ->eof()
        #   call), it might be called more than once. Use a flag to ignore subsequent calls
        if ($self->reactDisconnectFlag) {

            return undef;

        } else {

            # Ignore subsequent calls
            $self->ivPoke('reactDisconnectFlag', TRUE);
        }

        # Update the world's connection history object, if one was created for this session
        if ($self->connectHistoryObj) {

            $self->connectHistoryObj->set_disconnectedTime();
        }

        # If MXP is enabled, a link loss causes any open tags to be closed
        if ($self->mxpMode eq 'client_agree' && defined $self->mxpLineMode) {

            $self->emptyMxpStack();
        }

        # Display a confirmation message, if necessary
        if (! $flag) {

            $self->writeText('Connection terminated by host');
        }

        # Fire any hooks that are using the 'disconnect' hook event (but only while connected, and
        #   if allowed)
        if ($self->status eq 'connected') {

            $self->checkHooks('disconnect');
        }

        # Empty the repeat object and excess command lists - we don't want to continue sending
        #   commands after a disconnection
        $self->ivEmpty('repeatObjList');
        $self->ivPoke('excessCmdCount', 0);
        $self->ivEmpty('excessCmdList');
        $self->ivPoke('crawlModeFlag', FALSE);
        $self->ivPoke('crawlModeCmdLimit', undef);
        $self->ivPoke('crawlModeCheckTime', undef);

        # Save files (but only while connected, or while disconnecting; and only if allowed)
        if (
            (
                $self->status eq 'connected'
                || $self->status eq 'disconnected'
                || $self->status eq 'offline'
            ) && ! $self->disconnectNoSaveFlag
        ) {
            $self->pseudoCmd('save');
        }

        # Update IVs
        $self->ivUndef('protocol');
        $self->ivUndef('connectObj');
        $self->ivUndef('sshObj');
        $self->ivUndef('ptyObj');
        $self->ivUndef('sslObj');
        $self->ivPoke('status', 'disconnected');

        $self->ivPoke('mxpRelocateMode', 'none');

        $self->ivUndef('delayedQuitTime');
        $self->ivUndef('delayedQuitCmd');
        $self->ivPoke('disconnectTime', $axmud::CLIENT->localClock);

        if (! $flag) {

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $axmud::CLIENT->desktopObj->restrictWidgets();
        }

        # Remove all active interfaces
        $self->ivEmpty('interfaceHash');
        $self->ivEmpty('interfaceNumHash');
        $self->ivPoke('interfaceCount', 0);
        $self->ivEmpty('deleteInterfaceList');

        $self->ivEmpty('triggerHash');
        $self->ivEmpty('triggerOrderList');
        $self->ivEmpty('aliasHash');
        $self->ivEmpty('aliasOrderList');
        $self->ivEmpty('macroHash');
        $self->ivEmpty('macroOrderList');
        $self->ivEmpty('timerHash');
        $self->ivEmpty('timerOrderList');
        $self->ivEmpty('hookHash');
        $self->ivEmpty('hookOrderList');

        # Stop the session loop for this session (if running; not a fatal error if the loop can't be
        #   stopped, as we still need to terminate the connection itself)
        if ($self->sessionLoopObj && ! $self->stopSessionLoop()) {

            $self->writeError(
                'Could not stop the session loop',
                $self->_objClass . '->reactDisconnect',
            );
        }

        # Close any 'free' windows produced by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionFreeWins($self)) {

            # As one 'free' window is closed, its child 'free' windows are also closed, so we have
            #   to check the window still exists, before destroying it
            if ($axmud::CLIENT->desktopObj->ivExists('freeWinHash', $winObj->number)) {

                $winObj->winDestroy();
            }
        }

        # Check if there are any remaining 'grid' windows associated with this session and, if so,
        #   close them (but still don't close the 'main' window)
        $axmud::CLIENT->desktopObj->removeSessionWindows($self);

        # If this session has any 'external' windows on this session's workspace grid, and if this
        #   wasn't the current session, those 'external' windows may be invisible/minimised. Make
        #   them visible
        $axmud::CLIENT->desktopObj->revealGridWins($self);

        # Update this session's tab label to mark the session as disconnected. The TRUE flag forces
        #   the function to update the tab label
        $self->checkTabLabels(TRUE);

        # Update strip objects for any 'internal' windows used by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            my $stripObj;

            # Update information stored in each 'internal' window's connection info strip, if
            #   visible
            $winObj->setHostLabel($self->getHostLabelText());
            $winObj->setTimeLabel($self->getTimeLabelText());

            # Reset the 'internal' window's entry box
            $winObj->resetEntry();
            $stripObj = $winObj->ivShow('firstStripHash', 'Games::Axmud::Strip::Entry');
            if ($stripObj) {

                $stripObj->obscureEntry(FALSE);
            }

            # Reset the 'internal' window's blinkers, if any
            $self->turnOffBlinker(-1);      # Turn them all off
            $winObj->resetBlinkers();

            # Remove all gauges for this session, and redraw the gauge box
            # The TRUE flag means that the gauge box should be removed immediately if there are no
            #   gauges left (belonging to other sessions), rather than waiting a few seconds, as we
            #   normally would
            $stripObj = $winObj->ivShow('firstStripHash', 'Games::Axmud::Strip::GaugeBox');
            if ($stripObj) {

                $stripObj->removeSessionGauges($self, TRUE);
            }
        }

        # Make sure all changes are visible immediately
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->reactDisconnect');

        # Update gauge IVs stored by MXP
        $self->ivUndef('mxpGaugeLevel');
        $self->ivEmpty('mxpGaugeHash');

        return 1;
    }

    sub connectionError {

        # Callback, called by $self->doConnect when the GA::Net::Telnet object reports an error
        #   (usually due to a disconnection)
        #
        # Expected arguments
        #   $errorMsg   - The error message passed by GA::Net::Telnet
        #
        # Return values
        #   'undef'

        my ($self, $errorMsg, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->connectionError', @_);
        }

        # NB If attempting a connection to a host, where both the host address and host port are
        #   invalid (c.f. 'telnet deathmud'), this function is called twice. If we are already
        #   disconnected, don't display a second error
        if ($self->status eq 'disconnected' || $self->status eq 'offline') {

            return undef;
        }

        # If GA::Net::Telnet's error message is one we recognise, use our own error message
        if (
            $errorMsg =~ m/Name or service not known/i
            || $errorMsg =~ m/Unknown (remote|local) host/i
        ) {
            if ($self->mxpRelocateMode eq 'none') {

                $self->writeText(
                    'Unrecognised host \'' . $self->initHost . '\'',
                    $self->_objClass . '->connectionError',
                );

            } else {

                # During an MXP crosslinking operation, show a longer message so the user isn't
                #   left bewildered by a sudden disconnection message when the world specified a
                #   <QUIET> relocation
                $self->writeText(
                    'Relocation to new server failed, unrecognised host \''
                    . $self->mxpRelocateHost . '\'',
                    $self->_objClass . '->connectionError',
                );
            }

            # React to the disconnection. The TRUE flag means that we've already displayed a message
            $self->reactDisconnect(TRUE);

        } elsif ($errorMsg =~ m/problem connecting.*connection refused/i) {

            if ($self->mxpRelocateMode eq 'none') {

                $self->writeText(
                    'Connection to \'' . $self->initHost . '\' refused',
                    $self->_objClass . '->connectionError',
                );

            } else {

                $self->writeText(
                    'Relocation to new server failed, connection to \'' . $self->mxpRelocateHost
                    . '\' refused',
                    $self->_objClass . '->connectionError',
                );
            }

            # React to the disconnection
            $self->reactDisconnect(TRUE);

        } elsif ($errorMsg =~ m/problem connecting.*connect timed\-out/i) {

            if ($self->mxpRelocateMode eq 'none') {

                $self->writeText(
                    'Connection to \'' . $self->initHost . '\' timed out',
                    $self->_objClass . '->connectionError',
                );

            } else {

                $self->writeText(
                    'Relocation to new server failed, connection to \'' . $self->initHost
                    . '\' timed out',
                    $self->_objClass . '->connectionError',
                );
            }

            # React to the disconnection
            $self->reactDisconnect(TRUE);

        } else {

            # Otherwise, use the error message GA::Net::Telnet gave us
            $self->writeError(
                ucfirst($errorMsg),
                $self->_objClass . '->connectionError',
            );

            # React to the disconnection. Let $self->reactDisconnect display the standard
            #   'Connection terminated by host' message
            $self->reactDisconnect(FALSE);
        }

        # GA::Net::Telnet requires us to return 'undef'
        return undef;
    }

    sub connectionComplete {

        # Called by $self->spinIncomingLoop when the first text is received by the world, which
        #   signals that the connection is complete
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->connectionComplete', @_);
        }

        # We are now connected
        $self->ivPoke('status', 'connected');
        if (! $self->mxpRelocateQuietFlag) {

            $self->writeText('Connected');
        }

        # Store the time at which the connection was actually achieved, so GA::Strip::ConnectInfo
        #   can use it as a tooltip
        # (After an MXP crosslinking operation, the time connected to the new server is displayed)
        $self->ivPoke('connectedTimeString', $axmud::CLIENT->localTime());
        # Update the world's connection history object, if one was created for this session
        if ($self->connectHistoryObj) {

            $self->connectHistoryObj->set_connectedTime();
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $axmud::CLIENT->desktopObj->restrictWidgets();

        # Update the connection info strip object for any 'internal' windows used by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            $winObj->setHostLabel(
                $self->getHostLabelText(),
                'Connected since ' . $self->connectedTimeString,
            );
        }

        # (Make sure that message is visible immediately)
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->connectionComplete');

        if ($self->currentWorld->loginMode eq 'immediate') {

            # Automatic login mode 'immediate' - immediate login (character marked as 'logged in' as
            #   soon as the connection is established)
            $self->doLogin();

        } else {

            # Set the time at which $self->spinMaintainLoop should show a warning that the character
            #   hasn't logged in yet
            $self->ivPoke(
                'loginWarningTime',
                ($self->sessionTime + $axmud::CLIENT->loginWarningTime),
            );
        }

        # Fire any hooks that are using the 'connect' hook event
        $self->checkHooks('connect');

        return 1;
    }

    # Incoming data loop - process incoming data

    sub processIncomingData {

        # Called by $self->spinIncomingLoop when text is received from the world
        # Also called by $self->start when it's finished its setup jobs, and needs to display any
        #   text received from the world in the meantime
        # Called by GA::Cmd::SimulateWorld->do to simulate text received from the world
        #
        # Processes the received text, extracting escape sequences and MXP tags and applying Axmud
        #   triggers (by calling several different functions)
        #
        # Expected arguments
        #   $text           - The received text to process
        #
        # Optional arguments
        #   $noBlinkFlag    - If set to TRUE, a blinker in 'internal' windows for this session is
        #                       not turned on. If set to FALSE (or 'undef'), the blinker is turned
        #                       on as normal. This flag can be set to TRUE if this function is
        #                       called to display text that wasn't actually received from the world
        #                       (e.g. when called by the ';simulateworld' client command)
        #
        # Return values
        #   'undef' on improper arguments or if $text is an empty string
        #   1 otherwise

        my ($self, $text, $noBlinkFlag, $check) = @_;

        # Local variables
        my ($termOutput, $origText);

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processIncomingData', @_);
        }

        # Don't process an empty string
        if (! $text) {

            return undef;
        }

        $self->ivIncrement('packetCount');

        # If the global variable is set, all text received from the world (except out-of-bounds
        #   text) is also written to the terminal, with non-printable characters like ESC written as
        #   <27>
        if ($axmud::TEST_TERM_MODE_FLAG) {

            $termOutput = '';

            foreach my $char (split(//, $text)) {

                if (ord($char) >= 32 && ord($char) <= 127) {
                    $termOutput .= $char;
                } elsif ($char eq "\n" || $char eq "\r") {
                    $termOutput .= $char;
                } else {
                    $termOutput .= '<'. ord($char) . '>';
                }
            }

            print $termOutput;
        }

        # Turn on the window blinker (but only if this is the current session), and update IVs
        #   (unless $noBlinkFlag is set, in which case we're not displaying text that was actually
        #   received from the world)
        if (! $noBlinkFlag) {

            $self->turnOnBlinker(0);
        }

        # If the RawText task is open, display the received text (including any escape sequences)
        #   in the task window
        if ($self->rawTextTask && $self->rawTextTask->taskWinFlag) {

            $self->rawTextTask->insertText('<' . $self->packetCount . '>', 'RED', 'echo');
            foreach my $string (split(m/\n/, $text)) {

                $self->rawTextTask->insertText($string, 'after');
            }
        }

        # If the RawToken task is open, display the packet number immediately, and display each
        #   token as it's processed later in this function
        if ($self->rawTokenTask && $self->rawTokenTask->taskWinFlag) {

            $self->rawTokenTask->insertText('<' . $self->packetCount . '>', 'RED');
        }

        # If the last text received from the world didn't end with a newline character, it was
        #   stored as a possible prompt. If so, we can erase the stored details now
        if ($self->promptLine) {

            $self->ivUndef('promptLine');
            $self->ivUndef('promptStripLine');
            $self->ivUndef('promptCheckTime');
        }

        # If the emergency buffer was set on the previous call to this function, add its contents
        #   to the start of the received text
        if ($self->emergencyBuffer) {

            $text = $self->emergencyBuffer . $text;
            $self->ivUndef('emergencyBuffer');
        }

        # Insert an initial newline character in certain circumstances
        if (
            # There needs to be a newline character
            ($self->cmdPromptFlag || $self->nlEchoFlag)
            # ...and the world hasn't supplied one
            && ! ($text =~ m/^[\n\r]/)
            # And the current textview object hasn't just inserted one (e.g. after a system message)
            && ! $self->currentTabObj->textViewObj->insertNewLineFlag
            && $self->currentTabObj->textViewObj->bufferTextFlag
        ) {
            $text = "\n" . $text;
            $self->ivPoke('cmdPromptFlag', FALSE);
            $self->ivPoke('nlEchoFlag', FALSE);
        }

        # On the first line, detect Pueblo (if allowed)
        if (
            $axmud::CLIENT->usePuebloFlag
            && ! $self->currentWorld->ivExists('telnetOverrideHash', 'pueblo')
            && $self->puebloMode ne 'client_agree'
            && $text =~ m/This world is Pueblo (\d+\.\d+) Enhanced\./i
        ) {
            if ($self->mxpMode eq 'client_agree') {

                # Can't use Pueblo and MXP at the same time
                $self->ivPoke('puebloMode', 'client_refuse');

            } else {

                # Pueblo allowed
                $self->ivPoke('puebloMode', 'client_agree');
                $self->ivPoke('puebloVersion', $1);
                $self->send('PUEBLOCLIENT 2.01');       # Same response as zMud
            }
        }

        # Split the text into tokens:
        #   - Tokens consisting of a single newline character, "\n" or "\r"
        #   - Tokens consisting of a single escape character "\e" immediately followed by an "\e",
        #       "\n" or "\r" character, or an alphanumeric character (rare, but possible)
        #   - Tokens consisting of a valid escape sequence, starting with the escape character "\e"
        #   - If MXP is enabled, tokens which begin with "<", and the rest of an MXP tag
        #   - If MXP is enabled, tokens which begin with "&" and end ";", representing an MXP
        #       entity
        #   - Tokens of text which don't contain "\n", "\r" or "\e", and which don't contain (if
        #       MXP/Pueblo are enabled) "<" or "&"
        #   - If MSP is enabled, tokens which begin !!SOUND or !!MUSIC (case-sensitive)
        # In several situations, an invalid token is sent to $self->updateEmergencyBuffer, which
        #   decides if it's an incomplete token (we're still waiting for more text), or an
        #   invalid token, which must be discarded
        $origText = '';
        do {

            my (
                $nlTokenFlag, $tempMode, $firstChar, $mspString, $tokenFlag, $token, $data, $type,
                $successFlag, $length, $listRef, $modText, $value, $string, $showText, $colour,
                $reduceToken, $mspPosn,
                @tagList, @debugList,
            );

            # Remember the current setting of $self->mxpTempMode, so that this code block can
            #   notice if it has been applied
            $tempMode = $self->mxpTempMode;

            # When Pueblo list tags are processed during the previous iteration of this loop, they
            #   specify strings that are to be processed on the following iteration of this loop
            if ($self->puebloInsertString) {

                $text = $self->puebloInsertString . $text;
                $self->ivPoke('puebloInsertString', '');
            }

            # If Pueblo had been waiting for a new line which has since been processed, we can
            #   insert the Axmud style tag 'justify_default' on this loop
            if ($self->puebloJustifyMode eq 'wait_loop') {

                if ($self->ivExists('recvLineHash', 0)) {

                    # There are already tags stored at this offset
                    $listRef = $self->ivShow('recvLineHash', 0);
                    push (@$listRef, 'justify_default');

                } else {

                    # There are no tags stored at this offset yet
                    $self->ivAdd('recvLineHash', 0, ['justify_default']);
                }

                $self->ivPoke('puebloJustifyMode', 'normal');
            }

            # ANSI control sequences are not implemented yet, but cursor forward (ESC + [nC, where
            #   n is the number of spaces) is required by Duris opening screens, so we'll do a quick
            #   implementation here
            if ($text =~ m/^\x1b\x5b(\d+)C/) {

                my $spaces = ' ' x $1;
                $text =~ s/^\x1b\x5b(\d+)C/$spaces/;
            }

            # Extract tokens
            $firstChar = substr($text, 0, 1);
            $mspString = substr($text, 0, 7);
            if ($firstChar eq "\n" || $firstChar eq "\r") {

                # Newline token found
                $token = $firstChar;
                # ($self->nlTokenFlag is set later in the function)
                $nlTokenFlag = TRUE;
                # The Axmud colour tag to use in the RawToken task window, if it's running
                $colour = 'BLUE';

                if (
                    ($token eq "\n" && $self->crlfMode eq 'cr')
                    || ($token eq "\r" && $self->crlfMode eq 'lf')
                ) {
                    # Second part of a <CR><non-text tokens<LF> or <LF><non-text tokens><CR>
                    #   sequence, so ignore it
                    $origText .= $token;
                    $text = substr($text, length($token));

                    $self->ivPoke('crlfMode', '');

                } else {

                    if ($token eq "\n") {
                        $self->ivPoke('crlfMode', 'lf');
                    } else {
                        $self->ivPoke('crlfMode', 'cr');
                    }

                    if ($self->mxpIgnoreNewLineFlag || $self->mxpRelocateQuietLineFlag) {

                        # Ignore this newline character (but not the next one)
                        $self->ivPoke('mxpIgnoreNewLineFlag', FALSE);
                        $self->ivPoke('mxpRelocateQuietLineFlag', FALSE);
                        $origText .= $token;
                        $text = substr($text, length($token));

                    } elsif ($self->mxpParagraphFlag || $self->mxpRelocateQuietFlag) {

                        # Ignore this newline token (and any subsequent ones until the </P> tags is
                        #   processed, or the MXP crosslinking operation is complete)
                        $origText .= $token;
                        $text = substr($text, length($token));

                    } elsif ($self->puebloActiveFlag && ! $self->puebloLiteralFlag) {

                        # Ignore all newline characters when Pueblo is active
                        $origText .= $token;
                        $text = substr($text, length($token));

                    } else {

                        # Process the newline token
                        $text = substr($text, length($token));
                        @tagList = $self->processEndLine($origText, $token);
                        $origText = '';     # (Original text not required any more)
                    }
                }

            } elsif ($firstChar eq "\e") {

                if ($text =~ m/\e[\e\n\r\w]/) {

                    # Escape character followed immediately by an escape/newline character, or an
                    #   alphanumeric character. Definitely not a valid escape sequence, so use the
                    #   first "\e" as a token
                    $token = $firstChar;
                    $origText .= $token;
                    $text = substr($text, length($token));
                    @tagList = $self->processEscChar($token);

                    $colour = 'YELLOW';

                } else {

                    # Attempt to extract a valid escape sequence
                    ($token, $data, $type) = $self->extractEscSequence($text);
                    if (! defined $token) {

                        # Not a valid escape sequence; if there are no further newline/escape
                        #   characters, wait until the next packet arrives
                        $modText = $self->updateEmergencyBuffer($text, 'escape');
                        $origText .= substr($text, 0, (length($text) - length($modText)));
                        $text = $modText;

                    } else {

                        $origText .= $token;
                        $text = substr($text, length($token));
                        @tagList = $self->processEscSequence($token, $data, $type);

                        $colour = 'CYAN';
                    }
                }

            } else {

                # (From here, set $tokenFlag if a token is found, to keep the code simple)

                if (
                    ($axmud::CLIENT->allowMspFlexibleFlag || $self->nlTokenFlag)
                    && ($self->mspMode eq 'client_agree' || $self->mspMode eq 'client_simulate')
                    && ($mspString eq '!!SOUND' || $mspString eq '!!MUSIC')
                ) {
                    # Attempt to extract a valid MSP sound trigger. If the sound trigger is invalid,
                    #   it's treated as normal text
                    ($successFlag, $length) = $self->extractMspSoundTrigger($text);
                    if ($successFlag) {

                        # Valid MSP sound trigger. Process its parameters; if this process fails,
                        #   treat the token as normal text, instead
                        $token = substr($text, 0, $length);
                        if ($self->processMspSoundTrigger($token)) {

                            # Parameters were valid
                            $tokenFlag = TRUE;
                            $origText .= $token;
                            $text = substr($text, $length);

                            $colour = 'RED';
                        }
                    }
                }

                if (! $tokenFlag) {

                    # Extract MXP/Pueblo elements/entities
                    # (Don't extract MXP elements/entities in 'locked' line mode)
                    if (
                        (
                            $self->mxpMode eq 'client_agree'
                            && defined $self->mxpLineMode
                            && $self->mxpLineMode != 2
                        ) || $self->puebloMode eq 'client_agree'
                    ) {
                        if ($firstChar eq "<") {

                            $tokenFlag = TRUE;

                            # Attempt to extract a valid MXP/Pueblo element. If the element is
                            #   valid, $length contains the length of $text (starting from the
                            #   beginning) that is the valid element
                            $length = $self->extractMxpElement($text);
                            if (! defined $length) {

                                # Abnormally terminated element/tag. Treat it as ordinary, non-MXP
                                #   text
                                $tokenFlag = FALSE;

                            } elsif ($length == 0) {

                                # Incomplete element/tag; if there are no further newline/escape
                                #   characters, wait until the next packet arrives
                                $modText = $self->updateEmergencyBuffer($text, 'mxp');
                                $origText .= substr($text, 0, (length($text) - length($modText)));
                                $text = $modText;

                            } else {

                                if ($self->mxpMode eq 'client_agree') {

                                    # A valid MXP element
                                    $token = substr($text, 0, $length);
                                    $text = substr($text, $length);

                                    $colour = 'GREEN';

                                } else {

                                    # A valid Pueblo element
                                    $token = substr($text, 0, $length);
                                    $text = substr($text, $length);

                                    $colour = 'GREEN';
                                }

                                # Line spacing tags <NOBR>, <P>, </P>, <BR>, <SBR> are sent to
                                #   their own function, as is the HTML element <HR>
                                if (
                                    $self->mxpMode eq 'client_agree'
                                    && $axmud::CLIENT->ivExists(
                                        'constMxpLineSpacingHash',
                                        uc($token),
                                    )
                                ) {
                                    # (If the token is converted to a newline character, $origText
                                    #   will be converted to an empty string)
                                    $origText = $self->processMxpSpacingTag($origText, $token);

                                } elsif (
                                    $self->puebloMode eq 'client_agree'
                                    && uc($token) =~ m/\<\/?(BODY|BR|P|HR)\s?.*\>/
                                ) {
                                    # (If the token is converted to a newline character, $origText
                                    #   will be converted to an empty string)
                                    $origText = $self->processPuebloSpacingTag($origText, $token);

                                # Heading tags <H1>...<H6> are likewise sent to their own function
                                } elsif (uc($token) =~ m/\<\/?H[1-6]\s?.*\>/) {

                                    @tagList = $self->processMxpHeadingTag($origText, $token);
                                    $origText .= $token;

                                # Everything else goes here
                                } else {

                                    # Set an IV to handle <FRAME>, <DEST>, <A> and <SEND> tags by
                                    #   making the current value of $origText available to them
                                    if (defined $origText && $self->recvLineText) {

                                        $self->ivPoke('mxpOrigText', $origText);
                                    }

                                    # Proces the tag
                                    if ($self->mxpMode eq 'client_agree') {

                                        ($tokenFlag, @tagList)
                                            = $self->processMxpElement($token, $origText);

                                    } else {

                                        ($tokenFlag, @tagList)
                                            = $self->processPuebloElement($token, $origText);
                                    }

                                    # Reset the IV, whose value is no longer required
                                    $self->ivPoke('mxpOrigText', undef);

                                    if (! $tokenFlag) {

                                        # Treat this token as ordinary text, restore the changes to
                                        #   the $text to analyse from just above
                                        $text = $token . $text;
                                        $token = '';

                                    } else {

                                        $origText .= $token;
                                    }
                                }
                            }

                        } elsif ($firstChar eq "&") {

                            # Attempt to extract a valid MXP/Pueblo entity (invalid entities are
                            #   displayed 'as is'; and are processed by the next bit of code with a
                            #   call to ->processTextToken)
                            $token = $self->extractMxpEntity($text);
                            if ($token) {

                                $value = $self->processMxpEntity($token, $text);
                                if (defined $value) {

                                    # The entity has been substituted for its value. Treat the value
                                    #   as a normal text token
                                    $tokenFlag = TRUE;
                                    $origText .= $token;
                                    $text = substr($text, length($token));
                                    $self->processTextToken($value);

                                    $colour = 'MAGENTA';
                                }
                            }
                        }
                    }
                }

                if (! $tokenFlag) {

                    # The token is everything from the beginning of text up to:
                    #   - the last character which isn't an "\n", "\r", "\e"
                    #   - The last character which isn't "<" or "&", when MXP/Pueblo are enabled
                    #   - The last character before a !!SOUND or !!MUSIC tag, when MSP flexible tag
                    #       placement is enabled
                    $self->ivPoke('crlfMode', '');

                    $mspPosn = index($text, '!!SOUND');
                    if ($mspPosn == -1) {

                        $mspPosn = index($text, '!!MUSIC');
                    }

                    # (NB We want to guarantee one character; this clunky regex seems to be the only
                    #   way to do it, while capturing the backref $1)
                    if (
                        ($self->mxpMode eq 'client_agree' && defined $self->mxpLineMode)
                        || $self->puebloMode eq 'client_agree'
                    ) {
                        if ($text =~ m/^([^\n\r\e].*?)[\n\r\e\<\&]/) {

                            $token = $1;

                        } else {

                            # The rest of $text doesn't contain any of those characters, so $text is
                            #   the whole token
                            $token = $text;
                        }

                    } elsif ($text =~ m/^([^\n\r\e].*?)[\n\r\e]/) {

                        $token = $1;

                    } else {

                        $token = $text;
                    }

                    if (
                        $mspPosn > -1
                        && $mspPosn < length($token)
                        && ($self->mspMode eq 'client_agree' || $self->mspMode eq 'client_simulate')
                        && $axmud::CLIENT->allowMspFlexibleFlag
                    ) {
                        $token = substr($text, 0, $mspPosn);
                    }

                    # When Pueblo is active, reduce multiple whitespace characters to a single
                    #   space (or none, if there are no characters on the line)
                    $reduceToken = $token;
                    if (
                        $self->puebloActiveFlag
                        && ! $self->puebloLiteralFlag
                        && ! $self->puebloLiteralSampFlag
                    ) {
                        if (! ($self->recvWholeLineText =~ m/\S/)) {

                            $reduceToken =~ s/^\s+//;
                        }

                        $reduceToken =~ s/\s+/ /;
                    }

                    if (! $self->mxpRelocateQuietFlag && ! $self->mxpRelocateQuietLineFlag) {

                        $origText .= $reduceToken;
                        $text = substr($text, length($token));

                        $self->processTextToken($reduceToken);

                    } else {

                        # MXP crosslinking operation is in progress, in quiet mode, so simply
                        #   discard this token
                        $text = substr($text, length($token));
                    }

                    $colour = 'white';
                }
            }

            # For the benefit of MSP, remember whether the last token processed was a newline token,
            #   or not
            if ($nlTokenFlag) {
                $self->ivPoke('nlTokenFlag', TRUE);
            } else {
                $self->ivPoke('nlTokenFlag', FALSE);
            }

            if ($self->rawTokenTask && $self->rawTokenTask->taskWinFlag) {

                if (! $token) {

#                    # Usually not needed, because it's probably an incomplete escape sequence now
#                    #   stored in $self->emergencyBuffer
#                    $self->rawTokenTask->insertText('EMPTY TOKEN', 'red', 'ul_white');

                } elsif ($token =~ m/[\n\r]/) {

                    $self->rawTokenTask->insertText('NEWLINE', $colour);

                } else {

                    $self->rawTokenTask->insertText($token, $colour);
                }
            }

            # Handle temp secure mode, if it is marked as 'on'
            if (defined $self->mxpTempMode) {

                push (@tagList, $self->checkMxpSecureMode($text, $tempMode));
            }

            # If any Axmud colour/style tags have been generated, store them
            if (@tagList) {

                if ($self->ivExists('recvLineHash', $self->recvLineLength)) {

                    # There are already tags stored at this offset
                    $listRef = $self->ivShow('recvLineHash', $self->recvLineLength);
                    push (@$listRef, @tagList);

                } else {

                    # There are no tags stored at this offset yet
                    $self->ivAdd('recvLineHash', $self->recvLineLength, \@tagList);
                }
            }

            # If any MXP debug messages have been generated, display them, then display a summary of
            #   the token that caused the problem
            if ($self->mxpPuebloDebugList) {

                do {

                    my ($protocol, $token, $num, $msg);

                    $protocol = $self->ivShift('mxpPuebloDebugList');
                    $token = $self->ivShift('mxpPuebloDebugList');
                    $num = $self->ivShift('mxpPuebloDebugList');
                    $msg = $self->ivShift('mxpPuebloDebugList');

                    $self->writeDebug(uc($protocol) . ': ' . $msg . ' (#' . $num . ')');
                    $self->writeDebug('   Token: ' . $token);

                    # (In this debug message, replace newline characters with visible '\n' and '\r'
                    #   strings, so that the user can see the first line, plus any newline character
                    #   which terminates it)
                    $string = $origText . $text;
                    if ($string =~ m/(.*[\n\r])/) {
                        $showText = $1;
                    } else {
                        $showText = $string;
                    }

                    $showText =~ s/[\n]/\\n/;
                    $showText =~ s/[\r]/\\r/;

                    if ($showText) {
                        $self->writeDebug('   Line : ' . $showText);
                    } else {
                        $self->writeDebug('   Line : (empty line)');
                    }

                } until (! $self->mxpPuebloDebugList);
            }

        } until (! $text);

        # If the packet doesn't end with a newline character, but there is text stored in
        #   $self->recvLineText that hasn't been displayed yet, then display it
        if ($self->recvLineText) {

            $self->processIncompleteLine($origText);
        }

        # Apply any links created by MXP <A> and <SEND> tags to the current textview (if the current
        #   textview was changed during the call to this function, any links for other textviews
        #   have already been applied)
        foreach my $linkObj ($self->mxpTempLinkList) {

            $self->currentTabObj->textViewObj->add_incompleteLink($linkObj);
        }

        $self->ivEmpty('mxpTempLinkList');

        # Update the connection info strip object for any 'internal' windows used by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            my $stripObj;

            # Update information stored in each 'internal' window's connection info strip,
            #   if visible
            $winObj->setTimeLabel($self->getTimeLabelText());
        }

        # Set the 'main' window's urgency hint, if allowed
        if ($axmud::CLIENT->mainWinUrgencyFlag || $axmud::CLIENT->tempUrgencyFlag) {

            # The TRUE argument means only set the hint, if it's not already set
            $self->mainWin->setUrgent(TRUE);

            # If ->tempUrgencyFlag is set (and assuming ->mainWinUrgencyFlag is not set), the 'main'
            #   window's urgency hint should only be set once
            if ($axmud::CLIENT->tempUrgencyFlag) {

                $axmud::CLIENT->set_tempUrgencyFlag(FALSE);
            }
        }

        # Play a sound effect to signal that some text has been received, if allowed
        if ($axmud::CLIENT->tempSoundFlag) {

            $axmud::CLIENT->playSound('afk');

            # The sound should only be played once
            $axmud::CLIENT->set_tempSoundFlag(FALSE);
        }

        return 1;
    }

    sub extractEscSequence {

        # Called by $self->processIncomingData when it encounters an "\e" escape character, which
        #   probably starts an escape sequence
        # Attempts to extract a valid escape sequence
        #
        # Valid escape sequences are one of the following:
        #   - OSC colour palette escape sequences, in the form 'ESC[Pxxxxxxx'
        #   - MXP escape sequences in the form 'ESC[#z'
        #   - ANSI escape sequences in the form 'ESC[Value;...;Valuem' or 'ESC[c'
        #   - xterm titlebar escape sequences in the form 'ESC]0;xxxBEL'
        #
        # Expected arguments
        #   $text   - The remaining portion of the received text, which in this case starts with an
        #               "\e" escape character
        #
        # Return values
        #   An empty list on improper arguments or if no valid escape sequence can be extracted
        #   Otherwise, returns a list in the form (token, data, type), where 'token' is the
        #       complete escape sequence token, 'data' is the middle portion of the escape
        #       sequence (the x's in 'ESC[Pxxxxxxx' / the # in 'ESC[#z' / everything after the '['
        #       character in 'ESC[Value;...;Valuem' / the x's in 'ESC]0;xxxBEL' ) and 'type' is one
        #       of the strings 'osc', 'mxp', 'ansi' or 'xterm'

        my ($self, $text, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->extractEscSequence', @_);
            return @emptyList;
        }

        # Check there is some text after the initial "\e" character
        if (length $text <= 1) {

            return @emptyList;
        }

        # OSC colour palette escape sequences
        #   (...)               - Backref contains the whole escape sequence
        #   \x1b\x5bP           - The escape character (ASCII 27) + [P
        #   [0-9A-F]{7}         - Seven hexadecimal characters
        if ($axmud::CLIENT->oscPaletteFlag && $text =~ m/^((\x1b\x5bP)([0-9A-Fa-f]{7}))/) {

            return ($1, $3, 'osc');

        # MXP escape sequences (only extracted when MXP is enabled)
        #   (...)               - Backref contains the whole escape sequence
        #   \x1b\x5b            - The escape character (ASCII 27) + [
        #   \d+                 - One or more integers
        #   z                   - Final z character
        } elsif ($self->mxpMode eq 'client_agree' && $text =~ m/^((\x1b\x5b)(\d+)z)/) {

            return ($1, $3, 'mxp');

        # ANSI escape sequences
        #   (...)               - Backref contains the whole escape sequence
        #   \x1b\x5b            - The escape character (ASCII 27) + [
        #   \x9b                - ASCII 155
        #   .*                  - The middle section
        #   ?                   - Ungreedy match of the final character...
        #   [HfABCDsuJKmhIp]    - ...which can be any of these
        # v1.0.291 - Changed the regex so that it correctly catches an escape sequence used by
        #   Avalon (avalon.mud.de 7777), 'ESC[c' - not sure what it is (probably VT220), but in any
        #   case, it's now caught correctly, and ignored in the subsequent call to
        #   ->processEscSequence
#       } elsif ($text =~ m/^(\x1b\x5b|\x9b.*?[HfABCDsuJKmhIp])/) {
        } elsif ($text =~ m/^((\x1b\x5b|\x9b)(.*?[HfABCDsuJKmhIpc]))/) {

            return ($1, $3, 'ansi');

        # xterm escape sequences
        #   (...)               - Backref contains the whole escape sequence
        #   \x1b\x5d\0\;        - The escape character (ASCII 27) + ]0;
        #   .*                  - The middle section
        #   ?                   - Ungreedy match of the final character...
        #   \x{7}               - ...which is the BEL character (wrapped in {..} to prevent a
        #                           Perl error)
        } elsif ($text =~ m/((\x21\x5d\0\;)(.*?)\x{7})/) {

            return ($1, $3, 'xterm');

        } else {

            # No valid escape sequence found
            return @emptyList;
        }
    }

    sub extractMxpElement {

        # Called by $self->processIncomingData when it encounters a "<" character, which probably
        #   starts an MXP element
        # Attempts to extract a valid MXP element in the form <...>, which may itself contain
        #   embedded <...> structures
        #
        # NB MXP comments are in the form <!-- this is a comment -->. The intermediate section 'this
        #   is a comment' can contain anything except '-->' itself and the "\n", "\r" and "\e"
        #   characters; this function tries to extract an MXP comment, before extracting an MXP
        #   element
        # NB MXP elements are also abnormally terminated by "\n", "\r" and "\e"
        # NB The calling function should have checked that MXP is enabled, i.e. $self->mxpMode is
        #   'client_agree'
        #
        # Expected arguments
        #   $text   - The remaining portion of the received text, which in this case starts with a
        #               "<" character
        #
        # Return values
        #   'undef' on improper arguments or if an abnormally terminated element is found
        #   0 if an incomplete element is found (i.e. a valid element has probably been split
        #       across two packets)
        #   The length of a valid element otherwise

        my ($self, $text, $check) = @_;

        # Local variables
        my (
            $bracketCount, $elemLength, $endPosn, $comment, $nlPosn, $escPosn, $singleFlag,
            $doubleFlag,
        );

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->extractMxpElement', @_);
        }

        # Check there is some text after the initial "<" character
        if (length $text <= 1) {

            # Treat like an abnormally terminated element, so that the calling function uses it as
            #   non-MXP text
            return undef;
        }

        # Extract an MXP comment, if one is found
        if (substr($text, 0, 4) eq '<!--') {

            $endPosn = index($text, '-->');
            if ($endPosn == -1) {

                # Incomlete MXP comment, probably because it is split across two packets
                return 0;
            }

            $comment = substr($text, 0, ($endPosn + 3));

            # The comment is abnormally terminated by an "\n" or "\e" escape character
            $nlPosn = index($comment, "\n");
            if ($nlPosn == -1) {

                $nlPosn = index($comment, "\r");
            }

            if ($nlPosn != -1) {

                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<!--',         # We don't have the whole token yet
                        'Comment abnormally terminated by newline character',
                        1001,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<!--',
                        'Comment abnormally terminated by newline character',
                        6001,
                    );
                }

                # Abnormally terminated element
                return undef;
            }

            $escPosn = index($comment, "\e");
            if ($escPosn != -1) {

                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<!--',         # We don't have the whole token yet
                        'Comment abnormally terminated by escape character',
                        1002,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<!--',
                        'Comment abnormally terminated by escape character',
                        6002,
                    );
                }

                # Abnormally terminated element
                return undef;
            }

            # Valid comment found
            if ($self->mxpMode eq 'client_agree' && $axmud::CLIENT->debugMxpCommentFlag) {

                $self->writeDebug('MXP COMMENT: ' . $comment);

            } elsif (
                $self->puebloMode eq 'client_agree'
                && $axmud::CLIENT->debugPuebloCommentFlag
            ) {
                $self->writeDebug('PUEBLO COMMENT: ' . $comment);
            }

            return length($comment);
        }

        # Remove the first character, which is guaranteed to be a "<"
        $text = substr($text, 1);
        # The initial "<" character is the first bracket
        $bracketCount = 1;
        # (The length of the element, so far, is 1 character - i.e. the initial '<'
        $elemLength = 1;

        # Go throught the line, increasing $bracketCount for every '<' found, and decreasing it for
        #   every '>' found until $bracketCount is 0 (or the end of the line is reached)
        # However, we ignore '<' and '>' character inside a set of single '..' or double ".."
        #   quotes. (Literal ' or " quote characters can be specified within element arguments by
        #   using '' or ""; the algorithm takes account of this)
        do {

            my (
                $thisPosn, $type, $leftPosn, $rightPosn, $singlePosn, $doublePosn, $nlPosn,
                $escPosn,
            );

            $leftPosn = index($text, "<");
            if ($leftPosn > -1) {

                $thisPosn = $leftPosn;
                $type = 'left';
            }

            $rightPosn = index($text, ">");
            if ($rightPosn > -1 && (! defined $thisPosn || $rightPosn < $thisPosn)) {

                $thisPosn = $rightPosn;
                $type = 'right';
            }

            $singlePosn = index($text, "'");
            if ($singlePosn > -1 && (! defined $thisPosn || $singlePosn < $thisPosn)) {

                $thisPosn = $singlePosn;
                $type = 'single';
            }

            $doublePosn = index($text, "\"");
            if ($doublePosn > -1 && (! defined $thisPosn || $doublePosn < $thisPosn)) {

                $thisPosn = $doublePosn;
                $type = 'double';
            }

            # The element is abnormally terminated by an "\n", "\r" or "\e" escape character
            $nlPosn = index($text, "\n");
            if ($nlPosn != -1 && (! defined $thisPosn || $nlPosn < $thisPosn)) {

                $thisPosn = $nlPosn;
                $type = 'nl';
            }

            $nlPosn = index($text, "\r");
            if ($nlPosn != -1 && (! defined $thisPosn || $nlPosn < $thisPosn)) {

                $thisPosn = $nlPosn;
                $type = 'nl';
            }

            $escPosn = index($text, "\e");
            if ($escPosn != -1 && (! defined $thisPosn || $escPosn < $thisPosn)) {

                $thisPosn = $escPosn;
                $type = 'esc';
            }

            if (! $type) {

                # Incomplete MXP element, probably because it is split across two packets
                return 0;

            } elsif ($type eq 'left') {

                if (! $singleFlag && ! $doubleFlag) {

                    # < found and it's not inside a pair of single/double quotes
                    $bracketCount++;
                }

            } elsif ($type eq 'right') {

                if (! $singleFlag && ! $doubleFlag) {

                    # > found, and it's not inside a pair of single/double quotes
                    $bracketCount--;
                }

            } elsif ($type eq 'single') {

                # Ignore single quotes inside a pair of double quotes
                if (! $doubleFlag) {

                    if (! $singleFlag) {
                        $singleFlag = TRUE;
                    } else {
                        $singleFlag = FALSE;
                    }
                }

            } elsif ($type eq 'double') {

                # Ignore double quotes inside a pair of single quotes
                if (! $singleFlag) {

                    if (! $doubleFlag) {
                        $doubleFlag = TRUE;
                    } else {
                        $doubleFlag = FALSE;
                    }
                }

            } elsif ($type eq 'nl') {

                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<',            # We don't have the whole token yet
                        'Element abnormally terminated by newline character',
                        1011,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<',
                        'Element abnormally terminated by newline character',
                        6011,
                    );
                }

                # Abnormally terminated element
                return undef;

            } elsif ($type eq 'esc') {

                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<',            # We don't have the whole token yet
                        'Element abnormally terminated by escape character',
                        1012,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<',
                        'Element abnormally terminated by escape character',
                        6012,
                    );
                }

                # Abnormally terminated element
                return undef;
            }

            # We can dispense with all text up to and including this bracket or quote character
            $text = substr($text, $thisPosn + 1);
            $elemLength += ($thisPosn + 1);

        } until (! $text || ! $bracketCount);

        if ($bracketCount) {

            # Incomlete MXP element, probably because it is split across two packets
            return 0;

        } else {

            # End of the element found
            return $elemLength;
        }
    }

    sub extractMxpEntity {

        # Called by $self->processIncomingData when it encounters a "&" character, which probably
        #   starts an MXP entity
        # Attempts to extract a valid MXP entity in the form &keyword;
        # The entity keyword must start with a letter (A-Za-z) and then consist of letters, numbers
        #   or underline characters. No other characters are permitted.
        # (This function also recognises entities in the form '&#nnn;' )
        # If a valid entity isn't found, the calling function displays the text 'as is'
        #
        # NB The calling function should have checked that MXP is enabled, i.e. $self->mxpMode is
        #   'client_agree'
        #
        # Expected arguments
        #   $text   - The remaining portion of the received text, which in this case starts with a
        #               "&" character
        #
        # Return values
        #   'undef' on improper arguments or if an invalid entity is found
        #   Otherwise, returns the portion of $text comprising the entity

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->extractMxpEntity', @_);
        }

        # Check there is some text after the initial "&" character
        if (length $text <= 1) {

            return undef;
        }

        # Try to extract an entity
        if ($text =~ m/(\&[A-Za-z][A-Za-z0-9_]*\;)/) {

            return $1;

        # Entities in the form '&#nnn;' are also recognised
        } elsif ($text =~ m/(\&\#[0-9]{1,3}\;)/) {

            return $1;

        } else {

            return undef;
        }
    }

    sub extractMxpArgument {

        # Called by $self->processMxpElement
        # Extracts a single argument from a token containing an MXP element; returns both the
        #   extracted argument and the modified token
        #
        # Expected arguments
        #   $token        - The remaining portion of a token containing an MXP element
        #
        # Return values
        #   An empty list on improper arguments or if a malformed MXP argument is found
        #   Otherwise, returns a list in the form:
        #       (remaining_token, argument_name, argument_value)
        #   ...in which:
        #       - 'remaining_token' is any text that remains, after the argument has been
        #           extracted (and leading whitespace removed from it)
        #       - (so, 'remaining_token' can be an empty string, if the last argument has been
        #           extracted)
        #       - If the argument is in the form 'argument_name=argument_value', then they are
        #           returned as 'argument_name' and 'argument_value', with the middle '='
        #           character and leading/trailing whitespace removed
        #       - If the argument is not in the form 'argument_name=argument_value', then
        #           'argument_name' contains the whole argument, and 'argument_value' is undef
        #       - ('argument_name' can be an empty string, if there was no argument left to
        #           extract)

        my ($self, $token, $check) = @_;

        # Local variables
        my (
            $firstChar, $argName, $argValue,
            @emptyList,
        );

        # Check for improper arguments
        if (! defined $token || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->extractMxpArgument', @_);
            return @emptyList;
        }

        # Remove leading whitespace
        $token =~ s/^\s*//;
        if (! $token) {

            # No MXP argument left to extract
            if ($self->mxpMode eq 'client_agree') {

                $self->mxpDebug(
                    '<',            # This function doesn't know the whole token
                    'Element contains missing argument(s)',
                    1101,
                );

            } elsif ($self->puebloMode eq 'client_agree') {

                $self->puebloDebug(
                    '<',
                    'Element contains missing argument(s)',
                    6101,
                );
            }

            return ('', '', undef);
        }

        # If the MXP argument begins wih a single (or double) quote, look for the closing single (or
        #   double quote)
        # The enclosed text can contain the opposite quote (e.g. 'some " text'), or the same
        #   quote doubled ('the boy''s dog')
        $firstChar = substr($token, 0, 1);
        if ($firstChar eq '\'' || $firstChar eq '"') {

            $argName = $self->extractMxpQuote($token);
            if (! defined $argName) {

                # Improper arguments or malformed MXP argument
                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<',            # This function doesn't know the whole token
                        'Malformed argument',
                        1102,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<',
                        'Malformed argument',
                        6102,
                    );
                }

                return @emptyList;

            } else {

                # (Add 2, because the quote characters were removed from $argName);
                $token = substr($token, (length($argName) + 2));
                return ($token, $argName, undef);
            }
        }

        # Arguments by keyword are in the form
        #   argument_name=argument_value
        # ...where 'argument_name' cannot be quoted, by 'argument_value' CAN be quoted
        if ($token =~ m/^([A-Za-z][A-Za-z0-9_]*)\=/) {

            $argName = $1;                                      # Doesn't include the '='
            $argValue = '';
            $token = substr($token, (length($argName) + 1));    # Doesn't include the '='

            $firstChar = substr($token, 0, 1);
            if ($firstChar eq '\'' || $firstChar eq '"') {

                # 'argument_value' is potentially within a pair of single or double quotes

                $argValue = $self->extractMxpQuote($token);
                if (! defined $argValue) {

                    # Improper arguments or malformed MXP argument
                    if ($self->mxpMode eq 'client_agree') {

                        $self->mxpDebug(
                            '<',            # This function doesn't know the whole token
                            'Malformed argument',
                            1103,
                        );

                    } elsif ($self->puebloMode eq 'client_agree') {

                        $self->puebloDebug(
                            '<',
                            'Malformed argument',
                            6103,
                        );
                    }

                    return @emptyList;

                } else {

                    # (Add 2, because the quote characters were removed from $argName);
                    $token = substr($token, (length($argValue) + 2));
                }

            } else {

                # 'argument_value' is definitely not within a pair of single or double quotes
                if ($token =~ m/^([\S]*)\s/) {

                    $argValue = $1;
                    $token = substr($token, length($argValue));

                } else {

                    # No whitespace characters found
                    $argValue = $token;
                    $token = '';
                }

                # 'argument_name=' constructions are not allowed
                if (! $argValue) {

                    # Improper arguments or malformed MXP argument
                    if ($self->mxpMode eq 'client_agree') {

                        $self->mxpDebug(
                            '<',            # This function doesn't know the whole token
                            'Malformed argument',
                            1104,
                        );

                    } elsif ($self->puebloMode eq 'client_agree') {

                        $self->puebloDebug(
                            '<',
                            'Malformed argument',
                            6104,
                        );
                    }

                    return @emptyList;

                # argument_name=argument_value=something_else is not valid (however, '=' characters
                #   can be used inside a pair of quotes; the code above has already dealth with
                #   that)
                } elsif (index ($argValue, '=') > -1) {

                    # Improper arguments or malformed MXP argument
                    if ($self->mxpMode eq 'client_agree') {

                        $self->mxpDebug(
                            '<',            # This function doesn't know the whole token
                            'Malformed argument',
                            1105,
                        );

                    } elsif ($self->puebloMode eq 'client_agree') {

                        $self->puebloDebug(
                            '<',
                            'Malformed argument',
                            6105,
                        );
                    }

                    return @emptyList;
                }
            }

        # Arguments by position are in the form '...', "..." or a string text containing no
        #   whitespace characters / quotes / '>' characters
        } else {

            # The MXP argument is everything up to the first whitespace character
            if ($token =~ m/^(.*?)\s/) {

                $argName = $1;
                $token = substr($token, length($argName));

            } else {

                # No whitespace characters found
                $argName = $token;
                $token = '';
            }
        }

        return ($token, $argName, $argValue);
    }

    sub extractMxpQuote {

        # Called by $self->extractMxpArgument
        # Extracts an argument (or an argument value, in arguments of the type
        #   'argument_name=argument_value') from the remaining portion of an MXP element, in which
        #   the argument/argument value is enclosed by single or double quotes
        # The enclosed text can contain the opposite quote (e.g. 'some " text'), or the same
        #   quote doubled ('the boy''s dog')
        # The first and last characters, which are both ' or "s, are removed from the returned
        #   string
        #
        # Expected arguments
        #   $token        - The remaining portion of a token containing an MXP element
        #
        # Return values
        #   'undef' on improper arguments. if the first character in $token isn't a single or double
        #       quote or if $token is less than two characters long
        #   Otherwise, returns the argument/argument value, including its containing quotes

        my ($self, $token, $check) = @_;

        # Local variables
        my ($firstChar, $arg);

        # Check for improper arguments
        if (! defined $token || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->extractMxpArgument', @_);
        }

        # Check that $token contains at least two characters (the opening and closing quotes)
        if (length ($token) < 2) {

            # Improper arguments or malformed MXP argument
            if ($self->mxpMode eq 'client_agree') {

                $self->mxpDebug(
                    '<',            # This function doesn't know the whole token
                    'Element contained malformed quoted text',
                    1201,
                );

            } elsif ($self->puebloMode eq 'client_agree') {

                $self->puebloDebug(
                    '<',
                    'Element contained malformed quoted text',
                    6201,
                );
            }

            return undef;
        }

        # Remove the first character
        $firstChar = substr($token, 0, 1);
        if ($firstChar ne '\'' && $firstChar ne '"') {

            if ($self->mxpMode eq 'client_agree') {

                $self->mxpDebug(
                    '<',            # This function doesn't know the whole token
                    'Element contained malformed quoted text',
                    1202,
                );

            } elsif ($self->puebloMode eq 'client_agree') {

                $self->puebloDebug(
                    '<',
                    'Element contained malformed quoted text',
                    6202,
                );
            }

            return undef;

        } else {

            $token = substr($token, 1);
        }

        # Now go through the text of $token, looking for a closing quote matching $firstChar, but
        #   ignoring a duplicate quote (i.e. '' or "")
        $arg = '';

        do {

            my ($posn, $doublePosn);

            $posn = index($token, $firstChar);
            $doublePosn = index ($token, $firstChar. $firstChar);

            if ($posn == -1) {

                # $token does not contain a matching quote, so it's a malformed element
                if ($self->mxpMode eq 'client_agree') {

                    $self->mxpDebug(
                        '<',            # This function doesn't know the whole token
                        'Element contained malformed quoted text',
                        1203,
                    );

                } elsif ($self->puebloMode eq 'client_agree') {

                    $self->puebloDebug(
                        '<',
                        'Element contained malformed quoted text',
                        6203,
                    );
                }

                return undef;

            } elsif ($doublePosn > -1 && $doublePosn == $posn) {

                # The first quote found is a duplicate quote. Remove the text up to that point
                #   (including the duplicate quote), and then carry on
                $arg .= substr($token, 0, ($posn + 2));
                $token = substr($token, length($arg));

            } else {

                # Closing quote found. Remove it
                $arg .= substr($token, 0, $posn);
                return $arg;
            }

        } until (! $token);

        # Duplicate quote possibly found, but no matching closing quote, so it's a malformed element
        if ($self->mxpMode eq 'client_agree') {

            $self->mxpDebug(
                '<',            # This function doesn't know the whole token
                'Element contained malformed quoted text',
                1204,
            );

        } elsif ($self->puebloMode eq 'client_agree') {

            $self->puebloDebug(
                '<',
                'Element contained malformed quoted text',
                6204,
            );
        }

        return undef;
    }

    sub extractMspSoundTrigger {

        # Called by $self->processIncomingData when it encounters a token starting "!!SOUND" or
        #   "!!MUSIC", at the start of a line, which probably starts an MSP sound trigger
        # Attemps to extract a valid MSP sound trigger in the form
        #   !!SOUND(...)
        #   !!MUSIC(...)
        #
        # Expected arguments
        #   $text   - The remaining portion of the received text, which in this case starts with
        #               "!!SOUND" or "!!MUSIC"
        #
        # Return values
        #   An empty list on improper arguments or if an incomplete MSP sound trigger is found
        #   Otherwise, returns a list in the form (success_flag, length_of_token), where:
        #       'success_flag' is TRUE if a valid MSP sound trigger is found, and 'length_of_token'
        #           is the amount of text the calling function must extract from the beginning of
        #           $text
        #       'success_flag' is FALSE if an abnormally terminated MSP sound trigger is found, and
        #           'length_of_token' is returned as 0 (as it's not required by the calling
        #           function - for consistency, we return the same kind of data as
        #           ->processMxpElement returns)

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->extractMspSoundTrigger', @_);
        }

        # Extract the MSP sound trigger
        if ($text =~ m/^(\!\!SOUND\(.*\))/) {

            return (TRUE, length($1));

        } elsif ($text =~ m/^(\!\!MUSIC\(.*\))/) {

            return (TRUE, length($1));

        } else {

            return (FALSE, 0);
        }
    }

    sub processIncompleteLine {

        # Called by $self->processIncomingData when a packet of data does not end with a newline
        #   character
        #
        # Calls $self->processLinePortion to display the text we've already received (stored in
        #   $self->recvLineText, etc), and updates IVs
        #
        # Expected arguments
        #   $origText   - The original text received from the world, before any tokens were
        #                   extracted (reset to an empty string each time $self->processEndLine is
        #                   called)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $origText, $check) = @_;

        # Local variables
        my (
            $recvUsedText, $recvUsedLength,
            @emptyList,
            %recvUsedHash,
        );

        # Check for improper arguments
        if (! defined $origText || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processIncompleteLine', @_);
        }

        # Backup some IVs, in case the call to ->processLinePortion fails in strict prompts mode
        $recvUsedText = $self->recvUsedText;
        $recvUsedLength = $self->recvUsedLength;
        %recvUsedHash = $self->recvUsedHash;

        # If any text on this line has already been displayed, it is stored in $self->recvUsedText.
        #   If any tags have already been applied, they are stored in $self->recvUsedHash.
        # Update IVs, so that ->recvUsedText/->recvUsedHash contains everything that has been
        #   displayed added to anything we're about to display (in $self->recvLineText and
        #   ->recvLineHash)
        $self->combineLineHashes();

        # Prepare everything that needs to be displayed (after checking it for triggers, and so on),
        #   and then display it. The FALSE means that the received line doesn't with a newline
        #   character
        if (! $self->processLinePortion($origText, FALSE)) {

            # Failure because this line isn't a recognised prompt, and 'strict prompts mode' is
            #   enabled. Restore IVs
            $self->ivPoke('recvUsedText', $recvUsedText);
            $self->ivPoke('recvUsedLength', $recvUsedLength);
            $self->ivPoke('recvUsedHash', %recvUsedHash);

            # Store the un-displayed text in the emergency buffer, assuming that the next packet
            #   received will contain the rest of this incomplete line
            $self->updateEmergencyBuffer($origText, 'prompt');
        }

        # Update IVs ready for the next portion of this line
        $self->ivPoke('recvLineText', '');
        $self->ivPoke('recvLineLength', 0);
        $self->ivEmpty('recvLineHash');
        $self->ivPoke('recvImgLineText', '');

        # (Later code is simpler, if hashes of Axmud colour/style tags always have at least one
        #   entry, representing an empty list at the beginning of the line)
        $self->ivPoke('recvLineHash', 0, \@emptyList);

        return 1;
    }

    sub processEndLine {

        # Called by $self->processIncomingData when it encounters a single or double newline
        #    character, i.e. "\n", "\n\r", "\r\n" or "\r"
        # Also called by $self->processMxpSpacingTag when processing a <BR> or </P> tag
        #
        # Processes a newline token. Calls $self->displayLine the display the received line of
        #   text; then updates IVs ready for the next received line
        #
        # Expected arguments
        #   $origText       - The original text received from the world, before any tokens were
        #                       extracted (reset to an empty string after this function is called)
        #   $token          - An extracted token containing the single or double newline character
        #
        # Optional arguments
        #   $noCloseFlag    - Flag set to TRUE when called by $self->processMxpSpacingTag, after
        #                       processing an MXP line spacing tag like <BR>, in which case this
        #                       function doesn't close all open MXP tags
        #
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of Axmud colour/style tags generated by closing any open MXP
        #       tags (may be an empty list)

        my ($self, $origText, $token, $noCloseFlag, $check) = @_;

        # Local variables
        my (
            $wholeText,
            @emptyList, @emptyList2, @tagList,
        );

        # Check for improper arguments
        if (! defined $origText || ! defined $token || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processEndLine', @_);
            return @emptyList;
        }

        # If any text on this line has already been displayed, it is stored in $self->recvUsedText.
        #   If any tags have already been applied, they are stored in $self->recvUsedHash.
        # Update IVs, so that ->recvUsedText/->recvUsedHash contains everything that has been
        #   displayed added to anything we're about to display (in $self->recvLineText and
        #   ->recvLineHash)
        $self->combineLineHashes();

        # If empty line suppression is turned on (and the character is marked as logged in, when
        #   required), suppress empty lines, as necessary
        $wholeText = $self->recvUsedText;
        if (
            ! (
                $self->currentWorld->suppressEmptyLineCount
                && ($self->currentWorld->suppressBeforeLoginFlag || $self->loginFlag)
                && $wholeText =~ m/^\s*$/
                && (
                    # Suppress all empty lines
                    $self->currentWorld->suppressEmptyLineCount == 1
                    # Suppress consecutive empty lines
                    || $self->checkSuppressLine()
                )
            )
        ) {
            # Prepare everything that needs to be displayed (after checking it for triggers, and so
            #   on), and then display it. The TRUE means that the received line ends with a newline
            #   character
            $self->processLinePortion($origText, TRUE);
        }

        # Update IVs ready for the next line
        $self->ivPoke('recvLineText', '');
        $self->ivPoke('recvLineLength', 0);
        $self->ivEmpty('recvLineHash');
        $self->ivPoke('recvUsedText', '');
        $self->ivPoke('recvUsedLength', 0);
        $self->ivEmpty('recvUsedHash');
        $self->ivPoke('recvWholeLineText', '');
        $self->ivPoke('recvImgLineText', '');

        # (Later code is simpler, if hashes of Axmud colour/style tags always have at least one
        #   entry, representing an empty list at the beginning of the line)
        $self->ivPoke('recvLineHash', 0, \@emptyList);
        $self->ivPoke('recvUsedHash', 0, \@emptyList2);

        if (defined $self->mxpLineMode) {

            # If we're in the middle of a <V>...</V> construction, the construction is abnormally
            #   terminated
            if ($self->mxpCurrentVar) {

                $self->mxpDebug(
                    $token,
                    'Variable abnormally terminated by newline character',
                    1301,
                );

                $self->ivUndef('mxpCurrentVar');
            }

            # If we're in the middle of an <A>...</A> construction, the construction is abnormally
            #   terminated
            if ($self->mxpCurrentLink) {

                $self->mxpDebug(
                    $token,
                    'Link abnormally terminated by newline character',
                    1302,
                );

                $self->ivUndef('mxpCurrentLink');
            }

            # If we're in the middle of a <SEND>...</SEND> construction, the construction is
            #   abnormally terminated
            if ($self->mxpCurrentSend) {

                $self->mxpDebug(
                    $token,
                    'Send abnormally terminated by newline character',
                    1303,
                );

                $self->ivUndef('mxpCurrentSend');
            }

            # All outstanding tags are closed after a newline character (but not after a line
            #   spacing tag like <BR>, any only in 'open line mode')
            if (! $noCloseFlag && $self->mxpLineMode == 0) {

                push (@tagList, $self->emptyMxpStack());
            }

            # Newline characters cause the MXP line mode to be reset to the default mode (but line
            #   spacing tags like <BR> do not)
            if (! $noCloseFlag) {

                if (! $self->mxpDefaultMode) {

                    # When ->mxpDefaultMode is 0, the default mode is 'open'
                    push (@tagList, $self->setMxpLineMode(0));

                } else {

                    # ->mxpDefaultMode values of 5-7 correspond to ->mxpLineMode values of 0-2
                    push (@tagList, $self->setMxpLineMode($self->mxpDefaultMode - 5));
                }
            }
        }

        # If we're in the middle of two matching custom tags which defined tag properties, e.g.
        #   from the MXP spec, <RName>...</RName>, update the stored text
        # Represent the newline character as a space, but only if the existing stored text ends with
        #   a non-whitespace character (an acceptable compromise over textual purity)
        foreach my $key ($self->ivKeys('mxpFlagTextHash')) {

            my $text = $self->ivShow('mxpFlagTextHash', $key);

            if ($text ne '' && $text =~ m/\S$/) {

                $self->ivAdd('mxpFlagTextHash', $key, $text . ' ');
            }
        }

        # If Pueblo is waiting for a new line to insert the Axmud style tag 'justify_default',
        #   inform $self->processIncomingData that it's now safe to do so
        if ($self->puebloJustifyMode eq 'wait_newline') {

            $self->ivPoke('puebloJustifyMode', 'wait_loop');
        }

        return @tagList;
    }

    sub processLinePortion {

        # Called by $self->processIncompleteLine or ->processEndLine to display a partial or
        #   complete line of received text
        # If it's a partial line and earlier portions of this line have already been displayed,
        #   then $self->recvUsedText/->recvUsedHash, which contain the portion that's been
        #   processed so far, won't be the same as $self->recvLineText / $self->recvLineHash,
        #   which contain the portion that hasn't been displayed yet
        #
        # Expected arguments
        #   $origText       - The original text received from the world, before any tokens were
        #                       extracted (reset to an empty string each time $self->processEndLine
        #                       is called)
        #   $newLineFlag    - Flag set to TRUE if this line ends with a newline character; set to
        #                       FALSE if it doesn't (because it's a prompt, or because the whole
        #                       line hasn't been received yet)
        #
        # Return values
        #   'undef' on improper arguments or if (in 'strict prompts mode') an unrecognised
        #       prompt is found, which is assumed to be the first part of a complete line we haven't
        #       received yet
        #   1 otherwise

        my ($self, $origText, $newLineFlag, $check) = @_;

        # Local variables
        my (
            $stripText, $matchFlag, $previousOffset,
            @offsetList,
            %tagHash,
        );

        # Check for improper arguments
        if (! defined $newLineFlag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->displayLine', @_);
        }

        # Import IVs (for convenience)

        # $stripText is $origText, but stripped of everything except text tokens (so, stripped of
        #   escape sequences, MXP elements/entities, newline/escape characters, etc etc)
        $stripText = $self->recvLineText;
        # Many of the stripped tokens have been converted to Axmud colour/style tags. A hash of the
        #   tags, linked to their equivalent position (offset) in $stripText. Hash in the form
        #   $tagHash{offset} = reference_to_list_of_Axmud_colour_and_style_tags
        %tagHash = $self->recvLineHash;

        # Check for strict prompts, if necessary
        if (
            ! $newLineFlag
            && $self->currentWorld->strictPromptsMode
            # If $self->processIncomingData found an incomplete escape sequence at the end of a
            #   packet, it's been saved in $self->emergencyBuffer. In that case, there is no doubt
            #   that $stripText is not a prompt, so we don't have to check strict prompts
            && ! $self->emergencyBuffer
        ) {
            # GA::Profile::World->cmdPromptPatternList contains a list of patterns which are the
            #   world's command prompts (prompts don't have a newline character after them, as most
            #   other lines sent by the world do)
            # Test the line for recognised command prompts. If none are found, divert this
            #    text into the emergency buffer until some more text is received
            INNER: foreach my $pattern ($self->currentWorld->cmdPromptPatternList) {

                if ($stripText =~ m/$pattern/g) {

                    $matchFlag = TRUE;
                }
            }

            if (! $matchFlag) {

                # This line and any remaining lines not yet processed by the calling function will
                #   be copied into $self->emergencyBuffer
                return undef;
            }
        }

        # Because of the lack of a newline character in a command prompt, if the user types
        #   'north;north;north', the command prompt and the following text appear on the same line
        # Test $stripText against known command prompt patterns. If the line matches any of the
        #   patterns, we have to split the line at that point (unless the matching text occurs at
        #   the end of $stripText)
        # @offsetList contains the offsets of the first character after any matching text
        #
        # At the same time we'll check splitter triggers. @offsetList contains the offset of the
        #   first character after the point at which a line is split into two
        @offsetList = $self->checkLineSplit($stripText, $newLineFlag);
        if (! @offsetList) {

            # There are no command prompts in the middle of $stripText, and no splitter trigger has
            #   split the line into two or more portions, so process the whole line portion as a
            #   single line segment
            $self->processLineSegment(
                $origText,
                $stripText,
                $newLineFlag,
                FALSE,         # Let the function decide whether the segment is a prompt, or not
                %tagHash,
            );

        } else {

            # Split the line portion into separate line segments, as if we were inserting a newline
            #   character at all the positions in @offsetList
            # This job is complicated by the fact that we also have to divide %tagHash, adjusting
            #   the position of each group of tags
            $previousOffset = 0;        # Start at the beginning of $stripText
            for (my $offsetCount = 0; $offsetCount <= scalar @offsetList; $offsetCount++) {

                my (
                    $offset, $segmentText, $thisNewLineFlag, $promptFlag,
                    @emptyList,
                    %thisTagHash,
                );

                # Get the segment of the line between the last offset used (or the beginning of the
                #   line, if this is the first offset used) and the end of the matching text
                if ($offsetCount < scalar @offsetList) {

                    # Segment is not at the end of $stripText
                    $offset = $offsetList[$offsetCount];
                    $segmentText = substr(
                        $stripText,
                        $previousOffset,
                        ($offset - $previousOffset),
                    );

                    # We treat the segment as if it ended with a newline character
                    $thisNewLineFlag = TRUE;
                    # The segment definitely ends prompt
                    $promptFlag = TRUE;

                } else {

                    # Segment is at the end of $stripText
                    $segmentText = substr($stripText, $previousOffset);
                    # We treat the segment as ending with a newline character, or not, depending on
                    #   whether $line ended with a newline character
                    $thisNewLineFlag = $newLineFlag;
                    # The segment doesn't with a prompt
                    $promptFlag = FALSE;
                }

                # Create a new hash of Axmud colour/style tags that occur in the matched text,
                #   adjusting their positions (offsets) accordingly
                foreach my $posn (keys %tagHash) {

                    my ($listRef, $newPosn);

                    $listRef = $tagHash{$posn};
                    $newPosn = $posn - $previousOffset;

                    if (
                        $posn >= $previousOffset
                        && (! defined $offset || $posn < $offset)
                    ) {
                        $thisTagHash{$newPosn} = $listRef;
                    }
                }

                # %tagHash (as well as %thisTagHash) always have a key-value pair at offset 0,
                #   because it keeps the code simple
                # If a line has been split into segments, only the first portion will now have a
                #   tag hash with a key-value pair at offset 0. Add a new key-value pair at offset 0
                #   for all segments which now lack it
                if (! exists $thisTagHash{0}) {

                    $thisTagHash{0} = \@emptyList;
                }

                # Process the line segment
                $self->processLineSegment(
                    $origText,
                    $segmentText,           # Equivalent to part of $stripText
                    $thisNewLineFlag,
                    $promptFlag,
                    %thisTagHash,
                );

                # The next line segment begins after this one
                $previousOffset = $offset;
            }
        }

        if (! $newLineFlag) {

            # $stripText (the whole line, before it was divided into segments) ends with a prompt
            #   (i.e., doesn't end with a newline character). If it's a recognised command prompt,
            #   we have to set $self->cmdPromptFlag
            OUTER: foreach my $pattern ($self->currentWorld->cmdPromptPatternList) {

                if ($stripText =~ m/$pattern$/) {

                    $self->ivPoke('cmdPromptFlag', TRUE);
                    last OUTER;
                }
            }
        }

        # Write the 'receive' logfile (all other logfiles have already been written by the call to
        #   $self->processLineSegment)
        $self->writeReceiveDataLog($stripText, $self->recvImgLineText, $newLineFlag);

        return 1;
    }

    sub processLineSegment {

        # Called by $self->processLinePortion, which received a complete or partial line of
        #   received text
        # If that line portion matched recognised command prompts or matched splitter triggers, it
        #   will have been split into two or more segments; this function is called for each segment
        # Otherwise, this function is called for the whole line portion
        #
        # Expected arguments
        #   $origText       - The original text received from the world, before any tokens were
        #                       extracted (reset to an empty string each time $self->processEndLine
        #                       is called)
        #   $stripText      - A segment of the received text, comprising some or all of a line of
        #                       text received from a world, which has now been stripped of non-text
        #                       tokens like newline characters, escape sequences, etc
        #   $newLineFlag    - Flag set to TRUE if this line segment is to be treated as if it ends
        #                       with a newline character, FALSE if is to be treated as if it does
        #                       not end with a newline character
        #   $promptFlag     - Flag set to TRUE if this line segment definitely ends in a prompt; set
        #                       to FALSE if this function should decide if it ends in a prompt, or
        #                       not
        #
        # Optional arguments
        #   %tagHash        - A hash of Axmud colour/style tags, in the form:
        #                           $tagHash{offset} = reference_to_list_of_colour_and_style_tags
        #                   - (Can be an empty hash)
        #
        # Return values
        #   'undef' on improper arguments or if the line is empty, and has been suppressed due to
        #       line suppression IVs in the current world
        #   1 otherwise

        my ($self, $origText, $stripText, $newLineFlag, $promptFlag, %tagHash) = @_;

        # Local variables
        my (
            $modText, $gagFlag, $gagLogFlag, $instructListRef, $dependentCallListRef, $modFlag,
            $bufferObj, $testLine, $char, $modCmd, $bufferText, $addText, $informFlag,
            @instructList, @dependentCallList, @offsetList, @initialTagList, @specialList,
            %modTagHash, %mxpFlagTextHash,
        );

        # Check for improper arguments
        if (
            ! defined $origText || ! defined $stripText || ! defined $newLineFlag
            || ! defined $promptFlag
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->processLineSegment', @_);
        }

        # Now check the stripped line for valid URLs and valid email addresses. If any are found,
        #   modify %tagHash to include start start/stop positions of each link
        $self->extractClickLinks($stripText, \%tagHash);

        # Test the stripped line against triggers. If any of them fire, $self->checkTriggers will
        #   return a modified version of $splitText and %tagHash
        # (NB Splitter triggers have already been tested against this line)
        ($modText, $gagFlag, $gagLogFlag, $instructListRef, $dependentCallListRef, %modTagHash)
            = $self->checkTriggers($origText, $stripText, $newLineFlag, %tagHash);

        if (defined $modText) {

            # At least one trigger fired
            $modFlag = TRUE;
            @instructList = @$instructListRef;
            @dependentCallList = @$dependentCallListRef;
            %tagHash = %modTagHash;

        } else {

            # No triggers fired
            $modText = $stripText;
        }

        # Get a sorted list of offsets at which Axmud colour/style tags occur. Even if %tagHash
        #   contains no colour/style tags at all, there will still be an key-value pair
        #   corresponding to position 0 (the beginning of this line segment) - where the key is 0
        #   and the corresponding value is an empty list
        @offsetList = sort {$a <=> $b} (keys %tagHash);

        # Import the hash of stored text appearing between MXP custom elements, and reset the IV
        #   ready for the next line
        %mxpFlagTextHash = $self->mxpFlagTextStoreHash;
        $self->ivEmpty('mxpFlagTextStoreHash');

        # Process each piece of the line in turn
        OUTER: for (my $offsetCount = 0; $offsetCount < scalar @offsetList; $offsetCount++) {

            my (
                $textViewObj, $offset, $tagListRef, $nextOffset, $piece, $afterFlag, $string,
                $numString,
                @pieceTagList,
                %currentTagHash,
            );

            $textViewObj = $self->currentTabObj->textViewObj;

            # $offset is the position in $line where one or more Axmud colour/style tags occur
            $offset = $offsetList[$offsetCount];
            $tagListRef = $tagHash{$offset};
            # $nextOffset is the position in $line where the next set of Axmud colour/style tags
            #   occur. If there are no more tags after those at position $offset, then we leave
            #   $nextOffset set to 'undef'
            if ($offsetCount < (scalar @offsetList - 1)) {

                $nextOffset = $offsetList[$offsetCount + 1];
            }

            # Process the Axmud colour/style tags applying to this part of the line; the function
            #   returns a list of tags which apply to this portion of the line
            %currentTagHash = $textViewObj->colourStyleHash;
            %currentTagHash = $self->applyColourStyleTags(\%currentTagHash, $tagListRef);
            $textViewObj->set_colourStyleHash(%currentTagHash);
            # Get a list of colour/style tags that actually apply now (because
            #   GA::Obj::TextView->colourStyleHash also records those that don't)
            @pieceTagList = $textViewObj->listColourStyleTags();

            # Remember which tags applied at the beginning of this line segment, so we can pass it
            #   to $self->updateDisplayBuffer
            if ($offset == 0) {

                @initialTagList = @pieceTagList;
            }

            # Prepare a piece of the line segment to display in the current textview
            if (defined $nextOffset) {

                $piece = substr($modText, $offset, ($nextOffset - $offset));

            } else {

                $piece = substr($modText, $offset);   # Rest of the $line

                # If this segment is at the end of a received line of text which ended in a newline
                #   character (which has already been stripped away), the call to the textview
                #   object should instruct it add the newline character to this $piece
                if ($newLineFlag) {

                    $afterFlag = TRUE;
                }
            }

            # Display the piece (if allowed)
            if (
                (
                    $piece ne ''                # We have some text to display...
                    || $afterFlag               # ...or, at least, a newline character
                ) && (
                    ($modFlag && ! $gagFlag)    # Trigger fired, but doesn't have 'gag' attribute
                    || ! $modFlag               # No trigger fired
                )
            ) {
                # If this function is inserting text into the session's default textview object,
                #   then a GA::Buffer::Display is going to be created (or updated)
                # This function should inform the textview object what the number of the
                #   GA::Buffer::Display object will be, so the the textview object can compare it to
                #   its internal buffer line number (and, from there, display appropriate tooltips)
                if (! $informFlag && $self->currentTabObj eq $self->defaultTabObj) {

                    # This only needs to happen once for every call of this function
                    $informFlag = TRUE;
                    # Inform the textview object
                    $textViewObj->useDisplayBufferNum($self->displayBufferCount);
                }

                # The first time this function calls ->insertText (but not subsequent times), inform
                #   the textview object that it's about to receive some text from the session

                # When GA::Client->debugLineNumsFlag is set, show explicit display buffer line
                #   numbers at the beginning of the line
                if ($axmud::CLIENT->debugLineNumsFlag && $textViewObj->insertNewLineFlag) {

                    # Display the line number and/or explicit tags (in contrasting colours)
                    $numString = '<' . $self->displayBufferCount . '> ',
                    $textViewObj->insertText($numString, 'RED', 'ul_white', 'echo');

                    # If an incomplete link is being processed, its recorded position in its
                    #   textview will change because of the insertion of text at this point
                    if ($self->mxpCurrentLink) {

                        $self->mxpCurrentLink->updatePosn($numString);
                    }

                    if ($self->mxpCurrentSend) {

                        $self->mxpCurrentSend->updatePosn($numString);
                    }
                }

                # When GA::Client->debugLineTagsFlag is set, show explicit colour/style tags
                #   throughout the line
                $string = '';
                if ($axmud::CLIENT->debugLineTagsFlag) {

                    foreach my $tag (@$tagListRef) {

                        $string .= '[' . $tag . ']';
                    }

                    # Display the line number and/or explicit tags (in contrasting colours)
                    $textViewObj->insertText(
                        $string,
                        'white',
                        'ul_blue',
                        'echo',
                    );
                }

                if ($afterFlag) {

                    # Display the line piece and add a newline character to the end of it
                    $textViewObj->insertText(
                        $piece,
                        'after',
                        @pieceTagList,
                    );

                } else {

                    # Display the line piece without adding a newline character
                    $textViewObj->insertText(
                        $piece,
                        'echo',
                        @pieceTagList,
                    );
                }
            }
        }

        if (
            ($modFlag && ! $gagFlag)   # Trigger fired, but doesn't have 'gag' attribute
            || ! $modFlag
        ) {
            # Write to logs, if allowed (even for an empty line)
            # NB The 'receive' logfile is written by $self->writeReceiveDataLog, which is called by
            #   $self->processLinePortion, not this function
            $self->writeIncomingDataLogs($modText, $newLineFlag);
        }

        # Update the received display buffer
        $self->updateDisplayBuffer(
            $origText,
            $stripText,
            $modText,
            $newLineFlag,
            \@offsetList,
            \%tagHash,
            \@initialTagList,
            \%mxpFlagTextHash,
        );

        # If text-to-speech conversion is required, add the received text to the TTS buffer, which
        #   will be read aloud when control passes back to $self->incomingDataLoop
        if (
            $axmud::CLIENT->systemAllowTTSFlag
            && $axmud::CLIENT->ttsReceiveFlag
            # (Don't bother add any line segments which contain no readable characters, since the
            #   TTS can't read them and it may mess up the artificial full stop added below)
            && $modText =~ m/\w/
        ) {
            $bufferText = $self->ttsBuffer;
            $addText = $modText;

            if ($axmud::CLIENT->ttsSmoothFlag) {

                # To make the text sound more natural, when spoken by the TTS engine, if the
                #   existing contents of the TTS buffer ends with a newline character which is not
                #   preceded by a punctuation mark, and if the new text starts with a capital
                #   letter, insert an artificial full stop
                if (
                    # The last line stored in ->ttsBuffer ended with a newline character
                    $bufferText =~ m/\n$/
                    # The most recent line contains alphanumeric characters but doesn't end with a
                    #   punctuation mark, and is optionally followed by one or more empty lines
                    && ! ($bufferText =~ m/\w\s*[\.\,\:\;\!\?][\s*\n]+$/)
                    # The new line starts with a capital letter
                    && $addText =~ m/^\s*[A-Z]/
                ) {
                    $bufferText =~ s/\n$/\.\n/;
                }

                # If the new segment contains larget gaps (specifically, three or more consecutive
                #   whitespace characters), also insert an artificial full stop there
                $addText =~ s/(\w)\s{3,}/$1\. /;
            }

            $bufferText .= $addText;
            if ($newLineFlag) {

                $bufferText .= "\n";
            }

            # Update the buffer (if allowed)
            if (
                # Automatic login already processed
                $self->loginFlag
                # We Don't have to wait for a login before converting text
                || ! $axmud::CLIENT->ttsLoginFlag
                # We do have to wait for a login before converting text, but this is a prompt, and
                #   prompts are still converted before a login
                || ! $newLineFlag
            ) {
                $self->ivPoke('ttsBuffer', $bufferText);
            }
        }

        # Check for an MXP prompt notification
        if ($self->ivExists('mxpFlagTextHash', 'Prompt')) {

            $promptFlag = TRUE;
        }

        # Handle prompts generally
        if ($promptFlag || (! $newLineFlag)) {

            # Record details of the prompt, in case anything needs to react to it. If the last
            #   batch of text received was also a prompt
            $self->ivPoke('promptLine', $origText);
            $self->ivPoke('promptStripLine', $origText);

            if (! $promptFlag) {

                # If $self->sessionTime reaches this time without any more text being received
                #   from the world, treat it as a prompt
                $self->ivPoke(
                    'promptCheckTime',
                    $self->sessionTime + $axmud::CLIENT->promptWaitTime,
                );

            } else {

                # Process the prompt on the next spin of the maintain loop
                $self->ivPoke('promptCheckTime', $self->sessionTime);
            }
        }

        # Perform any instructions created by any triggers that fired, but don't allow Perl
        #   commands (which should already have been evaluated)
        foreach my $instruction (@instructList) {

            $self->doInstruct($instruction, TRUE);
        }

        # For any dependent triggers that fired, call the class and method specified by the fired
        #   trigger
        if (@dependentCallList) {

            do {

                my ($listRef, $class, $method);

                $listRef = shift @dependentCallList;

                $class = shift @$listRef;
                $method = shift @$listRef;

                $class->$method(@$listRef);

            } until ( ! @dependentCallList);
        }

        # Fire any hooks that are using the 'receive_text' hook event
        $self->checkHooks('receive_text', $stripText);

        # Deal with any login stuff
        if ($self->displayBufferCount) {

            $bufferObj = $self->ivShow('displayBufferHash', $self->displayBufferLast);
            $testLine = $bufferObj->modLine;

            if ($self->loginPromptsMode ne 'none') {

                # Test login patterns against the whole of the most recently-received line (in case
                #   the line matching a login pattern is split across packets)

                # In login modes 'lp' and 'world_cmd', if we're looking out for login success
                #   patterns, see if the line matches any of them
                if ($self->loginSuccessPatternList) {

                    INNER: foreach my $pattern ($self->loginSuccessPatternList) {

                        if ($testLine =~ m/$pattern/) {

                            # Success! Complete the login
                            $self->doLogin();
                            last INNER;
                        }
                    }

                # In login mode 'tiny', if we're looking out for the text which signals that the
                #   world is ready to receive the login, set the flag
                # Likewise for login mode 'world_cmd', but only if ->loginConnectPatternList is set)
                } elsif (
                    $self->loginPromptsMode eq 'tiny'
                    || ($self->loginPromptsMode eq 'world_cmd' && $self->loginConnectPatternList)
                ) {
                    INNER: foreach my $pattern ($self->loginConnectPatternList) {

                        if ($testLine =~ m/$pattern/) {

                            # Success! Set the flag that allows $self->spinMaintainLoop to call
                            #   $self->processCmdLoginMode
                            $self->ivPoke('loginConnectFoundFlag', TRUE);
                            last INNER;
                        }
                    }

                # In login mode 'mission', a login success pattern interrupts the mission
                #   immediately
                } elsif (
                    $self->loginPromptsMode eq 'mission'
                    && $self->currentWorld->loginSuccessPatternList
                ) {
                    INNER: foreach my $pattern ($self->currentWorld->loginSuccessPatternList) {

                        if ($testLine =~ m/$pattern/) {

                            # Success! Complete the login
                            $self->doLogin();
                            last INNER;
                        }
                    }
                }

            } elsif ($self->loginSpecialList && $self->initChar) {

                # Test the patterns in ->loginPatternList against the whole of the most
                #   recently-received line (in case the line matching a pattern is split across
                #   packets), initially looking for a line which matches the character's name
                #   (case-insensitively)
                $char = $self->initChar;
                if ($testLine =~ m/$char/i) {

                    # Line matches the character's name. Now, does it match the requirements of
                    #   ->loginPatternList? A list in groups of 3, in the form
                    #       (pattern, character_backref, world_command)
                    @specialList = $self->loginSpecialList;
                    do {

                        my (
                            $pattern, $backRef, $worldCmd,
                            @backRefList,
                        );

                        $pattern = shift @specialList;
                        $backRef = shift @specialList;
                        $worldCmd = shift @specialList;

                        if (@backRefList = ($testLine =~ m/$pattern/i)) {

                            # Line matches one of the specified patterns. Now check that the
                            #   character name appears in the right place (if required)
                            # NB In @backRefList, the first backreference is at index 0, so need to
                            #   subtract one
                            $backRef--;

                            if (
                                # Character names appear only once per line, so it doesn't matter
                                #   where in the line it appears
                                $backRef == -1
                                # Character names appear multiple lines per line; the one we need to
                                #   check is in backreference number $backRef
                                || lc($backRefList[$backRef]) eq lc($char)
                            ) {
                                # Success! Now we can subtitute backreferences
                                # If the corresponding world command, $worldCmd, is enclosed in
                                #   double-quotes, it's safe to use the ee modifier which will
                                #   convert $1, $2 etc into the contents of the matching
                                #   backreferences (otherwise we could end up executing arbitrary
                                #   Perl code)
                                $modCmd = $testLine;

                                if (
                                    substr($worldCmd, 0, 1) eq '"'
                                    && substr($worldCmd, -1) eq '"'
                                ) {
                                    $modCmd =~ s/$pattern/$worldCmd/iee;
                                } else {
                                    $modCmd =~ s/$pattern/$worldCmd/i
                                }
                            }
                        }

                    } until ($modCmd || ! @specialList);

                    # Absolutely no reason why $worldCmd shouldn't be set now, but just to be
                    #   safe...
                    if ($modCmd) {

                        # Send the world command, telling the world which character to login
                        $self->writeText(
                            'Automatic login: Logging in character \'' . $char . '\', sending world'
                            . ' command \'' . $modCmd . '\'',
                        );
                        $self->worldCmd($modCmd);
                    }

                    # Stop checking received lines for these patterns
                    $self->ivEmpty('loginSpecialList');
                }
            }
        }

        return 1;
    }

    sub processEscChar {

        # Called by $self->processIncomingData when it encounters an "\e" escape character which
        #   doesn't start a valid escape sequence
        #
        # Processes the escape token
        #
        # Expected arguments
        #   $token      - An extracted token containing the escape character
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of Axmud colour/style tags generated by closing any open MXP
        #       tags (may be an empty list)

        my ($self, $token, $check) = @_;

        # Local variables
        my (@emptyList, @tagList);

        # Check for improper arguments
        if (! defined $token || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processEscChar', @_);
            return @emptyList;
        }

        # Update IVs as if this token were a text token (so, the escape character is in fact
        #   displayed on a subsequent call to $self->processLinePortion)
        # v1.0.421 - Commented out, because some worlds send lone escape characters randomly
#        $self->ivPoke('recvLineText', $self->recvLineText . $token);
#        $self->ivPoke('recvLineLength', $self->recvLineLength + length($token));

        if (defined $self->mxpLineMode) {

            # If we're in the middle of a <V>...</V> construction, the construction is abnormally
            #   terminated
            if ($self->mxpCurrentVar) {

                $self->mxpDebug(
                    $token,
                    'Variable abnormally terminated by escape character',
                    1401,
                );

                $self->ivUndef('mxpCurrentVar');
            }

            # If we're in the middle of an <A>...</A> construction, the construction is abnormally
            #   terminated
            if ($self->mxpCurrentLink) {

                $self->mxpDebug(
                    $token,
                    'Link abnormally terminated by escape character',
                    1402,
                );

                $self->ivUndef('mxpCurrentLink');
            }

            # If we're in the middle of a <SEND>...</SEND> construction, the construction is
            #   abnormally terminated
            if ($self->mxpCurrentSend) {

                $self->mxpDebug(
                    $token,
                    'Send abnormally terminated by escape character',
                    1403,
                );

                $self->ivUndef('mxpCurrentSend');
            }

            # All outstanding tags are closed after a newline character (but only in 'open line
            #   mode')
            if ($self->mxpLineMode == 0) {

                @tagList = $self->emptyMxpStack();
            }
        }

        return @tagList;
    }

    sub processEscSequence {

        # Called by $self->processIncomingData when its call to $self->extractEscSequence
        #   successfully extracts an escape sequence token
        #
        # For OSC colour palette sequences, stores the newly-defined colour in $self->oscColourHash
        # For MXP escape sequences, updates IVs
        # For ANSI escape sequences, converts the sequence into a list of Axmud colour/style tags
        # For xterm titlebar escape sequences, updates IVs
        #
        # NB If we're in the middle of an MXP <V>...</V> construction, a valid escape sequence
        #   doesn't abnormally terminate the construction (but an invalid escape sequence does,
        #   handled by ->processEscChar, does)
        #
        # Expected arguments
        #   $token  - An extracted token containing the escape sequence
        #   $data   - The middle portion of the escape sequence (the x's in 'ESC[Pxxxxxxx' / the #
        #               in 'ESC[#z' / everything after the '[' character in 'ESC[Value;...;Valuem' /
        #               the x's in 'ESC]0;xxxBEL' )
        #   $type   - The type of escape sequence: 'osc', 'mxp', 'ansi', 'xterm'
        #
        # Return values
        #   An empty list on improper arguments or if the escape sequence is invalid
        #   Otherwise, returns a list of equivalent Axmud colour/style tags, when required (may be
        #       an empty list)

        my ($self, $token, $data, $type, $check) = @_;

        # Local variables
        my (
            $char, $colour, $tag,
            @emptyList, @valueList, @tagList,
            %colourHash, %styleHash,
        );

        # Check for improper arguments
        if (! defined $token || ! defined $data || ! defined $type || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processEscSequence', @_);
            return @emptyList;
        }

        # Import IVs (for quick lookup)
        %colourHash = $axmud::CLIENT->constANSIColourHash;
        %styleHash = $axmud::CLIENT->constANSIStyleHash;

        # Process the escape sequence
        if ($type eq 'osc') {

            # OSC colour palette escape sequence in the form 'ESC[Pxxxxxxx'
            # (The earlier call to $self->extractEscSequence checked that OSC colour paletters are
            #   enabled, i.e. GA::Client->oscPaletteFlag is TRUE)

            # Split the 'xxxxxxx' part into two components
            $char = substr($data, 0, 1);    # Which basic ANSI colour to change, in the range 0-F
            $colour = substr($data, 1);     # The RGB colour to use instead, e.g. FF0000

            # Update $self->oscColourHash. If $char is invalid, however, ignore this escape
            #   sequence
            $tag = $axmud::CLIENT->ivShow('constOscPaletteHash', uc($char));
            if (defined $tag) {

                $self->ivAdd('oscColourHash', $tag, '#' . uc($colour));
            }

        } elsif ($type eq 'mxp') {

            # MXP escape sequences in the form 'ESC[#z'
            # (The earlier call to $self->extractEscSequence checked that MXP is allowed, i.e.
            #   $self->mxpMode is 'client_agree')

            # As soon as the first MXP escape sequence is received, the MXP mode IVs must be set to
            #   their default values
            if (! defined $self->mxpLineMode) {

                $self->ivPoke('mxpLineMode', 0);
                $self->ivPoke('mxpDefaultMode', 0);
                $self->ivPoke('mxpTempMode', undef);
            }

            # Turn on the window blinker, and update IVs
            $self->turnOnBlinker(1);

            # $data contains an integer, the '#' in 'ESC[#\'
            # $data should be a value in the range 0-7, 10-12, 19-99; but it's allowed for
            #   these values to have leading zeros. Make sure any leading zeros are removed
            $data += 0;
            # Check that it's a valid line value
            if (
                ! (
                    ($data >= 0 && $data <= 7)
                    || ($data >= 10 && $data <= 12)
                    || ($data >= 19 && $data <= 99)
                )
            ) {
                # Invalid MXP escape sequence; ignore it
                $self->mxpDebug(
                    $token,
                    'Invalid value \'' . $data . '\' in MXP escape sequence (expected 0-7,'
                    . ' 10-12, 19, 20-99)',
                    1501,
                );

                return @emptyList;
            }

            # Process the sequence (modes 0-7 are the most frequent)
            if ($data <= 7) {

                # 0-7: Line mode escape sequences
                if ($data >= 0 && $data <= 2) {

                    # 0 - Open line, 1 - Secure line, 2 - Locked line
                    push (@tagList, $self->setMxpLineMode($data));
                    $self->ivUndef('mxpTempMode');

                } elsif ($data == 3) {

                    # 3 - Reset
                    # Close all open tags
                    push (@tagList, $self->emptyMxpStack());

                    # Update IVs
                    push (@tagList, $self->setMxpLineMode(0));
                    $self->ivPoke('mxpDefaultMode', 0);
                    $self->ivUndef('mxpTempMode');

                } elsif ($data == 4) {

                    # 4 - Temp secure mode
                    $self->ivPoke('mxpTempMode', $self->mxpLineMode);
                    push (@tagList, $self->setMxpLineMode(1, TRUE));

                } elsif ($data >= 5 && $data <= 7) {

                    # 5 - Lock open mode, 6 - Lock secure mode, 7 - Lock locked mode
                    $self->ivPoke('mxpDefaultMode', $data);
                    $self->ivUndef('mxpTempMode');
                    push (@tagList, $self->setMxpLineMode($data - 5));
                }

            } elsif ($data >=10 && $data <= 12) {

                # Room modes
                push (@tagList, 'mxpm_' . $data);       # e.g. 'mxpm_10'

            } elsif ($data == 19) {

                # Welcome text
                push (@tagList, 'mxpm_19');
                # ...which is not displayed during an MXP relocate operation
                if ($self->mxpRelocateMode ne 'none') {

                    $self->ivPoke('mxpRelocateQuietLineFlag', TRUE);
                }

            } elsif ($data >= 20 && $data <= 99) {

                # User-defined modes
                push (@tagList, 'mxpm_' . $data);       # e.g. 'mxpm_20'
            }

        } elsif ($type eq 'ansi') {

            # ANSI escape sequences in the form ESC[Value;...;Valuem or ESC[c

            # Split the part after the 'ESC[' into two components
            # A single character in the range HfABCDsuJKmhIpc
            $char = substr($data, -1, 1);
            # A list of integers separated by ';' characters
            $data = substr($data, 0, (length($data) - 1));

            # $char is a character in the range HfABCDsuJKmhIpc, but Axmud currently ignores all
            #   sequences that aren't 'Set Graphics Mode' escape sequences
            if ($char eq 'm') {

                # Some worlds (e.g. Viking MUD) use the escape sequence 'Esc[m' instead of 'Esc[0m'.
                #   Convert the former to the latter, if found
                if (! $data) {

                    $data = '0';
                }

                # $data is in the form 'Value;...;Value'. where Value is in the range 0-1, 3-9,
                #   22-25, 27-29, 30-39, 40-49
                @valueList = split(/;/, $data);
                # It's valid to use 'Value's with leading 0s. Remove the leading zeros
                foreach my $value (@valueList) {

                    if ($value =~ m/^\d+$/) {

                        $value += 0;
                    }
                }

                # NB We're using 'eq' rather than '==' to prevent a Perl error, if we analyse a
                #   sequence containing 'ESC[1,31m' rather than the correct 'ESC[1;31m', or even if
                #   the string contains invalid non-numerical characters
                #
                # Special case: xterm-256 colours will set @valueList to (38, 5, n) or (48, 5, n)
                #   (corresponding to escape sequences 'Esc[38;5;nm' and 'Esc[48;5;nm'). In this
                #   case, @valueList must contain exactly 3 values
                if ($valueList[0] eq '38' || $valueList[0] eq '48') {

                    # If it's not a valid sequence, ignore it
                    if (
                        scalar @valueList == 3
                        && $valueList[1] eq '5'
                        && $axmud::CLIENT->ivExists('xTermColourHash', 'x' . $valueList[2])
                    ) {
                        if ($valueList[0] eq '38') {
                            push (@tagList, 'x' . $valueList[2]);       # e.g. 'x255'
                        } else {
                            push (@tagList, 'ux' . $valueList[2]);      # e.g. 'ux255'
                        }
                    }

                # Otherwise, it's an ANSI escape sequence, which can contain an arbitary number of
                #   values
                } else {

                    INNER: foreach my $value (@valueList) {

                        # (Sequences listed roughly in order of popularity)

                        # 0 - All attributes off
                        if ($value eq '0') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'attribs_off');

                        # 1 - Bold on
                        } elsif ($value eq '1') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'bold');

                        # 22 - Bold off
                        } elsif ($value eq '22') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'bold_off');

                        # 30-37 - Text colours
                        # 40-47 - Underlay colours
                        } elsif (exists $colourHash{$value}) {

                            push (@tagList, $colourHash{$value});   # e.g. 'red', 'ul_red'

                        # 3 - Italics
                        # 4 - Underline on
                        # 5 - Blink (slow) on
                        # 6 - Blink (rapid) on
                        } elsif (exists $styleHash{$value}) {

                            push (@tagList, $styleHash{$value});

                        # 7 - Reverse video on
                        } elsif ($value eq '7') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'reverse');

                        # 27 - Reverse video off
                        } elsif ($value eq '27') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'reverse_off');

                        # 8 - Conceal on
                        } elsif ($value eq '8') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'conceal');

                        # 28 - Conceal off
                        } elsif ($value eq '28') {

                            # Use a dummy Axmud style tag
                            push (@tagList, 'conceal_off');

                        # 39 - Default text colour
                        } elsif ($value eq '39') {

                            push (@tagList, $self->session->currentTabObj->textViewObj->textColour);

                        # 49 - Default underlay colour
                        } elsif ($value eq '49') {

                            push (
                                @tagList,
                                $self->session->currentTabObj->textViewObj->underlayColour,
                            );
                        }
                    }
                }
            }

        } elsif ($type eq 'xterm') {

            # xterm titlebar escape sequences in the form 'ESC]0;xxxBEL'

            # Update IVs
            $self->ivPoke('xTermTitleFlag', $data);

            if ($axmud::CLIENT->xTermTitleFlag) {

                # The tab title must be updated (the routine call to $self->checkTabLabels by
                #   $self->spinMaintainLoop will handle it)
                $self->ivPoke('showXTermTitleFlag', TRUE);
            }
        }

        # Return any Axmud colour/style tags generated (may be an empty list)
        return @tagList;
    }

    sub processTextToken {

        # Called by $self->processIncomingData
        # Also called by $self->processMxpSpacingTag when processing a <SBR> or <HR> tag
        #
        # Process a text token (a string which doesn't contain any of the none-text tokens
        #   removed by the calling function, such as newline characters, escape sequences, etc)
        #
        # Expected arguments
        #   $token      - The token containing the text
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $token, $check) = @_;

        # Check for improper arguments
        if (! defined $token || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processTextToken', @_);
        }

        # Update IVs
        $self->ivPoke('recvLineText', $self->recvLineText . $token);
        $self->ivPoke('recvLineLength', $self->recvLineLength + length($token));
        $self->ivPoke('recvWholeLineText', $self->recvWholeLineText . $token);
        $self->ivPoke('recvImgLineText', $self->recvImgLineText . $token);

        # If we're in the middle of a <V>...</V> construction, update the variable's value
        if ($self->mxpCurrentVar) {

            $self->mxpCurrentVar->ivPoke('value', $self->mxpCurrentVar->value . $token);
        }

        # If we're in the middle of an <A>...</A> construction, update the link's visible text
        if ($self->mxpCurrentLink) {

            $self->mxpCurrentLink->ivPoke('text', $self->mxpCurrentLink->text . $token);
        }

        # If we're in the middle of a <SEND>...</SEND> construction, update the link's visible text
        if ($self->mxpCurrentSend) {

            $self->mxpCurrentSend->ivPoke('text', $self->mxpCurrentSend->text . $token);
        }

        # If we're in the middle of two matching custom tags which defined tag properties, e.g.
        #   from the MXP spec, <RName>...</RName>, update the stored text
        foreach my $key ($self->ivKeys('mxpFlagTextHash')) {

            $self->ivAdd('mxpFlagTextHash', $key, $self->ivShow('mxpFlagTextHash', $key) . $token);
        }

        return 1;
    }

    # Incoming data loop - process MXP/MSP/Pueblo tags

    sub processMxpElement {

        # Called by $self->processIncomingData when it encounters an MXP element (a tag like '<B>'),
        #   or by $self->processMxpCustomElement recursively
        #
        # Processes the MXP element, updating IVs and returning a corresponding list of Axmud
        #   colour/style tags, where necessary
        #
        # NB If we're in the middle of a <V>...</V> construction, an element that isn't either the
        #   opening or closing tag doesn't abnormally terminate the construction
        #
        # Expected arguments
        #   $token      - An extracted token containing the MXP element
        #
        # Optionl arguments
        #   $origText   - Specified when called by $self->processIncomingData, representing the
        #                   received line of text, up to (but not including) the MXP tag
        #   $parseMode  - Sometimes specified when this function calls itself recursively
        #               - Set to 'simple' when processing element definitions. Only simple atomic
        #                   elements like <B> (so don't allow <!ELEMENT> for example), and don't
        #                   allow closing elements like </B>
        #               - When set to 'simple_close', the same limitations apply as when set to
        #                   'simple'. In addition, the element itself isn't processed, instead the
        #                   corresponding closing element is processed (i.e. if $token is <B>, we
        #                   process </B>)
        #   %attHash    - When called by $self->processMxpCustomElement, a hash that specifies any
        #                   attribute values that apply to this tag
        #               - For example, in the element '<COLOR &col;>', this function substitutes
        #                   the '&col;' for the attribute value matching the attribute 'col', where
        #                   'col' is a key in %attHash, and the attribute value is the key's
        #                   matching value
        #               - Both the argument name and value can be substituted; so <COLOR &col;> and
        #                   <COLOR FORE=&col;> have an identical effect
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns a recognition flag (set to FALSE if the token is nothing to do with
        #       MXP which should be processed as ordinary text; set to TRUE if it's an MXP tag, even
        #       an invalid one) followed by an equivalent list of Axmud colour/style tags otherwise
        #       (may be an empty list)

        my ($self, $token, $origText, $parseMode, %attHash) = @_;

        # Local variables
        my (
            $origToken, $firstChar, $tagMode, $keyword,
            @emptyList, @argList,
        );

        # Check for improper arguments
        if (! defined $token) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpElement', @_);
            return @emptyList;
        }

        # ($token will be modified during this function, but some parts of the function require the
        #   the original token text)
        $origToken = $token;

        # Ignore comments, in the form '<!-- this is a comment -->' ($self->extractMxpElement has
        #   already checked that an element beginning with '<!--' ends with a '-->', so we only need
        #   to check the first part of the string
        if (substr($token, 0, 4) eq '<!--') {

            # Recognition flag, followed by empty Axmud colour/style tag list
            return TRUE;
        }

        # MXP elements are in the form:
        #   <keyword [args]>            ($tagMode 'open')
        #   </keyword>                  ($tagMode 'close')
        #   <!keyword [args]>           ($tagMode 'defn')

        # Remove the initial < followed by optional whitespace, and the final > preceded by optional
        #   whitespace
        $token =~ s/^\<\s*//;
        $token =~ s/\s*\>$//;
        # In case there's nothing left, don't bother looking for keywords or arguments...
        if (! $token) {

            $self->mxpDebug($origToken, 'Processed an empty element', 1601);

            # Treat token as ordinary text
            return @emptyList;
        }

        # Remove the initial / or !, if present
        $firstChar = substr($token, 0, 1);
        if ($firstChar eq '/') {

            # </keyword>
            $token = substr($token, 1);
            $tagMode = 'close';

        } elsif ($firstChar eq '!') {

            # <!keyword [args]>
            $token = substr($token, 1);
            $tagMode = 'defn';

        } else {

            # <keyword [args]>
            $tagMode = 'open';
        }

        # Remove the keyword, which 'must start with a letter (A-Z) and then consist of letters,
        #   numbers or the underline character'
        if ($token =~ m/^([A-Za-z][A-Za-z0-9_]*)/) {

            # (Simplify things by converting all keywords to upper-case)
            $keyword = uc($1);
            $token = substr($token, length($keyword));

        } else {

            if ($token =~ m/^[\'\"].*[\'\"]$/) {

                # Keyword not found
                $self->mxpDebug($origToken, 'Element contains quoted keyword', 1611);

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;

            } else {

                # Keyword not found
                $self->mxpDebug($origToken, 'Element contains invalid keyword', 1612);

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;
            }
        }

        # Some keywords are synonyms of others, e.g. <B>, <BOLD> and <STRONG> are all equivalent
        if ($axmud::CLIENT->ivExists('constMxpConvertHash', $keyword)) {

            $keyword = $axmud::CLIENT->ivShow('constMxpConvertHash', $keyword);
        }

        # Check that the keyword is either an official MXP element (e.g. <B>) or a user-defined
        #   element, stored as a key in $self->mxpElementHash
        if (
            # ($keyword is in upper case, but element names are stored in lower case)
            ! $axmud::CLIENT->ivExists('constMxpOfficialHash', $keyword)
            && ! $self->ivExists('mxpElementHash', lc($keyword))
        ) {
            # Ignore obsolete keywords
            if ($keyword eq 'SCRIPT') {

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;

            } else {

                # Treat token as ordinary text
                return @emptyList;
            }
        }

        # Handle $parseMode, if specified by the calling function
        if (defined $parseMode) {

            if ($tagMode eq 'close') {

                $self->mxpDebug(
                    $origToken,
                    'Malformed element (closing tag not valid in element definitions)',
                    1621,
                );

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;

            } elsif ($tagMode eq 'defn') {

                $self->mxpDebug(
                    $origToken,
                    'Malformed element (non-atomic tag not valid in element definitions)',
                    1622,
                );

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;

            } elsif ($parseMode eq 'simple_close') {

                # Don't process $token, which is an opening element like <B>; instead process the
                #   corresponding closing element, which is something like </B>
                $tagMode = 'close';     # Converts <B> to </B>
                $token = '';            # Converts <COLOR ...> to </COLOR>
            }
        }

        # Most keywords are only allowed in secure line mode
        if (
            ! $axmud::CLIENT->ivExists('constMxpModalHash', $keyword)
            && (! defined $self->mxpLineMode || $self->mxpLineMode != 1)
        ) {
            $self->mxpDebug($origToken, 'Secure element used in unsecure line', 1623);

            # Recognition flag, followed by empty Axmud colour/style tag list
           return TRUE;
        }

        # Get a list of arguments, separated by one or more whitespace characters
        # (The whole argument can be single-quoted, or double-quoted, in which case it can contain
        #   embedded whitespace or the '>' character)
        if ($token) {

            do {

                my (
                    $argName, $argValue, $key, $value,
                    @backRefList,
                );

                ($token, $argName, $argValue) = $self->extractMxpArgument($token);
                if (! defined $token) {

                    # Improper arguments, or malformed argument
                    $self->mxpDebug($origToken, 'Malformed element', 1631);

                    # Recognition flag, followed by empty Axmud colour/style tag list
                    return TRUE;

                } else {

                    # Substitute the argument name or value, if they match one or more of the
                    #   attributes specified in %attHash
                    # e.g. In <COLOR &col;> and <COLOR FORE=&col;>, if there's a key in %attHash
                    #   called 'col', subsitute the &col% for the key's corresponding value
                    if (! defined $argValue) {

                        @backRefList = ($argName =~ m/\&(\w+)\;/);
                        foreach my $backRef (@backRefList) {

                            if (exists $attHash{$backRef}) {

                                $value = $attHash{$backRef};
                                $argName =~ s/\&\w+\;/$value/g;
                            }
                        }

                    } elsif (defined $argValue) {

                        @backRefList = ($argValue =~ m/\&(\w+)\;/);
                        foreach my $backRef (@backRefList) {

                            if (exists $attHash{$backRef}) {

                                $value = $attHash{$backRef};
                                $argValue =~ s/\&\w+\;/$value/g;
                            }
                        }
                    }

                    # If the argument is not in the form 'argument_name=argument_value', then
                    #   $argValue is 'undef', and $argName contains the whole argument
                    push (@argList, $argName, $argValue);

                    # After removing the 'argument' or the 'argument_name=argument_value'
                    #   construction, if there's anything left in $token, it must start with a
                    #   whitespace character
                    # This ensures that there are whitespace character(s) between each argument,
                    #   and prevents constructions like: name='value'name='value'
                    if ($token && $token =~ m/^\S/) {

                        $self->mxpDebug($origToken, 'Malformed element', 1632);

                        # Recognition flag, followed by empty Axmud colour/style tag list
                        return TRUE;
                    }
                }

            } until (! $token);
        }

        # Process each type of MXP element in its own function

        # Process modal elements: <B> <I> <U> <S> <H> <COLOR> <FONT>
        if ($axmud::CLIENT->ivExists('constMxpModalHash', $keyword)) {

            return (TRUE, $self->processMxpModalElement($origToken, $tagMode, $keyword, @argList));

        # Process HTML elements: <SMALL> <TT>
        } elsif ($keyword eq 'SMALL' || $keyword eq 'TT') {

            return (TRUE, $self->processMxpHtmlElement($origToken, $tagMode, $keyword, @argList));

        # Process element definitions: <!ELEMENT>
        } elsif ($keyword eq 'EL') {

            return (TRUE, $self->processMxpElementDefn($origToken, $tagMode, $keyword, @argList));

        # Process attribute lists for user-defined elements: <!ATTLIST>
        } elsif ($keyword eq 'AT') {

            return (TRUE, $self->processMxpAttElement($origToken, $tagMode, $keyword, @argList));

        # Process entity element: <!ENTITY>
        } elsif ($keyword eq 'EN') {

            return (TRUE, $self->processMxpEntElement($origToken, $tagMode, $keyword, @argList));

        # Process direct setting of variable (entity values): <V>...</V>
        } elsif ($keyword eq 'V') {

            return (TRUE, $self->processMxpVarElement($origToken, $tagMode, $keyword, @argList));

        # Process support requests: <SUPPORT>
        } elsif ($keyword eq 'SUPPORT') {

            return (
                TRUE,
                $self->processMxpSupportElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process frames: <FRAME>
        } elsif ($keyword eq 'FRAME') {

            return (TRUE, $self->processMxpFrameElement($origToken, $tagMode, $keyword, @argList));

        # Process cursor control: <DEST>...</DEST>
        } elsif ($keyword eq 'DEST') {

            return (TRUE, $self->processMxpDestElement($origToken, $tagMode, $keyword, @argList));

        # Process direct setting of clickable links: <A>...</A>
        } elsif ($keyword eq 'A') {

            return (
                TRUE,
                $self->processMxpLinkElement($origToken, $tagMode, $keyword, FALSE, @argList),
            );

        # Process direct setting of send links: <SEND>...</SEND>
        } elsif ($keyword eq 'SEND') {

            return (
                TRUE,
                $self->processMxpSendElement($origToken, $tagMode, $keyword, FALSE, @argList),
            );

        # Process sounds: <SOUND>, <MUSIC>
        } elsif ($keyword eq 'SOUND' || $keyword eq 'MUSIC') {

            return (TRUE, $self->processMxpSoundElement($origToken, $tagMode, $keyword, @argList));

        # Process images: <IMAGE>
        } elsif ($keyword eq 'IMAGE') {

            return (TRUE, $self->processMxpImageElement($origToken, $tagMode, $keyword, @argList));

        # Process sound/image filters: <FILTER>
        } elsif ($keyword eq 'FILTER') {

            return (TRUE, $self->processMxpFilterElement($origToken, $tagMode, $keyword, @argList));

        # Process gauges: <GAUGE>, <STAT>
        } elsif ($keyword eq 'GAUGE' || $keyword eq 'STAT') {

            return (TRUE, $self->processMxpGaugeElement($origToken, $tagMode, $keyword, @argList));

        # Process MXP crosslinking: <RELOCATE>
        } elsif ($keyword eq 'RELOCATE') {

            return (
                TRUE,
                $self->processMxpCrosslinkElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process MXP logins: <USER>, <PASSWORD>
        } elsif ($keyword eq 'USER' || $keyword eq 'PASSWORD') {

            return (TRUE, $self->processMxpLoginElement($origToken, $tagMode, $keyword, @argList));

        # Process other official MXP elements
        } elsif ($axmud::CLIENT->ivExists('constMxpOfficialHash', $keyword)) {

            return (
                TRUE,
                $self->processMxpOfficialElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process custom elements
        } elsif ($self->ivExists('mxpElementHash', lc($keyword))) {

            return (TRUE, $self->processMxpCustomElement($origToken, $tagMode, $keyword, @argList));

        # (This should never be executed)
        } else {

            $self->mxpDebug($origToken, 'Internal error while processing element', 1641);

            # Recognition flag, followed by empty Axmud colour/style tag list
            return TRUE;
        }
    }

    sub processMxpModalElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP model element: <B> <I> <U> <S> <H> <COLOR> <FONT>
        # (NB Other modal elements, such as <A> and <SEND>, are processed in their own functions)
        # (NB In Pueblo, <STRIKE> is used in place of <S>)
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $textViewObj, $string, $blinkFlag, $string2, $blinkFlag2, $stackObj, $converted,
            @emptyList, @origList, @checkList, @tagList,
            %checkHash, %ivHash, %stackHash, %convertHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpModalElement', @_);
            return @emptyList;
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # Convert Pueblo <STRIKE> to MXP <S>
        if ($self->puebloMode eq 'client_agree' && $keyword eq 'STRIKE') {

            $keyword = 'S';
        }

        # Process atomic modal elements that have arguments
        if ($keyword eq 'C' || $keyword eq 'F') {

            # <COLOR FORE=foreground [BACK=background]>
            # </COLOR>
            # <FONT FACE=name [SIZE=size] [COLOR=foreground] [BACK=background]>
            # </FONT>
            if (
                ($tagMode eq 'open' && ! @argList)
                || ($tagMode eq 'close' && @argList)
                || $tagMode eq 'defn'
            ) {
                $self->mxpDebug($origToken, 'Malformed element', 2001);

                return @emptyList;

            # <COLOR FORE=foreground [BACK=background]>
            # <FONT FACE=name [SIZE=size] [COLOR=foreground] [BACK=background]>
            } elsif ($tagMode eq 'open') {

                # Process @argList
                if ($keyword eq 'C') {

                    @origList = @checkList = ('fore', 'back');
                    # Hash of argument names which don't take a corresponding value
                    %checkHash = ();
                    # Default argument values
                    %ivHash = (
                        'fore'      => undef,
                        'back'      => undef,
                    );

                } else {

                    @origList = @checkList = ('face', 'size', 'color', 'back');
                    # Hash of argument names which don't take a corresponding value
                    %checkHash = ();
                    # Default argument values
                    %ivHash = (
                        'face'      => undef,
                        'size'      => undef,
                        'color'     => undef,
                        'back'      => undef,
                    );
                }

                if (@argList) {

                    do {

                        my ($argName, $argValue) = $self->findMxpArgsByPosn(
                            \@origList,
                            \@checkList,
                            \%ivHash,
                            \%checkHash,
                            shift @argList,     # not 'undef'
                            shift @argList,     # might be 'undef'
                        );

                        if (! defined $argName) {

                            # Unrecognised argument name, or repeating argument name
                            $self->mxpDebug($origToken, 'Malformed element', 2002);

                            return @emptyList;

                        } else {

                            $ivHash{$argName} = $argValue;
                        }

                    } until (! @argList);
                }

                # 'foreground' and 'background' can be standard HTML colours (keys in
                #   GA::Client->constHtmlColourHash), RGB colour tags (in the form '#000000')
                # http://www.zuggsoft.com/zmud/mxp.htm states that we can use 'color attribute names
                #   such as 'blink', but neglects to specify WHICH colour attributes we can use - so
                #   'blink' is the only one Axmud implements
                # If used, Axmud expects that 'foreground'/'background' will be in the form
                #   'colour,blink' or 'blink,colour'
                if ($ivHash{'fore'}) {

                    ($string, $blinkFlag) = $self->convertMxpColour($ivHash{'fore'}, FALSE);
                    if (! defined $string) {

                        $self->mxpDebug(
                            $origToken,
                            'Unrecognised foreground colour \'' . $ivHash{'fore'} . '\'',
                            2003,
                        );

                        return @emptyList;

                    } else {

                        push (@tagList, $string);
                        $stackHash{'colour_foreground'} = $string;
                    }

                } elsif ($ivHash{'color'}) {

                    ($string, $blinkFlag) = $self->convertMxpColour($ivHash{'color'}, FALSE);
                    if (! defined $string) {

                        $self->mxpDebug(
                            $origToken,
                            'Unrecognised foreground colour \'' . $ivHash{'color'} . '\'',
                            2004,
                        );

                        return @emptyList;

                    } else {

                        push (@tagList, $string);
                        $stackHash{'colour_foreground'} = $string;
                    }
                }

                if ($ivHash{'back'}) {

                    ($string2, $blinkFlag2) = $self->convertMxpColour($ivHash{'back'}, TRUE);
                    if (! defined $string2) {

                        $self->mxpDebug(
                            $origToken,
                            'Unrecognised background colour \'' . $ivHash{'back'} . '\'',
                            2005,
                        );

                        return @emptyList;

                    } else {

                        push (@tagList, $string2);
                        $stackHash{'colour_background'} = $string2;
                    }
                }

                if ($blinkFlag || $blinkFlag2) {

                    push (@tagList, 'blink_slow');
                    $stackHash{'blink_flag'} = TRUE;
                }

                # If the GA::Client flag is not set, still allow a <FONT> tag to change the font
                #   colour, but don't allow it to change the font name (and don't display an error
                #   in complaint)
                if ($keyword eq 'F' && $axmud::CLIENT->allowMxpFontFlag) {

                    if ($ivHash{'face'}) {

                        $stackHash{'font_name'} = $ivHash{'face'};
                    }

                    if ($ivHash{'size'}) {

                        $stackHash{'font_size'} = $ivHash{'size'};
                    }

                    # Create a dummy style tag that $self->applyColouStyleTags can interpret
                    #   e.g. 'mxpf_monospace_bold_12'
                    push (@tagList, $self->createMxpFontTag(%stackHash));
                }

                # Create a new MXP stack object and store it in the current textview object,
                #   updating the latter's IVs
                if (! $textViewObj->createMxpStackObj($self, $keyword, %stackHash)) {

                    $self->mxpDebug($origToken, 'Internal error while processing element', 2006);

                    return @emptyList;

                } else {

                    # Operation complete
                    return @tagList;
                }

            # </C>
            # </F>
            } else {

                # Cancel the MXP text attribute
                return $self->popMxpStack($keyword);
            }

        # Process atomic modal elements that don't have arguments, i.e. <B> <I> <U> <S>/<STRIKE> <H>
        } else {

            if (@argList || $tagMode eq 'defn') {

                # These modal elements shouldn't have arguments
                $self->mxpDebug($origToken, 'Malformed element', 2011);

                return @emptyList;

            } elsif ($tagMode eq 'open') {

                # Convert keywords like 'B' into the corresponding keys used in
                #   GA::Obj::TextView->mxpModalStackHash, e.g. convert 'B' to 'bold_flag'
                $converted = $axmud::CLIENT->ivShow('constMxpStackConvertHash', $keyword);
                if (defined $converted) {

                    # (In ->mxpModalStackHash, the corresponding value is always TRUE or FALSE)
                    $convertHash{$converted} = TRUE;
                }

                # Create a new MXP stack object and store it in the current textview object,
                #   updating the latter's IVs
                if (! $textViewObj->createMxpStackObj($self, $keyword, %convertHash)) {

                    $self->mxpDebug($origToken, 'Internal error while processing element', 2012);

                    return @emptyList;
                }

                # Return the equivalent Axmud style tag, e.g. 'italics'
                if ($keyword ne 'B' && $keyword ne 'H') {

                    push (@tagList, $axmud::CLIENT->ivShow('constMxpModalOnHash', $keyword));

                } else {

                    # Create a dummy style tag that $self->applyColourStyleTags can interpret
                    #   e.g. 'mxpf_monospace_bold_12'
                    push (@tagList, $self->createMxpFontTag($textViewObj->mxpModalStackHash));
                }

                return @tagList;

            } else {

                # Cancel the MXP text attribute
                return $self->popMxpStack($keyword);
            }
        }
    }

    sub processMxpHtmlElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP HTML element: <SMALL> <TT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            @emptyList, @tagList,
            %stackHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpHtmlElement', @_);
            return @emptyList;
        }

        # HTML elements don't have arguments
        if ($tagMode eq 'defn' || @argList) {

            $self->mxpDebug($origToken, 'Malformed element', 2101);

            return @emptyList;
        }

        # Ignore these tags if the client flag is set
        if (! $axmud::CLIENT->allowMxpFontFlag) {

            return @emptyList;
        }

        # <SMALL>
        # <TT>
        if ($tagMode eq 'open') {

            if ($keyword eq 'SMALL') {

                # Use same size as <H6>
                $stackHash{'font_size'} = $axmud::CLIENT->ivShow('constHeadingSizeHash', 6)
                                            * $axmud::CLIENT->constFontSize;
                $stackHash{'spacing'} = $axmud::CLIENT->ivShow('constHeadingSpacingHash', 6)
                                            * $axmud::CLIENT->constFontSize;
                $stackHash{'font_name'} = $axmud::CLIENT->constFont;
                $stackHash{'bold_flag'} = FALSE;

            } elsif ($keyword eq 'TT') {

                # Use plain old monospace for a teletype text tag
                $stackHash{'font_size'}
                    = $self->currentTabObj->textViewObj->ivShow('mxpModalStackHash', 'font_size');
                $stackHash{'spacing'}
                    = $self->currentTabObj->textViewObj->ivShow('mxpModalStackHash', 'spacing');
                $stackHash{'font_name'} = $axmud::CLIENT->constFont;
                $stackHash{'bold_flag'} = FALSE;
            }

            # Create a dummy style tag that $self->applyColourStyleTags can interpret
            #   e.g. 'mxpf_monospace_bold_12'
            push (@tagList, $self->createMxpFontTag(%stackHash));

            # Create a new MXP stack object and store it in the current textview object,
            #   updating the latter's IVs
            if (
                ! $self->currentTabObj->textViewObj->createMxpStackObj(
                    $self,
                    $keyword,
                    %stackHash,
                )
            ) {
                $self->mxpDebug($origToken, 'Internal error while processing element', 2102);

                return @emptyList;

            } else {

                # Operation complete
                return @tagList;
            }

        # </SMALL>
        # </TT>
        } else {

            # Cancel the MXP text attribute
            return $self->popMxpStack($keyword);
        }
    }

    sub processMxpElementDefn {

        # Called by $self->processMxpElement
        #
        # Process an MXP element definition: <!ELEMENT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $elementObj,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpElementDefn', @_);
            return @emptyList;
        }

        # <!ELEMENT element-name [definition] [ATT=attribute-list] [TAG=tag] [FLAG=flags] [OPEN]
        #   [DELETE] [EMPTY]>
        # <!ELEMENT>
        if ($tagMode ne 'defn') {

            $self->mxpDebug($origToken, 'Malformed element', 2201);

            return @emptyList;
        }

        # Process @argList
        @origList = @checkList = ('name', 'defn', 'att', 'tag', 'flag', 'open', 'delete', 'empty');
        # Hash of argument names which don't take a corresponding value
        %checkHash = (
            'open'      => undef,
            'delete'    => undef,
            'empty'     => undef,
        );
        # Default argument values
        %ivHash = (
            'name'      => undef,
            'defn'      => undef,
            'att'       => undef,
            'tag'       => undef,
            'flag'      => undef,
            'open'      => FALSE,
            'delete'    => FALSE,
            'empty'     => FALSE,
        );

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 2202);

                    return @emptyList;

                } elsif ($argName eq 'open' || $argName eq 'delete' || $argName eq 'empty') {

                    $ivHash{$argName} = TRUE;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check the validity of the compulsory 'element-name' argument
        if (! $ivHash{'name'}) {

            $self->mxpDebug($origToken, 'Malformed element', 2211);

            return @emptyList;
        }

        # <!ELEMENT> tags can't be used to redefine standard elements, e.g. this is not allowed:
        #       <!ELEMENT expire '<COLOR red><B>'>
        if ($axmud::CLIENT->ivExists('constMxpOfficialHash', uc($ivHash{'name'}))) {

            $self->mxpDebug($origToken, 'Can\'t redefine MXP standard element', 2212);

            return @emptyList;
        }

        # If the 'DELETE' argument was present, then the other arguments are irrelevant
        if ($ivHash{'delete'}) {

            if (! $self->ivExists('mxpElementHash', lc($ivHash{'name'}))) {

                $self->mxpDebug(
                    $origToken,
                    'Can\'t delete non-existent element \'' . $ivHash{'name'} . '\'',
                    2213,
                );

            } else {

                $self->ivDelete('mxpElementHash', lc($ivHash{'name'}));
            }

            return @emptyList;
        }

        # Otherwise, create a new element object
        $elementObj = Games::Axmud::Mxp::Element->new($self, lc($ivHash{'name'}));
        if (! $elementObj) {

            # Improper arguments
            $self->mxpDebug($origToken, 'Internal error while processing element', 2214);

            return @emptyList;
        }

        # Set the element object's remaining IVs
        if (defined $ivHash{'defn'}) {

            # (Sets both ->defnArg and ->defnList)
            if (! $elementObj->setDefn($self, $ivHash{'defn'})) {

                $self->mxpDebug($origToken, 'Invalid element definition', 2215);

                return @emptyList;
            }
        }

        if (defined $ivHash{'att'}) {

            if (! $elementObj->setAttList($self, $ivHash{'att'})) {

                $self->mxpDebug($origToken, 'Invalid attribute list argument', 2216);

                return @emptyList;
            }
        }

        $elementObj->ivPoke('tagArg', $ivHash{'tag'});
        $elementObj->ivPoke('flagArg', $ivHash{'flag'});
        $elementObj->ivPoke('openFlag', $ivHash{'open'});
        $elementObj->ivPoke('emptyFlag', $ivHash{'empty'});

        # No problems encountered, so we can store the element object, replacing any existing
        #   element object with the same name
        $self->ivAdd('mxpElementHash', lc($ivHash{'name'}), $elementObj);

        # There are no Axmud colour/style tags to return
        return @emptyList;
    }

    sub processMxpAttElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP attlist element: <!ATTLIST>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $eleName, $blank, $elementObj, $string,
            @emptyList, @newList,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpAttElement', @_);
        }

        # <!ATTLIST element-name attribute-list>
        # <!ATTLIST>
        if ($tagMode ne 'defn') {

            $self->mxpDebug($origToken, 'Malformed element', 2301);

            return @emptyList;
        }

        # Get 'element-name'. Axmud stores user-defined elements in lower-case, since they are
        #   case-insensitive (and stores keywords in upper case, to easily tell them apart)
        $eleName = lc(shift @argList);
        $blank = shift @argList;
        if (! $eleName || $blank) {

            $self->mxpDebug($origToken, 'Malformed element', 2302);

            return @emptyList;

        } elsif (! $self->ivExists('mxpElementHash', $eleName)) {

            $self->mxpDebug($origToken, 'Unrecognised custom element', 2303);

            return @emptyList;

        } else {

            $elementObj = $self->ivShow('mxpElementHash', $eleName);
        }

        # In '<!ATTLIST element-name attribute-list>', the MXP spec gives two forms for
        #   'attribute-list':
        #
        #       <!ATTLIST boldtext 'color=red background=white flags'>
        #       <!ATTLIST Sound FName V=100 L=1 P=50 T U>

        # In the first example, if @argList contains a single pair, in the form (argument, 'undef'),
        #   and if the argument is surrounded by quotes, expand it
        if (
            scalar @argList == 2
            && ! defined $argList[0]
            && $argList[0] =~ m/^[\'\"].*[\'\"]$/
        ) {
            # Expanded the quoted 'attribute-list'
            $string = $argList[0];
            do {

                my ($attName, $attValue);

                ($string, $attName, $attValue) = $self->extractMxpArgument($string);
                if (! defined $string) {

                    $self->mxpDebug($origToken, 'Invalid attribute list argument', 2304);

                    return @emptyList;

                } else {

                    # If the argument is not in the form 'argument_name=argument_value', then
                    #   $argValue is 'undef', and $argName contains the whole argument
                    push (@newList, $attName, $attValue);

                    # After removing the 'argument' or the 'argument_name=argument_value'
                    #   construction, if there's anything left in $token, it must start with a
                    #   whitespace character
                    # This ensures that there are whitespace character(s) between each argument,
                    #   and prevents constructions like: name='value'name='value'
                    if ($string && $string =~ m/^\S/) {

                        $self->mxpDebug($origToken, 'Malformed element', 2305);

                        return @emptyList;
                    }
                }

            } until (! $string);

            # Expansion complete
            @argList = @newList;
        }

        # @argList is now in the form (attrib_name, default_value, attrib_name, default_value...)
        #   where 'default_value' is 'undef', if the attribute has no default value
        if (@argList) {

            do {

                my ($attName, $attValue);

                $attName = shift @argList;
                $attValue = shift @argList;
                # The GA::Mxp::Element object uses empty strings for arguments with no default
                #   values
                if (! defined $attValue) {

                    $attValue = '';
                }

                # If the attribute already exists, replace its default value
                if ($elementObj->ivExists('attHash', $attName)) {

                    $elementObj->ivAdd('attHash', $attName, $attValue);

                # Otherwise, add a new attribute to the end of the ordered list of attributes
                } else {

                    $elementObj->ivAdd('attHash', $attName, $attValue);
                    $elementObj->ivPush('attList', $attName);
                }

            } until (! @argList);
        }

        # There are no Axmud colour/style tags to return
        return @emptyList;
    }

    sub processMxpEntElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP entity element: <!ENTITY>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $entityObj, $string, $newValue,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpEntElement', @_);
            return @emptyList;
        }

        # <!ENTITY Name Value [DESC=description] [PRIVATE] [PUBLISH] [DELETE] [ADD] [REMOVE]>
        # <!ENTITY>
        if ($tagMode ne 'defn') {

            $self->mxpDebug($origToken, 'Malformed element', 2401);

            return @emptyList;
        }

        # Process @argList
        @origList = @checkList = (
            'name', 'value', 'desc', 'private', 'publish', 'delete', 'add', 'remove',
        );

        # Hash of argument names which don't take a corresponding value
        %checkHash = (
            'private'   => undef,
            'publish'   => undef,
            'delete'    => undef,
            'add'       => undef,
            'remove'    => undef,
        );
        # Default argument values
        %ivHash = (
            'name'      => undef,
            'value'     => undef,
            'desc'      => undef,
            'private'   => FALSE,
            'publish'   => FALSE,
            'delete'    => FALSE,
            'add'       => FALSE,
            'remove'    => FALSE,
        );

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 2402);

                    return @emptyList;

                } elsif ($argName ne 'name' && $argName ne 'value' && $argName ne 'descrip') {

                    $ivHash{$argName} = TRUE;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check the validity of the compulsory 'Name' and 'Value' arguments
        if (! $ivHash{'name'} || ! defined $ivHash{'value'}) {

            $self->mxpDebug($origToken, 'Malformed element', 2411);

            return @emptyList;
        }

        # If the 'DELETE' argument was present, then the other arguments are irrelevant
        if ($ivHash{'DELETE'}) {

            # Delete this entity
            $self->ivDelete('mxpEntityHash', $ivHash{'name'});
            # Remove any MXP gauges that depend on this entity
            $self->removeMxpGauges($ivHash{'name'});

            # There are no Axmud colour/style tags to return
            return @emptyList;
        }

        # Get the existing entity object or, if there isn't one, create it
        $entityObj = $self->ivShow('mxpEntityHash', $ivHash{'name'});
        if (! $entityObj) {

            # Create a new entity object
            $entityObj = Games::Axmud::Mxp::Entity->new($self, $ivHash{'name'});
            if (! $entityObj) {

                # Improper arguments
                $self->mxpDebug($origToken, 'Internal error while processing element', 2412);

                return @emptyList;

            } else {

                # Store the entity object, replacing any existing entity object with the same name
                $self->ivAdd('mxpEntityHash', $ivHash{'name'}, $entityObj);
            }
        }

        # Store 'Value' in the appropriate place. If the 'ADD' and 'REMOVE' arguments are both
        #   present (for some unfathomable reason), then the 'Value' isn't stored anywhere
        if (! ($ivHash{'add'} && $ivHash{'remove'})) {

            # Store $'Value' in the appropriate place
            if ($ivHash{'add'}) {

                # Add the value to the entity's existing value list
                $entityObj->ivPoke('value', $entityObj->value . '|' . $ivHash{'value'});

            } elsif ($ivHash{'remove'}) {

                # Delete the value from the entity's existing value list
                $string = $entityObj->value;
                $newValue = $ivHash{'value'};
                if (! ($string =~ m/$newValue/)) {

                    $self->mxpDebug($origToken, 'Entity value not found', 2413);

                    return @emptyList;

                } else {

                    $string =~ s/$newValue//;
                    # Replace any || sequence, which contained the  with a single |
                    $string =~ s/\|\|/\|/;
                    # Remove any |s from the beginning/end of the string
                    $string =~ s/^\|//;
                    $string =~ s/\|$//;

                    $entityObj->ivPoke('value', $string);
                }

            } else {

                # Store 'Value' as it was received
                $entityObj->ivPoke('value', $ivHash{'value'});
            }
        }

        # Set the entity object's remaining IVs
        $entityObj->ivPoke('descArg', $ivHash{'descrip'});
        $entityObj->ivPoke('privateFlag', $ivHash{'private'});
        $entityObj->ivPoke('publishFlag', $ivHash{'publish'});

        # Mark any corresponding gauges to be updated
        $self->ivAdd('mxpGaugeUpdateHash', $ivHash{'name'}, undef);

        # There are no Axmud colour/style tags to return
        return @emptyList;
    }

    sub processMxpVarElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP variable element: <V>...</V>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $entityObj, $varObj, $value, $string,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpVarElement', @_);
            return @emptyList;
        }

        # <V Name [DESC=description] [PRIVATE] [PUBLISH] [DELETE] [ADD] [REMOVE]>Value</V>
        # <VAR Name [DESC=description] [PRIVATE] [PUBLISH] [DELETE] [ADD] [REMOVE]>Value</VAR>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
            || $tagMode eq 'defn'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 2501);

            return @emptyList;
        }

        # Process the <V> tag
        if ($tagMode eq 'open') {

            # Watch out for a <V> ... <V> ... </V> construction
            if ($self->mxpCurrentVar) {

                # This a second <V> tag after an earlier </V> tag. Remove the earlier variable
                #   object, and show an error (ignoring both tags)
                $self->ivUndef('mxpCurrentVar');

                $self->mxpDebug($origToken, '<VAR> tag after earlier unclosed <VAR> tag', 2502);

                return @emptyList;
            }

            # Process @argList
            @origList = @checkList = (
                'name', 'desc', 'private', 'publish', 'delete', 'add', 'remove',
            );

            # Hash of argument names which don't take a corresponding value
            %checkHash = (
                'private'   => undef,
                'publish'   => undef,
                'delete'    => undef,
                'add'       => undef,
                'remove'    => undef,
            );
            # Default argument values
            %ivHash = (
                'name'      => undef,
                'desc'      => undef,
                'private'   => FALSE,
                'publish'   => FALSE,
                'delete'    => FALSE,
                'add'       => FALSE,
                'remove'    => FALSE,
            );

            if (@argList) {

                do {

                    my ($argName, $argValue) = $self->findMxpArgsByPosn(
                        \@origList,
                        \@checkList,
                        \%ivHash,
                        \%checkHash,
                        shift @argList,     # not 'undef'
                        shift @argList,     # might be 'undef'
                    );

                    if (! defined $argName) {

                        # Unrecognised argument name, or repeating argument name
                        $self->mxpDebug($origToken, 'Malformed element', 2503);

                        return @emptyList;

                    } elsif ($argName ne 'name' && $argName ne 'descrip') {

                        $ivHash{$argName} = TRUE;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check the validity of the compulsory 'Name' argument
            if (! $ivHash{'name'}) {

                $self->mxpDebug($origToken, 'Malformed element', 2511);

                return @emptyList;
            }

            # Find the corresponding entity object or, if there isn't one, create it (entity names
            #   are case-sensitive)
            $entityObj = $self->ivShow('mxpEntityHash', $ivHash{'name'});
            if (! $entityObj) {

                # Create a new entity object, whose (emergency default) value is an empty string
                $entityObj = Games::Axmud::Mxp::Entity->new($self, $ivHash{'name'}, '');
                if (! $entityObj) {

                    # Improper arguments
                    $self->mxpDebug($origToken, 'Internal error while processing variable', 2512);

                    return @emptyList;

                } else {

                    # Store the entity object, replacing any existing entity object with the same
                    #   name
                    $self->ivAdd('mxpEntityHash', $ivHash{'name'}, $entityObj);
                }
            }

            # We can't modify the entity's IVs until the closing </V> tag is found, so store
            #   arguments temporarily in a GA::Mxp::Var object
            $varObj = Games::Axmud::Mxp::Var->new($self, $ivHash{'name'});
            if (! $varObj) {

                $self->mxpDebug($origToken, 'Internal error while processing variable', 2513);

                return @emptyList;
            }

            # Set the variable object's IVs
            $varObj->ivPoke('descArg', $ivHash{'descrip'});
            $varObj->ivPoke('privateFlag', $ivHash{'private'});
            $varObj->ivPoke('publishFlag', $ivHash{'publish'});
            $varObj->ivPoke('deleteFlag', $ivHash{'delete'});
            $varObj->ivPoke('addFlag', $ivHash{'add'});
            $varObj->ivPoke('removeFlag', $ivHash{'remove'});

            # Store the current variable object
            $self->ivPoke('mxpCurrentVar', $varObj);

            # There are no Axmud colour/style tags to return
            return @emptyList;

        # Process the <\V> tag
        } else {

            # Watch out for a <V> ... </V> ... </V> construction
            if (! $self->mxpCurrentVar) {

                # This a second </V> tag after an earlier </V> tag
                $self->mxpDebug($origToken, '</VAR> tag does not match earlier <VAR> tag', 2521);

                return @emptyList;

            } else {

                $varObj = $self->mxpCurrentVar;
                $self->ivUndef('mxpCurrentVar');
            }

            if (! $varObj->value) {

                # There were no valid text tokens between the <V>...</V> tags
                $self->mxpDebug($origToken, 'Invalid <VAR>...</VAR> construction', 2522);

                return @emptyList;

            } else {

                $value = $varObj->value;
            }

            # Find the equivalent GA::Mxp::Entity object
            $entityObj = $self->ivShow('mxpEntityHash', $varObj->name);
            if (! $entityObj) {

                $self->mxpDebug(
                    $origToken,
                    'Can\'t find entity matching variable \'' . $varObj->name . '\'',
                    2523,
                );

                return @emptyList;
            }

            # If the DELETE argument was used, the entity gets removed altogether (and we don't need
            #   to know what was between the <V>...</V> tabs)
            if ($varObj->deleteFlag) {

                # Delete this entity
                $self->ivDelete('mxpEntityHash', $entityObj->name);
                # Remove any MXP gauges that depend on this entity
                $self->removeMxpGauges($ivHash{'name'});

                # (There are no Axmud colour/style tags to return)
                return @emptyList;
            }

            # Update the entity's value. If the 'ADD' and 'REMOVE' arguments are both present (for
            #   some unfathomable reason), then the variable object's ->value isn't stored anywhere
            if (! ($varObj->addFlag && $varObj->deleteFlag)) {

                if ($varObj->addFlag) {

                    $entityObj->ivPoke('value', $entityObj->value . '|' . $value);

                } elsif ($varObj->removeFlag) {

                    $string = $entityObj->value;
                    if (! ($string =~ m/$value/)) {

                        $self->mxpDebug(
                            $origToken,
                            'Entity \'' . $entityObj->name . '\' value \'' . $value
                            . '\' not found',
                            2524,
                        );

                        return @emptyList;

                    } else {

                        $string =~ s/$value//;
                        # Replace any || sequence, which contained the  with a single |
                        $string =~ s/\|\|/\|/;
                        # Remove any |s from the beginning/end of the string
                        $string =~ s/^\|//;
                        $string =~ s/\|$//;

                        $entityObj->ivPoke('value', $string);
                    }

                } else {

                    $entityObj->ivPoke('value', $value);
                }
            }

            # Process the other arguments
            $entityObj->ivPoke('descArg', $varObj->descArg);
            $entityObj->ivPoke('privateFlag', $varObj->privateFlag);
            $entityObj->ivPoke('publishFlag', $varObj->publishFlag);

            # Mark any corresponding gauges to be updated
            $self->ivAdd('mxpGaugeUpdateHash', $ivHash{'name'}, undef);

            # There are no Axmud colour/style tags to return
            return @emptyList;
        }
    }

    sub processMxpSupportElement {

        # Called by $self->processSupportElement (and also by GA::Client->set_allowMxpFlag and
        #   ->set_allowSoundFlag)
        #
        # Process an MXP support element: <SUPPORT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (will always
        #       be an empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $msg,
            @emptyList,
            %tagHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpSupportElement', @_);
            return @emptyList;
        }

        # <SUPPORT>
        # <SUPPORT [arg] [arg] [arg]...>
        if ($tagMode ne 'open') {

            $self->mxpDebug($origToken, 'Malformed element', 2601);

            return @emptyList;
        }

        # Axmud offers support for all MXP tags, but not all of them may be available at the moment
        #   (e.g. if sound is turned off, the world shouldn't bother sending <MUSIC> tags
        # Import the constant hash of MXP tags and their attributes, and remove both tags and
        #   attributes which aren't available at the moment
        %tagHash = $axmud::CLIENT->constMxpAttribHash;

        if (! $axmud::CLIENT->allowMxpFontFlag) {

            delete $tagHash{'h1'};
            delete $tagHash{'h2'};
            delete $tagHash{'h3'};
            delete $tagHash{'h4'};
            delete $tagHash{'h5'};
            delete $tagHash{'h6'};
            delete $tagHash{'small'};
            delete $tagHash{'tt'};

            # If FALSE, <FONT> tags can still change the text colour
            %tagHash = $self->deleteMxpAttrib('font', 'face', %tagHash);
            %tagHash = $self->deleteMxpAttrib('font', 'size', %tagHash);
        }

        if (! $axmud::CLIENT->allowMxpImageFlag) {

            delete $tagHash{'image'};
        }

        if (! $axmud::CLIENT->allowMxpLoadImageFlag) {

            %tagHash = $self->deleteMxpAttrib('image', 'url', %tagHash);
        }

        if (! $axmud::CLIENT->allowMxpFilterImageFlag) {

            delete $tagHash{'filter'};
        }

        if (! $axmud::CLIENT->allowSoundFlag || ! $axmud::CLIENT->allowMxpSoundFlag) {

            delete $tagHash{'sound'};
            delete $tagHash{'music'};
        }

        # (No need to use GA::Client->allowMxpLoadSoundFlag here)

        if (
            ! $axmud::CLIENT->allowMxpGaugeFlag
            || ! $self->mainWin->ivShow('firstStripHash', 'Games::Axmud::Strip::GaugeBox')
        ) {
            delete $tagHash{'gauge'};
            delete $tagHash{'stat'};
        }

        if (
            $axmud::BLIND_MODE_FLAG
            || ! $axmud::CLIENT->allowMxpFrameFlag
            || $self->mxpDisableFrameFlag
        ) {
            delete $tagHash{'frame'};
            # (Axmud chooses to ignore <DEST> tags if frames have been disabled generally; even
            #   though some world might want cursor control in the main MUD window, that's not very
            #   practical if it's scrolling)
            delete $tagHash{'dest'};
        }

        if ($axmud::CLIENT->shareMainWinFlag || ! $axmud::CLIENT->allowMxpInteriorFlag) {

            %tagHash = $self->deleteMxpAttrib('frame', 'internal', %tagHash);
        }

        if (! $axmud::CLIENT->allowMxpCrosslinkFlag) {

            delete $tagHash{'relocate'};
        }

        # Process the <SUPPORT> tag

        if (! @argList) {

            # Return a list of all supported tags
            $msg = '<SUPPORTS';

            foreach my $key (sort {$a cmp $b} (keys %tagHash)) {

                $msg .= ' +' . uc($key);
            }

            $msg .= '>';

            # The response must be sent securely
            $self->optSendMxpSecure($msg);

        } else {

            # Respond to every item in @argList
            $msg = '<SUPPORTS';

            do {

                my ($argName, $argValue, $tag, $attrib, $listRef);

                $argName = shift @argList;
                $argValue = shift @argList;     # Should be 'undef'; ignored, in any case

                # Remove the initial/final quotation marks, if present
                $argName =~ s/^\"//;
                $argName =~ s/\"$//;

                # Split the item into its component parts
                #   e.g. <SUPPORT color.*>          > ('color', '*')
                #   e.g. <SUPPORT send.expire>      > ('send', 'expire')
                if ($argName =~ m/(\w+)\.(.*)/) {

                    $tag = lc($1);
                    $attrib = lc($2);

                #   e.g. <SUPPORT image>            > ('image')
                } else {

                    $tag = lc($argName);
                }

                # Convert a long tag to its abbreviation (e.g. <DESTINATION> to <DEST>)
                if ($axmud::CLIENT->ivExists('constMxpConvertHash', uc($tag))) {

                    $tag = lc($axmud::CLIENT->ivShow('constMxpConvertHash', uc($tag)));
                }

                # Check the MXP is both recognised and currently supported
                if (! exists $tagHash{$tag}) {

                    $msg .= ' -' . $tag;

                } elsif (! defined $attrib) {

                    # e.g. <SUPPORT image>
                    $msg .= ' +' . $tag;

                } elsif ($attrib eq '*') {

                    # e.g. <SUPPORT color.*>
                    $listRef = $tagHash{$tag};
                    # MXP tags without attributes, such as <B>, appear as keys in %tagHash, but
                    #   their corresponding value is 'undef' rather than a list reference
                    if (defined $listRef) {

                        foreach my $thisAttrib (@$listRef) {

                            $msg .= ' +' . $tag . '.' . $thisAttrib;
                        }
                    }

                } else {

                    # e.g. <SUPPORT send.expire>
                    $listRef = $tagHash{$tag};
                    if (defined $listRef) {

                        INNER: foreach my $thisAttrib (@$listRef) {

                            if ($thisAttrib eq lc($attrib)) {

                                $msg .= ' +' . $tag . '.' . $thisAttrib;
                                last INNER;
                            }
                        }
                    }
                }

            } until (! @argList);

            $msg .= '>';

            # The response must be sent securely
            $self->optSendMxpSecure($msg);
        }

        # There are no Axmud colour/style tags to return
        return @emptyList;
    }

    sub processMxpFrameElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP frame element: <FRAME>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (will always
        #       be an empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $origFrameObj, $frameObj, $left, $right, $top, $bottom, $newFlag, $gridObj, $zonemapObj,
            $width, $height, $zoneModelObj, $taskObj, $currentLeft, $currentRight, $currentTop,
            $currentBottom, $currentWidth, $currentHeight, $reduceWidth, $reduceHeight, $resizeLeft,
            $resizeRight, $resizeTop, $resizeBottom, $newLeft, $newRight, $newTop, $newBottom,
            $newPaneObj, $tabObj, $paneObj, $spinFlag,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash, %reservedHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpFrameElement', @_);
            return @emptyList;
        }

        print "16459 $origToken\n";

        # <FRAME NAME=name [ACTION=action] [TITLE=title] [PARENT=parent] [INTERNAL] [ALIGN=align]
        #   [LEFT=left] [TOP=top] [WIDTH=width] [HEIGHT=height] [SCROLLING=scrolling] [FLOATING]>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 2702);

            return @emptyList;
        }

        # Ignore this tag if the client/session flags are set
        if (
            $axmud::BLIND_MODE_FLAG
            || ! $axmud::CLIENT->allowMxpFrameFlag
            || $self->mxpDisableFrameFlag
        ) {
            return @emptyList;
        }

        # Process the <FRAME> tag

        # Process @argList
        @origList = @checkList = (
            'name', 'action', 'title', 'internal', 'align', 'left', 'top', 'width',
            'height', 'scrolling', 'floating',
        );
        # Hash of argument names which don't take a corresponding value
        %checkHash = (
            'internal'  => undef,
            'floating'  => undef,
        );
        # Default argument values
        %ivHash = (
            'name'      => undef,
            'action'    => 'open',
            'title'     => undef,
            'parent'    => undef,       # Removed in MXP v1.0; ignored if specified
            'internal'  => FALSE,
            'align'     => 'top',
            'left'      => 0,
            'top'       => 0,
            'width'     => '50%',
            'height'    => '50%',
            'scrolling' => FALSE,
            'floating'  => FALSE,
        );

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 2702);

                    return @emptyList;

                } elsif ($argName eq 'action') {

                    $ivHash{$argName} = lc($argValue);

                } elsif (
                    $argName eq 'internal'
                    && ! $axmud::CLIENT->shareMainWinFlag
                    && $axmud::CLIENT->allowMxpInteriorFlag
                ) {
                    # (If internal frames can't be created, created external frames instead)
                    $ivHash{$argName} = TRUE;

                } elsif ($argName eq 'scrolling') {

                    # Default is 'no'
                    if ($argValue && $argValue eq 'yes') {

                        $ivHash{$argName} = TRUE;
                    }

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check the validity of the compulsory 'Name' argument
        if (! $ivHash{'name'}) {

            $self->mxpDebug($origToken, 'Malformed element', 2711);

            return @emptyList;
        }

        # Check the validity of optional arguments
        if (
            $ivHash{'action'} ne 'open'
            && $ivHash{'action'} ne 'close'
            && $ivHash{'action'} ne 'redirect'
        ) {
            # Invalid value
            $self->mxpDebug($origToken, 'Invalid frame action \'' . $ivHash{'action'} . '\'', 2712);

            return @emptyList;
        }

        if (
            $ivHash{'align'} ne 'left'
            && $ivHash{'align'} ne 'right'
            && $ivHash{'align'} ne 'bottom'
            && $ivHash{'align'} ne 'top'
        ) {
            # Invalid value
            $self->mxpDebug($origToken, 'Invalid frame align \'' . $ivHash{'align'} . '\'', 2713);

            return @emptyList;
        }

        foreach my $att ('left', 'top', 'width', 'height') {

            my $value = $ivHash{$att};

            if (! ($value =~ m/^\-?\d+[\%c]?/)) {

                $self->mxpDebug(
                    $origToken,
                    'Invalid frame ' . $att . ' coordinate \'' . $value . '\'',
                    2714,
                );

                return @emptyList;
            }
        }

        # Any <FRAME> tag abnormally terminates a <DEST>...</DEST> construction
        if ($self->mxpCurrentDest) {

            # Process a fake </DEST> tag to reset IVs correctly
            $self->processMxpDestElement('</DEST>', 'close', 'DEST');

            # Display error, if allowed
            $self->mxpDebug($origToken, '<FRAME> tag within <DEST> tags are not acceptable', 2715);

            return @emptyList;
        }

        # The <FRAME> and <DEST> tags can change $self->currentTabObj, corresponding to the frame in
        #   which text received from the world is displayed. The calling function may have processed
        #   some text tokens which haven't been displayed yet; if so, display them now so that
        #   they're displayed in the right frame
        if ($self->recvLineText) {

            $self->processIncompleteLine($self->mxpOrigText);
            $self->ivUndef('mxpOrigText');
        }

        # Apply any links created by MXP <A> and <SEND> tags to the current textview (if the current
        #   textview was changed during the call to this function, any links for other textviews
        #   have already been applied)
        foreach my $linkObj ($self->mxpTempLinkList) {

            $self->currentTabObj->textViewObj->add_incompleteLink($linkObj);
        }

        $self->ivEmpty('mxpTempLinkList');

        # Find the specified frame object
        $frameObj = $self->getMxpFrame($ivHash{'name'});

        # If there are no frame objects at all, before creating a new frame, we need to create a
        #   frame object for this session's default tab, and assign it to the default tab's pane
        #   object
        # (If the world actually specified a frame named '_top', then create a frame object for that
        #   frame only)
        if (
            ! $self->mxpFrameHash
            && ($ivHash{'action'} eq 'open' || $ivHash{'action'} eq 'redirect')
        ) {
            $origFrameObj = Games::Axmud::Mxp::Frame->new($self, '_top');
            if (! $origFrameObj) {

                # Improper arguments
                $self->mxpDebug($origToken, 'Internal error while processing frame', 2716);

                return @emptyList;

            } else {

                # Store the frame object and assign it to the default tab's pane object
                $self->ivAdd('mxpFrameHash', $origFrameObj->name, $origFrameObj);

                $origFrameObj->ivPoke('tabObj', $self->defaultTabObj);
                $origFrameObj->ivPoke('paneObj', $self->defaultTabObj->paneObj);
                $origFrameObj->ivPoke('textViewObj', $self->defaultTabObj->textViewObj);

                # Set other session IVs
                $self->ivPoke('mxpCurrentFrame', '_top');
                $self->ivPoke('mxpPrevFrame', '_top');

                ($left, $right, $top, $bottom) = $self->defaultTabObj->paneObj->stripObj->getPosn(
                    $self->defaultTabObj->paneObj,
                );

                # (The size of all additional frames is a factor of the size of the default frame,
                #   when the first additional frame is added)
                $self->ivPoke('mxpFrameWidth', $right - $left + 1);
                $self->ivPoke('mxpFrameHeight', $bottom - $top + 1);

                if ($ivHash{'name'} eq '_top') {

                    $frameObj = $origFrameObj;
                }
            }
        }

        # If the specified frame doesn't exist, create it (unless we created one just above)
        if (! $frameObj) {

            if ($ivHash{'action'} eq 'close') {

                $self->mxpDebug(
                    $origToken,
                    'Unrecognised frame name \'' . $ivHash{'name'} . '\'',
                    2717,
                );

                return @emptyList;

            } else {

                # Create a new frame object for the specified frame
                $frameObj = Games::Axmud::Mxp::Frame->new($self, $ivHash{'name'});
                if (! $frameObj) {

                    # Improper arguments
                    $self->mxpDebug($origToken, 'Internal error while processing frame', 2718);

                    return @emptyList;

                } else {

                    # Store the frame object, replacing any existing frame object with the same name
                    $self->ivAdd('mxpFrameHash', $frameObj->name, $frameObj);
                    $newFlag = TRUE;
                }
            }

        } elsif ($ivHash{'action'} eq 'open') {

            $self->mxpDebug(
                $origToken,
                'Can\'t open frame \'' . $ivHash{'name'} . '\', it already exists',
                2719,
            );

            return @emptyList;
        }

        # Set the frame object's IVs
        $frameObj->ivPoke('align', $ivHash{'align'});
        $frameObj->ivPoke('left', $ivHash{'left'});
        $frameObj->ivPoke('top', $ivHash{'top'});
        $frameObj->ivPoke('width', $ivHash{'width'});
        $frameObj->ivPoke('height', $ivHash{'height'});
        $frameObj->ivPoke('title', $ivHash{'title'});
        $frameObj->ivPoke('internalFlag', $ivHash{'internal'});
        $frameObj->ivPoke('scrollingFlag', $ivHash{'scrolling'});
        $frameObj->ivPoke('floatingFlag', $ivHash{'floating'});

        # If a new frame was specified, create it
        if ($newFlag) {

            # Implement an external frame as a new Frame task, placed on a temporary zonemap (if
            #   workspace grids are in use)
            if (! $frameObj->internalFlag) {

                # If workspace grids are available and the workspace containing the session's 'main'
                #   window isn't using a temporary zonemap, then create a temporary zonemap
                $gridObj = $self->mainWin->workspaceObj->findWorkspaceGrid($self);
                if ($gridObj && $gridObj->zonemap) {

                    $zonemapObj = $axmud::CLIENT->ivShow('zonemapHash', $gridObj->zonemap);
                    if ($zonemapObj && ! $zonemapObj->tempFlag) {

                        $zonemapObj = $axmud::CLIENT->createTempZonemap($self);
                        if ($zonemapObj) {

                            $gridObj->applyZonemap($zonemapObj);
                        }
                    }
                }

                if ($frameObj->name ne '_top') {

                    # Create the new Frame task
                    $taskObj = Games::Axmud::Task::Frame->new($self, 'current');
                    if (! $taskObj) {

                        $self->mxpDebug($origToken, 'Internal error while processing link', 2720);

                        return @emptyList;
                    }

                    # If the world has specified a size and position for this window, work out the
                    #   equivalent size and position in pixels on the same workspace that the
                    #   session's 'main' window is using, and store this size/position in the Frame
                    #   task so that the task window can be opened using them
                    $taskObj->set_winPosn($self->convertMxpWinSize($frameObj));

                    # Update IVs. The frame object's ->tabObj, ->paneObj and ->textViewObj IVs are
                    #   set when the task window is opened
                    $frameObj->ivPoke('taskObj', $taskObj);
                    $taskObj->set_frameObj($frameObj);

                    # Give the task window a chance to open immediately by spinning the task loop
                    #   (in case the next token processed by $self->processIncomingData wants to
                    #   display text in the task window)
                    # Normally the task loop won't spin if the incoming data loop (which was
                    #   ultimately responsible for calling this function) is spinning; so we have
                    #   to cheat, and temporarily reset a flag
                    if ($self->childLoopSpinFlag) {

                        $spinFlag = TRUE;
                        $self->ivPoke('childLoopSpinFlag', FALSE);
                    }

                    $self->spinTaskLoop();

                    if ($spinFlag) {

                        $self->ivPoke('childLoopSpinFlag', TRUE);
                    }
                }

            # Implement an internal frame as a new pane object in the session's 'main' window
            } else {

                # Reduce the size of the default tab's pane object (the original frame) to make
                #   way for a new frame
                # On the assumption that the world won't create endles frames with the same
                #   alignment, reduce the original frame's current size by a factor of it original
                #   size (e.g. original width = 60, current width = 40, new width = 20)
                # If this would make the original frame's width/height too small (e.g. original
                #   width = 60, current width = 20, new width = 0), halve the size instead
                #   (e.g. new width = 10)
                $origFrameObj = $self->ivShow('mxpFrameHash', '_top');
                ($currentLeft, $currentRight, $currentTop, $currentBottom)
                    = $self->defaultTabObj->paneObj->stripObj->getPosn(
                        $self->defaultTabObj->paneObj,
                    );

                # Original frame's current size
                $currentWidth = $currentRight - $currentLeft + 1;
                $currentHeight = $currentBottom - $currentTop + 1;
                # Original frame should be reduced by this much
                $reduceWidth = int($self->mxpFrameWidth * $self->mxpFrameXFactor);
                $reduceHeight = int($self->mxpFrameHeight * $self->mxpFrameYFactor);

                if ($currentWidth <= $reduceWidth) {

                    $reduceWidth = int($currentWidth / 2);
                }

                if ($currentHeight <= $reduceHeight) {

                    $reduceHeight = int($currentHeight / 2);
                }

                # Reduce the original frame on one axis only
                $newLeft = $resizeLeft = $currentLeft;
                $newRight = $resizeRight = $currentRight;
                $newTop = $resizeTop = $currentTop;
                $newBottom = $resizeBottom = $currentBottom;
                if ($frameObj->align eq 'left') {

                    # (The size/position of the new frame we're about to create)
                    $newLeft = $resizeLeft;
                    $newRight = $resizeLeft + $reduceWidth - 1;

                    # (The size/position of the original frame)
                    $resizeLeft += $reduceWidth;

                } elsif ($frameObj->align eq 'right') {

                    $resizeRight -= $reduceWidth;

                    $newLeft = $resizeRight + 1;
                    $newRight = $newLeft + $reduceWidth - 1;

                } elsif ($frameObj->align eq 'top') {

                    $newTop = $resizeTop;
                    $newBottom = $resizeTop + $reduceHeight - 1;

                    $resizeTop += $reduceHeight;

                } elsif ($frameObj->align eq 'bottom') {

                    $resizeBottom -= $reduceHeight;

                    $newTop = $resizeBottom + 1;
                    $newBottom = $newTop + $reduceHeight - 1;
                }

                $self->defaultTabObj->paneObj->stripObj->resizeTableObj(
                    $self->defaultTabObj->paneObj,
                    $resizeLeft,
                    $resizeRight,
                    $resizeTop,
                    $resizeBottom,
                );

                # Add a new pane object in the newly-available space
                $newPaneObj = $self->defaultTabObj->paneObj->stripObj->addTableObj(
                    'Games::Axmud::Table::Pane',
                    $newLeft,
                    $newRight,
                    $newTop,
                    $newBottom,
                    'mxp_frame_' . $frameObj->name,
                    # Configuration hash
                    'frame_title'       => $ivHash{'name'},
                    'no_label_flag'     => TRUE,
                );

                if (! $newPaneObj) {

                    $self->mxpDebug(
                        $origToken,
                        'Internal error creating frame \'' . $ivHash{'name'} . '\'',
                        2721,
                    );

                    return @emptyList;
                }

                # Add a tab
                $tabObj = $newPaneObj->addSimpleTab($self);
                if (! $tabObj) {

                    $self->mxpDebug(
                        $origToken,
                        'Internal error creating frame \'' . $ivHash{'name'} . '\'',
                        2722,
                    );

                    return @emptyList;
                }

                # This call makes the original frame's textview scroll to the bottom, as it's
                #   supposed to
                $axmud::CLIENT->desktopObj->updateWidgets(
                    $self->_objClass . '->processMxpFrameElement',
                );

                # Update IVs
                $frameObj->ivPoke('tabObj', $tabObj);
                $frameObj->ivPoke('paneObj', $tabObj->paneObj);
                $frameObj->ivPoke('textViewObj', $tabObj->textViewObj);
            }

        # Close an existing frame, if specified
        } elsif ($ivHash{'action'} eq 'close') {

            # Do not close the default tab's pane object, even if the world wants to
            # (The MXP spec doesn't specify what to do, but Axmud will not allow it)
            if ($ivHash{'name'} eq '_top') {

                $self->mxpDebug(
                    $origToken,
                    'Cannot close MXP frame corresponding to the default tab',
                    2723,
                );

                return @emptyList;
            }

            # Close the frame
            if ($frameObj->internalFlag) {

                # Remove an internal frame
                $self->defaultTabObj->paneObj->stripObj->removeTableObj(
                    $self->defaultTabObj->paneObj,
                );

            } else {

                # Halt the Frame task
                $frameObj->taskObj->set_shutdownFlag(TRUE);
            }

            # Update IVs
            $self->ivDelete('mxpFrameHash', $frameObj->name);
            if ($self->mxpCurrentFrame eq $frameObj->name) {

                # If the current frame is deleted, resume using the original frame
                # (The MXP spec doesn't specify what to do, so Axmud will do this)
                $self->ivPoke('mxpCurrentFrame', '_top');
                $self->ivPoke('currentTabObj', $self->defaultTabObj);
            }

            if ($self->mxpPrevFrame eq $frameObj->name) {

                # Same applies to the previous frame
                $self->ivPoke('mxpPrevFrame', '_top');
            }

        # Redirect text received from the world to the frame
        } elsif ($ivHash{'action'} eq 'redirect') {

            $self->ivPoke('mxpPrevFrame', $self->mxpCurrentFrame);
            $self->ivPoke('mxpCurrentFrame', $frameObj->name);
            $self->ivPoke('currentTabObj', $frameObj->tabObj);
        }

        # There are no Axmud colour/style tags to return
        return @emptyList;
    }

    sub processMxpDestElement {

        # Called by $self->processMxpElement (and by $self->processMxpFrameElement to process a
        #   fake </DEST> tag)
        #
        # Process an MXP destination element: <DEST>...</DEST>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $frameObj, $destObj, $textViewObj, $mark,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpDestElement', @_);
            return @emptyList;
        }

        # If the user manually closed an exterior frame (implemented as a Frame task window), ignore
        #   <DEST> tags
        if ($self->mxpDisableFrameFlag) {

            return @emptyList;
        }

        # <DEST Frame_name [X=int] [Y=int] [EOL] [EOF]>text</DEST>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
            || $tagMode eq 'defn'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 2801);

            return @emptyList;
        }

        # The <FRAME> and <DEST> tags can change $self->currentTabObj, corresponding to the frame in
        #   which text received from the world is displayed. The calling function may have processed
        #   some text tokens which haven't been displayed yet; if so, display them now so that
        #   they're displayed in the right frame
        if ($self->recvLineText) {

            $self->processIncompleteLine($self->mxpOrigText);
            $self->ivUndef('mxpOrigText');
        }

        # Apply any links created by MXP <A> and <SEND> tags to the current textview (if the current
        #   textview was changed during the call to this function, any links for other textviews
        #   have already been applied)
        foreach my $linkObj ($self->mxpTempLinkList) {

            $self->currentTabObj->textViewObj->add_incompleteLink($linkObj);
        }

        $self->ivEmpty('mxpTempLinkList');

        # Process the <DEST> tag
        if ($tagMode eq 'open') {

            # Watch out for a <DEST> ... <DEST> ... </DEST> construction
            if ($self->mxpCurrentDest) {

                # This a second <DEST> tag after an earlier </DEST> tag. Remove the earlier
                #   destination object, and show an error (ignoring both tags)
                $self->ivUndef('mxpCurrentDest');

                $self->mxpDebug($origToken, '<DEST> tag after earlier unclosed <DEST> tag', 2802);

                return @emptyList;
            }

            # Process @argList
            @origList = @checkList = ('name', 'x', 'y', 'eol', 'eof');
            # Hash of argument names which don't take a corresponding value
            %checkHash = (
                'eol'       => undef,
                'eof'       => undef,
            );
            # Default argument values
            %ivHash = (
                'name'      => undef,
                'x'         => undef,
                'y'         => undef,
                'eol'       => FALSE,
                'eof'       => FALSE,
            );

            if (@argList) {

                do {

                    my ($argName, $argValue) = $self->findMxpArgsByPosn(
                        \@origList,
                        \@checkList,
                        \%ivHash,
                        \%checkHash,
                        shift @argList,     # not 'undef'
                        shift @argList,     # might be 'undef'
                    );

                    if (! defined $argName) {

                        # Unrecognised argument name, or repeating argument name
                        $self->mxpDebug($origToken, 'Malformed element', 2803);

                        return @emptyList;

                    } elsif ($argName eq 'eol' || $argName eq 'eof') {

                        $ivHash{$argName} = TRUE;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check the validity of the compulsory 'Name' argument
            if (! $ivHash{'name'}) {

                # MXP spec says to use the main MUD window in this situation
                $ivHash{'name'} = '_top';
            }

            # Check the validity of the x/y coordinates, if specified
            if (
                (defined $ivHash{'x'} && ! $axmud::CLIENT->intCheck($ivHash{'x'}, 0))
                || (defined $ivHash{'y'} && ! $axmud::CLIENT->intCheck($ivHash{'y'}, 0))
            ) {
                $self->mxpDebug($origToken, 'Malformed element', 2812);

                return @emptyList;
            }

            # Find the specified frame object. '_top' and '_previous' are recognised, but this
            #   function won't create a new frame as $self->processMxpFrameElement would
            $frameObj = $self->getMxpFrame($ivHash{'name'});

            if (! $frameObj) {

                $self->mxpDebug($origToken, 'Unrecognised frame name in <DEST> tag', 2813);

                return @emptyList;
            }

            # In case an external frame is attached to a Frame task whose window hasn't opened yet,
            #   display the text in the session's default tab instead
            if (! $frameObj->tabObj) {

                $frameObj = $self->ivShow('mxpFrameHash', '_top');
            }

            # Store the arguments temporarily in a GA::Mxp::Dest object until the closing </DEST>
            #   tag is found
            $destObj = Games::Axmud::Mxp::Dest->new($self, $ivHash{'name'});
            if (! $destObj) {

                $self->mxpDebug(
                    $origToken,
                    'Internal error while processing destination tag',
                    2814,
                );

                return @emptyList;
            }

            # Set the destination object's IVs
            $destObj->ivPoke('xPos', $ivHash{'x'});
            $destObj->ivPoke('yPos', $ivHash{'y'});
            $destObj->ivPoke('eolFlag', $ivHash{'eol'});
            $destObj->ivPoke('eofFlag', $ivHash{'eof'});

            # Store the current destination object
            $self->ivPoke('mxpCurrentDest', $destObj);

            # The MXP spec suggests that text inside a <DEST>...</DEST> construction shouldn't be
            #   displayed until the final </DEST> tag is received, but makes no mention of what
            #   should be done with any tags inside the construction
            # The spec doesn't even specify what should be done with a newline character (for
            #   example, should it reset ->mxpLineMode, or not?)
            # Because of this, and because of they way Axmud handles incoming text token by token,
            #   Axmud handles a <DEST> tag as a special kind kind of <FRAME name REDIRECT> tag. Text
            #   is redirected into the specified frame until the </DEST> tag is processed, at which
            #   time text is redirected into whichever frame was used before the <DEST> tag
            # (NB In actual fact, if there are no newline characters inside the <DEST>...</DEST>
            #   construction, any text tokens inside won't be displayed until just before the
            #   </DEST> tag is processed; it's only if the construction contains newline tokens that
            #   the tokens might not be displayed all at the same time)
            # To keep things simple, a <FRAME> tag abnormally terminates the <DEST>...</DEST>
            #   construction

            # Store the frame being used at the moment, so it can be restored when the </DEST> tag
            #   is processed
            $destObj->ivPoke('mxpPrevFrame', $self->mxpPrevFrame);
            $destObj->ivPoke('mxpCurrentFrame', $self->mxpCurrentFrame);
            # Redirect text to the new frame
            $self->ivPoke('mxpPrevFrame', $self->mxpCurrentFrame);
            $self->ivPoke('mxpCurrentFrame', $frameObj->name);
            $self->ivPoke('currentTabObj', $frameObj->tabObj);

            # Put the textview(s) into overwrite mode (we do this here, rather than when the frame
            #   is created, so that if the world uses <DEST> tags in the session's current tab,
            #   overwrite mode is applied to that tab, too)
            $frameObj->tabObj->textViewObj->set_overwrite(TRUE);
            # If cursor control was specified, apply it now
            if (defined $destObj->xPos || defined $destObj->yPos) {

                $frameObj->tabObj->textViewObj->setInsertPosn($destObj->yPos, $destObj->xPos);
            }

            # There are no Axmud colour/style tags to return
            return @emptyList;

        # Process the <\DEST> tag
        } else {

            # Watch out for a <DEST> ... </DEST> ... </DEST> construction
            if (! $self->mxpCurrentDest) {

                # This a second </DEST> tag after an earlier </DEST> tag
                $self->mxpDebug(
                    $origToken,
                    '</DEST> tag does not match earlier <DEST> tag',
                    2815,
                );

                return @emptyList;

            } else {

                $destObj = $self->mxpCurrentDest;
                $self->ivUndef('mxpCurrentDest');
            }

            # Restore the insert position in the current tab's textview object to that which was
            #   in use before the opening <DEST> tag
            if (defined $destObj->xPos || defined $destObj->yPos) {

                $textViewObj = $self->currentTabObj->textViewObj;
                $mark = $textViewObj->insertMark;
                $textViewObj->resetInsertPosn();

                # Empty the buffer at the point immediately after the last part of the text from
                #   inside the <DEST>...</DEST> construction was displayed
                # (If both EOF and EOL were specified, just apply EOF)
                if ($destObj->eofFlag) {
                    $textViewObj->clearBufferAfterMark($mark);
                } elsif ($destObj->eolFlag) {
                    $textViewObj->clearLineAfterMark($mark);
                }
            }

            # Go back to using the frame that was in use before the <DEST> tag was processed, first
            #   making sure it still exists
            $frameObj = $self->ivShow('mxpFrameHash', $destObj->mxpCurrentFrame);
            if (! $frameObj) {

                $self->mxpDebug(
                    $origToken,
                    'Missing frame \'' . $frameObj->name . ' after </DEST> tag processed',
                    2816,
                );

                # (Use the original frame as an emergency default)
                $frameObj = $self->ivShow('mxpFrameHash', '_top');
            }

            $self->ivPoke('mxpPrevFrame', $destObj->mxpPrevFrame);
            $self->ivPoke('mxpCurrentFrame', $destObj->mxpCurrentFrame);
            $self->ivPoke('currentTabObj', $frameObj->tabObj);

            # There are no Axmud colour/style tags to return
            return @emptyList;
        }
    }

    sub processMxpLinkElement {

        # Called by $self->processMxpElement or ->popMxpStack
        #
        # Process an MXP link element: <A>...</A>
        # NB GA::Obj::Link objects are used to store both <A>..</A> and <SEND>..</SEND>
        #   constructions
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #   $noPopFlag  - Set to TRUE when called by $self->popMxpStack, in which case we don't
        #                   need to call ->popMxpStack again. Set to FALSE otherwise
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, $noPopFlag, @argList) = @_;

        # Local variables
        my (
            $textViewObj, $line, $offset, $linkObj, $type, $stackObj,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (
            ! defined $origToken || ! defined $tagMode || ! defined $keyword
            || ! defined $noPopFlag
        ) {
            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpLinkElement', @_);
            return @emptyList;
        }

        # <A href=URL [hint=text] [expire=name]>Text</A>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
            || $tagMode eq 'defn'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 2901);

            return @emptyList;
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # Process the <A> tag
        if ($tagMode eq 'open') {

            # Watch out for an <A> ... <A> ... </A> construction
            if ($self->mxpCurrentLink) {

                # This a second <A> tag after an earlier </A> tag. Remove the earlier link object,
                #   and show an error (ignoring both tags)
                $self->ivUndef('mxpCurrentLink');
                $self->mxpDebug($origToken, '<A> tag after earlier unclosed <A> tag', 2902);

                return @emptyList;

            # Also watch out for a <SEND> ... <A> ... </SEND> construction
            } elsif ($self->mxpCurrentSend) {

                $self->ivUndef('mxpCurrentSend');
                $self->mxpDebug($origToken, '<A> tag after earlier unclosed <SEND> tag', 2903);

                return @emptyList;
            }

            # Process @argList
            @origList = @checkList = ('href', 'hint', 'expire');
            # Hash of argument names which don't take a corresponding value
            %checkHash = ();
            # Default argument values
            %ivHash = (
                'href'      => undef,
                'hint'      => undef,
                'expire'    => undef,
            );

            if (@argList) {

                do {

                    my ($argName, $argValue) = $self->findMxpArgsByPosn(
                        \@origList,
                        \@checkList,
                        \%ivHash,
                        \%checkHash,
                        shift @argList,     # not 'undef'
                        shift @argList,     # might be 'undef'
                    );

                    if (! defined $argName) {

                        # Unrecognised argument name, or repeating argument name
                        $self->mxpDebug($origToken, 'Malformed element', 2904);

                        return @emptyList;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check the validity of the compulsory 'href=URL' argument
            if (! $ivHash{'href'}) {

                $self->mxpDebug($origToken, 'Malformed element', 2911);

                return @emptyList;
            }

            if (
                index (lc($ivHash{'href'}), 'http://') > -1
                || index (lc($ivHash{'href'}), 'https://') > -1
            ) {
                $type = 'www';
            } elsif (index (lc($ivHash{'href'}), 'telnet://') > -1) {
                $type = 'telnet';
            } elsif (index (lc($ivHash{'href'}), 'mailto://') > -1) {
                $type = 'mail';
            } else {

                $self->mxpDebug($origToken, 'Invalid URL', 2912);

                return @emptyList;
            }

            # We can't process the link until the closing </A> tag is found, so store arguments
            #   temporarily in a GA::Obj::Link object
            ($line, $offset) = $textViewObj->getInsertPosn();
            if (! defined $line) {

                $self->mxpDebug($origToken, 'Internal error while processing link', 2913);

                return @emptyList;
            }

            # The position of the link will be the textview's current insert position, plus the
            #   length of any already-processed text, that hasn't been displayed in the textview
            #   object yet, which has been stored for us in ->mxpOrigText
            # (The -1 argument means this is an incomplete link object, not yet applied to the
            #   current textview and not yet stored in the textview object's registries)
            if (defined $self->mxpOrigText) {

                $offset += length($self->recvLineText);
            }

            $linkObj = Games::Axmud::Obj::Link->new(
                -1,
                $textViewObj,
                $line,
                $offset,
                $type,
            );

            if (! $linkObj) {

                $self->mxpDebug($origToken, 'Internal error while processing link', 2914);

                return @emptyList;
            }

            # Set the link object's IVs
            $linkObj->ivPoke('href', $ivHash{'href'});          # Compulsory arg, never 'undef'
            $linkObj->ivPoke('hint', $ivHash{'hint'});          # May be 'undef'
            $linkObj->ivPoke('expireName', $ivHash{'expire'});  # May be 'undef'
            $linkObj->ivPoke('mxpFlag', TRUE);

            # Store the current link object (the link isn't applied to the textview until the
            #   corresponding </A> tag is processed, although any text inside the <A>...</A> is
            #   displayed immediately, as normal)
            $self->ivPoke('mxpCurrentLink', $linkObj);

            # Also need to create a new MXP stack object and store it in the current textview
            #   object, updating the latter's IVs
            if (! $textViewObj->createMxpStackObj($self, 'A')) {

                $self->mxpDebug($origToken, 'Internal error while processing element', 2915);

                return @emptyList;

            } else {

                # Operation complete
                return @emptyList;
            }

        # Process the <\A> tag
        } else {

            # Watch out for an <A> ... </A> ... </A> construction
            if (! $self->mxpCurrentLink) {

                # This a second </A> tag after an earlier </A> tag
                $self->mxpDebug($origToken, '</A> tag does not match earlier <A> tag', 2916);

                return @emptyList;

            } else {

                $linkObj = $self->mxpCurrentLink;
                $self->ivUndef('mxpCurrentLink');
            }

            if (! $linkObj->text) {

                # There were no valid text tokens between the <A>...</A> tags
                $self->mxpDebug($origToken, 'Invalid <A>...</A> construction', 2917);

                return @emptyList;

            } else {

                # The incomplete link object is now complete, but we can't apply the link to the
                #   current textview yet, as there may be some text before the link which hasn't
                #   been displayed yet
                # Instead, store it temporarily in an IV, and let $self->processIncomingData call
                #   GA::Obj::TextView->add_incompleteLink as soon as it's ready
                $self->ivPush('mxpTempLinkList', $linkObj);
            }

            # Close the <A>..</A> construction, if the calling function isn't doing that (the TRUE
            #   argument means don't call this function back, as you normally would)
            if (! $noPopFlag) {

               return $self->popMxpStack($keyword, TRUE);

            } else {

                # Operation complete
                return @emptyList;
            }
        }
    }

    sub processMxpSendElement {

        # Called by $self->processMxpElement or ->popMxpStack
        #
        # Process an MXP send element: <SEND>...</SEND>
        # NB GA::Obj::Link objects are used to store both <A>..</A> and <SEND>..</SEND>
        #   constructions
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #   $noPopFlag  - Set to TRUE when called by $self->popMxpStack, in which case we don't
        #                   need to call ->popMxpStack again. Set to FALSE otherwise
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, $noPopFlag, @argList) = @_;

        # Local variables
        my (
            $textViewObj, $line, $offset, $linkObj, $stackObj, $href, $hint, $text,
            @emptyList, @origList, @checkList, @cmdList, @optionList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (
            ! defined $origToken || ! defined $tagMode || ! defined $keyword
            || ! defined $noPopFlag
        ) {
            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpSendElement', @_);
            return @emptyList;
        }

        # <SEND [href=command] [hint=text] [prompt] [expire=name]>Text</SEND>
        if (
            ($tagMode eq 'close' && @argList)
            || $tagMode eq 'defn'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3001);

            return @emptyList;
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # Process the <SEND> tag
        if ($tagMode eq 'open') {

            # Watch out for a <SEND> ... <SEND> ... </SEND> construction
            if ($self->mxpCurrentSend) {

                # This a second <SEND> tag after an earlier </SEND> tag. Remove the earlier link
                #   object, and show an error (ignoring both tags)
                $self->ivUndef('mxpCurrentSend');
                $self->mxpDebug($origToken, '<SEND> tag after earlier unclosed <SEND> tag', 3002);

                return @emptyList;

            # Also watch out for an <A> ... <SEND> ... </A> construction
            } elsif ($self->mxpCurrentLink) {

                $self->ivUndef('mxpCurrentSend');
                $self->mxpDebug($origToken, '<A> tag after earlier unclosed <SEND> tag', 3003);

                return @emptyList;
            }

            # Process @argList

            # List of argument names, in the expected order
            @origList = @checkList = ('href', 'hint', 'prompt', 'expire');
            # Hash of argument names which don't take a corresponding value
            %checkHash = (
                'prompt'    => undef,
            );
            # Default argument values
            %ivHash = (
                'href'      => undef,
                'hint'      => undef,
                'prompt'    => FALSE,
                'expire'    => undef,
            );

            if (@argList) {

                do {

                    my ($argName, $argValue) = $self->findMxpArgsByPosn(
                        \@origList,
                        \@checkList,
                        \%ivHash,
                        \%checkHash,
                        shift @argList,     # not 'undef'
                        shift @argList,     # might be 'undef'
                    );

                    if (! defined $argName) {

                        # Unrecognised argument name, or repeating argument name
                        $self->mxpDebug($origToken, 'Malformed element', 3011);

                        return @emptyList;

                    } elsif ($argName eq 'prompt') {

                        $ivHash{$argName} = TRUE;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # We can't process the link until the closing </SEND> tag is found, so store arguments
            #   temporarily in a GA::Obj::Link object
            ($line, $offset) = $textViewObj->getInsertPosn();
            if (! defined $line) {

                $self->mxpDebug($origToken, 'Internal error while processing send', 3012);

                return @emptyList;
            }

            # The position of the link will be the textview's current insert position, plus the
            #   length of any already-processed text, that hasn't been displayed in the textview
            #   object yet, which has been stored for us in ->mxpOrigText
            # (The -1 argument means this is an incomplete link object, not yet applied to the
            #   current textview and not yet stored in the textview object's registries)
            if (defined $self->mxpOrigText) {

                $offset += length($self->recvLineText);
            }

            $linkObj = Games::Axmud::Obj::Link->new(
                -1,
                $textViewObj,
                $line,
                $offset,
                'cmd',
            );

            if (! $linkObj) {

                $self->mxpDebug($origToken, 'Internal error while processing send', 3013);

                return @emptyList;
            }

            # Set the link object's IVs
            $linkObj->ivPoke('href', $ivHash{'href'});              # May be 'undef'
            $linkObj->ivPoke('hint', $ivHash{'hint'});              # May be 'undef'
            $linkObj->ivPoke('mxpPromptFlag', $ivHash{'prompt'});   # Never 'undef'
            $linkObj->ivPoke('expireName', $ivHash{'expire'});      # May be 'undef'
            $linkObj->ivPoke('mxpFlag', TRUE);

            $href = $linkObj->href;
            $hint = $linkObj->hint;
            if (
                $href
                && (! $hint || $href =~ m/.\|./ || $hint =~ m/.\|./)
            ) {
                # When the link is clicked, a popup menu should be displayed. Set IVs to prepare
                #   for that
                $linkObj->ivPoke('popupFlag', TRUE);

                @cmdList = split('\|', $href);
                if ($hint) {

                    @optionList = split('\|', $hint);

                    if (((scalar @cmdList) + 1) == (scalar @optionList)) {

                        # The first item in @optionList is the hint; the rest are menu items
                        $linkObj->ivPoke('hint', shift @optionList);

                    } elsif ((scalar @cmdList) != (scalar @optionList)) {

                        $self->mxpDebug($origToken, 'Invalid SEND menu option format', 3014);

                        return @emptyList;

                    } else {

                        $linkObj->ivPoke('hint', undef);
                    }

                } else {

                    $linkObj->ivPoke('hint', undef);
                }

                $linkObj->ivPoke('popupCmdList', @cmdList);
                $linkObj->ivPoke('popupItemList', @optionList);
            }

            # Store the current link object (the link isn't applied to the textview until the
            #   corresponding </SEND> tag is processed, although any text inside the
            #   <SEND>...</SEND> is displayed immediately, as normal)
            $self->ivPoke('mxpCurrentSend', $linkObj);

            # Also need to create a new MXP stack object and store it in the current textview
            #   object, updating the latter's IVs
            if (! $textViewObj->createMxpStackObj($self, 'SEND')) {

                $self->mxpDebug($origToken, 'Internal error while processing element', 3015);

                return @emptyList;

            } else {

                # Operation complete
                return @emptyList;
            }

        # Process the <\SEND> tag
        } else {

            # Watch out for a <SEND> ... </SEND> ... </SEND> construction
            if (! $self->mxpCurrentSend) {

                # This a second </SEND> tag after an earlier </SEND> tag
                $self->mxpDebug($origToken, '</SEND> tag does not match earlier <SEND> tag', 3021);

                return @emptyList;

            } else {

                $linkObj = $self->mxpCurrentSend;
                $self->ivUndef('mxpCurrentSend');
            }

            if (! $linkObj->text && $linkObj->type ne 'image') {

                # There were no valid text tokens (or clickable images) between the <SEND>...</SEND>
                #   tags
                $self->mxpDebug($origToken, 'Invalid <SEND>...</SEND> construction', 3022);

                return @emptyList;

            } else {

                # The incomplete link object is now complete, but we can't apply the link to the
                #   current textview yet, as there may be some text before the link which hasn't
                #   been displayed yet
                # Instead, store it temporarily in an IV, and let $self->processIncomingData call
                #   GA::Obj::TextView->add_incompleteLink as soon as it's ready
                $self->ivPush('mxpTempLinkList', $linkObj);

                # Process any &text; entities (e.g. <!ELEMENT Item '<send href="buy &text;">'> ),
                #   but not for clickable images
                $href = $linkObj->href;
                $text = $linkObj->text;
                if ($href && $text && $linkObj->type ne 'image') {

                    $href =~ s/\&text\;/$text/g;
                    $linkObj->ivPoke('href', $href);
                }

                # Do the same, if a popup menu is to be created
                foreach my $item ($linkObj->popupCmdList) {

                    $item =~ s/\&text\;/$text/g;
                    push (@cmdList, $item);
                }

                foreach my $item ($linkObj->popupItemList) {

                    $item =~ s/\&text\;/$text/g;
                    push (@optionList, $item);
                }

                $linkObj->ivPoke('popupCmdList', @cmdList);
                $linkObj->ivPoke('popupItemList', @optionList);
            }

            # Close the <SEND>...</SEND> construction, if the calling function isn't doing that (the
            #   TRUE function means don't call this function back, as you normally would)
            if (! $noPopFlag) {

                return $self->popMxpStack($keyword, TRUE);

            } else {

                # Operation complete
                return @emptyList;
            }
        }
    }

    sub processMxpSoundElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP sound element: <SOUND> or <MUSIC>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $token,
            @emptyList,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpSoundElement', @_);
            return @emptyList;
        }

        # <SOUND FName V=volume L=repeats P=priority C=continue T=type U=url>
        # <MUSIC FName V=volume L=repeats P=priority C=continue T=type U=url>
        # <SOUND Off>
        # <SOUND Off U=default_url>
        # <MUSIC Off>
        # <MUSIC Off U=default_url>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3101);

            return @emptyList;
        }

        # Ignore this tag if the client flag is set
        if (! $axmud::CLIENT->allowMxpSoundFlag) {

            return @emptyList;
        }

        # The @argList should be identical to the one used in an MSP sound token
        # Instead of processing the @argList twice, once here and again in
        #   $self->processMspSoundTrigger, we'll just create a fake MSP sound token and pass it
        #   straight to $self->processMspSoundTrigger
        if ($keyword eq 'SOUND') {
            $token = '!!SOUND(';
        } else {
            $token = '!!MUSIC(';
        }

        # @argList is now in the form
        #   (FName, <undef>, param_name, param_value, param_name, param_value...)
        $token .= shift @argList;
        shift @argList;

        if (@argList) {

            do {
                my $name = shift @argList;
                my $value = shift @argList;

                $token .= ' ' . $name . '=' . $value;
            } until (! @argList);
        }

        $token .= ')';

        # Use MSP code to deal with playing the sound. (The TRUE argument means that this is an
        #   MXP sound, not an MSP one, and that MXP file filters can be used)
        if (! $self->processMspSoundTrigger($token, TRUE)) {

            $self->mxpDebug($origToken, 'Malformed element', 3102);
        }

        # (No Axmud colour/style tags involved)
        return @emptyList;
    }

    sub processMxpImageElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP image element: <IMAGE>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $textViewObj, $url, $urlRegex, $height, $width, $align, $path, $file, $dir, $ext,
            $convertFlag, $fetchObj, $pixbuf, $pixbuf2, $pbw, $pbh, $horiz, $vert, $hFactor,
            $vFactor, $moveX, $moveY, $rgba,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpImageElement', @_);
            return @emptyList;
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # <IMAGE FName URL=url T=type H=height W=width HSPACE=hspace VSPACE=vspace
        #       ALIGN=left|right|top|middle|bottom ISMAP>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3201);

            return @emptyList;
        }

        # Ignore this tag if the client flag is set
        if (! $axmud::CLIENT->allowMxpImageFlag) {

            return @emptyList;
        }

        # Process @argList

        # List of argument names, in the expected order
        @origList = @checkList = (
            'fname', 'url', 't', 'h', 'w', 'hspace', 'vspace', 'align', 'ismap',
        );

        # Hash of argument names which don't take a corresponding value
        %checkHash = (
            'url'       => undef,
            'hspace'    => undef,
            'vspace'    => undef,
            'ismap'     => undef,
        );
        # Default argument values
        %ivHash = (
            'fname'     => undef,
            'url'       => undef,
            't'         => undef,
            'h'         => undef,
            'w'         => undef,
            'hspace'    => FALSE,
            'vspace'    => FALSE,
            'align'     => undef,       # Cannot be implemented in Gtk2 - text tags broken
            'ismap'     => FALSE,
        );

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 3202);

                    return @emptyList;

                } elsif ($argName eq 'hspace' || $argName eq 'vspace' || $argName eq 'ismap') {

                    $ivHash{$argName} = TRUE;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check any specified argument values are valid. In some cases, give up; in other cases,
        #   use default values
        $url = $ivHash{'url'};
        $urlRegex = $axmud::CLIENT->constUrlRegex;
        if (defined $url) {

            if (! ($url =~ m/$urlRegex/)) {

                return @emptyList;

            } else {

                # Currently (v1.0.836), File::Fetch doesn't support https:// addresses, so
                #   substitute in http:// and hope for the best
                $url =~ s/^https\:/http\:/;
            }
        }

        # The height and width attributes can be in the form of integers (size in pixels), in the
        #   form of 'nc' (n character heights) or in the form 'n%' (percentage of the available
        #   space)
        # If specified, convert these values into a size in pixels
        $height = $ivHash{'h'};
        if (defined $height && $height =~ m/^\d+[\%c]?/) {

            $height = $self->convertMxpImageSize($height, 'height');
        }

        $width = $ivHash{'w'};
        if (defined $width && $width =~ m/^\d+[\%c]?/) {

            $width = $self->convertMxpImageSize($width, 'width');
        }

        $align = $ivHash{'align'};
        if (defined $align) {

            $align = lc($align);
            if (
                $align ne 'left'
                && $align ne 'right'
                && $align ne 'top'
                && $align ne 'middle'
                && $align ne 'bottom'
            ) {
                $align = undef;
            }
        }

        # Convert FName into a full filepath (must a relative filepath, pointing at a directory used
        #   to store images for the current world)
        if ($ivHash{'t'}) {

            $path = $axmud::DATA_DIR . '/mxp/' . $self->currentWorld->name
                        . '/' . $ivHash{'t'} . '/' . $ivHash{'fname'};

        } else {

            $path = $axmud::DATA_DIR . '/mxp/' . $self->currentWorld->name . '/' . $ivHash{'fname'};
        }

        # If this file actually exists, use it. Otherwise, try downloading it from the specified
        #   URL (if any)
        if (! -e $path) {

            # If a partial URL was not specified, or if we're not allowed to download, then there's
            #   no image to display
            # NB The MXP spec doesn't specify wildcards, as the MSP spec does, so wildcards are not
            #   implemented here
            if (! $url || ! $axmud::CLIENT->allowMxpLoadImageFlag) {

                return @emptyList;

            } else {

                # Compile the full URL from which to download
                if (substr($url, -1, 1) ne '/') {

                    $url .= '/';
                }

                $url .= $ivHash{'fname'};

                # Attempt to download the file
                ($file, $dir) = File::Basename::fileparse($path);
                $fetchObj = File::Fetch->new(uri => $url);
                if (! $fetchObj->fetch(to => $dir)) {

                    # Download error - display nothing
                    return @emptyList;
                }
            }
        }

        # For image files, Axmud only supports file extensions in GA::Client->constMxpFormatHash
        ($file, $dir, $ext) = File::Basename::fileparse($path, qr/\.[^.]*/);
        if (! $ext) {

            # Ignore any files with no extension at all
            return @emptyList;
        }

        $ext =~ s/^\.//;
        if (! $axmud::CLIENT->ivExists('constMxpFormatHash', $ext)) {

            # For any other file extension, apply the MXP filter (if the world has specified one)
            $path = $self->applyMxpFileFilter($path);
            if (! $path) {

                # No MXP file filter supplied, or file conversion failed
                return @emptyList;
            }

            # Check the file format of the converted file is supported
            ($file, $dir, $ext) = File::Basename::fileparse($path, qr/\.[^.]*/);
            $ext =~ s/^\.//;
            if (! $axmud::CLIENT->ivExists('constMxpFormatHash', $ext)) {

                # File format not supported
                return @emptyList;

            } else {

                $convertFlag = TRUE;
            }
        }

        # Convert the image file to a pixbuf
        if (! defined $width || ! defined $height) {

            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($path);
            if ($pixbuf) {

                $width = $pixbuf->get_width();
                $height = $pixbuf->get_height();
            }

        } else {

            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file_at_scale(
                $path,
                $width,
                $height,
                FALSE,                          # Don't preserve aspect ratio
            );
        }

        if ($pixbuf) {

            # Add some padding, if specified (increase the image size by 20%, without increasing the
            #   image itself)
            if ($ivHash{'hspace'} || $ivHash{'vspace'}) {

                if ($ivHash{'hspace'}) {
                    $hFactor = 1.2;
                } else {
                    $hFactor = 1;
                }

                if ($ivHash{'vspace'}) {
                    $vFactor = 1.2;
                } else {
                    $vFactor = 1;
                }

                $pixbuf2 = Gtk2::Gdk::Pixbuf->new(
                    'GDK_COLORSPACE_RGB',
                    FALSE,
                    $pixbuf->get_bits_per_sample(),
                    ($width * $hFactor),
                    ($height * $vFactor),
                );

                # Need to specify an RGBA colour to use as padding. Would like to specify a 0 alpha
                #   value, so that $pixbuf2's 'colour' automatically matches the background, but it
                #   doesn't work for some reason
                $rgba = $axmud::CLIENT->returnRGBColour($textViewObj->backgroundColour);
                $rgba =~ s/^#//;
                $rgba = ((hex $rgba) * 256) + 255;
                $pixbuf2->fill($rgba);

                $moveX = (($width * $hFactor) - $width) / 2;        # May be 0
                $moveY = (($height * $vFactor) - $height) / 2;

                $pixbuf->composite(
                    $pixbuf2,
                    $moveX,
                    $moveY,
                    $width,
                    $height,
                    $moveX,
                    $moveY,
                    1,
                    1,
                    'GDK_INTERP_BILINEAR',
                    255,
                );

                $pixbuf = $pixbuf2;
            }
        }

        if ($pixbuf) {

            # Handle clickable images. Ignore ISMAP if the <IMAGE> tag doesn't appear between
            #   <SEND>...</SEND>, or if anything else (such as some text) appears between
            #   <SEND>...</SEND>
            if (
                $ivHash{'ismap'}
                && $self->mxpCurrentSend
                && ! $self->mxpCurrentSend->text
            ) {
                $textViewObj->showImage($pixbuf, $self->mxpCurrentSend, 'echo');
                # Update the GA::Obj::Link object, setting its type to 'image' so that
                #   $self->processMxpSendElement knows not to complain that there's no clickable
                #   text
                $self->mxpCurrentSend->ivPoke('type', 'image');
                # The world command should be sent invisibly
                $self->mxpCurrentSend->ivPoke('mxpInvisFlag', TRUE);
                # For <SEND showmap><IMAGE map.jpg ISMAP></SEND>, ->href is currently set to
                #   'showmap'. The MXP spec requires that we append the position of the click to the
                #   world command sent. GA::Obj::TextView->showImage substitutes %x and %y for the
                #   click coordinates
                $self->mxpCurrentSend->ivPoke(
                    'href',
                    $self->mxpCurrentSend->href . '?%x,%y',
                );

            # Non-clickable images
            } else {

                $textViewObj->showImage($pixbuf, undef, 'echo');
            }

            # $self->recvImgLineText must be updated for all processed images
            $self->ivPoke('recvImgLineText', $self->recvImgLineText . '[' . $file . ']');
        }

        # Delete any converted file (leaving the original in place)
        if ($convertFlag) {

            unlink $path;
        }

        # (No Axmud colour/style tags involved)
        return @emptyList;
    }

    sub processMxpFilterElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP file filter element: <FILTER>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $filterObj,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpFilterElement', @_);
            return @emptyList;
        }

        # <FILTER SRC=ext DEST=ext NAME=plugin PROC=num>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3301);

            return @emptyList;
        }

        # Ignore this tag if the client flag is set
        if (! $axmud::CLIENT->allowMxpFilterImageFlag) {

            return @emptyList;
        }

        # Process @argList

        # List of argument names, in the expected order
        @origList = @checkList = ('src', 'dest', 'name', 'proc');
        # Hash of argument names which don't take a corresponding value
        %checkHash = ();
        # Default argument values
        %ivHash = (
            'src'     => undef,
            'dest'    => 'bmp',
            'name'    => undef,
            'proc'    => 0,
        );

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 3302);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # All arguments except PROC=num are compulsory
        if (! $ivHash{'src'} || ! $ivHash{'dest'} || ! $ivHash{'name'}) {

            $self->mxpDebug($origToken, 'Malformed element', 3311);

            return @emptyList;
        }

        # MXP supports .gif and .bmp images, so ignore any attempt to create a file filter for
        #   them
        if (lc($ivHash{'src'}) eq 'gif' || lc($ivHash{'src'}) eq 'bmp') {

            return @emptyList;
        }

        # Create a GA::Mxp::Filter to store the details until they're required
        $filterObj = Games::Axmud::Mxp::Filter->new(
            $self,
            $ivHash{'src'},
            $ivHash{'dest'},
            $ivHash{'name'},
            $ivHash{'proc'},
        );

        # Update the registry. If (for some reason) the world has already specified a file filter
        #   with the same source file extention, replace it with this one
        if ($filterObj) {

            $self->ivAdd('mxpFilterHash', $filterObj->src, $filterObj);
        }

        # (No Axmud colour/style tags involved)
        return @emptyList;
    }

    sub processMxpGaugeElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP gauge element: <GAUGE>, <STAT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments or if there is no GA::Strip::GaugeBox in this
        #       sessiOn's 'main' window in which to draw gauges
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $entityObj, $maxEntityObj, $stripObj, $level, $gaugeObj, $fullCol, $emptyCol, $maxValue,
            $label, $labelCol, $labelFlag,
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpGaugeElement', @_);
            return @emptyList;
        }

        # <GAUGE EntityName [Max=EntityName] [Caption=text] [Color=colour]>
        # <STAT EntityName [Max=EntityName] [Caption=text]>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3401);

            return @emptyList;
        }

        # Ignore these tags if the client flag is set
        if (! $axmud::CLIENT->allowMxpGaugeFlag) {

            return @emptyList;
        }

        # Process @argList
        if ($keyword eq 'GAUGE') {

            @origList = @checkList = ('name', 'max', 'caption', 'color');
            # Hash of argument names which don't take a corresponding value
            %checkHash = (
                'name'      => undef,
            );
            # Default argument values
            %ivHash = (
                'name'      => undef,
                'max'       => undef,
                'caption'   => undef,
                'color'     => undef,
            );

        } else {

            @origList = @checkList = ('name', 'max', 'caption');
            # Hash of argument names which don't take a corresponding value
            %checkHash = (
                'name'      => undef,
            );
            # Default argument values
            %ivHash = (
                'name'      => undef,
                'max'       => undef,
                'caption'   => undef,
            );
        }

        if (@argList) {

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 3402);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check the validity of the compulsory 'EntityName' argument
        if (! $ivHash{'name'}) {

            $self->mxpDebug($origToken, 'Malformed element', 3411);

            return @emptyList;
        }

        # Check that the specified entities exist. Discworld sends <STAT> tags for entities that
        #   don't exist yet, so create an entity using a fake <!ENTITY> tag as well as displaying an
        #   error
        $entityObj = $self->ivShow('mxpEntityHash', $ivHash{'name'});
        if (! $entityObj) {

            $self->processMxpEntElement(
                '<!ENTITY ' . $ivHash{'name'} . ' 0>',
                'defn',
                'EN',
                'Name',
                $ivHash{'name'},
                'Value',
                0,
            );

            # (Check the entity was created)
            $entityObj = $self->ivShow('mxpEntityHash', $ivHash{'name'});
            if (! $entityObj) {

                # (This was the original error message, when Axmud didn't accept <GAUGE>/<STAT> tags
                #   with non-existent entities)
                $self->mxpDebug($origToken, 'Entity not found', 3412);

                return @emptyList;
            }
        }

        if ($ivHash{'max'}) {

            $maxEntityObj = $self->ivShow('mxpEntityHash', $ivHash{'max'});
            if (! $maxEntityObj) {

                $self->processMxpEntElement(
                    '<!ENTITY ' . $ivHash{'max'} . ' 0>',
                    'defn',
                    'EN',
                    'Name',
                    $ivHash{'max'},
                    'Value',
                    0,
                );

                $maxEntityObj = $self->ivShow('mxpEntityHash', $ivHash{'max'});
                if (! $maxEntityObj) {

                    $self->mxpDebug($origToken, 'Entity not found', 3413);

                    return @emptyList;
                }
            }
        }

        # Only one gauge per entity is allowed. Ignore repeat requests
        if ($self->ivExists('mxpGaugeHash', $ivHash{'name'})) {

            return @emptyList;
        }

        # Find the GA::Strip::GaugeBox object for this session's 'main' window
        if (! $self->mxpGaugeStripObj) {

            $stripObj = $self->session->mainWin->ivShow(
                'firstStripHash',
                'Games::Axmud::Strip::GaugeBox',
            );

            if (! $stripObj) {

                return @emptyList;

            } else {

                $self->ivPoke('mxpGaugeStripObj', $stripObj);
            }
        }

        # All gauges created by MXP are drawn on their own gauge level
        if (! defined $self->mxpGaugeLevel) {

            $level = $self->mxpGaugeStripObj->addGaugeLevel($self);
            if (! defined $level) {

                # Error, or gauge box is full up, so draw nothing
                return @emptyList;

            } else {

                $self->ivPoke('mxpGaugeLevel', $level);
            }

        } else {

            $level = $self->mxpGaugeLevel;
        }

        # If specified, convert the colour to RGB (otherwise, the gauge uses default colours)
        if ($ivHash{'color'}) {

            ($fullCol) = $self->convertMxpColour($ivHash{'color'}, FALSE);
            if ($fullCol) {

                $emptyCol = '#000000';
            }
        }

        # Set the maximum value
        if ($maxEntityObj) {

            $maxValue = $maxEntityObj->value;

        } else {

            # Gauge is always full
            $maxValue = $entityObj->value;
        }

        # Set the gauge label to use. If none was specified, the call to ->addGauge needs 'undef'
        #   arguments
        if ($ivHash{'caption'}) {

            $label = $ivHash{'caption'};
            $labelFlag = TRUE;

            # GA::Client->constHtmlContrastHash provides a list of label colours that fit well with
            #   standard HTML 4 colours. If $fullCol is one of those, consult that hash. Otherwise
            #   use white as the label colour - not perfect, but at least it can be seen over the
            #   black empty portion of the gauge
            if ($fullCol) {

                $labelCol = $axmud::CLIENT->ivShow('constHtmlContrastHash', $ivHash{'color'});
            }

            if (! $labelCol) {

                $labelCol = '#FFFFFF';
            }
        }

        # Draw the gauge
        if ($keyword eq 'GAUGE') {

            $gaugeObj = $self->mxpGaugeStripObj->addGauge(
                $self,
                $level,
                $entityObj->value,
                $maxValue,
                FALSE,                  # Total size is $maxValue
                $label,
                $fullCol,
                $emptyCol,
                $labelCol,
                $labelFlag,
            );

        } else {

            $gaugeObj = $self->mxpGaugeStripObj->addTextGauge(
                $self,
                $level,
                $entityObj->value,
                $maxValue,
                FALSE,                  # Total size is $maxValue
                $label,
            );
        }

        if ($gaugeObj) {

            # Add this GA::Obj::Gauge object to a registry, so that other parts of the code know
            #   that the gauge must be redrawn when either entity's value is updated
            $self->ivAdd('mxpGaugeHash', $entityObj->name, $gaugeObj);
            $gaugeObj->ivPoke('mxpEntity', $ivHash{'name'});

            if ($maxEntityObj) {

                $self->ivAdd('mxpGaugeHash', $maxEntityObj->name, $gaugeObj);
                $gaugeObj->ivPoke('mxpMaxEntity', $ivHash{'max'});
            }
        }

        return @emptyList;
    }

    sub processMxpCrosslinkElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP crosslink element: <RELOCATE>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            @emptyList, @origList, @checkList,
            %checkHash, %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpCrosslinkElement', @_);
            return @emptyList;
        }

        # <RELOCATE hostname port>
        # </RELOCATE>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
            || $tagMode eq 'defn'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3501);

            return @emptyList;
        }

        # Ignore this tag if the client flag is set
        if (! $axmud::CLIENT->allowMxpCrosslinkFlag) {

            return @emptyList;
        }

        if ($tagMode eq 'open') {

            # Process @argList
            @origList = @checkList = ('hostname', 'port');
            # Hash of argument names which don't take a corresponding value
            %checkHash = ();
            # Default argument values
            %ivHash = (
                'hostname'  => undef,
                'port'      => undef,
            );

            do {

                my ($argName, $argValue) = $self->findMxpArgsByPosn(
                    \@origList,
                    \@checkList,
                    \%ivHash,
                    \%checkHash,
                    shift @argList,     # not 'undef'
                    shift @argList,     # might be 'undef'
                );

                if (! defined $argName) {

                    # Unrecognised argument name, or repeating argument name
                    $self->mxpDebug($origToken, 'Malformed element', 3502);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);

            # We'll let the hostname be anything, but the port should at least be a valid integer
            #   in the usual range
            if (
                ! $ivHash{'hostname'}
                || ! defined $ivHash{'port'}
                || ! $axmud::CLIENT->floatCheck($ivHash{'port'}, 0, 65535)
            ) {
                $self->mxpDebug($origToken, 'Invalid relocation hostname and/or port', 3511);

                return @emptyList;
            }

            # Don't allow a crosslink operation if one is already in progress, or if a delayed
            #   quit has been set up
            if ($self->mxpRelocateMode eq 'none' && ! defined $self->delayedQuitTime) {

                # The crosslinking process will start on the next incoming data loop (allowing the
                #   server to send a <QUIET> tag right after this one
                $self->ivPoke('mxpRelocateMode', 'wait_start');
                $self->ivPoke('mxpRelocateHost', $ivHash{'hostname'});
                $self->ivPoke('mxpRelocatePort', $ivHash{'port'});
            }

        } else {

            # Mark the character as logged in, if it isn't already (this sets
            #   $self->mxpRelocateMode to 'none', which terminates the crosslinking operation)
            $self->doLogin();
        }

        return @emptyList;
    }

    sub processMxpLoginElement {

        # Called by $self->processMxpElement
        #
        # Process an MXP login element: <USER>, <PASSWORD>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpLoginElement', @_);
            return @emptyList;
        }

        # <USER>
        # <PASSWORD>
        if (
            ($tagMode eq 'open' && @argList)
            || $tagMode ne 'open'
        ) {
            $self->mxpDebug($origToken, 'Malformed element', 3601);

            return @emptyList;
        }

        # If the current character's password was not specified by the call to $self->new, don't use
        #   either tag (even if the user has subsequently typed ';setchar', which may make the
        #   correct password available. This makes sure that, if the user opted to prevent
        #   auto-logins, an automatic login won't happen as a response to these tags)
        if ($self->initChar && $self->initPass) {

            if ($keyword eq 'USER') {

                $self->send($self->initChar);
                $self->ivPoke('mxpLoginMode', 'user_tag');

            } elsif ($keyword eq 'PASSWORD') {

                $self->send($self->initPass);

                if ($self->mxpLoginMode eq 'user_tag') {

                    $self->ivPoke('mxpLoginMode', 'pwd_tag');
                    # During an MXP crosslinking operation, wait for the </RELOCATE> tag to mark
                    #   the character as 'logged in'
                    if ($self->mxpRelocateMode eq 'none') {

                        $self->doLogin();
                    }
                }
            }
        }

        return @emptyList;
    }

    sub processMxpOfficialElement {

        # Called by $self->processMxpElement
        #
        # Process an official MXP element, not covered by other functions (<EXPIRE>, <VERSION> and
        #   <QUIET>)
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $argName, $argValue, $string, $termTypeMode, $customClientName, $customClientVersion,
            @emptyList, @tagList, @textViewList,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper(
                $self->_objClass . '->processMxpOfficialElement',
                @_,
            );

            return @emptyList;
        }

        # <EXPIRE [Name]>
        # <VERSION [styleversion]>
        # <QUIET>
        if ($tagMode ne 'open') {

            $self->mxpDebug($origToken, 'Malformed element', 3701);

            return @emptyList;
        }


        # <EXPIRE [Name]>
        if ($keyword eq 'EXPIRE') {

            $argName = shift @argList;
            $argValue = shift @argList;     # Must be 'undef'

            if (@argList || defined $argValue) {

                $self->mxpDebug($origToken, 'Malformed element', 3711);

                return @emptyList;
            }

            # Compile a list of textview objects used by MXP frames. If $self->mxpFrameHash is
            #   empty (because no frames have been added), use the default tab's textview which
            #   would be added to that hash with the name of '_top')
            if (! $self->mxpFrameHash) {

                push (@textViewList, $self->defaultTabObj->textViewObj);

            } else {

                foreach my $frameObj ($self->ivValues('mxpFrameHash')) {

                    push (@textViewList, $frameObj->textViewObj);
                }
            }

            # Check all GA::Obj::Link objects, and expire any MXP links which match this expiry
            #   name (or if $name was not supplied, any MXP links that match any expiry name, but
            #   not those which have no expire name at all)
            foreach my $textViewObj (@textViewList) {

                foreach my $linkObj ($textViewObj->ivValues('linkObjHash')) {

                    if (
                        $linkObj->mxpFlag
                        && $linkObj->expireName
                        && (! $argName || $argName eq $linkObj->expireName)
                    ) {
                        $linkObj->ivPoke('expiredFlag', TRUE);

                        if (
                            $textViewObj->currentLinkObj
                            && $textViewObj->currentLinkObj eq $linkObj
                        ) {
                            # Hide tooltips, reset the cursor, close any 'dialogue' windows
                            $textViewObj->resetCurrentLink();
                        }
                    }
                }
            }

        # <VERSION [styleversion]>
        } elsif ($keyword eq 'VERSION') {

            $argName = shift @argList;
            $argValue = shift @argList;     # Must be 'undef'

            if (@argList || defined $argValue) {

                $self->mxpDebug($origToken, 'Malformed element', 3721);

                return @emptyList;
            }

            if (defined $argName) {

                # MXP can send a style sheet number, indicating "the current version of the optional
                #   style sheet" which seems to be something to do with the subset of elements and
                #   entities currently in use. The MXP spec only requires us to store this number,
                #   and to return it in response to a VERSION tag
                $self->ivPoke('mxpStyleSheetNum', $argName);

            } else {

                # Return version information to the world, in the form
                #   <VERSION MXP=mxpversion STYLE=styleversion CLIENT=clientname
                #   VERSION=clientversion REGISTERED=yes/no>
                $string = '<VERSION';
                $string .= ' MXP=' . $axmud::CLIENT->constMxpVersion;

                if (defined $self->mxpStyleSheetNum) {

                    $string .= ' STYLE=' . $self->mxpStyleSheetNum;
                }

                # (Use the same rules about disclosing this client's identity, as are used in
                #   TTYPE negotiations)
                if ($self->currentWorld->ivExists('termOverrideHash', 'termTypeMode')) {
                    $termTypeMode = $self->currentWorld->ivShow('termOverrideHash', 'termTypeMode');
                } else {
                    $termTypeMode = $axmud::CLIENT->termTypeMode;
                }

                if ($self->currentWorld->ivExists('termOverrideHash', 'customClientName')) {

                    $customClientName
                        = $self->currentWorld->ivShow('termOverrideHash', 'customClientName');

                } else {

                    $customClientName = $axmud::CLIENT->customClientName;
                }

                if ($self->currentWorld->ivExists('termOverrideHash', 'customClientVersion')) {

                    $customClientVersion
                        = $self->currentWorld->ivShow('termOverrideHash', 'customClientVersion');

                } else {

                    $customClientVersion = $axmud::CLIENT->customClientVersion;
                }

                if ($termTypeMode eq 'send_client') {

                    $string .= ' CLIENT=' . $axmud::NAME_SHORT;

                } elsif ($termTypeMode eq 'send_client_version') {

                    $string .= ' CLIENT=' . $axmud::NAME_SHORT;
                    $string .= ' VERSION=' . $axmud::VERSION;

                } elsif ($termTypeMode eq 'send_custom_client') {

                    if ($customClientName) {

                        $string .= ' CLIENT=' . $customClientName;

                        if ($customClientVersion) {

                            $string .= ' VERSION=' . $customClientVersion;
                        }
                    }
                }

                # (All version of Axmud are registered ;) )
                $string .= ' REGISTERED=YES>';

                $self->optSendMxpSecure($string);
            }

        # <QUIET>
        } elsif ($keyword eq 'QUIET') {

            if (@argList) {

                $self->mxpDebug($origToken, 'Malformed element', 3731);

                return @emptyList;

            # Ignore <QUIET> tags outside of <RELOCATE>...</RELOCATE> constructions
            } elsif ($self->mxpRelocateMode ne 'none') {

                $self->ivPoke('mxpRelocateQuietFlag', TRUE);
            }

        } else {

            $self->mxpDebug($origToken, 'Internal error while processing element', 3741);

            return @emptyList;
        }

        return @tagList,
    }

    sub processMxpCustomElement {

        # Called by $self->processMxpElement
        #
        # Process a custom MXP element, created by an earlier <!ELEMENT> tag
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements, 'defn' for <!..>
        #                   elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $elementObj, $attCount, $recogniseFlag,
            @emptyList, @tagList, @miniTagList,
            %attHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpCustomElement', @_);
            return @emptyList;
        }

        if (($tagMode eq 'close' && @argList) || $tagMode eq 'defn') {

            $self->mxpDebug($origToken, 'Malformed element', 3801);

            return @emptyList;
        }

        # Get the stored element object
        $elementObj = $self->ivShow('mxpElementHash', lc($keyword));

        # <some_element>
        if ($tagMode eq 'open') {

            # We need to check that </some_element> we receive in the future is matched by an
            #   earlier <some_element> tag (i.e. the tag we're processing now), so we create a new
            #   MXP stack object for this purpose
            # (If the element has a definition (e.g. '<COLOR red><B>'), the definition is used as a
            #   macro expansion for the element, and stack objects are also created for those tags)

            # Create a new MXP stack object and store it in the current textview object,
            #   updating the latter's IVs
            if (! $self->currentTabObj->textViewObj->createMxpStackObj($self, $keyword)) {

                $self->mxpDebug($origToken, 'Internal error while processing element', 3811);

                return @emptyList;
            }

            # If the custom element defines any tag properties, start storing text between the
            #   opening tag we're processing now, and the matching closing tag we haven't processed
            #   yet
            # (The hash is updated with every call to $self->processTextToken)
            if (
                $elementObj->flagArg
                && ! $self->ivExists('mxpFlagTextHash', $elementObj->flagArg)
            ) {
                $self->ivAdd('mxpFlagTextHash', $elementObj->flagArg, '');
            }

            # Deal with supplied attributes, e.g. in elements like these:
            #   <colorbold col=blue>This text is in bold blue</colortext>
            #   <colorbold blue>This text is in bold blue</colortext>
            # First, import this element's hash of attributes and their default values
            %attHash = $elementObj->attHash;
            # Next, modify the hash, using any attributes values supplied by this tag
            if (@argList) {

                $attCount = -1;

                do {

                    my ($attName, $attValue, $thisAtt);

                    $attName = shift @argList;
                    $attValue = shift @argList;

                    # e.g. <colorbold col=blue>
                    if (defined $attValue) {

                        if (! exists $attHash{$attName}) {

                            $self->mxpDebug(
                                $origToken,
                                'Element specified unrecognised attribute \''
                                . $attName . '\'',
                                3812,
                            );

                        } else {

                            # Use the specified value, instead of the default value
                            $attHash{$attName} = $attValue;
                        }

                    # e.g. <colorbold blue>
                    } else {

                        # We expect that this will set the attribute in $elementObj->attList, whose
                        #   index is $attCount
                        $attCount++;
                        $thisAtt = $elementObj->ivIndex('attList', $attCount);
                        if (! defined $thisAtt) {

                            $self->mxpDebug($origToken, 'Attribute \'' . $attName . '\'', 3813);

                        } else {

                            # Here, $attName is actually an attribute value, not an attribute name
                            $attHash{$thisAtt} = $attName;
                        }
                    }

                } until (! @argList);
            }

            # The macro expansion is stored as a list of MXP tags in $elementObj->defnList
            # Process each one in turn by calling ->processMxpElement recursively;
            #   GA::Obj::TextView->mxpModalStackHash will be modified by them
            foreach my $thisTag ($elementObj->defnList) {

                # Mode 1 - Only allow simple atomic elements like <B> (so don't allow <!ELEMENT>
                #   for example), and don't allow closing elements like </B>
                # NB ->processMxpElement returns a flag indicating whether this tag should be
                #   treated as MXP, or ordinary text, but this function can ignore it
                ($recogniseFlag, @miniTagList)
                    = $self->processMxpElement($thisTag, undef, 'simple', %attHash);
                push (@tagList, @miniTagList);
            }

        # </some_element>
        } else {

            # Remove the corresponding stack object from the stack; any tags in the macro expansion
            #   will also get popped, returning text attributes to their previous state
            return $self->popMxpStack($keyword);
        }

        return @tagList;
    }

    sub processMxpEntity {

        # Called by $self->processIncomingData
        #
        # Process an MXP entity in the form %...;, replacing it with the named entity's value
        # For entities in the form '&#nnn;' replaces the nnn with the ASCII character of that value,
        #   but only for value in the range 32-255
        #
        # NB If we're in the middle of a <V>...</V> construction, an entity in the form %...'
        #   doesn't abnormally terminate the construction. Therefore the following construction is
        #   valid:
        #       <V>I have %number; gold coins</V>
        # ...and results in the variable's ->value being set to something 'I have 100 gold coins'
        #
        # Expected arguments
        #   $token      - The token containing the entity in the form &...;
        #   $text       - The remaining portion of text, which begins with the entity $token
        #
        # Return values
        #   'undef' on improper arguments or if the entity is unrecognised
        #   Otherwise returns the entity's value

        my ($self, $token, $text, $check) = @_;

        # Local variables
        my ($enNum, $enName, $entityObj);

        # Check for improper arguments
        if (! defined $token || ! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpEntity', @_);
        }

        # Handle entities in the form '&#nnn;'
        if ($token =~ m/\&\#([0-9]{1,3})\;/) {

            $enNum = $1;
            # We'll assume that 'nnn' can be '064' as well as '64'. Convert it to a numeric value
            $enNum += 0;

            # Ignore numbers not in the range 32-255
            if ($enNum < 32 || $enNum > 255) {

                return undef;

            } else {

                return chr($enNum);
            }
        }

        # Otherwise, get the entity name (the token will have at least three characters, due to the
        #   regex used in $self->extractMxpEntity, so there's no need to check for a minimum length)
        $enName = substr($token, 1, (length($token) - 2));

        # Does an entity called $enName exist?
        if (! $self->ivExists('mxpEntityHash', $enName)) {

            # Standard entity names don't have their own GA::Mxp::Entity object
            if (! $axmud::CLIENT->ivExists('constMxpEntityHash', $enName)) {

                $self->mxpDebug($token, 'Unrecognised entity \'' . $enName . '\'', 3901);

                return undef;

            } else {

                # Use the standard entity's value (an ASCII character)
                return $axmud::CLIENT->ivShow('constMxpEntityHash', $enName);
            }

        } else {

            # Replace the named entity with its value
            $entityObj = $self->ivShow('mxpEntityHash', $enName);
            return $entityObj->value;
        }
    }

    sub processMxpSpacingTag {

        # Called by $self->processIncomingData
        #
        # Process an MXP line spacing token: <NOBR>, <P>, </P>, <BR>, <SBR>
        # Also process one kind of MXP HTML element, <HR>
        #
        # Expected arguments
        #   $origText   - The original text received from the world, before this token was
        #                   extracted
        #   $token      - An extracted token containing the MXP element
        #
        # Return values
        #   'undef' on improper arguments or if an invalid closing tag like </BR> is used
        #   Otherwise, returns a modified $origText. If the token was converted into a newline
        #       character, $origText is set by this function to be an empty string (as happens after
        #       ->processIncomingData calls ->processEndLine directly); otherwise $token is added
        #       to the existing value of $origText

        my ($self, $origText, $token, $check) = @_;

        # Local variables
        my ($origToken, $width, $height, $string, $fontSize);

        # Check for improper arguments
        if (! defined $token || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpSpacingTag', @_);
        }

        # Remove the < and > characters
        $origToken = $token;
        $token = uc($token);
        $token =~ s/^\<//;
        $token =~ s/\>$//;

        # Only the </P> element can be used as a closing tag (e.g. </NOBR> is invalid)
        if ($token ne '/P' && substr($token, 0, 1) eq '/') {

            $self->mxpDebug($origToken, 'Invalid line spacing tag \'' . $origToken . '\'', 4001);
        }

        # Process the tag
        if ($token eq 'BR') {

            # Force a line break inside or outside a paragraph. The TRUE argument means 'don't close
            #   open MXP tags, as we would for a true newline character'
            $self->processEndLine($origText, $origToken, TRUE);
            # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
            #   string, so this function does the same)
            return '';

        } elsif ($token eq 'NOBR') {

            # Ignore the next newline character
            $self->ivPoke('mxpIgnoreNewLineFlag', TRUE);

        } elsif ($token eq 'P') {

            if ($self->mxpParagraphFlag) {

                # In successive <P>...<P> tags, the second <P> tag is treated as though it were
                #   preceded by a closing </P> tag
                # Force a line break. The TRUE argument means 'don't close open MXP tags, as we
                #   would for a true newline character'
                $self->processEndLine($origText, $origToken, TRUE);
                # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
                #   string, so this function does the same)
                return '';

            } else {

                # Ignore all newline characters until the closing </P> tag
                $self->ivPoke('mxpParagraphFlag', TRUE);
            }

        } elsif ($token eq '/P') {

            if (! $self->mxpParagraphFlag) {

                # An invalid </P>...</P> construction
                $self->mxpDebug($origToken, '</P> tag after earlier closing </P> tag', 4002);

            } else {

                # Stop ignoring newline characters
                $self->ivPoke('mxpParagraphFlag', FALSE);

                # Force a line break. The TRUE argument means 'don't close open MXP tags, as we
                #   would for a true newline character'
                $self->processEndLine($origText, $origToken, TRUE);
                # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
                #   string, so this function does the same)
                return '';
            }

        } elsif ($token eq 'SBR') {

            # Axmud doesn't yet implement soft line breaks; treat it as an ordinary space character
            $self->processTextToken(' ');

        } elsif ($token eq 'HR') {

            # An HTML element. Draw a poor man's 'horizontal rule' with simple ASCII characters

            # Force a line break inside or outside a paragraph. The TRUE argument means 'don't close
            #   open MXP tags, as we would for a true newline character'
            $self->processEndLine($origText, $origToken, TRUE);

            # Adjust the width to take account of different font sizes, especially inside headings
            #   (<H1>...</H1>, etc)
            ($width, $height) = $self->getTextViewSize();
            $fontSize = $self->currentTabObj->textViewObj->ivShow('mxpModalStackHash', 'font_size');

            if ($fontSize ne '' && $fontSize != $axmud::CLIENT->constFontSize) {

                # (Subtracting 1 seems to produce the right answer more often, for unknown reasons)
                $width = int($width / ($fontSize / $axmud::CLIENT->constFontSize)) - 1;
            }

            # Draw the horizontal rule
            $self->processTextToken(chr(0x2501) x $width);
            # ...which ends in another newline character
            $self->processEndLine($origText, $origToken, TRUE);

            # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
            #   string, so this function does the same)
            return '';
        }

        return $origText . $origToken;
    }

    sub processMxpHeadingTag {

        # Called by $self->processIncomingData
        #
        # Process an MXP heading token: <H1>...</H1> - <H6>...</H6>
        #
        # Expected arguments
        #   $origText   - The original text received from the world, before this token was
        #                   extracted
        #   $token      - An extracted token containing the MXP element
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origText, $token, $check) = @_;

        # Local variables
        my (
            $flag, $num,
            @emptyList, @tagList,
            %stackHash,
        );

        # Check for improper arguments
        if (! defined $token || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processMxpHeadingTag', @_);
            return @emptyList;
        }

        # In MXP, the format is
        #   <H1>...</H1>, etc
        # In Pueblo, the format is
        #   <H1 align=xxx>...</H1>, etc

        # Extract the forward slash (if present) and the number, removing the diamond brackets and
        #   ignoring any attributes
        if ($token =~ m/\<(\/?)H(\d)/) {

            $flag = $1;
            $num = $2;

        } else {

            # This shouldn't be possible, as the calling function already used a regex
            return @emptyList;
        }

        # <H1>, etc
        if (! $flag) {

            if ($self->mxpHeadingFlag) {

                # In successive <Hx>...<Hx> tags, the second <Hx> tag is treated as though it were
                #   preceded by a closing </Hx> tag (regardless of whether the number x is the
                #   same for both tags)
                # Force a line break. The TRUE argument means 'don't close open MXP tags, as we
                #   would for a true newline character'
                $self->processEndLine($origText, $token, TRUE);

            } else {

                # Ignore all newline characters until the closing </Hx> tag
                $self->ivPoke('mxpHeadingFlag', TRUE);
            }

            # If the client flag is set, the font isn't changed (but the line breaks are used as
            #   normal)
            if ($axmud::CLIENT->allowMxpFontFlag) {

                $stackHash{'font_size'} = $axmud::CLIENT->ivShow('constHeadingSizeHash', $num)
                                            * $axmud::CLIENT->constFontSize;
                $stackHash{'spacing'} = $axmud::CLIENT->ivShow('constHeadingSpacingHash', $num)
                                            * $axmud::CLIENT->constFontSize;
                $stackHash{'font_name'} = $axmud::CLIENT->constFont;
                $stackHash{'bold_flag'} = TRUE;

                # Create a dummy style tag that $self->applyColourStyleTags can interpret
                #   e.g. 'mxpf_monospace_bold_12'
                push (@tagList, $self->createMxpFontTag(%stackHash));

                # Create a new MXP stack object and store it in the current textview object,
                #   updating the latter's IVs
                if (
                    ! $self->currentTabObj->textViewObj->createMxpStackObj(
                        $self,
                        'H' . $num,
                        %stackHash,
                    )
                ) {
                    $self->mxpDebug($token, 'Internal error while processing element', 4101);

                    return @emptyList;
                }
            }

            # Operation complete
            return @tagList;

        # </H1>, etc
        } else {

            if (! $self->mxpHeadingFlag) {

                # An invalid </Hx>...</Hx> construction
                $self->mxpDebug($token, '</Hx> tag after earlier closing </Hx> tag', 4102);

                return @emptyList;
            }

            # Stop ignoring newline characters
            $self->ivPoke('mxpHeadingFlag', FALSE);

            # Force a line break. The TRUE argument means 'don't close open MXP tags, as we
            #   would for a true newline character'
            $self->processEndLine($origText, $token, TRUE);

            # If the client flag is set, restore the previous font
            if ($axmud::CLIENT->allowMxpFontFlag) {

                return $self->popMxpStack('H' . $num);

            } else {

                return @emptyList;
            }
        }
    }

    sub processMspSoundTrigger {

        # Called by $self->processIncomingData when it encounters an MSP sound trigger, in the form
        #   "!!SOUND(...)" or "!!MUSIC(...)"
        # (Also called by $self->processMxpSoundElement when processing MXP sounds)
        #
        # Processes the MSP sound trigger, producing a list of parameters which are passed to
        #   GA::Client->playSoundFile for playing
        #
        # Expected arguments
        #   $token      - An extracted token containing the MSP sound trigger
        #
        # Optional arguments
        #   $mxpFlag    - Set to TRUE for MXP <SOUND> and <MUSIC> tags, to which MXP file filters
        #                   can be applied; set to FALSE (or 'undef') for MSP sound triggers
        #
        # Return values
        #   'undef' on improper arguments or if the token is invalid, and should be displayed as
        #       normal text
        #   1 otherwise

        my ($self, $token, $mxpFlag, $check) = @_;

        # Local variables
        my (
            $paramString, $type, $fName, $v, $l, $p, $c, $t, $u, $path, $file, $dir, $ext,
            $fetchObj, $string, $urlRegex, $convertFlag,
            @list, @objList,@fileList,
        );

        # Check for improper arguments
        if (! defined $token || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processMspSoundTrigger', @_);
        }

        # Extract the parameter string (everything between the opening and closing brackets)
        if ($token =~ m/\!\!SOUND\((.*)\)/) {

            $paramString = $1;
            $type = 'sound';

        } elsif ($token =~ m/\!\!MUSIC\((.*)\)/) {

            $paramString = $1;
            $type = 'music';

        } else {

            # Invalid MSP sound token
            return undef;
        }

        if (! $paramString) {

            # Invalid MSP sound token (empty parameter string is not allowed - first parameter is
            #   compulsory)
            return undef;
        }

        # Extract the individual parameters (which are separated by one or more spaces, but do not
        #   contain spaces)
        @list = split(/\s+/, $paramString);

        # The first parameter (file name) is compulsory
        $fName = shift(@list);

        # FName=OFF is a special case
        if ($fName eq 'Off') {

            # The file name 'Off' is reserved; must playing sound or music triggers, if no other
            #   parameters are received
            if (! @list) {

                @objList = $self->ivValues('soundHarnessHash');
                foreach my $soundObj (@objList) {

                    if ($type eq $soundObj->type) {

                        # Stop this sound
                        $soundObj->stop();
                        # ...and update the registry
                        $self->ivDelete('soundHarnessHash', $soundObj->number);
                    }
                }

                return 1;

            # Otherwise, we expect @list to be in the form (U=some_url) which sets a default
            #   download URL
            } else {

                $u = shift @list;

                if (substr($u, 0, 2) ne 'U=') {

                    # Invalid parameter list - play nothing
                    return 1;
                }

                $u = substr($u, 2);

                $urlRegex = $axmud::CLIENT->constUrlRegex;
                if (! ($u =~ m/$urlRegex/)) {

                    # Invalid URL - play nothing
                    return 1;

                } else {

                    # The code below is a little simpler, if the partial URL ends with a slash
                    if (substr($u, -1, 1) ne '/') {

                        $u .= '/';
                    }

                    # Set the default MSP download URL, and play no sound
                    $self->ivPoke('mspDefaultURL', $u);

                    # Nothing to play, this time
                    return 1;
                }
            }
        }

        # Remaining parameters are in the form <param-name>=<param-value>, where <param-name> is a
        #   one-letter upper-case identifier (A-Z)
        # If any parameters are not specified, use default values
        $v = 100;
        $l = 1;
        $p = 50;
        $c = 1;
        $t = '';

        if ($self->mspDefaultURL) {
            $u = $self->mspDefaultURL;
        } else {
            $u = '';
        }

        # Extract parameter values, and check they're valid values
        foreach my $item (@list) {

            my ($name, $value);

            if ($item =~ m/(.*)\=(.*)/) {

                $name = $1;
                $value = $2;

                if ($name eq '' || $value eq '') {

                    # Invalid parameter format - play nothing
                    return 1;

                } elsif ($type eq 'sound' && $name eq 'C') {

                    # C parameters only used with !!MUSIC - play nothing
                    return 1;

                } elsif ($type eq 'music' && $name eq 'P') {

                    # P parameters only used with !!SOUND - play nothing
                    return 1;

                } elsif ($name eq 'V') {

                    # (If the value is invalid, $v remains set to its default value. This is also
                    #   true for $l, $p and $c)
                    if (! $axmud::CLIENT->floatCheck($value, 0, 100)) {

                        $v = $value;
                    }

                } elsif ($name eq 'L') {

                    if (! $axmud::CLIENT->intCheck($value, 1) && $value ne '-1') {

                        $l = $value;
                    }

                } elsif ($name eq 'P') {

                    if (! $axmud::CLIENT->floatCheck($value, 0, 100)) {

                        $p = $value;
                    }

                } elsif ($name eq 'C') {

                    if ($value eq '0' || $value eq '1') {

                        $c = $value;
                    }

                } elsif ($name eq 'T') {

                    $t = $value;

                } elsif ($name eq 'U') {

                    $u = $value;

                } else {

                    # Invalid parameter name - play nothing
                    return 1;
                }

            } else {

                # Invalid parameter - play nothing
                return 1;
            }
        }

        # If no file extension is specified, provide one
        ($file, $dir, $ext) = File::Basename::fileparse($fName, qr/\.[^.]*/);
        if (! $ext) {

            if ($type eq 'sound') {
                $fName .= '.wav';
            } else {
                $fName .= '.mid';
            }
        }

        # Convert $fName into a full filepath ($fName must a relative filepath, pointing at a
        #   directory used to store sounds for the current world)
        if ($t) {

            $path = $axmud::DATA_DIR . '/msp/' . $self->currentWorld->name
                        . '/' . $t . '/' . $fName;

        } else {

            $path = $axmud::DATA_DIR . '/msp/' . $self->currentWorld->name . '/' . $fName;
        }

        # If this file actually exists, use it. Otherwise...
        if (! -e $path) {

            # If a partial URL was specified...
            if ($u) {

                # If we're not allowed to download, then there's nothing to play
                if (
                    (! $mxpFlag && ! $axmud::CLIENT->allowMspLoadSoundFlag)
                    || ($mxpFlag && ! $axmud::CLIENT->allowMxpLoadSoundFlag)
                ) {
                    # Play nothing
                    return 1;
                }

                # Compile the full URL from which to download
                if (substr($u, -1, 1) ne '/') {

                    $u .= '/';
                }

                $u .= $fName;

                # Attempt to download the file
                ($file, $dir) = File::Basename::fileparse($path);
                $fetchObj = File::Fetch->new(uri => $u);
                if (! $fetchObj->fetch(to => $dir)) {

                    # Download error - play nothing
                    return 1;
                }

            # Otherwise, if the file doesn't exist, $path may contain wildcards (the MSP spec
            #   specifies * and ? specifically). Get a list of matching files, and choose a random
            #   one
            } else {

                ($file, $dir) = File::Basename::fileparse($path);

                # Convert MSP wildcards into regex equivalents (but only for the file itself, not
                #   its containing directory)
                # * must match the remainder of the file name
                $file =~ s/\*.*/.*/;
                # ? must match exactly one character
                $file =~ s/\?/.?/;
                $path = $dir . $file;

                # Get a list of matching files
                @fileList = grep {/$path/} glob "$dir*";
                if (! @fileList) {

                    # No matching file found. If the server specified a relative filepath (e.g.
                    #   'zone231/room22.wav', and that file (the equivalent of
                    #   /home/.../axmud-data/msp/world_name/zone231/room22.wav) doesn't exist, we
                    #   can strip away the subdirectory and look in the main directory instead
                    #   (i.e., in /home/.../axmud-data/msp/world_name/room22.wav)
                    $path = $axmud::DATA_DIR . '/msp/' . $self->currentWorld->name . '/' . $file;
                    $dir = $axmud::DATA_DIR . '/msp/' . $self->currentWorld->name . '/';
                    @fileList = grep {/$path/} glob "$dir*";
                }

                if (@fileList == 1) {

                    $path = $fileList[0];

                } else {

                    $path = $fileList[rand(scalar @fileList)];
                }

                if (! $path) {

                    # No matching sound file - play nothing
                    return 1;
                }
            }
        }

        # For sound files, Axmud only supports file extensions in GA::Client->constSoundFormatHash
        ($file, $dir, $ext) = File::Basename::fileparse($path, qr/\.[^.]*/);
        $ext =~ s/^\.//;
        if (! $axmud::CLIENT->ivExists('constSoundFormatHash', $ext)) {

            if (! $mxpFlag) {

                # File format not supported
                return 1;

            } else {

                # For any other file extension, apply the MXP filter (if the world has specified
                #   one)
                $path = $self->applyMxpFileFilter($path);
                if (! $path) {

                    # No MXP file filter supplied, or file conversion failed
                    return 1;
                }

                # Check the file format of the converted file is supported
                ($file, $dir, $ext) = File::Basename::fileparse($path, qr/\.[^.]*/);
                $ext =~ s/^\.//;
                if (! $axmud::CLIENT->ivExists('constSoundFormatHash', $ext)) {

                    # File format not supported
                    return 1;

                } else {

                    $convertFlag = TRUE;
                }
            }
        }

        # We can only play multiple sound triggers concurrently, if the flag is set
        # We can never play more than one music trigger concurrently
        @objList = $self->ivValues('soundHarnessHash');
        foreach my $soundObj (@objList) {

            if (
                (
                    # Concurrent sound triggers not allowed (at the moment)
                    ! $axmud::CLIENT->allowMspMultipleFlag
                    && $type eq 'sound'
                    && $soundObj->type eq 'sound'
                ) || (
                    # Concurrent music triggers not allowed (ever)
                    $type eq 'music'
                    && $soundObj->type eq 'music'
                )
            ) {
                # Stop this sound
                $soundObj->stop();
                # ...and update the registry
                $self->ivDelete('soundHarnessHash', $soundObj->number);
            }
        }

        # Apply the sound priority (but not for music triggers)
        if ($type eq 'sound') {

            @objList = $self->ivValues('soundHarnessHash');
            foreach my $soundObj (@objList) {

                if ($soundObj->type eq 'sound' && $soundObj->priority < $p) {

                    # This sound has a lower priority, so stop it
                    $soundObj->stop();
                    # ...and update the registry
                    $self->ivDelete('soundHarnessHash', $soundObj->number);
                }
            }
        }

        # Apply the continue flag
        if ($type eq 'music') {

            @objList = $self->ivValues('soundHarnessHash');
            foreach my $soundObj (@objList) {

                if ($soundObj->type eq 'music' && $soundObj->path eq $path) {

                    # This music trigger is already playing

                    # C=1
                    if ($c) {

                        # Allow the existing music trigger to continue playing, but modify its
                        #   repeat count
                        if ($soundObj->repeat != -1) {

                            $soundObj->ivPoke('repeat', $soundObj->repeat + $l);
                        }

                        return 1;

                    # C=0
                    } else {

                        # Restart the music trigger. In fact, stop the current music trigger, and
                        #   let the new one start playing in a moment, using the same file.  (The
                        #   TRUE argument means 'don't delete the sound file itself')
                        $soundObj->stop(TRUE);
                        # ...and update the registry
                        $self->ivDelete('soundHarnessHash', $soundObj->number);

                        # Combine the repeat count for the two duplicate sounds
                        if ($soundObj->repeat == -1 || $l == -1) {

                            $l = -1;

                        } else {

                            $l = $l + $soundObj->repeat;
                        }
                    }
                }
            }
        }

        # Play the sound file, passing any optional parameters
        $axmud::CLIENT->playSoundFile(
            $self,
            $path,
            $convertFlag,
            $type,
            $v,
            $l,
            $p,
            $c,
        );

        return 1;
    }

    sub processPuebloElement {

        # Called by $self->processIncomingData when it encounters a Pueblo element (e.g. '<B>')
        #
        # Processes the Pueblo element, updating IVs and returning a corresponding list of Axmud
        #   colour/style tags, where necessary
        #
        # Expected arguments
        #   $token      - An extracted token containing the Pueblo element
        #   $origText   - Specified when called by $self->processIncomingData, representing the
        #                   received line of text, up to (but not including) the Pueblo element
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns a recognition flag (set to FALSE if the token is nothing to do with
        #       Pueblo which should be processed as ordinary text; set to TRUE if it's a Pueblo
        #       element, even an invalid one) followed by an equivalent list of Axmud colour/style
        #       tags otherwise (may be an empty list)

        my ($self, $token, $origText, $check) = @_;

        # Local variables
        my (
            $origToken, $firstChar, $tagMode, $keyword,
            @emptyList, @argList,
        );

        # Check for improper arguments
        if (! defined $token) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloElement', @_);
            return @emptyList;
        }

        print "20165 $token\n";

        # ($token will be modified during this function, but some parts of the function require the
        #   the original token text)
        $origToken = $token;

        # Ignore comments, in the form '<!-- this is a comment -->' ($self->extractMxpElement has
        #   already checked that an element beginning with '<!--' ends with a '-->', so we only need
        #   to check the first part of the string
        if (substr($token, 0, 4) eq '<!--') {

            # Recognition flag, followed by empty Axmud colour/style tag list
            return TRUE;
        }

        # Pueblo elements are in the form:
        #   <keyword [args]>            ($tagMode 'open')
        #   </keyword>                  ($tagMode 'close')

        # Remove the initial < followed by optional whitespace, and the final > preceded by optional
        #   whitespace
        $token =~ s/^\<\s*//;
        $token =~ s/\s*\>$//;
        # In case there's nothing left, don't bother looking for keywords or arguments...
        if (! $token) {

            $self->puebloDebug($origToken, 'Processed an empty element', 6601);

            # Treat token as ordinary text
            return @emptyList;
        }

        # Remove the initial /, if present
        $firstChar = substr($token, 0, 1);
        if ($firstChar eq '/') {

            # </keyword>
            $token = substr($token, 1);
            $tagMode = 'close';

        } else {

            # <keyword [args]>
            $tagMode = 'open';
        }

        # Remove the keyword, assuming that the same rules apply that apply to MXP keyword ('must
        #   start with a letter (A-Z) and then consist of letters, numbers or the underline
        #   character'
        if ($token =~ m/^([A-Za-z][A-Za-z0-9_]*)/) {

            # (Simplify things by converting all keywords to upper-case)
            $keyword = uc($1);
            $token = substr($token, length($keyword));

        } else {

            if ($token =~ m/^[\'\"].*[\'\"]$/) {

                # Keyword not found
                $self->puebloDebug($origToken, 'Element contains quoted keyword', 6611);

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;

            } else {

                # Keyword not found
                $self->puebloDebug($origToken, 'Element contains invalid keyword', 6612);

                # Recognition flag, followed by empty Axmud colour/style tag list
                return TRUE;
            }
        }

        # Some keywords are synonyms of others, e.g. <B>, <EM> and <STRONG> are all equivalent
        if ($axmud::CLIENT->ivExists('constPuebloConvertHash', $keyword)) {

            $keyword = $axmud::CLIENT->ivShow('constPuebloConvertHash', $keyword);
        }

        # Check that the keyword is an official Pueblo element, whether implemented or not
        if (! $axmud::CLIENT->ivExists('constPuebloOfficialHash', $keyword)) {

            # Treat token as ordinary text
            return @emptyList;

        # Ignore unimplemented keywords
        } elsif (! $axmud::CLIENT->ivExists('constPuebloImplementHash', $keyword)) {

            # Recognition flag, followed by empty Axmud colour/style tag list
            return TRUE;
        }

        # Get a list of arguments, separated by one or more whitespace characters
        # (The whole argument can be single-quoted, or double-quoted, in which case it can contain
        #   embedded whitespace or the '>' character)
        if ($token) {

            do {

                my ($argName, $argValue, $key, $value);

                ($token, $argName, $argValue) = $self->extractMxpArgument($token);
                if (! defined $token) {

                    # Improper arguments, or malformed argument
                    $self->puebloDebug($origToken, 'Malformed element', 6621);

                    # Recognition flag, followed by empty Axmud colour/style tag list
                    return TRUE;

                } else {

                    # If the argument is not in the form 'argument_name=argument_value', then
                    #   $argValue is 'undef', and $argName contains the whole argument
                    push (@argList, $argName, $argValue);

                    # After removing the 'argument' or the 'argument_name=argument_value'
                    #   construction, if there's anything left in $token, it must start with a
                    #   whitespace character
                    # This ensures that there are whitespace character(s) between each argument,
                    #   and prevents constructions like: name='value'name='value'
                    if ($token && $token =~ m/^\S/) {

                        $self->puebloDebug($origToken, 'Malformed element', 6622);

                        # Recognition flag, followed by empty Axmud colour/style tag list
                        return TRUE;
                    }
                }

            } until (! $token);
        }

        # Process each type of Pueblo element in its own function

        # Process modal elements: <B> <I> <U> <STRIKE>
        if ($axmud::CLIENT->ivExists('constPuebloModalHash', $keyword)) {

            return (TRUE, $self->processMxpModalElement($origToken, $tagMode, $keyword, @argList));

        # Process HTML element: <TT>
        } elsif ($keyword eq 'TT') {

            return (TRUE, $self->processMxpHtmlElement($origToken, $tagMode, $keyword, @argList));

        # Process colours: <COLOR>...</COLOR>
        } elsif ($keyword eq 'COLOR') {

            return (
                TRUE,
                $self->processPuebloColourElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process fonts: <FONT>...</FONT>
        } elsif ($keyword eq 'FONT') {

            return (
                TRUE,
                $self->processPuebloFontElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process base font: <BASEFONT>
        } elsif ($keyword eq 'BASEFONT') {

            return (
                TRUE,
                $self->processPuebloBaseFontElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process hyperlinks to send to MUD: <SEND>...</SEND>
        } elsif ($keyword eq 'SEND') {

            # Not in Pueblo 2.50 spec, so assume it's the same as the MXP tag
            return (
                TRUE,
                $self->processMxpSendElement($origToken, $tagMode, $keyword, FALSE, @argList),
            );

        # Process direct setting of clickable links: <A>...</A>
        } elsif ($keyword eq 'A') {

            return (
                TRUE,
                $self->processPuebloLinkElement($origToken, $tagMode, $keyword, FALSE, @argList),
            );

        # Process list: <UL>...</UL>, <OL>...</OL>
        } elsif ($keyword eq 'UL' || $keyword eq 'OL') {

            return (
                TRUE,
                $self->processPuebloListElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process list items: <LI>
        } elsif ($keyword eq 'LI') {

            return (
                TRUE,
                $self->processPuebloListItemElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process text justification: <CENTER>
        } elsif ($keyword eq 'CENTER') {

            return (
                TRUE,
                $self->processPuebloJustifyElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process literal text: <CODE>...</CODE>, <PRE>...</PRE>, <SAMP>...</SAMP>
        } elsif ($keyword eq 'CODE' || $keyword eq 'PRE' || $keyword eq 'SAMP') {

            return (
                TRUE,
                $self->processPuebloLiteralElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process images: <IMG>
        } elsif ($keyword eq 'IMG') {

            return (
                TRUE,
                $self->processPuebloImageElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process panes: <XCH_PANE>
        } elsif ($keyword eq 'XCH_PANE') {

            return (
                TRUE,
                $self->processPuebloPaneElement($origToken, $tagMode, $keyword, @argList),
            );

        # Process other implemented MXP elements
        } elsif ($axmud::CLIENT->ivExists('constPuebloImplementHash', $keyword)) {

            return (
                TRUE,
                $self->processPuebloImplementedElement($origToken, $tagMode, $keyword, @argList),
            );

        # (This should never be executed)
        } else {

            $self->puebloDebug($origToken, 'Internal error while processing element', 6631);

            # Recognition flag, followed by empty Axmud colour/style tag list
            return TRUE;
        }
    }

    sub processPuebloColourElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo font element: <COLOR>...</COLOUR>
        # (NB Specified at http://www.gammon.com.au/forum/?id=281, but not specified in the spec for
        #   Pueblo 2.50)
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            @emptyList, @tagList,
            %ivHash, %stackHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloColourElement', @_);
            return @emptyList;
        }

        # <COLOR [FORE=foreground] [BACK=background]>
        # </COLOR>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
        ) {
            $self->puebloDebug($origToken, 'Malformed element', 6701);

            return @emptyList;

        # <COLOR [FORE=foreground] [BACK=background]>
        } elsif ($tagMode eq 'open') {

            # Process @argList

            # Default argument values
            %ivHash = (
                'fore'      => undef,
                'back'      => undef,
            );

            if (@argList) {

                do {

                    my $argName = lc(shift @argList);
                    my $argValue = shift @argList;

                    if (! defined $argName || ! exists $ivHash{$argName}) {

                        # Unrecognised argument name
                        $self->puebloDebug($origToken, 'Malformed element', 6711);

                        return @emptyList;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check any specified argument values are valid. In some cases, give up; in other cases,
            #   use default values
            if (defined $ivHash{'fore'} && ! ($ivHash{'fore'} =~ m/^#[A-Fa-f0-9]{6}$/)) {

                # Invalid foreground colour
                $self->puebloDebug($origToken, 'Invalid foreground colour', 6721);

                return @emptyList;

            } elsif (defined $ivHash{'back'} && ! ($ivHash{'back'} =~ m/^#[A-Fa-f0-9]{6}$/)) {

                # Invalid background colour
                $self->puebloDebug($origToken, 'Invalid background colour', 6722);

                return @emptyList;
            }

            # Prepare the colour, using a hash containing a subset of key-value pairs from
            #   GA::Obj::TextView->mxpModalStackHash, and containing only changes specified by
            #   this Pueblo tag
            if ($ivHash{'fore'}) {

                $stackHash{'colour_foreground'} = $ivHash{'fore'};
            }

            if ($ivHash{'back'}) {

                $stackHash{'colour_background'} = $ivHash{'back'};
            }

            # Create a new MXP stack object and store it in the current textview object, updating
            #   the latter's IVs
            if (
                ! $self->currentTabObj->textViewObj->createMxpStackObj($self, $keyword, %stackHash)
            ) {
                $self->puebloDebug($origToken, 'Internal error while processing element', 6723);

                return @emptyList;

            } else {

                # Operation complete
                return @tagList;
            }

        # </COLOR>
        } else {

            # Cancel the MXP text attribute
            return $self->popMxpStack($keyword);
        }
    }

    sub processPuebloFontElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo font element: <FONT>...</FONT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $token, $sign, $num,
            @emptyList, @tagList,
            %ivHash, %stackHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloFontElement', @_);
            return @emptyList;
        }

        # <FONT [SIZE=[+|-]size] [FGCOLOR=foreground] [COLOR=foreground] [TEXT=foreground]
        #   [BGCOLOR=background]>
        # </FONT>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
        ) {
            $self->puebloDebug($origToken, 'Malformed element', 6801);

            return @emptyList;

        # <FONT [SIZE=[+|-]size] [FGCOLOR=foreground] [COLOR=foreground] [TEXT=foreground]
        #   [BGCOLOR=background]>
        } elsif ($tagMode eq 'open') {

            # Process @argList

            # Default argument values
            %ivHash = (
                'size'      => $self->puebloBaseFontSize,
                'fgcolor'   => undef,
                'color'     => undef,
                'text'      => undef,
                'bgcolor'   => undef,
            );

            if (@argList) {

                do {

                    my $argName = lc(shift @argList);
                    my $argValue = shift @argList;

                    if (! defined $argName || ! exists $ivHash{$argName}) {

                        # Unrecognised argument name
                        $self->puebloDebug($origToken, 'Malformed element', 6811);

                        return @emptyList;

                    } elsif ($argName eq 'color' || $argName eq 'text') {

                        $ivHash{'fgcolor'} = $argValue;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check any specified argument values are valid. In some cases, give up; in other cases,
            #   use default values
            if ($ivHash{'size'} =~ m/^([\+\-]?)([1-7])$/) {

                $sign = $1;
                $num = $2;

                if ($sign) {

                    if ($sign eq '+') {
                        $ivHash{'size'} = $ivHash{'size'} + $num;
                    } else {
                        $ivHash{'size'} = $ivHash{'size'} - $num;
                    }

                    # Permitted size range is 1-7
                    if ($ivHash{'size'} < 1) {
                        $ivHash{'size'} = 1;
                    } elsif ($ivHash{'size'} > 7) {
                        $ivHash{'size'} = 7;
                    }
                }

            } else {

                # Invalid font size
                $self->puebloDebug($origToken, 'Invalid font size', 6821);
            }

            if (defined $ivHash{'fgcolor'} && ! ($ivHash{'fgcolor'} =~ m/^#[A-Fa-f0-9]{6}$/)) {

                # Invalid foreground colour
                $self->puebloDebug($origToken, 'Invalid foreground colour', 6822);

                return @emptyList;

            } elsif (defined $ivHash{'bgcolor'} && ! ($ivHash{'bgcolor'} =~ m/^#[A-Fa-f0-9]{6}$/)) {

                # Invalid background colour
                $self->puebloDebug($origToken, 'Invalid background colour', 6823);

                return @emptyList;
            }

            # Prepare the font, using a hash containing a subset of key-value pairs from
            #   GA::Obj::TextView->mxpModalStackHash, and containing only changes specified by this
            #   Pueblo tag
            # Use similar sizes/spacings used by GA::Client->constHeadingSizeHash and
            #   ->constHeadingSpacingHash. I don't know which sizes web browsers actually use
            if ($ivHash{'size'} == 1) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 0.67;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize * 1.67;

            } elsif ($ivHash{'size'} == 2) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 0.83;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize * 1.67;

            } elsif ($ivHash{'size'} == 3) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize * 1.33;

            } elsif ($ivHash{'size'} == 4) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 1.17;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize;

            } elsif ($ivHash{'size'} == 5) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 1.33;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize;

            } elsif ($ivHash{'size'} == 6) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 1.5;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize * 0.83;

            } elsif ($ivHash{'size'} == 7) {

                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize * 2;
                $stackHash{'spacing'} = $axmud::CLIENT->constFontSize * 0.67;
            }

            if ($ivHash{'fgcolor'}) {

                $stackHash{'colour_foreground'} = $ivHash{'fgcolor'};
            }

            if ($ivHash{'bgcolor'}) {

                $stackHash{'colour_background'} = $ivHash{'bgcolor'};
            }

            # Create a dummy style tag that $self->applyColouStyleTags can interpret
            #   e.g. 'mxpf_monospace_bold_12'
            push (@tagList, $self->createMxpFontTag(%stackHash));

            # Create a new MXP stack object and store it in the current textview object, updating
            #   the latter's IVs
            if (
                ! $self->currentTabObj->textViewObj->createMxpStackObj($self, $keyword, %stackHash)
            ) {
                $self->puebloDebug($origToken, 'Internal error while processing element', 6824);

                return @emptyList;

            } else {

                # Operation complete
                return @tagList;
            }

        # </FONT>
        } else {

            # Cancel the MXP text attribute
            return $self->popMxpStack($keyword);
        }
    }

    sub processPuebloBaseFontElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo base font element: <BASEFONT>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            @emptyList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloBaseFontElement', @_);
            return @emptyList;
        }

        # <BASEFONT [SIZE=[+|-]size]>
        if (
            ($tagMode eq 'open' && ! @argList)
            || $tagMode eq 'close'
        ) {
            $self->puebloDebug($origToken, 'Malformed element', 6901);

            return @emptyList;
        }

        # <BASEFONT [SIZE=[+|-]size]>

        # Process @argList

        # Default argument values
        %ivHash = (
            'size'      => $self->puebloBaseFontSize,
        );

        if (@argList) {

            do {

                my $argName = lc(shift @argList);
                my $argValue = shift @argList;

                if (! defined $argName || ! exists $ivHash{$argName}) {

                    # Unrecognised argument name
                    $self->puebloDebug($origToken, 'Malformed element', 6911);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check any specified argument values are valid. In some cases, give up; in other cases,
        #   use default values
        if (! $ivHash{'size'} =~ m/^[1-7]$/) {

            # Invalid font size
            $self->puebloDebug($origToken, 'Invalid base font size', 6921);

        } else {

            # Update IVs
            $self->ivPoke('puebloBaseFontSize', $ivHash{'size'});

            return @emptyList;
        }
    }

    sub processPuebloLinkElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo font element: <A>...</A>
        # Partially implemented. <A HREF=...> and <A XCH_CMD=...> are implemented, but (for example)
        #   <A NAME=...> is not
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #   $noPopFlag  - Set to TRUE when called by $self->popMxpStack, in which case we don't
        #                   need to call ->popMxpStack again. Set to FALSE otherwise
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, $noPopupFlag, @argList) = @_;

        # Local variables
        my (
            $textViewObj, $type, $offset, $line, $linkObj,
            @emptyList,
            %ivHash,
        );

        # Check for improper arguments
        if (
            ! defined $origToken || ! defined $tagMode || ! defined $keyword
            || ! defined $noPopupFlag
        ) {
            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloLinkElement', @_);
            return @emptyList;
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # <A [HREF=href] [XCH_CMD=command] [XCH_HINT=hint]
        # </A>
        if (
            ($tagMode eq 'open' && ! @argList)
            || ($tagMode eq 'close' && @argList)
        ) {
            $self->puebloDebug($origToken, 'Malformed element', 7001);

            return @emptyList;

        # <A [HREF=href] [XCH_CMD=command] [XCH_HINT=hint]
        } elsif ($tagMode eq 'open') {

            # Process @argList

            # Default argument values
            %ivHash = (
                'href'      => undef,
                'name'      => undef,   # not implemented
                'title'     => undef,   # not implemented
                'rel'       => undef,   # not implemented
                'rev'       => undef,   # not implemented
                'urn'       => undef,   # not implemented
                'methods'   => undef,   # not implemented
                'xch_cmd'   => undef,
                'xch_hint'  => undef,
            );

            if (@argList) {

                do {

                    my $argName = lc(shift @argList);
                    my $argValue = shift @argList;

                    if (! defined $argName || ! exists $ivHash{$argName}) {

                        # Unrecognised argument name
                        $self->puebloDebug($origToken, 'Malformed element', 7011);

                        return @emptyList;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check any specified argument values are valid. In some cases, give up; in other cases,
            #   use default values
            if (
                (! $ivHash{'href'} && ! $ivHash{'xch_cmd'})
                || ($ivHash{'href'} && $ivHash{'xch_cmd'})
            ) {
                # Missing or multiple hyperlinks
                $self->puebloDebug($origToken, 'Malformed element', 7021);
            }

            # Pueblo <A HREF=...> is implemented like MXP <A>. Pueblo <A XCH_CMD=...> is implemented
            #   like MXP <SEND>.
            if ($ivHash{'href'}) {

                # The // appears to be optional in Pueblo
                if (lc($ivHash{'href'}) =~ m/telnet\:/) {
                    $type = 'telnet';
                } elsif (lc($ivHash{'href'}) =~ m/mailto\:/) {
                    $type = 'mail';
                } else {

                    # Otherwise assume it's a hyperlink in the form '[http[s]://]website.com'
                    $type = 'www';
                }

            } else {

                $type = 'cmd';
            }

            # We can't process the link until the closing </A> tag is found, so store arguments
            #   temporarily in a GA::Obj::Link object
            ($line, $offset) = $textViewObj->getInsertPosn();
            if (! defined $line) {

                $self->puebloDebug($origToken, 'Internal error while processing hyperlink', 7022);

                return @emptyList;
            }

            # The position of the link will be the textview's current insert position, plus the
            #   length of any already-processed text, that hasn't been displayed in the textview
            #   object yet, which has been stored for us in ->mxpOrigText
            # (The -1 argument means this is an incomplete link object, not yet applied to the
            #   current textview and not yet stored in the textview object's registries)
            if (defined $self->mxpOrigText) {

                $offset += length($self->recvLineText);
            }

            $linkObj = Games::Axmud::Obj::Link->new(
                -1,
                $textViewObj,
                $line,
                $offset,
                $type,
            );

            if (! $linkObj) {

                $self->puebloDebug($origToken, 'Internal error while processing hyperlink', 7023);

                return @emptyList;
            }

            # Set the link object's IVs
            if ($ivHash{'href'}) {

                $linkObj->ivPoke('href', $ivHash{'href'});
                $linkObj->ivPoke('puebloMode', 'href_link');

            } else {

                $linkObj->ivPoke('href', $ivHash{'xch_cmd'});
                $linkObj->ivPoke('hint', $ivHash{'xch_hint'});
                $linkObj->ivPoke('puebloMode', 'cmd_link');
            }

            # Store the current link object (the link isn't applied to the textview until the
            #   corresponding </A> tag is processed, although any text inside the <A>...</A> is
            #   displayed immediately, as normal)
            $self->ivPoke('mxpCurrentLink', $linkObj);

            # Also need to create a new MXP stack object and store it in the current textview
            #   object, updating the latter's IVs
            if (! $textViewObj->createMxpStackObj($self, 'A')) {

                $self->puebloDebug($origToken, 'Internal error while processing element', 7024);

                return @emptyList;

            } else {

                # Operation complete
                return @emptyList;
            }

        # </A>
        } elsif ($tagMode eq 'close') {

            # Watch out for an <A> ... </A> ... </A> construction
            if (! $self->mxpCurrentLink) {

#                # This a second </A> tag after an earlier </A> tag
#                $self->puebloDebug($origToken, '</A> tag does not match earlier <A> tag', 7031);

                # For some reason I can't explain, many Pueblo-enhanced worlds are spamming </A>
                #   tags; so rather than display an error message, simply ignore the tag

                return @emptyList;

            } else {

                $linkObj = $self->mxpCurrentLink;
                $self->ivUndef('mxpCurrentLink');
            }

            if (! $linkObj->text) {

#                # There were no valid text tokens between the <A>...</A> tags
#                $self->puebloDebug($origToken, 'Invalid <A>...</A> construction', 7032);

                # Same complaint as just above; ignore this tag, too

                return @emptyList;

            } else {

                # The incomplete link object is now complete, but we can't apply the link to the
                #   current textview yet, as there may be some text before the link which hasn't
                #   been displayed yet
                # Instead, store it temporarily in an IV, and let $self->processIncomingData call
                #   GA::Obj::TextView->add_incompleteLink as soon as it's ready
                $self->ivPush('mxpTempLinkList', $linkObj);
            }

            # Close the <A>..</A> construction, if the calling function isn't doing that (the TRUE
            #   argument means don't call this function back, as you normally would)
            if (! $noPopupFlag) {

               return $self->popMxpStack($keyword, TRUE);

            } else {

                # Operation complete
                return @emptyList;
            }
        }
    }

    sub processPuebloListElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo list element: <UL>...</UL>, <OL>...</OL>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $start, $type, $listObj,
            @emptyList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloListElement', @_);
            return @emptyList;
        }

        # <UL [TYPE=disc|circle|square]>
        # </UL>
        # <OL [START=number] [TYPE=A|a|I|i|1]>
        # </OL>
        if ($tagMode eq 'close' && @argList) {

            $self->puebloDebug($origToken, 'Malformed element', 7101);

            return @emptyList;

        # <UL [TYPE=disc|circle|square]>
        # <OL [START=number] [TYPE=A|a|I|i|1]>
        } elsif ($tagMode eq 'open') {

            # Process @argList

            # Default argument values
            if ($keyword eq 'UL') {

                %ivHash = (
                    'type'      => 'disc',
                );

            } else {

                %ivHash = (
                    'compact'   => undef,       # not implemented
                    'start'     => 1,
                    'type'      => 1,
                );
            }

            if (@argList) {

                do {

                    my $argName = lc(shift @argList);
                    my $argValue = shift @argList;

                    if (! defined $argName || ! exists $ivHash{$argName}) {

                        # Unrecognised argument name
                        $self->puebloDebug($origToken, 'Malformed element', 7111);

                        return @emptyList;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check any specified argument values are valid. In some cases, give up; in other cases,
            #   use default values
            $start = $ivHash{'start'};
            # Assume that the specified number must be a positive integer
            if (defined $start && ! $axmud::CLIENT->intCheck($start, 1)) {

                # Invalid start number
                $self->puebloDebug($origToken, 'Invalid start number in list', 7121);

                return @emptyList;
            }

            $type = $ivHash{'type'};
            if (
                defined $type
                && (
                    (
                        $keyword eq 'UL' && $type ne 'disc' && $type ne 'circle'
                        && $type ne 'square'
                    ) || (
                        $keyword eq 'OL' && $type ne 'A' && $type ne 'a' && $type ne 'I'
                        && $type ne 'i' && $type ne '1'
                    )
                )
            ) {
                # Invalid type
                $self->puebloDebug($origToken, 'Invalid type in list', 7122);

                return @emptyList;
            }

            # Create an object to store details from the list
            $listObj = Games::Axmud::Pueblo::List->new($self, lc($keyword));
            if (! $listObj) {

                $self->puebloDebug($origToken, 'Internal error while processing element', 7123);

                return @emptyList;
            }

            # Add it to a stack of such objects (we don't use $self->mxpModalStackList, as we do for
            #   some similar tags)
            $self->ivPush('puebloStackList', $listObj);

            # Update remaining IVs
            if ($keyword eq 'UL') {

                $listObj->ivPoke('bulletType', $type);

            } else {

                $listObj->ivPoke('itemCount', $start);
                $listObj->ivPoke('itemType', $type);
            }

            # Operation complete
            return @emptyList;

        # </UL>
        # </OL>
        } else {

            # (If the stack is empty, do nothing)
            if ($self->puebloStackList) {

                $self->ivShift('puebloStackList');

                # Insert a newline character into the received text that's being processed by
                #   $self->processIncomingData
                $self->ivPoke(
                    'puebloInsertString',
                    $self->puebloInsertString . "\n",
                );
            }

            # Operation compelte
            return @emptyList;
        }
    }

    sub processPuebloListItemElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo list item element: <LI>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $listObj, $listType, $bulletType, $value, $string, $bullet,
            @emptyList, @stackList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloListItemElement', @_);
            return @emptyList;
        }

        # <LI [type=disc|circle|square|A|a|I|i|1>
        if ($tagMode eq 'close') {

            $self->puebloDebug($origToken, 'Malformed element', 7201);

            return @emptyList;
        }

        # <LI>

        # Process @argList

        # Default argument values
        %ivHash = (
            'type'      => undef,
            'value'     => undef,
        );

        if (@argList) {

            do {

                my $argName = lc(shift @argList);
                my $argValue = shift @argList;

                if (! defined $argName || ! exists $ivHash{$argName}) {

                    # Unrecognised argument name
                    $self->puebloDebug($origToken, 'Malformed element', 7211);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Retrieve the currently processed list object GA::Pueblo::List
        if (! $self->puebloStackList) {

            # The stack is empty, so this list item occurs outside an unordered/ordered list.
            #   Behave as though it were inside an unordered list
            $listType = 'ul';

        } else {

            @stackList = $self->puebloStackList;
            $listObj = shift @stackList;
            $listType = $listObj->listType;
        }

        # Check any specified argument values are valid. In some cases, give up; in other cases,
        #   use default values
        if (defined $ivHash{'type'}) {

            # <LI> attribute overrules the bullet type specified by the earlier <UL>/<OL> tag
            $bulletType = $ivHash{'type'};

        } elsif ($listObj) {

            if ($listType eq 'ul') {
                $bulletType = $listObj->bulletType;
            } else {
                $bulletType = $listObj->itemType;
            }

        } else {

            $bulletType = 'disc';
        }

        if ($listType eq 'ul') {

            if ($bulletType ne 'disc' && $bulletType ne 'circle' && $bulletType ne 'square') {

                # Default
                $bulletType = 'disc';
            }

        } else {

            if (
                $bulletType ne 'A' && $bulletType ne 'a' && $bulletType ne 'I' && $bulletType ne 'i'
                && $bulletType ne '1'
            ) {
                # Default
                $bulletType = '1';
            }
        }

        $value = $ivHash{'value'};
        if (! defined $value || ! $axmud::CLIENT->intCheck($value, 1)) {

            $value = $listObj->itemCount;
        }

        $listObj->ivPoke('itemCount', $value + 1);

        # If the <LI> tag appears outside a <UL>...</UL> or <OL>...</OL> construction, its text is
        #   displayed with no gap on the left
        $string = ' ' x $self->puebloColumnSize;            # Default - 3 characters
        # Set the bullet character to use, and update IVs as appropriate
        if ($bulletType eq 'disc' || $bulletType eq 'circle') {

            $bullet = chr(149) . ' ';

        } elsif ($bulletType eq 'square') {

            $bullet = chr(164) . ' ';

        } else {

            # Numbered (or lettered) bullet type
            if ($bulletType eq '1') {

                $bullet = $value . '. ';

            } elsif ($bulletType eq 'A' || $bulletType eq 'a') {

                # After 26 letters, go back to A
                $value = $value % 26;
                if ($bulletType eq 'A') {
                    $bullet = chr($value + 64) . '. ';
                } else {
                    $bullet = chr($value + 96) . '. ';
                }

            } else {

                # Roman numerals, for some reason. After 4000, go back to 1
                $value = $value % 4000;
                if ($bulletType eq 'I') {
                    $bullet = $axmud::CLIENT->convertRoman($value) . '. ';
                } else {
                    $bullet = lc($axmud::CLIENT->convertRoman($value)) . '. ';
                }
            }
        }

        # Insert these characters into the received text that's being processed by
        #   $self->processIncomingData
        $self->ivPoke(
            'puebloInsertString',
            $self->puebloInsertString . "\n" . $string x scalar ($self->puebloStackList) . $bullet,
        );

        # Operation compelte
        return @emptyList;
    }

    sub processPuebloJustifyElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo text justification element: <CENTER>...</CENTER>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloJustifyElement', @_);
            return @emptyList;
        }

        # <CENTER>
        # </CENTER>
        if (@argList) {

            $self->puebloDebug($origToken, 'Malformed element', 7301);

            return @emptyList;

        # <CENTER>
        } elsif ($tagMode eq 'open') {

            # The Axmud style tag is applied to the current line
            return 'justify_centre';

        # </CENTER>
        } else {

            # The Axmud style tag 'justify_default' must be applied to the next line
            $self->ivPoke('puebloJustifyMode', 'wait_newline');

            return @emptyList;
        }
    }

    sub processPuebloLiteralElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo literal text element: <CODE>...</CODE>, <PRE>...</PRE>, <SAMP>...</SAMP>
        # NB <CODE> and <PRE> are implemented identically by Axmud. <SAMP> uses the same font as
        #   <CODE> and <PRE>, but doesn't cause Axmud to ignore line breaks/reduce whitespace
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            @emptyList, @tagList,
            %stackHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloLiteralElement', @_);
            return @emptyList;
        }

        # <CODE>
        # </CODE>
        # <PRE>
        # </PRE>
        # <SAMP>
        # </SAMP>
        if ($keyword eq 'CODE' && @argList) {

            # (Ignore attributes in <PRE> tags)
            $self->puebloDebug($origToken, 'Malformed element', 7401);

            return @emptyList;
        }

        if ($keyword eq 'CODE' || $keyword eq 'PRE' || $keyword eq 'SAMP') {

            # <CODE>
            # <PRE>
            # <SAMP>
            if ($tagMode eq 'open') {

                # In successive <CODE>...<CODE> tags, the second <CODE> tag is treated as though
                #   it were preceded by a closing <CODE> tag
                # (Same applies to <PRE>, but not <SAMP>
                if ($keyword eq 'SAMP' && ! $self->puebloLiteralSampFlag) {

                    $self->ivPoke('puebloLiteralSampFlag', TRUE);

                } elsif ($keyword ne 'SAMP' && ! $self->puebloLiteralFlag) {

                    $self->ivPoke('puebloLiteralFlag', TRUE);
                }

                # Prepare a monospaced font
                $stackHash{'font_name'} = $axmud::CLIENT->constFont;
                $stackHash{'font_size'} = $axmud::CLIENT->constFontSize;
                $stackHash{'bold_flag'} = FALSE;

                # Create a dummy style tag that $self->applyColourStyleTags can interpret
                #   e.g. 'mxpf_monospace_bold_12'
                push (@tagList, $self->createMxpFontTag(%stackHash));

                # Create a new MXP stack object and store it in the current textview object,
                #   updating the latter's IVs
                if (
                    ! $self->currentTabObj->textViewObj->createMxpStackObj(
                        $self,
                        $keyword,
                        %stackHash,
                    )
                ) {
                    $self->puebloDebug($origToken, 'Internal error while processing element', 7411);

                    return @emptyList;
                }

            } else {

                # </CODE>
                # </PRE>
                # </SAMP>
                if ($keyword eq 'SAMP') {

                    if (! $self->puebloLiteralSampFlag) {

                        # An invalid </SAMP>...</SAMP> construction
                        $self->puebloDebug(
                            $origToken,
                            '</SAMP> tag after earlier closing </SAMP> tag',
                            7421,
                        );

                        return @emptyList;
                    }

                    # End of <SAMP> block
                    $self->ivPoke('puebloLiteralSampFlag', FALSE);

                } else {

                    if (! $self->puebloLiteralFlag) {

                        # An invalid </CODE>...</CODE> or <PRE>...</PRE> construction
                        $self->puebloDebug(
                            $origToken,
                            '</CODE> or </PRE> tag after earlier closing </CODE> or </PRE> tag',
                            7422,
                        );

                        return @emptyList;
                    }

                    # End of <CODE> or <PRE> block
                    $self->ivPoke('puebloLiteralFlag', FALSE);
                }

                # Restore the previous font
                return $self->popMxpStack($keyword);
            }
        }

        return @tagList;
    }

    sub processPuebloImageElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo image element: <IMG>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $tag, $href, $file, $dir, $ext,
            @emptyList, @attList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloImageElement', @_);
            return @emptyList;
        }

        # <IMG [ISMAP] [SRC|HREF] [WIDTH=width] [HEIGHT=height] VSPACE HSPACE>
        # <IMG XCH_MODE=text|html|purehtml>
        # <IMG XCH_SOUND=play|loop|stop|stoploop HREF=url EVENT=complete MD5=md5 ACTION=action
        #   OPTIONS=queue>
        if (($tagMode eq 'open' && ! @argList) || $tagMode eq 'close') {

            $self->puebloDebug($origToken, 'Malformed element', 7501);

            return @emptyList;
        }

        # Process @argList

        # Default argument values
        %ivHash = (
            'align'     => undef,       # not implemented
            'ismap'     => FALSE,
            'src'       => undef,
            'href'      => undef,
            'border'    => undef,       # not implemented
            'width'     => undef,
            'height'    => undef,
            'vspace'    => FALSE,
            'hspace'    => FALSE,
            'alt'       => undef,       # not implemented
            'lowsrc'    => undef,       # not implemented
            'xch_mode'  => undef,
            'xch_sound' => undef,       # only xch_sound=play implemented
            'complete'  => undef,       # not implemented
            'md5'       => undef,       # not implemented
            'action'    => undef,       # not implemented
            'options'   => undef,       # not implemented
        );

        if (@argList) {

            do {

                my $argName = lc(shift @argList);
                my $argValue = shift @argList;

                if (! defined $argName || ! exists $ivHash{$argName}) {

                    # Unrecognised argument name
                    $self->puebloDebug($origToken, 'Malformed element', 7511);

                    return @emptyList;

                } elsif ($argName eq 'ismap' || $argName eq 'vspace' || $argName eq 'hspace') {

                    $ivHash{$argName} = TRUE;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Deal with <IMG XCH_MODE=text|html|purehtml> (implemented the same way as <XCH_MUDTEXT>
        if ($ivHash{'xch_mode'}) {

            # Don't bother checking the usual <IMG> attributes, but we will check the value
            if ($ivHash{'xch_mode'} eq 'text') {

                $self->ivPoke('puebloActiveFlag', FALSE);

            } elsif ( $ivHash{'xch_mode'} eq 'html' || $ivHash{'xch_mode'} eq 'purehtml') {

                $self->ivPoke('puebloActiveFlag', TRUE);

            } else {

                $self->puebloDebug($origToken, 'Invalid Pueblo interpret mode', 7521);
            }

            return @emptyList;
        }

        # Deal with <IMG XCH_SOUND=play|loop|stop|stoploop HREF=url EVENT=complete MD5=md5
        #   ACTION=action OPTIONS=queue>
        # Partially implemented. <IMG XCH_SOUND=play HREF=url> will play the specified sound file
        if ($ivHash{'xch_sound'}) {

            # Don't bother checking the usual <IMG> attributes, but we will check the URL
            if ($ivHash{'xch_sound'} ne 'play') {

                # Ignore this sound tag
                return @emptyList;
            }

            ($file, $dir, $ext) = File::Basename::fileparse($ivHash{'href'}, qr/\.[^.]*/);
            if (! $file) {

                # Unreadable sound source
                $self->puebloDebug($origToken, 'Unreadable sound source', 7531);

                return @emptyList;
            }

            # Convert the tag to an MXP tag, and let the MXP code handle it
            $tag = '<SOUND ' . $file . $ext . ' U="' . $dir . '>';
            push (@attList, 'fname', $file . $ext);
            push (@attList, 'u', $dir);

            $self->processMxpImageElement(
                $tag,
                'open',
                'SOUND',
                @attList,
            );

            return @emptyList;
        }

        # Check any specified argument values are valid
        if (defined $ivHash{'href'}) {
            $href = $ivHash{'href'};
        } else {
            $href = $ivHash{'src'};
        }

        if (! defined $href) {

            # Invalid foreground colour
            $self->puebloDebug($origToken, 'Image tag with no image source', 7532);

            return @emptyList;
        }

        # Need to convert the source URL into its components
        ($file, $dir, $ext) = File::Basename::fileparse($href, qr/\.[^.]*/);
        if (! $file) {

            # Unreadable image source
            $self->puebloDebug($origToken, 'Unreadable image source', 7533);

            return @emptyList;
        }

        # Convert the tag to MXP tag(s), and let the MXP code handle it
        if ($ivHash{'ismap'}) {

            $self->processMxpSendElement(
                '<SEND showmap>',
                'open',
                'SEND',
                FALSE,
                'href',
                'showmap',
            );
        }

        $tag = '<IMAGE ' . $file . $ext . ' url="' . $dir;
        push (@attList, 'fname', $file . $ext);
        push (@attList, 'url', $dir);

        if (defined $ivHash{'width'}) {

            $tag .= ' w=' . $ivHash{'width'};
            push (@attList, 'w', $ivHash{'width'});
        }

        if (defined $ivHash{'height'}) {

            $tag .= ' h=' . $ivHash{'height'};
            push (@attList, 'h', $ivHash{'height'});
        }

        if ($ivHash{'hspace'}) {

            $tag .= ' hspace';
            push (@attList, 'hspace', undef);
        }

        if ($ivHash{'vspace'}) {

            $tag .= ' vspace';
            push (@attList, 'vspace', undef);
        }

        if ($ivHash{'ismap'}) {

            $tag .= ' ismap';
            push (@attList, 'ismap', undef);
        }

        $tag .= '>';

        $self->processMxpImageElement(
            $tag,
            'open',
            'IMAGE',
            @attList,
        );

        if ($ivHash{'ismap'}) {

            $self->processMxpSendElement(
                '</SEND>',
                'close',
                'SEND',
                FALSE,
            );
        }

        return @emptyList;
    }

    sub processPuebloPaneElement {

        # Called by $self->processPuebloElement
        #
        # Process a Pueblo pane element: <XCH_PANE>
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $tag, $action, $name, $count, $href, $imgTag, $scrolling, $internalFlag, $frameObj,
            @emptyList, @attList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->processPuebloPaneElement', @_);
            return @emptyList;
        }

        # <XCH_PANE ACTION=open|close|redirect [HREF=url] [NAME=name] [PANETITLE=title]
        #   [WIDTH=width] [HEIGHT=height] [MINWIDTH=width] [MINHEIGHT=height] [VSPACE=pixels]
        #   [HSPACE=pixels] [SCROLLING=yes|no|auto]
        #   [OPTIONS=
        #       "overlapped|floating|internal|browser|nonsizeable|noclose|small_title|fit|persistent
        #       |force|viewbottom|webtracker"
        #   ]
        #
        # Implementation notes:
        #   - ACTION=open - the Pueblo spec says "The HTML string must include a href attribute
        #       indicating what should be displayed in the pane.". If the href attribute is a
        #       .gif, .bmp or .jpg URL, that image file is loaded and displayed in the new pane;
        #       otherwise a new pane is created with a clickable link inside it. Axmud creates the
        #       pane even if href= is not specified at all
        #   - NAME= - "If this attribute contains no value, then the pane will be created without a
        #       name and cannot be closed by the world author.". Unnamed panes or panes named
        #       '_blank' are given the internal name pueblo_n, where n is a positive integer, and
        #       can be closed at any time by the user (just like any other internal/external window)
        #   - The MINWIDTH=, MINHEIGHT=, VSPACE= and HSPACE= attributes are not implemented
        #   - The only OPTIONS= values which are implemented are 'floating' and 'internal'. If both
        #       values are specified, only 'internal' is used
        #   - For ACTION=redirect is handled in the same way as ACTION=open, following the behaviour
        #       of the corresponding MXP <FRAME> tag attributes
        #
        if (($tagMode eq 'open' && ! @argList) || $tagMode eq 'close') {

            $self->puebloDebug($origToken, 'Malformed element', 7601);

            return @emptyList;
        }

        # The <XCH_PANE> tag can change $self->currentTabObj, corresponding to the frame in which
        #   text received from the world is displayed. The calling function may have processed some
        #   text tokens which haven't been displayed yet; if so, display them now so that they're
        #   displayed in the right frame
        if ($self->recvLineText) {

            $self->processIncompleteLine($self->mxpOrigText);
            $self->ivUndef('mxpOrigText');
        }

        # Apply any links created by Pueblo <A> tags to the current textview (if the current
        #   textview was changed during the call to this function, any links for other textviews
        #   have already been applied)
        foreach my $linkObj ($self->mxpTempLinkList) {

            $self->currentTabObj->textViewObj->add_incompleteLink($linkObj);
        }

        $self->ivEmpty('mxpTempLinkList');

        # Process @argList

        # Default argument values
        %ivHash = (
            'action'    => 'open',
            'href'      => '',
            'name'      => '',
            'panetitle' => '',
            'width'     => undef,
            'height'    => undef,
            'minwidth'  => undef,       # not implemented
            'minheight' => undef,       # not implemented
            'vspace'    => undef,
            'hspace'    => undef,
            'scrolling' => 'auto',
            'options'   => undef,
            'align'     => undef,       # not implemented
        );

        if (@argList) {

            do {

                my $argName = lc(shift @argList);
                my $argValue = shift @argList;

                if (! defined $argName || ! exists $ivHash{$argName}) {

                    # Unrecognised argument name
                    $self->puebloDebug($origToken, 'Malformed element', 7611);

                    return @emptyList;

                } else {

                    $ivHash{$argName} = $argValue;
                }

            } until (! @argList);
        }

        # Check any specified argument values are valid
        $action = $ivHash{'action'};
        if ($action ne 'open' && $action ne 'close' && $action ne 'redirect') {

            $self->puebloDebug($origToken, 'Invalid action attribute', 7621);

            return @emptyList;

        } elsif ($action eq 'close' && $name eq '') {

            # Must specify name= attribute, too
            $self->puebloDebug($origToken, 'Invalid close attribute', 7622);

            return @emptyList;
        }

        $name = $ivHash{'name'};
        if ($name eq '') {

            # Choose a name for the unnamed pane. To avoid infinite loops, max number is 1000
            $count = 0;
            do {

                $count++;
                $name = 'pueblo_' . $count;

            } until (! $self->ivExists('mxpFrameHash', $name) || $count > 1000);

            if (! $name) {

                $self->puebloDebug($origToken, 'Maximum number of Pueblo panes exceeded', 7623);

                return @emptyList;
            }
        }

        $scrolling = $ivHash{'scrolling'};
        if ($scrolling ne 'yes' && $scrolling ne 'no' && $scrolling ne 'auto') {

            $self->puebloDebug($origToken, 'Invalid scrolling attribute', 7624);

            return @emptyList;
        }

        # Convert the tag to MXP tag(s), and let the MXP code handle it
        if ($action eq 'open' || $action eq 'redirect') {

            $tag = '<FRAME name="' . $name . '"';
            push (@attList, 'name', $name);

            $tag .= ' action=' . $action;
            push (@attList, 'action', $action);

            if ($ivHash{'panetitle'}) {

                $tag .= ' title="' . $ivHash{'panetitle'} . '"';
                push (@attList, 'title', $ivHash{'panetitle'});
            }

            if (
                defined $ivHash{'options'}
                && $ivHash{'options'} =~ m/internal/
                && ! $axmud::CLIENT->shareMainWinFlag
            ) {
                $tag .= ' internal';
                push (@attList, 'internal', undef);
                $internalFlag = TRUE;
            }

            if (defined $ivHash{'width'}) {

                $tag .= ' width=' . $ivHash{'width'};
                push (@attList, 'width', $ivHash{'width'});
            }

            if (defined $ivHash{'height'}) {

                $tag .= ' height=' . $ivHash{'height'};
                push (@attList, 'height', $ivHash{'height'});
            }

            if ($scrolling eq 'yes' || $scrolling eq 'no') {

                $tag .= ' scrolling=' . $scrolling;
                push (@attList, 'scrolling', $scrolling);
            }

            if (
                ! $internalFlag
                && defined $ivHash{'options'}
                && $ivHash{'options'} =~ m/floating/
            ) {
                $tag .= ' floating';
                push (@attList, 'floating', undef);
            }

            $tag .= '>';

            $self->processMxpFrameElement(
                $tag,
                'open',
                'FRAME',
                @attList,
            );

            # Make sure the pane actually exists now
            $frameObj = $self->getMxpFrame($name);
            if (! $frameObj) {

                $self->puebloDebug($origToken, 'Internal error while creating MXP frame', 7625);

                return @emptyList;

            } elsif (! $frameObj->paneObj) {

                # The pane isn't visible yet, for some reason, so don't try to apply the href=
                #   attribute
                return @emptyList;
            }

            # Apply the href= attribute, if specified
            $href = $ivHash{'href'};
            if ($href ne '') {

                if ($action eq 'open') {

                    $self->processMxpFrameElement(
                        '<FRAME ' . $name . ' REDIRECT>',
                        'open',
                        'FRAME',
                        'name'      => $name,
                        'action'    => 'REDIRECT',
                    );
                }

                if ($href =~ m/(\.gif|\.bmp|\.jpg)$/) {

                    $self->processPuebloImageElement(
                        '<IMG src=' . $href . '>',
                        'open',
                        'IMG',
                        'src'       => $href,
                    );

                } else {

                    $self->currentTabObj->textViewObj->insertWithLinks($href, 'echo');
                }

                if ($action eq 'open') {

                    $self->processMxpFrameElement(
                        '<FRAME _previous REDIRECT>',
                        'open',
                        'FRAME',
                        'name'      => '_previous',
                        'action'    => 'REDIRECT',
                    );
                }
            }

        } elsif ($action eq 'close') {

            $tag = '<FRAME name=' . $name;
            push (@attList, 'name', $name);

            $tag .= ' action=' . $action;
            push (@attList, 'action', $action);

            $tag .= '>';

            $self->processMxpFrameElement(
                $tag,
                'open',
                'FRAME',
                @attList,
            );
        }

        return @emptyList;
    }

    sub processPuebloImplementedElement {

        # Called by $self->processMxpElement
        #
        # Process an implemented Pueblo element, not covered by other functions (<XCH_MUDTEXT>,
        #   <XCH_PAGE>, <XCH_ALERT>)
        #
        # Expected arguments
        #   $origToken  - The original token text, before anything was extracted
        #   $tagMode    - 'open' for <..> elements, 'close' for </..> elements
        #   $keyword    - The element keyword (already converted to upper case)
        #
        # Optional arguments
        #   @argList    - If the element has arguments, a list in the form
        #                    (arg_name, arg_value, arg_name, arg_value...)
        #                 ...where each 'arg_value' is 'undef' is the argument wasn't a name=value
        #                   construction
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise returns an equivalent list of Axmud colour/style tags otherwise (may be an
        #       empty list)

        my ($self, $origToken, $tagMode, $keyword, @argList) = @_;

        # Local variables
        my (
            $clear,
            @emptyList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $origToken || ! defined $tagMode || ! defined $keyword) {

            $axmud::CLIENT->writeImproper(
                $self->_objClass . '->processPuebloImplementedElement',
                @_,
            );

            return @emptyList;
        }

        # <XCH_MUDTEXT>
        # </XCH_MUDTEXT>
        # <XCH_PAGE CLEAR=links|plugins|images|text FGCOLOR|TEXT="#rrggbb">
        # <XCH_ALERT>
        if (
            (($keyword eq 'XCH_MUDTEST' || $keyword eq 'XCH_ALERT') && @argList)
            || (($keyword eq 'XCH_PAGE' || $keyword eq 'XCH_ALERT') && $tagMode eq 'close')
        ) {
            $self->puebloDebug($origToken, 'Malformed element', 7701);

            return @emptyList;
        }

        if ($keyword eq 'XCH_MUDTEXT') {

            # <XCH_MUDTEXT>
            if ($tagMode eq 'close') {

                $self->ivPoke('puebloActiveFlag', FALSE);

            # </XCH_MUDTEXT>
            } else {

                $self->ivPoke('puebloActiveFlag', TRUE);
            }

            return @emptyList;

        } elsif ($keyword eq 'XCH_PAGE') {

            # Process @argList

            # Default argument values
            %ivHash = (
                'clear'     => 'links',     # 'links'/'text' implemented; 'plugins'/'images' not
                'fgcolor'   => undef,       # not implemented
                'text'      => undef,       # same as 'fgcolor', not implemented
            );

            if (@argList) {

                do {

                    my $argName = lc(shift @argList);
                    my $argValue = shift @argList;

                    if (! defined $argName || ! exists $ivHash{$argName}) {

                        # Unrecognised argument name
                        $self->puebloDebug($origToken, 'Malformed element', 7711);

                        return @emptyList;

                    } else {

                        $ivHash{$argName} = $argValue;
                    }

                } until (! @argList);
            }

            # Check any specified argument values are valid
            $clear = $ivHash{'clear'};
            if (
                defined $clear && $clear ne 'links' && $clear ne 'plugins' && $clear ne 'images'
                && $clear ne 'text'
            ) {
                # Unrecognised argument name
                $self->puebloDebug($origToken, 'Invalid <XCH_PAGE> attribute', 7721);

                return @emptyList;
            }

            # Process the tag. The 'plugins' value is not implemented
            if ($clear eq 'links') {

                foreach my $linkObj ($self->currentTabObj->textViewObj->ivValues('linkObjHash')) {

                    $linkObj->expiredFlag = TRUE;
                }

            } elsif ($clear eq 'text') {

                $self->currentTabObj->textViewObj->clearBuffer();
            }

            return @emptyList;

        # <XCH_ALERT>
        } elsif ($keyword eq 'XCH_ALERT') {

            $axmud::CLIENT->playSound('beep');

            return @emptyList;
        }
    }

    sub processPuebloSpacingTag {

        # Called by $self->processIncomingData
        #
        # Process a Pueblo line spacing token: <BODY>, </BODY>, <P>, </P>, <BR>. <HR>
        #
        # Expected arguments
        #   $origText   - The original text received from the world, before this token was
        #                   extracted
        #   $token      - An extracted token containing the Pueblo element
        #
        # Return values
        #   'undef' on improper arguments or if an invalid closing tag like </BR> is used
        #   Otherwise, returns a modified $origText. If the token was converted into a newline
        #       character, $origText is set by this function to be an empty string (as happens after
        #       ->processIncomingData calls ->processEndLine directly); otherwise $token is added
        #       to the existing value of $origText

        my ($self, $origText, $token, $check) = @_;

        # Local variables
        my ($origToken, $bufferObj, $width, $height, $fontSize);

        # Check for improper arguments
        if (! defined $token || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->processPuebloSpacingTag',
                @_,
            );
        }

        # Remove the < and > characters
        $origToken = $token;
        $token = uc($token);
        $token =~ s/^\<//;
        $token =~ s/\>$//;
        # These Pueblo tags can have attributes, but Axmud ignores those attributes
        $token =~ s/\s.*//;

        # Only the </BODY> and </P> elements can be used as a closing tag (e.g. </BR> is invalid)
        if ($token ne '/BODY' && $token ne '/P' && substr($token, 0, 1) eq '/') {

            $self->puebloDebug($origToken, 'Invalid line spacing tag \'' . $origToken . '\'', 7801);
        }

        # Process the tag
        if ($token eq 'BODY' || $token eq '/BODY') {

            # The Pueblo spec states that <BODY>...</BODY> is not necessary for world output.
            #   However, some worlds (e.g. Epoch) display some introductory text before the
            #   <BODY>...</BODY> construction, so we'll replace both tags with empty lines

            # If the current line contains any non-whitespace characters, we must insert a newline
            #   character to terminate it
            if ($origText =~ m/\S/) {

                $self->processEndLine($origText, $origToken);
                $origText = '';
            }

            # Insert a second newline character to put space between this paragraph, and anything
            #   that follows it
            $self->processEndLine('', $origToken);

        } elsif ($token eq 'BR') {

            # Force a line break inside or outside a paragraph
            $self->processEndLine($origText, $origToken);
            # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
            #   string, so this function does the same)
            return '';

        } elsif ($token eq 'P') {

            # Special case: inside a <PRE>...</PRE> construction, <P> tags are implemented as a
            #   simple newline
            if ($self->puebloLiteralFlag) {

                # Force a line break
                $self->processEndLine($origText, $origToken);
                # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
                #   string, so this function does the same)
                return '';
            }

            $self->ivPoke('puebloParagraphFlag', TRUE);

            # If the current line contains any non-whitespace characters, we must insert a newline
            #   character to terminate it
            if ($self->recvWholeLineText =~ m/\S/) {

                $self->processEndLine($origText, $origToken);
                $origText = '';
            }

            # If the line before that contained any newline characters, we must insert a newline
            #   character to put an empty line between the that text, and the beginning of the
            #   paragraph
            if (defined $self->displayBufferLast) {

                $bufferObj = $self->ivShow('displayBufferHash', $self->displayBufferLast);
                if ($bufferObj->modLine =~ m/\S/) {

                    $self->processEndLine($origText, $origToken);
                    $origText = '';
                }
            }

            return $origText;       # Will almost certainly be an empty string

        } elsif ($token eq '/P') {

            if (! $self->puebloParagraphFlag) {

                # An invalid </P>...</P> construction
                $self->puebloDebug($origToken, '</P> tag after earlier closing </P> tag', 7811);

            } else {

                $self->ivPoke('puebloParagraphFlag', FALSE);

                # If the current line contains any non-whitespace characters, we must insert a
                #   newline character to terminate it
                if ($self->recvWholeLineText =~ m/\S/) {

                    $self->processEndLine($origText, $origToken);
                    $origText = '';
                }

                # Insert a second newline character to put space between this paragraph, and
                #   anything that follows it
                $self->processEndLine('', $origToken);

                return '';
            }

        } elsif ($token eq 'HR') {

            # An HTML element. Draw a poor man's 'horizontal rule' with simple ASCII characters

            # Force a line break inside or outside a paragraph. The TRUE argument means 'don't close
            #   open MXP tags, as we would for a true newline character'
            $self->processEndLine($origText, $origToken, TRUE);

            # Adjust the width to take account of different font sizes, especially inside headings
            #   (<H1>...</H1>, etc)
            ($width, $height) = $self->getTextViewSize();
            $fontSize = $self->currentTabObj->textViewObj->ivShow('mxpModalStackHash', 'font_size');
            if ($fontSize ne '' && $fontSize != $axmud::CLIENT->constFontSize) {

                # (Subtracting 1 seems to produce the right answer more often, for unknown reasons)
                $width = int($width / ($fontSize / $axmud::CLIENT->constFontSize)) - 1;
            }

            # Draw the horizontal rule
            $self->processTextToken(chr(0x2501) x $width);
            # ...which ends in another newline character
            $self->processEndLine($origText, $origToken, TRUE);

            # (When ->processIncomingData calls ->processEndLine, it sets $origText to an empty
            #   string, so this function does the same)
            return '';
        }

        return $origText . $origToken;
    }

    # Incoming data loop - misc incoming stuff

    sub applyColourStyleTags {

        # Called by $self->processLineSegment and ->applyTriggerStyle
        #
        # This function is two arguments
        # The first argument is a reference to a hash of Axmud colour/style tags, based on
        #   GA::Client->constColourStyleHash, representing the colours and styles that applied at
        #   some specific offset on a specific line of text received from the world, before any
        #   colour/style tags were applied
        # The second argument is a reference to a list of Axmud colour/style tags representing the
        #   colours and styles that were applied at that offset
        #
        # This function applies the list of tags to the hash, and returns the modified hash
        #
        # When called by $self->processLineSegment, the colour/style tags are those that apply
        #   right now. When called by ->applyColourStyleTags, the colour/style tags are those that
        #   applied at some point in the recent past, at the beginning or end of a line segment to
        #   which trigger styles are being applied
        #
        # Expected arguments
        #   $hashRef    - A reference to a hash of Axmud colour/style tags, based on
        #                   GA::CLIENT->constColourStyleHash, in the form
        #
        #                   'text'       => 'undef' or an Axmud text colour tag, e.g. 'red' or
        #                                       'x230'
        #                   'underlay'   => 'undef' or an Axmud underlay colour tag, e.g. 'ul_white'
        #                   'italics'    => TRUE or FALSE
        #                   'underline'  => TRUE or FALSE
        #                   'blink_slow' => TRUE or FALSE
        #                   'blink_fast' => TRUE or FALSE
        #                   'strike'     => TRUE or FALSE
        #                   'link'       => TRUE or FALSE
        #                   'mxp_font'   => TRUE or FALSE
        #                   'justify'    => 'left', 'right', 'centre', or 'undef' to represent the
        #                                       style tag 'justify_default'
        #   $listRef    - A rerefence to a list of Axmud colour/style tags which are used to modify
        #                   the hash (can be an empty list)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns the modified hash itself (not a reference to it)

        my ($self, $hashRef, $listRef, $check) = @_;

        # Local variables
        my (
            $textViewObj, $attribsOffFlag,
            @tagList,
            %hash, %styleHash, %justifyHash, %dummyHash,
        );

        # Check for improper arguments
        if (! defined $hashRef || ! defined $listRef || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->applyColourStyleTags', @_);
        }

        # De-reference arguments
        %hash = %$hashRef;
        @tagList = @$listRef;

        # Import IVs (for quick lookup)
        %styleHash = $axmud::CLIENT->constStyleTagHash;
        %justifyHash = $axmud::CLIENT->constJustifyTagHash;
        %dummyHash = $axmud::CLIENT->constDummyTagHash;
        # Import the current textview object (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        foreach my $tag (@tagList) {

            my ($type, $underlayFlag);

            # When called by $self->applyTriggerStyle, @tagList can contain bold text colour tags
            #   like 'BLUE'. When called by $self->processLineSegment, @tagList would instead
            #   contain the pair 'bold', 'blue'
            # @tagList might also contain bold underlay colours like 'UL_BLUE', but that doesn't
            #   affect the setting of $hash{'bold'} (which only applies to text colours)
            $type = $axmud::CLIENT->checkBoldTags($tag);
            if ($type && $type eq 'text') {

                # $tag is a bold colour tag. Pretend that we processed a 'bold' dummy tag during
                #   the last iteration of this loop
                $hash{'bold'} = TRUE;
                # Since we're about to change the text colour, we can rely on the following code to
                #   take care of $hash{'text'}, $hash{'real_text'} and so on
                $tag = lc($tag);
            }

            # PART 1: 'Dummy' style tags
            if (exists $dummyHash{$tag}) {

                if ($tag eq 'attribs_off') {

                    # A dummy tag created by $self->processEscSequence, which corresponds to 'all
                    #   attributes off'. We implement this by resetting %hash
                    %hash = $axmud::CLIENT->constColourStyleHash;
                    $attribsOffFlag = TRUE;

                } elsif ($tag eq 'bold') {

                    # A 'dummy' style tag
                    if (! $hash{'bold'}) {

                        $hash{'bold'} = TRUE;

                        if ($hash{'real_text'}) {

                            $hash{'real_text'} = uc($hash{'real_text'});
                        }

                        # In conceal mode we don't make any changes to $hash{'text'} and
                        #   $hash{'underlay'}
                        if ($hash{'reverse'}) {
                            $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});
                        } elsif (! $hash{'conceal'}) {
                            $hash{'text'} = $hash{'real_text'};
                        }

                        # In conceal mode and reverse video mode, changing the underlay colour has
                        #   no effect
                        if (! $hash{'conceal'} && ! $hash{'reverse'}) {

                            $hash{'underlay'} = $hash{'real_underlay'};
                        }
                    }

                    # (A second 'bold' tag is ignored)

                } elsif ($tag eq 'bold_off') {

                    # A 'dummy' style tag
                    if ($hash{'bold'}) {

                        $hash{'bold'} = FALSE;

                        if ($hash{'real_text'}) {

                            $hash{'real_text'} = lc($hash{'real_text'});
                        }

                        if ($hash{'reverse'}) {
                            $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});
                        } elsif (! $hash{'conceal'}) {
                            $hash{'text'} = $hash{'real_text'};
                        }

                        # In conceal mode and reverse video mode, changing the underlay colour has
                        #   no effect
                        if (! $hash{'conceal'} && ! $hash{'reverse'}) {

                            $hash{'underlay'} = $hash{'real_underlay'};
                        }
                    }

                    # (A second consecutive 'bold_off' tag is ignored)

                } elsif ($tag eq 'reverse') {

                    # A 'dummy' style tag
                    if (! $hash{'reverse'}) {

                        $hash{'reverse'} = TRUE;

                        # In reverse video mode, the existing text colour is used as the underlay
                        #   colour. The existing underlay colour is ignored, if it is set, because
                        #   reverse video was designed for monochrome monitors
                        # In addition, the textview background colour is used as the new text colour

                        # Conceal mode takes priority over reverse video mode, if that is already on
                        if (! $hash{'conceal'}) {

                            $hash{'text'} = $textViewObj->backgroundColour;
                            if ($hash{'real_text'}) {

                                $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});

                            } else {

                                $hash{'underlay'}
                                    = $axmud::CLIENT->swapColours($textViewObj->textColour);
                            }
                        }
                    }

                    # (A second 'reverse' tag is ignored)

                } elsif ($tag eq 'reverse_off') {

                    # A 'dummy' style tag
                    if ($hash{'reverse'}) {

                        $hash{'reverse'} = FALSE;
                        if (! $hash{'conceal'}) {

                            $hash{'text'} = $hash{'real_text'};
                            $hash{'underlay'} = $hash{'real_underlay'};
                        }
                    }

                    # (A second consecutive 'reverse_off' tag is ignored)

                } elsif ($tag eq 'conceal') {

                    # A 'dummy' style tag
                    if (! $hash{'conceal'}) {

                        $hash{'conceal'} = TRUE;

                        # In conceal mode, both the text and underlay colours are set to the same as
                        #   the textview background colour, so that the text is invisible unless the
                        #   user selects it with the mouse
                        # Conceal mode takes priority over reverse video mode, if that is already on
                        $hash{'text'} = $textViewObj->backgroundColour;
                        $hash{'underlay'}
                            = $axmud::CLIENT->swapColours($textViewObj->backgroundColour);
                    }

                    # (A second 'conceal' tag is ignored)

                } elsif ($tag eq 'conceal_off') {

                    # A 'dummy' style tag
                    if ($hash{'conceal'}) {

                        $hash{'conceal'} = FALSE;
                        if ($hash{'reverse'}) {

                            $hash{'text'} = $textViewObj->backgroundColour;
                            $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});

                        } else {

                            $hash{'text'} = $hash{'real_text'};
                            $hash{'underlay'} = $hash{'real_underlay'};
                        }
                    }

                    # (A second consecutive 'conceal_off' tag is ignored)

                } elsif ($tag eq 'mxpf_off') {

                    # (Other 'mxpf_...' tags are handled below)
                    $hash{'mxp_font'} = undef;
                }

            # PART 2: Justification style tags
            } elsif (exists $justifyHash{$tag}) {

                if ($tag eq 'justify_default') {
                    $hash{'justify'} = undef;
                } else {
                    $hash{'justify'} = substr($tag, 8);
                }

            # PART 3: Style tags
            } elsif (exists $styleHash{$tag}) {

                # (Codes listed in rough order of popularity)
                if ($tag eq 'link') {

                    if ($hash{'link'}) {
                        $hash{'link'} = FALSE;
                    } else {
                        $hash{'link'} = TRUE;
                    }

                } elsif ($tag eq 'link_off') {

                    $hash{'link'} = FALSE;

                } elsif (
                    $tag eq 'italics'
                    || $tag eq 'underline'
                    || $tag eq 'blink_slow'
                    || $tag eq 'blink_fast'
                    || $tag eq 'strike'
                ) {
                    # (A second consecutive tag reinforces the previous one)
                    $hash{$tag} = TRUE;

                } elsif ($tag eq 'italics_off') {

                    $hash{'italics'} = FALSE;

                } elsif ($tag eq 'underline_off') {

                    $hash{'underline'} = FALSE;

                } elsif ($tag eq 'blink_off') {

                    $hash{'blink_slow'} = FALSE;
                    $hash{'blink_fast'} = FALSE;

                } elsif ($tag eq 'strike_off') {

                    $hash{'strike'} = FALSE;
                }

            # PART 4: MXP style tags
            } elsif (substr($tag, 0, 5) eq 'mxpf_') {

                # Dummy style tags used for MXP fonts
                # Store the whole dummy tag, e.g. 'mxpf_monospace_bold_12' (NB 'mxpf_off' was
                #   handled further above)
                $hash{'mxp_font'} = $tag;

            } elsif (substr($tag, 0, 5) eq 'mxpm_') {

                # Dummy style tag used for MXP modes in the range 10-12, 19, 20-99 (which don't
                #   affect text attributes, so we don't add them to %hash)
                # ...

            # PART 5: Colour tags
            } else {

                ($type, $underlayFlag) = $axmud::CLIENT->checkColourTags($tag);
                if ($type && ($type eq 'xterm' || $type eq 'rgb')) {

                    # xterm/RGB colour tags should be case insensitive
                    $tag = lc($tag);
                }

                # PART 5a: Text colour tags
                if ($type && ! $underlayFlag) {

                    if (
                        defined $hash{'real_text'}
                        && $hash{'real_text'} eq $tag
                    ) {
                        # (The re-occuring tag is ignored)

                    } elsif ($attribsOffFlag && $tag eq $textViewObj->textColour) {

                        # After an 'attribs off', if the tag matches the 'main' window's normal text
                        #   colour, let that be the text colour
                        # (This takes care of ANSI escape sequences like '^[0;37;40m', meaning
                        #   'attribs off - white text - black underlay'. If we set $hash{'underlay'}
                        #   to 'ul_black', the next 'bold' tag - which was intended to apply only to
                        #   the text colour - will be applied to the underlay colour, too.)
                        $hash{'real_text'} = undef;
                        if ($hash{'reverse'}) {

                            $hash{'text'} = $textViewObj->backgroundColour;
                            $hash{'underlay'}
                                = $axmud::CLIENT->swapColours($textViewObj->textColour);

                        } else {

                            $hash{'text'} = $hash{'real_text'};
                        }

                    } elsif ($hash{'bold'} && $type eq 'standard') {

                        # Bold text colour
                        $hash{'real_text'} = uc($tag);
                        # If the world is using an OSC colour palette to modify this colour, use the
                        #   modified form
                        if ($self->ivExists('oscColourHash', $hash{'real_text'})) {

                            $hash{'real_text'} = $self->ivShow('oscColourHash', $hash{'real_text'});
                        }

                        # In conceal mode we don't make any changes to $hash{'text'} and
                        #   $hash{'underlay'}
                        if ($hash{'reverse'}) {
                            $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});
                        } elsif (! $hash{'conceal'}) {
                            $hash{'text'} = $hash{'real_text'};
                        }

                    } else {

                        # Normal colour, or a bold underlay colour like 'UL_BLUE', specified
                        #   directly by a call from $self->applyTriggerStyle
                        $hash{'real_text'} = $tag;
                        # If the world is using an OSC colour palette to modify this colour, use the
                        #   modified form
                        if ($self->ivExists('oscColourHash', $hash{'real_text'})) {

                            $hash{'real_text'} = $self->ivShow('oscColourHash', $hash{'real_text'});
                        }

                        if ($hash{'reverse'}) {
                            $hash{'underlay'} = $axmud::CLIENT->swapColours($hash{'real_text'});
                        } elsif (! $hash{'conceal'}) {
                            $hash{'text'} = $hash{'real_text'};
                        }
                    }

                # PART 5b: Underlay colour tags
                } elsif ($type && $underlayFlag) {

                    if (
                        defined $hash{'real_underlay'}
                        && $hash{'real_underlay'} eq $tag
                    ) {
                        # (The re-occuring tag is ignored)

                    } elsif ($attribsOffFlag && $tag eq $textViewObj->underlayColour) {

                        # After an 'attribs off', if the tag matches the 'main' window's normal
                        #   underlay colour, let that be the underlay colour
                        $hash{'real_underlay'} = undef;

                    } else {

                        $hash{'real_underlay'} = $tag;

                        # If the world is using an OSC colour palette to modify this colour, use the
                        #   modified form
                        if ($self->ivExists('oscColourHash', $hash{'real_underlay'})) {

                            $hash{'real_underlay'}
                                = $self->ivShow('oscColourHash', $hash{'real_underlay'});
                        }
                    }

                    # In conceal mode and reverse video mode, changing the underlay colour has no
                    #   effect
                    if (! $hash{'conceal'} && ! $hash{'reverse'}) {

                        $hash{'underlay'} = $hash{'real_underlay'};
                    }
                }
            }
        }

        return %hash;
    }

    sub checkLineSplit {

        # Called by $self->processLinePortion to work out if a received line of text should be
        #   split into two or more pieces, because the lines matches one of the world profile's
        #   command prompt patterns, or because it matches a splitter trigger
        # (This only affects the way Axmud treats received text, after all non-text tokens have been
        #   extracted by $self->processIncomingData; for example, it doesn't affect MXP at all)
        #
        # Returns a list of offsets of the first character after any split. However, it's not
        #   possible to split the line at the beginning (or end), so the list of offsets never
        #   contains 0 or n, where n = the length of the line. The list of offsets is in ascending
        #   order, and there are no duplicate offsets (so if a command prompt and a splitter trigger
        #   both split the line at the same place, only one offset is added to the list)
        #
        # Expected arguments
        #   $stripText      - A portion of the received text, comprising a complete or partial line
        #                       of text received from the world, which has now been stripped of non-
        #                       text tokens like newline characters, escape sequences, etc
        #   $newLineFlag    - Flag set to TRUE if this line ends in a newline character, or FALSE if
        #                       it doesn't
        #
        # Return values
        #   An empty list on improper arguments or if no command prompt patterns or splitter
        #       triggers match the line
        #   Otherwise, returns a list of offsets

        my ($self, $stripText, $newLineFlag, $check) = @_;

        # Local variables
        my (
            @emptyList, @offsetList, @deleteList, @sortedList,
            %checkHash,
        );

        # Check for improper arguments
        if (! defined $stripText || ! defined $newLineFlag || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->checkLineSplit', @_);
            return @emptyList;
        }

        # First split the line, according to the command prompt patterns specified by the current
        #   world profile
        foreach my $pattern ($self->currentWorld->cmdPromptPatternList) {

            while ($stripText =~ m/($pattern).+/g) {

                my $offset = $+[1];

                if ($offset != 0 && $offset != length($stripText)) {

                    push (@offsetList, $offset);
                    # We'll use a hash to eliminate duplicates, once we start checking splitter
                    #   triggers
                    $checkHash{$offset} = undef;
                }
            }
        }

        # Check every active trigger interface, in the correct order, looking for splitter triggers
        #   that match $stripText
        OUTER: foreach my $number ($self->triggerOrderList) {

            my ($obj, $stimulus, $response, $afterFlag, $ignoreFlag, $count, $copyLine);

            $obj = $self->ivShow('interfaceNumHash', $number);

            # If the trigger isn't a splitter trigger, ignore it
            # If the trigger is a dependent trigger, don't fire it
            # If the trigger is disabled, don't fire it
            # If the trigger requires a line ending with a newline character and the line doesn't
            #   end with one, don't fire it
            # If the trigger requires a login and the character isn't logged in, don't fire it
            if (
                ! $obj->ivShow('attribHash', 'splitter')
                || ! $obj->indepFlag
                || ! $obj->enabledFlag
                || ($obj->ivShow('attribHash', 'need_prompt') && $newLineFlag)
                || ($obj->ivShow('attribHash', 'need_login') && ! $self->loginFlag)
            ) {
               next OUTER;
            }

            $stimulus = $obj->stimulus;
            $response = $obj->response;
            $afterFlag = $obj->ivShow('attribHash', 'split_after');
            $ignoreFlag = $obj->ivShow('attribHash', 'ignore_case');
            $count = 0;

            # An independent splitter trigger.
            # The trigger's 'stimulus' attribute must match $stripText for the trigger to fire
            # (NB Using $stripText in the block breaks the algorithm. No idea why, but a workaround
            #   is to perform the pattern match on another variable)
            $copyLine = $stripText;
            if (
                (! $ignoreFlag && ($copyLine =~ m/$stimulus/g))
                || ($ignoreFlag && ($copyLine =~ m/$stimulus/gi))
            ) {
                # Now we test the line against the trigger's 'response' attribute. We split the line
                #   immediately before the matching portion of text or, if the trigger's
                #   'keep_splitting' attribute is TRUE, immediately after it
                if ($ignoreFlag) {

                    while ($stripText =~ m/($response)/gi) {

                        my $offset = $-[0];             # Split at beginning of matching portion

                        if ($afterFlag) {

                            $offset += length($1) ;     # Split at end of matching portion
                        }

                        if (
                            $offset > 0
                            && $offset < length($stripText)
                            && ! exists $checkHash{$offset}
                        ) {
                            push (@offsetList, $offset);
                            $checkHash{$offset} = undef;
                            $count++;
                        }
                    }

                } else {

                    while ($stripText =~ m/($response)/g) {

                        my $offset = $-[0];             # Split at beginning of matching portion

                        if ($afterFlag) {

                            $offset += length($1) ;     # Split at end of matching portion
                        }

                        if (
                            $offset > 0
                            && $offset < length($stripText)
                            && ! exists $checkHash{$offset}
                        ) {
                            push (@offsetList, $offset);
                            $checkHash{$offset} = undef;
                            $count++;
                        }
                    }
                }

                # The trigger has fired only if both the stimulus and response match $stripText
                if ($count) {

                    # Temporary triggers should be marked for deletion, after firing for the first
                    #   time
                    if ($obj->ivShow('attribHash', 'temporary')) {

                        push (@deleteList, $obj);
                    }

                    # Should we continue checking other triggers?
                    if (! $obj->ivShow('attribHash', 'keep_checking')) {

                        # Don't check any more triggers
                        last OUTER;
                    }
                }
            }
        }

        # Any temporary triggers which fired can now be deleted
        foreach my $obj (@deleteList) {

            $self->removeInterface($obj);
        }

        # Sort the list of offsets, before returning it
        @sortedList = sort {$a <=> $b} (@offsetList);
        return @sortedList;
    }

    sub combineLineHashes {

        # Called by $self->processIncompleteLine and ->processEndLine, just before displayed
        #   calling $self->displayLine to display some text from a received line
        # If any text on this line has already been displayed, it is stored in $self->recvUsedText.
        #   If any tags have already been applied, they are stored in $self->recvUsedHash.
        # Update IVs, so that ->recvUsedText/->recvUsedHash contains everything that has been
        #   displayed added to anything we're about to display (in $self->recvLineText and
        #   ->recvLineHash)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (%recvLineHash, %recvUsedHash);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->combineLineHashes', @_);
        }

        if (! $self->recvUsedText && ! $self->recvUsedHash) {

            # Nothing from this line has been displayed yet
            $self->ivPoke('recvUsedText', $self->recvLineText);
            $self->ivPoke('recvUsedLength', $self->recvLineLength);
            $self->ivPoke('recvUsedHash', $self->recvLineHash);

        } else {

            # Text and/or colour/style tags on this line have already been displayed, so we need to
            #   merge IVs
            # Import some IVs for quick lookup
            %recvLineHash = $self->recvLineHash;
            %recvUsedHash = $self->recvUsedHash;

            foreach my $lineOffset (keys %recvLineHash) {

                my ($lineListRef, $usedListRef, $usedOffset);

                # Get the list of Axmud colour/style tags at this offset
                $lineListRef = $self->ivShow('recvLineHash', $lineOffset);
                # Find the offset, once the new text has been added to the old
                $usedOffset = $lineOffset + $self->recvUsedLength;

                if (exists $recvUsedHash{$usedOffset}) {

                    $usedListRef = $recvUsedHash{$usedOffset};
                    push (@$usedListRef, @$lineListRef);

                } else {

                    $recvUsedHash{$usedOffset} = $lineListRef;
                }
            }

            $self->ivPoke('recvUsedText', $self->recvUsedText . $self->recvLineText);
            $self->ivPoke('recvUsedLength', $self->recvUsedLength . $self->recvLineLength);
            $self->ivPoke('recvUsedHash', %recvUsedHash);
        }

        return 1;
    }

    sub updateEmergencyBuffer {

        # Called by $self->processIncomingData (for an incomplete escape sequence or an incomplete
        #   MXP tag) and by ->processIncompleteLine (for an unrecognised prompt in 'strict prompts'
        #   mode)
        # When the calling function tries to extract a token but fails because the token may be
        #   incomplete, this function is called
        # If the incomplete token is followed by a newline or escape character, we discard
        #   everything up to (but not including) that newline/escape character
        # Otherwise, we assume that the token has been split between two packets, only the first of
        #   which has been received. The token text is stored in an emergency buffer, which is added
        #   to the contents of the next packet received
        #
        # Expected arguments
        #   $text       - The remaining portion of the received text, starting with an incomplete
        #                   token
        #   $mode       - Set to 'escape' for incomplete escape sequences, 'mxp' for incomplete MXP
        #                   tags, 'prompt' for unrecognised prompts in 'strict prompts' mode
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns the modified $text. If there are no newline or escape characters,
        #       $text is converted to an empty string; otherwise, $text is stripped of everything
        #       before the first newline or escape character

        my ($self, $text, $mode, $check) = @_;

        # Local variables
        my ($modText, $thisPosn, $nlPosn, $escPosn);

        # Check for improper arguments
        if (! defined $text || ! defined $mode || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateEmergencyBuffer', @_);
        }

        # We're not interested in newline/escape characters at the beginning of $text, so don't
        #   test the first character
        $modText = substr($text, 1);

        # Now, look for the first newline/escape character in the rest of $text
        $nlPosn = index($modText, "\n");
        if ($nlPosn > -1) {

            $thisPosn = $nlPosn;
        }

        $nlPosn = index($modText, "\r");
        if ($nlPosn > -1 && (! defined $thisPosn || $nlPosn < $thisPosn)) {

            $thisPosn = $nlPosn;
        }

        # (Unrecognised prompts are allowed to contain escape characters, so don't look for them
        #   when $mode is 'prompt')
        if ($mode ne 'prompt') {

            $escPosn = index($modText, "\e");
            if ($escPosn > -1 && (! defined $thisPosn || $escPosn < $thisPosn)) {

                $thisPosn = $escPosn;
            }
        }

        if (defined $thisPosn) {

            # (Take account of the first character that was removed earlier)
            $thisPosn++;

            # Invalid, not incomplete, token. Discard everything up to the newline or escape
            #   character
            if ($mode eq 'mxp') {

                $self->mxpDebug(
                    '<',                            # Token unknown to this function
                    'Invalid token discarded',
                    5001,
                );
            }

            return (substr($text, $thisPosn));

        } else {

            # Incomplete token. Store everything in the emergency buffer
            if (defined $self->emergencyBuffer) {
                $self->ivPoke('emergencyBuffer', $self->emergencyBuffer . $text);
            } else {
                $self->ivPoke('emergencyBuffer', $text);
            }

            return '';
        }
    }

    sub writeIncomingDataLogs {

        # Called by $self->processLineSegment to write logs after each line segment (usually
        #   comprising a whole line) is received from the world, and after any matching triggers
        #   have fired)
        # NB $self->writeReceiveDataLog is used to write the 'receive' logfile; this function is
        #   used to write all other logfiles
        #
        # Expected arguments
        #   $modLine        - A segment of the received text, comprising some or all of a line of
        #                       text received from a world, which has now been stripped of non-text
        #                       tokens like newline characters, escape sequences, etc; and then
        #                       possibly modified after all matching triggers have fired
        #   $newLineFlag    - Flag set to TRUE if this line segment is to be treated as if it ends
        #                       with a newline character, FALSE if is to be treated as if it does
        #                       not end with a newline character
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $modLine, $newLineFlag, $check) = @_;

        # Local variables
        my @list;

        # Check for improper arguments
        if (! defined $modLine || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->writeIncomingDataLogs', @_);
        }

        # Write logs (if allowed)
        $axmud::CLIENT->writeLog(
            $self,
            TRUE,                           # Not world-specific logs
            $modLine,
            FALSE,                          # Don't precede with a newline character
            $newLineFlag,
            'main',                         # Write to these files
        );

        $axmud::CLIENT->writeLog(
            $self,
            FALSE,                          # World-specific logs
            $modLine,                       # After triggers fired
            FALSE,                          # Don't precede with a newline character
            $newLineFlag,
            'display',                      # Write to these files
        );

        # If we're recording lines in a separate logfile because of a character falling asleep,
        #   passing out or dying, do that now
        if ($self->logAsleepUntilLine) {

            @list = ($modLine);
            if ($self->displayBufferLast >= $self->logAsleepUntilLine) {

                $self->ivUndef('logAsleepUntilLine');
                push (@list, '--- (End of sleep record) ---');
            }

            foreach my $item (@list) {

                $axmud::CLIENT->writeLog(
                    $self,
                    FALSE,                          # World-specific logs
                    $item,
                    FALSE,                          # Don't precede with a newline character
                    $newLineFlag,
                    'sleep' ,                       # Write to these files
                );
            }
        }

        if ($self->logPassedOutUntilLine) {

            @list = ($modLine);
            if ($self->displayBufferLast >= $self->logPassedOutUntilLine) {

                $self->ivUndef('logPassedOutUntilLine');
                push (@list, '--- (End of passout record) ---');
            }

            foreach my $item (@list) {

                $axmud::CLIENT->writeLog(
                    $self,
                    FALSE,                          # World-specific logs
                    $item,
                    FALSE,                          # Don't precede with a newline character
                    $newLineFlag,
                    'passout',                      # Write to these files
                );
            }
        }

        if ($self->logDeadUntilLine) {

            @list = ($modLine);
            if ($self->displayBufferLast >= $self->logDeadUntilLine) {

                $self->ivUndef('logDeadUntilLine');
                push (@list, '--- (End of dead record) ---');
            }

            foreach my $item (@list) {

                $axmud::CLIENT->writeLog(
                    $self,
                    FALSE,                          # World-specific logs
                    $item,
                    FALSE,                          # Don't precede with a newline character
                    $newLineFlag,
                    'dead' ,                        # Write to these files
                );
            }
        }

        return 1;
    }

    sub writeReceiveDataLog {

        # Called by $self->processLinePortion to write the 'receive' logfile
        # Unlike other logfiles, lines written to 'receive' are not split into multiple lines by
        #   splitter triggers or by recognised command prompts
        #
        # Expected arguments
        #   $stripLine      - A segment of the received text, comprising some or all of a line of
        #                       text received from a world, which has now been stripped of non-text
        #                       tokens like newline characters, escape sequences, etc
        #   $imgLine        - The same line but with extra text added for any processed images
        #   $newLineFlag    - Flag set to TRUE if this line portion is to be treated as if it ends
        #                       with a newline character, FALSE if is to be treated as if it does
        #                       not end with a newline character
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $stripLine, $imgLine, $newLineFlag, $check) = @_;

        # Check for improper arguments
        if (! defined $stripLine || ! defined $imgLine || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->writeReceiveDataLog', @_);
        }

        # Write logs (if allowed)
        if ($axmud::CLIENT->logImageFlag) {

            $axmud::CLIENT->writeLog(
                $self,
                FALSE,                          # World-specific logs
                $imgLine,                       # Additional text representing processed images
                FALSE,                          # Don't precede with a newline character
                $newLineFlag,
                'receive',                      # Write to these files
            );

        } else {

            $axmud::CLIENT->writeLog(
                $self,
                FALSE,                          # World-specific logs
                $stripLine,                     # No text representing processed images
                FALSE,                          # Don't precede with a newline character
                $newLineFlag,
                'receive',                      # Write to these files
            );
        }

        return 1;
    }

    sub extractClickLinks {

        # Called by $self->processLineSegment
        # Before testing a received line against triggers, we need to see if there are any valid
        #   web URLs/email addresses and, if so, note their positions (so they can be used when the
        #   line is displayed in the current textview object)
        #
        # The calling function uses a hash of Axmud colour/style tags, in the form
        #   $tagHash{position} = reference_to_a_list_of_Axmud_colour_and_style_tags
        # ...where $position is the position in the line segment at which the colour/style tags
        #   apply (the first character is position 0)
        #
        # For any URLs/email addresses found, adds 'link' and 'link_off' style tags to the hash
        #
        # Expected arguments
        #   $stripText  - A segment of the received text, comprising some or all of a line of text
        #                   received from a world, which has now been stripped of non-text tokens
        #                   like newline characters, escape sequences, etc
        #   $tagHashRef - Reference to the hash of Axmud colour/style tags described above (which
        #                   this function may modify)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $stripText, $tagHashRef, $check) = @_;

        # Local variables
        my ($urlRegex, $emailRegex);

        # Check for improper arguments
        if (! defined $stripText || ! defined $tagHashRef || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->extractClickLinks', @_);
        }

        # (Don't bother checking for URLs, if there is no command set to open an external web
        #   browser)
        if ($axmud::CLIENT->browserCmd) {

            # Import the regexes for recognising URLs
            $urlRegex = $axmud::CLIENT->constUrlRegex;

            while ($stripText =~ m/($urlRegex)/gi) {

                my ($match, $start, $stop, $listRef);

                $match = $1;
                $start = length ($`);
                $stop = $start + length($&);

                # Ignore URLS which already occur between two 'link' tags
                if ($self->checkLinkTags($tagHashRef, $start, $stop)) {

                    if (exists $$tagHashRef{$start}) {

                        $listRef = $$tagHashRef{$start};
                        push (@$listRef, 'link');

                    } else {

                        $$tagHashRef{$start} = ['link'];
                    }

                    if (exists $$tagHashRef{$stop}) {

                        $listRef = $$tagHashRef{$stop};
                        push (@$listRef, 'link_off');

                    } else {

                        $$tagHashRef{$stop} = ['link_off'];
                    }
                }
            }
        }

        # (Don't bother checking for email addresses, if there is no command set to open an external
        #   email application)
        if ($axmud::CLIENT->emailCmd) {

            # Import the regexes for recognising email addresses
            $emailRegex = $axmud::CLIENT->constEmailRegex;

            while ($stripText =~ m/($emailRegex)/gi) {

                my ($match, $start, $stop, $listRef);

                $match = $1;
                $start = length ($`);
                $stop = $start + length($&);

                # Ignore email addresses which already occurs between two 'link' tags
                if ($self->checkLinkTags($tagHashRef, $start, $stop)) {

                    if (exists $$tagHashRef{$start}) {

                        $listRef = $$tagHashRef{$start};
                        push (@$listRef, 'link');

                    } else {

                        $$tagHashRef{$start} = ['link'];
                    }

                    if (exists $$tagHashRef{$stop}) {

                        $listRef = $$tagHashRef{$stop};
                        push (@$listRef, 'link_off');

                    } else {

                        $$tagHashRef{$stop} = ['link_off'];
                    }
                }
            }
        }

        return 1;
    }

    sub checkLinkTags {

        # Called by $self->extractClickLinks when a URL or email address is found in some received
        #   text
        #
        # The calling function was supplied with a hash of Axmud colour/style tags, in the form
        #   $tagHash{position} = reference_to_a_list_of_Axmud_colour_and_style_tags
        # ...where $position is the position in the line segment at which the colour/style tags
        #   apply (the first character is position 0)
        #
        # This function checks the position of the URL/email address. If the whole URL/email address
        #   occurs outside of two matching 'link' tags, then it's safe to add new 'link' tags
        # e.g. ________EMAIL_______<link_tag>URL<link_off_tag>______    < We can add new tags
        # e.g. ________<link_tag>...EMAIL...<link_off_tag>__________    < We can't add new tags
        #
        # Expected arguments
        #   $tagHashRef     - Reference to the hash of Axmud colour/style tags described above
        #   $start, $stop   - Offsets for the beginning/end of the newly-found URL or email address
        #                       ($start is 0, if the URL/email address occurs at the start of the
        #                       line)
        #
        # Return values
        #   'undef' on improper arguments or if the URL/email address occurs between two matching
        #       'link' tags
        #   1 if it's safe to create two new matching 'link' tags

        my ($self, $tagHashRef, $start, $stop, $check) = @_;

        # Local variables
        my (
            $prevFlag, $thisStart, $thisStop, $thisOffset,
            @onOffsetList, @offOffsetList, @sortedList
        );

        # Check for improper arguments
        if (! defined $tagHashRef || ! defined $start || ! defined $stop || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkLinkTags', @_);
        }

        # Get a list of offsets at which 'link' and 'link_off' tags occur
        foreach my $offset (keys %$tagHashRef) {

            my $listRef = $$tagHashRef{$offset};

            if (grep m/link/, @$listRef) {

                push (@onOffsetList, $offset);
            }

            if (grep m/link_off/, @$listRef) {

                push (@offOffsetList, $offset);
            }
        }

        if (@onOffsetList || @offOffsetList) {

            # Sort the lists
            if (@onOffsetList) {

                @onOffsetList = sort {$a <=> $b} (@onOffsetList);
            }

            if (@offOffsetList) {

                @offOffsetList = sort {$a <=> $b} (@offOffsetList);
            }

            do {

                my ($onOffset, $offOffset, $thisFlag);

                $onOffset = $onOffsetList[0];
                $offOffset = $offOffsetList[0];

                # Remove 'link' and 'link_off' tags, one by one, in the order in which they occured
                if (defined $onOffset && defined $offOffset && $offOffset < $onOffset) {

                    $thisOffset = $offOffset;
                    $thisFlag = FALSE;              # 'link_off'
                    shift @offOffsetList;

                } elsif (defined $onOffset) {

                    $thisOffset = $onOffset;
                    $thisFlag = TRUE;               # 'link'
                    shift @onOffsetList;

                } elsif (defined $offOffset) {

                    $thisOffset = $offOffset;
                    $thisFlag = FALSE;              # 'link_off'
                    shift @offOffsetList;
                }

                # Special case: the first tag found should have been a 'link' tag, but if it was a
                #   'link_off' tag, implying that a link has been spread over two lines, then the
                #   URL/email address can't be found before it
                if (! defined $prevFlag && ! $thisFlag) {

                    if ($start <= $thisOffset) {

                        return undef;
                    }

                } else {

                    # 'link' tag found
                    if ($thisFlag) {

                        # Wait for the corresponding 'link_off'. If the previously found tag was
                        #   also 'link', ignore the new tag (two consecutive 'link' tags shouldn't
                        #   happen, but just in case, that's what we'll do)
                        if (! defined $prevFlag) {

                            $thisStart = $thisOffset;
                            $thisStop = undef;
                        }

                    # 'link_off' flag found
                    } else {

                        $thisStop = $thisOffset;
                        # Does any part of the URL/email address fall between the 'link' and
                        #   'link_off' tags?
                        if (
                            ($start >= $thisStart && $start <= $thisStop)
                            || ($stop >= $thisStart && $stop <= $thisStop)
                        ) {
                            # Either the beginning or the end of the URL/email address occurs
                            #   between two matching 'link'/'link_off' tags
                            return undef;

                        } else {

                            # Move on to the next two matching 'link'/'link_off' tags
                            $thisStart = undef;
                            $thisStop = undef;
                        }
                    }
                }

                $prevFlag = $thisFlag;

            } until (! @onOffsetList && ! @offOffsetList);

            # If the last tag found was a 'link' tag, then the URL/email address must not occur
            #   after it
            if (defined $thisStart && ! defined $thisStop && $start >= $thisOffset) {

                return undef;
            }
        }

        # The URL/email address doesn't occur between two matching 'link' tags
        return 1;
    }

    sub checkSuppressLine {

        # Called by $self->processEndLine when processing a newline character at the end of an
        #   empty line (i.e. one which contains no text tokens, or only text tokens consisting of
        #   whitespace)
        # This function assumes that the current world's ->suppressEmptyLineCount is set to an
        #   integer greater than 1 (other values are dealt with by the calling function)
        # Any number of consecutive empty lines below this value are preserved, but any number of
        #   consecutive empty lines at or above this value are suppresssed (ignored)
        # e.g. Set to 3; the first two consecutive empty lines are preserved, but the 3rd, 4th and
        #   5th are suppressed
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the line currently being processed by
        #       ->processLineSegment shouldn't be suppressed
        #   1 if the line should be suppressed

        my ($self, $check) = @_;

        # Local variables
        my ($line, $count, $target);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkSuppressLine', @_);
        }

        # If the display buffer is empty, there are no previous lines to check
        if (! $self->displayBufferCount) {

            return undef;
        }

        $line = $self->displayBufferLast;
        $count = 1;     # The line being processed is itself empty
        $target = $self->currentWorld->suppressEmptyLineCount;

        do {

            my $bufferObj;

            $count++;

            $bufferObj = $self->ivShow('displayBufferHash', $line);
            if (! $bufferObj->emptyFlag) {

                # Not enough consecutive empty lines; don't suppress the current empty line
                return undef;

            } else {

                # On the next DO loop, check the previous line in the buffer
                $line--;
            }

        } until ($count >= $target || $line < $self->displayBufferFirst);

        # There are enough consecutive empty lines; suppress the current empty line
        return 1;
    }

    sub processPrompt {

        # Called by $self->spinMaintainLoop after receiving some text that looks like a prompt, and
        #   after the waiting time has expired
        # Also called by $self->optCallback after a TELOPT_EOR is received from the server
        # Processes the next part of an automatic login, if necessary, and causes the 'prompt'
        #   hook event
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $pattern, $initChar, $initAccount, $initPass,
            @loginCmdList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->processPrompt', @_);
        }

        $self->ivIncrement('promptCount');

        # In login mode 'lp', 'world_cmd' and 'telnet', we're waiting for prompts so that we can do
        #   an automatic login
        if ($self->loginPromptsMode ne 'none') {

            # Mode 'lp' - LP/Diku/AberMUD login (consecutive prompts for character/password)
            if ($self->loginPromptsMode eq 'lp') {

                if ($self->promptCount == 1) {

                    # Send the character name
                    $self->worldCmd($self->initChar);

                } elsif ($self->promptCount == 2) {

                    # Send the password. The second argument is the substring in the first argument
                    #   which should be obscured (the whole string, in this case)
                    $self->worldCmd($self->initPass, $self->initPass);
                    # Wait for login success patterns (if there are any), otherwise mark the
                    #   character as logged in
                    $self->setLoginPatterns();
                }

            # Mode 'world_cmd' - send a sequence of world commands at the first prompt
            } elsif ($self->loginPromptsMode eq 'world_cmd' && $self->promptCount == 1) {

                $self->processCmdLoginMode();

            # Mode 'telnet' - basic telnet login (e.g. 'login:' 'password:')
            } elsif ($self->loginPromptsMode eq 'telnet' && $self->loginPromptPatternList) {

                $pattern = $self->ivShift('loginPromptPatternList');
                if ($self->promptStripLine =~ m/$pattern/i) {

                    if ($self->loginPromptPatternList) {

                        # Send the character name
                        $self->worldCmd($self->initChar);

                    } else {

                        # Send the password. The second argument is the substring in the first
                        #   argument which should be obscured (the whole string, in this case)
                        $self->worldCmd($self->initPass, $self->initPass);

                        # Wait for login success patterns (if there are any), otherwise mark the
                        #   character as logged in
                        $self->setLoginPatterns();
                    }

                } else {

                    # The pattern doesn't match this prompt. Re-insert it into the list, so that
                    #   it can be tested against the next prompt
                    $self->ivUnshift('loginPromptPatternList', $pattern);
                }
            }
        }

        # Fire any hooks that are using the 'login' hook event
        $self->checkHooks('prompt', $self->promptStripLine);

        # Cancel the prompt
        $self->ivUndef('promptLine');
        $self->ivUndef('promptStripLine');
        $self->ivUndef('promptCheckTime');

        return 1;
    }

    # Incoming data loop - misc MXP/Pueblo stuff

    sub setMxpLineMode {

        # Called by $self->processEndLine, ->processEscSequence and ->checkMxpSecureMode
        # Sets a new value for $self->mxpLineMode
        # Also closes any open tags when open mode changes to secure mode (and vice versa)
        #
        # Expected arguments
        #   $newMode   - The new MXP line mode (a value in the range 0-2)
        #
        # Optional arguments
        #   $tempFlag    - Used when turning on/off temp secure mode. When TRUE, this function
        #                   doesn't call $self->emptyMxpStack; FALSE (or 'undef') when
        #                   $self->mxpLineMode is being set for any other reason
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of equivalent Axmud colour/style tags, when required (may be
        #       an empty list)

        my ($self, $newMode, $tempFlag, $check) = @_;

        # Local variables
        my (@emptyList, @tagList);

        # Check for improper arguments
        if (! defined $newMode || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->setMxpLineMode', @_);
            return @emptyList;
        }

        if (
            ! $tempFlag
            && (
                ($self->mxpLineMode == 1 && $newMode == 0)
                || ($self->mxpLineMode == 0 && $newMode == 1)
            )
        ) {
            # Switching between open/secure mode closes any open tags
            @tagList = $self->emptyMxpStack();
        }

        # Update the IV
        $self->ivPoke('mxpLineMode', $newMode);

        return @tagList;
    }

    sub convertMxpColour {

        # Called by $self->processMxpElement to convert the colours in a <COLOR> or <FONT>
        #   construction, in the form
        #       <COLOR FORE=foreground [BACK=background]>
        #       <FONT FACE=name [SIZE=size] [COLOR=foreground] [BACK=background]>
        # Also called by $self->processMxpGaugeElement for <GAUGE> tags
        #
        # 'foreground' and 'background' can be standard HTML colours (keys in
        #   GA::Client->constHtmlColourHash), RGB colour tags (in the form '#000000')
        # http://www.zuggsoft.com/zmud/mxp.htm states that we can use 'color attribute names such as
        #   blink', but neglects to specify WHICH colour attributes we can use - so 'blink' is the
        #   only one Axmud implements. If used, Axmud expects that 'foreground'/'background' will be
        #   in the form 'colour,blink' or 'blink,colour'
        #
        # This function converts a valid HTML colour to an RGB colour tag, in the form '#FFFFFF'
        #
        # Expected arguments
        #   $colour         - From the constructions above, the colour to convert: 'foreground' or
        #                       'background'
        #   $underlayFlag   - Flag set to TRUE for 'BACK=background' arguments, FALSE for
        #                       'COLOUR=foreground' arguments
        #
        # Return values
        #   An empty list on improper arguments or if $colour is invalid
        #   Otherwise, returns a list in the form
        #       (rgb_colour, blink_flag)
        #   ...where 'blink_flag' is TRUE or FALSE

        my ($self, $colour, $underlayFlag, $check) = @_;

        # Local variables
        my (
            $rgb, $blinkFlag,
            @emptyList,
        );

        # Check for improper arguments
        if (! defined $colour || ! defined $underlayFlag || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->convertMxpColour', @_);
            return @emptyList;
        }

        # Axmud treats MXP/HTML colours as case-insensitive
        $colour = lc($colour);

        # Remove 'blink', if present
        if ($colour =~ s/^blink\,//) {
            $blinkFlag = TRUE;
        } elsif ($colour =~ s/\,blink$//) {
            $blinkFlag = TRUE;
        } else {
            $blinkFlag = FALSE;
        }

        # Convert standard HTML colours to RGB format
        if ($axmud::CLIENT->ivExists('constHtmlColourHash', $colour)) {

            $rgb = $axmud::CLIENT->ivShow('constHtmlColourHash', $colour);

        } elsif ($colour =~ m/^\#[A-Fa-f0-9]{6}$/) {

            $rgb = $colour;

        } else {

            # Invalid colour
            return @emptyList;
        }

        if ($underlayFlag) {

            $rgb = 'u' . $rgb;
        }

        return ($rgb, $blinkFlag);
    }

    sub popMxpStack {

        # Called by $self->processMxpModalElement, ->processMxpLinkElement, ->processMxpSendElement,
        #   ->processMxpCustomElement or ->emptyMxpStack
        # After a modal tag (like <B> or <FONT>) is processed, a GA::Mxp::StackObj object is
        #   created, recording the then-current state of the current textview object's
        #   ->mxpModalStackHash
        # When the matching closing tag (</B> or </FONT>) is processed, we go through the current
        #   textview object's ->mxpModalStackList, 'popping' objects until we find one with the
        #   right keyword
        # Then we create a list of Axmud colour/style tags that restore the text attributes that
        #   applied when the original opening tag was processed
        # In this way, we can nest text attributes (but only those created with MXP)
        #
        # This functions updates the current textview's IVs, removing the popped stack object(s), if
        #   the operation is successful
        #
        # Expected arguments
        #   $keyword        - The closing tag's keyword, e.g. 'B' or 'FONT'
        #
        # Optional arguments
        #   $noCloseFlag    - Set to TRUE when called by $self->processMxpLinkElement or
        #                       ->processMxpSendElement, in which case those functions don't need to
        #                       be called back to close a <A>..</A> construction. Set to FALSE (or
        #                       'undef') in all other situations
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of Axmud colour/style tags restoring the text attributes that
        #       applied when the original opening tag was processed

        my ($self, $keyword, $noCloseFlag, $check) = @_;

        # Local variables
        my (
            $textViewObj, $stackObj, $fontFlag, $fontTag, $elementObj, $text, $flagArg, $entityObj,
            @emptyList, @stackList, @tagList,
            %sessionStackHash, %objStackHash,
        );

        # Check for improper arguments
        if (! defined $keyword || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->popMxpStack', @_);
            return @emptyList;
        }

        # Import the current textview object (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;
        # Import the stack of MXP modal tags
        @stackList = $textViewObj->mxpModalStackList;
        if (! @stackList) {

            $self->mxpDebug(
                '</' . uc($keyword) . '>',
                'Encountered closing \'</' . uc($keyword) . '>\' tag without matching opening'
                . ' tag',
                5002,
            );

            return @emptyList;
        }

        do {

            my $thisObj = pop @stackList;

            # GA::Obj::TextView->mxpModalStackList doesn't account for the following elements:
            #   <A>, <SEND>
            # While popping the stack, if we encounter one of those elements, we must close them not
            #   by modifying the textview object's ->mxpModalStackList, but by calling another
            #   function directly
            # The TRUE argument means 'don't call $self->popMxpStack as you normally would'. If this
            #   function was called by $self->processMxpSendElement (etc), $noCloseFlag is TRUE, so
            #   we don't need to call that function again
            # Other modal elements are closed entirely by the code in this function
            if (! $noCloseFlag) {

                if ($thisObj->keyword eq 'A') {

                    if ($self->mxpMode eq 'client_agree') {

                        push (@tagList,
                            $self->processMxpLinkElement(
                                '</A>',
                                'close',
                                'A',
                                TRUE,
                            ),
                        );

                    } elsif ($self->puebloMode eq 'client_agree') {

                        push (@tagList,
                            $self->processPuebloLinkElement(
                                '</A>',
                                'close',
                                'A',
                                TRUE,
                            ),
                        );
                    }

                } elsif ($thisObj->keyword eq 'SEND') {

                    push (@tagList,
                        $self->processMxpSendElement(
                            '</SEND>',
                            'close',
                            'SEND',
                            TRUE,
                        ),
                    );
                }
            }

            if ($thisObj->keyword eq $keyword) {

                # Matching opening tag found
                $stackObj = $thisObj;
            }

        } until ($stackObj || ! @stackList);

        if (! $stackObj) {

            $self->mxpDebug(
                '</' . uc($keyword) . '>',
                'Encountered closing \'</' . uc($keyword) . '>\' tag without matching opening'
                . ' tag',
                5003,
            );

            return @emptyList;
        }

        # Matching opening tag found. Import the current textview object's current MXP text
        #   attributes (for quick comparison)
        %sessionStackHash = $textViewObj->mxpModalStackHash;
        # ...and the stored MXP text attributes (those in use, before the original opening tag)
        %objStackHash = $stackObj->stackHash;

        # Check each key-value pair in turn. If they are different, add an Axmud colour/style tag to
        #   the return list, which will re-created conditions that existed before the original
        #   opening tag
        # NB The keys in $stackHash don't have the same name as corresponding Axmud colour/style
        #   tags
        if ($sessionStackHash{'bold_flag'} ne $objStackHash{'bold_flag'}) {

            $fontFlag = TRUE;
        }

        if ($sessionStackHash{'italics_flag'} ne $objStackHash{'italics_flag'}) {

            if (! $objStackHash{'italics_flag'}) {
                push (@tagList, 'italics_off');
            } else {
                push (@tagList, 'italics');
            }
        }

        if ($sessionStackHash{'underline_flag'} ne $objStackHash{'underline_flag'}) {

            if (! $objStackHash{'underline_flag'}) {
                push (@tagList, 'underline_off');
            } else {
                push (@tagList, 'underline');
            }
        }

        if ($sessionStackHash{'strike_flag'} ne $objStackHash{'strike_flag'}) {

            if (! $objStackHash{'strike_flag'}) {
                push (@tagList, 'strike_off');
            } else {
                push (@tagList, 'strike');
            }
        }

        if ($sessionStackHash{'colour_foreground'} ne $objStackHash{'colour_foreground'}) {

            if (! $objStackHash{'colour_foreground'}) {
                push (@tagList, $self->session->currentTabObj->textViewObj->textColour);
            } else {
                push (@tagList, $objStackHash{'colour_foreground'});
            }
        }

        if ($sessionStackHash{'colour_background'} ne $objStackHash{'colour_background'}) {

            if (! $objStackHash{'colour_background'}) {
                push (@tagList, $self->session->currentTabObj->textViewObj->underlayColour);
            } else {
                push (@tagList, $objStackHash{'colour_background'});
            }
        }

        # (<BOLD> and <HIGH> are implemented the same way)
        if ($sessionStackHash{'high_flag'} ne $objStackHash{'high_flag'}) {

            $fontFlag = TRUE;
        }

        if ($sessionStackHash{'font_name'} ne $objStackHash{'font_name'}) {

            $fontFlag = TRUE;
        }

        if ($sessionStackHash{'font_size'} ne $objStackHash{'font_size'}) {

            $fontFlag = TRUE;
        }

        if ($sessionStackHash{'blink_flag'} ne $objStackHash{'blink_flag'}) {

            if (! $objStackHash{'blink_flag'}) {
                push (@tagList, 'blink_off');
            } else {
                push (@tagList, 'blink_slow');
            }
        }

        # Update the current textview object's IVs
        $textViewObj->set_mxpModalStackList(@stackList);
        $textViewObj->set_mxpModalStackHash($stackObj->stackHash);

        if ($fontFlag) {

            # Create a dummy style tag to describe the changed font/font size (including bold
            #   on/off)
            push (@tagList, $self->createMxpFontTag($stackObj->stackHash));
        }

        # For custom elements, get the matching element object
        $elementObj = $self->ivShow('mxpElementHash', lc($stackObj->keyword));
        if ($elementObj && $elementObj->flagArg) {

            # We have been storing the text between two matching custom elements, e.g. from the MXP
            #   spec, <RName>...</RName>, which define a tag property
            # Stop storing text for this tag property, and move it to ->mxpFlagTextStoreHash, ready
            #   to be stored in the display buffer at the next opportunity
            $text = $self->ivShow('mxpFlagTextHash', $elementObj->flagArg);
            if (defined $text) {

                # (Guard against possibility that two <RName>...</RName> constructions appear on the
                #   same line)
                if (! $self->ivShow('mxpFlagTextStoreHash', $elementObj->flagArg)) {

                    $self->ivAdd('mxpFlagTextStoreHash', $elementObj->flagArg, $text);

                } else {

                    $self->ivAdd(
                        'mxpFlagTextStoreHash',
                        $elementObj->flagArg,
                        $self->ivShow('mxpFlagTextStoreHash', $elementObj->flagArg) . $text,
                    );
                }

                # Of the six standard tag properties, only one ('Set xxx') must be implemented
                #   immediately
                # ('Prompt' is handled by $self->processLineSegment, and 'RoomName' etc are handled
                #   by the Locator task)
                $flagArg = $elementObj->flagArg;
                if ($flagArg =~ m/^Set\s+\w/) {

                    $flagArg =~ s/^Set\s+//;

                    # Find the matching entity object. If it doesn't exist, don't create it
                    $entityObj = $self->ivShow('mxpEntityHash', $elementObj->name);
                    if ($entityObj) {

                        $entityObj->ivPoke('value', $text);
                        # Mark any corresponding gauges to be updated
                        $self->ivAdd('mxpGaugeUpdateHash', $elementObj->name, undef);
                    }
                }
            }

            $self->ivDelete('mxpFlagTextHash', $elementObj->flagArg);
        }

        # Operation complete
        return @tagList;
    }

    sub emptyMxpStack {

        # Called by $self->processEndLine, ->processEscChar, etc
        # Empties the current textview's stack of GA::Mxp::StackObj objects by calling
        #   $self->popMxpStack for each one in turn
        # This has the effect of closing all open MXP tags
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of Axmud colour/style tags restoring the text attributes that
        #       applied before any of the existing open tags were processed

        my ($self, $check) = @_;

        # Local variables
        my (
            $textViewObj,
            @emptyList, @tagList,
        );

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->emptyMxpStack', @_);
            return @emptyList;
        }

        # Import the current textview (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;
        if (! $textViewObj->mxpModalStackList) {

            # There are no open MXP tags
            return @emptyList;
        }

        # Otherwise, pop stack objects one by one, and in reverse order
        do {

            my $stackObj = $textViewObj->ivIndex(
                'mxpModalStackList',
                $textViewObj->ivLast('mxpModalStackList'),
            );

            push (@tagList, $self->popMxpStack($stackObj->keyword));

        } until (! $textViewObj->mxpModalStackList);

        # Operation complete
        return @tagList;
    }

    sub findMxpArgsByPosn {

        # From http://www.mushclient.com/mushclient/mxp.htm :
        #
        # Arguments may be by keyword or positional. If by keyword the syntax is:
        #   argument_name=argument_value
        # If no argument name is provided then the argument is assumed to be the next argument by
        #   position from the previous argument, or if no previous argument the first argument. This
        #   means, that following a keyword argument, the next argument that does not have a keyword
        #   is now considered to be the argument in sequence after the keyword. Thus you could use a
        #   single keyword argument to "jump" to the middle of an argument list.
        #
        # This function takes a single 'name=value' construction (or a 'name' construction, or a
        #   'value' construction whose corresponding name can be inferred by position) and works out
        #   the compulsory 'name' part and the optional 'value' part
        # If an argument 'name' is found, everything up to and including that 'name' is removed from
        #   %$checkListRef so that, if the next call to this function is a 'value' construction, we
        #   can work out the 'name' by using the first remaining item in %$checkListRef (this is
        #   'arguments by position')
        # If the world specifies argument 'names' in the wrong order, resets $checkListRef so the
        #   current and subsequent calls to the function work as intended
        #
        # Expected arguments
        #   $origListRef    - Reference to a list of argument 'name's in the order they're
        #                       expected; e.g. for <SEND>, this list will be
        #                           ('href', 'hint', 'prompt', 'expire')
        #                   - This list is not modified during successive calls to this function
        #   $checkListRef   - Reference to a list of argument 'name's in the order they're
        #                       expected; e.g. for <SEND>, this list will be
        #                           ('href', 'hint', 'prompt', 'expire')
        #                   - Argument 'name's are removed from this list, as they are processed
        #                   - If this list is empty, it is reset using the contents of $origListRef
        #   $ivHashRef      - Reference to a hash of argument 'name's, stored as keys, and their
        #                       default values, e.g. for <SEND>, will contain
        #                           $ivHash{'href'} = undef,
        #                           $ivHash{'prompt'} = FALSE, (...and so on)
        #   $checkHashRef   - Reference to a hash of argument 'name's which never take a
        #                       corresponding value, those argument names stored as keys in the
        #                       hash; e.g. for <SEND>, will contain a single key-value pair:
        #                           $checkHash{'prompt'} = undef
        #   $argName        - An argument name or value to process...
        #
        # Optional argumements
        #   $argValue       - ...for 'name=value' constructions, both $argName and $argValue will be
        #                       set. For 'name' or 'value' constructions, $argName will be set but
        #                       $argValue will be 'undef'
        #
        # Return Values
        #   An empty list on improper arguments or if there's an error
        #   Otherwise returns a list in the form (name, value), where 'name' is set and 'value' may
        #       be either set or 'undef'

        my (
            $self, $origListRef, $checkListRef, $ivHashRef, $checkHashRef, $argName, $argValue,
            $check,
        ) = @_;

        # Local variables
        my (
            $matchFlag,
            @emptyList,
        );

        # Check for improper arguments
        if (
            ! defined $origListRef || ! defined $checkListRef || ! defined $ivHashRef
            || ! defined $checkHashRef || ! defined $argName || defined $check
        ) {
            $axmud::CLIENT->writeImproper($self->_objClass . '->findMxpArgByPosn', @_);
            return @emptyList;
        }


        # If the world specifies argument 'names' in the wrong order, reset $checkListRef so the
        #   current and subsequent calls to the function work as intended
        if (exists $$ivHashRef{lc($argName)}) {

            OUTER: foreach my $item (@$checkListRef) {

                if ($item eq lc($argName)) {

                    $matchFlag = TRUE;
                    last OUTER;
                }
            }

            if (! $matchFlag) {

                @$checkListRef = (@$origListRef);
            }
        }

        # If $argValue is 'undef', then it's not a 'name=value' construction but a 'value'
        #   construction. In this situation, the argument 'name' is the first remaining item in
        #   @$checkListRef
        # Exception: if 'name' exists in %$checkHashRef, then it's a 'name' construction (with no
        #   corresponding argument value)
        if (! defined $argValue && ! exists $$checkHashRef{lc($argName)}) {

            # 'value' construction
            $argValue = $argName;
            $argName = shift @$checkListRef;

            return ($argName, $argValue);
        }

        # Otherwise, it's a 'name' or a 'name=value' construction.
        # We must remove everything in @$checkListRef up to and including the matching argument
        #   name so that, if the next argument is not a name=value construction, we can set the
        #   'name' part to the following item in @$checkListRef
        $argName = lc($argName);
        do {

            my $item = shift @$checkListRef;

            if ($item eq $argName) {

                return ($argName, $argValue);
            }

        } until (! @$checkListRef);

        # Ran out of argument names, the calling function will report an error
        return @emptyList;
    }

    sub deleteMxpAttrib {

        # Called by $self->processMxpSupportElement
        # Updates a copy of GA::Client->constMxpAttribHash, which is in the form
        #   $tagHash{mxp_tag} = reference_to_list_of_tab_attributes
        # Removes an attribute from the list corresponding to a specified tag, and returns the
        #   updated hash
        #
        # Expected arguments
        #   $tag        - An MXP tag in lower-case (e.g. 'frame')
        #   $attrib     - One of the tag's attributes, also in lower-case (e.g. 'internal')
        #
        # Optional arguments
        #   %tagHash    - The hash to update. If the specified tag and/or attribute don't exist
        #                   in the hash, it is returned unmodified
        #
        # Return values
        #   An empty hash on improper arguments
        #   Otherwise, returns the modified hash

        my ($self, $tag, $attrib, %tagHash) = @_;

        # Local variables
        my (
            $listRef,
            @newList,
            %emptyHash,
        );

        # Check for improper arguments
        if (! defined $tag || ! defined $attrib) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->deleteMxpAttrib', @_);
            return %emptyHash;
        }

        # If the specified tag doesn't exist (has already been removed), make no changes
        if (! exists $tagHash{$tag}) {

            return %tagHash;
        }

        # Check the list corresponding to a specified tag and, if found, remove it
        $listRef = $tagHash{$tag};
        # MXP tags without attributes, such as <B>, appear as keys in %tagHash, but their
        #   corresponding value is 'undef' rather than a list reference
        if (defined $listRef) {

            foreach my $thisAttrib (@$listRef) {

                if ($thisAttrib ne $attrib) {

                    push (@newList, $thisAttrib);
                }
            }

            $tagHash{$tag} = \@newList;
        }

        return %tagHash;
    }

    sub createMxpFontTag {

        # Called by $self->processMxpModalElement, ->processMxpHtmlElement and ->popMxpStack
        # When a <FONT>...</FONT> construction is encountered, when both the opening and closing
        #   tags are processed, this function is called
        # Creates a dummy style tags in the form 'mxpf_monospace_bold_12' or 'mxpf_off'
        # $self->applyColourStyleTags extracts everything after the 'mxpf', and uses the extracted
        #   text to modify the 'mxp_font' Gtk2::TextTag
        #
        # Expected arguments
        #   %stackHash      - A hash of MXP text attributes and their settings, in the same format
        #                       as GA::Obj::TextView->mxpModalStackHash, but containing only
        #                       key-value pairs for changes that are about to be applied to
        #                       the current textview object's ->mxpModalStackHash
        #                   - e.g. { 'font_name' => 'Monospace', 'font_size' => '12' }
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns a dummy style tag in the forms 'mxpf_monospace_bold_12' or 'mxpf_off'

        my ($self, %stackHash) = @_;

        # Local variables
        my ($textViewObj, $tag);

        # Check for improper arguments
        if (! %stackHash) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->createMxpFontTag', @_);
        }

        # Import IVs (for convenience)
        $textViewObj = $self->currentTabObj->textViewObj;

        # Create the dummy style tag
        if (
            ! $stackHash{'font_name'}
            && ! $stackHash{'font_size'}
            && ! $stackHash{'bold_flag'}
        ) {
            $tag = 'mxpf_off';

        } else {

            $tag = 'mxpf';

            if ($stackHash{'font_name'}) {
                $tag .= '_' . $stackHash{'font_name'};
            } else {
                $tag .= '_' . $textViewObj->font;
            }

            # (<BOLD> and <HIGH> are implemented the same way)
            if ($stackHash{'bold_flag'} || $stackHash{'high_flag'}) {

                $tag .= '_bold';
            }

            if ($stackHash{'font_size'}) {
                $tag .= '_' . int($stackHash{'font_size'});
            } else {
                $tag .= '_' . $textViewObj->fontSize;
            }

            if ($stackHash{'spacing'}) {

                $tag .= '_p' . int($stackHash{'spacing'});
            }
        }

        return $tag;
    }

    sub checkMxpSecureMode {

        # Called by $self->processIncomingData just after a token is extracted from text received
        #   from the world
        # When temp secure mode is on, check whether it needs to be turned off and, if not,
        #   whether the next character in the stream is a "<" character
        #
        # Expected arguments
        #   $text       - Any remaining text from the world that's not been processed yet (may be
        #                   an empty string)
        #
        # Optional arguments
        #   $tempMode   - The value of $self->mxpTempMode, before the most recent token was
        #                   extracted from text received from the world (may be 'undef')
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list of equivalent Axmud colour/style tags, when required (may be
        #       an empty list)

        my ($self, $text, $tempMode, $check) = @_;

        # Local variables
        my (@emptyList, @tagList);

        # Check for improper arguments
        if (! defined $text || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->checkMxpSecureMode', @_);
            return @emptyList;
        }

        if (defined $tempMode) {

            # MXP temporary secure mode has expired with the token just processed by the calling
            #   function
            push (@tagList, $self->setMxpLineMode($self->mxpTempMode, TRUE));
            $self->ivUndef('mxpTempMode');

        } else {

            # MXP temporary secure mode was applied by the token just processed by the calling
            #   function. Check that the next character in the stream is the compulsory "<"
            #   character
            # (For consistency with other functions, we'll allow whitesace before the "<" character,
            #   but not newline characters, etc)
            if (! $text || ! ($text =~ m/^\s*\</)) {

                $self->mxpDebug('n/a', 'Temp secure mode not followed by a possible MXP tag', 5004);

                # Disable temp secure mode
                push (@tagList, $self->setMxpLineMode($self->mxpTempMode, TRUE));
                $self->ivUndef('mxpTempMode');
            }
        }

        return @tagList;
    }

    sub convertMxpWinSize {

        # Called by $self->processMxpFrameElement
        # When the world specifies a new frame using a <FRAME> tag, it can optionally specify the
        #   frame's size and position
        # Work out the equivalent size and position on the workspace, in pixels
        #
        # Expected arguments
        #   $frameObj   - The GA::Mxp::Frame object created in response to the <FRAME> tag
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, returns a list in the form (left, top, width, height)
        #   ...where 'left' and 'top' are the workspace coordinates of the top-left of the proposed
        #       window position, and 'width' / 'height' is the size of the window, all in pixels

        my ($self, $frameObj, $check) = @_;

        # Local variables
        my (
            $workspaceObj, $availableWidth, $availableHeight, $charWidth, $charHeight, $right,
            $bottom,
            @emptyList, @returnList,
        );

        # Check for improper arguments
        if (! defined $frameObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->convertMxpWinSize', @_);
        }

        # The frame is opened in the same workspace used by the session's 'main' window
        # The size of the available workspace is the current width and height, minus any space
        #   reserved for panels
        $workspaceObj = $self->mainWin->workspaceObj;
        $availableWidth = $workspaceObj->currentWidth - $workspaceObj->panelLeftSize
                            - $workspaceObj->panelRightSize;
        $availableHeight = $workspaceObj->currentHeight - $workspaceObj->panelTopSize
                            - $workspaceObj->panelBottomSize;

        # Get size of an 'X' character (because the MXP specification demands it)
        ($charWidth, $charHeight) = $self->getTextSize('X');

        # $frameObj->left (etc) can be in the form 'n%' (a percentage), 'nc' (a multiple of
        #   character widths/heights) or 'n' (a value in pixels), relative to the left or top of the
        #   available desktop
        # A minus value, i.e. '-n%', '-nc' or '-n' signifies that the value is relative to the
        #   right or bottom of the available desktop
        # If an invalid value was specified, use a default value
        foreach my $iv ('left', 'top', 'width', 'height') {

            my ($value, $minus, $num, $type, $newValue);

            $value = $frameObj->$iv;

            if ($value =~ m/(\-)?(\d+)([\%c]?)/) {

                $minus = $1;
                $num = $2;
                $type = $3;

                if (! $type) {

                    if ($iv eq 'left' || $iv eq 'width') {

                        if ($minus) {
                            $newValue = $availableWidth - $num;
                        } else {
                            $newValue = $num;
                        }

                    } else {

                        if ($minus) {
                            $newValue = $availableHeight - $num;
                        } else {
                            $newValue = $num;
                        }
                    }

                } elsif ($type eq '%') {

                    if ($iv eq 'left' || $iv eq 'width') {

                        if ($minus) {
                            $newValue = $availableWidth - ($availableWidth * ($num / 100));
                        } else {
                            $newValue = $availableWidth * ($num / 100);
                        }

                    } else {

                        if ($minus) {
                            $newValue = $availableHeight - ($availableHeight * ($num / 100));
                        } else {
                            $newValue = $availableHeight * ($num / 100);
                        }
                    }

                } elsif ($type eq 'c') {

                    if ($iv eq 'left' || $iv eq 'width') {

                        # (Take into account spacing around the grid window's strip and table
                        #   objects by using $axmud::CLIENT->constGridSpacingPixels; it's not
                        #   exact, but it's good enough)
                        $newValue = ($charWidth * $num) + $workspaceObj->controlsLeftSize
                                        + $workspaceObj->controlsRightSize
                                        + ($axmud::CLIENT->constGridSpacingPixels * 2);

                        if ($minus) {

                            $newValue = $availableWidth - $newValue;
                        }

                    } else {

                        $newValue = ($charHeight * $num) + $workspaceObj->controlsTopSize
                                        + $workspaceObj->controlsBottomSize
                                        + ($axmud::CLIENT->constGridSpacingPixels * 2);

                        if ($minus) {

                            $newValue = $availableHeight - $newValue;
                        }
                    }
                }

            } else {

                # Invalid value, so use a default value
                if ($iv eq 'left' || $iv eq 'top') {
                    $newValue = 0;
                } elsif ($iv eq 'width') {
                    $newValue = int($availableWidth / 2);
                } else {
                    $newValue = int($availableHeight / 2);
                }
            }

            push (@returnList, $newValue);
        }

        # Sanity checking, for the benefit of an MXP frame tag which tries to draw a window outside
        #   the bounds of the desktop

        # Left
        if ($returnList[0] < 0) {

            $returnList[0] = 0;
        }

        # Top
        if ($returnList[1] < 0) {

            $returnList[1] = 0;
        }

        # Width
        $right = $returnList[0] + $returnList[2];                   # left + width
        if ($right > $availableWidth) {

            $returnList[2] = $availableWidth - $returnList[0];      # total width - left
        }

        # Height
        $bottom = $returnList[1] + $returnList[3];                  # top + height
        if ($bottom > $availableHeight) {

            $returnList[3] = $availableHeight - $returnList[1];     # total height - top
        }

        return @returnList;
    }

    sub getMxpFrame {

        # Can be called by anything
        # Looks up the name of an MXP frame (implemented as a Frame task window) and returns the
        #   corresponding frame object
        # The special name '_previous' refers to $self->mxpPrevFrame, a frame in $self->mxpFrameHash
        #   that could have any name, so all code should call this function rather than looking up a
        #   frame in $self->mxpFrameHash directly
        #
        # Expected arguments
        #   $name   - An MXP frame name - one of the keys in $self->mxpFrameHash, or '_previous'
        #
        # Return values
        #   'undef' on improper arguments or if the name doesn't match an MXP frame object
        #   Otherwise returns the matching MXP frame object

        my ($self, $name, $check) = @_;

        # Check for improper arguments
        if (! defined $name || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->getMxpFrame', @_);
        }

        if ($name eq '_previous') {
            return $self->ivShow('mxpFrameHash', $self->mxpPrevFrame);
        } else {
            return $self->ivShow('mxpFrameHash', $name);
        }
    }

    sub removeMxpFrames {

        # Called by GA::Task::Frame->closeWin and ->del_winObj
        # If an MXP frame (implemented as a Frame task window) is closed, close all remanining
        #   frames (interior and exterior) and inform the world that frames are no longer supported
        # (In other words, if the user manually closes a Frame task window, there's no way of
        #   informing the world that one frame is not available but any others still are, so we
        #   have to stop using frames altogether)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeMxpFrames', @_);
        }

        # Close exterior frames (Frame task windows), but don't close interior frames (pane objects)
        foreach my $frameObj ($self->ivValues('mxpFrameHash')) {

            if ($frameObj->taskObj && $frameObj->taskObj->winObj) {

                $frameObj->taskObj->closeWin();
            }
        }

        # Reset IVs. Internal frames will continue to exist, but nothing will write to them
        $self->ivEmpty('mxpFrameHash');
        $self->ivUndef('mxpCurrentFrame');
        $self->ivUndef('mxpPrevFrame');
        $self->ivPoke('currentTabObj', $self->defaultTabObj);

        # Inform the world by processing a fake <SUPPORT> tag
        $self->ivPoke('mxpDisableFrameFlag', TRUE);
        $self->processMxpSupportElement(
            '<SUPPORT>',
            'open',
            'SUPPORT',
        );

        return 1;
    }

    sub convertMxpImageSize {

        # Called by $self->processMxpImageElement
        # When the world specifies an image using an <IMAGE> tag, it can optionally specify the
        #   image's width and height. These values can be expressed as in integers (the size in
        #   pixels), in the form 'nc' (size in characters) or 'n%' (size in percentage of the
        #   available space).
        # Convert a value (either width or height) to a size in pixels
        #
        # Expected arguments
        #   $value      - The value to convert (in the form 'n', 'nc' or 'n%')
        #   $mode       - 'width' to convert a width, 'height' to convert a height
        #
        # Return values
        #   'undef' on improper arguments or if $val isn't in the form 'n', 'nc' or 'n%'
        #   Otherwise, the converted size in pixels

        my ($self, $value, $mode, $check) = @_;

        # Local variables
        my ($num, $type, $charWidth, $charHeight, $rectObj);

        # Check for improper arguments
        if (! defined $value || ! defined $mode || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->convertMxpImageSize', @_);
        }

        if ($value =~ m/(\d+)([\%c]?)/) {

            $num = $1;
            $type = $2;

            if (! $type) {

                # $value was already in pixels
                return int($value);

            } elsif ($type eq 'c') {

                # Get size of an X character
                ($charWidth, $charHeight) = $self->getTextSize('X');
                if ($mode eq 'width') {

                    return int($num * $charWidth);

                } elsif ($mode eq 'height') {

                    return int($num * $charHeight);

                } else {

                    # Emergency default
                    return undef;
                }

            } else {

                # Get the size of the default tab's textview, in pixels
                # Convert the percentage into a fraction (e.g. convert 50% into 0.5)
                $num /= 100;

                # Get a Gtk2::Gdk::Rectangle
                $rectObj = $self->defaultTabObj->textViewObj->textView->get_visible_rect();
                if ($mode eq 'width') {

                    return int($num * $rectObj->width);

                } elsif ($mode eq 'height') {

                    return int($num * $rectObj->height);

                } else {

                    # Emergency default
                    return undef;
                }
            }

        } else {

            # Invalid image size format (not 'n', 'nc' or 'n%'
            return undef;
        }
    }

    sub updateMxpGauges {

        # Called by $self->spinMaintainLoop
        # When an MXP entity is modified (including being created or deleted), an entry is added
        #   to $self->mxpGaugeUpdateHash
        # Once per maintenance loop, this function is called. The function checks whether any of the
        #   modified entities has corresponding 'main' window gauges and, if so, updates the
        #   GA::Obj::Gauge objects and redraws the gauges
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $updateFlag,
            @deleteList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateMxpGauges', @_);
        }

        foreach my $entName ($self->ivKeys('mxpGaugeUpdateHash')) {

            my ($entityObj, $gaugeObj);

            # If the entity has been deleted, its gauge must be removed
            if (! $self->ivExists('mxpEntityHash', $entName)) {

                push (@deleteList, $entName);

            } else {

                # Update the corresponding gauge object, if it has actually been created
                $gaugeObj = $self->ivShow('mxpGaugeHash', $entName);
                if ($gaugeObj) {

                    $updateFlag = TRUE;
                    $entityObj = $self->ivShow('mxpEntityHash', $entName);

                    if ($gaugeObj->mxpEntity eq $entName) {
                        $gaugeObj->ivPoke('value', $entityObj->value);
                    } else {
                        $gaugeObj->ivPoke('maxValue', $entityObj->value);
                    }
                }
            }
        }

        if (@deleteList) {

            $self->removeMxpGauges(@deleteList);

        } elsif ($updateFlag) {

            $self->mxpGaugeStripObj->updateGauges();
        }

        # All corresponding gauges have been updated
        $self->ivEmpty('mxpGaugeUpdateHash');

        return 1;
    }

    sub removeMxpGauges {

        # Called by various functions
        # Removes MXP gauges created by $self->processMxpGaugeElement and updates IVs.
        # If there are no more gauges left on the gauge level allocated to MXP, the gauge level
        #   itself is also removed
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   @list       - A list of MXP entity names; any matching MXP gauges are removed. If it's
        #                   an empty list, ALL MXP gauges are removed
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, @list) = @_;

        # Local variables
        my @modList;

        # (No improper arguments to check)

        if ($self->mxpGaugeStripObj) {

            # If no entity names were specified, remove all MXP gauges
            if (! @list) {

                if ($self->mxpGaugeLevel) {

                    $self->mxpGaugeStripObj->removeGaugeLevel($self, $self->mxpGaugeLevel);
                }

                $self->ivUndef('mxpGaugeLevel');
                $self->ivEmpty('mxpGaugeHash');

            # Otherwise, compile a list of MXP gauges to remove. If entity names were specified,
            #   check corresponding MXP gauges actually exist.
            } else {

                foreach my $name (@list) {

                    if ($self->ivExists('mxpGaugeHash', $name)) {

                        push (@modList, $self->ivShow('mxpGaugeHash', $name));
                    }
                }

                if (@modList) {

                    # The FALSE argument means 'don't keep an empty gauge level'
                    $self->mxpGaugeStripObj->removeGauges($self, FALSE, @modList);

                    foreach my $obj (@modList) {

                        $self->ivDelete('mxpGaugeHash', $obj->number);
                    }

                    if (! $self->mxpGaugeHash) {

                        # All gauges have been removed
                        $self->mxpGaugeStripObj->removeGaugeLevel($self, $self->mxpGaugeLevel);
                        $self->ivUndef('mxpGaugeLevel');
                    }
                }
            }
        }

        return 1;
    }

    sub mxpDoRelocate {

        # Called by $self->incomingDataLoop
        # Initiaties an MXP crosslinking operation. Closes the current connection and opens a new
        #   one
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->mxpDoRelocate', @_);
        }

        # If auto-saves are turned on, do an auto-save now
        if ($axmud::CLIENT->autoSaveFlag && $self->autoSaveCheckTime) {

            # Perform the auto-save
            $self->pseudoCmd('save');
            $self->ivPoke('autoSaveLastTime', $self->sessionTime);
            # Set the time at which the next auto-save will occur
            $self->resetAutoSave();
        }

        # Update the connection info strip object for any 'internal' windows used by this session
        foreach my $winObj ($axmud::CLIENT->desktopObj->listSessionGridWins($self, TRUE)) {

            $winObj->setHostLabel(
                $self->getHostLabelText(),
                'MXP crosslinking operation in progress...',
            );
        }

        # Terminate the current connection
        $self->doTempDisconnect();
        $self->ivPoke('mxpRelocateMode', 'started');

        # Make sure all changes are visible immediately
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->dispatchCmd');

        # Intitiate the new connection
        if (! $self->doConnect($self->mxpRelocateHost, $self->mxpRelocatePort, $self->protocol)) {

            # Reconnection failed
            $self->doDisconnect();

        } else {

            $self->ivPoke('mxpRelocateMode', 'wait_login');
        }

        return 1;
    }

    sub applyMxpFileFilter {

        # Called by $self->processMxpImageElement and ->processMspSoundTrigger
        # Given a full file path, applies the MXP file filter
        # If the world has provided a plugin to convert an image/sound file its own format into a
        #   format supported by Axmud, call the plugin, which performs the conversion and returns
        #   the path to the converted file
        #
        # Expected arguments
        #   $path       - Full file path to the image/sound file to convert, e.g.
        #                   '/home/myname/axmud-data/deathmud/mxp/myimage.gff'
        #
        # Return values
        #   'undef' on improper arguments or if the file can't be converted
        #   Otherwise returns the file path to the converted file (which the calling function will
        #       delete, after it's used), e.g. '/home/myname/axmud-data/deathmud/mxp/myimage.gif'

        my ($self, $path, $check) = @_;

        # Local variables
        my ($file, $dir, $ext, $filterObj, $pluginObj, $funcRef);

        # Check for improper arguments
        if (! defined $path || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->applyMxpFileFilter', @_);
        }

        ($file, $dir, $ext) = File::Basename::fileparse($path, qr/\.[^.]*/);
        $ext =~ s/^\.//;

        # Has the world specified a file filter for this file extension?
        $filterObj = $self->ivShow('mxpFilterHash', $ext);
        if (! defined $filterObj) {

            # No file filter found, so ignore the file
            return undef;
        }

        # Does the named plugin exist, and is it enabled?
        $pluginObj = $axmud::CLIENT->ivShow('pluginHash', $filterObj->name);
        if (! defined $pluginObj || ! $pluginObj->enabledFlag) {

            # Plugin not available, so ignore the file
            return undef;
        }

        # Has the plugin informed the GA::Client of which function to call, and is it a valid
        #   function?
        $funcRef = $axmud::CLIENT->ivShow('pluginMxpFilterHash', $filterObj->name);
        if (! defined $funcRef || ref($funcRef) ne 'CODE') {

            # Conversion function not specified, so ignore the file
            return undef;
        }

        # Apply the file filter to convert the file from the world's own image format into a
        #   .gif or .bmp format
        $path = &$funcRef($path, $filterObj->src, $filterObj->dest, $filterObj->proc);
        if (! $path) {

            # Conversion failed, so ignore the file
            return undef;

        } else {

            return $path;
        }
    }

    sub mxpDebug {

        # Called by various functions
        # Stores an MXP debug message until $self->processIncomingData is ready to display it (by
        #   not displaying it immediately, we can avoid some very ugly Gtk2 errors)
        #
        # Expected arguments
        #   $token      - The MXP token that caused the error
        #   $msg        - The debug message
        #
        # Optional arguments
        #   $num       - An optional 4-digit error number, specified literally in the Axmud code
        #                   (could be set to 'undef' if we just need a quick, temporary message).
        #                   Currently, MXP errors use the range 1000-4999 and mixed MXP/Pueblo
        #                   errors use the range 5000-5999
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $token, $msg, $num, $check) = @_;

        # Check for improper arguments
        if (! defined $token || ! defined $msg || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->mxpDebug', @_);
        }

        # (Do nothing if the client flag is not set)
        if ($axmud::CLIENT->debugMxpFlag) {

            if (! defined $num) {

                $num = 9998;
            }

            $self->ivPush('mxpPuebloDebugList', 'mxp', $token, $num, $msg);
        }

        return 1;
    }

    sub puebloDebug {

        # Called by various functions
        # Stores an Pueblo debug message until $self->processIncomingData is ready to display it (by
        #   not displaying it immediately, we can avoid some very ugly Gtk2 errors)
        #
        # Expected arguments
        #   $token      - The MXP token that caused the error
        #   $msg        - The debug message
        #
        # Optional arguments
        #   $num       - An optional 4-digit error number, specified literally in the Axmud code
        #                   (could be set to 'undef' if we just need a quick, temporary message).
        #                   Currently, Pueblo errors use the range 6000-8999 and mixed MXP/Pueblo
        #                   errors use the range 5000-5999
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $token, $msg, $num, $check) = @_;

        # Check for improper arguments
        if (! defined $token || ! defined $msg || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->puebloDebug', @_);
        }

        # (Do nothing if the client flag is not set)
        if ($axmud::CLIENT->debugPuebloFlag) {

            if (! defined $num) {

                $num = 9999;
            }

            $self->ivPush('mxpPuebloDebugList', 'pueblo', $token, $num, $msg);
        }

        return 1;
    }

    # Incoming data loop - misc MSP stuff

    sub setPseudoMSP {

        # Called by GA::Cmd::MSP->do
        # Some worlds are not able to negotiate telnet options to enable MSP, but nevertheless
        #   are able to send MSP sound/music triggers to the client
        # Therefore we need a setting of $self->mspMode which means something like 'the server did
        #   not negotiate MSP, but Axmud is responding to MSP sound/music triggers'
        # This function is called to turn on/off pseudo-MSP recognition. The server is informed
        #   using IAC DONT MSP or IAC DO MSP, even if it doesn't seem to recognise those
        #   telnet options
        # NB Pseudo-MSP recognition can be turned on, even if the general setting
        #   (GA::Client->useMspFlag) is FALSE
        #
        # Expected arguments
        #   $flag   - Set to TRUE to turn on pseudo-MSP recognition, or FALSE to turn it off
        #
        # Return values
        #   'undef' on improper arguments or if full MSP or pseudo-MSP recognition is already on/off
        #   1 otherwise

        my ($self, $flag, $check) = @_;

        # Local variables
        my %telConstHash;

        # Check for improper arguments
        if (! defined $flag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setPseudoMSP', @_);
        }

        # Import the hash of telnet constants (for convenience)
        %telConstHash = $axmud::CLIENT->constTelnetHash;

        if ($flag) {

            if ($self->mspMode eq 'client_agree' || $self->mspMode eq 'client_simulate') {

                # Full MSP or pseudo-MSP recognition is already turned on
                return 1;

            } else {

                # Turn on pseudo-MSP recognition
                $self->optSendDo($telConstHash{'TELOPT_MSP'});
                $self->ivPoke('mspMode', 'client_simulate');
            }

        } else {

            if ($self->mspMode eq 'no_invite' || $self->mspMode eq 'client_refuse') {

                # Full MSP or pseudo-MSP recognition is already turned off
                return 1;

            } else {

                # Turn off pseudo-MSP recognition
                $self->optSendDont($telConstHash{'TELOPT_MSP'});
                $self->ivPoke('mspMode', 'client_refuse');
            }
        }

        return 1;
    }

    # Buffers

    sub updateDisplayBuffer {

        # Called by $self->processLineSegment when a complete line of text has been received from
        #   the world and displayed in a textview
        # Updates the display buffer
        #
        # Expected arguments
        #   $line           - The original line of text received from the world
        #   $stripLine      - $line after being stripped of escape sequences
        #   $modLine        - $stripLine after being modified by any matching interfaces (identical
        #                       to $stripLine if none match)
        #   $newLineFlag    - TRUE if $line ends with a newline character, FALSE if it doesn't
        #   $offsetListRef  - Reference to a sorted list containing the offsets (positions in
        #                       $modLine) at which escape sequences occured, before they were
        #                       stripped away
        #   $offsetHashRef  - Reference to a hash in the form
        #                       $tagHash{offset} = reference_to_list_of_colour_and_style_tags
        #                   - Each offset represents the position of a character in $modLine
        #                   - Axmud colour and style tags each correspond to an escape sequence
        #   $appliedListRef - Reference to a list of Axmud colour/style tags that actually applied
        #                       at the beginning of the line (may be an empty list)
        #   $mxpFlagTextHash
        #                   - Reference to the contents of $self->mxpFlagTextStoreHash, just before
        #                       it was reset (may be an empty hash)
        #
        # Return values
        #   'undef' on improper arguments, if the session is not connected to a world or if the
        #       buffer can't be updated
        #   Otherwise returns the new buffer object created (or the existing buffer object
        #       modified)

        my (
            $self, $line, $stripLine, $modLine, $newLineFlag, $offsetListRef, $offsetHashRef,
            $appliedListRef, $mxpFlagTextHash, $check,
        ) = @_;

        # Local variables
        my (
            $lastObj, $thisObj,
            %tagHash,
        );

        # Check for improper arguments
        if (
            ! defined $line || ! defined $stripLine || ! defined $modLine || ! defined $newLineFlag
            || ! defined $offsetListRef || ! defined $offsetHashRef || ! defined $appliedListRef
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateDisplayBuffer', @_);
        }

        # Don't update the buffer after a disconnection (but do update it in 'connect offline'
        #   mode); in addition, only text displayed in the default tab is added to the display
        #   buffer
        if ($self->status eq 'disconnected' || $self->currentTabObj ne $self->defaultTabObj) {

            return undef;
        }

        if (! defined $self->displayBufferFirst) {

            # This the first line ever received
            $self->ivPoke('displayBufferFirst', 0);
        }

        # Use $offsetListRef and $offsetHashRef to compile a new hash of Axmud colour/style tags.
        #   The new hash is in the form
        #   $tagHash{tag} = reference_to_list_of_offsets_where_the_tag_occurs
        foreach my $offset (@$offsetListRef) {

            my $tagListRef = $$offsetHashRef{$offset};

            foreach my $tag (@$tagListRef) {

                my $newListRef;

                if (exists $tagHash{$tag}) {

                    $newListRef = $tagHash{$tag};
                    push (@$newListRef, $offset);

                } else {

                    $tagHash{$tag} = [$offset];
                }
            }
        }

        # If the previous received line didn't end with a newline character, we add this
        #   line to its GA::Buffer::Display object. Otherwise, we create a new
        #   GA::Buffer::Display object
        if ($self->displayBufferCount) {

            $lastObj = $self->ivShow('displayBufferHash', $self->displayBufferLast);
        }

        if (! $lastObj || $lastObj->newLineFlag) {

            # Previous line did end with a newline character (or this is the first line in the
            #   buffer)

            # Create a new buffer object for this line
            $thisObj = Games::Axmud::Buffer::Display->new(
                $self,
                'session',
                $self->displayBufferCount,
                $line,
                $stripLine,
                $modLine,
                $self->sessionTime,
                $newLineFlag,
                $offsetHashRef,
                \%tagHash,
                $appliedListRef,
                $mxpFlagTextHash,
            );

            if (! $thisObj) {

                return undef;

            } else {

                # Update the display buffer
                $self->ivAdd('displayBufferHash', $thisObj->number, $thisObj);
                $self->ivIncrement('displayBufferCount');
                $self->ivPoke('displayBufferLast', ($self->displayBufferCount - 1));

                # If the buffer is full, remove the oldest line
                if ($self->displayBufferCount > $axmud::CLIENT->customDisplayBufferSize) {

                    $self->ivDelete('displayBufferHash', $self->displayBufferFirst);
                    $self->ivIncrement('displayBufferFirst');
                }
            }

        } else {

            # Previous line didn't end with a newline character. Append the new text to the
            #   previous line
            $lastObj->update(
                $line,
                $stripLine,
                $modLine,
                $newLineFlag,
                $offsetHashRef,
                \%tagHash,
                $mxpFlagTextHash,
            );
        }

        # Set the time at which text was most recently received from the world and displayed in the
        #   default tab
        if ($self->defaultTabObj eq $self->currentTabObj) {

            $self->ivPoke('lastDisplayTime', $self->sessionTime);
        }

        # Allow the 'world_idle' hook event to happen ($self->constHookIdleTime seconds from now)
        $self->ivPoke('disableWorldIdleFlag', FALSE);

        if ($thisObj) {
            return $thisObj;
        } else {
            return $lastObj;
        }
    }

    sub updateInstructBuffer {

        # Called by $self->doInstruct after the user types an instruction in a 'main' window (when
        #   this is the window's visible session), or when any other part of the code calls
        #   $self->doInstruct
        # Updates the instruction buffer
        #
        # Expected arguments
        #   $instruct   - The instruction itself (e.g. ';setworld deathmud' or 'north;kill orc')
        #   $type       - The type of instruction: 'client' for a client command, 'world' for a
        #                   world command, 'perl' for a Perl command and 'echo' for an echo command
        #
        # Return values
        #   'undef' on improper arguments, if the session is not connected to a world or if the
        #       buffer is not updated
        #   Otherwise returns the buffer object created

        my ($self, $instruct, $type, $check) = @_;

        # Local variables
        my $obj;

        # Check for improper arguments
        if (! defined $instruct || ! defined $type || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateInstructBuffer', @_);
        }

        # Don't update the buffer after a disconnection (but do update it in 'connect offline' mode)
        if ($self->status eq 'disconnected') {

            return undef;
        }

        if (! defined $self->instructBufferFirst) {

            # This the first instruction ever processed
            $self->ivPoke('instructBufferFirst', 0);
        }

        # Create a new buffer object for this instruction
        $obj = Games::Axmud::Buffer::Instruct->new(
            $self,
            'session',
            $self->instructBufferCount,
            $instruct,
            $type,
            $self->sessionTime,
        );

        if (! $obj) {

            return undef;

        } else {

            # Update the instruction buffer
            $self->ivAdd('instructBufferHash', $obj->number, $obj);
            $self->ivIncrement('instructBufferCount');
            $self->ivPoke('instructBufferLast', ($self->instructBufferCount - 1));

            # If the buffer is full, remove the oldest line
            if ($self->instructBufferCount > $axmud::CLIENT->customInstructBufferSize) {

                $self->ivDelete('instructBufferHash', $self->instructBufferFirst);
                $self->ivIncrement('instructBufferFirst');
            }

            # Also add a separate buffer object (with a different ->number) to the equivalent
            #   registry in the GA::Client
            $axmud::CLIENT->updateInstructBuffer($self, $instruct, $type);
        }

        # Set the time at which the last instruction was executed
        $self->ivPoke('lastInstructTime', $self->sessionTime);

        return $obj;
    }

    sub updateCmdBuffer {

        # Called by $self->worldCmd for each individual command sent to the world
        # Also called by $self->teleportCmd, after an earlier call by GA::Cmd::Teleport->do
        #
        # Updates the world command buffer
        #
        # Expected arguments
        #   $cmd            - The world command itself (e.g. 'north', 'kill orc')
        #
        # Optional arguments
        #   $cage          - The highest-priority command cage (quite unlikely that this is set to
        #                       'undef')
        #   $redirectCmd    - For redirect mode commands, the substitute command (e.g. if $cmd is
        #                       'north', $redirectCmd might be 'sail north')
        #   $standardCmd    - For assisted moves, the standard primary direction equivalent to the
        #                       custom primary direction stored in $cmd. Set to 'undef' for
        #                       everything else
        #   $assistedCmd    - For assisted moves, the sequence of world commands corresponding to
        #                       the standard primary direction, $cmd (e.g. 'open door;north'). Set
        #                       to 'undef' for everything else
        #   $exitObj        - For assisted moves, the GA::Obj::Exit used for the movement (an exit
        #                       somewhere in the exit model). Set to 'undef' for everything else
        #   $teleportFlag   - When called by $self->teleportCmd, flag set to TRUE ('undef'
        #                       otherwise)
        #   $destRoom       - When called by $self->teleportCmd, the world model number of the
        #                       destination room (if known; 'undef' if not, or if not called by
        #                       $self->teleportCmd)
        #
        # Return values
        #   'undef' on improper arguments, if the session is not connected to a world or if the
        #       buffer can't be updated
        #   Otherwise returns the buffer object created

        my (
            $self, $cmd, $cage, $redirectCmd, $standardCmd, $assistedCmd, $exitObj,
            $teleportFlag, $destRoom, $check
        ) = @_;

        # Local variables
        my ($obj, $newGhost, $dir, $unabbrevDir, $exitNum);

        # Check for improper arguments
        if (! defined $cmd || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateCmdBuffer', @_);
        }

        # Don't update the buffer after a disconnection (but do update it in 'connect offline' mode)
        if ($self->status eq 'disconnected') {

            return undef;
        }

        if (! defined $self->cmdBufferFirst) {

            # This the first world command ever sent
            $self->ivPoke('cmdBufferFirst', 0);
        }

        # Create a new buffer object for this world command
        $obj = Games::Axmud::Buffer::Cmd->new(
            $self,
            'session',
            $self->cmdBufferCount,
            $cmd,
           