package App::Multigit;

use 5.014;
use strict;
use warnings FATAL => 'all';

use Capture::Tiny qw(capture);
use File::Find::Rule;
use Future::Utils qw(fmap);
use Path::Class;
use Config::Any;
use IPC::Run;

use App::Multigit::Repo;
use App::Multigit::Loop qw(loop);

=head1 NAME

App::Multigit - Run commands on a bunch of git repositories without having to
deal with git subrepositories.

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 FUNCTIONS

These are not currently exported.

=head2 mgconfig

Returns C<.mgconfig>. This is a stub to be later configurable, but also
to stop me typoing it all the time.

=cut

sub mgconfig() {
    return '.mgconfig';
}

=head2 mg_parent

Tries to find the closest directory with an C<mgconfig> in it. Dies if there is
no mgconfig here.

=cut

sub mg_parent {
    my $pwd = shift // dir->absolute;

    PARENT: {
        do {
            return $pwd if -e $pwd->file(mgconfig);
            last PARENT if $pwd eq $pwd->parent;
        }
        while ($pwd = $pwd->parent);
    }

    die "Could not find .mgconfig in any parent directory";
}

=head2 all_repositories

Returns a hashref of all repositories under C<mg_parent>.

The keys are the repository directories relative to C<mg_parent>, and the values
are the hashrefs from the config, if any.

=cut

sub all_repositories {
    my $pwd = shift // dir->absolute;
    my $mg_parent = mg_parent $pwd;

    my $cfg = Config::Any->load_files({
        files => [ mgconfig ],
        use_ext => 0,
        force_plugins => [
            qw/Config::Any::INI/
        ]
    });

    my $repos = +{
        map { %$_ } values %{ $cfg->[0] }
    };

    for (keys %$repos) {
        $repos->{$_}->{dir} //= dir($_)->basename =~ s/\.git$//r
    }

    return $repos;
}

=head2 each($command)

For each configured repository, C<$command> will be run. Each command is run in
a separate process which C<chdir>s into the repository first.

It returns a convergent L<Future> that represents all tasks. When this Future
completes, all tasks are complete.

In the arrayref form, the C<$command> is passed directly to C<run> in
L<App::Multigit::Repo>.  The Futures returned thus are collated and the list of
return values is thus collated. The list will be an alternating list of STDOUT
and STDERRs from the commands thus run.

    my $future = App::Multigit::each([qw/git reset --hard HEAD/]);
    my @stdios = $future->get;

The subref form is more useful. The subref is run with the Repo object, allowing
you to chain functionality thus.
    
    use curry;
    my $future = App::Multigit::each(sub {
        my $repo = shift;
        $repo
            ->run(\&do_a_thing)
            ->then($repo->curry::run(\&do_another_thing))
        ;
    });

In this case, the subref given to C<run> is passed the STDOUT and STDERR from
the previous command; for convenience, they start out as the empty strings,
rather than C<undef>.

    sub do_a_thing {
        my ($repo_obj, $stdout, $stderr) = @_;
        ...
    }

Thus you can chain them in any order.

Observe also that the interface to C<run> allows for the arrayref form as well:

    use curry;
    my $future = App::Multigit::each(sub {
        my $repo = shift;
        $repo
            ->run([qw/git checkout master/])
            ->then($repo->curry::run(\&do_another_thing))
        ;
    });

Notably, the returned Future will gather the return values of all the other Futures.
This means your final C<< ->then >> can be something other than a curried
C<run>. The helper function C<report> produces a pair whose first value is the
repo name and the second value is STDOUT concatenated with STDERR.

    use curry;
    my $future = App::Multigit::each(sub {
        my $repo = shift;
        $repo
            ->run([qw/git checkout master/])
            ->then($repo->curry::run(\&do_another_thing))
            ->then(App::Multigit::report($repo))
        ;
    });

    my %results = $future->get;

=cut

sub each {
    my $command = shift;
    my $repos = all_repositories;

    return fmap { _run_in_repo($command, $_[0], $repos->{$_[0]}) } 
        foreach => [ keys %$repos ],
        concurrent => 20,
    ;
}

sub _run_in_repo {
    my ($cmd, $repo, $config) = @_;

    if (ref $cmd eq 'ARRAY') {
        App::Multigit::Repo->new(
            name => $repo,
            config => $config
        )->run($cmd);
    }
    else {
        App::Multigit::Repo->new(
            name => $repo,
            config => $config
        )->$cmd;
    }
}

=head2 loop

Returns the L<IO::Async::Loop> object. This is essentially a singleton.

=cut

# sub loop used to be here but I moved it.

=head2 init($workdir)

Scans C<$workdir> for git directories and registers each in C<.mgconfig>

=cut

sub init {
    my $workdir = shift;
    my @dirs = File::Find::Rule
        ->relative
        ->directory
        ->maxdepth(1)
        ->mindepth(1)
        ->in($workdir);

    my %config;
    for my $dir (@dirs) {
        my ($remotes) = capture {
            system qw(git -C), $dir, qw(remote -v)
                and return;
        };

        # FIXME: This seems fragile
        next if $?;

        if (not $remotes) {
            warn "No remotes configured for $dir\n";
            next;
        }
        my ($first_remote) = split /\n/, $remotes;
        my ($name, $url) = split ' ', $first_remote;

        $config{$url} = $dir;
    }

    {
        my $config_filename = dir($workdir)->file(App::Multigit::mgconfig);
        open my $config_out, ">", $config_filename;

        for (keys %config) {
            say $config_out "[$_]";
            say $config_out "dir=$config{$_}";
        }
    }
}

=head2 report

This adapts a C<< ->then >> chain using C<run> in App::Multigit::Repo to return
the name of the repository and the output.

Returns a Future that yields a two-element list of the directory - from the
config - and the STDOUT from the command, indented with tabs.

Intended for use as a hash constructor.

    my %report = Future->wait_all(@futures)->get;

    for my $dir (sort keys %report) {
        say for $dir, $report{$dir};
    }

=cut

sub report {
    my $repo = shift;
    return sub {
        return Future->done if not @_;
        my ($stdout, $stderr) = @_;
        my $dir = $repo->config->{dir};

        my $output = do { 
            no warnings 'uninitialized';
            indent($stdout, 1) . indent($stderr, 1);
        };

        return Future->done(
            $dir => $output
        );
    }
}

=head2 indent

Returns a copy of the first argument indented by the number of tabs in the
second argument

=cut

sub indent {
    return if not defined $_[0];
    $_[0] =~ s/^/"\t" x $_[1]/germ
}

1;

__END__

=head1 AUTHOR

Alastair McGowan-Douglas, C<< <altreus at perl.org> >>

=head1 BUGS

Please report bugs on the github repository L<https://github.com/Altreus/App-Multigit>.

=head1 LICENSE

Copyright 2015 Alastair McGowan-Douglas.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

