package Evo::Guard;
use Evo::Base -strict;

sub new {
  my $class = shift;
  Carp::croak "You have to save Evo::Guard object to the variable"
    unless defined wantarray;
  bless {cb => $_[0]}, $class;
}

sub DESTROY { (delete $_[0]->{cb})->($@ ? $@ : ()) if $_[0]->{cb} }

1;

# ABSTRACT: A guard

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo::Guard - A guard

=head1 VERSION

version 0.0160

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Evo::Guard;

  eval {
    my $guard
      = Evo::Guard->new(sub($err=undef) { say $err ? "$err" : "alive" });
    die "Error";
  };

  eval {
    my $guard
      = Evo::Guard->new(sub($err=undef) { say $err ? "$err" : "alive" });
  };

=head1 DESCRIPTION

Executes a code when no refs are pointed to this object.
The behavoiur is similar to L<Guard/"guard">, but this module is pure
perl (and about 70-80% slower) and covers more cases.

It can be used to do some cleanups no matter was exception thrown or not.
Or it can do something usefull only on exceptions
Or it can do both

=head1 METHODS

=head2 new

Creates a guard object. It will raise an exception if you forget to save it
in somewhere.

The first argument will be executed when guard is out of scope.

If there was an exception in scope, the first argument will be passed to the
cb. If all is ok, no arguments will be passed. So you can check the first
argument to do some job on errors only

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
