package Evo;
use Evo::Base -strict;
use Evo::Manager;

use Exporter 'import';

my @DSL  = qw(dsl_stash dsl_check_stash dsl_call dsl_depth);
my @FUNC = qw(asap);
our @EXPORT_OK = (@DSL, @FUNC);
our %EXPORT_TAGS = (all => \@EXPORT_OK, dsl => \@DSL);

our $VERSION = '0.015';    # VERSION

sub asap : prototype(&@)            { Evo::Manager->singleton->asap(@_) }
sub dsl_depth : prototype()         { Evo::Manager->singleton->dsl_depth; }
sub dsl_call : prototype($;@)       { Evo::Manager->singleton->dsl_call(@_); }
sub dsl_stash : prototype(;$$)      { Evo::Manager->singleton->dsl_stash(@_); }
sub dsl_check_stash : prototype(;$) { Evo::Manager->singleton->dsl_stash(@_); }

1;

# ABSTRACT: Perl Evo design pattern

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo - Perl Evo design pattern

=head1 VERSION

version 0.015

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Evo;

  use Evo::Manager;
  my $mngr = Evo::Manager->new;

  # next tick
  our $state = 'non blocking';

  # always nb
  Evo::asap {
    say $state;
    local $state = 'blocking?';
    Evo::asap { say $state };
  };

  # dsl
  # prints 1;2;foo
  my @args = (1, 2);
  my $print = sub(@args) { local $, = ';'; say @args, $mngr->dsl_stash('foo') };
  $mngr->dsl_call({foo => 'foo'}, @args, $print);

  # currying
  my $debug = sub($evo, $next) {
    sub(@args) {
      say 'Got args: ', join ';', @args;
      $next->(@args);
      }
  };

  my $safe = sub($evo, $next) {
    sub(@args) {
      local $@;
      eval { $next->(@args) };
      warn "catched: $@" if $@;
      }
  };

  my $die = sub {die};
  my $curry = $mngr->curry_wrappers($debug, $safe);
  $curry->(sub { say "going to die"; die })->(1, 2);

  my $way = $mngr->build_way;
  my $train = $mngr->build_train(way => $way);
  $way->curry($curry)->add(sub { say 'step1'; })->add(sub { die "step 2" })
    ->add(sub { say "step 3"; });

  $train->start;

=head1 DESCRIPTION

  Tiny framework for organization non-trivial code into a simple logic.
  It includes benefits from functional and OO styles.

  Write fast, reusable and testable code.
  Familiar with OO? - Use it. Familiar with higher-order functions? - Use it.
  Familiar with non-blocking style? - Use it. Want DSL? - Use it.

  Do it the right way.

=head1 DESCRIPTION

  Attention. Don't use it right now. Wait for the stable release

=head1 METHODS

=head2 asap, dsl_depth, dsl_stash dsl_check_stash dsl_call dsl_depth

Calls corresponding L<Evo::Manager> methods passing a singleton as the first
argument

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
