package CryptoTron::GetAccountJsonStr;

# Load the Perl pragmas.
use 5.008009;
use strict;
use warnings;

# Load the Perl pragma Exporter.
use vars qw(@ISA @EXPORT @EXPORT_OK);
use Exporter 'import';

# Exporting the implemented subroutine.
our @EXPORT = qw(
    GetAccount_JsonStr
);

# Base class of this (tron_addr) module.
our @ISA = qw(Exporter);

# Set the package version. 
our $VERSION = '0.05';

# Load the required Perl modules or packages.
use Try::Catch;
use URI;
use POSIX;
use JSON::PP;
use LWP::UserAgent;

# Define the global variables.
our($HEADER, $SERVICE_URL);

# Set api url and api path.
our $API_URL = 'https://api.trongrid.io';
our $API_PATH = '/walletsolidity/getaccount';

# Set the request header.
$HEADER = [Accept => 'application/json',
           Content_Type => 'application/json'];

# Assemble the service url.
$SERVICE_URL = $API_URL.$API_PATH;

# ---------------------------------------------------------------------------- #
# Function encode()                                                            #
#                                                                              #
# Description:                                                                 # 
# The subroutine is decoding and encoding the given content.                   #
#                                                                              #
# @arg     $content     -> STRING                                              #
# @returns $json_encode -> STRING                                              #
# ---------------------------------------------------------------------------- #
sub encode {
    # Assign the argument to the local variable.
    my $content = $_[0];
    # Set up the options for the Perl module.
    my $json = 'JSON::PP'->new->pretty;
    # Decode the content of the response.
    my $json_decode = $json->decode($content);
    # Encode the decoded content of the response.
    my $json_encode = $json->encode($json_decode);
    # Return encoded and decoded data.
    return $json_encode;
};

# ---------------------------------------------------------------------------- #
# Function get_response()                                                      # 
#                                                                              #
# Description:                                                                 #
# The subroutine is using the HTTP method POST to retrieve the                 # 
# response from the given service url.                                         #
#                                                                              #
# @arg    $service_url -> STRING                                               #
# @return $content     -> STRING                                               #
# ---------------------------------------------------------------------------- #
sub get_response {
    # Assign the argument to the local variable.
    my $payload = $_[0];
    # Declare the variable.
    my $content = undef;
    # Create a new uri object.
    my $uri = URI->new($SERVICE_URL);
    # Create a new user agent object.
    my $ua = LWP::UserAgent->new;
    # Get the response from the uri.
    my $response = $ua->post($uri, $HEADER, Content => $payload);
    # Check success of operation.
    if ($response->is_success) {
        # Get the content from the response.
        $content = $response->content;
    } else {
        # Print the error code and message to the terminal window.
        print "HTTP error code: ", $response->code, "\n";
        print "HTTP error message: ", $response->message, "\n";
        # Exit script.
        exit 1;
    }
    # Return the content.
    return $content;
};

# ---------------------------------------------------------------------------- #
# Subroutine GetAccount_JsonStr()                                              #
# ---------------------------------------------------------------------------- #
sub GetAccount_JsonStr {
    # Assign the argument to the local variable.
    my $address = $_[0];
    # Assemble the payload from the address.
    my $payload = "\{\"address\":\"$address\",\"visible\":\"True\"\}";
    # Get the content from the service url.
    my $content = get_response($payload);
    # Encode the content.
    my $json_encode = encode($content);
    # Return the raw json data string to the terminal window.
    return $json_encode;
};

1;

__END__

=head1 NAME

CryptoTron::AccountTools - Perl extension for use with crypto coin Tron addresses

=head1 SYNOPSIS

  use CryptoTron::GetAccountJsonStr;

  # Declare the public keys.
  my $PublicKeyBase58 = 'TQHgMpVzWkhSsRB4BzZgmV8uW4cFL8eaBr';

  # Get the account info as JSON string.
  my $account = GetAccount_JsonStr($PublicKeyBase58);
  print $account;

=head1 DESCRIPTION

Get account info from the Tron blockchain.

=head1 SEE ALSO

Try::Catch;
URI;
POSIX;
JSON::PP;
LWP::UserAgent;

=head1 AUTHOR

Dr. Peter Netz, E<lt>ztenretep@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2022 by Dr. Peter Netz

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.30.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
