package Dist::Zilla::App::Command::deploy;

use P;
use Dist::Zilla::App qw[-command];
use Config();

sub abstract ($self) {
    return 'deploy distribution locally';
}

sub opt_spec ( $self, $app ) {
    return return
      [ install    => 'install to PATH and PERL5LIB' ],
      [ develop    => 'cpanm --with-develop' ],
      [ recommends => 'cpanm --with-recommends' ],
      [ suggests   => 'cpanm --with-suggests' ];
}

sub validate_args ( $self, $opt, $args ) {

    # NOTE args is just raw array or params, that not described as options

    die 'no args expected' if $args->@*;

    return;
}

sub execute ( $self, $opt, $args ) {

    # chmod
    $self->_chmod;

    # cpanm
    exit 100 if !$self->_cpanm($opt);

    # install
    exit 101 if $opt->install && !$self->_install;

    return;
}

sub _chmod ($self) {
    say 'chmod ... ';

    if ( !$MSWIN ) {
        P->file->find(
            {   wanted => sub {
                    if (-d) {
                        P->file->chmod( 'rwx------', $_ ) or say qq[$!: $_];
                    }
                    else {
                        if ( $self->_is_exec($_) ) {
                            P->file->chmod( 'r-x------', $_ ) or say qq[$!: $_];
                        }
                        else {
                            P->file->chmod( 'rw-------', $_ ) or say qq[$!: $_];
                        }
                    }

                    chown $>, $), $_ or say qq[$!: $_];    # EUID, EGID
                },
                no_chdir => 1,
            },
            q[.]
        );
    }

    say 'done';

    return;
}

sub _is_exec ( $self, $path ) {
    $path = P->file->path($path);

    if ( ( $path->dirname eq 'bin/' || $path->dirname eq 'script/' ) && !$path->suffix ) {
        return 1;
    }
    elsif ( $path->suffix eq 'sh' || $path->suffix eq 'pl' || $path->suffix eq 't' ) {
        return 1;
    }

    return;
}

sub _cpanm ( $self, $opt ) {
    if ( -f 'cpanfile' ) {
        my $cfg = P->cfg->load( $P->{SHARE_DIR} . 'pcore.perl' );

        if ( exists $cfg->{cpanm}->{ $Config::Config{archname} } && $cfg->{cpanm}->{ $Config::Config{archname} }->@* ) {
            P->sys->system( 'cpanm', '--notest', $cfg->{cpanm}->{ $Config::Config{archname} }->@* ) or return;
        }

        my @args = (    #
            'cpanm',
            '--with-feature', ( $MSWIN ? 'windows' : 'linux' ),
            ( $opt->develop    ? '--with-develop'    : () ),
            ( $opt->recommends ? '--with-recommends' : () ),
            ( $opt->suggests   ? '--with-suggests'   : () ),
            '--cpanfile',    'cpanfile',
            '--installdeps', q[.],
        );

        say join q[ ], @args;

        P->sys->system(@args) or return;
    }

    return 1;
}

sub _install ($self) {
    if ( !P->pm->is_superuser ) {
        say q[Root privileges required to deploy pcore at system level.];

        return;
    }

    my $canon_dist_root = P->file->cwd->realpath->canonpath;

    my $canon_bin_dir = P->file->path('./bin/')->realpath->canonpath;

    my $canon_dist_lib_dir = P->file->path('./../')->realpath->canonpath;

    if ($MSWIN) {

        # set $ENV{PERL5LIB}
        P->sys->system(qq[setx.exe /M PERL5LIB "$canon_dist_root/lib;"]) or return;

        say q[%PERL5LIB% updated];

        # set $ENV{PCORE_DIST_LIB}
        P->sys->system(qq[setx.exe /M PCORE_DIST_LIB "$canon_dist_lib_dir"]) or return;

        say q[%PCORE_DIST_LIB% updated];

        # update $ENV{PATH}
        my $env_path = lc $ENV{PATH} =~ s[/][\\]smgr;

        my $bin_dir_lc = lc $canon_bin_dir;

        if ( $env_path !~ m[\Q$bin_dir_lc\E(?:/|)(?:;|\Z)]sm ) {
            my @path = grep { $_ && !/\A\h+\z/sm } split /;/sm, $ENV{PATH};    # remove empty path tokens, and tokens, consisting only from spaces

            push @path, $canon_bin_dir;

            $ENV{PATH} = join q[;], @path;                                     ## no critic qw(Variables::RequireLocalizedPunctuationVars)

            P->sys->system(qq[setx.exe /M PATH "$ENV{PATH};"]) or return;

            say q[%PATH% updated];
        }
    }
    else {
        my $data = <<"SH";
export PERL5LIB="$canon_dist_root/lib:\$PERL5LIB"
export PCORE_DIST_LIB="$canon_dist_lib_dir"
export PATH="\$PATH:$canon_bin_dir"
SH

        P->file->write_bin( '/etc/profile.d/pcore.sh', { mode => q[rw-r--r--] }, \$data );
    }

    return 1;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 7                    │ NamingConventions::ProhibitAmbiguousNames - Ambiguously named subroutine "abstract"                            │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    2 │ 119, 121, 123        │ ValuesAndExpressions::ProhibitLongChainsOfMethodCalls - Found method-call chain of length 4                    │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    1 │ 1                    │ NamingConventions::Capitalization - Package "Dist::Zilla::App::Command::deploy" does not start with a upper    │
## │      │                      │ case letter                                                                                                    │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=cut
