package Pcore::App::Sphinx;

use Pcore qw[-cli -class];

extends qw[Pcore::App::Alien];

has '+name' => ( default => 'sphinx' );
has '+ns'   => ( default => 'Pcore::Sphinx' );

has alien_cfg_dir          => ( is => 'lazy', isa => Str, init_arg => undef );
has alien_indexer_bin_path => ( is => 'lazy', isa => Str, init_arg => undef );

has sphinx_ver => ( is => 'lazy', isa => Str, init_arg => undef );

our $CFG = {
    SPHINX => {    #
        SPHINX_VER => undef,
    }
};

sub _build_sphinx_ver {
    my $self = shift;

    return $self->cfg->{SPHINX}->{SPHINX_VER} || $PROC->{SPHINX}->{SPHINX_VER};
}

# APP
around run => sub {
    my $orig = shift;
    my $self = shift;

    if ( $ARGV{indexer} ) {
        say q[start indexer];

        $self->generate_alien_cfg;

        P->sys->system( $self->alien_indexer_bin_path, q[--config], $self->alien_cfg_path, q[--all] );

        exit 0;
    }

    $self->$orig;

    return;
};

around _build_cfg => sub {
    my $orig = shift;
    my $self = shift;

    return P->hash->merge(
        $self->$orig,
        $CFG,
        {   SPHINX => {    #
                SERVER => {
                    searchd => {
                        listen                       => q[9306:mysql41],
                        log                          => $PROC->{LOG_DIR} . $self->name . '-searchd.log',
                        query_log                    => $PROC->{LOG_DIR} . $self->name . '-searchd-query.log',
                        read_timeout                 => 5,
                        client_timeout               => 300,
                        max_children                 => 30,
                        persistent_connections_limit => 30,
                        pid_file                     => $self->app_dir . 'searchd.pid',
                        max_matches                  => 1000,
                        seamless_rotate              => 1,
                        preopen_indexes              => 1,
                        unlink_old                   => 1,
                        attr_flush_period            => 0,
                        ondisk_dict_default          => 0,
                        mva_updates_pool             => '1M',
                        max_packet_size              => '8M',
                        max_filters                  => 256,
                        max_filter_values            => 4096,
                        listen_backlog               => 5,
                        read_buffer                  => '256K',
                        read_unhinted                => '32K',
                        max_batch_queries            => '32',
                        subtree_docs_cache           => '4M',
                        subtree_hits_cache           => '8M',
                        workers                      => 'threads',
                        dist_threads                 => 4,
                        binlog_path                  => $self->alien_data_dir,
                        binlog_flush                 => 2,
                        binlog_max_log_size          => '128M',
                        thread_stack                 => '64K',
                        expansion_limit              => 0,
                        rt_flush_period              => 0,
                        query_log_format             => 'sphinxql',
                        mysql_version_string         => undef,
                        plugin_dir                   => q[],
                        collation_server             => 'utf8_general_ci',
                        collation_libc_locale        => 'C',
                        watchdog                     => 1,
                        compat_sphinxql_magics       => 0,
                        predicted_time_costs         => q[doc=64, hit=48, skip=2048, match=64],
                        sphinxql_state               => $self->alien_data_dir . 'sphinxvars.sql',
                        rt_merge_iops                => 0,
                        rt_merge_maxiosize           => 0,
                        ha_ping_interval             => 1000,
                        ha_period_karma              => 60,
                        prefork_rotation_throttle    => 0,
                        snippets_file_prefix         => q[],
                    },
                    indexer => {
                        mem_limit             => '32M',
                        max_iops              => 0,
                        max_iosize            => 0,
                        max_xmlpipe2_field    => '2M',
                        write_buffer          => '1M',
                        max_file_field_buffer => '8M',
                        on_file_field_error   => 'ignore_field',
                        on_json_attr_error    => 'ignore_attr',
                        json_autoconv_numbers => 0,

                        # json_autoconv_keynames => 'lowercase',
                        # lemmatizer_base  => $self->app_dir . q[],
                        lemmatizer_cache => '256K',
                    },
                },
            }
        }
    );
};

around _create_local_cfg => sub {
    my $orig = shift;
    my $self = shift;

    my $local_cfg = {
        SPHINX => {
            SPHINX_VER => $self->cfg->{SPHINX}->{SPHINX_VER},
            SERVER     => $self->cfg->{SPHINX}->{SERVER},
        },
    };

    return P->hash->merge( $self->$orig, $local_cfg );
};

sub _build_alien_dir {
    my $self = shift;

    return $self->app_dir . 'sphinx-' . $self->sphinx_ver . q[/];
}

sub _build_alien_data_dir {
    my $self = shift;

    my $dir = $self->app_dir . 'sphinx-data/';

    P->file->mkpath($dir);

    return $dir;
}

sub _build_alien_cfg_dir {
    my $self = shift;

    my $dir = $self->app_dir . 'indexes';

    P->file->mkpath($dir);

    return $dir;
}

sub _build_alien_bin_path {
    my $self = shift;

    return $self->alien_dir . 'bin/searchd';
}

sub _build_alien_indexer_bin_path {
    my $self = shift;

    return $self->alien_dir . 'bin/indexer';
}

sub _build_alien_cfg_path {
    my $self = shift;

    return $self->app_dir . 'sphinx.conf';
}

around app_build => sub {
    my $orig = shift;
    my $self = shift;

    $self->$orig;

    if ( -d $self->alien_dir ) {
        $self->_appx_report_warn( q["] . $self->alien_dir . q[" already exists. Remove it manually to rebuild] );
    }
    else {
        my $res = try {

            # re2
            my $re2_prefix = ( $ARGV{r2_system} ? q[/usr/local/] : $self->alien_dir ) . 're2/';
            P->file->rmtree($re2_prefix);
            P->file->mkpath($re2_prefix);

            P->sys->system(qq[wget -O - https://github.com/google/re2/archive/master.tar.gz | tar -C $re2_prefix --strip-components=1 -xzvf -]);

            {
                my $chdir_guard = P->file->chdir($re2_prefix) or die;

                P->sys->system( 'make', '-j' . P->sys->cpus_num );

                local $ENV{LDFLAGS} = '-pthread';

                P->sys->system( 'make', 'test' );
            }

            # sphinx
            P->sys->system( q[wget -O - http://sphinxsearch.com/files/sphinx-] . $self->sphinx_ver . qq[-release.tar.gz | tar -C $PROC->{TEMP_DIR} -xzvf -] );

            # libstemmer
            P->sys->system( qq[wget -O - http://snowball.tartarus.org/dist/libstemmer_c.tgz | tar -C $PROC->{TEMP_DIR}sphinx-] . $self->sphinx_ver . q[-release -xzvf -] );

            {
                my $chdir_guard = P->file->chdir( $PROC->{TEMP_DIR} . 'sphinx-' . $self->sphinx_ver . q[-release] ) or die;

                P->sys->system( q[./configure], q[--prefix=] . $self->alien_dir, q[--with-mysql], q[--with-libstemmer], q[--with-libexpat], q[--with-iconv], q[--enable-id64], qq[--with-re2-includes=$re2_prefix] );

                P->sys->system( 'make', '-j' . P->sys->cpus_num );

                P->sys->system( 'make', 'install' );

                return 1;
            }
        }
        catch {
            my $e = shift;

            return 0;
        };

        unless ($res) {
            P->file->rmtree( $self->alien_dir );

            $self->_appx_report_fatal(qq[Error building application. Maybe you need to manually install dependencies, try: 'sudo yum -y install expat-devel MariaDB-client'\nOr try to build with options '--UID=root --r2-system' if sphinx app dir located in vmware hgfs (i.e. symbolic links not supported)]);
        }
    }

    return;
};

around app_deploy => sub {
    my $orig = shift;
    my $self = shift;

    $self->$orig;

    return;
};

sub generate_alien_cfg {
    my $self = shift;

    my %cfg;

    for my $section ( sort keys %{ $self->cfg->{SPHINX}->{SERVER} } ) {
        next if !$self->cfg->{SPHINX}->{SERVER}->{$section};

        my $cfg;
        for my $key ( sort keys %{ $self->cfg->{SPHINX}->{SERVER}->{$section} } ) {
            next if !defined $self->cfg->{SPHINX}->{SERVER}->{$section}->{$key};

            if ( ref $self->cfg->{SPHINX}->{SERVER}->{$section}->{$key} eq 'ARRAY' ) {
                for my $val ( @{ $self->cfg->{SPHINX}->{SERVER}->{$section}->{$key} } ) {
                    $cfg .= qq[    $key = $val\n];
                }
            }
            else {
                $cfg .= qq[    $key = ] . $self->cfg->{SPHINX}->{SERVER}->{$section}->{$key} . $LF;
            }
        }

        $cfg{$section} = $cfg;
    }

    my $conf = qq[#!/bin/sh\n\ncat <<EOF\n];
    $conf .= join $LF, map {"$_ {\n$cfg{$_}}"} sort keys %cfg;
    $conf .= qq[\nEOF\n\ncat ] . $self->alien_cfg_dir . q[/*.conf];

    $self->store_alien_cfg( \$conf );

    return;
}

sub master_proc {
    my $self = shift;

    return;
}

sub alien_proc {
    my $self = shift;

    $self->generate_alien_cfg;

    exec $self->alien_bin_path, q[--config], $self->alien_cfg_path, q[--nodetach] or die;
}

1;
__END__
=pod

=encoding utf8

=head1 OPTIONS

=over

=item --indexer

=item --r2-system

Install re2 to /usr/local instead of sphinx directory

=back

=cut
