package Pcore::App::Redis;

use Pcore qw[-class];

extends qw[Pcore::App::Alien];

has '+name' => ( default => 'redis' );
has '+ns'   => ( default => 'Pcore::Redis' );

has redis_ver => ( is => 'lazy', isa => Str, init_arg => undef );

our $CFG = {
    REDIS => {    #
        REDIS_VER => undef,
    }
};

# CLI
sub cli_name ($self) {
    return 'redis';
}

sub _build_redis_ver ($self) {
    return $self->cfg->{REDIS}->{REDIS_VER} || $PROC->{CFG}->{REDIS}->{REDIS_VER};
}

# APP
around _build_cfg => sub ( $orig, $self ) {
    return P->hash->merge(
        $self->$orig,
        $CFG,
        {   REDIS => {
                SERVER => {
                    pidfile                     => $self->app_dir . 'redis.pid',
                    port                        => 6379,                                             # use 0 to stop TCP interface
                    tcp_backlog                 => 511,
                    bind                        => [qw[127.0.0.1]],
                    unixsocket                  => q[/tmp/] . $self->name . q[.sock],
                    unixsocketperm              => 755,
                    timeout                     => 0,
                    tcp_keepalive               => 0,
                    loglevel                    => 'notice',
                    logfile                     => $PROC->{LOG_DIR} . $self->name . q[-redis.log],
                    databases                   => 16,
                    save                        => [ [ 900, 1 ], [ 300, 10 ], [ 60, 10_000 ], ],
                    stop_writes_on_bgsave_error => 'yes',
                    rdbcompression              => 'yes',
                    rdbchecksum                 => 'yes',
                    dbfilename                  => $self->name . '.rdb',
                    dir                         => $self->alien_data_dir,
                    slave_serve_stale_data      => 'yes',
                    slave_read_only             => 'yes',
                    repl_disable_tcp_nodelay    => 'no',
                    slave_priority              => 100,
                    requirepass                 => undef,
                    appendonly                  => 'yes',
                    appendfilename              => $self->name . '.aof',
                },
            }
        }
    );
};

around _create_local_cfg => sub ( $orig, $self ) {
    my $local_cfg = {
        REDIS => {
            REDIS_VER => $self->cfg->{REDIS}->{REDIS_VER},
            SERVER    => $self->cfg->{REDIS}->{SERVER},
        },
    };

    $local_cfg->{REDIS}->{SERVER}->{requirepass} = P->random->bytes_hex(32);

    return P->hash->merge( $self->$orig, $local_cfg );
};

sub _build_alien_dir ($self) {
    return $self->app_dir . 'redis-' . $self->redis_ver . q[/];
}

sub _build_alien_data_dir ($self) {
    my $dir = $self->app_dir . 'redis-data/';

    P->file->mkpath($dir);

    return $dir;
}

sub _build_alien_bin_path ($self) {
    return $self->alien_dir . 'src/redis-server';
}

sub _build_alien_cfg_path ($self) {
    return $self->app_dir . 'redis.conf';
}

around app_build => sub ( $orig, $self ) {
    $self->$orig;

    if ( -d $self->alien_dir ) {
        $self->_appx_report_warn( q["] . $self->alien_dir . q[" already exists. Remove it manually to rebuild] );
    }
    else {
        my $res = try {
            P->file->mkpath( $self->alien_dir );

            P->sys->system( q[wget -O - http://download.redis.io/releases/redis-] . $self->redis_ver . q[.tar.gz | tar -C ] . $self->alien_dir . q[ --strip-components=1 -xzvf -] );

            {
                my $chdir_guard = P->file->chdir( $self->alien_dir ) or die;

                P->sys->system( q[make], q[-j] . P->sys->cpus_num );
            }

            return 1;
        }
        catch {
            my $e = shift;

            return 0;
        };

        unless ($res) {
            P->file->rmtree( $self->alien_dir );

            $self->_appx_report_fatal(q[Error building application. Maybe you need to execute "yum -y install jemalloc-devel"]);
        }
    }

    return;
};

around app_deploy => sub ( $orig, $self ) {
    $self->$orig;

    return;
};

sub generate_alien_cfg ($self) {
    my $cfg;

    my $add_key = sub {
        my $key = shift;
        my $val = shift;

        my $cfg_key = $key =~ s/_/-/smgr;

        if ( ref $val eq 'ARRAY' ) {
            $cfg .= qq[$cfg_key ] . join( q[ ], @{$val} ) . $LF;
        }
        else {
            $cfg .= qq[$cfg_key $val\n];
        }

        return;
    };

    for my $key ( sort keys %{ $self->cfg->{REDIS}->{SERVER} } ) {
        next if !defined $self->cfg->{REDIS}->{SERVER}->{$key};

        if ( ref $self->cfg->{REDIS}->{SERVER}->{$key} eq 'ARRAY' ) {
            for ( @{ $self->cfg->{REDIS}->{SERVER}->{$key} } ) {
                $add_key->( $key, $_ );
            }
        }
        else {
            $add_key->( $key, $self->cfg->{REDIS}->{SERVER}->{$key} );
        }
    }

    $self->store_alien_cfg( \$cfg );

    return;
}

sub master_proc ($self) {
    return;
}

sub alien_proc ($self) {
    $self->generate_alien_cfg;

    exec $self->alien_bin_path, $self->alien_dir . 'redis.conf', q[--include ] . $self->alien_cfg_path or die;
}

1;
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::App::Redis

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ATTRIBUTES

=head1 METHODS

=head1 SEE ALSO

=cut
