package Pcore::App::Controller::ExtJS;

use Pcore -role;
use Pcore::Util::Data qw[to_json];
use Pcore::ExtJS;
use JavaScript::Packer qw[];

with qw[Pcore::App::Controller];

requires qw[_build_ext_map];

has api_ver => ( is => 'ro', isa => Maybe [Str] );    # API version

has ext_map                => ( is => 'lazy', isa => HashRef, init_arg => undef );
has perl_ext_app_namespace => ( is => 'ro',   isa => Str,     init_arg => undef );
has ext_app_namespace      => ( is => 'ro',   isa => Str,     init_arg => undef );
has ext_namespace          => ( is => 'ro',   isa => Str,     init_arg => undef );

has _static_cache => ( is => 'ro', isa => HashRef, init_arg => undef );

# this method can be overrided in the child class
sub BUILD ( $self, $args ) {
    return;
}

# this method can be overrided in the child class
sub run ( $self, $req ) {

    if ( $req->{path_tail} && $req->{path_tail}->is_file ) {

        # try to return static content
        $self->return_static($req);
    }
    else {
        $req->(404)->finish;
    }

    return;
}

around BUILD => sub ( $orig, $self, $args ) {
    $self->$orig($args);

    my $app_namespace = ref $self->{app};

    my $ext_app;

    # this is ExtJS application class
    if ( $self->does('Pcore::App::Controller::ExtJS::App') ) {
        $ext_app = $self;

        $self->{perl_ext_app_namespace} = ref $self;

        $self->{ext_namespace} = $self->{ext_app_namespace} = ref($self) =~ s/:://smgr;
    }

    # this is ExtJS namespace class
    else {

        # find parent ExtJS app instance
        my $parent_class = ref $self;

        while (1) {

            # remove last ::label, break loop, if no last label
            last unless $parent_class =~ s/::[^:]+\z//sm;

            last if $parent_class eq $app_namespace;

            if ( eval { $parent_class->does('Pcore::App::Controller::ExtJS::App') } ) {
                $ext_app = $self->{app}->{router}->get_ctrl_by_class_name($parent_class);

                last;
            }
        }

        die q[ExtJS app wasn't found] if !$ext_app;

        $self->{perl_ext_app_namespace} = ref $ext_app;

        $self->{ext_app_namespace} = $ext_app->{ext_namespace};

        my ( $path, $ext_app_path ) = ( $self->path, $ext_app->path );

        $self->{ext_namespace} = $path =~ s/$ext_app_path/$self->{ext_app_namespace}./smr;

        # remove last "/"
        substr $self->{ext_namespace}, -1, 1, q[];

        $self->{ext_namespace} =~ s[/][.]smg;

        # inherit default API version from app, if not defined in current namespace
        $self->{api_ver} //= $ext_app->{api_ver};
    }

    # validate ExtJS class
    die qq[ExtJS namespace "$self->{ext_namespace}" is invalid] if $self->{ext_namespace} =~ /[^[:alpha:]\d.]/smg;

    # check ext_map
    my $ext_map = $self->ext_map;

    for my $class ( keys $ext_map->%* ) {
        die qq["$class" must be in lowercase alpha name] if $class !~ /\A[[:lower:]]+\z/sm;

        die qq["EXT_$class" method is required to generate ExtJS class] if !$self->can("EXT_$class");

        if ( ref $ext_map->{$class} ne 'HASH' ) {
            $ext_map->{$class} = {
                extend => $ext_map->{$class},
                static => undef,
            };
        }

        my $class_name = ucfirst lc $class;

        my $perl_class_name = ref($self) . "::$class_name";

        my $ext_class_name = "$self->{ext_namespace}.$class_name";

        # register perl class
        $Pcore::ExtJS::BOOTSTRAP->{classic}->{class}->{$perl_class_name} = $ext_class_name;

        # alias
        if ( $ext_map->{$class}->{extend} ) {

            # find real base class name
            my $real_base_class_name = $Pcore::ExtJS::BOOTSTRAP->{classic}->{class}->{ $ext_map->{$class}->{extend} };

            die qq[ExjJS class "$ext_map->{$class}->{extend}" is not registered] if !$real_base_class_name;

            # get base class alias
            if ( my $base_class_alias = $Pcore::ExtJS::BOOTSTRAP->{classic}->{class_to_alias}->{$real_base_class_name} ) {
                my ($alias_namespace) = ( $base_class_alias =~ /(.+)[.][^.]+\z/sm );

                # generate alias
                my $alias = "$alias_namespace." . ( lc $ext_class_name =~ s/[.]/-/smgr );

                # register ExtJS class alias
                $Pcore::ExtJS::BOOTSTRAP->{classic}->{class_to_alias}->{$ext_class_name} = $alias;

                $Pcore::ExtJS::BOOTSTRAP->{classic}->{alias_to_class}->{$alias} = $ext_class_name;
            }
        }

        # register ext class as app static class
        if ( $ext_map->{$class}->{static} ) {
            $ext_app->{static_classes}->{$ext_class_name} = {
                ctrl       => $self,
                class_name => $class_name,
            };
        }
    }

    return;
};

around run => sub ( $orig, $self, $req ) {

    # .js file request
    if ( $req->{path_tail} && $req->{path_tail} =~ /\A(.+)[.]js\z/sm ) {
        my $extjs_class_name = $1;

        my $internal_class_name = lc $extjs_class_name;

        if ( exists $self->{ext_map}->{$internal_class_name} ) {
            my $readable = 1;

            my $is_static = $self->{ext_map}->{$internal_class_name}->{static};

            # return static content from cache, if cached
            if ( $is_static && exists $self->{_static_cache}->{$internal_class_name} ) {
                $req->( 200, [ 'Content-Type' => 'application/javascript' ], $self->{_static_cache}->{$internal_class_name} )->finish;

                return;
            }

            # create ExtJS class generation context
            my $ext = bless {
                app        => $self->{app},
                req        => $req,
                readable   => $readable,
                namespace  => $self,
                class_name => $extjs_class_name,
                extend     => $self->{ext_map}->{$internal_class_name}->{extend},
                cb         => sub ( $ext, $cfg, %args ) {
                    $ext->{cfg} = $cfg;

                    my $headers = [ 'Content-Type' => 'application/javascript' ];

                    push $headers->@*, $args{headers}->@*, if $args{headers};

                    my $js = $ext->to_js;

                    # compress static javascript
                    if ($is_static) {
                        $self->minify_js($js);

                        # store minified js in cache
                        $self->{_static_cache}->{$internal_class_name} = $js;
                    }

                    # generate class
                    $req->( 200, $headers, $js )->finish;

                    return;
                },
              },
              'Pcore::ExtJS';

            my $method = "EXT_$internal_class_name";

            # call ExtJS class generation method
            $self->$method($ext);

            return;
        }
    }

    # fall back to the original method
    $self->$orig($req);

    return;
};

sub minify_js ( $self, $ref ) {
    state $js_packer = JavaScript::Packer->init;

    $js_packer->minify( $ref, { compress => 'obfuscate' } );    # clean

    return;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## +------+----------------------+----------------------------------------------------------------------------------------------------------------+
## | Sev. | Lines                | Policy                                                                                                         |
## |======+======================+================================================================================================================|
## |    3 | 1                    | Modules::ProhibitExcessMainComplexity - Main code has high complexity score (24)                               |
## +------+----------------------+----------------------------------------------------------------------------------------------------------------+
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::App::Controller::ExtJS -  - ExtJS namespace HTTP controller

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ATTRIBUTES

=head1 METHODS

=head1 SEE ALSO

=cut
