[[Raku PDF Project]](https://pdf-raku.github.io)
 / [PDF::ISO_32000](https://pdf-raku.github.io/PDF-ISO_32000-raku)

# PDF-ISO_32000-raku

The [PDF 32000-1:2008 1.7 Specification](http://www.adobe.com/content/dam/Adobe/en/devnet/acrobat/pdfs/PDF32000_2008.pdf) contains around 380 tables, of which about 280 can be considered data or object definitions.

This module has been used to extract and generate the roles and classes from the PDF
specification for use by PDF::Class.

PDF::Class (at last count) implements roles and classes for about 100 of these objects, most of which
have been derived and/or checked against the roles generated by this module.

This library contains:

- XHTML tables that have been mined from the PDF-32000 1.7 specification.
- JSON data for the tables
- Raku Roles for tables that represent PDF objects:

The roles are named ISO_32000::Table_NNN-Xxxx and contain method stubs and documentation for each entry in the role


## Tables

Data is available for all of the tables in the PDF-32000 1.7 specification:
```
use PDF::ISO_32000;
# Load data about the Document Information dictionary
my %info = PDF::ISO_32000.table: "Table_317-Entries_in_the_document_information_dictionary";
# -OR- by table number
%info = PDF::ISO_32000.[317];
say %info<caption>;             # Table 317 – Entries in the document information dictionary
say %info<head>.join(" | ");    # Key | Type | Value
say %info<rows>[0].join(" | "); # Title | text string | (Optional; PDF 1.1) The document’s title.
```

The `table-index` method returns a list that maps table numbers to table names:

```
say PDF::ISO_32000.table-index[317] # Table 317 - Info_entries
```

The `appendix` method returns a hash index into the Appendix:

```
my $stream-ops = PDF::ISO_32000.appendix<A.1>;
say $stream-ops, # PDF_content_stream_operators
say PDF::ISO_32000.table($stream-ops)<caption>; #  Table A.1 – PDF content stream operators
```

## Roles

Roles are available for tables named `*_entries`, or `*_attributes`.

```
% p6doc ISO_320000:Table_317-Entries_in_the_document_information_dictionary
% p6doc ISO_320000:Table_28-Entries_in_the_catalog_dictionary
```

The roles also contain [method stubs](https://docs.raku.org/language/objects#Stubs) for the entries that need to be implemented for the role. For example:

```
% cat << EOF > lib/Catalog.rakumod
use ISO_32000::Table_28-Entries_in_the_catalog_dictionary;
unit class Catalog does ISO_32000::Table_28-Entries_in_the_catalog_dictionary;
EOF
% raku -I . -M Catalog
===SORRY!=== Error while compiling lib/Catalog.rakumod (Catalog)
Method 'SpiderInfo' must be implemented by Catalog because it is required by roles: ISO_32000::Table_28-Entries_in_the_catalog_dictionary.
at lib/Catalog.rakumod (Catalog):1
```
This module contains:

- a copy of the [PDF-32000 specification](https://github.com/pdf-raku/PDF-ISO_32000-raku/blob/master/src/PDF32000_2008.pdf)
- [JSON tables](https://github.com/pdf-raku/PDF-ISO_32000-raku/blob/master/resources/) extracted from the above
- [generated interface roles](https://github.com/pdf-raku/PDF-ISO_32000-raku/blob/master/gen/lib/ISO_32000) for building and validating PDF objects
- scripts and Makefiles for regenerating the XML tables and roles

## ISO 3200 Roles

The following interface roles have been mined from the ISO-32000 specification

### Roles and Entries


ISO_32000 Reference|Entries
----|-----
[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod)|/Length /Filter /DecodeParms /F /FFilter /FDecodeParms /DL
[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod)|/Predictor /Colors /BitsPerComponent /Columns /EarlyChange
[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)|/K /EndOfLine /EncodedByteAlign /Columns /Rows /EndOfBlock /BlackIs1 /DamagedRowsBeforeError
[Table 12 – Optional parameter for the JBIG2Decode filter](gen/lib/ISO_32000/Table_12-Optional_parameter_for_the_JBIG2Decode_filter.rakumod)|/JBIG2Globals
[Table 13 – Optional parameter for the DCTDecode filter](gen/lib/ISO_32000/Table_13-Optional_parameter_for_the_DCTDecode_filter.rakumod)|/ColorTransform
[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.rakumod)|/Type /Name
[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod)|/Size /Prev /Root /Encrypt /Info /ID
[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.rakumod)|/Type /N /First /Extends
[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod)|/Type /Size /Index /Prev /W
[Table 19 – Additional entries in a hybrid-reference file’s trailer dictionary](gen/lib/ISO_32000/Table_19-Additional_entries_in_a_hybrid-reference_files_trailer_dictionary.rakumod)|/XRefStm
[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod)|/Filter /SubFilter /V /Length /CF /StmF /StrF /EFF
[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod)|/R /O /U /P /EncryptMetadata
[Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.rakumod)|/Recipients /P
[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.rakumod)|/Type /CFM /AuthEvent /Length
[Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.rakumod)|/Recipients /EncryptMetadata
[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)|/Type /Version /Extensions /Pages /PageLabels /Names /Dests /ViewerPreferences /PageLayout /PageMode /Outlines /Threads /OpenAction /AA /URI /AcroForm /Metadata /StructTreeRoot /MarkInfo /Lang /SpiderInfo /OutputIntents /PieceInfo /OCProperties /Perms /Legal /Requirements /Collection /NeedsRendering
[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.rakumod)|/Type /Parent /Kids /Count
[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)|/Type /Parent /LastModified /Resources /MediaBox /CropBox /BleedBox /TrimBox /ArtBox /BoxColorInfo /Contents /Rotate /Group /Thumb /B /Dur /Trans /Annots /AA /Metadata /PieceInfo /StructParents /ID /PZ /SeparationInfo /Tabs /TemplateInstantiated /PresSteps /UserUnit /VP
[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)|/Dests /AP /JavaScript /Pages /Templates /IDS /URLS /EmbeddedFiles /AlternatePresentations /Renditions
[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod)|/ExtGState /ColorSpace /Pattern /Shading /XObject /Font /ProcSet /Properties
[Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.rakumod)|/Kids /Names /Limits
[Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.rakumod)|/Kids /Nums /Limits
[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.rakumod)|/FunctionType /Domain /Range
[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod)|/Size /BitsPerSample /Order /Encode /Decode
[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.rakumod)|/C0 /C1 /N
[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.rakumod)|/Functions /Bounds /Encode
[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod)|/Type /FS /F /UF /DOS /Mac /Unix /ID /V /EF /RF /Desc /CI
[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.rakumod)|/Type /Subtype /Params
[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod)|/Size /CreationDate /ModDate /Mac /CheckSum
[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.rakumod)|/Subtype /Creator /ResFork
[Table 48 – Entries in a collection item dictionary](gen/lib/ISO_32000/Table_48-Entries_in_a_collection_item_dictionary.rakumod)|/Type
[Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.rakumod)|/Type /D /P
[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.rakumod)|/Type /BaseVersion /ExtensionLevel
[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.rakumod)|/CTM /color
[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)|/overprint /transfer /halftone /flatness /smoothness
[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)|/Type /LW /LC /LJ /ML /D /RI /OP /op /OPM /Font /BG /BG2 /UCR /UCR2 /TR /TR2 /HT /FL /SM /SA /BM /SMask /CA /ca /AIS /TK
[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.rakumod)|/WhitePoint /BlackPoint /Gamma
[Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.rakumod)|/WhitePoint /BlackPoint /Gamma /Matrix
[Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.rakumod)|/WhitePoint /BlackPoint /Range
[Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.rakumod)|/N /Alternate /Range /Metadata
[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.rakumod)|/Subtype /Colorants /Process /MixingHints
[Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.rakumod)|/ColorSpace /Components
[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.rakumod)|/Solidities /PrintingOrder /DotGain
[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod)|/Type /PatternType /PaintType /TilingType /BBox /XStep /YStep /Resources /Matrix
[Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod)|/Type /PatternType /Shading /Matrix /ExtGState
[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod)|/ShadingType /ColorSpace /Background /BBox /AntiAlias
[Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.rakumod)|/Domain /Matrix /Function
[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.rakumod)|/Coords /Domain /Function /Extend
[Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.rakumod)|/Coords /Domain /Function /Extend
[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod)|/BitsPerCoordinate /BitsPerComponent /BitsPerFlag /Decode /Function
[Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod)|/BitsPerCoordinate /BitsPerComponent /VerticesPerRow /Decode /Function
[Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod)|/BitsPerCoordinate /BitsPerComponent /BitsPerFlag /Decode /Function
[Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.rakumod)|/Type /Subtype /Level1
[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)|/Type /Subtype /Width /Height /ColorSpace /BitsPerComponent /Intent /ImageMask /Mask /Decode /Interpolate /Alternates /SMask /SMaskInData /Name /StructParent /ID /OPI /Metadata /OC
[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.rakumod)|/Image /DefaultForPrinting /OC
[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)|/Type /Subtype /FormType /BBox /Matrix /Resources /Group /Ref /Metadata /PieceInfo /LastModified /StructParent /StructParents /OPI /OC /Name
[Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.rakumod)|/Type /S
[Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.rakumod)|/F /Page /ID
[Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.rakumod)|/Type /Name /Intent /Usage
[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.rakumod)|/Type /OCGs /P /VE
[Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.rakumod)|/OCGs /D /Configs
[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)|/Name /Creator /BaseState /ON /OFF /Intent /AS /Order /ListMode /RBGroups /Locked
[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)|/CreatorInfo /Language /Export /Zoom /Print /View /User /PageElement
[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.rakumod)|/Event /OCGs /Category
[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod)|/Type /Subtype /Name /BaseFont /FirstChar /LastChar /Widths /FontDescriptor /Encoding /ToUnicode
[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)|/Type /Subtype /Name /FontBBox /FontMatrix /CharProcs /Encoding /FirstChar /LastChar /Widths /FontDescriptor /Resources /ToUnicode
[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.rakumod)|/Type /BaseEncoding /Differences
[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.rakumod)|/Registry /Ordering /Supplement
[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)|/Type /Subtype /BaseFont /CIDSystemInfo /FontDescriptor /DW /W /DW2 /W2 /CIDToGIDMap
[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod)|/Type /CMapName /CIDSystemInfo /WMode /UseCMap
[Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod)|/Type /Subtype /BaseFont /Encoding /DescendantFonts /ToUnicode
[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)|/Type /FontName /FontFamily /FontStretch /FontWeight /Flags /FontBBox /ItalicAngle /Ascent /Descent /Leading /CapHeight /XHeight /StemV /StemH /AvgWidth /MaxWidth /MissingWidth /FontFile /FontFile2 /FontFile3 /CharSet
[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.rakumod)|/Style /Lang /FD /CIDSet
[Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.rakumod)|/FontFile /FontFile2 /FontFile3
[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod)|/Length1 /Length2 /Length3 /Subtype /Metadata
[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod)|/Type /HalftoneType /HalftoneName /Frequency /Angle /SpotFunction /AccurateScreens /TransferFunction
[Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod)|/Type /HalftoneType /HalftoneName /Width /Height /TransferFunction
[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod)|/Type /HalftoneType /HalftoneName /Xsquare /Ysquare /TransferFunction
[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod)|/Type /HalftoneType /HalftoneName /Width /Height /Width2 /Height2 /TransferFunction
[Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.rakumod)|/Type /HalftoneType /HalftoneName /Default
[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod)|/Type /S /G /BC /TR
[Table 146 – Additional entry in a soft-mask image dictionary](gen/lib/ISO_32000/Table_146-Additional_entry_in_a_soft-mask_image_dictionary.rakumod)|/Matte
[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.rakumod)|/S /CS /I /K
[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)|/HideToolbar /HideMenubar /HideWindowUI /FitWindow /CenterWindow /DisplayDocTitle /NonFullScreenPageMode /Direction /ViewArea /ViewClip /PrintArea /PrintClip /PrintScaling /Duplex /PickTrayByPDFSize /PrintPageRange /NumCopies
[Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.rakumod)|/Type /First /Last /Count
[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod)|/Title /Parent /Prev /Next /First /Last /Count /Dest /A /SE /C /F
[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod)|/Type /Schema /D /View /Sort
[Table 156 – Entries in a collection schema dictionary](gen/lib/ISO_32000/Table_156-Entries_in_a_collection_schema_dictionary.rakumod)|/Type
[Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod)|/Type /Subtype /N /O /V /E
[Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.rakumod)|/Type /S /A
[Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.rakumod)|/Type /S /P /St
[Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.rakumod)|/Type /F /I
[Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod)|/Type /T /N /V /P /R
[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod)|/Type /S /D /Dm /M /Di /SS /B
[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod)|/Type /NA /PA /Next /Prev /Dur
[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)|/Type /Subtype /Rect /Contents /P /NM /M /F /AP /AS /Border /C /StructParent /OC
[Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.rakumod)|/Type /W /S /D
[Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.rakumod)|/S /I
[Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.rakumod)|/N /R /D
[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod)|/T /Popup /CA /RC /CreationDate /IRT /Subj /RT /IT /ExData
[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod)|/Subtype /Open /Name /State /StateModel
[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod)|/Subtype /A /Dest /H /PA /QuadPoints /BS
[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod)|/Subtype /DA /Q /RC /DS /CL /IT /BE /RD /BS /LE
[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod)|/Subtype /L /BS /LE /IC /LL /LLE /Cap /IT /LLO /CP /Measure /CO
[Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod)|/Subtype /BS /IC /BE /RD
[Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod)|/Subtype /Vertices /LE /BS /IC /BE /IT /Measure
[Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.rakumod)|/Subtype /QuadPoints
[Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.rakumod)|/Subtype /RD /Sy
[Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.rakumod)|/Subtype /Name
[Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.rakumod)|/Subtype /InkList /BS
[Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.rakumod)|/Subtype /Parent /Open
[Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.rakumod)|/Subtype /FS /Name
[Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.rakumod)|/Subtype /Sound /Name
[Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.rakumod)|/Subtype /T /Movie /A
[Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod)|/Subtype /T /MK /A /AA
[Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod)|/Subtype /H /MK /A /AA /BS /Parent
[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)|/R /BC /BG /CA /RC /AC /I /RI /IX /IF /TP
[Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.rakumod)|/Subtype /FixedPrint
[Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.rakumod)|/Type /Matrix /H /V
[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod)|/Subtype /QuadPoints /IC /RO /OverlayText /Repeat /DA /Q
[Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.rakumod)|/Type /S /Next
[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod)|/E /X /D /U /Fo /Bl /PO /PC /PV /PI
[Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.rakumod)|/O /C
[Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.rakumod)|/K /F /V /C
[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod)|/WC /WS /DS /WP /DP
[Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.rakumod)|/S /D
[Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.rakumod)|/S /F /D /NewWindow
[Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod)|/S /F /D /NewWindow /T
[Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod)|/R /N /P /A /T
[Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod)|/S /F /Win /Mac /Unix /NewWindow
[Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.rakumod)|/F /D /O /P
[Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.rakumod)|/S /F /D /B
[Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.rakumod)|/S /URI /IsMap
[Table 207 – Entry in a URI dictionary](gen/lib/ISO_32000/Table_207-Entry_in_a_URI_dictionary.rakumod)|/Base
[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod)|/S /Sound /Volume /Synchronous /Repeat /Mix
[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.rakumod)|/S /Annotation /T /Operation
[Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.rakumod)|/S /T /H
[Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.rakumod)|/S /N
[Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.rakumod)|/S /State /PreserveRB
[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod)|/S /R /AN /OP /JS
[Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.rakumod)|/S /Trans
[Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.rakumod)|/S /TA /V
[Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.rakumod)|/S /JS
[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod)|/Fields /NeedAppearances /SigFlags /CO /DR /DA /Q /XFA
[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod)|/FT /Parent /Kids /T /TU /TM /Ff /V /DV /AA
[Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.rakumod)|/DA /Q /DS /RV
[Table 227 – Additional entry specific to check box and radio button fields](gen/lib/ISO_32000/Table_227-Additional_entry_specific_to_check_box_and_radio_button_fields.rakumod)|/Opt
[Table 229 – Additional entry specific to a text field](gen/lib/ISO_32000/Table_229-Additional_entry_specific_to_a_text_field.rakumod)|/MaxLen
[Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.rakumod)|/Opt /TI /I
[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.rakumod)|/Lock /SV
[Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.rakumod)|/Type /Action /Fields
[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)|/Type /Ff /Filter /SubFilter /DigestMethod /V /Cert /Reasons /MDP /TimeStamp /LegalAttestation /AddRevInfo
[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)|/Type /Ff /Subject /SubjectDN /KeyUsage /Issuer /OID /URL /URLType
[Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.rakumod)|/S /F /Fields /Flags
[Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.rakumod)|/S /Fields /Flags
[Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.rakumod)|/S /F
[Table 241 – Entry in the FDF trailer dictionary](gen/lib/ISO_32000/Table_241-Entry_in_the_FDF_trailer_dictionary.rakumod)|/Root
[Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.rakumod)|/Version /FDF
[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)|/F /ID /Fields /Status /Pages /Encoding /Annots /Differences /Target /EmbeddedFDFs /JavaScript
[Table 244 – Additional entry in an embedded file stream dictionary for an encrypted FDF file](gen/lib/ISO_32000/Table_244-Additional_entry_in_an_embedded_file_stream_dictionary_for_an_encrypted_FDF_file.rakumod)|/EncryptionRevision
[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.rakumod)|/Before /After /AfterPermsReady /Doc
[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)|/Kids /T /V /Ff /SetFf /ClrFf /F /SetF /ClrF /AP /APRef /IF /Opt /A /AA /RV
[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.rakumod)|/SW /S /A /FB
[Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.rakumod)|/Templates /Info
[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.rakumod)|/TRef /Fields /Rename
[Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.rakumod)|/Name /F
[Table 251 – Additional entry for annotation dictionaries in an FDF file](gen/lib/ISO_32000/Table_251-Additional_entry_for_annotation_dictionaries_in_an_FDF_file.rakumod)|/Page
[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)|/Type /Filter /SubFilter /Contents /Cert /ByteRange /Reference /Changes /Name /M /Location /Reason /ContactInfo /R /V /Prop_Build /Prop_AuthTime /Prop_AuthType
[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod)|/Type /TransformMethod /TransformParams /Data /DigestMethod
[Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.rakumod)|/Type /P /V
[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)|/Type /Document /Msg /V /Annots /Form /Signature /EF /P
[Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.rakumod)|/Type /Action /Fields /V
[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.rakumod)|/DocMDP /UR3
[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)|/JavaScriptActions /LaunchActions /URIActions /MovieActions /SoundActions /HideAnnotationActions /GoToRemoteActions /AlternateImages /ExternalStreams /TrueTypeFonts /ExternalRefXobjects /ExternalOPIdicts /NonEmbeddedFonts /DevDepGS_OP /DevDepGS_HT /DevDepGS_TR /DevDepGS_UCR /DevDepGS_BG /DevDepGS_FL /Annotations /OptionalContent /Attestation
[Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.rakumod)|/Type /BBox /Name /Measure
[Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.rakumod)|/Type /Subtype
[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod)|/R /X /Y /D /A /T /S /O /CYX
[Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod)|/Type /U /C /F /D /FD /RT /RD /PS /SS /O
[Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.rakumod)|/Type /S /RH
[Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.rakumod)|/Type /S /Script
[Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod)|/Type /S /N /MH /BE
[Table 267 – Entries in a rendition MH/BE dictionary](gen/lib/ISO_32000/Table_267-Entries_in_a_rendition_MH-BE_dictionary.rakumod)|/C
[Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod)|/Type /A /C /O /S /R /D /Z /V /P /L
[Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.rakumod)|/Type /V /M
[Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.rakumod)|/Type /V /M
[Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.rakumod)|/C /P /SP
[Table 272 – Additional entries specific to a selector rendition dictionary](gen/lib/ISO_32000/Table_272-Additional_entries_specific_to_a_selector_rendition_dictionary.rakumod)|/R
[Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.rakumod)|/Type /S /N
[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod)|/D /CT /P /Alt /PL /MH /BE
[Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.rakumod)|/Type /TF
[Table 276 – Entries in a media clip data MH/BE dictionary](gen/lib/ISO_32000/Table_276-Entries_in_a_media_clip_data_MH-BE_dictionary.rakumod)|/BU
[Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.rakumod)|/D /Alt /MH /BE
[Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.rakumod)|/B /E
[Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.rakumod)|/Type /PL /MH /BE
[Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod)|/V /C /F /D /A /RC
[Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.rakumod)|/Type /S /T
[Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.rakumod)|/Type /MH /BE
[Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod)|/W /B /O /M /F
[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod)|/Type /D /RT /P /O /T /UC /R /TT
[Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.rakumod)|/Type /S
[Table 286 – Additional entries in a media offset time dictionary](gen/lib/ISO_32000/Table_286-Additional_entries_in_a_media_offset_time_dictionary.rakumod)|/T
[Table 287 – Additional entries in a media offset frame dictionary](gen/lib/ISO_32000/Table_287-Additional_entries_in_a_media_offset_frame_dictionary.rakumod)|/F
[Table 288 – Additional entries in a media offset marker dictionary](gen/lib/ISO_32000/Table_288-Additional_entries_in_a_media_offset_marker_dictionary.rakumod)|/M
[Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.rakumod)|/Type /S /V
[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.rakumod)|/Type /MU /A /NU
[Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.rakumod)|/Type /PID /MH /BE
[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod)|/Type /U /L /LI /H /HI /OS
[Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod)|/Type /R /C /B /E /CO /CP
[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.rakumod)|/F /Aspect /Rotate /Poster
[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)|/Start /Duration /Rate /Volume /ShowControls /Mode /Synchronous /FWScale /FWPosition
[Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.rakumod)|/Type /Subtype /Resources /StartResource
[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod)|/Subtype /3DD /3DV /3DA /3DI /3DB
[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod)|/A /AIS /D /DIS /TB /NP
[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)|/Type /Subtype /VA /DV /Resources /OnInstantiate /AN
[Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.rakumod)|/Type /Subtype /PC /TM
[Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.rakumod)|/Type /3DD
[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)|/Type /XN /IN /MS /C2W /U3DPath /CO /P /O /BG /RM /LS /SA /NA /NR
[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod)|/Subtype /CS /F /N /FOV /PS /OS /OB
[Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod)|/Type /Subtype /CS /C /EA
[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod)|/Type /Subtype /AC /FC /O /CV
[Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.rakumod)|/Type /Subtype
[Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod)|/Type /C /O /PO /PC /IV /IC
[Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod)|/Type /N /O /V /M
[Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod)|/Type /Subtype /MD5 /3DA /3DV
[Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.rakumod)|/Type /Subtype
[Table 316 – Additional entry for components having metadata](gen/lib/ISO_32000/Table_316-Additional_entry_for_components_having_metadata.rakumod)|/Metadata
[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)|/Title /Author /Subject /Keywords /Creator /Producer /CreationDate /ModDate /Trapped
[Table 318 – Entries in a page-piece dictionary](gen/lib/ISO_32000/Table_318-Entries_in_a_page-piece_dictionary.rakumod)|
[Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.rakumod)|/LastModified /Private
[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.rakumod)|/Marked /UserProperties /Suspects
[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)|/Type /K /IDTree /ParentTree /ParentTreeNextKey /RoleMap /ClassMap
[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)|/Type /S /P /ID /Pg /K /A /C /R /T /Lang /Alt /E /ActualText
[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod)|/Type /Pg /Stm /StmOwn /MCID
[Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.rakumod)|/Type /Pg /Obj
[Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.rakumod)|/StructParent /StructParents
[Table 327 – Entry common to all attribute object dictionaries](gen/lib/ISO_32000/Table_327-Entry_common_to_all_attribute_object_dictionaries.rakumod)|/O
[Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.rakumod)|/O /P
[Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.rakumod)|/N /V /F /H
[Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.rakumod)|/Type /BBox /Attached /Subtype
[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)|/Placement /WritingMode /BackgroundColor /BorderColor /BorderStyle /BorderThickness /Padding /Color
[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)|/SpaceBefore /SpaceAfter /StartIndent /EndIndent /TextIndent /TextAlign /BBox /Width /Height /BlockAlign /InlineAlign /TBorderStyle /TPadding
[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)|/BaselineShift /LineHeight /TextDecorationColor /TextDecorationThickness /TextDecorationType /RubyAlign /RubyPosition /GlyphOrientationVertical
[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.rakumod)|/ColumnCount /ColumnGap /ColumnWidths
[Table 347 – Standard list attribute](gen/lib/ISO_32000/Table_347-Standard_list_attribute.rakumod)|/ListNumbering
[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.rakumod)|/Role /checked /Desc
[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)|/RowSpan /ColSpan /Headers /Scope /Summary
[Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.rakumod)|/V /C
[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod)|/Type /S /ID /O /SI /CT /TS
[Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.rakumod)|/S /T /TID
[Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.rakumod)|/S /R
[Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod)|/AU /TS /E /S /C
[Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.rakumod)|/U /C
[Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod)|/URL /L /F /P /CT /H /S
[Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.rakumod)|/G /C
[Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.rakumod)|/CropBox /BleedBox /TrimBox /ArtBox
[Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.rakumod)|/C /W /S /D
[Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.rakumod)|/Subtype /MN
[Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.rakumod)|/MarkStyle /Colorants
[Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.rakumod)|/Pages /DeviceColorant /ColorSpace
[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)|/Type /S /OutputCondition /OutputConditionIdentifier /RegistryName /Info /DestOutputProfile
[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)|/Subtype /LastModified /Version /AnnotStates /FontFauxing
[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.rakumod)|/PCM /SeparationColorNames /TrapRegions /TrapStyles
[Table 368 – Entry in an OPI version dictionary](gen/lib/ISO_32000/Table_368-Entry_in_an_OPI_version_dictionary.rakumod)|
[Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)|/Linearized /L /H /O /E /N /T /P

## Entry to role mappings

Entry|ISO_32000 Roles
----|-----
/3DA|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod)
/3DB|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod)
/3DD|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod) [Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.rakumod)
/3DI|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod)
/3DV|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod)
/A|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.rakumod) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.rakumod) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod) [Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod) [Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.rakumod) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)
/AA|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/AC|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod)
/AIS|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod)
/AN|[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)
/AP|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/APRef|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/AS|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)
/AU|[Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod)
/AccurateScreens|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod)
/AcroForm|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Action|[Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.rakumod) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.rakumod)
/ActualText|[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)
/AddRevInfo|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)
/After|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.rakumod)
/AfterPermsReady|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.rakumod)
/Alt|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)
/Alternate|[Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.rakumod)
/AlternateImages|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/AlternatePresentations|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/Alternates|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/Angle|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod)
/AnnotStates|[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)
/Annotation|[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.rakumod)
/Annotations|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Annots|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/AntiAlias|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod)
/ArtBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.rakumod)
/Ascent|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Aspect|[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.rakumod)
/Attached|[Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.rakumod)
/Attestation|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/AuthEvent|[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.rakumod)
/Author|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/AvgWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/B|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.rakumod) [Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.rakumod) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod)
/BBox|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod) [Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.rakumod) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.rakumod) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/BC|[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)
/BE|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.rakumod) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.rakumod) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.rakumod) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.rakumod)
/BG|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/BG2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/BM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/BS|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod) [Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod)
/BU|[Table 276 – Entries in a media clip data MH/BE dictionary](gen/lib/ISO_32000/Table_276-Entries_in_a_media_clip_data_MH-BE_dictionary.rakumod)
/Background|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod)
/BackgroundColor|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/Base|[Table 207 – Entry in a URI dictionary](gen/lib/ISO_32000/Table_207-Entry_in_a_URI_dictionary.rakumod)
/BaseEncoding|[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.rakumod)
/BaseFont|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod)
/BaseState|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/BaseVersion|[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.rakumod)
/BaselineShift|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/Before|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.rakumod)
/BitsPerComponent|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/BitsPerCoordinate|[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod)
/BitsPerFlag|[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod)
/BitsPerSample|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod)
/Bl|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod)
/BlackIs1|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/BlackPoint|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.rakumod) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.rakumod) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.rakumod)
/BleedBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.rakumod)
/BlockAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/Border|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)
/BorderColor|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/BorderStyle|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/BorderThickness|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/Bounds|[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.rakumod)
/BoxColorInfo|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/ByteRange|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/C|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.rakumod) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 267 – Entries in a rendition MH/BE dictionary](gen/lib/ISO_32000/Table_267-Entries_in_a_rendition_MH-BE_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.rakumod) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod) [Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.rakumod) [Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.rakumod) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.rakumod)
/C0|[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.rakumod)
/C1|[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.rakumod)
/C2W|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/CA|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)
/CF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod)
/CFM|[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.rakumod)
/CI|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod)
/CIDSet|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.rakumod)
/CIDSystemInfo|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod) [Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod)
/CIDToGIDMap|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)
/CL|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod)
/CMapName|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod)
/CO|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/CP|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod)
/CS|[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod)
/CT|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod)
/CTM|[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.rakumod)
/CV|[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod)
/CYX|[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod)
/Cap|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod)
/CapHeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Category|[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.rakumod)
/CenterWindow|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/Cert|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Changes|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/CharProcs|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)
/CharSet|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/CheckSum|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod)
/ClassMap|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)
/ClrF|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/ClrFf|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/ColSpan|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)
/Collection|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Color|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/ColorSpace|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod) [Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.rakumod) [Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.rakumod)
/ColorTransform|[Table 13 – Optional parameter for the DCTDecode filter](gen/lib/ISO_32000/Table_13-Optional_parameter_for_the_DCTDecode_filter.rakumod)
/Colorants|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.rakumod) [Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.rakumod)
/Colors|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod)
/ColumnCount|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.rakumod)
/ColumnGap|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.rakumod)
/ColumnWidths|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.rakumod)
/Columns|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod) [Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/Components|[Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.rakumod)
/Configs|[Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.rakumod)
/ContactInfo|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Contents|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Coords|[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.rakumod) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.rakumod)
/Count|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.rakumod) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod)
/CreationDate|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/Creator|[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.rakumod) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/CreatorInfo|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/CropBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.rakumod)
/D|[Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.rakumod) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.rakumod) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.rakumod) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.rakumod) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.rakumod) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.rakumod) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.rakumod) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.rakumod)
/DA|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.rakumod)
/DIS|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod)
/DL|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod)
/DOS|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod)
/DP|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod)
/DR|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod)
/DS|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.rakumod)
/DV|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)
/DW|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)
/DW2|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)
/DamagedRowsBeforeError|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/Data|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod)
/Decode|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/DecodeParms|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod)
/Default|[Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.rakumod)
/DefaultForPrinting|[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.rakumod)
/Desc|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.rakumod)
/DescendantFonts|[Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod)
/Descent|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Dest|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod)
/DestOutputProfile|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/Dests|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/DevDepGS_BG|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DevDepGS_FL|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DevDepGS_HT|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DevDepGS_OP|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DevDepGS_TR|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DevDepGS_UCR|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/DeviceColorant|[Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.rakumod)
/Di|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod)
/Differences|[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/DigestMethod|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod)
/Direction|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/DisplayDocTitle|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/Dm|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod)
/Doc|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.rakumod)
/DocMDP|[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.rakumod)
/Document|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/Domain|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.rakumod) [Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.rakumod) [Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.rakumod) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.rakumod)
/DotGain|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.rakumod)
/Duplex|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/Dur|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod)
/Duration|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/E|[Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/EA|[Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod)
/EF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/EFF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod)
/EarlyChange|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod)
/EmbeddedFDFs|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/EmbeddedFiles|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/Encode|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod) [Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.rakumod)
/EncodedByteAlign|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/Encoding|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/Encrypt|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod)
/EncryptMetadata|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod) [Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.rakumod)
/EncryptionRevision|[Table 244 – Additional entry in an embedded file stream dictionary for an encrypted FDF file](gen/lib/ISO_32000/Table_244-Additional_entry_in_an_embedded_file_stream_dictionary_for_an_encrypted_FDF_file.rakumod)
/EndIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/EndOfBlock|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/EndOfLine|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/Event|[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.rakumod)
/ExData|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod)
/Export|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/ExtGState|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod)
/Extend|[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.rakumod) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.rakumod)
/Extends|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.rakumod)
/ExtensionLevel|[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.rakumod)
/Extensions|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/ExternalOPIdicts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/ExternalRefXobjects|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/ExternalStreams|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/F|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.rakumod) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.rakumod) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.rakumod) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.rakumod) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.rakumod) [Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod) [Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod) [Table 287 – Additional entries in a media offset frame dictionary](gen/lib/ISO_32000/Table_287-Additional_entries_in_a_media_offset_frame_dictionary.rakumod) [Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod)
/FB|[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.rakumod)
/FC|[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod)
/FD|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod)
/FDF|[Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.rakumod)
/FDecodeParms|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod)
/FFilter|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod)
/FL|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/FOV|[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod)
/FS|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.rakumod)
/FT|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod)
/FWPosition|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/FWScale|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/Ff|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/Fields|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod) [Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.rakumod) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.rakumod) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod) [Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.rakumod) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.rakumod)
/Filter|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod) [Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/First|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.rakumod) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod)
/FirstChar|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)
/FitWindow|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/FixedPrint|[Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.rakumod)
/Flags|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.rakumod) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.rakumod)
/Fo|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod)
/Font|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/FontBBox|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/FontDescriptor|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)
/FontFamily|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/FontFauxing|[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)
/FontFile|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.rakumod)
/FontFile2|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.rakumod)
/FontFile3|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.rakumod)
/FontMatrix|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)
/FontName|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/FontStretch|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/FontWeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Form|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/FormType|[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)
/Frequency|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod)
/Function|[Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.rakumod) [Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.rakumod) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.rakumod) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.rakumod) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.rakumod)
/FunctionType|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.rakumod)
/Functions|[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.rakumod)
/G|[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod) [Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.rakumod)
/Gamma|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.rakumod) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.rakumod)
/GlyphOrientationVertical|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/GoToRemoteActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Group|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)
/H|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.rakumod) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.rakumod) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/HI|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod)
/HT|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/HalftoneName|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.rakumod)
/HalftoneType|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.rakumod)
/Headers|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)
/Height|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/Height2|[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod)
/HideAnnotationActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/HideMenubar|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/HideToolbar|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/HideWindowUI|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/I|[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.rakumod) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.rakumod) [Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.rakumod)
/IC|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod)
/ID|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod)
/IDS|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/IDTree|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)
/IF|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/IN|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/IRT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod)
/IT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod)
/IV|[Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod)
/IX|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)
/Image|[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.rakumod)
/ImageMask|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/Index|[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod)
/Info|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod) [Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.rakumod) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/InkList|[Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.rakumod)
/InlineAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/Intent|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.rakumod) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/Interpolate|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/IsMap|[Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.rakumod)
/Issuer|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)
/ItalicAngle|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/JBIG2Globals|[Table 12 – Optional parameter for the JBIG2Decode filter](gen/lib/ISO_32000/Table_12-Optional_parameter_for_the_JBIG2Decode_filter.rakumod)
/JS|[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod) [Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.rakumod)
/JavaScript|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/JavaScriptActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/K|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod) [Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.rakumod) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.rakumod) [Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)
/KeyUsage|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)
/Keywords|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/Kids|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.rakumod) [Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.rakumod) [Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.rakumod) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/L|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/LC|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/LE|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod)
/LI|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod)
/LJ|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/LL|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod)
/LLE|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod)
/LLO|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod)
/LS|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/LW|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/Lang|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod)
/Language|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/Last|[Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod)
/LastChar|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)
/LastModified|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.rakumod) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)
/LaunchActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Leading|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Legal|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/LegalAttestation|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)
/Length|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.rakumod) [Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod) [Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.rakumod)
/Length1|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod)
/Length2|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod)
/Length3|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod)
/Level1|[Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.rakumod)
/Limits|[Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.rakumod) [Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.rakumod)
/LineHeight|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/Linearized|[Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/ListMode|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/ListNumbering|[Table 347 – Standard list attribute](gen/lib/ISO_32000/Table_347-Standard_list_attribute.rakumod)
/Location|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Lock|[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.rakumod)
/Locked|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/M|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.rakumod) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.rakumod) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod) [Table 288 – Additional entries in a media offset marker dictionary](gen/lib/ISO_32000/Table_288-Additional_entries_in_a_media_offset_marker_dictionary.rakumod) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod)
/MCID|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod)
/MD5|[Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod)
/MDP|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)
/MH|[Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.rakumod) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.rakumod) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.rakumod) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.rakumod)
/MK|[Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod)
/ML|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/MN|[Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.rakumod)
/MS|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/MU|[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.rakumod)
/Mac|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod)
/MarkInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/MarkStyle|[Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.rakumod)
/Marked|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.rakumod)
/Mask|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/Matrix|[Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.rakumod) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod) [Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.rakumod)
/Matte|[Table 146 – Additional entry in a soft-mask image dictionary](gen/lib/ISO_32000/Table_146-Additional_entry_in_a_soft-mask_image_dictionary.rakumod)
/MaxLen|[Table 229 – Additional entry specific to a text field](gen/lib/ISO_32000/Table_229-Additional_entry_specific_to_a_text_field.rakumod)
/MaxWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Measure|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.rakumod)
/MediaBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/Metadata|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod) [Table 316 – Additional entry for components having metadata](gen/lib/ISO_32000/Table_316-Additional_entry_for_components_having_metadata.rakumod)
/MissingWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Mix|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod)
/MixingHints|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.rakumod)
/ModDate|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/Mode|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/Movie|[Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.rakumod)
/MovieActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Msg|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/N|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.rakumod) [Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.rakumod) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.rakumod) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.rakumod) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod) [Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.rakumod) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/NA|[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/NM|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)
/NP|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod)
/NR|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/NU|[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.rakumod)
/Name|[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.rakumod) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod) [Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.rakumod) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.rakumod) [Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.rakumod) [Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.rakumod)
/Names|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.rakumod)
/NeedAppearances|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod)
/NeedsRendering|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/NewWindow|[Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.rakumod) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod)
/Next|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.rakumod)
/NonEmbeddedFonts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/NonFullScreenPageMode|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/NumCopies|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/Nums|[Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.rakumod)
/O|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.rakumod) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod) [Table 327 – Entry common to all attribute object dictionaries](gen/lib/ISO_32000/Table_327-Entry_common_to_all_attribute_object_dictionaries.rakumod) [Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.rakumod) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/OB|[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod)
/OC|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)
/OCGs|[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.rakumod) [Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.rakumod) [Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.rakumod)
/OCProperties|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/OFF|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/OID|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)
/ON|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/OP|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod)
/OPI|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)
/OPM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/OS|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod)
/Obj|[Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.rakumod)
/OnInstantiate|[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)
/Open|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.rakumod)
/OpenAction|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Operation|[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.rakumod)
/Opt|[Table 227 – Additional entry specific to check box and radio button fields](gen/lib/ISO_32000/Table_227-Additional_entry_specific_to_check_box_and_radio_button_fields.rakumod) [Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/OptionalContent|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Order|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/Ordering|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.rakumod)
/Outlines|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/OutputCondition|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/OutputConditionIdentifier|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/OutputIntents|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/OverlayText|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod)
/P|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod) [Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.rakumod) [Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.rakumod) [Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.rakumod) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.rakumod) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.rakumod) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.rakumod) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.rakumod) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/PA|[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod)
/PC|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod)
/PCM|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.rakumod)
/PI|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod)
/PID|[Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.rakumod)
/PL|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.rakumod) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.rakumod)
/PO|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod)
/PS|[Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod)
/PV|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod)
/PZ|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/Padding|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/Page|[Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.rakumod) [Table 251 – Additional entry for annotation dictionaries in an FDF file](gen/lib/ISO_32000/Table_251-Additional_entry_for_annotation_dictionaries_in_an_FDF_file.rakumod)
/PageElement|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/PageLabels|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/PageLayout|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/PageMode|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Pages|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod) [Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.rakumod)
/PaintType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod)
/Params|[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.rakumod)
/Parent|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod)
/ParentTree|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)
/ParentTreeNextKey|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)
/Pattern|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod)
/PatternType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod)
/Perms|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Pg|[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod) [Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.rakumod)
/PickTrayByPDFSize|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/PieceInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)
/Placement|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/Popup|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod)
/Poster|[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.rakumod)
/Predictor|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.rakumod)
/PresSteps|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/PreserveRB|[Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.rakumod)
/Prev|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod)
/Print|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/PrintArea|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/PrintClip|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/PrintPageRange|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/PrintScaling|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/PrintingOrder|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.rakumod)
/Private|[Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.rakumod)
/ProcSet|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod)
/Process|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.rakumod)
/Producer|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/Prop_AuthTime|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Prop_AuthType|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Prop_Build|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Properties|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod)
/Q|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.rakumod)
/QuadPoints|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod) [Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.rakumod) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod)
/R|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 272 – Additional entries specific to a selector rendition dictionary](gen/lib/ISO_32000/Table_272-Additional_entries_specific_to_a_selector_rendition_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.rakumod)
/RBGroups|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.rakumod)
/RC|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod)
/RD|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod) [Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod)
/RF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod)
/RH|[Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.rakumod)
/RI|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)
/RM|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/RO|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod)
/RT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod)
/RV|[Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/Range|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.rakumod) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.rakumod) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.rakumod)
/Rate|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/Reason|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Reasons|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)
/Recipients|[Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.rakumod) [Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.rakumod)
/Rect|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod)
/Ref|[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod)
/Reference|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Registry|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.rakumod)
/RegistryName|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/Rename|[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.rakumod)
/Renditions|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/Repeat|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod)
/Requirements|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/ResFork|[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.rakumod)
/Resources|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.rakumod) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)
/Role|[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.rakumod)
/RoleMap|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod)
/Root|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod) [Table 241 – Entry in the FDF trailer dictionary](gen/lib/ISO_32000/Table_241-Entry_in_the_FDF_trailer_dictionary.rakumod)
/Rotate|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.rakumod)
/RowSpan|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)
/Rows|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.rakumod)
/RubyAlign|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/RubyPosition|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/S|[Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.rakumod) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod) [Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.rakumod) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.rakumod) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.rakumod) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.rakumod) [Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.rakumod) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.rakumod) [Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.rakumod) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.rakumod) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.rakumod) [Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.rakumod) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod) [Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.rakumod) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.rakumod) [Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.rakumod) [Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.rakumod) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.rakumod) [Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.rakumod) [Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.rakumod) [Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.rakumod) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.rakumod) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.rakumod) [Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.rakumod) [Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.rakumod) [Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.rakumod) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.rakumod) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.rakumod) [Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.rakumod) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod) [Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.rakumod) [Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.rakumod) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.rakumod) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/SA|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/SE|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod)
/SI|[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod)
/SM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/SMask|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/SMaskInData|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod)
/SP|[Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.rakumod)
/SS|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod)
/SV|[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.rakumod)
/SW|[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.rakumod)
/Schema|[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod)
/Scope|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)
/Script|[Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.rakumod)
/SeparationColorNames|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.rakumod)
/SeparationInfo|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/SetF|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/SetFf|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod)
/Shading|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod)
/ShadingType|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.rakumod)
/ShowControls|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/SigFlags|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod)
/Signature|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod)
/Size|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.rakumod) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod) [Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.rakumod) [Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.rakumod)
/Solidities|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.rakumod)
/Sort|[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod)
/Sound|[Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.rakumod) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod)
/SoundActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/SpaceAfter|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/SpaceBefore|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/SpiderInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/SpotFunction|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod)
/St|[Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.rakumod)
/Start|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/StartIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/StartResource|[Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.rakumod)
/State|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod) [Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.rakumod)
/StateModel|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod)
/Status|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/StemH|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/StemV|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/Stm|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod)
/StmF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod)
/StmOwn|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod)
/StrF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod)
/StructParent|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.rakumod)
/StructParents|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.rakumod)
/StructTreeRoot|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Style|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.rakumod)
/SubFilter|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod)
/Subj|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod)
/Subject|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/SubjectDN|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)
/Subtype|[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.rakumod) [Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.rakumod) [Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.rakumod) [Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod) [Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.rakumod) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.rakumod) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.rakumod) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.rakumod) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.rakumod) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.rakumod) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod) [Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.rakumod) [Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.rakumod) [Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.rakumod) [Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.rakumod) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.rakumod) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.rakumod) [Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.rakumod) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.rakumod) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.rakumod) [Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.rakumod) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.rakumod) [Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.rakumod) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.rakumod) [Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.rakumod) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.rakumod) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.rakumod) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod) [Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.rakumod) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod) [Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.rakumod) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.rakumod) [Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.rakumod) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)
/Summary|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.rakumod)
/Supplement|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.rakumod)
/Suspects|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.rakumod)
/Sy|[Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.rakumod)
/Synchronous|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod) [Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/T|[Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.rakumod) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.rakumod) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.rakumod) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.rakumod) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.rakumod) [Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.rakumod) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.rakumod) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 286 – Additional entries in a media offset time dictionary](gen/lib/ISO_32000/Table_286-Additional_entries_in_a_media_offset_time_dictionary.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.rakumod) [Table F.1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.rakumod)
/TA|[Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.rakumod)
/TB|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.rakumod)
/TBorderStyle|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/TF|[Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.rakumod)
/TI|[Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.rakumod)
/TID|[Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.rakumod)
/TK|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/TM|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.rakumod)
/TP|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.rakumod)
/TPadding|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/TR|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod)
/TR2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/TRef|[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.rakumod)
/TS|[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.rakumod)
/TT|[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod)
/TU|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod)
/Tabs|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/Target|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.rakumod)
/TemplateInstantiated|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/Templates|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod) [Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.rakumod)
/TextAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/TextDecorationColor|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/TextDecorationThickness|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/TextDecorationType|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.rakumod)
/TextIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/Threads|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Thumb|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/TilingType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod)
/TimeStamp|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod)
/Title|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.rakumod) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/ToUnicode|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod)
/Trans|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.rakumod)
/TransferFunction|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod)
/TransformMethod|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod)
/TransformParams|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod)
/TrapRegions|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.rakumod)
/TrapStyles|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.rakumod)
/Trapped|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.rakumod)
/TrimBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.rakumod)
/TrueTypeFonts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/Type|[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.rakumod) [Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.rakumod) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod) [Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.rakumod) [Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.rakumod) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.rakumod) [Table 48 – Entries in a collection item dictionary](gen/lib/ISO_32000/Table_48-Entries_in_a_collection_item_dictionary.rakumod) [Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.rakumod) [Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.rakumod) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.rakumod) [Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.rakumod) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.rakumod) [Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.rakumod) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.rakumod) [Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.rakumod) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod) [Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.rakumod) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod) [Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.rakumod) [Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod) [Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.rakumod) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.rakumod) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.rakumod) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod) [Table 156 – Entries in a collection schema dictionary](gen/lib/ISO_32000/Table_156-Entries_in_a_collection_schema_dictionary.rakumod) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.rakumod) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.rakumod) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.rakumod) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.rakumod) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.rakumod) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.rakumod) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.rakumod) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.rakumod) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.rakumod) [Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.rakumod) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod) [Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.rakumod) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.rakumod) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.rakumod) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.rakumod) [Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.rakumod) [Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.rakumod) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.rakumod) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.rakumod) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.rakumod) [Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.rakumod) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.rakumod) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.rakumod) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.rakumod) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod) [Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.rakumod) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.rakumod) [Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.rakumod) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.rakumod) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.rakumod) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.rakumod) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.rakumod) [Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.rakumod) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.rakumod) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.rakumod) [Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.rakumod) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.rakumod) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.rakumod) [Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.rakumod) [Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.rakumod) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.rakumod) [Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.rakumod) [Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.rakumod) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.rakumod) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.rakumod) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.rakumod)
/U|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.rakumod) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.rakumod) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.rakumod) [Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.rakumod)
/U3DPath|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/UC|[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.rakumod)
/UCR|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/UCR2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/UF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod)
/UR3|[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.rakumod)
/URI|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.rakumod)
/URIActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.rakumod)
/URL|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.rakumod)
/URLS|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.rakumod)
/URLType|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.rakumod)
/Unix|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod)
/Usage|[Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.rakumod)
/UseCMap|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod)
/User|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/UserProperties|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.rakumod)
/UserUnit|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/V|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.rakumod) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.rakumod) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.rakumod) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.rakumod) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.rakumod) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.rakumod) [Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.rakumod) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.rakumod) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.rakumod) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.rakumod) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.rakumod) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.rakumod) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.rakumod) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.rakumod) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.rakumod) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.rakumod) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.rakumod) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.rakumod) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.rakumod) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.rakumod) [Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.rakumod)
/VA|[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.rakumod)
/VE|[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.rakumod)
/VP|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.rakumod)
/Version|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod) [Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.rakumod) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.rakumod)
/Vertices|[Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.rakumod)
/VerticesPerRow|[Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.rakumod)
/View|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.rakumod)
/ViewArea|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/ViewClip|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.rakumod)
/ViewerPreferences|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.rakumod)
/Volume|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.rakumod) [Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.rakumod)
/W|[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.rakumod) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.rakumod) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.rakumod) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.rakumod)
/W2|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.rakumod)
/WC|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod)
/WMode|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.rakumod)
/WP|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod)
/WS|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.rakumod)
/WhitePoint|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.rakumod) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.rakumod) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.rakumod)
/Width|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.rakumod) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.rakumod) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.rakumod)
/Width2|[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.rakumod)
/Widths|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.rakumod) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.rakumod)
/Win|[Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.rakumod)
/WritingMode|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.rakumod)
/X|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.rakumod) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod)
/XFA|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.rakumod)
/XHeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.rakumod)
/XN|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.rakumod)
/XObject|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.rakumod)
/XRefStm|[Table 19 – Additional entries in a hybrid-reference file’s trailer dictionary](gen/lib/ISO_32000/Table_19-Additional_entries_in_a_hybrid-reference_files_trailer_dictionary.rakumod)
/XStep|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod)
/Xsquare|[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod)
/Y|[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.rakumod)
/YStep|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.rakumod)
/Ysquare|[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.rakumod)
/Z|[Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.rakumod)
/Zoom|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.rakumod)
/ca|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/checked|[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.rakumod)
/color|[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.rakumod)
/flatness|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)
/halftone|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)
/op|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.rakumod)
/overprint|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)
/smoothness|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)
/transfer|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.rakumod)
