/*
 *  Copyright 1999-2001 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details.
 */

package com.vizdom.util;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.Reader;
import java.io.Writer;
import java.io.UnsupportedEncodingException;
import com.vizdom.test.TestAgent;
import junit.framework.TestCase;

/**
 * Performs some basic checking of the CharacterEncoder class. 
 * 
 * @author Gennis Emerson
 * @author John Lacey
 * @version $Revision: 1.13 $
 */
public class TestCharacterEncoder extends TestCase
{
    /**
     * Unit test.
     *
     * @param args optional test method names
     * @exception Exception if a test throws an error
     */
    public static void main(String[] args) throws Exception
    {
        TestAgent.mainForClass(TestCharacterEncoder.class, args);
    }

    
    /**
     * JUnit constructor.
     *
     * @param testName a test method name
     */
    public TestCharacterEncoder(String testName)
    {
        super(testName);
    }

    
    public void testByteToChar() throws Exception
    {
        final int SIZE = 256;
        
        byte[] pattern = new byte[SIZE];
        for (int b = 0; b < SIZE; b++)
            pattern[b] = (byte) b;

        String[] encodings = {
            "DEC_MCS", "HP_Roman8", "HP_Roman9", "Europa_3",
        };

        for (int i = 0; i < encodings.length; i++)
        {
            try
            {
                String s = CharacterEncoder.toString(pattern, encodings[i]);
                byte[] b = CharacterEncoder.toByteArray(s, encodings[i]);

                assertEquals(SIZE, s.length());
                assertEquals(SIZE, b.length);

                java.util.ArrayList mapped = new java.util.ArrayList(SIZE);
                for (int j = 0; j < SIZE; j++)
                {
                    if (s.charAt(j) != '.')
                    {
                        assertTrue(encodings[i] + " position " +
                            Integer.toHexString(j) + " already mapped",
                            !mapped.contains(new Character(s.charAt(j))));
                        mapped.add(new Character(s.charAt(j)));
                    
                        assertEquals(encodings[i] + " position " +
                            Integer.toHexString(j), j, b[j] & 0xFF);
                    }
                }
            }
            catch (UnsupportedEncodingException encEx)
            {
                System.out.println(encodings[i] + ": failure");
            }
        }
    }


    public void testCharToByte() throws Exception
    {
        final int SIZE = 256;
        
        char[] chars = new char[SIZE];
        for (int b = 0; b < SIZE; b++)
            chars[b] = (char) b;
        String pattern = new String(chars);

        String[] encodings = {
            "DEC_MCS", "HP_Roman8", "HP_Roman9", "Europa_3",
        };

        for (int i = 0; i < encodings.length; i++)
        {
            try
            {
                byte[] b = CharacterEncoder.toByteArray(pattern, encodings[i]);
                String s = CharacterEncoder.toString(b, encodings[i]);

                assertEquals(SIZE, b.length);
                assertEquals(SIZE, s.length());

                java.util.ArrayList mapped = new java.util.ArrayList(SIZE);
                for (int j = 0; j < SIZE; j++)
                {
                    if ((b[j] & 0xFF) != (int) '.')
                    {
                        assertTrue(encodings[i] + " position " +
                            Integer.toHexString(j) + " already mapped",
                            !mapped.contains(new Byte(b[j])));
                        mapped.add(new Byte(b[j]));
                    
                        assertEquals(encodings[i] + " position " +
                            Integer.toHexString(j), j, s.charAt(j));
                    }
                }
            }
            catch (UnsupportedEncodingException encEx)
            {
                System.out.println(encodings[i] + ": failure");
            }
        }
    }


    public void testEuroEncodings() throws Exception
    {
        String data = "\u20ac";
        String[] encodings = {
            "DEC_MCS", "ISO8859_15_FDIS", "Cp858", "HP_Roman8", "HP_Roman9"
        };
        boolean[] hasEuro = { false, true, true, false, true };
        for (int i = 0; i < encodings.length; i++)
        {
            byte[] bytes = CharacterEncoder.toByteArray(data, encodings[i]);
            String s = CharacterEncoder.toString(bytes, encodings[i]);
            if (hasEuro[i])
                assertEquals('\u20ac', s.charAt(0));
            else
                assertEquals(0x2e, bytes[0] & 0xFF);
        }
    }
    

    public static void testToString(int dummyToAvoidJUnit)
    {
        byte[] b = {   
            (byte) 0x39, (byte) 0x38, (byte) 0x37, (byte) 0x20,
            (byte) 0xc1, (byte) 0xc2, (byte) 0xc3, (byte) 0xc4
        };
        
        try
        {
            String foo = null;
            //foo = CharacterEncoder.toString(b, "DEC_MCS");
            //foo = CharacterEncoder.toString(b, "HP_Roman8");
            //foo = CharacterEncoder.toString(b, "Europa_3");
            foo = CharacterEncoder.toString(b, "ASCII");
            System.out.println(foo);
            for (int i = 0; i < foo.length(); i++)
            {
                System.out.println(foo.charAt(i) + " " +
                    Integer.toHexString(foo.charAt(i)));
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    
    public static void testToByteArray(int dummyToAvoidJUnit)
    {
        String s = "987 \u00c0\u00c1\u00c2\u00c3";
        try
        {
            byte[] bar = null;
            //bar = CharacterEncoder.toByteArray(s, "DEC_MCS");
            bar = CharacterEncoder.toByteArray(s, "HP_Roman8");
            //bar = CharacterEncoder.toByteArray(s, "Europa_3");
            //bar = CharacterEncoder.toByteArray(s, "ASCII");
            
            for (int i = 0; i < bar.length; i++)
            {
                System.out.println(s.charAt(i) + " | " + bar[i] + " | " +
                    Integer.toHexString(bar[i] & 0xFF));
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    
    public static void testToWriter(int dummyToAvoidJUnit)
    {
        try
        {
            Writer out = null;
            //out = CharacterEncoder.toWriter(System.out, "DEC_MCS");
            //out = CharacterEncoder.toWriter(System.out, "HP_Roman8");
            out = CharacterEncoder.toWriter(System.out, "Europa_3");

            out.write("Hello there");
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    public static void testToReader(int dummyToAvoidJUnit)
    {
        String[] encs = { "DEC_MCS", "HP_Roman8", "Europa_3" };
        for (int i = 0; i < encs.length; i++)
        {
            try
            {
                System.out.println(encs[i]);
                FileOutputStream fout = new FileOutputStream("data.txt");
                Writer out = CharacterEncoder.toWriter(fout, encs[i]);
                out.write("This is some sample data.\n");
                out.write("This is some non-ASCII Unicode data: \n");
                out.write("\u00c0\u00c1\u00c2\u00c3\u00a1\u00a2\u00a3" +
                    "\u00a4\n");
                out.close();
            
                FileInputStream fin = new FileInputStream("data.txt");
                Reader in = CharacterEncoder.toReader(fin, encs[i]);
                int c;
                while ((c = in.read()) != -1)
                    System.out.print((char) c);
                in.close();
                System.out.println();
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
    }
}
