/*
 * Copyright 1999-2000 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details. 
 */

package com.vizdom.dbd.jdbc;

import com.vizdom.ber.BerObject;
import com.vizdom.ber.BerIdentifier;
import com.vizdom.ber.BerTypes;
import com.vizdom.util.Debug;
import com.vizdom.util.UnreachableCodeException;
import java.io.*;
import java.math.BigDecimal;
import java.net.Socket;
import java.sql.*;
import java.util.Hashtable;

/**
 * This class implements a single DBD server connection. It's 
 * intended to be run as a single thread started by a
 * connection-accepting server.
 *
 * @author Gennis Emerson
 * @version $Revision: 24 $
 */
/* The handleMessage methods may throw only SQLExceptions and 
 * RuntimeExceptions, the first if they can recover, the second
 * if the connection should exit.
 */
class Connection implements Runnable
{
    /** The DBI constant for non-nullable columns. */
    private static final Integer sDbiNoNulls = new Integer(0);
    /** The DBI constant for nullable columns. */
    private static final Integer sDbiNullable = new Integer(1);   
    /** The DBI constant for nullable-unknown columns. */
    private static final Integer sDbiNullableUnknown = new Integer(2);
    /** LONG fields will be read in chunks this size. */
    private static final int sLONG_READ_BUFFER_SIZE = 8192;

    /** The current thread name, used in tracing messages. */
    private String mThreadId;
    /** The client socket. */
    private Socket mSocket;
    /**
     * The client socket's input stream. This is a BufferedInputStream 
     * in order to avoid a problem I had with <code>available</code> 
     * being > 0 but <code>read</code> returning -1.
     */
    private BufferedInputStream mIn;
    /** The client socket's output stream. */
    private BufferedOutputStream mOut;

    /** This connection's BerModule. */
    private BerDbdModule mBerModule;
    /** The JDBC Connection. */
    private java.sql.Connection mConn;
    /** 
     * True if this Connection was constructed with a pre-existing
     * JDBC connection.
     */
    private boolean preExistingConnection;
    /** A collection of the Statements currently in use. */
    private Hashtable mStatementTable;
    /** 
     * The count of Statements created. This value is used as the
     * key into the statement table for each newly-created Statement.
     */
    /* Starts at 1 in case we want the statement handle to be true in perl. */
    private int mNextHandle;

    /**
     * Constructor - initializes fields.
     *
     * @param aClient the client socket for this connection
     * @param aBerModule the BerModule this connection will use
     *      to read from/write to the socket
     * @exception IOException if an error occurs getting the socket's
     *      input or output streams
     */
    Connection(Socket aClient, BerDbdModule aBerModule) throws IOException
    {
        mSocket = aClient;

        mIn = new BufferedInputStream(mSocket.getInputStream());
        mOut = new BufferedOutputStream(mSocket.getOutputStream());
        mStatementTable = new Hashtable();
        mNextHandle = 1;

        mBerModule = aBerModule;

        preExistingConnection = false;
    }

    /**
     * Constructor. Uses a pre-existing JDBC connection.
     *
     * @param aClient the client socket for this connection
     * @param aBerModule the BerModule this connection will use
     *      to read from/write to the socket
     * @param aJdbcConnection a JDBC connection to be used
     *      by the client
     * @exception IOException if an error occurs getting the socket's
     *      input or output streams
     */
    Connection(Socket aClient, BerDbdModule aBerModule, 
        java.sql.Connection aJdbcConnection) throws IOException
    {
        this(aClient, aBerModule);
        mConn = aJdbcConnection;
        preExistingConnection = true;
    }


    /**
     * Accepts requests from client and dispatches them to the appropriate
     * method for handling. Sends response to client on request
     * completion.
     */
    public void run()
    {
        mThreadId = "[" + Thread.currentThread().getName() + "] ";

        BerObject request;
        BerObject response = null;
        boolean connected = true;

        Debug.getLogWriter(Debug.BRIEF).println(mThreadId + 
            "client started");

        while (connected)
        {
            try
            {
                /* Re-implement this, treating Connection as a Visitor
                 * on the fooRequest classes. For example,
                 * this switch would become request.handleMessage(this)
                 * and each fooRequest would have a 
                 * handleMessage(Connection conn) {conn.handleMessage(this)}
                 */
                request = mBerModule.readFrom(mIn);
                if (request == null)
                {
                    throw new FatalException("Client disconnected");
                }
                Debug.getLogWriter(Debug.BRIEF).println(mThreadId + 
                    "request: " + request);

                BerIdentifier id = request.getIdentifier();
                Debug.assert(id.getTagClass() == BerTypes.APPLICATION);
                int tagNumber = id.getTagNumber();
                switch (tagNumber)
                {
                case BerDbdModule.gDISCONNECT_REQUEST:
                    connected = false;
                    response = handleRequest((DisconnectRequest) request); 
                    break;
                
                case BerDbdModule.gCONNECT_REQUEST:
                    try
                    {
                        response = handleRequest((ConnectRequest) request);
                    }
                    catch (SQLException sql)
                    {
                        connected = false;
                        throw sql;
                    }
                    break;

                case BerDbdModule.gPING_REQUEST:
                    response = handleRequest((PingRequest) request);
                    break;
                case BerDbdModule.gCOMMIT_REQUEST: 
                    response = handleRequest((CommitRequest) request); 
                    break;
                case BerDbdModule.gROLLBACK_REQUEST: 
                    response = handleRequest((RollbackRequest) request); 
                    break;
                case BerDbdModule.gPREPARE_REQUEST: 
                    response = handleRequest((PrepareRequest) request); 
                    break;
                case BerDbdModule.gEXECUTE_REQUEST: 
                    response = handleRequest((ExecuteRequest) request); 
                    break;
                case BerDbdModule.gFETCH_REQUEST: 
                    response = handleRequest((FetchRequest) request); 
                    break;
                case BerDbdModule.gGET_CONNECTION_PROPERTY_REQUEST: 
                    response = handleRequest(
                        (GetConnectionPropertyRequest) request); 
                    break;
                case BerDbdModule.gSET_CONNECTION_PROPERTY_REQUEST: 
                    response = handleRequest(
                        (SetConnectionPropertyRequest) request); 
                    break;
                case BerDbdModule.gGET_STATEMENT_PROPERTY_REQUEST: 
                    response = handleRequest(
                        (GetStatementPropertyRequest) request); 
                    break;
                case BerDbdModule.gSET_STATEMENT_PROPERTY_REQUEST: 
                    response = handleRequest(
                        (SetStatementPropertyRequest) request); 
                    break;
                case BerDbdModule.gSTATEMENT_FINISH_REQUEST: 
                    response = handleRequest(
                        (StatementFinishRequest) request); 
                    break;
                case BerDbdModule.gSTATEMENT_DESTROY_REQUEST: 
                    response = handleRequest(
                        (StatementDestroyRequest) request); 
                    break;
                default: 
                    throw new DbdException(DbdException.gUNKNOWN_REQUEST,
                        new String[] { String.valueOf(tagNumber) });
                }
                
                if (response != null)
                {
                    response.writeTo(mOut);
                    mOut.flush(); 
                    Debug.getLogWriter(Debug.BRIEF).println(mThreadId + 
                        "response: " + response);
                    response = null;
                }
                else
                    throw new DbdException(DbdException.gNO_RESPONSE);
            }
            catch (SQLException sqlError)
            {
                try
                {
                    mSendError(sqlError);
                }
                catch (FatalException fatal)
                {
                    Debug.getLogWriter(Debug.BRIEF).print(mThreadId + 
                        "error: ");
                    fatal.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
                }
                Debug.getLogWriter(Debug.BRIEF).print(mThreadId + "error: ");
                sqlError.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
            }
            catch (IOException ioError)
            {
                connected = false;
                Debug.getLogWriter(Debug.BRIEF).print(mThreadId + "error: ");
                ioError.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
                try
                {
                    mSendError(new DbdException(
                        DbdException.gGENERIC_EXCEPTION,
                        new String[] { ioError.toString() }));
                }
                catch (FatalException fatal)
                {
                    Debug.getLogWriter(Debug.BRIEF).print(mThreadId + 
                        "error: ");
                    fatal.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
                }
            }
            catch (RuntimeException runtime)
            {
                connected = false;
                Debug.getLogWriter(Debug.BRIEF).print(mThreadId + "error: ");
                runtime.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
                try
                {
                    mSendError(new DbdException(
                        DbdException.gGENERIC_EXCEPTION,
                        new String[] { "Fatal error" }));
                }
                catch (FatalException fatal)
                {
                    Debug.getLogWriter(Debug.BRIEF).print(mThreadId + 
                        "error: ");
                    fatal.printStackTrace(Debug.getLogWriter(Debug.BRIEF));
                }
            }
        }
        
        try { mOut.close(); } catch (IOException e) { }
        try { mIn.close(); } catch (IOException e) { }
        try { mSocket.close(); } catch (IOException e) { }

        Debug.getLogWriter(Debug.BRIEF).println(mThreadId + 
            "client done");
    }

    /**
     * Sets up the connection's character encoding, if the client 
     * requested a specific encoding. If this Connection was not
     * constructed with a pre-existing JDBC connection, 
     * establishes a JDBC connection.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if an error occurs establishing the
     *      JDBC connection
     */
     /* When we read the ConnectRequest, we don't yet know the client's 
      * character set. Therefore, the client is responsible for sending
      * the character set as an ASCII string. All other strings are
      * expected to be sent encoded in the client's character set.
      * We must explicitly pass the charset to the getXXX methods here,
      * since the charset wasn't set when the module decoded the
      * request.
      *
      * The rest of the code expects that the character encoding in the
      * BerModule is valid, based on its use here, and proceeds to
      * ignore UnsupportedEncodingExceptions.
      */
    BerObject handleRequest(ConnectRequest aRequest) 
        throws SQLException
    {
        Hashtable properties = aRequest.getProperties();
        // The property is always sent, but it might be "".
        String charset = properties.get("CharacterEncoding").toString();
        try
        {
            if (!charset.equals(""))
            {
                Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
                    "setting character encoding to " + charset);
                // Trigger UnsupportedEncodingException as soon as possible.
                com.vizdom.util.CharacterEncoder.toByteArray("test string", 
                    charset);
                mBerModule.setCharacterEncoding(charset);
            }

            // ??? Let user/pass be null.
            Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
                "connecting; url = " + aRequest.getURL(charset) + 
                "; user = " + aRequest.getUser(charset));
            if (!preExistingConnection)
            {
                mConn = DriverManager.getConnection(aRequest.getURL(charset), 
                    aRequest.getUser(charset), aRequest.getPassword(charset));
            }
        }
        catch (UnsupportedEncodingException unsupEnc)
        {
            throw new DbdException(DbdException.gUNSUPPORTED_ENCODING,
                new String[] { charset });
        }
        return new ConnectResponse();
    }
    

    /**
     * Closes the current set of Statements and the JDBC connection.
     * Does nothing if this Connection was constructed with a 
     * pre-existing JDBC connection.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if an error occurs while closing the
     *      Connection
     */
    /* StatementHolder.close doesn't throw exceptions, so we should always
     * get the chance to close the connection. 
     */
    BerObject handleRequest(DisconnectRequest aRequest)
        throws SQLException
    {
        if (!preExistingConnection)
        {
            java.util.Enumeration elements = mStatementTable.elements();
            while (elements.hasMoreElements())
            {
                StatementHolder holder = (StatementHolder) elements.nextElement();
                holder.close();
            }
            Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
                "closed " + mStatementTable.size() + " Statements on disconnect");
            
            mStatementTable = null;
            mConn.close();
        }
        return new DisconnectResponse();
    }
    
    /**
     * Checks to see whether the JDBC Connection is still open.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if the Connection is closed
     */
    BerObject handleRequest(PingRequest aRequest)
        throws SQLException
    {
        return new PingResponse(mConn.isClosed() ? 0 : 1);
    }

    /**
     * Calls <code>Connection.commit</code>.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if the <code>commit</code> fails
     */
    BerObject handleRequest(CommitRequest aRequest)
        throws SQLException
    {
        mConn.commit();
        return new CommitResponse();
    }
    
    /**
     * Calls <code>Connection.rollback</code>. 
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if the <code>rollback</code> fails
     */
    BerObject handleRequest(RollbackRequest aRequest)
        throws SQLException
    {
        mConn.rollback();
        return new RollbackResponse();
    }
    
    /**
     * Prepares a statement for later execution.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if the statement preparation fails
     */
    /* We use PreparedStatements here since we don't know if there are
     * any substitutable parameters. 
     */
    BerObject handleRequest(PrepareRequest aRequest)
        throws SQLException
    {
        PreparedStatement stmt = 
            mConn.prepareStatement(aRequest.getStatement());
        int stmtHandle = mNextHandle++;
        mStatementTable.put(new Integer(stmtHandle), 
            new StatementHolder(stmt));
        Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
            "assigned handle " + stmtHandle);
        return new PrepareResponse(stmtHandle);
    }
    
    /**
     * Sets statement parameters and executes a previously prepared 
     * statement. This method will try to convert the bytes sent to 
     * the type corresponding to the provided type hint and call the 
     * appropriate setXXX method. Type conversions are taken from
     * Table 21.2, p. 394, in JDBC Data Access with Java.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if an error occurs when setting a parameter
     *      with setXXX or executing the statement
     */
    /* The request will contain the statement handle for the statement
     * to be executed and a list of parameters and parameter type hints.
     * This method will throw an exception if one of the setXXX methods 
     * fails (or if a data conversion fails). We're playing some games
     * to make sure that the parameter number is included in the 
     * error message, since we're setting all the parameters at once
     * and the user might not know otherwise which one failed.
     */
    BerObject handleRequest(ExecuteRequest aRequest)
        throws SQLException
    {
        Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
            "executing statement handle " + aRequest.getHandle());
        StatementHolder holder = mGetStatementHolder(aRequest.getHandle());
        PreparedStatement stmt = holder.getStatement();
        Parameter[] params = aRequest.getParameters();
        Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
            "setting " + params.length + " parameters");
        for (int i = 0; i < params.length; i++)
        {
            try 
            {
                if (params[i].value == null)
                {
                    Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                        "setting parameter " + (i + 1) + "; value null; type " +
                        params[i].type);
                    stmt.setNull(i + 1, params[i].type);
                    continue;
                }
                Debug.getLogWriter(Debug.TEDIOUS).print(mThreadId + 
                    "setting parameter " + (i + 1) + "; value ");
                switch (params[i].type) 
                {
                case Types.BINARY:
                case Types.VARBINARY:
                case Types.LONGVARBINARY:
                    stmt.setBytes(i + 1, params[i].value.toByteArray());
                    break;
                case Types.TINYINT:
                case Types.SMALLINT:
                    stmt.setShort(i + 1, 
                        Short.parseShort(params[i].value.toString()));
                    break;
                case Types.INTEGER:
                    stmt.setInt(i + 1, 
                        Integer.parseInt(params[i].value.toString()));
                    break;
                case Types.BIGINT: 
                    stmt.setLong(i + 1, 
                        Long.parseLong(params[i].value.toString()));
                    break;
                case Types.REAL: 
                    stmt.setFloat(i + 1, 
                        new Float(params[i].value.toString()).floatValue());
                    break;
                case Types.FLOAT: 
                case Types.DOUBLE: 
                    stmt.setDouble(i + 1, 
                        new Double(params[i].value.toString()).doubleValue());
                    break;
                case Types.DECIMAL: 
                case Types.NUMERIC:
                    stmt.setBigDecimal(i + 1, 
                        new BigDecimal(params[i].value.toString()));
                    break;
                case Types.BIT:   // Clients must send "0" or "1"
                    stmt.setBoolean(i + 1, 
                        params[i].value.toString().equals("1"));
                    break;
                case Types.CHAR: 
                case Types.VARCHAR:
                case Types.LONGVARCHAR:  // Use a stream here?
                    stmt.setString(i + 1, params[i].value.toString());
                    break;
                case Types.DATE:      // ??? Perhaps we could accept
                case Types.TIME:      // ??? dates in the default JDBC
                case Types.TIMESTAMP: // ??? format.
                case Types.OTHER: 
                default: 
                    stmt.setString(i + 1, params[i].value.toString());
                    break;
                }
                if (params[i].type == Types.BINARY ||
                    params[i].type == Types.VARBINARY ||
                    params[i].type == Types.LONGVARBINARY)
                {
                    Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                        "(binary; length " + 
                        params[i].value.toByteArray().length + 
                        "); type " + params[i].type);
                }
                else
                {
                    Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                        params[i].value.toString() + "; type " + 
                        params[i].type);
                }
            }
            catch (NumberFormatException ne)
            {
                throw new DbdException(DbdException.gSET_PARAMETER,
                    new String[] { String.valueOf(i + 1), ne.toString() });
            }
            catch (SQLException se)
            {
                DbdException dbd = new DbdException(
                    DbdException.gSET_PARAMETER,
                    new String[] { String.valueOf(i + 1), se.toString() });
                dbd.setNextException(se);
                throw dbd;
            }            
        }
        
        ExecuteResponse resp;
        if (stmt.execute())
        {
            // execute returned a result set.
            Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                "getting result set");
            ResultSet rs = stmt.getResultSet();
            holder.setResultSet(rs);
            ResultSetMetaData rsmd = holder.getResultSetMetaData();
            int cols = rsmd.getColumnCount();
            resp = 
                new ExecuteResponse(
                new ExecuteResultSetResponse(cols));
        }
        else
        {
            // execute returned a row count.
            resp = new ExecuteResponse(new ExecuteRowsResponse(
                stmt.getUpdateCount()));
        }
        return resp;
    }

    /**
     * Fetches the next row of data from the ResultSet associated
     * with a given Statement. Implements the DBI specification 
     * with regard to LongReadLen, LongTruncOk, and ChopBlanks.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if <code>next</code> or <code>getXXX</code>
     *      fail, or if long data is truncated
     * @exception Exception if the statement has no result set, or the
     *      provided statement handle is invalid
     */
     /* Code here relies on prepare (or some other earlier method) to 
      * set the LongReadLen, LongTruncOk, and ChopBlanks properties.
      *
      * Some data types get special handling, but mostly we pass 
      * everything back as a string and let the client sort it out.
      */
    BerObject handleRequest(FetchRequest aRequest)
        throws SQLException
    {
        Debug.getLogWriter(Debug.VERBOSE).println(mThreadId + 
            "fetching row from statement handle " + aRequest.getHandle());
        StatementHolder holder = mGetStatementHolder(aRequest.getHandle());
        ResultSet rs = holder.getResultSet();
        if (rs == null)
            throw new DbdException(DbdException.gNO_DATA);
        
        Object[] row = null;
        boolean hasData;
        if (hasData = rs.next())
        {
            ResultSetMetaData rsmd = holder.getResultSetMetaData();
            int cols = rsmd.getColumnCount();
            row = new Object[cols];
            int longReadLen = ((Integer) 
                holder.getProperties().get("LongReadLen")).intValue();
            boolean longTruncOk = ((Boolean) 
                holder.getProperties().get("LongTruncOk")).booleanValue();
            boolean chopBlanks = ((Boolean) 
                holder.getProperties().get("ChopBlanks")).booleanValue();
            for (int i = 0; i < cols; i++)
            {
                try
                {
                    int type = rsmd.getColumnType(i + 1);
                    Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                        "getting column " + (i + 1) + "; type " + type);
                    switch (type)
                    {
                    case Types.BINARY: 
                    case Types.VARBINARY: 
                        row[i] = rs.getBytes(i + 1);
                        break;
                    case Types.LONGVARBINARY: 
                        if (longReadLen == 0)
                            row[i] = null;
                        else
                        {
                            row[i] = mReadLong(i + 1, rs.getBinaryStream(i + 1),
                                longReadLen, longTruncOk);
                        }
                        break;
                    case Types.LONGVARCHAR: 
                        if (longReadLen == 0)
                            row[i] = null;
                        else
                        {
                            byte[] bytes = mReadLong(i + 1, 
                                rs.getUnicodeStream(i + 1), longReadLen, 
                                longTruncOk);
                            row[i] = (bytes == null) ? 
                                null : new String(bytes, "UTF8");
                        }
                        break;
                    case Types.CHAR: 
                        if (chopBlanks)
                        {
                            row[i] = mChopBlanks(rs.getString(i + 1));
                            break;
                        }
                        // Fall through
                    default:  // This will include any Types.OTHER columns.
                        row[i] = rs.getString(i + 1);
                        break;
                    }
                }
                catch (IOException ioError)
                {
                    throw new DbdException(DbdException.gFETCH_EXCEPTION,
                        new String[] { String.valueOf(i + 1), 
                        ioError.toString() });
                }
            }
        }
        try
        {
            return new FetchResponse(hasData, row, 
                mBerModule.getCharacterEncoding());
        }
        catch (UnsupportedEncodingException unsupEnc)
        {
            throw new UnreachableCodeException();
        }
    }
    


    /**
     * Returns the value of a connection property. 
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception Exception if the requested connection property name
     *      is not recognized
     */
    BerObject handleRequest(GetConnectionPropertyRequest aRequest)
        throws SQLException
    {
        String property = aRequest.getPropertyName();
        if (property.equals("AutoCommit"))
        {
            Integer[] response = new Integer[1];
            response[0] = new Integer(mConn.getAutoCommit() ? 1 : 0);
            try
            {
                return new GetConnectionPropertyResponse(response, 
                    mBerModule.getCharacterEncoding());
            }
            catch (UnsupportedEncodingException unsupEnc)
            {
                throw new UnreachableCodeException();
            }
            
        }
        throw new DbdException(DbdException.gUNKNOWN_PROPERTY, 
            new String[] { property });
    }

    /**
     * Sets a connection property value.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception Exception if the connection property name
     *      is not recognized
     */
    BerObject handleRequest(SetConnectionPropertyRequest aRequest)
        throws SQLException
    {
        String property = aRequest.getPropertyName();
        if (property.equals("AutoCommit"))
        {
            mConn.setAutoCommit(aRequest.getPropertyValue().equals("1"));
            return new SetConnectionPropertyResponse();
        }
        throw new DbdException(DbdException.gUNKNOWN_PROPERTY, 
            new String[] { property });
    }

    /**
     * Gets a statement property value.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception SQLException if a database access error occurs
     * @exception Exception if the property name is unknown, or
     *      if the statement handle is invalid or has no ResultSet
     *      associated with it
     */
    BerObject handleRequest(GetStatementPropertyRequest aRequest)
        throws SQLException
    {
        try
        {
            StatementHolder holder = mGetStatementHolder(aRequest.getHandle());
            String property = aRequest.getPropertyName();
            if (property.equals("CursorName")) 
            {
                ResultSet rs = holder.getResultSet();
                if (rs == null)
                    throw new DbdException(DbdException.gNO_CURSOR);
                String[] cursorname = new String[1];
                // DBI specifies that the cursor name should be returned as undef
                // if cursor names are not supported. JDBC specifies that
                // getCursorName should throw an exception if named cursors
                // are not supported.
                try
                {
                    cursorname[0] = rs.getCursorName();
                }
                catch (SQLException e)
                {
                    cursorname[0] = null;
                    Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId + 
                        "getCursorName threw an exception");
                }
                return new GetStatementPropertyResponse(cursorname,
                    mBerModule.getCharacterEncoding());
            }
            
            ResultSetMetaData rsmd = holder.getResultSetMetaData();
            if (rsmd == null)
                throw new DbdException(DbdException.gNO_METADATA);
            int colcount = rsmd.getColumnCount();
            
            if (property.equals("NAME"))
            {
                String[] data = new String[colcount];
                for (int i = 1; i <= colcount; i++)
                    data[i - 1] = rsmd.getColumnName(i);
                return new GetStatementPropertyResponse(data, 
                    mBerModule.getCharacterEncoding());
            }
            else if (property.equals("TYPE"))
            {
                Integer[] data = new Integer[colcount];
                for (int i = 1; i <= colcount; i++)
                    data[i - 1] = new Integer(rsmd.getColumnType(i));
                return new GetStatementPropertyResponse(data,
                    mBerModule.getCharacterEncoding());
            }
            else if (property.equals("PRECISION"))
            {
                Integer[] data = new Integer[colcount];
                for (int i = 1; i <= colcount; i++)
                    data[i - 1] = new Integer(rsmd.getPrecision(i));
                return new GetStatementPropertyResponse(data,
                    mBerModule.getCharacterEncoding());
            }
            else if (property.equals("SCALE"))
            {
                Integer[] data = new Integer[colcount];
                for (int i = 1; i <= colcount; i++)
                {
                    // Scale might reasonably be unsupported on some columns.
                    try
                    {
                        data[i - 1] = new Integer(rsmd.getScale(i));
                    }
                    catch (SQLException e)
                    {
                        data[i - 1] = null;
                    }
                }
                return new GetStatementPropertyResponse(data,
                    mBerModule.getCharacterEncoding());
            }
            else if (property.equals("NULLABLE"))
            {
                Integer[] data = new Integer[colcount];
                for (int i = 1; i <= colcount; i++)
                {
                    int nullable = rsmd.isNullable(i);
                    switch (nullable)
                    {
                    case ResultSetMetaData.columnNoNulls:
                        data[i - 1] = sDbiNoNulls;
                        break;
                    case ResultSetMetaData.columnNullable:
                        data[i - 1] = sDbiNullable;
                        break;
                    case ResultSetMetaData.columnNullableUnknown:
                        data[i - 1] = sDbiNullableUnknown;
                        break;
                    default:  // UnreachableCodeException???
                        data[i - 1] = null;
                        Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId +
                            "isNullable returned an unknown value " + 
                            nullable);
                        break;
                    }
                }
                return new GetStatementPropertyResponse(data,
                    mBerModule.getCharacterEncoding());
            }
            throw new DbdException(DbdException.gUNKNOWN_PROPERTY, 
                new String[] { property });
            
        }
        catch (UnsupportedEncodingException unsupEnc)
        {       
            throw new UnreachableCodeException();
        }
    }


    /**
     * Sets a statement property value.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception Exception if the statement handle is invalid or
     *      the property name is not recognized
     */
    BerObject handleRequest(SetStatementPropertyRequest aRequest)
        throws SQLException
    {
        StatementHolder holder = mGetStatementHolder(aRequest.getHandle());
        String property = aRequest.getPropertyName();

        if (property.equals("LongReadLen"))
        {
            holder.getProperties().put(property, 
                new Integer(aRequest.getPropertyValue()));
            return new SetStatementPropertyResponse();
        }
        if (property.equals("LongTruncOk"))
        {
            holder.getProperties().put(property, 
                new Boolean(aRequest.getPropertyValue().equals("1")));
            return new SetStatementPropertyResponse();
        }
        if (property.equals("ChopBlanks"))
        {
            holder.getProperties().put(property, 
                new Boolean(aRequest.getPropertyValue().equals("1")));
            return new SetStatementPropertyResponse();
        }

        throw new DbdException(DbdException.gUNKNOWN_PROPERTY, 
            new String[] { property });
    }    
    

    /**
     * Finishes a statement. A statement is finished when all its data
     * has been read, so this method closes the result set. 
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception Exception if the statement handle is invalid
     */
    /* Accept this message, but don't do anything. DBI forbids finish
     * to have any effect on the session's transaction state, and we
     * can't guarantee that if we close the ResultSet, which is the 
     * obvious implementation here.
     *
     * JDBC 2.0 also has implications here, since reading all a 
     * result set's data doesn't necessarily mean you're done with it.
     */   
    BerObject handleRequest(StatementFinishRequest aRequest)
        throws SQLException
    {
        StatementHolder holder = mGetStatementHolder(aRequest.getHandle());
        //holder.finish();
        return new StatementFinishResponse();
    }    
    
    /**
     * Frees a statement for garbage collection.
     *
     * @param aRequest the request received from the client
     * @return a BER response object
     * @exception Exception if the statement handle is invalid
     */
    /* This should be called by $sth->DESTROY: the statement has gone
     * out of scope and is no longer usable, so remove it from the
     * cache.
     */
    BerObject handleRequest(StatementDestroyRequest aRequest)
        throws DbdException
    {
        // Note: we want to call mStatementTable.remove, so don't use
        // mGetStatementHolder.
        int handle = aRequest.getHandle();
        StatementHolder holder = (StatementHolder) mStatementTable.remove(
            new Integer(handle));
        // ??? Should this be a runtime exception? Can the user cause this?
        if (holder == null)
            throw new DbdException(DbdException.gINVALID_STATEMENT_HANDLE);
        return new StatementDestroyResponse();
    }    


    /**
     * Retrieves a StatementHolder from the statement table.
     *
     * @param aStatementHandle the statement handle to retrieve
     * @return the corresponding StatementHolder from the cache
     * @exception Exception if the statement handle does not correspond
     *      to an entry in the statement table
     */
    private StatementHolder mGetStatementHolder(int aStatementHandle)
        throws DbdException
    {
        StatementHolder holder = (StatementHolder) mStatementTable.get(
            new Integer(aStatementHandle));
        // ??? Should this be a runtime exception? Can the user invoke this?
        if (holder == null)
            throw new DbdException(DbdException.gINVALID_STATEMENT_HANDLE);
        return holder;
    }

    /**
     * Removes trailing blanks from the given string.
     *
     * @param aString a string, potentially with trailing blanks
     * @return the same string with any trailing blanks removed
     */
    private String mChopBlanks(String aString)
    {
        if (aString != null)
        {
            int last = aString.length() - 1;
            while (last >= 0 && aString.charAt(last) == ' ')
                last--;
            return aString.substring(0, last + 1);
        }
        else
            return null;
    }

    /**
     * Reads a LONG field. Implements the DBI semantics associated with
     * the LongTruncOk and LongReadLen properties.
     *
     * @param aColumnIndex the column index being read, for error reporting
     * @param anInputStream an input stream returned by a getXXXStream method
     * @param aLongReadLen the LongReadLen property for this statement
     * @param aLongTruncOk the LongTruncOk property for this statement
     * @return a byte array read from the stream, or null if the input
     *      stream is null
     * @exception DataTruncation if the data is truncated
     * @exception IOException if an error occurs reading from the stream
     */
    private byte[] mReadLong(int aColumnIndex, InputStream anInputStream, 
        int aLongReadLen, boolean aLongTruncOk) 
        throws DbdException, IOException
    {
        if (anInputStream == null)
            return null;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte[] buffer = new byte[sLONG_READ_BUFFER_SIZE];
        int read;
        int totalread = 0;
        while (totalread < aLongReadLen &&
            (read = anInputStream.read(buffer, 0, 
            Math.min(buffer.length, (aLongReadLen - totalread)))) != -1)
        {
            totalread += read;
            baos.write(buffer, 0, read);
        }
        baos.close();
        Debug.getLogWriter(Debug.VERBOSE).println(mThreadId +
            "read " + totalread + " bytes from LONG column " + aColumnIndex);
        if (!aLongTruncOk && anInputStream.read() != -1)
        {
            anInputStream.close();
            throw new DbdException(DbdException.gDATA_TRUNCATION);
        }
        anInputStream.close();
        return baos.toByteArray();
    }

    /** 
     * Sends the given error information to the client.
     *
     * @param aMessage the error text
     * @param aCode the error code
     * @param aSQLState the SQL state associated with this error
     * @exception FatalException if the error message can't be sent
     */
    private void mSendError(SQLException aSQLException)
    {
        try 
        {
            ErrorResponse error = new ErrorResponse(aSQLException,
                mBerModule.getCharacterEncoding());
            Debug.getLogWriter(Debug.TEDIOUS).println(mThreadId +
                "sending error: " + aSQLException.getMessage());
            error.writeTo(mOut);
            mOut.flush();
        }
        catch (UnsupportedEncodingException unsupEnc)
        {
            throw new UnreachableCodeException();
        }
        catch (Exception e )
        {
            throw new FatalException("Failed to send error message " +
                "to client: " + e.toString());
        }
    }
}


/**
 * This class encapsulates a PreparedStatement and its associated 
 * ResultSet, ResultSetMetaData, and properties.
 */
class StatementHolder
{
    /** The PreparedStatement. */
    PreparedStatement mStatement;
    /** The statement's result set, if relevant. */
    ResultSet mResultSet;
    /** The result set meta data. */
    ResultSetMetaData mResultSetMetaData;
    /** The statement properties (LongReadLen, etc.). */
    Hashtable mStatementProperties;

    /**
     * Constructor - initializes fields. 
     *
     * @param aStatement a PreparedStatement
     */
    StatementHolder(PreparedStatement aStatement)
    {
        mStatement = aStatement;
        mStatementProperties = new Hashtable();
        
        mResultSet = null;
        mResultSetMetaData = null;
    }

    /**
     * Closes this statement. Ignores any exceptions thrown by 
     * PreparedStatement.close.
     */
    void close()
    {
        if (mStatement != null)
        {
            try { mStatement.close(); } catch (Exception e) { }
        }
        mStatementProperties = null;
        mResultSetMetaData = null;
        mResultSet = null;
        mStatement = null;
    }

    /**
     * Finishes this statement. Ignores any exceptions thrown by 
     * ResultSet.close. 
     */
    /* This is like DBI's finish - no more data to be read from the 
     * statement, but execute may be called again. However, in JDBC, 
     * closing the result set may commit a transaction in AutoCommit 
     * mode, which is forbidden by the DBI spec. I'm leaving this 
     * here, but it shouldn't be called unless we decide to support
     * this behavior.
     */
    void finish()
    {
        if (mResultSet != null)
        {
            try { mResultSet.close(); } catch (Exception e) { }
        }
        mResultSet = null;
        mResultSetMetaData = null;
    }

    /**
     * Returns this holder's PreparedStatement.
     *
     * @return this holder's PreparedStatement
     */
    PreparedStatement getStatement()
    {
        return mStatement;
    }

    /**
     * Returns this holder's ResultSet.
     *
     * @return this holder's ResultSet
     */
    ResultSet getResultSet()
    {
        return mResultSet;
    }
    /**
     * Updates this holder's ResultSet and ResultSetMetaData
     * when the statement has been executed.
     *
     * @param aResultSet a new ResultSet for this statement
     * @exception SQLException if ResultSet.getMetaData fails
     */
    void setResultSet(ResultSet aResultSet) throws SQLException
    {
        mResultSet = aResultSet;
        mResultSetMetaData = 
            (aResultSet == null) ? null : aResultSet.getMetaData();
    }


    /**
     * Returns this holder's ResultSetMetaData.
     *
     * @return this holder's ResultSetMetaData
     */
    ResultSetMetaData getResultSetMetaData()
    {
        return mResultSetMetaData;
    }

    /**
     * Returns this holder's property list.
     *
     * @return this holder's property list
     */
    Hashtable getProperties()
    {
        return mStatementProperties;
    }
}
