# lib/Software/Licemse/OrLaterPack.pm #

#   This file is part of Software-License-OrLaterPack.
#
#   Software-License-OrLaterPack is free software: you can redistribute it
#   and/or modify it under the terms of the GNU General Public License as
#   published by the Free Software Foundation, either version 3 of the
#   License, or (at your option) any later version.
#
#   Software-License-OrLaterPack is distributed in the hope that it will be
#   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#   Public License for more details.
#
#   You should have received a copy of the GNU General Public License along
#   with Software-License-OrLaterPack.  If not, see
#   <http://www.gnu.org/licenses/>.

package Software::License::OrLaterPack;

# ABSTRACT: Add-ons for Software::License module providing "or any later" clause to some GNU licenses

use 5.008;
use strict;
use warnings;

our $VERSION = '0.005'; # VERSION

use Software::License::AGPL_3::or_later;
use Software::License::GPL_3::or_later;

1;

#pod =pod
#pod
#pod =encoding UTF-8
#pod
#pod =head1 SYNOPSIS
#pod
#pod In C<dist.ini>:
#pod
#pod     name             = Foo-Bar
#pod     version          = 0.001
#pod     author           = John Doe <john.doe@example.com>
#pod     license          = GPL_3::or_later ; <<== Note 'or_later' <<==
#pod         ; or another license, see the list of provided licenses below
#pod     copyright_holder = John Doe
#pod     copyright_year   = 2015
#pod     …
#pod
#pod Direct usage:
#pod
#pod     use Software::License::GPL_3::or_later; # <<== Note 'or_later' <<==
#pod         # or another license, see the list provided licenses below
#pod     my $license =
#pod         Software::License::GPL_3::or_later->new( {
#pod             holder => 'John Doe', …
#pod         } );
#pod     print( $license->notice );
#pod     …
#pod
#pod =head1 PURPOSE
#pod
#pod L<Dist::Zilla>/C<dzil>, a popular framework/tool for building packages for CPAN, allows you to
#pod specify software license in C<dist.ini> file, e. g.:
#pod
#pod     license = NAME
#pod
#pod where I<NAME> is a short license name, like C<Perl_5> or C<BSD>. When building your package,
#pod C<dzil> will add "COPYRIGHT AND LICENSE" section to the documentation, add appropriate C<LICENSE>
#pod file to your package, etc. C<Dist::Zilla> uses licenses provided by C<Software::License> package.
#pod The latter is bundled with a set of popular software licenses, including GNU General Public License
#pod (GNU GPL) versions 1, 2, and 3, denoted by names C<GPL_1>, C<GPL_2>, and C<GPL_3> respectively.
#pod
#pod L<Free Software Foundation|https://fsf.org/>, a developer of GNU GPL, recommends do not stick to a
#pod specific version of the GNU license, but allow "upgrading" it by using "license version I<N> or (at
#pod your option) any later version". See
#pod L<rationale|https://www.gnu.org/licenses/gpl-faq.html#VersionThreeOrLater>.
#pod
#pod However, L<Software::License> package does not provide (ready to use out of the box) a way to
#pod express such conditions. This module partially fulfills the lack.
#pod
#pod =head1 LIST OF PROVIDED LICENSES
#pod
#pod =for :list
#pod = L<Software::License::AGPL_3::or_later>
#pod = L<Software::License::GPL_3::or_later>
#pod
#pod Let me know if you need other licenses, e. g. GPLv2+.
#pod
#pod =head1 CAVEATS
#pod
#pod L<CPAN::Meta::Spec> hardcodes the list of "valid" licenses. In version 2.150001 of the module there
#pod are no "upgradable" GNU licenses, so in CPAN the GPLv3+ will look as ordinal GPLv3 (C<gpl_3>), and
#pod so on.
#pod
#pod =head1 SEE ALSO
#pod
#pod =for :list
#pod = L<Dist::Zilla>
#pod = L<Software::License>
#pod = L<Software::License::AGPL_3>
#pod = L<Software::License::AGPL_3::or_later>
#pod = L<Software::License::GPL_3>
#pod = L<Software::License::GPL_3::or_later>
#pod = L<Why should programs say “Version 3 of the GPL or any later version”?|https://www.gnu.org/licenses/gpl-faq.html#VersionThreeOrLater>
#pod
#pod =cut

# end of file #

__END__

=pod

=encoding UTF-8

=head1 NAME

Software::License::OrLaterPack - Add-ons for Software::License module providing "or any later" clause to some GNU licenses

=head1 VERSION

version 0.005

=head1 SYNOPSIS

In C<dist.ini>:

    name             = Foo-Bar
    version          = 0.001
    author           = John Doe <john.doe@example.com>
    license          = GPL_3::or_later ; <<== Note 'or_later' <<==
        ; or another license, see the list of provided licenses below
    copyright_holder = John Doe
    copyright_year   = 2015
    …

Direct usage:

    use Software::License::GPL_3::or_later; # <<== Note 'or_later' <<==
        # or another license, see the list provided licenses below
    my $license =
        Software::License::GPL_3::or_later->new( {
            holder => 'John Doe', …
        } );
    print( $license->notice );
    …

=head1 PURPOSE

L<Dist::Zilla>/C<dzil>, a popular framework/tool for building packages for CPAN, allows you to
specify software license in C<dist.ini> file, e. g.:

    license = NAME

where I<NAME> is a short license name, like C<Perl_5> or C<BSD>. When building your package,
C<dzil> will add "COPYRIGHT AND LICENSE" section to the documentation, add appropriate C<LICENSE>
file to your package, etc. C<Dist::Zilla> uses licenses provided by C<Software::License> package.
The latter is bundled with a set of popular software licenses, including GNU General Public License
(GNU GPL) versions 1, 2, and 3, denoted by names C<GPL_1>, C<GPL_2>, and C<GPL_3> respectively.

L<Free Software Foundation|https://fsf.org/>, a developer of GNU GPL, recommends do not stick to a
specific version of the GNU license, but allow "upgrading" it by using "license version I<N> or (at
your option) any later version". See
L<rationale|https://www.gnu.org/licenses/gpl-faq.html#VersionThreeOrLater>.

However, L<Software::License> package does not provide (ready to use out of the box) a way to
express such conditions. This module partially fulfills the lack.

=head1 LIST OF PROVIDED LICENSES

=over 4

=item L<Software::License::AGPL_3::or_later>

=item L<Software::License::GPL_3::or_later>

=back

Let me know if you need other licenses, e. g. GPLv2+.

=head1 CAVEATS

L<CPAN::Meta::Spec> hardcodes the list of "valid" licenses. In version 2.150001 of the module there
are no "upgradable" GNU licenses, so in CPAN the GPLv3+ will look as ordinal GPLv3 (C<gpl_3>), and
so on.

=head1 SEE ALSO

=over 4

=item L<Dist::Zilla>

=item L<Software::License>

=item L<Software::License::AGPL_3>

=item L<Software::License::AGPL_3::or_later>

=item L<Software::License::GPL_3>

=item L<Software::License::GPL_3::or_later>

=item L<Why should programs say “Version 3 of the GPL or any later version”?|https://www.gnu.org/licenses/gpl-faq.html#VersionThreeOrLater>

=back

=head1 AUTHOR

Van de Bugger <van.de.bugger@liamg.moc>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 Van de Bugger

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.

=cut
