#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: lib/Dist/Zilla/Plugin/Templates.pm

#pod =encoding UTF-8
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright © 2015 Van de Bugger
#pod
#pod This file is part of perl-Dist-Zilla-Plugin-Templates.
#pod
#pod perl-Dist-Zilla-Plugin-Templates is free software: you can redistribute it and/or modify it
#pod under the terms of the GNU General Public License as published by the Free Software Foundation,
#pod either version 3 of the License, or (at your option) any later version.
#pod
#pod perl-Dist-Zilla-Plugin-Templates is distributed in the hope that it will be useful, but WITHOUT
#pod ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#pod PURPOSE. See the GNU General Public License for more details.
#pod
#pod You should have received a copy of the GNU General Public License along with
#pod perl-Dist-Zilla-Plugin-Templates. If not, see <http://www.gnu.org/licenses/>.
#pod
#pod =cut

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#pod =for :this This is C<Dist::Zilla::Plugin::Templates> module documentation. Read this if you are going to hack or
#pod extend C<Dist-Zilla-Plugin-Templates>.
#pod
#pod =for :those If you want to convert any text file into template, read the L<manual|Dist::Zilla::Plugin::Templates::Manual>. General
#pod topics like getting source, building, installing, bug reporting and some others are covered in the
#pod F<README>.
#pod
#pod =head1 DESCRIPTION
#pod
#pod Implementation of the plugin is trivial. It just consumes few roles which do all the work:
#pod C<FileFinderUser> provides a list of templates, C<TextTemplater> process them.
#pod
#pod =head1 SEE ALSO
#pod
#pod =for :list
#pod = L<Dist::Zilla>
#pod = L<Dist::Zilla::Role::FileFinderUser>
#pod = L<Dist::Zilla::Role::TextTemplater>
#pod = L<Text::Template>
#pod = L<Dist::Zilla::Plugin::Templates::Manual>
#pod
#pod =cut

# --------------------------------------------------------------------------------------------------

package Dist::Zilla::Plugin::Templates;

use Moose;
use namespace::autoclean;

# ABSTRACT: Convert any text file into template
our $VERSION = 'v0.4.1'; # VERSION

with 'Dist::Zilla::Role::FileMunger';
with 'Dist::Zilla::Role::FileFinderUser' => {
    finder_arg_names => [ qw{ template templates } ],
    default_finders  => [ ':NoFiles' ],
};
with 'Dist::Zilla::Role::TextTemplater' => { -version => 0.007 };   # need nested packages support.
with 'Dist::Zilla::Role::ErrorLogger' => { -version => 0.005 };

use Dist::Zilla::Plugin::Templates::File;

# --------------------------------------------------------------------------------------------------

#pod =method munge_files
#pod
#pod This is the primary method of the plugin. It is called by C<Dist::Zilla> during build. The method
#pod iterates through the files provided by C<< $self->found_files >> (a method taken from
#pod C<FileFinderUser> role) and process each file with C<< $self->fill_in_file >> (a method taken from
#pod C<TextTemplater> role). That's all, folks.
#pod
#pod =cut

sub munge_files {
    my ( $self ) = @_;
    for my $file ( @{ $self->found_files } ) {
        $self->fill_in_file(
            $file,
            {
                include => \ &{ sub {       # `Text::Template` wants double reference.
                    return $self->include( @_ );
                } },
            },
        );
    };
    return;
};

# --------------------------------------------------------------------------------------------------

#pod =method include
#pod
#pod This method implements same-name template function. Normally, templates should call the function,
#pod not method:
#pod
#pod     {{ include( 'filename' ); }}
#pod
#pod However, if something wrong with the function, file inclusion can be made through the method call:
#pod
#pod     {{ $plugin->include( 'filename' ); }}
#pod
#pod =cut

sub include {
    my ( $self, $name ) = @_;
    return Dist::Zilla::Plugin::Templates::File->new( {
        name    => $name,
        _plugin => $self,
    } );
};

# --------------------------------------------------------------------------------------------------

__PACKAGE__->meta->make_immutable();

1;

# --------------------------------------------------------------------------------------------------

# doc/what.pod #

#pod =encoding UTF-8
#pod
#pod =head1 WHAT?
#pod
#pod C<Dist-Zilla-Plugin-Templates> (or just C<Templates>) is a C<Dist-Zilla> plugin allowing developers to insert
#pod fragments of Perl code into arbitrary text files, which become I<templates>. When building a
#pod distribution with C<Dist::Zilla>, C<Templates> plugin evaluates these fragments, and replaces each
#pod fragment with result of its evaluation.
#pod
#pod =cut

# end of file #


# end of file #

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Templates - Convert any text file into template

=head1 VERSION

Version v0.4.1, released on 2015-09-19 18:06 UTC.

=head1 WHAT?

C<Dist-Zilla-Plugin-Templates> (or just C<Templates>) is a C<Dist-Zilla> plugin allowing developers to insert
fragments of Perl code into arbitrary text files, which become I<templates>. When building a
distribution with C<Dist::Zilla>, C<Templates> plugin evaluates these fragments, and replaces each
fragment with result of its evaluation.

This is C<Dist::Zilla::Plugin::Templates> module documentation. Read this if you are going to hack or
extend C<Dist-Zilla-Plugin-Templates>.

If you want to convert any text file into template, read the L<manual|Dist::Zilla::Plugin::Templates::Manual>. General
topics like getting source, building, installing, bug reporting and some others are covered in the
F<README>.

=head1 DESCRIPTION

Implementation of the plugin is trivial. It just consumes few roles which do all the work:
C<FileFinderUser> provides a list of templates, C<TextTemplater> process them.

=head1 OBJECT METHODS

=head2 munge_files

This is the primary method of the plugin. It is called by C<Dist::Zilla> during build. The method
iterates through the files provided by C<< $self->found_files >> (a method taken from
C<FileFinderUser> role) and process each file with C<< $self->fill_in_file >> (a method taken from
C<TextTemplater> role). That's all, folks.

=head2 include

This method implements same-name template function. Normally, templates should call the function,
not method:

    {{ include( 'filename' ); }}

However, if something wrong with the function, file inclusion can be made through the method call:

    {{ $plugin->include( 'filename' ); }}

=head1 SEE ALSO

=over 4

=item L<Dist::Zilla>

=item L<Dist::Zilla::Role::FileFinderUser>

=item L<Dist::Zilla::Role::TextTemplater>

=item L<Text::Template>

=item L<Dist::Zilla::Plugin::Templates::Manual>

=back

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright © 2015 Van de Bugger

This file is part of perl-Dist-Zilla-Plugin-Templates.

perl-Dist-Zilla-Plugin-Templates is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

perl-Dist-Zilla-Plugin-Templates is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
perl-Dist-Zilla-Plugin-Templates. If not, see <http://www.gnu.org/licenses/>.

=cut
