package RDF::RDFa::Generator::HTML::Pretty;

use 5.008;
use base qw'RDF::RDFa::Generator::HTML::Hidden';
use common::sense;
use constant XHTML_NS => 'http://www.w3.org/1999/xhtml';
use XML::LibXML qw':all';

sub create_document
{
	my ($proto, $model) = @_;
	my $self = (ref $proto) ? $proto : $proto->new;
	
	my $html = sprintf(<<HTML, ($self->{'version'}||'1.0'), ($self->{'title'} || 'RDFa Document'), ref $self);
<html xmlns="http://www.w3.org/1999/xhtml" version="XHTML+RDFa %1\$s">
<head profile="http://www.w3.org/1999/xhtml/vocab">
<title>%2\$s</title>
<meta nane="generator" value="%3\$s" />
</head>
<body>
<h1>%2\$s</h1>
<p><small>Generated by %3\$s.</small></p>
</body>
</html>
HTML

	return $proto->inject_document($html, $model);
}

sub nodes
{
	my ($proto, $model) = @_;
	my $self = (ref $proto) ? $proto : $proto->new;
	
	my $stream = $self->_get_stream($model);
	my @nodes;
	
	my $root_node = XML::LibXML::Element->new('div');
	$root_node->setNamespace(XHTML_NS, undef, 1);
	
	my $prefixes = {};
	my $subjects = {};
	while (my $st = $stream->next)
	{
		my $s = $st->subject->is_resource ?
			$st->subject->uri :
			('_:'.$st->subject->blank_identifier);
		push @{ $subjects->{$s} }, $st;
	}
	
	foreach my $s (keys %$subjects)
	{
		my $subject_node = $root_node->addNewChild(XHTML_NS, 'div');
		
		$self->_process_subject($subjects->{$s}->[0], $subject_node, $prefixes);
		$self->_resource_heading($subjects->{$s}->[0]->subject, $subject_node, $subjects->{$s}, $prefixes);
		$self->_resource_classes($subjects->{$s}->[0]->subject, $subject_node, $subjects->{$s}, $prefixes);
		$self->_resource_statements($subjects->{$s}->[0]->subject, $subject_node, $subjects->{$s}, $prefixes);
		## TODO Query $model for statements that act as special notes for the subject (in a separate graph)
		#$self->_resource_notes($subjects->{$s}->[0]->subject, $subject_node, $model);
	}
	
	if ($self->{'version'} == 1.1
	and $self->{'prefix_attr'})
	{
		my $prefix_string = '';
		while (my ($u,$p) = each(%$prefixes))
		{
			$prefix_string .= sprintf("%s: %s ", $p, $u);
		}
		if (length $prefix_string)
		{
			$root_node->setAttribute('prefix', $prefix_string);
		}
	}
	else
	{
		while (my ($u,$p) = each(%$prefixes))
		{
			$root_node->setNamespace($u, $p, 0);
		}
	}
	
	push @nodes, $root_node;
	return @nodes if wantarray;
	my $nodelist = XML::LibXML::NodeList->new;
	$nodelist->push(@nodes);
	return $nodelist;
}

sub _resource_heading
{
	my ($self, $subject, $node, $statements, $prefixes) = @_;
	
	my $heading = $node->addNewChild(XHTML_NS, 'h3');
	$heading->appendTextNode( $subject->is_resource ? $subject->uri : ('_:'.$subject->blank_identifier) );
	$heading->setAttribute('class', $subject->is_resource ? 'resource' : 'blank' );
	
	return $self;
}

## TODO
## <span rel="rdf:type"><img about="[foaf:Person]" src="fsfwfwfr.png"
##                           title="http://xmlns.com/foaf/0.1/Person" /></span>

sub _resource_classes
{
	my ($self, $subject, $node, $statements, $prefixes) = @_;
	
	my @statements = sort {
		$a->predicate->uri cmp $b->predicate->uri
		or $a->object->uri cmp $b->object->uri
		}
		grep {
			$_->predicate->uri eq 'http://www.w3.org/1999/02/22-rdf-syntax-ns#type'
			and $_->object->is_resource
		}
		@$statements;

	return unless @statements;

	my $SPAN = $node->addNewChild(XHTML_NS, 'span');
	$SPAN->setAttribute('class', 'rdf-type');
	$SPAN->setAttribute('rel', $self->_make_curie('http://www.w3.org/1999/02/22-rdf-syntax-ns#type', $prefixes));

	foreach my $st (@statements)
	{
		my $IMG = $SPAN->addNewChild(XHTML_NS, 'img');
		$IMG->setAttribute('about', $st->object->uri);
		$IMG->setAttribute('alt',   $st->object->uri);
		$IMG->setAttribute('src',   $self->_img($st->object->uri));
		$IMG->setAttribute('title', $st->object->uri);
	}

	return $self;
}


sub _resource_statements
{
	my ($self, $subject, $node, $statements, $prefixes) = @_;
	
	my @statements = sort {
		$a->predicate->uri cmp $b->predicate->uri
		or $a->object->uri cmp $b->object->uri
		}
		grep {
			$_->predicate->uri ne 'http://www.w3.org/1999/02/22-rdf-syntax-ns#type'
			or !$_->object->is_resource
		}
		@$statements;

	return unless @statements;
	
	my $DL = $node->addNewChild(XHTML_NS, 'dl');
	
	my $current_property = undef;
	foreach my $st (@statements)
	{
		unless ($st->predicate->uri eq $current_property)
		{
			my $DT = $DL->addNewChild(XHTML_NS, 'dt');
			$DT->setAttribute('title', $st->predicate->uri);
			$DT->appendTextNode($self->_make_curie($st->predicate->uri, $prefixes));
		}
		
		my $DD = $DL->addNewChild(XHTML_NS, 'dd');
		
		if ($st->object->is_resource)
		{
			$DD->setAttribute('rel',  $self->_make_curie($st->predicate->uri, $prefixes));
			$DD->setAttribute('class', 'resource');
			
			my $A = $DD->addNewChild(XHTML_NS, 'a');
			$A->setAttribute('href', $st->object->uri);
			$A->appendTextNode($st->object->uri);
		}
		elsif ($st->object->is_blank)
		{
			$DD->setAttribute('rel',  $self->_make_curie($st->predicate->uri, $prefixes));
			$DD->setAttribute('class', 'blank');
			
			my $A = $DD->addNewChild(XHTML_NS, 'span');
			$A->setAttribute('about', '[_:'.$st->object->blank_identifier.']');
			$A->appendTextNode('_:'.$st->object->blank_identifier);
		}
		elsif ($st->object->is_literal
		&& !$st->object->has_datatype)
		{
			$DD->setAttribute('property',  $self->_make_curie($st->predicate->uri, $prefixes));
			$DD->setAttribute('class', 'plain-literal');
			$DD->setAttribute('xml:lang',  $st->object->literal_value_language);
			$DD->appendTextNode($st->object->literal_value);
		}
		elsif ($st->object->is_literal
		&& $st->object->has_datatype
		&& $st->object->literal_datatype eq 'http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral')
		{
			$DD->setAttribute('property',  $self->_make_curie($st->predicate->uri, $prefixes));
			$DD->setAttribute('class', 'typed-literal datatype-xmlliteral');
			$DD->setAttribute('datatype',  $self->_make_curie($st->object->literal_datatype, $prefixes));
			$DD->appendWellBalancedChunk($st->object->literal_value);
		}
		elsif ($st->object->is_literal
		&& $st->object->has_datatype)
		{
			$DD->setAttribute('property',  $self->_make_curie($st->predicate->uri, $prefixes));
			$DD->setAttribute('class', 'typed-literal');
			$DD->setAttribute('datatype',  $self->_make_curie($st->object->literal_datatype, $prefixes));
			$DD->appendTextNode($st->object->literal_value);
		}
	}
	
	return $self;
}

sub _img
{
	my ($self, $type) = @_;
	
	my $icons = {
		'http://xmlns.com/foaf/0.1/Document'                   => 'data:image/png;charset=binary;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABHNCSVQICAgIfAhkiAAAAz1JREFUWIXll0FPG0cUx38zuzgmllM7TQhRFFVVghGRkIBYqJz6HfoFyIEb36HXHnPiGIlP0I/QWy6oQkKqBAdXTghxD03BwnaK1zs7OeBZz65n2TVKT32StTPjnfd+83/PM2P4v5tID+zt7f3k+/4+UPU8DyklURShlIo/YRjGT7t90xjQC8Pw1f7+/q92PH+KSIg3Ozs71XGbMAzxfR+tdeI9u1+wXd3d3X0D3AyglKoBtNtthBAopahUKgRBYAARYiKc3Xb1DcjCwgJBENTS300BaK3jIMZZuu16ZgXPsymAKIoSQe1ArVYr/r5IUM/zaDQaicUVAjAqpEGWlpb+ewWUUgmHtgp5Cniex/Lycmawwgq4ggshaDQat1LAjBcCMAq4ivDo6KhQINs2Njbi9kwAAIeHh6yvr0/UkB4d+YQfGn9Tr97l7Ymm2687YQCe6nYCygUgXQCmCDc3N+NJxlHlzr98t9CgXnnB0uLnKYejwTk6CuO+rWRhADMxiqKpHA+CMp/67/kcvOPsn3JychQyPD+l9+73RA0ZH+kChpwUuFYRRSV+O6rhyZBgdB8hJ4C9sz8Qns/w4iNajUCSmwIngL0bpgEe3JUI4QElRHnifNjvct79QP3xM3S1zqNv5qGXBCikwPjkcprWms6f7RjKtuCqiyg9oPvpAi3nOWu1eL4oZgdI/wzTeexedjMOIIHSZa4GXaTnc9lv83zx+0QB3roGLAkY9C9v3ny8EqMwREeThZjcFwbIPhHhx5dPM4O7NqKZi9CugYODA5rNZtxvNpvO1LhgswLmAtgp2NraYjgcxk5PTk4SNeJ6Jpz7Pqurq7MDGHKlVGJlKysruQrk7XyFAIylV3V8fJx5WLnm3EoBUwMuSY0CUspCKmitc1W4MQVpMwpk5d/3fdbW1ma6GWWmwL6WGbNrIEuF9LU8T4WZtmLb0sd0eryopQGqUspLrfW9Wu36Cq+UYm5ubmbHxmzAUqnUAxaBC2CYBqgCDzudzi/b29s/A/O3ipgNcnV6evoaeAx4wF9AZOtXAh4B3wI14M74xa9hV1wvdgCcAx+APjj+nI4Dl8dAs1/03RYBI65lHwJxPr8AVhTGqEDsvuoAAAAASUVORK5CYII=' ,
		'http://xmlns.com/foaf/0.1/Person'                     => 'data:image/png;charset=binary;base64,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',
		'http://xmlns.com/foaf/0.1/Group'                      => 'data:image/png;charset=binary;base64,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',
		'http://www.w3.org/2003/01/geo/wgs84_pos#SpatialThing' => 'data:image/png;charset=binary;base64,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' ,
		'http://www.w3.org/2006/vcard/ns#Address'              => 'data:image/png;charset=binary;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABHNCSVQICAgIfAhkiAAABqVJREFUWIXVl2lsVNcVx393eW82j8dgx3YweAy4yBQIu9oiRwi6OGpQpaIqtGmKhJSoSiJBVAWp39pK+ZRPEVGVLhIEulBWKWkrlQYVNQSJuA2ECJpAYuJiMNhgPNhjz/buvf0wC16GVcqHHulKd94975z/Oed/ztwH/8+yT8qX/yDErb0w52FtiId5aT/4KLWnsaHhiZaamtrTvb39gTHfeBo++cIB7IHGiNbvLJg3b0HbrFnh3NAQI+k0p/v6BvJB8J0fQveD2JMPovw7WBELhU6vWr58cVtbWxgpcUDE91ne2toU9ry//hG++YUA2K/Uj2prav7e2dk5q76xUSIlKAVC4ICQ57EsmWwI+f6f9in1g/u1e88S/BzkQqVeb2hqenrFqlV1SkowBqzFplLY4WEyxmAABwTWcu7y5ZuZXO4Xm6zdcS/76q5RQ+IRrY+2d3Q8sWT16lqpSupCYIeGcMPDFIwhpBTGOYxzIAT1tbWRkVzuq9+1tvaAtf94qAzshgUxzzuybM2a2U2trRprwRhcEGAvXSK4dYszw0NcHbzBymSS5miUcWvJOocDLHBxYGBkNJ0++D1jnhXFBE2TqhzYq9STtTU173Vu2NDW1N6uUQqkxBmDvXiR8Zs3eX9okMRL21h16hT/TcToSY8SlZJwiRM4x9zGxtqZicRTB7R+6xjo+8rAAa1/Fm9o2PaVrq4ZXiRSjNpa3MgI5vx5bo6Oci6TJvnLN4gseQxjDMJarv30ZdSpMyyK15IDRozBUQx7IJXKDA4PfzAcBF0/hvGqAHZBOKbUoTkLFjy+eP36uJhItoEBzIULXEqnuRIJM//N3cj6eoIgQEqJlBJrLUO/eoP03r0sT9RhhCBlDBZwzpFKp3P9N26cTwfB2i2QKvtVAL+H2VGtTyzt7Fz5pbVro+X2Qghsby/BhQucS6dJf3kh7bv3IONxjDFEIhFmzpxJLBbDGEN4xUq81jl8fOQIzdojqjVZ57CA7/s66vv12fHxp77t3IG3IF3JwD4p+x/fuLE5MX9+MSPW4oIA89FHZPv6+HB0lBmbN9P0wgsYY7DWEovFCIfDk8qXyWTIZDJkz56l5/nnWRIKURMKcd0YCiVy5rNZ23v16hVrzLrvQ48E0ErlEu3tAqVAKZwxBN3djHz+Od0jI8x+9VWaX3wR5xxKKRKJBL7vY62tLGMMvu9TU1NDbOlSOg4d4qyU3MhkaNQav0ROHQrJuS0tc6RS/5zcBaXJ5sbGCN59l2uXL/OhtSw6eJDEunU459BaE41GS0ma7Ly8F0IQiUQIt7Sw+O23udTYSO/YGI1aEymBkJ6HVspNa0M3MEDh2DEuDA5y6dFHWX7kCOF58wDwPA/P8yY5nup84vI8Dy+RYOH+/YytXs1/Rkd5RCniUiLF7earADCffUbu+HFO9vcTrF/PY4cPI+NxnHNIKRFC3JfzqdmQnse8114j/MwzvD80xAwpmSFvx60BAmPC106e5HwqNTq+aVNsxSuvSGstuOLwcs7h3ORBVv4thEBrjRBimk5ZzxhDy9atHO/pKQRHjwYd8XgkHwQ1FQA5a9f/u6/vxNwtW35yuK5u19eVIggCjh59h7HxsYqjqRKNxOjq6mLHjtfJ53PTzgE8P8RL27aSz+cZnD8/aDtx4lsf9Pf/pWDtmgqAzfAxzs1k5062b9++q8LqkM+nPecnGRRCVMAs7FiEc458PodJPonvKTwtMcaSL1hyhQCu/A3nHIVCAaVUZmN//3tA3TQOTJRyDVvntKK1rhDQ9/3K3vM8kq1JrLUlYCAFKCGKJJuQsHIZrLXTalT1D8I5h7WWZDKJ1lVVAGhoaKgAGM8G5PKG8VzArbEcI2M5MjnD16LFgEpjO/9AAJTWlUhEaSMZR4te8q4DBBUA//rkWnWUUSZ2RWHq8V1LUHYsKKVUgCc+ZcOaP6PEKFCd+dXEGINzLpj6/I4ZMMaAc5PZLyDPMg4cX1wpTXHo+KyrO1PVsfaKI7tQKCClzD4QAO15FIOf3IKe5xXxCMhmszz33LNV27RsK5fLlct6zwyEylFZa5FCVjU8EVA+nyefv82taiWRUlIoFAiCwAINFO8DwVQAPlAvhAiy2ayOx+OTev5OEo/H73peBnT9+nUKhYIBmkt+BwE7EYAG/FQq9ebOnTu3OOfuemMG+M2vf3svlYoIIbLd3d17gTDFYAVMvhMKoL60aktKdx4CDyZ5irevDHALuAxkpwIoiwdESkgf6uO1ijigUHKaZcIV/X9F/F/h7eNregAAAABJRU5ErkJggg==',
		'http://www.w3.org/2006/vcard/ns#Vcard'                => 'data:image/png;charset=binary;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABHNCSVQICAgIfAhkiAAABSVJREFUWIXtl09sVFUUxn/3vZnXYVoKlFL6v8goUJoYqkBQy6KJYJQoYUPSAAlp3BE3uCBKdGNCgCFQaAhRlIVKYtyomEAkFROo0UpKB0K0SK3j0FJogRaGGdqZ985xAR3pXzTauNCTnNyXd+8533e+e+7Ne/Avm5nK5HqLGWT5Q6gzB9EsLEmico2c1CVjSAP4pgz8HuWkg0+hWZUYXwmGLNC7GDdGwhSrDn1rDIkpIaC3mEE6WI0VeA4C1ah/PlaWHx1KQvpnDNO5a1wYPDU1Cjj+xyCrEgJLMCWV2E86aK4FfVlIazVICrRXbzsRH0BjY+Mix3E+E5EFtm1bfxf/+Kl2yksuU15yBSewBJvnMHYJ4l7CS6dJ3r38km0N9MzM6f7aB+A4zudr165dMHfu3H+mKb2vwLVAboI9B6wysJeCHQdfPj7rWuDS5eK6mQu6G30AqVTqiYfBxXM59v6bnG36GIBlz2/klVd3YNk+VBVVBcg8j3ZLbSz1Y2kAY3qBdtCbIDGQm/idmSTu+WdhDQ0M98AI2U8e3UHHueOsfvFlUDj7w3FOHs3hhU1vjQs4moxoAWgRaDG2tANDQDZoPyrdGN8S7sQHDdncGHe/Tx87xNJlzzLb7ifPd4uly57h9LFDiAgiQmtrK7FYDFUlEokQiUQycyJCPDGXwx+cpjUyhCuPI+5NxP0Zz0viShVYVVy5Oh9j8CY8BYrieh6KosqD8b739PTQ0tJCLBYjPz+f9vZ2Vq5cSUtLCytWrGD16tUIy2k7f5543GVw8C5FhXn80nmNl9asw/GtIXnv07HSD9v0ikU0N5/ilpdHvzebM2dOMWv+kkyFTU1NlJWVEQgE8DyPdevW0dfXR01NDdevXwfgu+/b+CVaxG/d1Zxte4xzFxbREV3BoLsGTEEGa4wCXzV9RNv1dh4PTuPLLz5BPCGVM52+WISzrSd5uvp5du/ejaqyatWqjCq1tbUj+uDgwYPj9oiIjsAbQUBEOHzkDRZWluNKL26e4LrK/IpFzOpPcui913nvYBudnZ0MDg5mEj8MkNlCVRzHoby8fEzDTkggnU7R338DQ4h4/DbyICCVUnJziui5dgHXTTNv3rxHnoTJfEIClmVTWhIi+utvLKysxu/3A+C6aX66dIFZM/MxxqKzs5OhoaEx4JNV/qcIiAjbtx3lnZ11fPP1af5YaygqLGf7tg9RVSoqKh59F4hMODchAVWlrHQh7za2PjLxZJI/am6E6qMV2LlzJ6pKQ0MDu3btQkQyYzgcJhwOo6qEw2FEhD179mRI7d27NzPu27cPEaGhoYH9+/ejqhw4cAARmVyB4SqHF/5VecdTaLi44flJe2B04ocDRzfceEnHix1N8k8pUFdXh4iQSCTYsGEDyWSSjRs3AtDb20t9fT19fX1s3ryZgYEBRIT6+noSiQSbNm3Ctm3i8Tjr168nOzt7TOUTKjBM5OLFixhjSCQS+P1+iouLiUajBINBcnNzKSgooKOjg2nTptHV1YVlWSxevJhoNMrVq1cJBoOEQiG6u7vxPI/a2tpJCYiqWsYYtm7diohQU1MzRr5QKDRCyuXLl6OqVFVVZd4VFhaOiAuFQhmwLVu2YNv2cKGaIeD3+3/s6uqqKi0tNcZM6Zc6IkI0GlXLstqBXB8QbG5ufi2dTh8B5jHF/wqAuq7bdeLEibeB2QawgbwHPgPIBrKmCDzF/QKTwB2g5+FqbcDh/rY4U0TAA1wg/YDM2Kvxf/vP2e9P1+y59D+5HwAAAABJRU5ErkJggg==',
		'http://www.w3.org/2002/12/cal/ical#Vevent'            => 'data:image/png;charset=binary;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABHNCSVQICAgIfAhkiAAABMlJREFUWIXtlktoVFcYx3/n3LkzY5o0mcTExElCGvO0LooiphNfFAJtFhU0dFEVbZoGZ1aBbsSKD6q4sRZr46OgForRWnBhoAu70cigduETROJoAglUkjAkzWNe997TRTKXTDKx2sRFaf8wDHzfvef/u9/3nXsP/K//ukS6oN/vXyeE6FRKZS+EiaZpYcMwPjp9+vTvM3OOOW74zufzZVdVVfHo0SPKysrmBRAKhXIfPnx4DFj7SgBKqYqxsTG6u7sxDINQKDQvgFgshmVZ1elycmbgwIED0rKsDNM0GR8fn5exbSIlwNvpcmlnIBAI3ALqlFILAiCEQAhxrr29/fOZublakLF12zbqfb7XNrOmoJNPJoSg6+ZNLl28mJPu+rQAuq5XlJaWMjI6SiQa5XXqIIVACoGQEikli1wuampqcLvdH6S9fmagubk5S0q5aHFeHoZh2OaxWIxdra2sWbWKd6ur+bChgd+uXQNgfHycm11dHDt6lE+amvjhzBl7vYRhkJebi2EYOTt37nT/LYCUcrWUUrjdbgzDsONDg4MUFRVx8NAh9u7bR8/z5/hbW1FK8cvlyzTv2MGN69fpunGDp0+fTvYdUEohpKS8vFw5HI6tM/1mtUBK2VC+bBmGaWJNi3uLi9l/8CBMLXru7FkmJiYQQrClqYlt27cTjUapKi+nr68PgT18mKbJmro60d3d/Slw9qUAmqY1LF++nMS0p0/qx/PneRYKcf/uXfLy8vj+1CkAsrKy7P8cj4f+vj7bXAiBZVlUVVbicrnqXqUF1WVlZRiJxCyAZ6EQvT09/PHiBQ8ePOD+vXspeSEEJSUl9Pf3zwLweDwopTJaWlqWzAnQ3Nyc5XQ638rPzyeepgJfHz7MTx0d3LpzB93hoP3EiRRzIQTFxcVMTEwQDoftOUjma2trla7rX8wJoGnaSk3TyMjISBnAcDjMpY4OBgYG+HNkhKtXrxIOh9mwcWMqAFBSUgIwOQczqtDY2Ch0Xf9sTgAhxHqv1ytM00zZ+0ODg+zds4f3VqygprKSr3bvZpffz779+1PMhRCUlJYCTLZhWmUspfAWF+P1et9paWmptT2nA/j9/uDHmzb51m/YwMjoaEr5E/E4AwMDOHSd/Pz85Pt98imEQE69eJLlz1+8mJzsbDsupUSTklgsxpEjR4aGhobeP3nyZCgFIBAIDLe1tWUXFBYyEY3yKhKQYiKltIG0NDEhJi3b2tpIJBLr7G0YCAQyXS5X1pLCwrQDOCfAtD4nW2GbTYNI5oeHh7lw4YKladrPx48fD9oApmmu1HWdt6f29HyklEIphWmaRKNRwuEwT548IRgMYhgGIyMjZiKRaAGUY9pNa0ZHR2UgEJiXuRACp8uF7nDgdDrJzMwkMzOT3Nxcqqurefz4MUVFRbHe3t6NwK82gKZpa30+H7dv32bz5s243bO+G/NWJBIhGAxSUVGR0dPTsyUFQCm12uPxUFhYyJUrV7As6yVL/TNJKVm6dCkFBQXS4XA0As4kgFRKLRJCUF9f/0bMk9I0jUQigVIqDxBJACsSiXzZ2dn5rWmaac9uCykhhBmPx78B4jPPhFlMHh7dgHOBfZN7Ow6EgTFApT2UAtrUbyGlABN4c/39V+ovlKTEgEoJpX4AAAAASUVORK5CYII=',
		'http://purl.org/rss/1.0/channel'                      => 'data:image/png;charset=binary;base64,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',
		'?'                                                    => 'data:image/png;charset=binary;base64,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',
	};
	
	my $equiv = {
		'http://xmlns.com/foaf/0.1/PersonalProfileDocument'  => 'http://xmlns.com/foaf/0.1/Document' ,
		'http://www.w3.org/2003/01/geo/wgs84_pos#Point'      => 'http://www.w3.org/2003/01/geo/wgs84_pos#SpatialThing' , 
		'http://www.w3.org/2006/vcard/ns#Location'           => 'http://www.w3.org/2003/01/geo/wgs84_pos#SpatialThing' , 
		'http://purl.org/NET/c4dm/event.owl#Event'           => 'http://www.w3.org/2002/12/cal/ical#Vevent',
		'http://purl.org/rss/1.0/item'                       => 'http://xmlns.com/foaf/0.1/Document' ,
	};
	
	return $icons->{$type} || $icons->{ $equiv->{$type} } || $icons->{'*'};
}

1;
