package RPi::WiringPi::Core;

use strict;
use warnings;

our $VERSION = '0.02';

use Carp qw(croak);

require XSLoader;
XSLoader::load('RPi::WiringPi::Core', $VERSION);

# not yet implemented:

#extern int  wiringPiSetupSysPhys   (void) ;
#extern void pinModeAlt          (int pin, int mode) ;
#extern int  analogRead          (int pin) ;
#extern void analogWrite         (int pin, int value) ;

sub new {
    return bless {}, shift;
}
sub setup {
    return wiringPiSetup();
}
sub setup_sys {
    return wiringPiSetupSys();
}
sub setup_gpio {
    return wiringPiSetupGPIO();
}
sub pin_mode {
    my ($self, $pin, $mode) = @_;
    if ($mode != 0 && $mode != 1){
        croak "Core::pin_mode() mode param must be either 0 (input) or 1 (output)\n";
    }
    pinMode($pin, $mode);
}
sub pull_up_down {
    my ($self, $pin, $value);
    # off, down up = 0, 1, 2
    if ($value != 0 && $value != 1 && $value != 2){
        croak "Core::pull_up_down requires either 0, 1 or 2 for value";
    }
    pullUpDnControl($pin, $value);
}
sub read_pin {
    my ($self, $pin) = @_;
    return digitalRead($pin);
}
sub write_pin {
    my ($self, $pin, $value) = @_;
    if ($value != 0 && $value != 1){
        croak "Core::write_pin value must be 0 or 1\n";
    }
    digitalWrite($pin, $value);
}
sub pwm_write {
    my ($self, $pin, $value) = @_;
    if ($value > 1024 || $value < 0){
        croak "Core::pwm_write value must be 0-1024";
    }
    pwmWrite($pin, $value);
}
sub {};
1;
__END__

=head1 NAME

RPi::WiringPi::Core - wiringPi Core Functions

=head1 SYNOPSIS

  use RPi::WiringPi::Core;

  my $core = RPi::Wiring::Core->new;

=head1 DESCRIPTION

This is an XS-based module, and requires L<wiringPi|http://wiringpi.com> to be
installed.

It provides Perl method access to (at this time, most of) the
L<wiringPi Core|http://wiringpi.com/reference/core-functions> along with a few
of its system functions.

Although this module can be used directly, it's generally used as a base class
for other modules.

=head1 METHODS

=head2 new()

Returns a new C<RPi::WiringPi::Core> object.

=head2 setup()

See L<wiringPi setup functions|http://wiringpi.com/reference/setup> for
for information on this method.

=head2 setup_sys()

See L<wiringPi setup functions|http://wiringpi.com/reference/setup> for
for information on this method.

=head2 setup_gpio()

See L<wiringPi setup functions|http://wiringpi.com/reference/setup> for
for information on this method.

=head2 pin_mode($pin, $mode)

Puts the GPIO pin in either INPUT or OUTPUT mode.

Parameters:

    $pin

Mandatory: The GPIO pin number, using wiringPi's pin number representation.

    $mode

Mandatory: C<0> to have the pin listen on INPUT, and C<1> for OUTPUT.

=head2 read_pin($pin);

Returns the current state (HIGH/on, LOW/off) of a given pin.

Parameters:
    
    $pin

Mandatory: The wiringPi number representation of the GPIO pin.

=head2 write_pin($pin, $state)

Sets the state (HIGH/on, LOW/off) of a given pin.

Parameters:

    $pin

Mandatory: The wiringPi number representation of the GPIO pin.

    $state

Mandatory: C<1> to turn the pin on (HIGH), and C<0> to turn it LOW (off).

=head2 pull_up_down($pin, $direction)

Enable/disable the built-in pull up/down resistors for a specified pin.

Parameters:

    $pin

Mandatory: The wiringPi number representation of the GPIO pin.

    $direction

Mandatory: C<2> for UP, C<1> for DOWN and C<0> to disable the resistor.

=head2 pwm_write($pin, $value)

Sets the Pulse Width Modulation duty cycle (on time) of the pin.

Parameters:

    $pin

Mandatory: The wiringPi number representation of the GPIO pin.

    $value

Mandatory: C<0> to C<1024>. C<0> is 0% (off) and C<1024> is 100% (fully on).

=head1 AUTHOR

Steve Bertrand, E<lt>steveb@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2016 by Steve Bertrand

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.18.2 or,
at your option, any later version of Perl 5 you may have available.
