package Deep::Hash::Exists;

use strict;
use warnings;
use utf8;

use base qw( Exporter );

our @EXPORT = qw();
our @EXPORT_OK = qw( key_exists );
our %EXPORT_TAGS = ( all => [ @EXPORT, @EXPORT_OK ] );
our $VERSION = '0.02';

sub key_exists($$) {
    my $hash_ref = shift;
    my $array_ref = shift;
    
    return 0 if ref $hash_ref ne 'HASH';
    
    my $key = shift @$array_ref;
    
    if ( exists $hash_ref->{$key} ) {
        if ( scalar(@$array_ref) > 0 ) {
            return &key_exists( $hash_ref->{$key}, $array_ref );
        } else {
            return 1;
        }#fi
    } else {
        return 0;
    }#fi
}

1;


__END__


=head1 NAME

Deep::Hash::Exists - verify existence of keys hash

=head1 SYNOPSIS

 use Deep::Hash::Exists qw( key_exists );
 
 my $hash_ref = {
         A => 'one',
         B => [ 'one', 'two' ],
         C => { 
             'one' => 1, 
             'two' => 2, 
         },
 };
 
 if ( key_exists( $hash_ref, [ 'C', 'one' ] ) ) {
     printf "Output: %d", key_exists( $hash_ref, [ 'C', 'one' ] );
 }
 # Output: 1
 
 if ( key_exists( $hash_ref, [ 'D' ] ) ) {
     printf "Output: %d", key_exists( $hash_ref, [ 'D' ] );
 }
 # Output: 0
 
=head1 DESCRIPTION

Exists hash:

 my $hash_ref = {
         A => 'one',
         B => [ 'one', 'two' ],
         C => { 
             'one' => 1, 
             'two' => 2, 
         },
 };

If verify existence of keys standard way, will be created non existent keys:

 exists $hash_ref->{C}{three}{PI}{0};
 print Dumper $hash_ref;

Output:

 # $VAR1 = {
 #          'A' => 'one',
 #          'B' => [
 #                   'one',
 #                   'two'
 #                 ]
 #          'C' => {
 #                   'one' => 1,
 #                   'two' => 2,
 #                   'three' => {
 #                                'PI' => {}
 #                              }
 #                },
 #        };

Subroutine C<key_exists> does not create new keys:

 key_exists( $hash_ref, [ 'C', 'three', 'PI', 0 ] );
 print Dumper $hash_ref;
 
Output:

 # $VAR1 = {
 #          'A' => 'one',
 #          'B' => [
 #                   'one',
 #                   'two'
 #                 ],
 #          'C' => {
 #                   'one' => 1,
 #                   'two' => 2
 #                 }
 #        };


=head2 key_exists($$)

B<key_exists>( I<$hash_ref, $array_ref_keys> ) - verify existence of keys hash

B<Return>:

    1 - if exists hash key
    0 - if not exists hash key

B<Example>:
 
 my $hash_ref = {
         A => 'one',
         B => [ 'one', 'two' ],
         C => { 
             'one' => 1, 
             'two' => 2, 
         },
 };
 
 printf "Output: %s", key_exists( $hash_ref, [ 'A' ] );
 # Output: 1
 printf "Output: %s", key_exists( $hash_ref, [ 'B' ] );
 # Output: 1
 printf "Output: %s", key_exists( $hash_ref, [ 'B', 0 ] );
 # Output: 0
 printf "Output: %s", key_exists( $hash_ref, [ 'C', 'one' ] );
 # Output: 1
 printf "Output: %s", key_exists( $hash_ref, [ 'C', 'three' ] );
 # Output: 0
 printf "Output: %s", key_exists( $hash_ref, [ 'C', 'three', 'PI', 0 ] );
 # Output: 0
 # Subroutine does not create new keys.

=cut


=head1 SEE ALSO

L<Hash::Util>, L<Deep::Hash::Utils>


=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 Vitaly Simul <vitalysimul@gmail.com>

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.



=head1 AUTHOR

Vitaly Simul <vitalysimul@gmail.com>

=cut
