package PAR::StrippedPARL::Dynamic;
use 5.006;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut


__DATA__
M?T5,1@(!`0````````````(`/@`!````$"-```````!``````````)#"````
M`````````$``.``(`$``'0`<``8````%````0`````````!``$```````$``
M0```````P`$```````#``0````````@``````````P````0``````@``````
M```"0`````````)````````<`````````!P``````````0`````````!````
M!0````````````````!``````````$```````"QA````````+&$`````````
M`"````````$````&````,&$````````P86```````#!A8```````-&``````
M``#P<```````````(````````@````8```!880```````%AA8```````6&%@
M``````#P`0```````/`!````````"``````````$````!````!P"````````
M'`)````````<`D```````$0`````````1``````````$`````````%#E=&0$
M````$%L````````06T```````!!;0```````]`````````#T``````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````"``````````O;&EB-C0O;&0M;&EN=7@M>#@V+38T+G-O+C(`
M!````!`````!````1TY5```````"````!@```!H````$````%`````,```!'
M3E4`&[742<:<IAFD#!D?.<+1_GQP;>EA````90````````````````````4`
M`````````````#0`````````'0```"@````>`````````#T````)````5```
M```````4````1@```$<```!"````6````&(````&````/@``````````````
M`P```%`````[``````````````!+````4@```$X````W`````````&0```!*
M````/````"X`````````-0````````!/`````````$4````V````````````
M``!=````.0```!,````;``````````````!1````)0```#`````I````````
M```````(`````````"<``````````````$$``````````0```"0````*````
M0````%8```!@``````````````!:````30```"H```!7````/P```#(````O
M`````````%L```````````````````!<````3``````````K````4P```#$`
M```+````50````(`````````````````````````````````````````````
M``0```````````````````!>````````````````````````````````````
M````````````````%0````````!C```````````````'````%@``````````
M``````````````!A`````````!(````9````$0```%\````8````60```"8`
M``!#`````````"(`````````%P```"T````?````````````````````````
M``````!)`````````````````````````#H``````````````#@```!(````
M,P`````````````````````````````````````````/````&@````P`````
M```````````````<````(````"$`````````(P``````````````+````!``
M```-``````````````````````````````````````````X`````````````
M`$0``````````````!$```!"````!`````@````@'!``!4`$A"H"0`"0A@&H
M0B@`($D@!`NXHCA-N@0&"D(```!#````1P```$P```!-````5````%8```!8
M````6@```%P`````````7@```&$```!B````8P``````````````>_!*:V[%
MWQ_JT^\.SMY'+6M_FGPNQ9NV(/_D#GQ!CL6XC?$.U634?-EQ6!SB1\8!K$OC
MP/+;TQVZXY)\PN4_+/JA;&"S`8MO:@E#UH?YG(%FK\QFQ;F<0$)%U>S%B060
M#!3BGLG5LG':S>.>,V+;[0PZEPM6I)N6I]VC:W^IG)OAIP(*"HH05Z/@^TD`
M```````````````````````````````'`0``$@``````````````````````
M`````@``$@````````````````````````"/`@``$@``````````````````
M``````!V`@``$@````````````````````````!<`@``$@``````````````
M``````````"6`@``$@`````````````````````````-`P``$@``````````
M``````````````"=````$@````````````````````````"K`@``$@``````
M``````````````````#N`@``$@````````````````````````#F`@``$@``
M``````````````````````!Y````$@````````````````````````"Y`P``
M$@`````````````````````````"`0``$@````````````````````````#.
M`@``$@````````````````````````"Y`@``$@``````````````````````
M``!^`@``$@`````````````````````````&`P``$@``````````````````
M```````=`P``$@````````````````````````!K`0``$@``````````````
M```````````K`0``$@````````````````````````#``@``$@``````````
M``````````````"K````$@````````````````````````"U`0``$@``````
M```````````````````M`@``$@````````````````````````!M`@``$@``
M``````````````````````!2`0``$@`````````````````````````]`@``
M$@````````````````````````!D`@``$@````````````````````````"B
M`0``$@````````````````````````"D`@``$@``````````````````````
M``#[`0``$@````````````````````````!$`P``$@``````````````````
M``````#4`@``$@`````````````````````````]`P``$@``````````````
M```````````Y`0``$@`````````````````````````1````(```````````
M``````````````!"````$@````````````````````````#9````$@``````
M```````````````````9`@``$@````````````````````````"N`0``$@``
M``````````````````````"(````$@````````````````````````!@`0``
M$@`````````````````````````T````$@`````````````````````````,
M`0``$@````````````````````````"7`0``$@``````````````````````
M``#F````$@````````````````````````"<`@``$@``````````````````
M``````#T`0``$@`````````````````````````N`P``$@``````````````
M```````````D`P``$@````````````````````````!Y`0``$@``````````
M```````````````V`P``$@````````````````````````"R`@``$@``````
M```````````````````@````(`````````````````````````"]````$@``
M```````````````````````6`P``$@````````````````````````!)`0``
M$@````````````````````````#;`@``$@````````````````````````!4
M`@``$@````````````````````````!/`@``$@``````````````````````
M```1`@``$@````````````````````````!1````$@``````````````````
M``````#'`@``$@````````````````````````#X`@``$@``````````````
M``````````"'`0``$@```'`=0`````````````````#"`P``$@`.`+!60```
M````<`````````"0`P``$@`/`%A80`````````````````#G`P``$@`.`"!)
M0```````M0D```````!1`P``$@`.```A0```````#@(```````"6`P``$@`.
M``!#0```````!0````````!+!```$@`.`!!#0```````J`$```````!X!```
M$@`.`"!70```````.@````````!%!```$@`,`+`<0`````````````````"4
M!```$@`.`.!20```````RP,```````!U`P``$`#Q_V3!8```````````````
M``#.`P``$@`.`,!$0```````)0`````````!!```$0`0`&A80```````!```
M``````#``0``$0`:`*#!8```````$P````````"!`P``$`#Q_R#28```````
M``````````!#!```$@`.`-!!0```````*@````````!8!```$@`.`!!(0```
M````"0$```````!J!```$@`.`)!'0```````2`````````#T`P``$``9`(!E
M8`````````````````!H````$@```#`@0`````````````````#;`P``$@`.
M`/!$0```````S@````````!6`P``$0`:`)C!8```````"`````````!N`P``
M$`#Q_V3!8`````````````````!7`P``(0`:`)C!8```````"`````````"&
M`P``$@`.`(!70````````@`````````8`0``$@```)`?0```````````````
M```F!```$@`.`)!70```````B0````````#Z`P``$@`.`!`C0```````````
M``````#V`P``(``9`(!E8`````````````````#W````$0`:`(#!8```````
M!`````````!8`P``(0`:`)C!8```````"`````````"G`P``$@`.`.!'0```
M````)``````````0!```$@`.`%!"0```````J@````````#,````$0`:`)#!
M8````````0`````````V!```$@`.`,!%0```````RP$`````````;&EB<&5R
M;"YS;RXU+C$T`%]?9VUO;E]S=&%R=%]?`%]*=E]296=I<W1E<D-L87-S97,`
M4&5R;%]S>7-?=&5R;0!097)L7W-Y<U]I;FET,P!097)L7W-A=F5?9&5S=')U
M8W1O<E]X`%!E<FQ?871F;W)K7VQO8VL`<&5R;%]C;VYS=')U8W0`4&5R;%]C
M<F]A:U]N;V-O;G1E>'0`4&5R;%]A=E]S:&EF=`!097)L7W-V7S)P=E]F;&%G
M<P!097)L7W-V7W-E='!V;@!03%]D;U]U;F1U;7``4&5R;%]N97=35G!V`%!E
M<FQ?;F5W6%-?9FQA9W,`4$Q?=&AR7VME>0!P97)L7V9R964`4&5R;%]M9U]S
M970`4&5R;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<W9?<V5T<'8`4&5R;%]G=E]F
M971C:'!V`'!E<FQ?<G5N`%!E<FQ?9&]U;G=I;F0`4&5R;%]N97=84P!P97)L
M7V1E<W1R=6-T`%!E<FQ?879?9F5T8V@`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?
M<&%R<V4`4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?86QL;V,`4$Q?;65M;W)Y
M7W=R87``;&EB9&PN<V\N,@!L:6)M+G-O+C8`;&EB<'1H<F5A9"YS;RXP`&]P
M96XV-`!R96%D`%]?97)R;F]?;&]C871I;VX`;'-E96LV-`!P=&AR96%D7V=E
M='-P96-I9FEC`&-L;W-E`&QI8F,N<V\N-@!?7W)E9VES=&5R7V%T9F]R:P!E
M>&ET`'-P<FEN=&8`<W1R;F-M<`!?7W-T<F1U<`!C;&]S961I<@!S=')N8W!Y
M`%]?<W1A8VM?8VAK7V9A:6P`=6YL:6YK`&UK9&ER`')E86QL;V,`;65M8VAR
M`&=E='!I9`!S=')T;VL`<W1R;&5N`&UE;7-E=`!S=')S='(`<FUD:7(`;65M
M8W!Y`%]?;'AS=&%T-C0`;W!E;F1I<@!?7WAS=&%T-C0`7U]C='EP95]B7VQO
M8P!G971U:60`<F5A9&QI;FL`97AE8W9P`'-T<F-H<@!R96%D9&ER-C0`;65M
M;6]V90!A8V-E<W,`<W1R8VUP`%]?;&EB8U]S=&%R=%]M86EN`%]?96YV:7)O
M;@!L:6)C<GEP="YS;RXQ`%]E9&%T80!?7V)S<U]S=&%R=`!?96YD`%]?;&EB
M8U]C<W5?9FEN:0!P87)?8W5R<F5N=%]E>&5C`&=E=%]U<V5R;F%M95]F<F]M
M7V=E='!W=6ED`'!A<E]C;&5A;G5P`'!A<E]B87-E;F%M90!P87)?9&ER;F%M
M90!P87)?;6MT;7!D:7(`7U]D871A7W-T87)T`%])3U]S=&1I;E]U<V5D`'!A
M<E]C=7)R96YT7V5X96-?<')O8P!?7VQI8F-?8W-U7VEN:70`<&%R7VEN:71?
M96YV`'AS7VEN:70`<&%R7V9I;F1P<F]G`'!A<E]S971U<%]L:6)P871H`'!A
M<E]E;G9?8VQE86X`6%-?26YT97)N86QS7U!!4E]#3$5!4E-404-+`%A37TEN
M=&5R;F%L<U]005)?0D]/5`!'3$E"0U\R+C(N-0!'3$E"0U\R+C,`1TQ)0D-?
M,BXS+C(`1TQ)0D-?,BXT`````@`#``(``@`"``(``@````(``@`"`````@``
M``(``@`$``(``@```````@```````P`"````!0`"`````@`#``(``@`"````
M`````````P`"``````````````````(``P`"``(````"``(```````(````"
M``(``@`#`````@`&`````0`!``$``0`!``$``0`!``$``0`!``$````!``$`
M`0`!``$````!``(``0`"``$````!``$``0````(``0`!`````0`!``$`Y`$`
M`!`````@````=1II"0```P"J!`````````$`!``S`@``$``````````3:6D-
M```&`+8$```0````<AEI"0``!0#`!```$````!1I:0T```0`S`0``!````!U
M&FD)```"`*H$````````2&-@```````&````)0``````````````@,%@````
M```%````7P``````````````D,%@```````%````8P``````````````F,%@
M```````%````5P``````````````H,%@```````%````3P``````````````
M:&-@```````'`````0``````````````<&-@```````'`````@``````````
M````>&-@```````'`````P``````````````@&-@```````'````!```````
M````````B&-@```````'````!0``````````````D&-@```````'````!@``
M````````````F&-@```````'````!P``````````````H&-@```````'````
M"```````````````J&-@```````'````"0``````````````L&-@```````'
M````0@``````````````N&-@```````'````"@``````````````P&-@````
M```'````"P``````````````R&-@```````'````#```````````````T&-@
M```````'````#0``````````````V&-@```````'````#@``````````````
MX&-@```````'````#P``````````````Z&-@```````'````$```````````
M````\&-@```````'````$0``````````````^&-@```````'````$@``````
M`````````&1@```````'````$P``````````````"&1@```````'````%```
M````````````$&1@```````'````%0``````````````&&1@```````'````
M%@``````````````(&1@```````'````%P``````````````*&1@```````'
M````&```````````````,&1@```````'````&0``````````````.&1@````
M```'````&@``````````````0&1@```````'````&P``````````````2&1@
M```````'````'```````````````4&1@```````'````'0``````````````
M6&1@```````'````'@``````````````8&1@```````'````'P``````````
M````:&1@```````'````(```````````````<&1@```````'````(0``````
M````````>&1@```````'````(@``````````````@&1@```````'````(P``
M````````````B&1@```````'````)```````````````D&1@```````'````
M)@``````````````F&1@```````'````)P``````````````H&1@```````'
M````*```````````````J&1@```````'````*0``````````````L&1@````
M```'````*@``````````````N&1@```````'````*P``````````````P&1@
M```````'````6P``````````````R&1@```````'````+```````````````
MT&1@```````'````+0``````````````V&1@```````'````+@``````````
M````X&1@```````'````+P``````````````Z&1@```````'````,```````
M````````\&1@```````'````,0``````````````^&1@```````'````,@``
M`````````````&5@```````'````,P``````````````"&5@```````'````
M-```````````````$&5@```````'````50``````````````&&5@```````'
M````-0``````````````(&5@```````'````-@``````````````*&5@````
M```'````.```````````````,&5@```````'````.0``````````````.&5@
M```````'````.@``````````````0&5@```````'````.P``````````````
M2&5@```````'````/```````````````4&5@```````'````/0``````````
M````6&5@```````'````/@``````````````8&5@```````'````/P``````
M````````:&5@```````'````0```````````````<&5@```````'````00``
M````````````2(/L".B#!@``Z!('``#H73L``$B#Q`C#``````````#_-8)&
M(`#_)81&(``/'T``_R6"1B``:`````#IX/____\E>D8@`&@!````Z=#_____
M)7)&(`!H`@```.G`_____R5J1B``:`,```#IL/____\E8D8@`&@$````Z:#_
M____)5I&(`!H!0```.F0_____R521B``:`8```#I@/____\E2D8@`&@'````
MZ7#_____)4)&(`!H"````.E@_____R4Z1B``:`D```#I4/____\E,D8@`&@*
M````Z4#_____)2I&(`!H"P```.DP_____R4B1B``:`P```#I(/____\E&D8@
M`&@-````Z1#_____)1)&(`!H#@```.D`_____R4*1B``:`\```#I\/[___\E
M`D8@`&@0````Z>#^____)?I%(`!H$0```.G0_O___R7R12``:!(```#IP/[_
M__\EZD4@`&@3````Z;#^____)>)%(`!H%````.F@_O___R7:12``:!4```#I
MD/[___\ETD4@`&@6````Z8#^____)<I%(`!H%P```.EP_O___R7"12``:!@`
M``#I8/[___\END4@`&@9````Z5#^____);)%(`!H&@```.E`_O___R6J12``
M:!L```#I,/[___\EHD4@`&@<````Z2#^____)9I%(`!H'0```.D0_O___R62
M12``:!X```#I`/[___\EBD4@`&@?````Z?#]____)8)%(`!H(````.G@_?__
M_R5Z12``:"$```#IT/W___\E<D4@`&@B````Z<#]____)6I%(`!H(P```.FP
M_?___R5B12``:"0```#IH/W___\E6D4@`&@E````Z9#]____)5)%(`!H)@``
M`.F`_?___R5*12``:"<```#I</W___\E0D4@`&@H````Z6#]____)3I%(`!H
M*0```.E0_?___R4R12``:"H```#I0/W___\E*D4@`&@K````Z3#]____)2)%
M(`!H+````.D@_?___R4:12``:"T```#I$/W___\E$D4@`&@N````Z0#]____
M)0I%(`!H+P```.GP_/___R4"12``:#````#IX/S___\E^D0@`&@Q````Z=#\
M____)?)$(`!H,@```.G`_/___R7J1"``:#,```#IL/S___\EXD0@`&@T````
MZ:#\____)=I$(`!H-0```.F0_/___R721"``:#8```#I@/S___\ERD0@`&@W
M````Z7#\____)<)$(`!H.````.E@_/___R6Z1"``:#D```#I4/S___\ELD0@
M`&@Z````Z4#\____):I$(`!H.P```.DP_/___R6B1"``:#P```#I(/S___\E
MFD0@`&@]````Z1#\____)9)$(`!H/@```.D`_/___R6*1"``:#\```#I\/O_
M__\E@D0@`&A`````Z>#[____)7I$(`!H00```.G0^___4TB#["")?"0<2(ET
M)!!(C7PD'$B)5"0(2(UT)!!(C50D".@)_O__NI`?0`"_,"!``$B)UN@G-@``
M@#U0H"```'4OZ!G]__](A<!(B07'L"``#X2<`0``2(G'Z$'\__^+/1N@(`#H
MYOW__\:`S`0```"+/0F@(`#HU/W__X"(.P4```*+/?>?(`#HPOW__XM\)!R`
MB#L%```!@\<%2&/_2(7_#X@3`0``\D@/*L?R#U@%2SD``&8/+@5+.0``#X<Z
M`0``2,'G`^CT_/__2(M\)!!(B04HL"``QP4FL"``!````$B+%TC'0`A16D``
M2,=`$,!E8`!(QT`85%I``$B)$(M4)!R#^@%^*HU*_DR-!,T(````,<EF+@\?
MA```````2(MT#PA(B70(($B#P0A,.<%U[8U*!$B+/=NO(`"#P@-%,<"^`$)`
M`$ACR4C'1,CX`````$B)P>AD_?__A<")PW0V2(L]KZ\@`.BR^___,<"_%5E`
M`.@6&```2(7`="E(BSV2KR``Z#7[___H$/W__TB#Q"")V%O#2(L]>:\@`.C<
M_?__B</KNDB+/6&O(`!(A?]T).CW,P``Z\1(B?A(B?I(T>B#X@%("=#R2`\J
MP/(/6,#IU?[__[_*6$``Z*X7``!(A<!(B<=(B04AKR``=([KP;\!````Z+/]
M__^^H,%@`+_%6$``,<#H8OS__Y"0,>U)B=%>2(GB2(/D\%!42<?`@%=``$C'
MP9!70`!(Q\<`(4``Z+?[___TD)!(@^P(2(L%`4`@`$B%P'0"_]!(@\0(PY"0
MD)"0D)"0D)"0D)!52(GE4TB#[`B`/4B>(```=4N[2&%@`$B+!4*>(`!(@>M`
M86``2,'[`TB#ZP%(.=AS)&8/'T0``$B#P`%(B04=GB``_Q3%0&%@`$B+!0^>
M(`!(.=ARXL8%^YT@``%(@\0(6UW#9F9F+@\?A```````2(,]>#T@``!52(GE
M=!*X`````$B%P'0(7;]086``_^!=PY"0D)"0D)"0D)"0D)"005=,C4<X,<E!
M5D%505154TB![`@"``!F#Q^$``````!)BQ0(2(G02(G62,'@"$C!YAB#YO\E
M``#_`$@)\$B)UDC![@B!Y@#_``!("?!(B=9(P>X8@>;_````2`GP2(G62,'N
M($B)1$R(2(GP2,'F&$C!X`B#YO\E``#_`$@)\$B)UDC!ZCA(P>XH@>8`_P``
M2`GP2`G02(E$3)!(@\$(2(/Y0`^%=____TB-5"2(2(VT)/`!``!(@\)H9@\?
M1```2(M"V$@S`D@S0JA(,T*82(G!2`'`2,'I'T@)P8/A_TB)2AA(@\((2#GR
M==1,BP=(BU\@3(M/"$R+5Q!,BU\82(G:3(G`2`-4)(A,B<%(P>@;2,'A!4R)
MWD@#="202`G(3(G!@^#_2/?12(VL`IEY@EI,B<A,B=)(]]!,(<I,(=%,(=A(
M"=!,B<H!Q4R)R$C!XAY(P>@"28GM2`G0@^#_2(G"3"'"2`G*2(GI3(VD%IEY
M@EI(B>I(P>$%2,'J&TR)UD@#="282`G*3(G!00'43(G"2,'I`DC!XAY("=%(
MB>J#X?](]]))(<U((<),">I-B>5(C;06F7F"6DR)XDG!Y05(P>H;3`GJ38GE
M`=9(B>I(P>H"2,'E'D@#1"2@2`GJ3(GE@^+_2/?522'52"'-2`-,)*A,">U)
MB?5(C:PHF7F"6DB)\$G!Y05(P>@;3`GH28GU`<5,B>!)P>0>2,'H`DP)X$F)
M](/@_TGWU$DAQ4DAU$@#5"2P30GL28GM3HVD(9EY@EI(B>E)P>4%2,'I&TP)
MZ4F)[4$!S$B)\4C!YAY(P>D"2`GQ2(GN@^'_2/?622'-2"'&3`GN38GE2(VT
M,IEY@EI,B>)(P>H;2<'E!4@#1"2X3`GJ38GE`=9(B>I(P>4>2,'J`D@)ZDR)
MY8/B_TCWU4DAU4@AS4@#3"3`3`GM28GU2(VL*)EY@EI(B?!)P>4%2,'H&TP)
MZ$F)]0'%3(G@2<'D'DC!Z`),">!)B?2#X/])]]1)(<5)(=1(`U0DR$T)[$F)
M[4Z-I"&9>8):2(GI2<'E!4C!Z1M,">E)B>U!`<Q(B?%(P>8>2,'I`D@)\4B)
M[H/A_TCWUD@AQDDAS4@#1"303`GN38GE2(VT,IEY@EI,B>))P>4%2,'J&TP)
MZDV)Y0'62(GJ2,'E'DC!Z@)(">I,B>6#XO](]]5)(=5((<U(`TPDV$P)[4F)
M]4B-K"B9>8):2(GP2<'E!4C!Z!M,">A)B?4!Q4R)X$G!Y!Y(P>@"3`G@28GT
M@^#_2??422'%22'430GL28GM3HVD(9EY@EI(B>E)P>4%2,'I&TP)Z4F)[4$!
MS$B)\4C!YAY(P>D"2`GQ2(GN@^'_2`-4).!(]]9)(<U((<9(`T0DZ$P)[DV)
MY4G!Y05(C;0RF7F"6DR)XDC!ZAM,">I-B>4!UDB)ZDC!Y1Y(P>H"2`GJ3(GE
M@^+_2/?522'52"'-2`-,)/!,">U)B?5(C:PHF7F"6DB)\$G!Y05(P>@;3`GH
M28GU`<5,B>!)P>0>2,'H`DP)X$F)](/@_TGWU$DAQ4DAU$T)[$F)[4Z-I"&9
M>8):2(GI2<'E!4C!Z1M,">E)B>U!`<Q(B?%(P>D"2,'F'D@#5"3X2`GQ2(GN
M@^'_2/?622'-2"'&2`,$)$P)[DV)Y4B-M#*9>8):3(GB2<'E!4C!ZAM,">I-
MB>4!UDB)ZDC!Y1Y(P>H"2`GJ3(GE@^+_2/?522'52"'-2`-,)`A,">U)B?5(
MC:PHF7F"6DB)\$G!Y05(P>@;3`GH28GU`<5,B>!)P>0>2,'H`DP)X$F)](/@
M_TGWU$DAQ4DAU$T)[$F)[4Z-I"&9>8):2(GI2,'I&TG!Y05(`U0D$$P)Z4F)
M[4$!S$B)\4C!YAY(P>D"2`GQ2(GN@^'_2/?622'-2"'&2`-$)!A,">Y-B>5(
MC;0RF7F"6DR)XDG!Y05(P>H;3`GJ38GE`=9(B>I(P>4>2,'J`D@)ZDR)Y8/B
M_TCWU4DAU4@AS4@#3"0@3`GM28GU2(VL*)EY@EI(B?!)P>4%2,'H&TP)Z$F)
M]0'%3(G@2<'D'DC!Z`),">!)B?2#X/])]]1)(=1)(<5(`U0D*$T)[$Z-K"&9
M>8):2(GI28GL2<'D!4C!Z1M,">%!`<U(B?%(P>8>2,'I`D@)\8/A_TB)SD@Q
MQD@#1"0P2#'N3(VD,J'KV6Y,B>I,B>Y(P>8%2,'J&T@)\D$!U$B)ZDC!Y1Y(
MP>H"2`GJ@^+_2(G62#'.2`-,)#A,,>Y(C:PPH>O9;DR)X$R)YDC!Y@5(P>@;
M2`GP`<5,B>A)P>4>2,'H`DP)Z$F)[8/@_TB)QD@QUDG!Y05(`U0D0$PQYDB-
MM#&AZ]EN2(GI2,'I&TP)Z0'.3(GA2<'D'DC!Z0))B?5,">%)P>4%@^'_28G,
M23'$2`-$)$A),>Q.C:0BH>O9;DB)\DC!ZAM,">I!`=1(B>I(P>4>2,'J`DV)
MY4@)ZDG!Y06#XO](B=5(,<U(`TPD4$@Q]4B-K"BAZ]EN3(G@2,'H&TP)Z`'%
M2(GP2,'F'DC!Z`))B>U("?"#X/](B<9(,=9)P>4%2`-4)%A,,>9(C;0QH>O9
M;DB)Z4C!Z1M,">D!SDR)X4G!Y!Y(P>D"28GU3`GA2<'E!8/A_TF)S$DQQ$@#
M1"1@23'L3HVD(J'KV6Y(B?)(P>H;3`GJ00'42(GJ2,'E'DC!Z@)-B>5(">I)
MP>4%@^+_2(G52#'-2`-,)&A(,?5(C:PHH>O9;DR)X$C!Z!M,">@!Q4B)\$C!
MYAY(P>@"28GM2`GP@^#_2(G&2#'62<'E!4@#5"1P3#'F2(VT,:'KV6Y(B>E(
MP>D;3`GI`<Y,B>%)P>0>2,'I`DF)]4P)X4G!Y06#X?])B<Q),<1(`T0D>$DQ
M[$Z-I"*AZ]EN2(GR2,'J&TP)ZD$!U$B)ZDC!Y1Y(P>H"38GE2`GJ2<'E!8/B
M_TB)U4@QS4@#C"2`````2#'U2(VL**'KV6Y,B>!(P>@;3`GH`<5(B?!(P>8>
M2,'H`DF)[4@)\(/@_TB)QD@QUDG!Y05(`Y0DB````$PQYDB-M#&AZ]EN2(GI
M2,'I&TP)Z0'.3(GA2<'D'DC!Z0))B?5,">%)P>4%@^'_28G,23'$2`.$))``
M``!),>Q.C:0BH>O9;DB)\DC!ZAM,">I!`=1(B>I(P>4>2,'J`DV)Y4@)ZDG!
MY06#XO](B=5(,<U(`XPDF````$@Q]4B-K"BAZ]EN3(G@2,'H&TP)Z`'%2(GP
M2,'F'DC!Z`))B>U("?"#X/](B<9(,=9)P>4%2`.4)*````!,,>9(C;0QH>O9
M;DB)Z4C!Z1M,">D!SDR)X4G!Y!Y(P>D"28GU3`GA2<'E!8/A_TF)S$DQQ$@#
MA"2H````23'L3HVD(J'KV6Y(B?)(P>H;3`GJ00'42(GJ2,'E'DC!Z@)-B>5(
M">I)P>4%@^+_2(G52#'-2`.,)+````!(,?5(C:PHH>O9;DR)X$C!Z!M,">@!
MQ4B)\$C!YAY(P>@"2`GP@^#_2(G&2#'62`.4)+@```!,,>9,C:PQH>O9;DB)
MZ4B)[DC!Y@5(P>D;2`GQ3(GF2<'D'DC![@)!`<U,">:#YO](B?%(,<%(`X0D
MP````$@QZ4R-I`JAZ]EN3(GJ3(GI2,'A!4C!ZAM("<I(B>E(P>4>2,'I`D$!
MU$@)Z8/A_TB)RD@Q\DPQZDB-K!"AZ]EN3(G@3(GB2,'B!4C!Z!M("=!,B>I)
MP>4>`<5(P>H"N-R\&X],">I)B<5,`ZPDR````(/B_TF)UDP![DF)U4DASDD)
MS4TAY4T)]4F)[DD!]4B)[DG!Y@5(P>X;3`GV00'U3(GF2<'D'DC![@),">9)
MB<1,`Z0DT````(/F_TF)]DDAUDP!X4F)]$D)U$DA[$T)]$V)[DD!S$R)Z4G!
MY@5(P>D;3`GQ00',2(GI2,'E'DC!Z0)(">E(B<5(`ZPDV````(/A_TF)SD@!
MZDB)S4@)]4PA[4DA]DP)]4V)YD@!U4R)XDG!Y@5(P>H;3`GR`=5,B>I)P>4>
M2,'J`DP)ZDF)Q4P#K"3@````@^+_28G622'.3`'N28G520G-32'E30GU28GN
M20'U2(GN2<'F!4C![AM,"?9!`?5,B>9)P>0>2,'N`DP)YDF)Q$P#I"3H````
M@^;_28GV22'63`'A28GT20G422'L30GT38GN20',3(GI2,'I&TG!Y@5,"?%!
M`<Q(B>E(P>4>2,'I`D@)Z4B)Q4@#K"3P````@^'_28G.22'V2`'J2(G-2`GU
M3"'M3`GU38GF2`'53(GB2<'F!4C!ZAM,"?(!U4R)ZDG!Y1Y(P>H"3`GJ28G%
M3`.L)/@```"#XO])B=9)(<Y,`>Y)B=5)"<U-(>5-"?5)B>Y)`?5(B>Y)P>8%
M2,'N&TP)]D$!]4R)YDG!Y!Y(P>X"3`GF28G$3`.D)``!``"#YO])B?9)(=9,
M`>%)B?1)"=1)(>Q-"?1-B>Y)`<Q,B>E)P>8%2,'I&TP)\4$!S$B)Z4C!Y1Y(
MP>D"2`GI2(G%2`.L)`@!``"#X?])B<Y)(?9(`>I(B<U("?5,(>U,"?5-B>9(
M`=5,B>))P>8%2,'J&TP)\@'53(GJ2<'E'DC!Z@),">I)B<5,`ZPD$`$``(/B
M_TF)UDDASDP![DF)U4D)S4TAY4T)]4F)[DD!]4B)[DG!Y@5(P>X;3`GV00'U
M3(GF2<'D'DC![@),">9)B<1,`Z0D&`$``(/F_TF)]DDAUDP!X4F)]$D)U$DA
M[$T)]$V)[DD!S$R)Z4G!Y@5(P>D;3`GQ00',2(GI2,'E'DC!Z0)(">E(B<5(
M`ZPD(`$``(/A_TF)SD@!ZDB)S4DA]D@)]4PA[4P)]4V)YD@!U4R)XDG!Y@5(
MP>H;3`GR`=5,B>I)P>4>2,'J`DP)ZDF)Q4P#K"0H`0``@^+_28G622'.3`'N
M28G520G-32'E30GU28GN20'U2(GN2<'F!4C![AM,"?9!`?5,B>9)P>0>2,'N
M`DP)YDF)Q$P#I"0P`0``@^;_28GV3`'A28GT20G422'L22'630GT38GN20',
M3(GI2<'F!4C!Z1M,"?%!`<Q(B>E(P>4>2,'I`D@)Z4B)Q4@#K"0X`0``@^'_
M28G.22'V2`'J2(G-2`GU3"'M3`GU38GF2`'53(GB2<'F!4C!ZAM,"?(!U4R)
MZDG!Y1Y(P>H"3`GJ28G%3`.L)$`!``"#XO])B=9)(<Y,`>Y)B=5)"<U-(>5-
M"?5)B>Y)`?5(B>Y(P>X;2<'F!4P)]D$!]4R)YDG!Y!Y(P>X"3`GF28G$3`.D
M)$@!``"#YO])B?9)(=9,`>%)B?1)"=1)(>Q-"?1-B>Y)`<Q,B>E)P>8%2,'I
M&TP)\4$!S$B)Z4C!Y1Y(P>D"2`GI2(G%2`.L)%`!``"#X?])B<Y)(?9(`>I(
MB<U("?5,(>U,"?5-B>9(`=5,B>))P>8%2,'J&TP)\@'53(GJ2<'E'DC!Z@),
M">I)B<5,`ZPD6`$``(/B_T@#A"1@`0``28G622'.3`'N28G520G-2`'!32'E
M30GU28GN20'U2(GN2<'F!4C![AM,"?9!`?5,B>9)P>0>2,'N`DP)YH/F_TB)
M\$F)]$@)T$DAU$@AZ$P)X$F)[$C!Y1Y,C30!3(GH3(GI2,'A!4C!Z!M)P>P"
M2`G(20GL00'&N-;!8LI!@^3_2(G!2`.,)&@!``!,B?5(P>4%2`'*3(GA2#'Q
M3#'I2`'13(GR2,'J&T@)ZDR)[4G!Y1Y$C3P*2(G"2`.4)'`!``!(P>T"3`GM
M3(GY@^7_2,'A!4@!UDB)ZDPQXDPQ\DR-+!9,B?I(P>H;2`G*3(GQ2<'F'D$!
MU4B)PD@#E"1X`0``2,'I`DR)[DP)\4C!Y@6#X?])`=1(B<I(,>I,,?I)`=1,
MB>I(P>H;2`GR2(G&00'42`.T)(`!``!,B?I(P>H"2<'G'DV)YDP)^DG!Y@6#
MXO](`?5(B=9(,<Y,,>Y(`?5,B>9(P>X;3`GV`?5,B>Y)P>4>2,'N`DF)[DP)
M[DF)Q4P#K"2(`0``@^;_2<'F!4P!Z4F)]4DQU4TQY4D!S4B)Z4C!Z1M,"?%!
M`<U,B>%)P>0>2,'I`DV)[DP)X4F)Q$P#I"20`0``@^'_3`'B28G,23'T2<'F
M!4DQ[$D!U$R)ZDC!ZAM,"?)!`=1(B>I(P>4>2,'J`DV)YD@)ZDB)Q4@#K"28
M`0``@^+_2<'F!4@![DB)U4@QS4PQ[4@!]4R)YDC![AM,"?8!]4R)[DG!Y1Y(
MP>X"28GN3`GN28G%3`.L)*`!``"#YO])P>8%3`'I28GU23'533'E20'-2(GI
M2,'I&TP)\4$!S4R)X4C!Z0))P>0>38GN3`GA28G$3`.D)*@!``"#X?])P>8%
M3`'B28G,23'T23'L20'43(GJ2,'J&TP)\D$!U$B)ZDC!Y1Y(P>H"38GF2`GJ
M2(G%2`.L)+`!``"#XO])P>8%2`'N2(G52#'-3#'M2`'U3(GF2,'N&TP)]@'U
M3(GN2<'E'DC![@))B>Y,">Y)B<5,`ZPDN`$``(/F_TP!Z4F)]4DQU4TQY4G!
MY@5)`<U(B>E(P>D;3`GQ00'-3(GA2<'D'DC!Z0)-B>Y,">%)B<1,`Z0DP`$`
M`(/A_TG!Y@5,`>))B<Q),?1),>Q)`=1,B>I(P>H;3`GR00'42(GJ2,'E'DC!
MZ@)-B>9(">I(B<5(`ZPDR`$``(/B_TG!Y@5(`>Y(B=5(,<U,,>U(`?5,B>9(
MP>X;3`GV`?5,B>Y(P>X"2<'E'DF)[DP)[DF)Q4P#K"30`0``@^;_2<'F!4P!
MZ4F)]4DQU4TQY4D!S4B)Z4C!Z1M,"?%!`<U,B>%)P>0>2,'I`DV)[DP)X4F)
MQ$P#I"38`0``@^'_2<'F!4P!XDF)S$DQ]$DQ[$D!U$R)ZDC!ZAM,"?)!`=1(
MB>I(P>4>2,'J`DV)YD@)ZDB)Q4@#K"3@`0``@^+_2`'N2(G52#'-3#'M2`'U
M3(GF2<'F!4C![AM,"?8!]4R)[DG!Y1Y(P>X"28GN3`GN28G%3`.L).@!``"#
MYO])P>8%3`'I28GU23'533'E20'-2(GI2,'I&TP)\4$!S4R)X4G!Y!Y(P>D"
M38GN3`GA28G$3`.D)/`!``"#X?])P>8%3`'B28G,23'T23'L20'43(GJ2,'J
M&TP)\D$!U$B)ZDC!Y1Y(P>H"38GF2`GJ2(G%2`.L)/@!``"#XO])P>8%2`.$
M)``"``!(`>Y(B=5(,<U(`<%,,>U(`?5,B>9(P>X;3`GV1(TT+DR)[DG!Y1Y(
MP>X"3(GE3`GN38GE2,'M`H/F_TG!Y1Y%`?%(B?!,">U$`=Y(,=!!`>H!VDPQ
MX$V)]$@!P4R)\$G!Y`5(P>@;3`G@`<A!`<!,B0=,B4\(3(E7$$B)=QA(B5<@
M2('$"`(``%M=05Q!74%>05_#9@\?1```05=)B?=!5D%528G]05154TB#[`A(
MA?]T;4R+-;B'(`!-A?9T80^V!SP]='2$P'1P28G\28/$`4$/M@0D/#UT!(3`
M=>]%*>Q)BQXQP$B%VW0W36/D3(GU9@\?A```````3(GB3(GN2(G?Z/+B__^%
MP'4'0H`\(SUT+TB#Q0A(BUT`2(7;==HQP$B#Q`A;74%<05U!7D%?PV8/'X0`
M`````$4QY.NB#Q\`3"GU2HU$(P%(P?T#08DO2(/$"%M=05Q!74%>05_#9BX/
M'X0``````$B#[!@QP$B-="0,Z"#___](@\08PV9F+@\?A```````4TB)^TB#
M[!`/'X0``````$B-="0,,<!(B=_H\?[__TB%P'0N2&-$)`Q(BQ6XAB``2(T$
MPDB+4`A(A=)(B1!TT$B#P`A(BU`(2(722(D0=?#KODB#Q!!;PP\?A```````
M55-(@^P(BSUDAB``Z"_D__^+/5F&(`!(BX!@`@``2(MH".@9Y/__BSU#AB``
M#[:8Q`0``.@'Y/__2(M`"(/C?P^W0"#!X`<)V&:)10*+/1V&(`#HZ./__XL]
M$H8@`$B+@&`"``!(BU@(Z-+C__\QTKY+6D``2(G'Z+/C__^+/>V%(`!(B4,H
MZ+3C__^+/=Z%(`!(BX!@`@``2(M8".B>X___2(N`@`8``(L]P84@`$B)0S#H
MB./__XL]LH4@`$B+@&`"``!(BU@(Z'+C__\QTDB#N/@)````=!F+/9"%(`#H
M6^/__TB+@/@)``!(BY"H````BSUWA2``2(E3..@^X___2(N`8`(``(L]884@
M`$B+0`A(QT!``````.@@X___2(N`8`(``(L]0X4@`$B+0`A(QT`@`````.@"
MX___BSTLA2``2(N`8`(``$B+6`CH[.+__TB+@'`"``!(B4-(2(/$"%M=PV8N
M#Q^$``````!(B6PDX$B)7"382(G]3(ED).A,B6PD\$R)="3X2('LN````$B%
M_W0%@#\`=3-(BYPDD````$B+K"28````3(ND)*````!,BZPDJ````$R+M"2P
M````2('$N````,,/'P#HJ^#__TB%P$B)PW3`#Q\`2(G?Z!CC__](A<!)B<0/
MA)P```!!@'PD$RX/A*@```!-C6PD$TB)[^C#X/__3(GO28G&Z+C@__]%C70&
M`DEC_N@KXO__3(GI28G$2(G'2(GJOFQ80``QP.A3X___OP$```!(B>),B>;H
M(^#__X/X_TR)YW00BT0D&"4`\```/0!```!T<^B&W___3(GGZ%[?__](B=_H
MAN+__TB%P$F)Q`^%;O___V8N#Q^$``````!(B=_HV.#__TB)[^@@X/__Z?C^
M__\/'P!!@'PD%``/A"S___]!@'PD%"X/A4#___]!@'PD%0`/A!3____I+___
M_P\?@`````#HD_[__P\?`.N(9F9F9F8N#Q^$``````!(B5PD\$B);"3X2(G[
M2(/L&(L]:8,@`#'MZ#+A__](C9"8````2(G>OP$```#H7N+__X7`=!J)Z$B+
M7"0(2(ML)!!(@\08PV8/'X0``````(L]*H,@`.CUX/__BX"P````)0#P```]
M`$```'0=BST-@R``Z-C@__^+@+`````E`/```#T`H```=:R^`@```$B)WS'M
MZ*?A__^%P$`/E,7KE69F9F9F9BX/'X0``````$%7,<!!5D%505152(G]4TB)
M\TB#["B`/CT/E,!(`<-(B=_H"=___TB-="0<08G%2(GO,<#HU_K__TB%P$F)
MQ`^$"P$``$B)Q^CCWO__1(GI,=)(.<@/@\T```!$BV0D'`^V10`\/0^$I`$`
M`(3`#X2<`0``2(GHZPL/'X``````A-)T#$B#P`$/MA"`^CUU\$B+#5&"(`!!
M*>U!C7P%`DZ-).'H".#__TB%P$F)!"0/A&,!``"+5"0<2(L%*((@`$B+!-`/
MME4`A-*($'0I@/H]="1(@\4!ZPP/'X``````@/H]=!(/ME4`2(/``4B#Q0&$
MTH@0=>G&`#TQT@\?0``/M@P3B$P0`4B#P@&$R77P,<!(@\0H6UU!7$%=05Y!
M7\,/'X``````#[8$$T&(!!1(@\(!A,!TV`^V!!-!B`042(/"`83`=>#KQF8/
M'T0``$R+/9&!(`!%,>2^$````$F#/P!T)4R)^`\?A```````2(/`"$&#Q`%(
M@S@`=?)!C70D`DAC]DC!Y@.+/:"!(`"%_W0T3(G_Z)3?__](A<!)B<9(B04_
M@2``=&U)8\Q)QT3."`````!$B60D'.F?_O__#Q^``````$B)]\<%6X$@``$`
M``#HUM[__TB%P$F)QG0V26/,3(G^2(G'2(T4S0````!(B4PD".A3W___3(DU
MY(`@`$B+3"0(ZZ,/'T0``$B)Z.EX_O__N/_____I\/[__V9F9F9F+@\?A```
M````2(E<).A,B60D^$B)^TB);"3P2('L*!```$F)]+K_#P``2(GFOW)80`!D
M2(L$)2@```!(B80D"!```#'`Z!_<__^%P`^.?P```(U0_TACZH`\+`!T:TAC
MZ$B)ZKXO````2(GGZ(C=__](A<!T6XL]+8`@`.CXW?__2(GI2(GB2(G>2(G'
MZ/?>__](BX0D"!```&1(,P0E*````'5'2(N<)!`0``!(BZPD&!```$R+I"0@
M$```2('$*!```,,/'T``A=)UE`\?0`"+/=)_(`#HG=W__TR)XDB)WDB)Q^AO
MW/__ZZ;H*-S__P\?A```````2(/L"(L]IG\@`.AQW?__N8%80`"Z<!U``+Z+
M6$``2(G'2(/$".F&W?__9@\?1```2(/L"(L]=G\@`.A!W?__2(G'Z+G___^+
M/6-_(`#H+MW__T&X2UI``$4QR;K@4D``3(G!OJ=80`!(B<=(@\0(Z8S=__]F
M9F8N#Q^$``````!54S';2('L*"```&1(BP0E*````$B)A"08(```,<#H[=K_
M_[E]6$``B<*^O%A``#'`2(GGZ!?>__^%P'A(2(VT)!`0``"Z`!```$B)Y^B>
MVO__A<")Q7@MC7T!2&/_Z*W<__](A<!(B<-T&DAC[4B-M"00$```2(G'2(GJ
MZ"_<___&!"L`2(N4)!@@``!D2#,4)2@```!(B=AU"DB!Q"@@``!;7</H]MK_
M_V8/'T0``.E+____9F8N#Q^$``````!!54F)]4%454B)_;_*6$``4TB![!@0
M``!D2(L$)2@```!(B80D"!```#'`Z%#W__^^+P```$B)[TB)P^C`VO__2(7`
M=#=(B>Z_TUA``#'`Z$S[__](BY0D"!```&1(,Q0E*````$B)Z`^%-0$``$B!
MQ!@0``!;74%<05W#2(GOZ$S:__],B>^^X%A``$&)Q.BLW/__28G%9@\?A```
M````387M=*!!@'T``+C(6$``3`]$Z$B%VW033(GN2(G?Z#W;__^%P`^$E@``
M`$R)[^C]V?__2&/008!\%?\O=1E)C505_P\?1```Q@(`2(/J`8/H`8`Z+W3Q
M08U$!`$]_P\```^//O___TF)Z+GB6$``3(GJO@%:0`!(B><QP.AOW/__BSU)
M?2``Z!3;__](C9"8````2(GFOP$```#H,-G__X7`=1V+/29](`#H\=K__XN`
ML````"4`\```/0"```!T%+[@6$``,?_HT]O__TF)Q>DK____O@$```!(B>?H
MKMO__X7`==M(B>:_TUA``.@-^O__2(GGZ`7:__](B<7IL?[__^@XV?__#Q^$
M```````/M@](B?B$R7092(U7`6:0@/DO2`]$PDB#P@$/MDK_A,EU[?/#9F8N
M#Q^$``````!(A?]32(G[=">`/P!T(NC=V/__2(U4`_](.=-R#.L_D$B#Z@%(
M.=IT=X`Z+W3RZR6_`@```.@VVO__2(7`=`5FQP`N`%O#9BX/'X0``````(`Z
M+W0)2(/J`4@YTW+R2#G:=#T/'T0``$B#Z@%(.=-S!8`Z+W3R2"G:,<!(C4H"
M2('Y`!```'>\2(G>2(/"`;\`PF``Z(?7__^X`,)@`%O#@#HO=`M;O^)80`#I
M$-G__[\"````Z+;9__](B<(QP$B%TG2`9L<"+@!(B=!;PV:02(/L.+_D6$``
M9$B+!"4H````2(E$)"@QP$B+!7@4``!(B00D2(L%=10``$B)1"0(2(L%<10`
M`$B)1"002(L%;10``$B)1"08#[<%:10``&:)1"0@,<#HE?3__S'`O^U80`#H
MB?3__S'`O_580`#H??3__S'`O_Y80`#H<?3__S'`OP590`#H9?3__S'`OQ59
M0`#H6?3__S'`O\I80`#H3?3__S'`OR%90`#H0?3__S'`ORM90`#H-?3__S'`
MOS590`#H*?3__S'`O]-80`#H'?3__S'`OS]90`#H$?3__S'`OTA90`#H!?3_
M_S'`OU-90`#HV?/__TB%P'0/2(G&ORM90``QP.CE]___,<"_9%E``.BY\___
M2(7`=`](B<:_=EE``#'`Z,7W__\QP+^!64``Z)GS__](A<!T/$B)QK_*6$``
M,<#HI??__S'`OJU90`"_!5E``.B4]___2(M$)"AD2#,$)2@```!U6DB#Q#C#
M#Q^``````#'`OY%90`#H3//__TB%P'072(G&OR%90``QP.A8]___Z[%F#Q]$
M``!(C70D$;^B64``N0H```#SIG682(UT)!N_(5E``#'`Z"SW___KA>AEUO__
M#Q]$``!(@^P(BP4&'B``@_C_=`E(@\0(PP\?0``QP+\A64``Z-SR__](B<(Q
MP$B%TG0/#[82A-)T"#'`@/HP#Y7`B07-'2``2(/$",,/'X0``````$B#[`CH
M%];__XG'Z,#5__](A<!T"TB+`$B#Q`C##Q\`,<#K]&9F9BX/'X0``````$%6
M055!5$F)_%53N[E90`!(@^Q`2,<$)+E90`!(QT0D"*]90`!(C6PD"$C'1"00
MO%E``$C'1"08LEE``$C'1"0@MUE``$C'1"0HR5E``$C'1"0P2UI``.LA#Q^`
M`````$B)Q^AHV/__2(7`=%-(BUT`2(/%"(`[`'0Q,<!(B=_H_/'__TB%P$F)
MQDR)YG0%@#@`=<Q(B=\QP.@"]O__2(M=`$B#Q0B`.P!USTB#Q$!;74%<05U!
M7L,/'X0``````$R)Y^@(U?__3(GW28G%Z/W4__])C7P%`NASUO__38GP28G%
M2(G'N>!80`!,B>*^`5I``#'`Z);7__],B>Y(B=\QP.B9]?__ZY4/'X``````
M05=!5D%505154TB)^[_*6$``2('L>($``&1(BP0E*````$B)A"1H@0``,<!(
MQT0D8.!90`!(QT0D:,A80`!(QT0D<$M:0`!(QT0D,'990`!(QT0D.&]90`!(
MQT0D0.590`!(QT0D2(Q90`!(QT0D4.U90`!(QT0D6$M:0`!(QX0D@````-19
M0`!(QX0DB````/%90`!(QX0DD````$M:0`#HRO#__TB%P'0)@#@`#X7,`@``
M,<#H!?[__TB%P$F)Q@^$X0(``$$/MBY`A.UT+.C[UO__3(GR#Q^$``````!(
MBPA(#[[M]@1I"'4#Q@)?2(/"`0^V*D"$[77D0;P!````OW990`#K'`\?@```
M``!)8\1!@\0!2(M\Q#"`/P`/A/L!```QP.A$\/__2(7`2(G%=-Q(B<?HI//_
M_X7`=-!(B>_H2-3__TB%P$F)Q73`3(GOZ&C3__],B?=(B<7H7=/__T2-I`4$
M!```36/D3(GGZ,K4__],B>=(B<7HO]3__TV)\4&X^EE``+GB6$``3(GJ2(G'
M28G$OO]90``QP.C<U?__ONT!``!,B>?H7]+__S'`O[)90`#HL^___TB%P`^$
M.@(``$B)Q^C"T___2(L[2(G&Z!?X__](A<!)B<4/A!L"``!,B>_HT]+__X/X
M`WXV2&/0OPA:0`"Y!0```$F-=!7\\Z9U((/X!`^$KP8``(/H!$B808!\!?\O
M#X2>!@``9@\?1```,<#H0?S__X7`=14Q]DR)[^B3U/__A<!!B<8/A<@!``"^
MK5E``+\A64``,<#H-_/__^CBT?__0;E+6D``08G`N>)80`!,B>*^)EI``$B)
M[S'`,=OH`-7__^L]9@\?1```Z$/1__^#.!%U/X/#`>BFT?__08G908G`N>)8
M0`!,B>*^-%I``$B)[S'`2,<$)$M:0`#HP=3__[[M`0``2(GOZ$31__^#^/]T
MMTR)Y^CGT/__O\I80`!(B>XQP.BH\O__2(GOZ`#\__](BY0D:($``&1(,Q0E
M*````$B)Z`^%I@8``$B!Q'B!``!;74%<05U!7D%?PP\?0`!(BVPD8$&\`0``
M`(!]``!U&>L]#Q]$``!)8\1!@\0!2(MLQ&"`?0``="9(B>_HCO'__X7`=.)(
MB>_H,M+__TB%P$F)Q732Z>7]__\/'T0``$4Q[>G8_?__#Q^$``````!(B<=(
MB40D$.AC^___2(M$)!!(B<?H]M'__TB)Q>E.____9@\?1```O0$```"_U%E`
M`&8/'T0``#'`Z+'M__](A<!T%$B)Q^C$T?__2(7`28G&#X7O_/__2&/%@\4!
M2(N\Q(````"`/P!US4&^V5E``.D&_?__9BX/'X0``````$R+*^G=_?__#Q^$
M``````"Z`@```$C'QN[___^)Q^A]T___2(VT)*````"Z!@```$2)]^B(T?__
M@+PDH`````!U#H"\)*$```!##X3L!```,=(Q]D2)]^A$T___2(V<)-B```!!
MN(FKS>]!N?[<NIA!NO#ATL-(QX0DH(````$C16=,B80DJ(```$R)C"2P@```
M2,>$)+B```!V5#(03(F4),"```!(QX0DR(````````!(QX0DT(````````#'
MA"08@0````````\?@`````!(C;0DH````+H`@```1(GWZ-O0__^%P$&)Q0^.
M@`$``$B+C"3(@```26/%B[PD&($``$R-O"2@````C13!2#G*2(F4),B````/
MDL%(P>@=#[;)2`&,)-"```!(`80DT(```(7_=&I!OT````!(C;0DH````$$I
M_TAC_T4Y_44/3OU(`=])8]=(B50D&.A_T/__BX0D&($``$0!^(/X0(F$)!B!
M```/A57___](C;PDH(```$4I_4R-O"2@````3`-\)!CH2-7__P\?A```````
M08/]/P^.I````$&-1<#!Z`:)PHE$)"1(B=!(B50D*$C!X`9)C40'0$B)1"08
M3(GX2(L02(V\)*"```!(B1-(BU`(2(E3"$B+4!!(B5,02(M0&$B)4QA(BU`@
M2(E3($B+4"A(B5,H2(M0,$B)4S!(BU`X2(/`0$B)4SA(B40D$.B\U/__2(M$
M)!!(.T0D&'6@2(M$)"A(@\`!2,'@!DD!QXM$)"3WV,'@!D6-;`7`26/53(G^
M2(G?Z(3/__]$B:PD&($``.EG_O__#Q^``````$2)]^C8SO__2(N<),B```!,
MBZPDT(```$B)V$C!Z`.#X#^-2`%(8]#&A!38@```@(/Y.`^/4`(``+HW````
M2&/),?8IPDB-O`S8@```2&/2Z%O.__],B>A(C;PDH(```(B<)!>!``!(P>@8
M1(BL)!.!``"(A"00@0``3(GH2,'H$(B$)!&!``!,B>A(P>@(B(0D$H$``$B)
MV$C!Z!B(A"04@0``2(G82,'H$(B$)!6!``!(B=@QVTC!Z`B(A"06@0``Z*G3
M__](BX0DH(```$B)P8B$)%.!``!(P>D8B<J(C"10@0``2(G!2,'I$(B,)%&!
M``!(B<%(BX0DJ(```$C!Z0B(C"12@0``2(G!B(0D5X$``$C!Z1B(C"14@0``
M2(G!2,'I$(B,)%6!``!(B<%(BX0DL(```$C!Z0B(C"16@0``2(G!B(0D6X$`
M`$C!Z1B(C"18@0``2(G!2,'I$(B,)%F!``!(B<%(BX0DN(```$C!Z0B(C"1:
M@0``2(G!B(0D7X$``$C!Z1B(C"1<@0``2(G!2,'I$(B,)%V!``!(B<%(BX0D
MP(```$C!Z0B(C"1>@0``2(G!B(0D8X$``$C!Z1B(C"1@@0``2(G!2,'I$(B,
M)&&!``!(B<%(P>D(B(PD8H$``.L0#Q^$```````/MI0<4($``$B-O%P@@0``
M,<`/MM*^(5I``$B#PP'H_<[__TB#^Q1UUTR-A"0@@0``QH0D2($```!!N4M:
M0`"YXEA``$R)XKX26D``2(GO,<#HRL[__^D6^O__#Q]$``!(BUL(2(7;#X1;
M^?__2(G?Z.O+__](@_@##X9)^?__2(UT`_R_#5I``+D%````\Z9,#T3KZ2_Y
M__](8\FZ/P```#'V2(V\#-B````IPDACTN@+S/__2(V\)*"```#HOM'__TB-
ME"38@```N0<````QP$B)U_-(J^F)_?__@+PDH@```$$/A0;[__^`O"2C````
M0P^%^/K__X"\)*0```!(#X7J^O__@+PDI0```$4/A=SZ__^Z`@```$C'QL;_
M__]$B??H&,[__TB-M"2@````NBD```!$B??H(\S__T&Y2UI``$R-A"2@````
MZ>_^___H&\O__V9F+@\?A```````055!5%532(/L"(L]D&X@`.A;S/__OP59
M0`#&@,4$```!,<#HB.?__TB%P'0)@#@Q#X2*`@``,<#HH_+__XL]76X@`.@H
MS/__N0L```"Z`0```+Y%6D``2(G'Z.'+__](A<`/A+D```!(BT`0,=M,BV`H
MBP6P?B``@_@!#XZ5````O0@```#K-`\?A```````28MU$$B+!85^(`!(BSPH
MZ(S+__^%P'5H@\,!BP5W?B``2(/%"(U3`3G0?EJ+/=YM(`#HJ<O__S')B=I,
MB>9(B<?H:LS__TB%P'0U3(LH0?9%#01UJXL]M6T@`.B`R___3(GNN0(````Q
MTDB)Q^ANRO__2(G&ZXUF#Q^$``````"+!1)^(`"-4/\YV@^$-P$``(L]>6T@
M`.A$R___N00```"Z`0```+Y*6D``2(G'Z/W*__](A<!(B<-T,TB+0!"+/4MM
M(`!(BRCH$\O__[I,6D``2(G'2(GNZ./)__](BT,02(L8]D,.0`^%(@$``(L]
M'&T@`.CGRO__N00```"Z`0```+Y164``2(G'Z*#*__](A<!(B<-T.#'`O]-8
M0`#H_.7__TB%P$F)Q`^$&`$``$B+0Q"+/=9L(`!(BRCHGLK__TR)XDB)QTB)
M[NAPR?__BSVZ;"``Z(7*__^+/:]L(`#&@,4$````Z'/*__^_REA``$C'@$@'
M````````,<#HG.7__TB%P$B)!1)](``/A.````"^[0$``$B)Q^@?R/__@_@1
M=!"%P'0,@_C_#Q\`#X4E`0``2(/$"%M=05Q!7<,/'X``````@_@!#X[`_O__
MNP$```!F+@\?A```````BSTJ;"``@\,!Z/+)__],B>9(B<?HY\?__SD=F7P@
M`'_?Z8O^__]F+@\?A```````BSWZ:R``Z,7)__](B=Y(B<?H&LK__^G#_O__
M#Q]$``"`>`$`#X1S_?__9@\?1```Z6']__\/'P!(BP5!?"``BSV[:R``3(L@
M2(M#$$B+*.A\R?__3(GF2(GOZ`'K___IW/[__TB+/15\(`#H,//__TB)!1E\
M(`!(BT,02(LM_GL@`$R+($'V1"0-!'1\28M$)!!(B>Y(B<?H4\K__XG%2(M#
M$$B+&/9##01T/DB+0Q!(B<)(BP7&>R``B>F_@%I``$B+,#'`Z"7)__^)P4B+
M!:Q[(`!(BQ6U>R``O[!:0`!(BS`QP.@&R?__BST0:R``Z-O(__^Y`@```#'2
M2(G>2(G'Z,G'___KIXL]\6H@`.B\R/__N0(````QTDR)YDB)Q^BJQ___Z6?_
M__\/'T0``%5(B?U32(/L".@R[O__#[802(G#A-)T&TB-0`$/'T``@/HO2`]$
MV$B#P`$/ME#_A-)U[3'`Z*;P__^%P'0=2(7M=!B`?0``=!*^^EE``$B)W^C:
MR?__2#G8=`=(@\0(6UW#2(/$"$B)[UM=Z6#E__](@^P(BSU6:B``Z"'(__^^
M_____TB)Q^A4Q___BST^:B``Z`G(__\QTKX0.T``2(G'2(/$".EVR?__D)"0
MD)"02(T%(0X@`$B%P'0,2(L-%0X@`.DHQ___,<GK]Y"0D)#SPV9F9F9F+@\?
MA```````2(EL)-A,B60DX$B-+8L)(`!,C26$"2``3(EL).A,B70D\$R)?"3X
M2(E<)-!(@^PX3"GE08G]28GV2,']`TF)U^C;Q/__2(7M=!PQVP\?0`!,B?I,
MB?9$B>]!_Q3<2(/#`4@YZW+J2(M<)`A(BVPD$$R+9"083(ML)"!,BW0D*$R+
M?"0P2(/$.,.0D)"0D)"054B)Y5-(@^P(2(L%``D@`$B#^/]T&;LP86``#Q]$
M``!(@^L(_]!(BP-(@_C_=?%(@\0(6UW#D)!(@^P(Z/_*__](@\0(PP```0`"
M`"5S+R5S`"]P<F]C+W-E;&8O97AE`'!E<FQX<VDN8P!$>6YA3&]A9&5R.CIB
M;V]T7T1Y;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`+W!R;V,O)6DO
M)7,`+@!005)?5$5-4`!005)?4%)/1TY!344`.@`O`%!%4DPU3$E"`%!%4DQ,
M24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!5TY%
M1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!005)?05)'0P!005)?
M05)'5E\P`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]435!$25(`4$%2
M7U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@!005)?
M0TQ%04X]`#$`3$E"4$%42`!$64Q$7TQ)0E)!4EE?4$%42`!32$Q)0E]0051(
M`%5315(`4UE35$5-`"]T;7``5$5-4$1)4@!435``55-%4DY!344`<&%R+0`E
M<R5S)7,E<P!P87)L`"YP87(`)7,E<V-A8VAE+25S)7,`)3`R>``E<R5S=&5M
M<"TE=25S`"5S)7-T96UP+25U+25U)7,`05)'5@`8`'!E<FP`+64`+2T``%]?
M14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@```````````E<SH@97AE
M8W5T:6]N(&]F("5S(&9A:6QE9"`M(&%B;W)T:6YG('=I=&@@)6DN"@`E<SH@
M8W)E871I;VX@;V8@<')I=F%T92!T96UP;W)A<GD@<W5B9&ER96-T;W)Y("5S
M(&9A:6QE9"`M(&%B;W)T:6YG('=I=&@@)6DN"@``````````````````````
MP$,!&P,[]````!T```#`P?__$`$``/#%__^8!0``\,C__S@!``"PWO__B`$`
M`(#?___H`0``H-___P`"````X/__(`(``'#A__]0`@``\.+__W@"``"@X___
MF`(``-#E___H`@``P.;__Q`#``#PYO__*`,``$#G__]``P``\.?__W`#````
MZ/__B`,``+#I___(`P``X.G__^`#``"PZO__$`0``(#L__\P!```T.S__U`$
M````[?__<`0``!#N__^X!```T/?__P@%``"@^___2`4``!#\__^`!0``4/S_
M_\`%``!P_/__V`4``(#\___P!0```````!0``````````7I2``%X$`$;#`<(
MD`$``"0````<````J,#__S`$````#A!&#AA*#PMW"(``/QH[*C,D(@````!,
M````1````+#'__^Z%0```$(.$(\"2`X8C@-"#B"-!$(.*(P%00XPA@9!#CB#
M!T<.P`0#F!4..$$.,$$.*$(.($(.&$(.$$(."````````%P```"4````(-W_
M_\8`````0@X0CP)%#AB.`T(.((T$10XHC`5!#C"&!D$..(,'1`Y``G@*#CA!
M#C!!#BA"#B!"#AA"#A!"#@A*"UL..$$.,$$.*$(.($(.&$(.$$(."!0```#T
M````D-W__Q4`````1`X@4`X(`!P````,`0``F-W__U@`````00X0@P)'#B`"
M3@X000X(+````"P!``#8W?__9@$```!!#A"&`D$.&(,#1`X@`UT!#AA!#A!!
M#@@`````````)````%P!```8W___<@$```!-@P:&!58.P`&.`HT#C`1Y"@X(
M1`L``!P```"$`0``<.#__Z$`````388"@P-$#B!U"@X(2@L`3````*0!````
MX?__(@(```!"#A"/`D0.&(X#0@X@C01"#BB,!4$.,(8&1`XX@P='#F`#``$*
M#CA!#C!!#BA"#B!"#AA"#A!"#@A("P`````D````]`$``.#B___H`````$V,
M`H,$3`ZP((8#`J(*#@A%"P``````%````!P"``"HX___*@````!$#A!A#@@`
M%````#0"``#`X___1`````!$#A![#@@`+````$P"``#XX___J@````!!#A"&
M`D$.&(,#20[`0`*7"@X800X000X(00L`````%````'P"``!XY/__!0``````
M````````/````)0"``!PY/__J`$```!"#A"-`D4.&(P#00X@A@1)#BB#!4<.
MP"`"70H.*$$.($$.&$(.$$(."$$+`````!0```#4`@``X.7__R4`````````
M`````"P```#L`@``^.7__\X`````1`X0@P("00H."$L+`D\*#@A!"T8*#@A*
M"UT."````!P````<`P``F.;__\L!````1`Y``VP!"@X(2`L`````'````#P#
M``!(Z/__2`````!$#A!/"@X(10MO#@@````<````7`,``'CH__\D`````$0.
M$%@*#@A$"P```````$0```!\`P``B.C__PD!````0@X0C@)"#AB-`T(.((P$
M1`XHA@5!#C"#!DD.<`*;"@XP00XH00X@0@X80@X00@X(20L``````$P```#$
M`P``4.G__[4)````0@X0CP)"#AB.`T(.((T$0@XHC`5!#C"&!D$..(,'3PZP
M@P(#^`(*#CA!#C!!#BA"#B!"#AA"#A!"#@A%"P``/````!0$``#`\O__RP,`
M``!"#A"-`D(.&(P#00X@A@1!#BB#!40.,`-``@H.*$$.($$.&$(.$$(."$@+
M`````#0```!4!```4/;__W``````00X0A@)$#AB#`T0.(`)6"@X800X000X(
M00M$#AA$#A!!#@@`````%````(P$``"(]O__.@````!$#A!Q#@@`)````*0$
M``!0P/__#@(```!!#A"#`D0.,`./`0H.$$,."$$+`````!0```#,!```B/;_
M_QP``````````````!0```#D!```D/;__P(``````````````"0```#\!```
MB/;__XD`````48P%A@9?#D!&@P>/`HX#C00``````````````````/______
M____``````````#__________P`````````````````````!``````````$`
M`````````0````````#/`0````````$`````````V@$````````!````````
M`.0!`````````0`````````S`@````````$`````````8`,````````,````
M`````+`<0```````#0````````!86$````````0`````````8`)```````#U
M_O]O`````(`%0```````!0````````#X#T````````8`````````@`9`````
M```*`````````-8$````````"P`````````8`````````!4`````````````
M```````#`````````%!C8````````@`````````P!@```````!0`````````
M!P`````````7`````````(`60```````!P`````````(%D````````@`````
M````>``````````)`````````!@`````````_O__;P````"8%4```````/__
M_V\``````@````````#P__]O`````,X40```````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````!8
M86````````````````````````````#F'$```````/8<0```````!AU`````
M```6'4```````"8=0```````-AU```````!&'4```````%8=0```````9AU`
M``````!V'4```````(8=0```````EAU```````"F'4```````+8=0```````
MQAU```````#6'4```````.8=0```````]AU````````&'D```````!8>0```
M````)AY````````V'D```````$8>0```````5AY```````!F'D```````'8>
M0```````AAY```````"6'D```````*8>0```````MAY```````#&'D``````
M`-8>0```````YAY```````#V'D````````8?0```````%A]````````F'T``
M`````#8?0```````1A]```````!6'T```````&8?0```````=A]```````"&
M'T```````)8?0```````IA]```````"V'T```````,8?0```````UA]`````
M``#F'T```````/8?0```````!B!````````6($```````"8@0```````-B!`
M``````!&($```````%8@0```````9B!```````!V($```````(8@0```````
MEB!```````"F($```````+8@0```````QB!```````#6($```````.8@0```
M````]B!`````````````````````````````````````````````````````
M````````_____P`````````````````````````````````````C(2]U<W(O
M8FEN+W!E<FP*979A;"`G97AE8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K
M(B1`(GTG"B`@("!I9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE
M;&P*"G!A8VMA9V4@7U]P87)?<&P["@HC("TM+2!4:&ES('-C<FEP="!M=7-T
M(&YO="!U<V4@86YY(&UO9'5L97,@870@8V]M<&EL92!T:6UE("TM+0HC('5S
M92!S=')I8W0["@HC;&EN92`Q-3@*"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M
M92P@0'1M<&9I;&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["B`@
M("!R97%U:7)E($9I;&4Z.E1E;7`["B`@("!R97%U:7)E($9I;&4Z.D)A<V5N
M86UE.PH@("`@<F5Q=6ER92!&:6QE.CI3<&5C.PH@("`@;7D@)'1O<&1I<B`]
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["B`@("!O=71S
M*'%Q>U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E;7`B?2D["B`@("!&:6QE
M.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI
M;FL@?2P@)'!A<E]T96UP*3L*("`@(')M9&ER("1P87)?=&5M<#L*("`@(",@
M1&]N)W0@<F5M;W9E('1O<&1I<B!B96-A=7-E('1H:7,@8V%U<V5S(&$@<F%C
M92!W:71H(&]T:&5R(&%P<',*("`@(",@=&AA="!A<F4@=')Y:6YG('1O('-T
M87)T+@H*("`@(&EF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R
M)RD@>PH@("`@("`@(",@4V]M971H:6YG('=E;G0@=W)O;F<@=6YL:6YK:6YG
M('1H92!T96UP;W)A<GD@9&ER96-T;W)Y+B`@5&AI<PH@("`@("`@(",@='EP
M:6-A;&QY(&AA<'!E;G,@;VX@<&QA=&9O<FUS('1H870@9&ES86QL;W<@=6YL
M:6YK:6YG('-H87)E9`H@("`@("`@(",@;&EB<F%R:65S(&%N9"!E>&5C=71A
M8FQE<R!T:&%T(&%R92!I;B!U<V4N(%5N;&EN:R!W:71H(&$@8F%C:V=R;W5N
M9`H@("`@("`@(",@<VAE;&P@8V]M;6%N9"!S;R!T:&4@9FEL97,@87)E(&YO
M(&QO;F=E<B!I;B!U<V4@8GD@=&AI<R!P<F]C97-S+@H@("`@("`@(",@1&]N
M)W0@9&\@86YY=&AI;F<@;VX@5VEN9&]W<R!B96-A=7-E(&]U<B!P87)E;G0@
M<')O8V5S<R!W:6QL"B`@("`@("`@(R!T86ME(&-A<F4@;V8@8VQE86YI;F<@
M=&AI;F=S('5P+@H*("`@("`@("!M>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H
M"B`@("`@("`@("`@(%1%35!,051%(#T^("=T;7!86%A86"<L"B`@("`@("`@
M("`@($1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*
M("`@("`@("`@("`@4U5&1DE8(#T^("<N8VUD)RP*("`@("`@("`@("`@54Y,
M24Y+(#T^(#`L"B`@("`@("`@*3L*"B`@("`@("`@<')I;G0@)'1M<"`B(R$O
M8FEN+W-H"G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"B`@(')M("UR
M9B`G)'!A<E]T96UP)PH@("!I9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H
M96X*("`@("`@(&)R96%K"B`@(&9I"B`@('-L965P(#$*("`@>#U@97AP<B!<
M)'@@*R`Q8`ID;VYE"G)M("<B("X@)'1M<"T^9FEL96YA;64@+B`B)PHB.PH@
M("`@("`@("`@("!C:&UO9"`P-S`P+"1T;7`M/F9I;&5N86UE.PH@("`@("`@
M(&UY("1C;60@/2`D=&UP+3YF:6QE;F%M92`N("<@/B]D978O;G5L;"`R/B8Q
M("8G.PH@("`@("`@(&-L;W-E("1T;7`["B`@("`@("`@<WES=&5M*"1C;60I
M.PH@("`@("`@(&]U=',H<7$H4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@
M=&\@<&5R9F]R;2!C;&5A;G5P.B`I"B`@("`@("`@("`@("`N("1T;7`M/F9I
M;&5N86UE*3L*("`@('T*?2!]"@I"14=)3B!["B`@("!);G1E<FYA;',Z.E!!
M4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PH*
M("`@(&5V86P@>PH*7W!A<E]I;FET7V5N=B@I.PH*:68@*&5X:7-T<R`D14Y6
M>U!!4E]!4D=67S!](&%N9"`D14Y6>U!!4E]!4D=67S!]("D@>PH@("`@0$%2
M1U8@/2!M87`@)$5.5GLB4$%27T%21U9?)%\B?2P@*#$@+BX@)$5.5GM005)?
M05)'0WT@+2`Q*3L*("`@("0P(#T@)$5.5GM005)?05)'5E\P?3L*?0IE;'-E
M('L*("`@(&9O<B`H:V5Y<R`E14Y6*2!["B`@("`@("`@9&5L971E("1%3E9[
M)%]](&EF("]>4$%27T%21U9?+SL*("`@('T*?0H*;7D@)'%U:65T(#T@(21%
M3E9[4$%27T1%0E5'?3L*"B,@9FEX("1P<F]G;F%M92!I9B!I;G9O:V5D(&9R
M;VT@4$%42`IM>2`E0V]N9FEG(#T@*`H@("`@<&%T:%]S97`@("`@/3X@*"1>
M3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"B`@("!?97AE("`@("`@("`]
M/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@
M)R<I+`H@("`@7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@
M/R`G7%PG(#H@)R\G*2P**3L*"E]S971?<')O9VYA;64H*3L*7W-E=%]P87)?
M=&5M<"@I.PH*(R!-86=I8R!S=')I;F<@8VAE8VMI;F<@86YD(&5X=')A8W1I
M;F<@8G5N9&QE9"!M;V1U;&5S('M[>PIM>2`H)'-T87)T7W!O<RP@)&1A=&%?
M<&]S*3L*>PH@("`@;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["@H@
M("`@(R!#:&5C:R!F:6QE('1Y<&4L(&=E="!S=&%R="!O9B!D871A('-E8W1I
M;VX@>WM["B`@("!O<&5N(%]&2"P@)SPG+"`D<')O9VYA;64@;W(@;&%S=#L*
M("`@(&)I;FUO9&4H7T9(*3L*"B`@("!M>2`D8G5F.PH@("`@<V5E:R!?1D@L
M("TX+"`R.PH@("`@<F5A9"!?1D@L("1B=68L(#@["B`@("!L87-T('5N;&5S
M<R`D8G5F(&5Q(")<;E!!4BYP;5QN(CL*"B`@("!S965K(%]&2"P@+3$R+"`R
M.PH@("`@<F5A9"!?1D@L("1B=68L(#0["B`@("!S965K(%]&2"P@+3$R("T@
M=6YP86-K*").(BP@)&)U9BDL(#(["B`@("!R96%D(%]&2"P@)&)U9BP@-#L*
M"B`@("`D9&%T85]P;W,@/2`H=&5L;"!?1D@I("T@-#L*("`@(",@?7U]"@H@
M("`@(R!%>'1R86-T:6YG(&5A8V@@9FEL92!I;G1O(&UE;6]R>2![>WL*("`@
M(&UY("5R97%U:7)E7VQI<W0["B`@("!W:&EL92`H)&)U9B!E<2`B1DE,12(I
M('L*("`@("`@("!R96%D(%]&2"P@)&)U9BP@-#L*("`@("`@("!R96%D(%]&
M2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["@H@("`@("`@(&UY("1F=6QL
M;F%M92`]("1B=68["B`@("`@("`@;W5T<RAQ<2A5;G!A8VMI;F<@9FEL92`B
M)&9U;&QN86UE(BXN+BDI.PH@("`@("`@(&UY("1C<F,@/2`H("1F=6QL;F%M
M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["B`@("`@
M("`@;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_
M*"XJ*2A<+BXJ*7PI.PH*("`@("`@("!R96%D(%]&2"P@)&)U9BP@-#L*("`@
M("`@("!R96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["@H@("`@
M("`@(&EF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L
M?&EX?&%L*20O:2D@>PH@("`@("`@("`@("!M>2`H)&]U="P@)&9I;&5N86UE
M*2`](%]T96UP9FEL92@D97AT+"`D8W)C*3L*("`@("`@("`@("`@:68@*"1O
M=70I('L*("`@("`@("`@("`@("`@(&)I;FUO9&4H)&]U="D["B`@("`@("`@
M("`@("`@("!P<FEN="`D;W5T("1B=68["B`@("`@("`@("`@("`@("!C;&]S
M92`D;W5T.PH@("`@("`@("`@("`@("`@8VAM;V0@,#<U-2P@)&9I;&5N86UE
M.PH@("`@("`@("`@("!]"B`@("`@("`@("`@("1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*("`@("`@("`@("`@)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PH@
M("`@("`@('T*("`@("`@("!E;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H
M;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["B`@("`@("`@
M("`@(",@<VAO=6QD(&)E(&UO=F5D('1O(%]T96UP9FEL92@I"B`@("`@("`@
M("`@(&UY("1F:6QE;F%M92`]("(D14Y6>U!!4E]414U0?2\D8F%S96YA;64D
M97AT(CL*("`@("`@("`@("`@;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN(BD[
M"B`@("`@("`@("`@(&]P96X@;7D@)&]U="P@)SXG+"`D9FEL96YA;64@;W(@
M9&EE("0A.PH@("`@("`@("`@("!B:6YM;V1E*"1O=70I.PH@("`@("`@("`@
M("!P<FEN="`D;W5T("1B=68["B`@("`@("`@("`@(&-L;W-E("1O=70["B`@
M("`@("`@?0H@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`D<F5Q=6ER95]L
M:7-T>R1F=6QL;F%M97T@/0H@("`@("`@("`@("`D4$%2.CI(96%V>3HZ36]D
M=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*("`@("`@("`@("`@("`@(&)U9B`]
M/B`D8G5F+`H@("`@("`@("`@("`@("`@8W)C(#T^("1C<F,L"B`@("`@("`@
M("`@("`@("!N86UE(#T^("1F=6QL;F%M92P*("`@("`@("`@("`@?3L*("`@
M("`@("!]"B`@("`@("`@<F5A9"!?1D@L("1B=68L(#0["B`@("!]"B`@("`C
M('U]?0H*("`@(&QO8V%L($!)3D,@/2`H<W5B('L*("`@("`@("!M>2`H)'-E
M;&8L("1M;V1U;&4I(#T@0%\["@H@("`@("`@(')E='5R;B!I9B!R968@)&UO
M9'5L92!O<B`A)&UO9'5L93L*"B`@("`@("`@;7D@)&9I;&5N86UE(#T@9&5L
M971E("1R97%U:7)E7VQI<W1[)&UO9'5L97T@?'P@9&\@>PH@("`@("`@("`@
M("!M>2`D:V5Y.PH@("`@("`@("`@("!F;W)E86-H("AK97ES("5R97%U:7)E
M7VQI<W0I('L*("`@("`@("`@("`@("`@(&YE>'0@=6YL97-S("]<421M;V1U
M;&5<120O.PH@("`@("`@("`@("`@("`@)&ME>2`]("1?.R!L87-T.PH@("`@
M("`@("`@("!]"B`@("`@("`@("`@(&1E;&5T92`D<F5Q=6ER95]L:7-T>R1K
M97E](&EF(&1E9FEN960H)&ME>2D["B`@("`@("`@?2!O<B!R971U<FX["@H@
M("`@("`@("1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D9FEL96YA;64O)&UO
M9'5L92(["@H@("`@("`@(&EF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN
M960H)DE/.CI&:6QE.CIN97<I*2!["B`@("`@("`@("`@(&UY("1F:"`]($E/
M.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@)"$["B`@("`@("`@("`@(&)I
M;FUO9&4H)&9H*3L*("`@("`@("`@("`@<')I;G0@)&9H("1F:6QE;F%M92T^
M>V)U9GT["B`@("`@("`@("`@('-E96LH)&9H+"`P+"`P*3L*("`@("`@("`@
M("`@<F5T=7)N("1F:#L*("`@("`@("!]"B`@("`@("`@96QS92!["B`@("`@
M("`@("`@(&UY("@D;W5T+"`D;F%M92D@/2!?=&5M<&9I;&4H)RYP;2<L("1F
M:6QE;F%M92T^>V-R8WTI.PH@("`@("`@("`@("!I9B`H)&]U="D@>PH@("`@
M("`@("`@("`@("`@8FEN;6]D92@D;W5T*3L*("`@("`@("`@("`@("`@('!R
M:6YT("1O=70@)&9I;&5N86UE+3Y[8G5F?3L*("`@("`@("`@("`@("`@(&-L
M;W-E("1O=70["B`@("`@("`@("`@('T*("`@("`@("`@("`@;W!E;B!M>2`D
M9F@L("<\)RP@)&YA;64@;W(@9&EE("0A.PH@("`@("`@("`@("!B:6YM;V1E
M*"1F:"D["B`@("`@("`@("`@(')E='5R;B`D9F@["B`@("`@("`@?0H*("`@
M("`@("!D:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86YN;W0@9FEN9"`D
M;6]D=6QE(5QN(CL*("`@('TL($!)3D,I.PH*("`@(",@3F]W(&QO860@86QL
M(&)U;F1L960@9FEL97,@>WM["@H@("`@(R!I;FET:6%L:7IE('-H87)E9"!O
M8FIE8W0@<')O8V5S<VEN9PH@("`@<F5Q=6ER92!84TQO861E<CL*("`@(')E
M<75I<F4@4$%2.CI(96%V>3L*("`@(')E<75I<F4@0V%R<#HZ2&5A=GD["B`@
M("!R97%U:7)E($5X<&]R=&5R.CI(96%V>3L*("`@(%!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*"B`@("`C(&YO=R!L970G<R!T<GD@9V5T=&EN
M9R!H96QP97(@;6]D=6QE<R!F<F]M('=I=&AI;@H@("`@<F5Q=6ER92!)3SHZ
M1FEL93L*"B`@("`C(&QO860@<F5S="!O9B!T:&4@9W)O=7`@:6X*("`@('=H
M:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I
M6S!=*2!["B`@("`@("`@(VQO8V%L("1)3D-[)T-W9"YP;2=](#T@7U]&24Q%
M7U\@:68@)%Y/(&YE("=-4U=I;C,R)SL*("`@("`@("!U;FQE<W,@*"1)3D-[
M)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*("`@("`@
M("`@("`@(R!R97%U:7)E(&UO9'5L97,L(&1O(&]T:&5R(&5X96-U=&%B;&4@
M9FEL97,*("`@("`@("`@("`@:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_)"]I
M*2!["B`@("`@("`@("`@("`@("!R97%U:7)E("1F:6QE;F%M93L*("`@("`@
M("`@("`@?0H@("`@("`@("`@("!E;'-E('L*("`@("`@("`@("`@("`@(",@
M4VMI<"!!8W1I=F53=&%T92=S('-I=&5C=7-T;VUI>F4N<&P@9FEL93H*("`@
M("`@("`@("`@("`@(&1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PH@("`@("`@("`@("!]"B`@("`@("`@
M?0H@("`@("`@(&1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["B`@
M("!]"@H@("`@(R!]?7T*"B`@("!L87-T('5N;&5S<R`D8G5F(&5Q(")02UPP
M,#-<,#`T(CL*("`@("1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L*?0HC
M('U]?0H*(R!!<F=U;65N="!P<F]C97-S:6YG('M[>PIM>2!`<&%R7V%R9W,[
M"FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*"F1E
M;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M
M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0H*)'%U:65T(#T@,"!U;FQE<W,@
M)$5.5GM005)?1$5"54=].PHC($1O;B=T('-W86QL;W<@87)G=6UE;G1S(&9O
M<B!C;VUP:6QE9"!E>&5C=71A8FQE<R!W:71H;W5T("TM<&%R+6]P=&EO;G,*
M:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N
M<R<@)B8@<VAI9G0I*2!["B`@("!M>2`E9&ES=%]C;60@/2!Q=R@*("`@("`@
M("!P("`@8FQI8E]T;U]P87(*("`@("`@("!I("`@:6YS=&%L;%]P87(*("`@
M("`@("!U("`@=6YI;G-T86QL7W!A<@H@("`@("`@(',@("!S:6=N7W!A<@H@
M("`@("`@('8@("!V97)I9GE?<&%R"B`@("`I.PH*("`@(",@:68@=&AE(&%P
M<"!I<R!I;G9O:V5D(&%S(")A<'!N86UE("TM<&%R+6]P=&EO;G,@+2UR975S
M92!04D]'4D%-($!04D]'7T%21U8B+`H@("`@(R!U<V4@=&AE(&%P<"!T;R!R
M=6X@=&AE(&=I=F5N('!E<FP@8V]D92!I;G-T96%D(&]F(&%N>71H:6YG(&9R
M;VT@=&AE"B`@("`C(&%P<"!I='-E;&8@*&)U="!S=&EL;"!S970@=7`@=&AE
M(&YO<FUA;"!A<'`@96YV:7)O;FUE;G0@86YD($!)3D,I"B`@("!I9B`H0$%2
M1U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PH@("`@("`@('-H:69T
M($!!4D=6.PH@("`@("`@("1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`
M05)'5CL*("`@('T*("`@(&5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U
M<@H*("`@("`@("!M>2!`861D7W1O7VEN8SL*("`@("`@("!W:&EL92`H0$%2
M1U8I('L*("`@("`@("`@("`@)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I
M=7-4=ETI*"XJ*2\@;W(@;&%S=#L*"B`@("`@("`@("`@(&EF("@D,2!E<2`G
M22<I('L*("`@("`@("`@("`@("`@('!U<V@@0&%D9%]T;U]I;F,L("0R.PH@
M("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<VEF("@D,2!E<2`G32<I('L*
M("`@("`@("`@("`@("`@(&5V86P@(G5S92`D,B(["B`@("`@("`@("`@('T*
M("`@("`@("`@("`@96QS:68@*"0Q(&5Q("=!)RD@>PH@("`@("`@("`@("`@
M("`@=6YS:&EF="!`<&%R7V%R9W,L("0R.PH@("`@("`@("`@("!]"B`@("`@
M("`@("`@(&5L<VEF("@D,2!E<2`G3R<I('L*("`@("`@("`@("`@("`@("1O
M=70@/2`D,CL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-I9B`H)#$@
M97$@)V(G*2!["B`@("`@("`@("`@("`@("`D8G5N9&QE(#T@)W-I=&4G.PH@
M("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<VEF("@D,2!E<2`G0B<I('L*
M("`@("`@("`@("`@("`@("1B=6YD;&4@/2`G86QL)SL*("`@("`@("`@("`@
M?0H@("`@("`@("`@("!E;'-I9B`H)#$@97$@)W$G*2!["B`@("`@("`@("`@
M("`@("`D<75I970@/2`Q.PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&5L
M<VEF("@D,2!E<2`G3"<I('L*("`@("`@("`@("`@("`@(&]P96X@)&QO9V9H
M+"`B/CXB+"`D,B!O<B!D:64@(EA86#H@0V%N;F]T(&]P96X@;&]G.B`D(2([
M"B`@("`@("`@("`@('T*("`@("`@("`@("`@96QS:68@*"0Q(&5Q("=4)RD@
M>PH@("`@("`@("`@("`@("`@)&-A8VAE7VYA;64@/2`D,CL*("`@("`@("`@
M("`@?0H*("`@("`@("`@("`@<VAI9G0H0$%21U8I.PH*("`@("`@("`@("`@
M:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["B`@("`@("`@("`@("`@
M("!D96QE=&4@)$5.5GLG4$%27U1%35`G?3L*("`@("`@("`@("`@("`@(&EN
M:71?:6YC*"D["B`@("`@("`@("`@("`@("!R97%U:7)E(%!!4CHZ1&ES=#L*
M("`@("`@("`@("`@("`@("9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S
M($!!4D=6.PH@("`@("`@("`@("`@("`@)GLB4$%2.CI$:7-T.CHD8VUD(GTH
M)%\I(&9O<B!`05)'5CL*("`@("`@("`@("`@("`@(&5X:70["B`@("`@("`@
M("`@('T*("`@("`@("!]"@H@("`@("`@('5N<VAI9G0@0$E.0RP@0&%D9%]T
M;U]I;F,["B`@("!]"GT*"B,@6%A8("TM(&%D9"`M+7!A<BUD96)U9R!S=7!P
M;W)T(0H*(R!]?7T*"B,@3W5T<'5T(&UO9&4@*"U/*2!H86YD;&EN9R![>WL*
M:68@*"1O=70I('L*("`@('L*("`@("`@("`C;&]C86P@)$E.0WLG0W=D+G!M
M)WT@/2!?7T9)3$5?7R!I9B`D7D\@;F4@)TU35VEN,S(G.PH@("`@("`@(')E
M<75I<F4@24\Z.D9I;&4["B`@("`@("`@<F5Q=6ER92!!<F-H:79E.CI::7`[
M"B`@("!]"@H@("`@;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*("`@(&UY("1Z
M:7`["@H*("`@(&EF("AD969I;F5D("1P87(I('L*("`@("`@("!O<&5N(&UY
M("1F:"P@)SPG+"`D<&%R(&]R(&1I92`B0V%N;F]T(&9I;F0@)R1P87(G.B`D
M(2(["B`@("`@("`@8FEN;6]D92@D9F@I.PH@("`@("`@(&)L97-S*"1F:"P@
M)TE/.CI&:6QE)RD["@H@("`@("`@("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE
M=SL*("`@("`@("`H("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"B`@("`@("`@("`@(&]R
M(&1I92`B4F5A9"`G)'!A<B<@97)R;W(Z("0A(CL*("`@('T*"@H@("`@;7D@
M)65N=B`](&1O('L*("`@("`@("!I9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D
M>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PH@("`@("`@("`@("`D;65T
M82`]?B!S+RXJ7G!A<CHD+R]M<SL*("`@("`@("`@("`@)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B`@("`@("`@("`@("1M971A(#U^("]>("`H6UXZ72LI.B`H
M+BLI)"]M9SL*("`@("`@("!]"B`@("!].PH*("`@(",@3W!E;B!I;G!U="!A
M;F0@;W5T<'5T(&9I;&5S('M[>PH@("`@;&]C86P@)"\@/2!<-#L*"B`@("!I
M9B`H9&5F:6YE9"`D<&%R*2!["B`@("`@("`@;W!E;B!005(L("<\)RP@)'!A
M<B!O<B!D:64@(B0A.B`D<&%R(CL*("`@("`@("!B:6YM;V1E*%!!4BD["B`@
M("`@("`@9&EE("(D<&%R(&ES(&YO="!A(%!!4B!F:6QE(B!U;FQE<W,@/%!!
M4CX@97$@(E!+7#`P,UPP,#0B.PH@("`@?0H*("`@($-R96%T95!A=&@H)&]U
M="D@.PH@("`@"B`@("!M>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`H@("`@("`@
M("1O=70L"B`@("`@("`@24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE
M.CI/7U=23TY,62@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*("`@("`@("`P
M-S<W+`H@("`@*2!O<B!D:64@)"$["B`@("!B:6YM;V1E*"1F:"D["@H@("`@
M)"\@/2`H9&5F:6YE9"`D9&%T85]P;W,I(#\@7"1D871A7W!O<R`Z('5N9&5F
M.PH@("`@<V5E:R!?1D@L(#`L(#`["B`@("!M>2`D;&]A9&5R(#T@<V-A;&%R
M(#Q?1D@^.PH@("`@:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D
M97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["B`@("`@("`@<F5Q=6ER92!005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`["B`@("`@("`@4$%2.CI&:6QT97(Z.E!O9%-T
M<FEP+3YN97<M/F%P<&QY*%PD;&]A9&5R+"`D,"D*("`@('T*("`@(&9O<F5A
M8V@@;7D@)&ME>2`H<V]R="!K97ES("5E;G8I('L*("`@("`@("!M>2`D=F%L
M(#T@)&5N=GLD:V5Y?2!O<B!N97AT.PH@("`@("`@("1V86P@/2!E=F%L("1V
M86P@:68@)'9A;"`]?B`O7ELG(ETO.PH@("`@("`@(&UY("1M86=I8R`](")?
M7T5.5E]005)?(B`N('5C*"1K97DI("X@(E]?(CL*("`@("`@("!M>2`D<V5T
M(#T@(E!!4E\B("X@=6,H)&ME>2D@+B`B/21V86PB.PH@("`@("`@("1L;V%D
M97(@/7X@<WLD;6%G:6,H("LI?7L*("`@("`@("`@("`@)&UA9VEC("X@)'-E
M="`N("@G("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"B`@("`@
M("`@?65G.PH@("`@?0H@("`@)&9H+3YP<FEN="@D;&]A9&5R*3L*("`@("0O
M(#T@=6YD968["B`@("`C('U]?0H*("`@(",@5W)I=&4@8G5N9&QE9"!M;V1U
M;&5S('M[>PH@("`@:68@*"1B=6YD;&4I('L*("`@("`@("!R97%U:7)E(%!!
M4CHZ2&5A=GD["B`@("`@("`@4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E
M<B@I.PH@("`@("`@(&EN:71?:6YC*"D["@H@("`@("`@(')E<75I<F5?;6]D
M=6QE<R@I.PH*("`@("`@("!M>2!`:6YC(#T@<V]R="!["B`@("`@("`@("`@
M(&QE;F=T:"@D8BD@/#T^(&QE;F=T:"@D82D*("`@("`@("!](&=R97`@>PH@
M("`@("`@("`@("`A+T)31%!!3B\*("`@("`@("!](&=R97`@>PH@("`@("`@
M("`@("`H)&)U;F1L92!N92`G<VET92<I(&]R"B`@("`@("`@("`@("@D7R!N
M92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B`@("`@("`@("`@
M("`D7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PH@("`@("`@
M('T@0$E.0SL*"B`@("`@("`@(R!&:6QE(&5X:7-T<R!T97-T(&%D9&5D('1O
M(&9I>"!25"`C-#$W.3`Z"B`@("`@("`@(R!&=6YN>2P@;F]N+65X:7-T:6YG
M(&5N=')Y(&EN(%\\+BXN+F%U=&\O0V]M<')E<W,O4F%W+UIL:6(O875T;W-P
M;&ET+FEX+@H@("`@("`@(",@5&AI<R!I<R!A(&)A;F0M86ED(&9I>"!W:71H
M(&YO(&1E97!E<B!G<F%S<"!O9B!T:&4@:7-S=64N"B`@("`@("`@(R!3;VUE
M8F]D>2!P;&5A<V4@9V\@=&AR;W5G:"!T:&4@<&%I;B!O9B!U;F1E<G-T86YD
M:6YG('=H870G<R!H87!P96YI;F<L"B`@("`@("`@(R!)(&9A:6QE9"X@+2T@
M4W1E9F9E;@H@("`@("`@(&UY("5F:6QE<SL*("`@("`@("`O7E\\*"XK*20O
M(&%N9"`M92`D,2!A;F0@)&9I;&5S>R0Q?2LK(&9O<B!K97ES("4Z.CL*("`@
M("`@("`D9FEL97-[)%]]*RL@9F]R('9A;'5E<R`E24Y#.PH*("`@("`@("!M
M>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L*("`@("`@
M("!M>2`E=W)I='1E;CL*"B`@("`@("`@9F]R96%C:"`H<V]R="!K97ES("5F
M:6QE<RD@>PH@("`@("`@("`@("!M>2`H)&YA;64L("1F:6QE*3L*"B`@("`@
M("`@("`@(&9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PH@("`@("`@("`@("`@
M("`@:68@*"1N86UE(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD7WTI('L*
M("`@("`@("`@("`@("`@("`@("`D9FEL92`]("1?.PH@("`@("`@("`@("`@
M("`@("`@(&QA<W0["B`@("`@("`@("`@("`@("!]"B`@("`@("`@("`@("`@
M("!E;'-I9B`H+UXH7%$D9&ER7$5<+R@N*EM>0V-=*2E<6B]I*2!["B`@("`@
M("`@("`@("`@("`@("`@*"1F:6QE+"`D;F%M92D@/2`H)#$L("0R*3L*("`@
M("`@("`@("`@("`@("`@("!L87-T.PH@("`@("`@("`@("`@("`@?0H@("`@
M("`@("`@("`@("`@96QS:68@*&TA7B]L;V%D97(O6UXO72LO*"XJ6UY#8UTI
M7%HA*2!["B`@("`@("`@("`@("`@("`@("`@:68@*&UY("1R968@/2`D4$%2
M.CI(96%V>3HZ36]D=6QE0V%C:&5[)#%]*2!["B`@("`@("`@("`@("`@("`@
M("`@("`@("@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*("`@("`@("`@
M("`@("`@("`@("`@("`@;&%S=#L*("`@("`@("`@("`@("`@("`@("!]"B`@
M("`@("`@("`@("`@("`@("`@96QS:68@*"UF("(D9&ER+R0Q(BD@>PH@("`@
M("`@("`@("`@("`@("`@("`@("`H)&9I;&4L("1N86UE*2`]("@B)&1I<B\D
M,2(L("0Q*3L*("`@("`@("`@("`@("`@("`@("`@("`@;&%S=#L*("`@("`@
M("`@("`@("`@("`@("!]"B`@("`@("`@("`@("`@("!]"B`@("`@("`@("`@
M('T*"B`@("`@("`@("`@(&YE>'0@=6YL97-S(&1E9FEN960@)&YA;64@86YD
M(&YO="`D=W)I='1E;GLD;F%M97TK*SL*("`@("`@("`@("`@;F5X="!I9B`A
M<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"\["B`@
M("`@("`@("`@(&]U=',H(&IO:6X@(B(L"B`@("`@("`@("`@("`@("!Q<2A0
M86-K:6YG("(I+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4L
M"B`@("`@("`@("`@("`@("!Q<2@B+BXN*0H@("`@("`@("`@("`I.PH*("`@
M("`@("`@("`@;7D@)&-O;G1E;G0["B`@("`@("`@("`@(&EF("AR968H)&9I
M;&4I*2!["B`@("`@("`@("`@("`@("`D8V]N=&5N="`]("1F:6QE+3Y[8G5F
M?3L*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-E('L*("`@("`@("`@
M("`@("`@(&]P96X@1DE,12P@)SPG+"`D9FEL92!O<B!D:64@(D-A;B=T(&]P
M96X@)&9I;&4Z("0A(CL*("`@("`@("`@("`@("`@(&)I;FUO9&4H1DE,12D[
M"B`@("`@("`@("`@("`@("`D8V]N=&5N="`](#Q&24Q%/CL*("`@("`@("`@
M("`@("`@(&-L;W-E($9)3$4["@H@("`@("`@("`@("`@("`@4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4I"B`@
M("`@("`@("`@("`@("`@("`@:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@
M)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PH*("`@("`@("`@("`@("`@
M(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT
M96YT+"`D9FEL92P@)&YA;64I.PH@("`@("`@("`@("!]"@H@("`@("`@("`@
M("!O=71S*'%Q*%=R:71T96X@87,@(B1N86UE(BDI.PH@("`@("`@("`@("`D
M9F@M/G!R:6YT*")&24Q%(BD["B`@("`@("`@("`@("1F:"T^<')I;G0H<&%C
M:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2D["B`@("`@("`@("`@("1F:"T^
M<')I;G0H<W!R:6YT9B@*("`@("`@("`@("`@("`@("(E,#AX+R5S(BP@07)C
M:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M90H@("`@
M("`@("`@("`I*3L*("`@("`@("`@("`@)&9H+3YP<FEN="AP86-K*"=.)RP@
M;&5N9W1H*"1C;VYT96YT*2DI.PH@("`@("`@("`@("`D9F@M/G!R:6YT*"1C
M;VYT96YT*3L*("`@("`@("!]"B`@("!]"B`@("`C('U]?0H*("`@(",@3F]W
M('=R:71E(&]U="!T:&4@4$%2(&%N9"!M86=I8R!S=')I;F=S('M[>PH@("`@
M)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2!I9B`D>FEP.PH*("`@("1C
M86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P+"`T,#L*("`@(&EF
M("@A)&-A8VAE7VYA;64@86YD(&UY("1M=&EM92`]("AS=&%T*"1O=70I*5LY
M72D@>PH@("`@("`@(&UY("1C='@@/2!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z
M.E-(03L@1&EG97-T.CI32$$M/FYE=R@Q*2!]"B`@("`@("`@("`@('Q\(&5V
M86P@>R!R97%U:7)E($1I9V5S=#HZ4TA!,3L@1&EG97-T.CI32$$Q+3YN97<@
M?0H@("`@("`@("`@("!\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.DU$-3L@
M1&EG97-T.CI-1#4M/FYE=R!].PH*("`@("`@("`C(%=O<FMA<F]U;F0@9F]R
M(&)U9R!I;B!$:6=E<W0Z.E-(02`U+C,X(&%N9"`U+C,Y"B`@("`@("`@;7D@
M)'-H85]V97)S:6]N(#T@979A;"![("1$:6=E<W0Z.E-(03HZ5D524TE/3B!]
M('Q\(#`["B`@("`@("`@:68@*"1S:&%?=F5R<VEO;B!E<2`G-2XS."<@;W(@
M)'-H85]V97)S:6]N(&5Q("<U+C,Y)RD@>PH@("`@("`@("`@("`D8W1X+3YA
M9&1F:6QE*"1O=70L(")B(BD@:68@*"1C='@I.PH@("`@("`@('T*("`@("`@
M("!E;'-E('L*("`@("`@("`@("`@:68@*"1C='@@86YD(&]P96XH;7D@)&9H
M+"`B/"1O=70B*2D@>PH@("`@("`@("`@("`@("`@8FEN;6]D92@D9F@I.PH@
M("`@("`@("`@("`@("`@)&-T>"T^861D9FEL92@D9F@I.PH@("`@("`@("`@
M("`@("`@8VQO<V4H)&9H*3L*("`@("`@("`@("`@?0H@("`@("`@('T*"B`@
M("`@("`@)&-A8VAE7VYA;64@/2`D8W1X(#\@)&-T>"T^:&5X9&EG97-T(#H@
M)&UT:6UE.PH@("`@?0H@("`@)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M
M(&QE;F=T:"`D8V%C:&5?;F%M92D["B`@("`D8V%C:&5?;F%M92`N/2`B0T%#
M2$4B.PH@("`@)&9H+3YP<FEN="@D8V%C:&5?;F%M92D["B`@("`D9F@M/G!R
M:6YT*'!A8VLH)TXG+"`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*
M("`@("1F:"T^<')I;G0H(EQN4$%2+G!M7&XB*3L*("`@("1F:"T^8VQO<V4[
M"B`@("!C:&UO9"`P-S4U+"`D;W5T.PH@("`@(R!]?7T*"B`@("!E>&ET.PI]
M"B,@?7U]"@HC(%!R97!A<F4@)'!R;V=N86UE(&EN=&\@4$%2(&9I;&4@8V%C
M:&4@>WM["GL*("`@(&QA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*
M"B`@("!?9FEX7W!R;V=N86UE*"D["@H@("`@(R!.;W<@;&]A9"!T:&4@4$%2
M(&9I;&4@86YD('!U="!I="!I;G1O(%!!4CHZ3&EB0V%C:&4@>WM["B`@("!R
M97%U:7)E(%!!4CL*("`@(%!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H
M*3L*"@H@("`@>PH@("`@("`@("-L;V-A;"`D24Y#>R=#=V0N<&TG?2`](%]?
M1DE,15]?(&EF("1>3R!N92`G35-7:6XS,B<["B`@("`@("`@<F5Q=6ER92!&
M:6QE.CI&:6YD.PH@("`@("`@(')E<75I<F4@07)C:&EV93HZ6FEP.PH@("`@
M?0H@("`@;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PH@("`@;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=SL*("`@("1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I
M+"`G<B<I(&]R(&1I92`B)"$Z("1`(CL*("`@("1Z:7`M/G)E861&<F]M1FEL
M94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/
M2R@I(&]R(&1I92`B)"$Z("1`(CL*"B`@("!P=7-H($!005(Z.DQI8D-A8VAE
M+"`D>FEP.PH@("`@)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`[
M"@H@("`@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*("`@(&]U=',H<7$H
M7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](BDI.PH*("`@(&EF
M("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A
M="!T:&ES('!O:6YT(0H@("`@("`@(&9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z
M:7`M/FUE;6)E<G,@*2!["B`@("`@("`@("`@(&YE>'0@:68@)&UE;6)E<BT^
M:7-$:7)E8W1O<GD["B`@("`@("`@("`@(&UY("1M96UB97)?;F%M92`]("1M
M96UB97(M/F9I;&5.86UE.PH@("`@("`@("`@("!N97AT('5N;&5S<R`D;65M
M8F5R7VYA;64@/7X@;7L*("`@("`@("`@("`@("`@(%X*("`@("`@("`@("`@
M("`@("\_<VAL:6(O"B`@("`@("`@("`@("`@("`H/SHD0V]N9FEG.CI#;VYF
M:6=[=F5R<VEO;GTO*3\*("`@("`@("`@("`@("`@("@_.B1#;VYF:6<Z.D-O
M;F9I9WMA<F-H;F%M97TO*3\*("`@("`@("`@("`@("`@("A;7B]=*RD*("`@
M("`@("`@("`@("`@("0*("`@("`@("`@("`@?7@["B`@("`@("`@("`@(&UY
M("1E>'1R86-T7VYA;64@/2`D,3L*("`@("`@("`@("`@;7D@)&1E<W1?;F%M
M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R
M86-T7VYA;64I.PH@("`@("`@("`@("!I9B`H+68@)&1E<W1?;F%M92`F)B`M
M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PH@("`@("`@
M("`@("`@("`@;W5T<RAQ<2A3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC
M92!I="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92(I*3L*("`@("`@
M("`@("`@?2!E;'-E('L*("`@("`@("`@("`@("`@(&]U=',H<7$H17AT<F%C
M=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(I*3L*("`@("`@
M("`@("`@("`@("1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N
M86UE*3L*("`@("`@("`@("`@("`@(&-H;6]D*#`U-34L("1D97-T7VYA;64I
M(&EF("1>3R!E<2`B:'!U>"(["B`@("`@("`@("`@('T*("`@("`@("!]"B`@
M("!]"B`@("`C('U]?0I]"B,@?7U]"@HC($EF('1H97)E)W,@;F\@;6%I;BYP
M;"!T;R!R=6XL('-H;W<@=7-A9V4@>WM["G5N;&5S<R`H)%!!4CHZ3&EB0V%C
M:&5[)'!R;V=N86UE?2D@>PH@("`@9&EE(#P\("(N(B!U;FQE<W,@0$%21U8[
M"E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE
M(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*("`@("`@("0P(%L@+4)\
M+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*("`@("1%3E9[4$%27U!23T=.
M04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"B,@?7U]
M"@IS=6(@0W)E871E4&%T:"!["B`@("!M>2`H)&YA;64I(#T@0%\["B`@("`*
M("`@(')E<75I<F4@1FEL93HZ0F%S96YA;64["B`@("!M>2`H)&)A<V5N86UE
M+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N
M86UE+"`H)UPN+BHG*2D["B`@("`*("`@(')E<75I<F4@1FEL93HZ4&%T:#L*
M("`@(`H@("`@1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@
M)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0H*<W5B(')E<75I
M<F5?;6]D=6QE<R!["B`@("`C;&]C86P@)$E.0WLG0W=D+G!M)WT@/2!?7T9)
M3$5?7R!I9B`D7D\@;F4@)TU35VEN,S(G.PH*("`@(')E<75I<F4@;&EB.PH@
M("`@<F5Q=6ER92!$>6YA3&]A9&5R.PH@("`@<F5Q=6ER92!I;G1E9V5R.PH@
M("`@<F5Q=6ER92!S=')I8W0["B`@("!R97%U:7)E('=A<FYI;F=S.PH@("`@
M<F5Q=6ER92!V87)S.PH@("`@<F5Q=6ER92!#87)P.PH@("`@<F5Q=6ER92!#
M87)P.CI(96%V>3L*("`@(')E<75I<F4@17)R;F\["B`@("!R97%U:7)E($5X
M<&]R=&5R.CI(96%V>3L*("`@(')E<75I<F4@17AP;W)T97(["B`@("!R97%U
M:7)E($9C;G1L.PH@("`@<F5Q=6ER92!&:6QE.CI496UP.PH@("`@<F5Q=6ER
M92!&:6QE.CI3<&5C.PH@("`@<F5Q=6ER92!84TQO861E<CL*("`@(')E<75I
M<F4@0V]N9FEG.PH@("`@<F5Q=6ER92!)3SHZ2&%N9&QE.PH@("`@<F5Q=6ER
M92!)3SHZ1FEL93L*("`@(')E<75I<F4@0V]M<')E<W,Z.EIL:6(["B`@("!R
M97%U:7)E($%R8VAI=F4Z.EII<#L*("`@(')E<75I<F4@4$%2.PH@("`@<F5Q
M=6ER92!005(Z.DAE879Y.PH@("`@<F5Q=6ER92!005(Z.D1I<W0["B`@("!R
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*("`@(')E<75I<F4@4$%2
M.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*("`@(')E<75I<F4@871T<FEB=71E
M<SL*("`@(&5V86P@>R!R97%U:7)E($-W9"!].PH@("`@979A;"![(')E<75I
M<F4@5VEN,S(@?3L*("`@(&5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!]
M.PH@("`@979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T[
M"B`@("!E=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!]
M.PH@("`@979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ
M<V-A;&%R('T["GT*"B,@5&AE($,@=F5R<VEO;B!O9B!T:&ES(&-O9&4@87!P
M96%R<R!I;B!M>6QD<B]M:W1M<&1I<BYC"B,@5&AI<R!C;V1E(&%L<V\@;&EV
M97,@:6X@4$%2.CI3971U<%1E;7`@87,@<V5T7W!A<E]T96UP7V5N=B$*<W5B
M(%]S971?<&%R7W1E;7`@>PH@("`@:68@*&1E9FEN960@)$5.5GM005)?5$5-
M4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PH@("`@("`@("1P
M87)?=&5M<"`]("0Q.PH@("`@("`@(')E='5R;CL*("`@('T*"B`@("!F;W)E
M86-H(&UY("1P871H("@*("`@("`@("`H;6%P("1%3E9[)%]]+"!Q=R@@4$%2
M7U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*("`@("`@("!Q
M=R@@0SI<7%1%35`@+W1M<"`N("D*("`@("D@>PH@("`@("`@(&YE>'0@=6YL
M97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*
M("`@("`@("!M>2`D=7-E<FYA;64["B`@("`@("`@;7D@)'!W=6ED.PH@("`@
M("`@(",@9&]E<R!N;W0@=V]R:R!E=F5R>7=H97)E.@H@("`@("`@(&5V86P@
M>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*"B`@
M("`@("`@:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*("`@
M("`@("`@("`@)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["B`@("`@
M("`@?0H@("`@("`@(&5L<VEF("AD969I;F5D("1P=W5I9"D@>PH@("`@("`@
M("`@("`D=7-E<FYA;64@/2`D<'=U:60["B`@("`@("`@?0H@("`@("`@(&5L
M<V4@>PH@("`@("`@("`@("`D=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\
M?"`D14Y6>U5315)]('Q\("=365-414TG.PH@("`@("`@('T*("`@("`@("`D
M=7-E<FYA;64@/7X@<R]<5R]?+V<["@H@("`@("`@(&UY("1S=&UP9&ER(#T@
M(B1P871H)$-O;F9I9WM?9&5L:6U]<&%R+21U<V5R;F%M92(["B`@("`@("`@
M;6MD:7(@)'-T;7!D:7(L(#`W-34["B`@("`@("`@:68@*"$D14Y6>U!!4E]#
M3$5!3GT@86YD(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2![
M"B`@("`@("`@("`@(&]P96X@*&UY("1F:"P@(CPB+B`D<')O9VYA;64I.PH@
M("`@("`@("`@("!S965K("1F:"P@+3$X+"`R.PH@("`@("`@("`@("!S>7-R
M96%D("1F:"P@;7D@)&)U9BP@-CL*("`@("`@("`@("`@:68@*"1B=68@97$@
M(EPP0T%#2$4B*2!["B`@("`@("`@("`@("`@("!S965K("1F:"P@+34X+"`R
M.PH@("`@("`@("`@("`@("`@<WES<F5A9"`D9F@L("1B=68L(#0Q.PH@("`@
M("`@("`@("`@("`@)&)U9B`]?B!S+UPP+R]G.PH@("`@("`@("`@("`@("`@
M)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`D8G5F.PH@
M("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`@
M("`@;7D@)&-T>"`](&5V86P@>R!R97%U:7)E($1I9V5S=#HZ4TA!.R!$:6=E
M<W0Z.E-(02T^;F5W*#$I('T*("`@("`@("`@("`@("`@("`@("!\?"!E=F%L
M('L@<F5Q=6ER92!$:6=E<W0Z.E-(03$[($1I9V5S=#HZ4TA!,2T^;F5W('T*
M("`@("`@("`@("`@("`@("`@("!\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z
M.DU$-3L@1&EG97-T.CI-1#4M/FYE=R!].PH*("`@("`@("`@("`@("`@(",@
M5V]R:V%R;W5N9"!F;W(@8G5G(&EN($1I9V5S=#HZ4TA!(#4N,S@@86YD(#4N
M,SD*("`@("`@("`@("`@("`@(&UY("1S:&%?=F5R<VEO;B`](&5V86P@>R`D
M1&EG97-T.CI32$$Z.E9%4E-)3TX@?2!\?"`P.PH@("`@("`@("`@("`@("`@
M:68@*"1S:&%?=F5R<VEO;B!E<2`G-2XS."<@;W(@)'-H85]V97)S:6]N(&5Q
M("<U+C,Y)RD@>PH@("`@("`@("`@("`@("`@("`@("1C='@M/F%D9&9I;&4H
M)'!R;V=N86UE+"`B8B(I(&EF("@D8W1X*3L*("`@("`@("`@("`@("`@('T*
M("`@("`@("`@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`@("`@("`@(&EF
M("@D8W1X(&%N9"!O<&5N*&UY("1F:"P@(CPD<')O9VYA;64B*2D@>PH@("`@
M("`@("`@("`@("`@("`@("`@("!B:6YM;V1E*"1F:"D["B`@("`@("`@("`@
M("`@("`@("`@("`@("1C='@M/F%D9&9I;&4H)&9H*3L*("`@("`@("`@("`@
M("`@("`@("`@("`@8VQO<V4H)&9H*3L*("`@("`@("`@("`@("`@("`@("!]
M"B`@("`@("`@("`@("`@("!]"@H@("`@("`@("`@("`@("`@)'-T;7!D:7(@
M+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`H("1C='@@/R`D8W1X+3YH
M97AD:6=E<W0@.B`D;71I;64@*3L*("`@("`@("`@("`@?0H@("`@("`@("`@
M("!C;&]S92@D9F@I.PH@("`@("`@('T*("`@("`@("!E;'-E('L*("`@("`@
M("`@("`@)$5.5GM005)?0TQ%04Y](#T@,3L*("`@("`@("`@("`@)'-T;7!D
M:7(@+CT@(B1#;VYF:6=[7V1E;&EM?71E;7`M)"0B.PH@("`@("`@('T*"B`@
M("`@("`@)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*("`@("`@("!M:V1I
M<B`D<W1M<&1I<BP@,#<U-3L*("`@("`@("!L87-T.PH@("`@?0H*("`@("1P
M87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]4
M14U0?2`]?B`O*"XK*2\["GT*"G-U8B!?=&5M<&9I;&4@>PH@("`@;7D@*"1E
M>'0L("1C<F,I(#T@0%\["B`@("!M>2`H)&9H+"`D9FEL96YA;64I.PH*("`@
M("1F:6QE;F%M92`]("(D<&%R7W1E;7`O)&-R8R1E>'0B.PH*("`@(&EF("@D
M14Y6>U!!4E]#3$5!3GTI('L*("`@("`@("!U;FQI;FL@)&9I;&5N86UE(&EF
M("UE("1F:6QE;F%M93L*("`@("`@("!P=7-H($!T;7!F:6QE+"`D9FEL96YA
M;64["B`@("!]"B`@("!E;'-E('L*("`@("`@("!R971U<FX@*'5N9&5F+"`D
M9FEL96YA;64I(&EF("@M<B`D9FEL96YA;64I.PH@("`@?0H*("`@(&]P96X@
M)&9H+"`G/B<L("1F:6QE;F%M92!O<B!D:64@)"$["B`@("!B:6YM;V1E*"1F
M:"D["B`@("!R971U<FXH)&9H+"`D9FEL96YA;64I.PI]"@HC('-A;64@8V]D
M92!L:79E<R!I;B!005(Z.E-E='5P4')O9VYA;64Z.G-E=%]P<F]G;F%M90IS
M=6(@7W-E=%]P<F]G;F%M92!["B`@("!I9B`H9&5F:6YE9"`D14Y6>U!!4E]0
M4D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*
M("`@("`@("`D<')O9VYA;64@/2`D,3L*("`@('T*"B`@("`D<')O9VYA;64@
M?'P]("0P.PH*("`@(&EF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R
M;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PH@("`@("`@("1P<F]G
M;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1#
M;VYF:6=[7V1E;&EM?2D@*R`Q*3L*("`@('T*"B`@("!I9B`H(21%3E9[4$%2
M7U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM
M?2D@/CT@,"D@>PH@("`@("`@(&EF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O
M9VYA;64I('L*("`@("`@("`@("`@<F5T=7)N(&EF("US("1F:#L*("`@("`@
M("!]"B`@("`@("`@:68@*"US("(D<')O9VYA;64D0V]N9FEG>U]E>&5](BD@
M>PH@("`@("`@("`@("`D<')O9VYA;64@+CT@)$-O;F9I9WM?97AE?3L*("`@
M("`@("`@("`@<F5T=7)N.PH@("`@("`@('T*("`@('T*"B`@("!F;W)E86-H
M(&UY("1D:7(@*'-P;&ET("]<421#;VYF:6=[<&%T:%]S97!]7$4O+"`D14Y6
M>U!!5$A]*2!["B`@("`@("`@;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-
M4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B`@("`@("`@)&1I<B`]
M?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B`@("`@("`@*"@D<')O9VYA
M;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X
M97TB*2P@;&%S="D*("`@("`@("`@("`@:68@+7,@(B1D:7(D0V]N9FEG>U]D
M96QI;7TD<')O9VYA;64D0V]N9FEG>U]E>&5](CL*("`@("`@("`H*"1P<F]G
M;F%M92`]("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I
M"B`@("`@("`@("`@(&EF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N
M86UE(CL*("`@('T*?0H*<W5B(%]F:7A?<')O9VYA;64@>PH@("`@)#`@/2`D
M<')O9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*("`@(&EF("AI;F1E
M>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P*2!["B`@("`@("`@
M)'!R;V=N86UE(#T@(BXD0V]N9FEG>U]D96QI;7TD<')O9VYA;64B.PH@("`@
M?0H*("`@(",@6%A8("T@:&%C:R!T;R!M86ME(%!71"!W;W)K"B`@("!M>2`D
M<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*
M("`@("`@("`@("`@("`@(#H@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_
M(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*("`@(&-H;VUP*"1P=V0I.PH@
M("`@)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D0V]N9FEG>U]D96QI;7U<
M12DO)'!W9"1#;VYF:6=[7V1E;&EM?2\["@H@("`@)$5.5GM005)?4%)/1TY!
M345](#T@)'!R;V=N86UE.PI]"@IS=6(@7W!A<E]I;FET7V5N=B!["B`@("!I
M9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["B`@("`@("`@
M<F5T=7)N.PH@("`@?2!E;'-E('L*("`@("`@("`D14Y6>U!!4E])3DE424%,
M25I%1'T@/2`R.PH@("`@?0H*("`@(&9O<B`H<7<H(%-005=.140@5$5-4"!#
M3$5!3B!$14)51R!#04-(12!04D]'3D%-12!!4D=#($%21U9?,"`I("D@>PH@
M("`@("`@(&1E;&5T92`D14Y6>R=005)?)RXD7WT["B`@("!]"B`@("!F;W(@
M*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B`@("`@("`@)$5.
M5GLG4$%27R<N)%]](#T@)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I9B!E>&ES
M=',@)$5.5GLG4$%27T=,3T)!3%\G+B1??3L*("`@('T*"B`@("!M>2`D<&%R
M7V-L96%N(#T@(E]?14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@(CL*
M"B`@("!I9B`H)$5.5GM005)?5$5-4'TI('L*("`@("`@("!D96QE=&4@)$5.
M5GM005)?0TQ%04Y].PH@("`@?0H@("`@96QS:68@*"%E>&ES=',@)$5.5GM0
M05)?1TQ/0D%,7T-,14%.?2D@>PH@("`@("`@(&UY("1V86QU92`]('-U8G-T
M<B@D<&%R7V-L96%N+"`Q,B`K(&QE;F=T:"@B0TQ%04XB*2D["B`@("`@("`@
M)$5.5GM005)?0TQ%04Y](#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%.
M/2A<4RLI+SL*("`@('T*?0H*<W5B(&]U=',@>PH@("`@<F5T=7)N(&EF("1Q
M=6EE=#L*("`@(&EF("@D;&]G9F@I('L*("`@("`@("!P<FEN="`D;&]G9F@@
M(D!?7&XB.PH@("`@?0H@("`@96QS92!["B`@("`@("`@<')I;G0@(D!?7&XB
M.PH@("`@?0I]"@IS=6(@:6YI=%]I;F,@>PH@("`@<F5Q=6ER92!#;VYF:6<[
M"B`@("!P=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O
M;F9I9WLD7WTL('%W*`H@("`@("`@(&%R8VAL:6)E>'`@<')I=FQI8F5X<"!S
M:71E87)C:&5X<"!S:71E;&EB97AP"B`@("`@("`@=F5N9&]R87)C:&5X<"!V
M96YD;W)L:6)E>'`*("`@("D["GT*"B,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C(PHC(%1H92!M86EN('!A8VMA9V4@9F]R('-C<FEP="!E>&5C=71I;VX*
M"G!A8VMA9V4@;6%I;CL*"G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF
M4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*"F1I92!Q
M<2AP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M92(Z
M($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;BD*("`@('5N;&5S<R`M92`D
M<')O9VYA;64["@ID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D
M0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*"GT[
M"@HD.CI?7T524D]2(#T@)$`@:68@)$`["GT*"D-/4D4Z.F5X:70H)#$I(&EF
M("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z
M.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*"C$["@HC;&EN92`Q,#$Q"@I?7T5.
M1%]?"@````!'0T,Z("A$96)I86X@-"XV+C$M,RD@-"XV+C$`1T-#.B`H1&5B
M:6%N(#0N-"XV+38I(#0N-"XV```N<VAS=')T86(`+FEN=&5R<``N;F]T92Y!
M0DDM=&%G`"YN;W1E+F=N=2YB=6EL9"UI9``N9VYU+FAA<V@`+F1Y;G-Y;0`N
M9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L82YD>6X`
M+G)E;&$N<&QT`"YI;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+F5H7V9R86UE
M7VAD<@`N96A?9G)A;64`+F-T;W)S`"YD=&]R<P`N:F-R`"YD>6YA;6EC`"YG
M;W0`+F=O="YP;'0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````
M````````````````````````````````````````````````````````````
M````````````````"P````$````"```````````"0`````````(````````<
M`````````````````````0```````````````````!,````'`````@``````
M```<`D```````!P"````````(`````````````````````0`````````````
M```````A````!P````(`````````/`)````````\`@```````"0`````````
M```````````$````````````````````.`````4````"`````````&`"0```
M````8`(````````@`P````````8`````````"``````````$`````````#0`
M``#V__]O`@````````"`!4```````(`%``````````$````````&````````
M``@````````````````````^````"P````(`````````@`9```````"`!@``
M`````'@)````````!P````$````(`````````!@`````````1@````,````"
M`````````/@/0```````^`\```````#6!````````````````````0``````
M`````````````$X```#___]O`@````````#.%$```````,X4````````R@``
M```````&``````````(``````````@````````!;````_O__;P(`````````
MF!5```````"8%0```````'``````````!P````(````(````````````````
M````:@````0````"``````````@60```````"!8```````!X``````````8`
M````````"``````````8`````````'0````$`````@````````"`%D``````
M`(`6````````,`8````````&````#0````@`````````&`````````!^````
M`0````8`````````L!Q```````"P'````````!@````````````````````$
M````````````````````>0````$````&`````````-`<0```````T!P`````
M```P!```````````````````$``````````0`````````(0````!````!@``
M````````(4`````````A````````6#<``````````````````!``````````
M``````````"*`````0````8`````````6%A```````!86`````````X`````
M```````````````$````````````````````D`````$````"`````````&A8
M0```````:%@```````"H`@``````````````````"```````````````````
M`)@````!`````@`````````06T```````!!;````````]```````````````
M``````0```````````````````"F`````0````(`````````"%Q````````(
M7````````"0%```````````````````(````````````````````L`````$`
M```#`````````#!A8```````,&$````````0````````````````````"```
M`````````````````+<````!`````P````````!`86```````$!A````````
M$`````````````````````@```````````````````"^`````0````,`````
M````4&%@``````!080````````@````````````````````(````````````
M````````PP````8````#`````````%AA8```````6&$```````#P`0``````
M``<`````````"``````````0`````````,P````!`````P````````!(8V``
M`````$AC````````"`````````````````````@`````````"`````````#1
M`````0````,`````````4&-@``````!08P```````"@"````````````````
M```(``````````@`````````V@````$````#`````````(!E8```````@&4`
M``````#D6P``````````````````(````````````````````.`````(````
M`P````````"`P6```````&3!````````H!```````````````````"``````
M``````````````#E`````0```#````````````````````!DP0```````#@`
M```````````````````!``````````$``````````0````,`````````````
M````````````G,$```````#N`````````````````````0``````````````
$````````
