package PAR::StrippedPARL::Dynamic;
use 5.006;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut


__DATA__
M?T5,1@(!`0````````````(`/@`!````\"5```````!``````````"C*````
M`````````$``.``(`$``'@`=``8````$````0`````````!``$```````$``
M0```````P`$```````#``0````````@``````````P````0``````@``````
M```"0`````````)````````<`````````!P``````````0`````````!````
M!0````````````````!``````````$```````.##````````X,,`````````
M$`````````$````&````X,,```````#@TT```````.#30```````Z`0`````
M``"#%0`````````0`````````@````8```#@PP```````.#30```````X--`
M`````````@`````````"````````"``````````$````!````!P"````````
M'`)````````<`D```````$0`````````1``````````$`````````%#E=&0$
M````_,(```````#\PD```````/S"0```````Y`````````#D``````````0`
M````````4>5T9`8`````````````````````````````````````````````
M```````````````````````O;&EB-C0O;&0M;&EN=7@M>#@V+38T+G-O+C(`
M!````!`````!````1TY5```````"````!@```!H````$````%`````,```!'
M3E4`_(:7DU-B=]\_&``Y`BD`MR<6/;H`````````````````````````````
M``!'````$@````````````````````````!\````(```````````````````
M``````"7````(`````````````````````````"S````$@``````````````
M``````````"[````$@````````````````````````#-````$@``````````
M``````````````#Q````$@`````````````````````````(`0``$@``````
M```````````````````:`0``$@`````````````````````````G`0``$@``
M```````````````````````T`0``$@````````````````````````!(`0``
M$@````````````````````````!9`0``$@````````````````````````!K
M`0``$@````````````````````````!S`0``$@``````````````````````
M``!]`0``$@````````````````````````"$`0``$@``````````````````
M``````"+`0``$@````````````````````````"0`0``$@``````````````
M``````````"8`0``$@````````````````````````"B`0``$@``````````
M``````````````"I`0``$@````````````````````````"N`0``$@``````
M``````````````````"W`0``$@````````````````````````"]`0``$@``
M``````````````````````#$`0``$@````````````````````````#6`0``
M$@````````````````````````#A`0``$@````````````````````````#H
M`0``$@````````````````````````#O`0``$@``````````````````````
M``#W`0``$@`````````````````````````7`@``$@``````````````````
M```````X`@``$@````````````````````````!?`@``$@``````````````
M``````````!F`@``$@````````````````````````!M`@``$@``````````
M``````````````!V`@``$@````````````````````````!]`@``$@``````
M``````````````````"$`@``$@````````````````````````"-`@``$@``
M``````````````````````"H`@``$@````````````````````````"_`@``
M$@````````````````````````#3`@``$@````````````````````````#:
M`@``$@````````````````````````#A`@``$@``````````````````````
M``#F`@``$@````````````````````````#M`@``$@``````````````````
M``````#S`@``$@````````````````````````#\`@``$@``````````````
M```````````:`P``$@`````````````````````````N`P``$@``````````
M``````````````!#`P``$@````````````````````````!8`P``$@``````
M``````````````````"P`P``$@````````````````````````#(`P``$@``
M``````````````````````#:`P``$@````````````````````````#J`P``
M$@````````````````````````#X`P``$@`````````````````````````*
M!```$@`````````````````````````8!```$@``````````````````````
M```I!```$@`````````````````````````W!```$@``````````````````
M``````!#!```$@````````````````````````!2!```$@``````````````
M``````````"/!```$@````````````````````````"=!```$@``````````
M``````````````"T!```$@````````````````````````#G!```$@``````
M``````````````````#[!```$@`````````````````````````&!0``$@``
M```````````````````````5!0``$@`````````````````````````S!0``
M$@````````````````````````!(!0``$@````````````````````````!J
M!0``$@````````````````````````!U!0``$@``````````````````````
M``"#!0``$@````````````````````````"-!0``$@``````````````````
M``````";!0``$@`````````````````````````'`@``$@```)@B0```````
M``````````#'!0``$0`6`)C80```````"``````````H````$@`.`%!<0```
M````*P(```````"1````$@`/`'A?0`````````````````"C`P``$@`.`#!-
M0```````O0D```````!G!```$@`.`*!;0```````;P````````"S!0``$0`6
M`)#80```````"`````````"+````$@`,`-`@0``````````````````B`@``
M$@`.`/!60```````IP0```````"N`@``$@`.`.!&0```````!0````````#&
M`@``$@`.`/!&0```````RP$```````!S!```$@`.`!!<0```````0```````
M``#U!0``$`#Q_\C80``````````````````X````$0`0`*!?0```````!```
M``````#_`0``$@`.`(!%0```````+0`````````-`P``$@`.`,!(0```````
M)`````````!*`P``$@`.`*!+0```````2`````````"1`P``$@`.`"!,0```
M````#`$```````"D!0``$0`:`%#I0```````$P`````````!!@``$`#Q_V/I
M0`````````````````!O````$``6`'C80`````````````````#6!```$@``
M`-@D0``````````````````B`P``$@`.`/!(0```````V@````````#;!0``
M$0`:`$#I0```````"`````````#E!0``$0`:`$#I0```````"`````````#N
M!0``$`#Q_\C80``````````````````!````$@`.`*!>0````````@``````
M``##!```$@```,@D0`````````````````!;!0``$0`0`*!B0```````H5L`
M```````1````$@`.`/`E0``````````````````8````$@`.`+!>0```````
MB0`````````M````(``6`'C80`````````````````"K````$0`:`$#I0```
M````"`````````!V`P``$@`.`/!+0```````(@````````!)`@``$@`.``!&
M0```````U@`````````V`P``$@`.`-!)0```````RP$`````````7U]L:6)C
M7V-S=5]F:6YI`%]S=&%R=`!?7VQI8F-?8W-U7VEN:70`;6%I;@!D871A7W-T
M87)T`%])3U]S=&1I;E]U<V5D`%]?;&EB8U]S=&%R=%]M86EN`$=,24)#7S(N
M,BXU`&QI8F,N<V\N-@!?7V1A=&%?<W1A<G0`7U]G;6]N7W-T87)T7U\`7VEN
M:70`7V9I;FD`7TIV7U)E9VES=&5R0VQA<W-E<P!E;G9I<F]N`'-T<FYC;7``
M4&5R;%]'=&AR7VME>5]P='(`<'1H<F5A9%]G971S<&5C:69I8P!L:6)P=&AR
M96%D+G-O+C``4&5R;%])8W5R<W1A8VMI;F9O7W!T<@!097)L7TEI;E]E=F%L
M7W!T<@!097)L7TEO<%]P='(`4&5R;%]N97=35G!V`%!E<FQ?265V86Q?<F]O
M=%]P='(`4&5R;%])<&%R<V5R7W!T<@!097)L7TET;W!?96YV7W!T<@!O<&5N
M9&ER`')E861D:7(V-`!S=')L96X`;6%L;&]C`&5X:70`<W!R:6YT9@!?7WAS
M=&%T-C0`=6YL:6YK`&9R964`8VQO<V5D:7(`<FUD:7(`97AE8W9P`%!E<FQ?
M27-T871B=69?<'1R`%]?;'AS=&%T-C0`;65M8VAR`&%C8V5S<P!R96%L;&]C
M`&UE;6UO=F4`>'-?:6YI=`!B;V]T7T1Y;F%,;V%D97(`4&5R;%]N97=84P!8
M4U]);G1E<FYA;'-?4$%27T)/3U0`4&5R;%]N97=84U]F;&%G<P!P87)?8W5R
M<F5N=%]E>&5C7W!R;V,`9V5T<&ED`'-T<G-T<@!R96%D;&EN:P!G971U:60`
M;65M8W!Y`&=E='!W=6ED`%]?<W1A8VM?8VAK7V9A:6P`1TQ)0D-?,BXT`&UK
M9&ER`'!A<E]C=7)R96YT7V5X96,`;W!E;C8T`'!A<E]F:6YD<')O9P!S=')C
M:'(`<W1R=&]K`')E860`<W1R8VUP`&-L;W-E`%]?<W1R9'5P`%]?97)R;F]?
M;&]C871I;VX`<&%R7V)A<V5N86UE`&QS965K-C0`<&%R7V1I<FYA;64`<W1R
M;F-P>0!P87)?:6YI=%]E;G8`;65M<V5T`'!A<E]E;G9?8VQE86X`7U]R96=I
M<W1E<E]A=&9O<FL`1TQ)0D-?,BXS+C(`9V5T7W5S97)N86UE7V9R;VU?9V5T
M<'=U:60`<&%R7W-E='5P7VQI8G!A=&@`<&%R7VUK=&UP9&ER`%]?8W1Y<&5?
M8E]L;V,`1TQ)0D-?,BXS`%!E<FQ?271A:6YT961?<'1R`%!E<FQ?9W9?9F5T
M8VAP=@!097)L7V%V7V9E=&-H`%!E<FQ?<W9?,G!V7V9L86=S`%!E<FQ?<W9?
M<V5T<'8`4&5R;%])8V]M<&-V7W!T<@!097)L7V%V7W-H:69T`%!E<FQ?;6=?
M<V5T`%!E<FQ?<W9?<V5T<'9N`%!E<FQ?8W)O86M?;F]C;VYT97AT`'!A<E]C
M;&5A;G5P`%A37TEN=&5R;F%L<U]005)?0TQ%05)35$%#2P!097)L7V1O=6YW
M:6YD`%!E<FQ?<V%V95]D97-T<G5C=&]R7W@`4&5R;%]S>7-?:6YI=#,`4&5R
M;%]A=&9O<FM?=6YL;V-K`%!E<FQ?871F;W)K7VQO8VL`4&5R;%]'9&]?=6YD
M=6UP7W!T<@!P97)L7V%L;&]C`'!E<FQ?8V]N<W1R=6-T`%!E<FQ?27!E<FQ?
M9&5S=')U8W1?;&5V96Q?<'1R`%!E<FQ?265X:71?9FQA9W-?<'1R`%!E<FQ?
M<V%F97-Y<VUA;&QO8P!L;V%D7VUY7W!A<E]P;`!P97)L7W!A<G-E`'!E<FQ?
M9&5S=')U8W0`<&5R;%]F<F5E`%!E<FQ?<WES7W1E<FT`<&5R;%]R=6X`4$Q?
M;65M;W)Y7W=R87``;F%M95]L;V%D7VUY7W!A<E]P;`!S:7IE7VQO861?;7E?
M<&%R7W!L`%]?96YV:7)O;@!?96YV:7)O;@!?961A=&$`7U]B<W-?<W1A<G0`
M7V5N9`!L:6)P97)L+G-O+C4N,3(`;&EB9&PN<V\N,@!L:6)M+G-O+C8`;&EB
M8W)Y<'0N<V\N,0``````````80```',````5```````````````7````````
M`&,```!0`````````$````!O````9````"0````<````,`````@`````````
M9P````0````X````*P```'`````O````-P```%8``````````````!\```!R
M````:@``````````````+0```$L```!2````#````&T```!&````90```!T`
M``!F`````````"<`````````:P````````!,````60```````````````@``
M`",````[````6@``````````````;````#8````H````:```````````````
M5`````````!@``````````````!-`````````"````!?````'@```#$```!$
M````#0```````````````0```$(```!8````;@```%L````I````50``````
M``!=`````````````````````P```'$`````````+````$@````Y````8@``
M`&D```!!````````````````````````````````````````````````````
M``````4`````````````````````````````````````````````````````
M``````````H````.````````````````````````````````````````````
M````````````````````````$P```````````````````"8`````````$0``
M```````````````````````+`````````!D`````````````````````````
M"0```````````````````"(`````````(0`````````S````````````````
M```````````````:```````````````&`````````$<````'````/@``````
M````````/0```#\````/`````````$D````Z````10```"X```!#````````
M`%<````\````,@```%,````;````$@```#0```!/````%````!8````8````
M``````````!A````40````````!<````7@```"4```!.````*@`````````U
M````$```````````````2@```!$```!/````!`````@````@$!``!4`$A*H"
M``&`A@&H0B@`($\@!`NXHCA-NH0&"D\```!1````5@```%L```!<````8P``
M`&4```!G````:0```&P`````````;@```'````!Q````<@``````````````
M>O!*:Z$B"R!J?YI\ZM/O#L[>1RUNQ=\?1Y@%QKB-\0[49-1\+L6;MB#_Y`Y]
M08[%V7%8'*Q+X\#"Y3\LXD?&`;(!BV_ZH6Q@\MO3';OCDGQJ"4/6A_F<@6:O
MS&;%N9Q`Q(D%D$-%U>P,%.*>R-6R<>?O7WLR8MOMV\WCG@PZEPNGW:-K?ZF<
MF^&G`@JCX/M)```"```````"````!@````````````````````(``@`"``(`
M`@`"``(``@`"``(``@`"`````@`"``(``@`"```````"``(``@`"``(``@`#
M``(`!@`"``(`!@`"``8``@`&``8``@`"``0`!0``````````````````````
M``````````````````````````````````````````````$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0````$``@`"``$``0````$`
M`0`!``$``@`!``$``0````$`!`!E````$````%````!U&FD)```"`%D````0
M````%&EI#0```P">`@``$````'(9:0D```0`:@,``!`````3:6D-```%`+X#
M`````````0`!`.$````0`````````'4::0D```8`60``````````````X-5`
M```````&`````@``````````````4.E````````%````80``````````````
M0.E````````%````;P```````````````-9````````'`````0``````````
M````"-9````````'````!```````````````$-9````````'````!0``````
M````````&-9````````'````!@``````````````(-9````````'````!P``
M````````````*-9````````'````"```````````````,-9````````'````
M"0``````````````.-9````````'````"@``````````````0-9````````'
M````"P``````````````2-9````````'````#```````````````4-9`````
M```'````#0``````````````6-9````````'````#@``````````````8-9`
M```````'````#P``````````````:-9````````'````$```````````````
M<-9````````'````$0``````````````>-9````````'````$P``````````
M````@-9````````'````%```````````````B-9````````'````%0``````
M````````D-9````````'````%@``````````````F-9````````'````%P``
M````````````H-9````````'````&```````````````J-9````````'````
M&@``````````````L-9````````'````&P``````````````N-9````````'
M````'0``````````````P-9````````'````'@``````````````R-9`````
M```'````'P``````````````T-9````````'````3P``````````````V-9`
M```````'````(@``````````````X-9````````'````)```````````````
MZ-9````````'````)@``````````````\-9````````'````*```````````
M````^-9````````'````*P```````````````-=````````'````+```````
M````````"-=````````'````+@``````````````$-=````````'````,```
M````````````&-=````````'````,P``````````````(-=````````'````
M)0``````````````*-=````````'````)P``````````````,-=````````'
M````(P``````````````.-=````````'````-@``````````````0-=`````
M```'````*0``````````````2-=````````'````*@``````````````4-=`
M```````'````,0``````````````6-=````````'````,@``````````````
M8-=````````'````+0``````````````:-=````````'````+P``````````
M````<-=````````'````-```````````````>-=````````'````-P``````
M````````@-=````````'````.```````````````B-=````````'````.0``
M````````````D-=````````'````.@``````````````F-=````````'````
M.P``````````````H-=````````'````/```````````````J-=````````'
M````/0``````````````L-=````````'````/@``````````````N-=`````
M```'````'```````````````P-=````````'````/P``````````````R-=`
M```````'````&0``````````````T-=````````'````0```````````````
MV-=````````'````00``````````````X-=````````'````0P``````````
M````Z-=````````'````:@``````````````\-=````````'````9```````
M````````^-=````````'````1````````````````-A````````'````10``
M````````````"-A````````'````1@``````````````$-A````````'````
M1P``````````````&-A````````'````2```````````````(-A````````'
M````20``````````````*-A````````'````2@``````````````,-A`````
M```'````2P``````````````.-A````````'````3```````````````0-A`
M```````'````30``````````````2-A````````'````3@``````````````
M4-A````````'````$@``````````````6-A````````'````(```````````
M````8-A````````'````(0``````````````:-A````````'````0@``````
M````````<-A````````'````-0``````````````2(/L".A#!0``Z-(%``#H
M73X``$B#Q`C#_S4"M0``_R4$M0``D)"0D/\E`K4``&@`````Z>#_____)?JT
M``!H`0```.G0_____R7RM```:`(```#IP/____\EZK0``&@#````Z;#_____
M)>*T``!H!````.F@_____R7:M```:`4```#ID/____\ETK0``&@&````Z8#_
M____)<JT``!H!P```.EP_____R7"M```:`@```#I8/____\ENK0``&@)````
MZ5#_____);*T``!H"@```.E`_____R6JM```:`L```#I,/____\EHK0``&@,
M````Z2#_____)9JT``!H#0```.D0_____R62M```:`X```#I`/____\EBK0`
M`&@/````Z?#^____)8*T``!H$````.G@_O___R5ZM```:!$```#IT/[___\E
M<K0``&@2````Z<#^____)6JT``!H$P```.FP_O___R5BM```:!0```#IH/[_
M__\E6K0``&@5````Z9#^____)5*T``!H%@```.F`_O___R5*M```:!<```#I
M</[___\E0K0``&@8````Z6#^____)3JT``!H&0```.E0_O___R4RM```:!H`
M``#I0/[___\E*K0``&@;````Z3#^____)2*T``!H'````.D@_O___R4:M```
M:!T```#I$/[___\E$K0``&@>````Z0#^____)0JT``!H'P```.GP_?___R4"
MM```:"````#IX/W___\E^K,``&@A````Z=#]____)?*S``!H(@```.G`_?__
M_R7JLP``:",```#IL/W___\EXK,``&@D````Z:#]____)=JS``!H)0```.F0
M_?___R72LP``:"8```#I@/W___\ERK,``&@G````Z7#]____)<*S``!H*```
M`.E@_?___R6ZLP``:"D```#I4/W___\ELK,``&@J````Z4#]____):JS``!H
M*P```.DP_?___R6BLP``:"P```#I(/W___\EFK,``&@M````Z1#]____)9*S
M``!H+@```.D`_?___R6*LP``:"\```#I\/S___\E@K,``&@P````Z>#\____
M)7JS``!H,0```.G0_/___R5RLP``:#(```#IP/S___\E:K,``&@S````Z;#\
M____)6*S``!H-````.F@_/___R5:LP``:#4```#ID/S___\E4K,``&@V````
MZ8#\____)4JS``!H-P```.EP_/___R5"LP``:#@```#I8/S___\E.K,``&@Y
M````Z5#\____)3*S``!H.@```.E`_/___R4JLP``:#L```#I,/S___\E(K,`
M`&@\````Z2#\____)1JS``!H/0```.D0_/___R42LP``:#X```#I`/S___\E
M"K,``&@_````Z?#[____)0*S``!H0````.G@^____R7ZL@``:$$```#IT/O_
M__\E\K(``&A"````Z<#[____)>JR``!H0P```.FP^____R7BL@``:$0```#I
MH/O___\EVK(``&A%````Z9#[____)=*R``!H1@```.F`^____R7*L@``:$<`
M``#I</O___\EPK(``&A(````Z6#[____);JR``!H20```.E0^____R6RL@``
M:$H```#I0/O___\EJK(``&A+````Z3#[____):*R``!H3````.D@^____R6:
ML@``:$T```#I$/O___\EDK(``&A.````Z0#[__\``````````#'M28G17DB)
MXDB#Y/!05$G'P*!>0`!(Q\&P7D``2,?'4%Q``.C?^O__])"02(/L"$B+!;FO
M``!(A<!T`O_02(/$",-F9F9F+@\?A```````54B)Y5-(@^P(@#V0L@```'5+
MN[C80`!(BP6*L@``2('KL-A``$C!^P-(@^L!2#G8<R1F#Q]$``!(@\`!2(D%
M9;(``/\4Q;#80`!(BP57L@``2#G8<N+&!4.R```!2(/$"%O)PV9F9BX/'X0`
M`````$B#/0BR````54B)Y702N`````!(A<!T"+_`V$``R?_@R<-F9F9F9BX/
M'X0``````$%7,<E!5D%505154TB-7SA(@>P0`@``2(UT)(@/'T``2(L4"TB)
MT$F)T$C!X`A)P>`808/@_R4``/\`3`G`28G02<'H"$&!X`#_``!,"<!)B=!)
MP>@808'@_P```$P)P$F)T$G!Z"!(B01.3(G`2<'@&$C!X`A!@^#_)0``_P!,
M"<!)B=!(P>HX2<'H*$&!X`#_``!,"<!("=!(B41."$B#P0A(@_E`#X5S____
M2(U6:$B!QF@"```/'X0``````$B+0MA(,P)(,T*H2#-"F$B)P4@!P$C!Z1](
M"<&#X?](B4H82(/""$@Y\G743(L'2(M?($R+3PA,BU<03(M?&$B)VDR)P$@#
M5"2(3(G!2,'H&TC!X05,B=Y(`W0DD$@)R$R)P8/@_TCWT4B-K`*9>8):3(G(
M3(G22/?03"'*3"'13"'82`G03(G*`<5,B<A(P>(>2,'H`DF)[4@)T(/@_TB)
MPDPAPD@)RDB)Z4R-I!:9>8):2(GJ2,'A!4C!ZAM,B=9(`W0DF$@)RDR)P4:-
M)"),B<)(P>D"2,'B'D@)T4B)ZH/A_TCWTDDAS4@APDP)ZDV)Y4B-M!:9>8):
M3(GB2<'E!4C!ZAM,">I-B>6--#)(B>I(P>4>2,'J`D@)ZD@#1"2@3(GE@^+_
M2/?522'52"'-2`-,)*A,">U)B?5(C:PHF7F"6DB)\$G!Y05(P>@;3`GH08GM
M0HTL*$R)X$G!Y!Y(P>@"28GU3`G@28GT@^#_2??422'%22'42`-4)+!-">Q)
MB>U.C:0AF7F"6DB)Z4G!Y05(P>D;3`GI28GM1HTD(4B)\4C!YAY(P>D"2`GQ
M2(GN@^'_2/?622'-2"'&3`GN38GE2(VT,IEY@EI,B>))P>4%2,'J&TP)ZDV)
MY8TT,DB)ZDC!Y1Y(P>H"2`GJ2`-$)+A,B>6#XO](]]5)(=5((<U(`TPDP$P)
M[4F)]4B-K"B9>8):2(GP2<'E!4C!Z!M,">A!B>Y)B?5"C2PP3(G@2<'D'DC!
MZ`),">!)B?2#X/])]]1)(<5)(=1(`U0DR$T)[$F)[4Z-I"&9>8):2(GI2<'E
M!4C!Z1M,">E)B>U&C20A2(GQ2,'F'DC!Z0)("?%(B>Z#X?](]]9)(<U((<9,
M">Y-B>5(C;0RF7F"6DR)XDG!Y05(P>H;3`GJ38GEC30R2(GJ2,'E'DC!Z@)(
M">I(`T0DT$R)Y8/B_TCWU4DAU4@AS4@#3"383`GM28GU2(VL*)EY@EI(B?!)
MP>4%2,'H&TP)Z$&)[TF)]4*-+#A,B>!)P>0>2,'H`DP)X$F)](/@_TGWU$DA
MQ4DAU$@#5"3@30GL28GM3HVD(9EY@EI(B>E)P>4%2,'I&TP)Z4F)[4:-)"%(
MB?%(P>8>2,'I`D@)\4B)[H/A_TCWUDDAS4@AQDP)[DV)Y4B-M#*9>8):3(GB
M2<'E!4C!ZAM,">I-B>6--#)(B>I(P>4>2,'J`D@)ZD@#1"3H3(GE@^+_2/?5
M22'52"'-2`-,)/!,">U)B?5(C:PHF7F"6DB)\$G!Y05(P>@;3`GH08GM0HTL
M*$R)X$G!Y!Y(P>@"28GU3`G@28GT@^#_2??422'%22'42`-4)/A-">Q)B>U.
MC:0AF7F"6DB)Z4G!Y05(P>D;3`GI28GM1HTD(4B)\4C!YAY(P>D"2`GQ2(GN
M@^'_2/?622'-2"'&3`GN38GE2(VT,IEY@EI,B>))P>4%2,'J&TP)ZDV)Y8TT
M,DB)ZDC!Y1Y(P>H"2`GJ2`,$)$R)Y8/B_TCWU4DAU4@AS4@#3"0(3`GM28GU
M2(VL*)EY@EI(B?!)P>4%2,'H&TP)Z$&)[DF)]4*-+#!,B>!)P>0>2,'H`DP)
MX$F)](/@_TGWU$DAQ4DAU$@#5"0030GL28GM3HVD(9EY@EI(B>E)P>4%2,'I
M&TP)Z4F)[4:-)"%(B?%(P>8>2,'I`D@)\4B)[H/A_TCWUDDAS4@AQDP)[DV)
MY4B-M#*9>8):3(GB2<'E!4C!ZAM,">I-B>6--#)(B>I(P>4>2,'J`D@)ZD@#
M1"083(GE@^+_2/?522'52"'-2`-,)"!,">U)B?5(C:PHF7F"6DB)\$G!Y05(
MP>@;3`GH08GO28GU0HTL.$R)X$G!Y!Y(P>@"3`G@28GT@^#_2??422'%22'4
M2`-4)"A-">Q.C:PAF7F"6DB)Z4F)[$G!Y`5(P>D;3`GA1HTL*4B)\4C!YAY(
MP>D"2`GQ@^'_2(G.2#'&2#'N3(VD,J'KV6Y,B>I,B>Y(P>8%2,'J&T@)\D:-
M)")(B>I(P>4>2,'J`D@)ZH/B_T@#1"0P2(G62#'.2`-,)#A,,>Y(C:PPH>O9
M;DR)X$R)YDC!Y@5(P>@;2`GPB>Z-+#!,B>A)P>4>2,'H`DP)Z$F)[8/@_TG!
MY05(B<9(,=9(`U0D0$PQYDB-M#&AZ]EN2(GI2,'I&TP)Z8TT,4R)X4G!Y!Y(
MP>D"3`GA28GU@^'_2<'E!4F)S$DQQ$@#1"1(23'L3HVD(J'KV6Y(B?)(P>H;
M3`GJ1HTD(DB)ZDC!Y1Y(P>H"2`GJ38GE@^+_2(G52#'-2#'U2<'E!4@#3"10
M2(VL**'KV6Y,B>!(P>@;3`GH08GM0HTL*$B)\$C!YAY(P>@"2`GP28GM@^#_
M2<'E!4B)QD@QUD@#5"183#'F2(VT,:'KV6Y(B>E(P>D;3`GIC30Q3(GA2<'D
M'DC!Z0),">%)B?6#X?])P>4%28G,23'$2`-$)&!),>Q.C:0BH>O9;DB)\DC!
MZAM,">I&C20B2(GJ2,'E'DC!Z@)(">I-B>6#XO])P>4%2(G52#'-2#'U2(VL
M**'KV6Y,B>!(P>@;3`GH08GN2`-,)&A"C2PP2(GP2,'F'DC!Z`)("?!)B>V#
MX/])P>4%2(G&2#'62`-4)'!,,>9(C;0QH>O9;DB)Z4C!Z1M,">F--#%,B>%)
MP>0>2,'I`DP)X4F)]8/A_TG!Y05)B<Q),<1(`T0D>$DQ[$Z-I"*AZ]EN2(GR
M2,'J&TP)ZD:-)")(B>I(P>4>2,'J`D@)ZDV)Y8/B_TG!Y05(B=5(,<U(,?5(
MC:PHH>O9;DR)X$C!Z!M,">A!B>]"C2PX2(GP2,'F'DC!Z`)("?!(`XPD@```
M`$F)[8/@_TG!Y05(B<9(,=9(`Y0DB````$PQYDB-M#&AZ]EN2(GI2,'I&TP)
MZ8TT,4R)X4G!Y!Y(P>D"3`GA28GU@^'_2<'E!4F)S$DQQ$@#A"20````23'L
M3HVD(J'KV6Y(B?)(P>H;3`GJ1HTD(DB)ZDC!Y1Y(P>H"2`GJ38GE@^+_2<'E
M!4B)U4@QS4@#C"28````2#'U2(VL**'KV6Y,B>!(P>@;3`GH08GM0HTL*$B)
M\$C!YAY(P>@"2`GP28GM@^#_2(G&2#'62<'E!4@#E"2@````3#'F2(VT,:'K
MV6Y(B>E(P>D;3`GIC30Q3(GA2<'D'DC!Z0),">%)B?6#X?])P>4%28G,23'$
M2`.$)*@```!),>Q.C:0BH>O9;DB)\DC!ZAM,">I&C20B2(GJ2,'E'DC!Z@)(
M">I-B>6#XO])P>4%2(G52#'-2`.,)+````!(,?5(C:PHH>O9;DR)X$C!Z!M,
M">A!B>Y"C2PP2(GP2,'F'DC!Z`)("?"#X/](B<9(,=9,,>9,C:PQH>O9;DB)
MZ4B)[DC!Z1M(P>8%2`.4)+@```!("?%,B>9)P>0>2,'N`D:-+"E,">:#YO](
MB?%(,<%(`X0DP````$@QZ4R-I`JAZ]EN3(GJ3(GI2,'A!4C!ZAM("<I(B>E(
MP>4>2,'I`D:-)")(">F#X?](B<I(,?),,>I(C:P0H>O9;DR)X$R)XDC!X@5(
MP>@;2`G008GO3(GJ0HTL.$G!Y1ZXW+P;CTC!Z@),">I)B<5,`ZPDR````(/B
M_TF)UDDASDF-=#4`28G520G-32'E30GU28GN3HTL+DB)[DG!Y@5(P>X;3`GV
M1HTL+DR)YDG!Y!Y(P>X"3`GF28G$3`.D)-````"#YO])B?9)(=9)C0P,28GT
M20G422'L30GT38GN3HTD(4R)Z4G!Y@5(P>D;3`GQ1HTD(4B)Z4C!Y1Y(P>D"
M2`GI2(G%2`.L)-@```"#X?])B<Y)(?9(C505`$B)S4@)]4PA[4P)]4V)YDB-
M+"I,B>))P>8%2,'J&TP)\D&)[D*-+#),B>I)P>4>2,'J`DP)ZDF)Q4P#K"3@
M````@^+_28G622'.28UT-0!)B=5)"<U-(>5-"?5)B>Y.C2PN2(GN2<'F!4C!
M[AM,"?9&C2PN3(GF2<'D'DC![@),">9)B<1,`Z0DZ````(/F_TF)]DDAUDF-
M#`Q)B?1)"=1)(>Q-"?1-B>Y.C20A3(GI2<'F!4C!Z1M,"?%&C20A2(GI2,'E
M'DC!Z0)(">E(B<5(`ZPD\````(/A_TF)SDDA]DB-5!4`2(G-2`GU3"'M3`GU
M38GF2(TL*DR)XDG!Y@5(P>H;3`GR08GO0HTL.DR)ZDG!Y1Y(P>H"3`GJ28G%
M3`.L)/@```"#XO])B=9)C70U`$F)U4D)S4DASDTAY4T)]4F)[DZ-+"Y(B>Y)
MP>8%2,'N&TP)]D:-+"Y,B>9)P>0>2,'N`DP)YDF)Q$P#I"0``0``@^;_28GV
M22'628T,#$F)]$D)U$DA[$T)]$V)[DZ-)"%,B>E)P>8%2,'I&TP)\4:-)"%(
MB>E(P>4>2,'I`D@)Z4B)Q4@#K"0(`0``@^'_28G.22'V2(U4%0!(B<U("?5,
M(>U,"?5-B>9(C2PJ3(GB2<'F!4C!ZAM,"?)!B>Y"C2PR3(GJ2<'E'DC!Z@),
M">I)B<5,`ZPD$`$``(/B_TF)UDDASDF-=#4`28G520G-32'E30GU28GN3HTL
M+DB)[DG!Y@5(P>X;3`GV1HTL+DR)YDG!Y!Y(P>X"3`GF28G$3`.D)!@!``"#
MYO])B?9)(=9)C0P,28GT20G422'L30GT38GN3HTD(4R)Z4G!Y@5(P>D;3`GQ
M1HTD(4B)Z4C!Y1Y(P>D"2`GI2(G%2`.L)"`!``"#X?])B<Y)(?9(C505`$B)
MS4@)]4PA[4P)]4V)YDB-+"I,B>))P>8%2,'J&TP)\D&)[T*-+#I,B>I)P>4>
M2,'J`DP)ZDF)Q4P#K"0H`0``@^+_28G622'.28UT-0!)B=5)"<U-(>5-"?5)
MB>Y.C2PN2(GN2<'F!4C![AM,"?9&C2PN3(GF2<'D'DC![@),">9)B<1,`Z0D
M,`$``(/F_TF)]DDAUDF-#`Q)B?1)"=1)(>Q-"?1-B>Y.C20A3(GI2<'F!4C!
MZ1M,"?%&C20A2(GI2,'E'DC!Z0)(">E(B<5(`ZPD.`$``(/A_TF)SDDA]DB-
M5!4`2(G-2`GU3"'M3`GU38GF2(TL*DR)XDG!Y@5(P>H;3`GR08GN0HTL,DR)
MZDG!Y1Y(P>H"3`GJ28G%3`.L)$`!``"#XO])B=9)(<Y)C70U`$F)U4D)S4TA
MY4T)]4F)[DZ-+"Y(B>Y)P>8%2,'N&TP)]D:-+"Y,B>9)P>0>2,'N`DP)YDF)
MQ$P#I"1(`0``@^;_28GV22'628T,#$F)]$D)U$DA[$T)]$V)[DZ-)"%,B>E)
MP>8%2,'I&TP)\4:-)"%(B>E(P>4>2,'I`D@)Z4B)Q4@#K"10`0``@^'_28G.
M2(U4%0!(B<U("?5)(?9,(>U,"?5-B>9(C2PJ3(GB2<'F!4C!ZAM,"?)!B>]"
MC2PZ3(GJ2<'E'DC!Z@),">I)B<5,`ZPD6`$``(/B_T@#A"1@`0``28G622'.
M28UT-0!)B=5)"<U(C0P(32'E30GU28GN3HTL+DB)[DG!Y@5(P>X;3`GV1HTL
M+DR)YDG!Y!Y(P>X"3`GF@^;_2(GP28GT2`G022'42"'H3`G@28GL2,'E'DR-
M-`%,B>A,B>E(P>$%2,'H&TG![`)("<A)">Q&C30PN-;!8LI!@^3_2(G!2`.,
M)&@!``!,B?5(P>4%2(T4$4R)X4@Q\4PQZ4B-#`I,B?)(P>H;2`GJ3(GM2<'E
M'D2-/`I(B<)(`Y0D<`$``$C![0),">U,B?F#Y?](P>$%2(TT,DB)ZDPQXDPQ
M\DR-+!9,B?I(P>H;2`G*3(GQ2<'F'D:-+"I(B<)(`Y0D>`$``$C!Z0),"?%,
MB>Z#X?](P>8%3HTD(DB)RD@QZDPQ^DD!U$R)ZDC!ZAM("?)(B<9&C20B3(GZ
M2<'G'DC!Z@),"?I-B>:#XO](`[0D@`$``$G!Y@5(C2PN2(G62#'.3#'N2`'U
M3(GF2,'N&TP)]D&)[D*-+#9,B>Y)P>4>2,'N`DP)[DF)Q4P#K"2(`0``@^;_
M28GN2<'F!4F-3`T`28GU23'533'E3HTL*4B)Z4C!Z1M,"?%&C2PI3(GA2<'D
M'DC!Z0),">%)B<1,`Z0DD`$``(/A_TV)[DG!Y@5)C10428G,23'T23'L3HTD
M(DR)ZDC!ZAM,"?)&C20B2(GJ2,'E'DC!Z@)(">I(B<5(`ZPDF`$``(/B_TV)
MYDB-=#4`2(G52#'-3#'M2<'F!4B-+"Y,B>9(P>X;3`GV08GO0HTL/DR)[DG!
MY1Y(P>X"3`GN28G%3`.L)*`!``"#YO])B>Y)P>8%28U,#0!)B?5),=5-,>5.
MC2PI2(GI2,'I&TP)\4:-+"E,B>%)P>0>2,'I`DP)X4F)Q$P#I"2H`0``@^'_
M38GN2<'F!4F-%!1)B<Q),?1),>Q.C20B3(GJ2,'J&TP)\D:-)")(B>I(P>4>
M2,'J`D@)ZDB)Q4@#K"2P`0``@^+_38GF2<'F!4B-=#4`2(G52#'-3#'M2(TL
M+DR)YDC![AM,"?9!B>Y"C2PV3(GN2<'E'DC![@),">Y)B<5,`ZPDN`$``(/F
M_TF)[DG!Y@5)C4P-`$F)]4DQU4TQY4Z-+"E(B>E(P>D;3`GQ1HTL*4R)X4G!
MY!Y(P>D"3`GA28G$3`.D),`!``"#X?]-B>Y)P>8%28T4%$F)S$DQ]$DQ[$Z-
M)"),B>I(P>H;3`GR1HTD(DB)ZDC!Y1Y(P>H"2`GJ2(G%2`.L),@!``"#XO]-
MB>9)P>8%2(UT-0!(B=5(,<U,,>U(C2PN3(GF2,'N&TP)]D&)[T*-+#Y,B>Y)
MP>4>2,'N`DP)[DF)Q4P#K"30`0``@^;_28GN2<'F!4F-3`T`28GU23'533'E
M3HTL*4B)Z4C!Z1M,"?%&C2PI3(GA2<'D'DC!Z0),">%)B<1,`Z0DV`$``(/A
M_TV)[DG!Y@5)C10428G,23'T23'L3HTD(DR)ZDC!ZAM,"?)&C20B2(GJ2,'E
M'DC!Z@)(">I(B<5(`ZPDX`$``(/B_TV)YDG!Y@5(C70U`$B)U4@QS4PQ[4B-
M+"Y,B>9(P>X;3`GV08GN0HTL-DR)[DG!Y1Y(P>X"3`GN28G%3`.L).@!``"#
MYO])B>Y)C4P-`$F)]4DQU4G!Y@5-,>5.C2PI2(GI2,'I&TP)\4:-+"E,B>%)
MP>0>2,'I`DP)X4F)Q$P#I"3P`0``@^'_38GN2<'F!4F-%!1)B<Q),?1),>Q.
MC20B3(GJ2,'J&TP)\D:-)")(B>I(P>4>2,'J`D@)ZDB)Q4@#K"3X`0``@^+_
M38GF2`.$)``"``!)P>8%2(UT-0!(B=5(,<U(C0P(3#'M2(TL+DR)YDC![AM,
M"?9!B>],B>5&C30^3(GN2<'E'DC![@)(P>T"3`GN38GE1XT,#H/F_TG!Y1Y(
MB?!,">U(,=!$`=4!VDPQX$V)]$0!WD@!P4R)\$G!Y`5(P>@;3`G@`<A&C00`
M3(D'2(EO$$R)3PA(B7<82(E7($B!Q!`"``!;74%<05U!7D%?PP\?1```3(ED
M).!,B7PD^#'`2(E<)-!(B6PDV$F)_$R);"3H3(ET)/!(@^PX2(7_28GW=2E(
MBUPD"$B+;"003(MD)!A,BVPD($R+="0H3(M\)#!(@\0XPV8/'T0``$R+-3&K
M``!-A?9TRP^V!SP]=`2$P'5)13'M28L>,<!(A=MTLTUC[4R)]68/'T0``$R)
MZDR)YDB)W^C"XO__A<!U!T*`/"L]=#=(@\4(2(M=`$B%VW7:,<#I>O___P\?
M`$F)_4F#Q0%!#[9%`#P]=`2$P'7O12GEZZ$/'X``````3"GU2HU$*P%(P?T#
M08DOZ4/___\/'T``2(/L&#'`2(UT)`SH`/___TB#Q!C#9F8N#Q^$``````!5
M2(G]4TB#[!A(C5PD#&:0,<!(B=Y(B>_HT_[__TB%P'0H2&-$)`Q(P>`#2`,%
M3JH``&8/'T0``$B#P`A(BQ!(A=)(B5#X=?#KQDB#Q!A;7<-F9F9F9F8N#Q^$
M``````!5,?]32(/L".CKX?__BSCH].'__TB)Q^C\X?__2(L`,?](BV@(Z,[A
M__^+..C7X?__2(G'Z._A__\Q_P^V&.BUX?__BSB#XW_HN^'__TB)Q^CCX?__
M2(L`,?\/MT`@P>`'"=AFB44"Z(SA__^+..B5X?__2(G'Z)WA__](BP`Q_TB+
M6`CH;^'__XLXZ'CA__\QTKZD7T``2(G'Z*GA__\Q_TB)0RCH3N'__XLXZ%?A
M__](B<?H7^'__TB+`#'_2(M8".@QX?__BSCH.N'__TB)Q^B"X?__2(L`,?](
MB4,PZ!3A__^+..@=X?__2(G'Z"7A__](BP`Q_TB+6`CH]^#__XLXZ`#A__](
MB<?H6.'__S'22(,X`'0@,?_HV>#__XLXZ.+@__](B<?H.N'__TB+`$B+D*@`
M``!(B5,X,?_HM>#__XLXZ+[@__](B<?HQN#__TB+`#'_2(M`"$C'0$``````
MZ)#@__^+..B9X/__2(G'Z*'@__](BP`Q_TB+0`A(QT`@`````.AKX/__BSCH
M=.#__TB)Q^A\X/__2(L`,?](BU@(Z$[@__^+..A7X/__2(G'Z+_@__](BP!(
MB4-(2(/$"%M=PV8/'X0``````$B);"3@2(E<)-A(B?U,B60DZ$R);"3P3(ET
M)/A(@>RX````2(7_=`6`/P!U,TB+G"20````2(NL))@```!,BZ0DH````$R+
MK"2H````3(NT)+````!(@<2X````PP\?`.A3X/__2(7`2(G#=,`/'P!(B=_H
M4.#__TB%P`^$MP```$R-<!,/MA4I'@``#[9`$SG0=10/MA4;'@``03A6`73/
M#Q^```````^V%0@>```YT`^$F0```$B)[^@9X/__3(GW28G$Z`[@__]%C60$
M`DEC_.@1X/__3(GQ2(G'28G$2(GJOJI?0``QP.@)X/__2(GB3(GFOP$```#H
M">#__X/X_W00BT0D&"4`\```/0!```!T;$R)Y^C\W___3(GGZ`3@__](B=_H
MG-___TB%P`^%3/___P\?`$B)W^CXW___2(GOZ`#@___IX/[__P\?``^V!6$=
M``!!.$8!#X56____#[8541T``$$X5@(/A/[^___I0/___V8/'X0``````$R)
MY^AX_O__ZY)F#Q]$``!5,>U32(G[,?](@^P(Z(;>__^+..B/WO__2(G'Z*??
M__](B=Y(B<*_`0```.BGW___A<!T"TB#Q`B)Z%M=PV:0,?_H4=[__XLXZ%K>
M__](B<?H<M___XM`&"4`\```/0!```!T)3'_Z"S>__^+..@UWO__2(G'Z$W?
M__^+0!@E`/```#T`H```=:M(B=^^`@```#'MZ$_?__^%P$`/E,5(@\0(6XGH
M7<,/'X0``````$B)7"302(EL)-@QP$R)9"3@3(EL).A(B?-,B70D\$R)?"3X
M2(/L6(`^/4F)_`^4P$@!PTB)W^A=WO__2(UT)!Q!B<5,B><QP.@S^O__2(7`
M2(G%=%M(B<?H.][__T2)ZD@YT`^"U@$```\?@``````/M@-(@\,!B$4`2(/%
M`83`=>XQP$B+7"0H2(ML)#!,BV0D.$R+;"1`3(MT)$A,BWPD4$B#Q%C#9@\?
MA```````3(LU6:4``#'MOA````!)@SX`="-,B?!F#Q^$``````!(@\`(@\4!
M2(,X`'7SC74"2&/V2,'F`XL%`Y4``(7`#X3[````3(GWZ$O>__])B<=(B04)
MI0``2&/-387_N/____\/A&S___])QT3/"`````");"0<00^V!"2$P`^$%@$`
M`#P]#X0.`0``3(G@ZP60A-)T#$B#P`$/MA"`^CUU\$4IY8GM2,'E`T&-?`4"
M2`,MJZ0``.@^W?__2(G"2(722(E%`+C_____#X0%____BU0D'$B+!8:D``!(
MBQ3000^V!"2$P(@"="8\/70B28/$`>L)#Q]$```\/70300^V!"1(@\(!28/$
M`83`B`)UZ<8"/4B#P@%F+@\?A```````#[8#2(/#`8@"2(/"`83`=>_IG/[_
M_V8N#Q^$``````!(B??'!?.3```!````Z*;<__])B<>X_____TV%_P^$<?[_
M_TACS4R)]DR)_TB-%,T`````2(E,)`CH*MW__TR)/=NC``!(BTPD".G9_O__
MBVPD'.G=_O__3(G@Z0#___](@^P(,?_HC=O__XLXZ);;__^YL%]``+J8(D``
MOKI?0`!(B<=(@\0(Z?O?__\/'P!(@^P(,?_H7=O__XLXZ&;;__](B<?HMO__
M_S'_Z$?;__^+..A0V___0;BD7T``13')NO!60`!,B<&^UE]``$B)QTB#Q`CI
MOM___V8/'T0``$B)7"3H2(EL)/`Q[4R)9"3X2('L."```$B-G"00$```9$B+
M!"4H````2(F$)!@@```QP.APW/__N>M?0`")PK[O7T``,<!(B=_HFMO__X7`
M>3Y(BY0D&"```&1(,Q0E*````$B)Z'5I2(N<)"`@``!(BZPD*"```$R+I"0P
M(```2('$."```,,/'X0``````$B)W[H`$```2(GFZ!C<__^%P(G#>*R->P%(
M8__H)]O__TB%P$B)Q7292&/;2(GF2(G'2(G:Z/[;___&1!T``.N!Z`+<__]F
M+@\?A```````Z1O___]F9BX/'X0``````$%7059!54%428GT54B)_;_[7T``
M4TB![!@0``!D2(L$)2@```!(B80D"!```#'`Z'SW__^^+P```$B)[TB)P^BT
MV___2(7`=#M(B>Z_!&!``#'`Z.C[__](BY0D"!```&1(,Q0E*````$B)Z`^%
M5`$``$B!Q!@0``!;74%<05U!7D%?PTB)[^A,VO__OA%@0`!,B>=!B<?H;-O_
M_TB%P$F)QG2E0;VE7T``08/'`6:008`^`$T/1/5(A=MT$TR)]DB)W^A0V___
MA<`/A*L```!,B??H`-K__TACT$&`?!;_+W4@28U4%O\/'X0``````$B)T<8"
M`(/H`4B#Z@&`>?\O=.U!C00'/?\/```/CS?___])B>BY$V!``$R)\KX58$``
M2(GG,<#HS-G__S'_Z/78__^+..C^V/__2(G'Z!;:__](B>9(B<*_`0```.BV
MV?__A<!U)3'_Z,O8__^+..C4V/__2(G'Z.S9__^+0!@E`/```#T`@```=!TQ
M_[X18$``Z(':__](A<!)B<8/A1W____IL?[__[X!````2(GGZ-/9__^%P'72
M2(GFOP1@0`#HBOK__TB)Y^AJVO__2(G%Z9+^___H'=K__P\?1```#[8/2(GX
MA,ET&$B-5P%FD(#Y+P^V"D@/1,)(@\(!A,EU[O/#9F9F+@\?A```````2(7_
M4TB)^W0G@#\`="+HQ=C__TB-1`/_2#G#<@SK/Y!(@^@!2#G#<S6`."]T\NLE
MOP(```#HKMC__TB%P'0%9L<`+@!;PV8N#Q^$``````"`."]T"4B#Z`%(.<-R
M\D@YV'45@#@O=%!;OQ-@0`#IM=G__P\?1```2(/H`4@YPW,%@#@O=/)(B<(Q
MP$@IVDB-2@)(@?D`$```=ZE(B=Y(@\(!OT#90`#HC-G__[A`V4``6\,/'T0`
M`+\"````Z";8__](B<(QP$B%T@^$=/___V;'`BX`2(G06\-F#Q]$``!(@^PX
MOQQ@0`!D2(L$)2@```!(B40D*#'`2(L%Z!<``$B)!"1(BP7E%P``2(E$)`A(
MBP7A%P``2(E$)!!(BP7=%P``2(E$)!@/MP79%P``9HE$)"`QP.B5]/__,<"_
M)6!``.B)]/__,<"_+6!``.A]]/__,<"_-F!``.AQ]/__,<"_/6!``.AE]/__
M,<"_36!``.A9]/__,<"_^U]``.A-]/__,<"_66!``.A!]/__,<"_8V!``.@U
M]/__,<"_;6!``.@I]/__,<"_!&!``.@=]/__,<"_=V!``.@1]/__,<"_@&!`
M`.@%]/__,<"_BV!``.C9\___2(7`=`](B<:_8V!``#'`Z%7X__\QP+^<8$``
MZ+GS__](A<!T#TB)QK^N8$``,<#H-?C__S'`O[E@0`#HF?/__TB%P'0\2(G&
MO_M?0``QP.@5^/__,<"^Y6!``+\]8$``Z`3X__](BT0D*&1(,P0E*````'5:
M2(/$.,,/'X``````,<"_R6!``.A,\___2(7`=!=(B<:_66!``#'`Z,CW___K
ML68/'T0``$B-="01O]I@0`"Y"@```/.F=9A(C70D&[]98$``,<#HG/?__^N%
MZ#W7__\/'T0``$B#[`B+!=Z,``"#^/]T"4B#Q`C##Q]``#'`OUE@0`#HW/+_
M_TB)PC'`2(72=`\/MA*$TG0(,<"`^C`/E<")!:6,``!(@\0(PP\?A```````
M2(/L".@_U___B<?H2-?__TB)PC'`2(72=`-(BP)(@\0(PV9F9F9F+@\?A```
M````059!54%428G\55.[YV!``$B#[$!(QP0DYV!``$C'1"0(]V!``$B-;"0(
M2,=$)!#_8$``2,=$)!@,84``2,=$)"`184``2,=$)"@C84``2,=$)#"D7T``
MZR0/'X``````3(GF2(G'Z,W6__](A<!T4$B+70!(@\4(@#L`=#$QP$B)W^CY
M\?__2(7`28G&=`6`.`!US$B)WS'`3(GFZ&_V__](BUT`2(/%"(`[`'7/2(/$
M0%M=05Q!74%>PP\?1```3(GGZ.#4__],B?=)B<7HU=3__TF-?`4"Z-O4__]-
MB?!)B<5(B<>Y$6!``$R)XKX58$``,<#HSM3__TR)[DB)WS'`Z`GV___I9/__
M_P\?0`!!5T%6055!5%532(G[O_M?0`!(@>R(@0``9$B+!"4H````2(F$)'B!
M```QP$C'A"20````.F%``$C'A"28````I5]``$C'A"2@````I%]``$C'1"1`
MKF!``$C'1"1(/V%``$C'1"101F%``$C'1"183F%``$C'1"1@4V%``$C'1"1H
MI%]``$C'1"1P+F%``$C'1"1X5V%``$C'A"2`````I%]``.C'\/__2(7`=`F`
M.``/A>D"```QP.@"_O__2(7`28G&#X3^`@``00^V+D"$[70IZ&#5__],B?(/
M'T0``$B+"$@/ONWV!&D(=0/&`E](@\(!#[8J0(3M=>1!O`$```"_KF!``.L<
M#Q^``````$ECQ$&#Q`%(BWS$0(`_``^$"P(``#'`Z$3P__](A<!(B<5TW$B)
MQ^@4]/__A<!TT$B)[^B@U/__2(7`28G%=,!,B>_H0-/__TR)]TB)Q>@UT___
MB>I$C:0"!`0``$UCY$R)Y^@PT___3(GG2(G%2(E$)"CH(-/__TV)\4&X8&%`
M`+D38$``3(GJ2(G'28G$OF5A0``QP.@-T___ONT!``!,B>?HD-3__S'`OPQA
M0`#HK.___TB%P`^$2P(``$B)Q^@3U/__2(L[2(G&Z.#W__](A<!)B<4/A"P"
M``!,B>_HI-+__X/X`WXW2&/0OVYA0`"Y!0```$F-=!7\\Z9U(8/X!`^$X`8`
M`(/H!$B808!\!?\O#X3/!@``#Q^``````#'`Z#G\__^%P'45,?9,B>_H$]3_
M_X7`08G'#X70`0``ON5@0`"_66!``#'`Z)_S___H$M/__T&YI%]``$&)P+D3
M8$``3(GBOHQA0`!(B>\QP#';Z##2___K/68/'T0``.C3T___@S@1=3^#PP'H
MUM+__T&)V4&)P+D38$``3(GBOIIA0`!(B>\QP$C'!"2D7T``Z/'1__^^[0$`
M`$B)[^ATT___@_C_=+=,B>?H!]+__[_[7T``2(GN,<#H$//__TB)[^CX^___
M2(N4)'B!``!D2#,4)2@```!(BT0D*`^%I`8``$B!Q(B!``!;74%<05U!7D%?
MPV8N#Q^$``````!(BZPDD````$&\`0```(!]``!U(>M*9BX/'X0``````$EC
MQ$&#Q`%(BZS$D````(!]``!T*TB)[^CC\?__A<!TWTB)[^AOTO__2(7`28G%
M=,_IROW__V8N#Q^$``````!%,>WIN/W__P\?A```````2(G'2(E$)!CH0_O_
M_TB+1"082(G'Z"[2__](B40D*.DT____#Q]``+T!````ORYA0`!F#Q]$```Q
MP.B1[?__2(7`=!1(B<?H_-'__TB%P$F)Q@^%TOS__TACQ8/%`4B+?,1P@#\`
M==!!OC-A0`#IZ?S__P\?1```3(LKZ<S]__],C:PDL````+H"````2,?&[O__
M_XG'Z#W2__^Z!@```$R)[D2)_^@]TO__@+PDL`````!U#H"\)+$```!##X3K
M!```,=*[B:O-[S'V1(G_Z`32__](B9PDN(```$B-G"2P@```N?[<NIBZ\.'2
MPTC'A"2P@````2-%9TB)C"3`@```2(U#.$C'A"3(@```=E0R$$B)E"30@```
M2,>$)-B`````````2,>$)."`````````QX0D*($```````!(B40D,&8/'T0`
M`+H`@```3(GN1(G_Z)#1__^%P$&)Q@^.C0$``$B+C"38@```26/&B[PD*($`
M`$R)[HT4P4@YRDB)E"38@```#Y+!2,'H'0^VR4@!C"3@@```2`&$)."```"%
M_W1LN4`````I^4AC_T$YSD$/3LY(`WPD,$ACT8E,)"!(B50D..@MT/__BX0D
M*($``(M,)"`!R(/X0(F$)"B!```/A6#___](BW0D.$B)WT$ISDF-=#4`2(ET
M)"#H#M3__TB+="0@9@\?A```````08/^/P^.M@```#'`#Q]``$B+%`9(B=](
MB90DZ(```$B+5`8(2(F4)/"```!(BU0&$$B)E"3X@```2(M4!AA(B90D`($`
M`$B+5`8@2(F4)`B!``!(BU0&*$B)E"00@0``2(M4!C!(B90D&($``$B+5`8X
M2(E$)!A(B70D($B)E"0@@0``Z'?3__](BT0D&$2)\DB+="0@2(/`0"G"@_H_
M#X]K____08/N0$2)\L'J!HG0P>(&2,'@!D$IUDB-=#!`2(M\)#!)8];H&\__
M_T2)M"0H@0``Z5[^__]F#Q]$``!$B?_H`-#__TR+K"38@```3(NT)."```!,
MB>A(P>@#@^`_2&/0@\`!@_@XQH04Z(```(`/CF@"``!,C7LXND`````Q]BG"
M2)A)C3P'2&/2Z,3/__](B=_HQ-+__[D'````,<!,B?_S2*M,B?!(B=]$B+0D
M(X$``$C!Z!A$B*PD)X$``$B-G"0P@0``B(0D(($``$R)\$C!Z!"(A"0A@0``
M3(GP3(VT)&&!``!(P>@(B(0D(H$``$R)Z$C!Z!B(A"0D@0``3(GH2,'H$(B$
M)"6!``!,B>A%,>U(P>@(B(0D)H$``.@XTO__2(N$)+"```!(B<&(A"1C@0``
M2,'I&(G*B(PD8($``$B)P4C!Z1"(C"1A@0``2(G!2(N$)+B```!(P>D(B(PD
M8H$``$B)P8B$)&>!``!(P>D8B(PD9($``$B)P4C!Z1"(C"1E@0``2(G!2(N$
M),"```!(P>D(B(PD9H$``$B)P8B$)&N!``!(P>D8B(PD:($``$B)P4C!Z1"(
MC"1I@0``2(G!2(N$),B```!(P>D(B(PD:H$``$B)P8B$)&^!``!(P>D8B(PD
M;($``$B)P4C!Z1"(C"1M@0``2(G!2(N$)-"```!(P>D(B(PD;H$``$B)P8B$
M)'.!``!(P>D8B(PD<($``$B)P4C!Z1"(C"1Q@0``2(G!2,'I"(B,)'*!``#I
M#`````\?0`!!#[8628/&`4J-/"LQP`^VTKZ'84``28/%`NCYR___28/]*'7;
MQH0D6($```!!N:1?0`!)B=BY$V!``$R)XKYX84``2(GO,<#HR\O__^GG^?__
M9@\?1```2(M;"$B%VP^$*_G__TB)W^B+R___2(/X`P^&&?G__TB-=`/\OW-A
M0`"Y!0```/.F3`]$Z^G_^/__NC@````Q]BG"2)A(C7P#.$ACTNA?S?__Z:O]
M__^`O"2R````00^%!_O__X"\)+,```!##X7Y^O__@+PDM````$@/A>OZ__^`
MO"2U````10^%W?K__[H"````2,?&QO___T2)_^C>S/__NBD```!,B>Y$B?_H
MWLS__T&YI%]``$V)Z.D>____Z.O+__\/'P!!5C'_055!5%532('L$!```&1(
MBP0E*````$B)A"0($```,<#H_\G__XLXZ`C*__](B<?HP,S__[\]8$``Q@`!
M,<#H:>?__TB%P'0)@#@Q#X2[`@``,<#HA/+__S'_Z,7)__^+..C.R?__N0L`
M``"Z`0```+ZK84``2(G'Z(?,__](A<`/A+<```!(BT`0,=M,BV`HBP6&@0``
M@_@!#XZ3````O0@```#K,F8/'T0``$F+=1!(BP5=@0``2(L\*.A4R___A<!U
M:(/#`8L%3X$``$B#Q0B-4P$YT'Y:,?_H1<G__XLXZ$[)__\QR8G:3(GF2(G'
MZ!_,__](A<!T,DR+*$'V10T$=:@Q_^@9R?__BSCH(LG__TR)[KD"````,=)(
MB<?H`,S__TB)QNN'#Q\`BP7J@```C5#_.=H/A&\!```Q_^C@R/__BSCHZ<C_
M_[D$````N@$```"^L&%``$B)Q^BBR___2(7`2(G#=#9(BT`0,?](BRCHK,C_
M_XLXZ+7(__^ZLF%``$B)QTB)[NBER___2(M#$$B+&/9##D`/A4P!```Q_^A]
MR/__BSCHALC__[D$````N@$```"^MV%``$B)Q^@_R___2(7`2(G#=#LQP+\$
M8$``Z-/E__](A<!)B<0/A#\!``!(BT,0,?](BRCH,<C__XLXZ#K(__],B>)(
MB>Y(B<?H+,O__S'_Z!7(__^+..@>R/__2(G'Z-;*__\Q_\8``.C\Q___BSCH
M!<C__TB)Q^@-R___O_M?0`!(QP``````,<#H8N7__TB%P$B)!=!_```/A',!
M``"^[0$``$B)Q^@=RO__@_@1=`V%P'0)@_C_#X7$`0``2(N$)`@0``!D2#,$
M)2@```!FD`^%I@$``$B!Q!`0``!;74%<05U!7L-F+@\?A```````@_@!#XZ(
M_O__NP$```!FD#'_@\,!Z%['__^+..AGQ___3(GF2(G'Z'S*__\Y'3Y_``!_
MW.E8_O__#Q^``````#'_Z#''__^+..@ZQ___2(G>2(G'Z%_*___IEO[__V:0
M@'@!``^$0OW__V8/'T0``.DP_?__#Q\`2(L%Z7X``#'_3(LP2(M#$$R+(.CH
MQO__BSCH\<;__[K_#P``2(GFO[EA0`#H;\C__X7`08G%?E!!C47_2&/0@#P4
M`'4'A<!T/T&)Q4UC[;XO````2(GG3(GJZ/')__](A<!T)#'_Z)7&__^+..B>
MQO__3(GI2(GB3(GF2(G'Z-W)___I7/[__S'_Z''&__^+..AZQO__3(GR3(GF
M2(G'Z&S)___I._[__TB+/4!^``#H8_+__TB)!41^``!(BT,02(LM*7X``$R+
M($'V1"0-!`^$A````$F+1"002(GN2(G'Z(K)__^)Q4B+0Q!(BQCV0PT$=$-(
MBT,02(G"2(L%[7T``(GIOP!B0`!(BS`QP.ALR?__Z*?'__^)P4B+!<Y]``!(
MBQ77?0``OS!B0`!(BS`QP.A(R?__,?_HP<7__XLXZ,K%__^Y`@```#'22(G>
M2(G'Z*C(___KGS'_Z)_%__^+..BHQ?__N0(````QTDR)YDB)Q^B&R/__Z5S_
M__]F#Q^$``````!52(G]4TB#[`CH0NW__P^V$$B)PX32=!I(C4`!#Q]``(#Z
M+P^V$$@/1-A(@\`!A-)U[C'`Z,?O__^%P'0=2(7M=!B`?0``=!*^8&%``$B)
MW^ACQ___2#G8=`=(@\0(6UW#2(/$"$B)[UM=Z>'D__^02(/L"#'_Z/W$__^+
M..@&Q?__OO____](B<?H><C__S'_Z.+$__^+..CKQ/__,=*^(#]``$B)QTB#
MQ`CI>,G__U-(@^P@B7PD'$B)="002(U\)!Q(B50D"$B-="002(U4)`CH0<C_
M_[K()$``O]@D0`!(B=;H]P$``#'_Z%C(__^`.`!U-NA>R/__2(7`2(D%7'P`
M``^$M@$``$B)Q^A6R/__,?_H7\3__XLXZ&C$__](B<?H4,C__\8``#'_Z$;$
M__^+..A/Q/__2(G'Z$?(__^`"`(Q_^@MQ/__BSCH-L3__TB)Q^@NR/__@`@!
MBWPD'(/'!4AC_TB%_P^(&`$``/)(#RK'\@]8!6,%``!F#RX%8P4```^'/P$`
M`$C!YP/H!,C__TB+="002(D%R'L``$B+%DC'0`C+84``2,=`$*!B0`!(QT`8
MSF%``$B)$(M4)!S'!:1[```$````@_H!?B*-2OY(C3S-"````#')9I!(BUP.
M"$B)7`@@2(/!"$@Y^77MC4H$13'`@\(#OK!%0`!(8\E(QT3(^`````!(BSU,
M>P``2(G!Z(S'__^%P(G#=#Y(BSTW>P``Z(K'__\QP+]-8$``Z,;@__](A<!T
M,4B+/1I[``#H?<?__^B(Q___B=A(@\0@6\,/'X0``````$B+/?EZ``#H?,?_
M_XG#Z[)(BST!>P``2(7_="GH?_W__^N\#Q]$``!(B?A(B?I(T>B#X@%("=#R
M2`\JP/(/6,#IT/[__[_[7T``Z%'@__](A<!(B<=(B06\>@``=('KO+\!````
MZ"['__^^4.E``+_(84``,<#H'<;__Y`/'T``2(T%^7D``$B%P'0,2(L-[7D`
M`.E`Q___,<GK]P\?0`#SPV9F9F9F+@\?A```````2(EL)-A,B60DX$B-+3^A
MO_],C24XH;__3(EL).A,B70D\$R)?"3X2(E<)-!(@^PX3"GE08G]28GV2,']
M`TF)U^C;P?__2(7M=!PQVP\?0`!,B?I,B?9$B>]!_Q3<2(/#`4@YZW+J2(M<
M)`A(BVPD$$R+9"083(ML)"!,BW0D*$R+?"0P2(/$.,,/'X``````54B)Y5-(
M@^P(2(L%4'D``$B#^/]T&;N@V$``#Q]$``!(@^L(_]!(BP-(@_C_=?%(@\0(
M6\G#9I!(@^P(Z+_&__](@\0(PP``````````````````````````````````
M`0`"```N`"XN`"5S+R5S`'!E<FQX<VDN8P!$>6YA3&]A9&5R.CIB;V]T7T1Y
M;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`97AE`"]P<F]C+R5I+R5S
M`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`)7,E<R5S`%!%4DPU3$E"`%!%
M4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!
M5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!005)?05)'0P!0
M05)?05)'5E\P`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]435!$25(`
M4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@!0
M05)?0TQ%04X]`#$`3$1?3$E"4D%265]0051(`$Q)0E!!5$@`3$E"4D%265]0
M051(`%!!5$@`1%E,1%],24)205)97U!!5$@`4TA,24)?4$%42`!54T52`%-9
M4U1%30`O=&UP`%1-4$1)4@!414U01$E2`%1%35``5$U0`%5315).04U%`'!A
M<BT`)7,E<R5S)7,`<&%R;``N<&%R`"5S)7-C86-H92TE<R5S`"4P,G@`)7,E
M<W1E;7`M)74E<P`E<R5S=&5M<"TE=2TE=25S`$%21U8`&`!P97)L`#``+W!R
M;V,O<V5L9B]E>&4`)7,`+64`+2T`<&%R+G!L`%]?14Y67U!!4E]#3$5!3E]?
M("`@("`@("`@("`@("`@```````````E<SH@97AE8W5T:6]N(&]F("5S(&9A
M:6QE9"`M(&%B;W)T:6YG('=I=&@@)6DN"@`E<SH@8W)E871I;VX@;V8@<')I
M=F%T92!T96UP;W)A<GD@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`M(&%B;W)T
M:6YG('=I=&@@)6DN"@``````````````````````P$,`````````````````
M````(R$O=7-R+V)I;B]P97)L"F5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@
M+5,@)#`@)'LQ*R(D0")])PH@("`@:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E
M<B!S;VUE('-H96QL"@IP86-K86=E(%]?<&%R7W!L.PH*(R`M+2T@5&AI<R!S
M8W)I<'0@;75S="!N;W0@=7-E(&%N>2!M;V1U;&5S(&%T(&-O;7!I;&4@=&EM
M92`M+2T*(R!U<V4@<W1R:6-T.PH*(VQI;F4@,34X"@IM>2`H)'!A<E]T96UP
M+"`D<')O9VYA;64L($!T;7!F:6QE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,
M14%.?2D@>PH@("`@<F5Q=6ER92!&:6QE.CI496UP.PH@("`@<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*("`@(')E<75I<F4@1FEL93HZ4W!E8SL*("`@(&UY
M("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D<&%R7W1E;7`I
M.PH@("`@;W5T<RAQ<7M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(GTI
M.PH@("`@1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M
M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["B`@("!R;61I<B`D<&%R7W1E
M;7`["B`@("`C($1O;B=T(')E;6]V92!T;W!D:7(@8F5C875S92!T:&ES(&-A
M=7-E<R!A(')A8V4@=VET:"!O=&AE<B!A<'!S"B`@("`C('1H870@87)E('1R
M>6EN9R!T;R!S=&%R="X*"B`@("!I9B`H+60@)'!A<E]T96UP("8F("1>3R!N
M92`G35-7:6XS,B<I('L*("`@("`@("`C(%-O;65T:&EN9R!W96YT('=R;VYG
M('5N;&EN:VEN9R!T:&4@=&5M<&]R87)Y(&1I<F5C=&]R>2X@(%1H:7,*("`@
M("`@("`C('1Y<&EC86QL>2!H87!P96YS(&]N('!L871F;W)M<R!T:&%T(&1I
M<V%L;&]W('5N;&EN:VEN9R!S:&%R960*("`@("`@("`C(&QI8G)A<FEE<R!A
M;F0@97AE8W5T86)L97,@=&AA="!A<F4@:6X@=7-E+B!5;FQI;FL@=VET:"!A
M(&)A8VMG<F]U;F0*("`@("`@("`C('-H96QL(&-O;6UA;F0@<V\@=&AE(&9I
M;&5S(&%R92!N;R!L;VYG97(@:6X@=7-E(&)Y('1H:7,@<')O8V5S<RX*("`@
M("`@("`C($1O;B=T(&1O(&%N>71H:6YG(&]N(%=I;F1O=W,@8F5C875S92!O
M=7(@<&%R96YT('!R;V-E<W,@=VEL;`H@("`@("`@(",@=&%K92!C87)E(&]F
M(&-L96%N:6YG('1H:6YG<R!U<"X*"B`@("`@("`@;7D@)'1M<"`](&YE=R!&
M:6QE.CI496UP*`H@("`@("`@("`@("!414U03$%412`]/B`G=&UP6%A86%@G
M+`H@("`@("`@("`@("!$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H
M)'1O<&1I<BDL"B`@("`@("`@("`@(%-51D9)6"`]/B`G+F-M9"<L"B`@("`@
M("`@("`@(%5.3$E.2R`]/B`P+`H@("`@("`@("D["@H@("`@("`@('!R:6YT
M("1T;7`@(B,A+V)I;B]S:`IX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D
M;PH@("!R;2`M<F8@)R1P87)?=&5M<"<*("`@:68@6R!<(2`M9"`G)'!A<E]T
M96UP)R!=.R!T:&5N"B`@("`@("!B<F5A:PH@("!F:0H@("!S;&5E<"`Q"B`@
M('@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G(B`N("1T;7`M/F9I;&5N86UE
M("X@(B<*(CL*("`@("`@("`@("`@8VAM;V0@,#<P,"PD=&UP+3YF:6QE;F%M
M93L*("`@("`@("!M>2`D8VUD(#T@)'1M<"T^9FEL96YA;64@+B`G(#XO9&5V
M+VYU;&P@,CXF,2`F)SL*("`@("`@("!C;&]S92`D=&UP.PH@("`@("`@('-Y
M<W1E;2@D8VUD*3L*("`@("`@("!O=71S*'%Q*%-P87=N960@8F%C:V=R;W5N
M9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@*0H@("`@("`@("`@("`@
M+B`D=&UP+3YF:6QE;F%M92D["B`@("!]"GT@?0H*0D5'24X@>PH@("`@26YT
M97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!
M4CHZ0D]/5#L*"B`@("!E=F%L('L*"E]P87)?:6YI=%]E;G8H*3L*"FEF("AE
M>&ES=',@)$5.5GM005)?05)'5E\P?2!A;F0@)$5.5GM005)?05)'5E\P?2`I
M('L*("`@($!!4D=6(#T@;6%P("1%3E9[(E!!4E]!4D=67R1?(GTL("@Q("XN
M("1%3E9[4$%27T%21T-]("T@,2D["B`@("`D,"`]("1%3E9[4$%27T%21U9?
M,'T["GT*96QS92!["B`@("!F;W(@*&ME>7,@)45.5BD@>PH@("`@("`@(&1E
M;&5T92`D14Y6>R1??2!I9B`O7E!!4E]!4D=67R\["B`@("!]"GT*"FUY("1Q
M=6EE="`]("$D14Y6>U!!4E]$14)51WT["@HC(&9I>"`D<')O9VYA;64@:68@
M:6YV;VME9"!F<F]M(%!!5$@*;7D@)4-O;F9I9R`]("@*("`@('!A=&A?<V5P
M("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`H@("`@7V5X
M92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_
M("<N97AE)R`Z("<G*2P*("`@(%]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-
M4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["@I?<V5T7W!R;V=N86UE*"D[
M"E]S971?<&%R7W1E;7`H*3L*"B,@36%G:6,@<W1R:6YG(&-H96-K:6YG(&%N
M9"!E>'1R86-T:6YG(&)U;F1L960@;6]D=6QE<R![>WL*;7D@*"1S=&%R=%]P
M;W,L("1D871A7W!O<RD["GL*("`@(&QO8V%L("1324=[7U]705).7U]](#T@
M<W5B('M].PH*("`@(",@0VAE8VL@9FEL92!T>7!E+"!G970@<W1A<G0@;V8@
M9&%T82!S96-T:6]N('M[>PH@("`@;W!E;B!?1D@L("<\)RP@)'!R;V=N86UE
M(&]R(&QA<W0["B`@("!B:6YM;V1E*%]&2"D["@H@("`@;7D@)&)U9CL*("`@
M('-E96L@7T9(+"`M."P@,CL*("`@(')E860@7T9(+"`D8G5F+"`X.PH@("`@
M;&%S="!U;FQE<W,@)&)U9B!E<2`B7&Y005(N<&U<;B(["@H@("`@<V5E:R!?
M1D@L("TQ,BP@,CL*("`@(')E860@7T9(+"`D8G5F+"`T.PH@("`@<V5E:R!?
M1D@L("TQ,B`M('5N<&%C:R@B3B(L("1B=68I+"`R.PH@("`@<F5A9"!?1D@L
M("1B=68L(#0["@H@("`@)&1A=&%?<&]S(#T@*'1E;&P@7T9(*2`M(#0["B`@
M("`C('U]?0H*("`@(",@17AT<F%C=&EN9R!E86-H(&9I;&4@:6YT;R!M96UO
M<GD@>WM["B`@("!M>2`E<F5Q=6ER95]L:7-T.PH@("`@=VAI;&4@*"1B=68@
M97$@(D9)3$4B*2!["B`@("`@("`@<F5A9"!?1D@L("1B=68L(#0["B`@("`@
M("`@<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PH*("`@("`@
M("!M>2`D9G5L;&YA;64@/2`D8G5F.PH@("`@("`@(&]U=',H<7$H56YP86-K
M:6YG(&9I;&4@(B1F=6QL;F%M92(N+BXI*3L*("`@("`@("!M>2`D8W)C(#T@
M*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N
M9&5F.PH@("`@("`@(&UY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@
M;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*"B`@("`@("`@<F5A9"!?1D@L("1B
M=68L(#0["B`@("`@("`@<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B
M=68I.PH*("`@("`@("!I9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O
M7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*("`@("`@("`@("`@;7D@*"1O=70L
M("1F:6QE;F%M92D@/2!?=&5M<&9I;&4H)&5X="P@)&-R8RD["B`@("`@("`@
M("`@(&EF("@D;W5T*2!["B`@("`@("`@("`@("`@("!B:6YM;V1E*"1O=70I
M.PH@("`@("`@("`@("`@("`@<')I;G0@)&]U="`D8G5F.PH@("`@("`@("`@
M("`@("`@8VQO<V4@)&]U=#L*("`@("`@("`@("`@("`@(&-H;6]D(#`W-34L
M("1F:6QE;F%M93L*("`@("`@("`@("`@?0H@("`@("`@("`@("`D4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B`@("`@
M("`@("`@("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F
M=6QL;F%M93L*("`@("`@("!]"B`@("`@("`@96QS:68@*"`D9G5L;&YA;64@
M/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D("1%3E9[4$%27U1%35!]("D@
M>PH@("`@("`@("`@("`C('-H;W5L9"!B92!M;W9E9"!T;R!?=&5M<&9I;&4H
M*0H@("`@("`@("`@("!M>2`D9FEL96YA;64@/2`B)$5.5GM005)?5$5-4'TO
M)&)A<V5N86UE)&5X="(["B`@("`@("`@("`@(&]U=',H(E-(3$E".B`D9FEL
M96YA;65<;B(I.PH@("`@("`@("`@("!O<&5N(&UY("1O=70L("<^)RP@)&9I
M;&5N86UE(&]R(&1I92`D(3L*("`@("`@("`@("`@8FEN;6]D92@D;W5T*3L*
M("`@("`@("`@("`@<')I;G0@)&]U="`D8G5F.PH@("`@("`@("`@("!C;&]S
M92`D;W5T.PH@("`@("`@('T*("`@("`@("!E;'-E('L*("`@("`@("`@("`@
M)')E<75I<F5?;&ES='LD9G5L;&YA;65](#T*("`@("`@("`@("`@)%!!4CHZ
M2&5A=GDZ.DUO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["B`@("`@("`@("`@
M("`@("!B=68@/3X@)&)U9BP*("`@("`@("`@("`@("`@(&-R8R`]/B`D8W)C
M+`H@("`@("`@("`@("`@("`@;F%M92`]/B`D9G5L;&YA;64L"B`@("`@("`@
M("`@('T["B`@("`@("`@?0H@("`@("`@(')E860@7T9(+"`D8G5F+"`T.PH@
M("`@?0H@("`@(R!]?7T*"B`@("!L;V-A;"!`24Y#(#T@*'-U8B!["B`@("`@
M("`@;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PH*("`@("`@("!R971U<FX@
M:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["@H@("`@("`@(&UY("1F:6QE
M;F%M92`](&1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5]('Q\(&1O('L*
M("`@("`@("`@("`@;7D@)&ME>3L*("`@("`@("`@("`@9F]R96%C:"`H:V5Y
M<R`E<F5Q=6ER95]L:7-T*2!["B`@("`@("`@("`@("`@("!N97AT('5N;&5S
M<R`O7%$D;6]D=6QE7$4D+SL*("`@("`@("`@("`@("`@("1K97D@/2`D7SL@
M;&%S=#L*("`@("`@("`@("`@?0H@("`@("`@("`@("!D96QE=&4@)')E<75I
M<F5?;&ES='LD:V5Y?2!I9B!D969I;F5D*"1K97DI.PH@("`@("`@('T@;W(@
M<F5T=7)N.PH*("`@("`@("`D24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&9I
M;&5N86UE+R1M;V1U;&4B.PH*("`@("`@("!I9B`H)$5.5GM005)?0TQ%04Y]
M(&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PH@("`@("`@("`@("!M
M>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE("0A.PH@("`@
M("`@("`@("!B:6YM;V1E*"1F:"D["B`@("`@("`@("`@('!R:6YT("1F:"`D
M9FEL96YA;64M/GMB=69].PH@("`@("`@("`@("!S965K*"1F:"P@,"P@,"D[
M"B`@("`@("`@("`@(')E='5R;B`D9F@["B`@("`@("`@?0H@("`@("`@(&5L
M<V4@>PH@("`@("`@("`@("!M>2`H)&]U="P@)&YA;64I(#T@7W1E;7!F:6QE
M*"<N<&TG+"`D9FEL96YA;64M/GMC<F-]*3L*("`@("`@("`@("`@:68@*"1O
M=70I('L*("`@("`@("`@("`@("`@(&)I;FUO9&4H)&]U="D["B`@("`@("`@
M("`@("`@("!P<FEN="`D;W5T("1F:6QE;F%M92T^>V)U9GT["B`@("`@("`@
M("`@("`@("!C;&]S92`D;W5T.PH@("`@("`@("`@("!]"B`@("`@("`@("`@
M(&]P96X@;7D@)&9H+"`G/"<L("1N86UE(&]R(&1I92`D(3L*("`@("`@("`@
M("`@8FEN;6]D92@D9F@I.PH@("`@("`@("`@("!R971U<FX@)&9H.PH@("`@
M("`@('T*"B`@("`@("`@9&EE(")";V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N
M;F]T(&9I;F0@)&UO9'5L92%<;B(["B`@("!]+"!`24Y#*3L*"B`@("`C($YO
M=R!L;V%D(&%L;"!B=6YD;&5D(&9I;&5S('M[>PH*("`@(",@:6YI=&EA;&EZ
M92!S:&%R960@;V)J96-T('!R;V-E<W-I;F<*("`@(')E<75I<F4@6%-,;V%D
M97(["B`@("!R97%U:7)E(%!!4CHZ2&5A=GD["B`@("!R97%U:7)E($-A<G`Z
M.DAE879Y.PH@("`@<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["B`@("!005(Z
M.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["@H@("`@(R!N;W<@;&5T)W,@
M=')Y(&=E='1I;F<@:&5L<&5R(&UO9'5L97,@9G)O;2!W:71H:6X*("`@(')E
M<75I<F4@24\Z.D9I;&4["@H@("`@(R!L;V%D(')E<W0@;V8@=&AE(&=R;W5P
M(&EN"B`@("!W:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q
M=6ER95]L:7-T*5LP72D@>PH@("`@("`@("-L;V-A;"`D24Y#>R=#=V0N<&TG
M?2`](%]?1DE,15]?(&EF("1>3R!N92`G35-7:6XS,B<["B`@("`@("`@=6YL
M97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO
M*2!["B`@("`@("`@("`@(",@<F5Q=6ER92!M;V1U;&5S+"!D;R!O=&AE<B!E
M>&5C=71A8FQE(&9I;&5S"B`@("`@("`@("`@(&EF("@D9FEL96YA;64@/7X@
M+UPN<&UC/R0O:2D@>PH@("`@("`@("`@("`@("`@<F5Q=6ER92`D9FEL96YA
M;64["B`@("`@("`@("`@('T*("`@("`@("`@("`@96QS92!["B`@("`@("`@
M("`@("`@("`C(%-K:7`@06-T:79E4W1A=&4G<R!S:71E8W5S=&]M:7IE+G!L
M(&9I;&4Z"B`@("`@("`@("`@("`@("!D;R`D9FEL96YA;64@=6YL97-S("1F
M:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*("`@("`@("`@("`@
M?0H@("`@("`@('T*("`@("`@("!D96QE=&4@)')E<75I<F5?;&ES='LD9FEL
M96YA;65].PH@("`@?0H*("`@(",@?7U]"@H@("`@;&%S="!U;FQE<W,@)&)U
M9B!E<2`B4$M<,#`S7#`P-"(["B`@("`D<W1A<G1?<&]S(#T@*'1E;&P@7T9(
M*2`M(#0["GT*(R!]?7T*"B,@07)G=6UE;G0@<')O8V5S<VEN9R![>WL*;7D@
M0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?
M;F%M92D["@ID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ
M92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*"B1Q=6EE="`]
M(#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*(R!$;VXG="!S=V%L;&]W(&%R
M9W5M96YT<R!F;W(@8V]M<&EL960@97AE8W5T86)L97,@=VET:&]U="`M+7!A
M<BUO<'1I;VYS"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM
M<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PH@("`@;7D@)61I<W1?8VUD(#T@
M<7<H"B`@("`@("`@<"`@(&)L:6)?=&]?<&%R"B`@("`@("`@:2`@(&EN<W1A
M;&Q?<&%R"B`@("`@("`@=2`@('5N:6YS=&%L;%]P87(*("`@("`@("!S("`@
M<VEG;E]P87(*("`@("`@("!V("`@=F5R:69Y7W!A<@H@("`@*3L*"B`@("`C
M(&EF('1H92!A<'`@:7,@:6YV;VME9"!A<R`B87!P;F%M92`M+7!A<BUO<'1I
M;VYS("TM<F5U<V4@4%)/1U)!32!`4%)/1U]!4D=6(BP*("`@(",@=7-E('1H
M92!A<'`@=&\@<G5N('1H92!G:79E;B!P97)L(&-O9&4@:6YS=&5A9"!O9B!A
M;GET:&EN9R!F<F]M('1H90H@("`@(R!A<'`@:71S96QF("AB=70@<W1I;&P@
M<V5T('5P('1H92!N;W)M86P@87!P(&5N=FER;VYM96YT(&%N9"!`24Y#*0H@
M("`@:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*("`@
M("`@("!S:&EF="!`05)'5CL*("`@("`@("`D14Y6>U!!4E]!4%!?4D554T5]
M(#T@<VAI9G0@0$%21U8["B`@("!]"B`@("!E;'-E('L@(R!N;W)M86P@<&%R
M;"!B96AA=FEO=7(*"B`@("`@("`@;7D@0&%D9%]T;U]I;F,["B`@("`@("`@
M=VAI;&4@*$!!4D=6*2!["B`@("`@("`@("`@("1!4D=66S!=(#U^("]>+2A;
M04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["@H@("`@("`@("`@("!I
M9B`H)#$@97$@)TDG*2!["B`@("`@("`@("`@("`@("!P=7-H($!A9&1?=&]?
M:6YC+"`D,CL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-I9B`H)#$@
M97$@)TTG*2!["B`@("`@("`@("`@("`@("!E=F%L(")U<V4@)#(B.PH@("`@
M("`@("`@("!]"B`@("`@("`@("`@(&5L<VEF("@D,2!E<2`G02<I('L*("`@
M("`@("`@("`@("`@('5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*("`@("`@("`@
M("`@?0H@("`@("`@("`@("!E;'-I9B`H)#$@97$@)T\G*2!["B`@("`@("`@
M("`@("`@("`D;W5T(#T@)#(["B`@("`@("`@("`@('T*("`@("`@("`@("`@
M96QS:68@*"0Q(&5Q("=B)RD@>PH@("`@("`@("`@("`@("`@)&)U;F1L92`]
M("=S:71E)SL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-I9B`H)#$@
M97$@)T(G*2!["B`@("`@("`@("`@("`@("`D8G5N9&QE(#T@)V%L;"<["B`@
M("`@("`@("`@('T*("`@("`@("`@("`@96QS:68@*"0Q(&5Q("=Q)RD@>PH@
M("`@("`@("`@("`@("`@)'%U:65T(#T@,3L*("`@("`@("`@("`@?0H@("`@
M("`@("`@("!E;'-I9B`H)#$@97$@)TPG*2!["B`@("`@("`@("`@("`@("!O
M<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE(")86%@Z($-A;FYO="!O<&5N
M(&QO9SH@)"$B.PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<VEF("@D
M,2!E<2`G5"<I('L*("`@("`@("`@("`@("`@("1C86-H95]N86UE(#T@)#([
M"B`@("`@("`@("`@('T*"B`@("`@("`@("`@('-H:69T*$!!4D=6*3L*"B`@
M("`@("`@("`@(&EF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PH@("`@
M("`@("`@("`@("`@9&5L971E("1%3E9[)U!!4E]414U0)WT["B`@("`@("`@
M("`@("`@("!I;FET7VEN8R@I.PH@("`@("`@("`@("`@("`@<F5Q=6ER92!0
M05(Z.D1I<W0["B`@("`@("`@("`@("`@("`F>R)005(Z.D1I<W0Z.B1C;60B
M?2@I('5N;&5S<R!`05)'5CL*("`@("`@("`@("`@("`@("9[(E!!4CHZ1&ES
M=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["B`@("`@("`@("`@("`@("!E>&ET
M.PH@("`@("`@("`@("!]"B`@("`@("`@?0H*("`@("`@("!U;G-H:69T($!)
M3D,L($!A9&1?=&]?:6YC.PH@("`@?0I]"@HC(%A86"`M+2!A9&0@+2UP87(M
M9&5B=6<@<W5P<&]R="$*"B,@?7U]"@HC($]U='!U="!M;V1E("@M3RD@:&%N
M9&QI;F<@>WM["FEF("@D;W5T*2!["B`@("!["B`@("`@("`@(VQO8V%L("1)
M3D-[)T-W9"YP;2=](#T@7U]&24Q%7U\@:68@)%Y/(&YE("=-4U=I;C,R)SL*
M("`@("`@("!R97%U:7)E($E/.CI&:6QE.PH@("`@("`@(')E<75I<F4@07)C
M:&EV93HZ6FEP.PH@("`@?0H*("`@(&UY("1P87(@/2!S:&EF="A`05)'5BD[
M"B`@("!M>2`D>FEP.PH*"B`@("!I9B`H9&5F:6YE9"`D<&%R*2!["B`@("`@
M("`@;W!E;B!M>2`D9F@L("<\)RP@)'!A<B!O<B!D:64@(D-A;FYO="!F:6YD
M("<D<&%R)SH@)"$B.PH@("`@("`@(&)I;FUO9&4H)&9H*3L*("`@("`@("!B
M;&5S<R@D9F@L("=)3SHZ1FEL92<I.PH*("`@("`@("`D>FEP(#T@07)C:&EV
M93HZ6FEP+3YN97<["B`@("`@("`@*"`D>FEP+3YR96%D1G)O;49I;&5(86YD
M;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0H@("`@
M("`@("`@("!O<B!D:64@(E)E860@)R1P87(G(&5R<F]R.B`D(2(["B`@("!]
M"@H*("`@(&UY("5E;G8@/2!D;R!["B`@("`@("`@:68@*"1Z:7`@86YD(&UY
M("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*("`@("`@
M("`@("`@)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B`@("`@("`@("`@("1M
M971A(#U^(',O7EQ3+BHO+VUS.PH@("`@("`@("`@("`D;65T82`]?B`O7B`@
M*%M>.ETK*3H@*"XK*20O;6<["B`@("`@("`@?0H@("`@?3L*"B`@("`C($]P
M96X@:6YP=70@86YD(&]U='!U="!F:6QE<R![>WL*("`@(&QO8V%L("0O(#T@
M7#0["@H@("`@:68@*&1E9FEN960@)'!A<BD@>PH@("`@("`@(&]P96X@4$%2
M+"`G/"<L("1P87(@;W(@9&EE("(D(3H@)'!A<B(["B`@("`@("`@8FEN;6]D
M92A005(I.PH@("`@("`@(&1I92`B)'!A<B!I<R!N;W0@82!005(@9FEL92(@
M=6YL97-S(#Q005(^(&5Q(")02UPP,#-<,#`T(CL*("`@('T*"B`@("!#<F5A
M=&50871H*"1O=70I(#L*("`@(`H@("`@;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=R@*("`@("`@("`D;W5T+`H@("`@("`@($E/.CI&:6QE.CI/7T-214%4*"D@
M?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL
M"B`@("`@("`@,#<W-RP*("`@("D@;W(@9&EE("0A.PH@("`@8FEN;6]D92@D
M9F@I.PH*("`@("0O(#T@*&1E9FEN960@)&1A=&%?<&]S*2`_(%PD9&%T85]P
M;W,@.B!U;F1E9CL*("`@('-E96L@7T9(+"`P+"`P.PH@("`@;7D@)&QO861E
M<B`]('-C86QA<B`\7T9(/CL*("`@(&EF("@A)$5.5GM005)?5D520D%424U]
M(&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PH@("`@("`@(')E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PH@("`@("`@(%!!4CHZ1FEL
M=&5R.CI0;V13=')I<"T^;F5W+3YA<'!L>2A<)&QO861E<BP@)#`I"B`@("!]
M"B`@("!F;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["B`@("`@
M("`@;7D@)'9A;"`]("1E;G9[)&ME>7T@;W(@;F5X=#L*("`@("`@("`D=F%L
M(#T@979A;"`D=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*("`@("`@("!M>2`D
M;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U8R@D:V5Y*2`N(")?7R(["B`@("`@
M("`@;7D@)'-E="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*("`@
M("`@("`D;&]A9&5R(#U^('-[)&UA9VEC*"`K*7U["B`@("`@("`@("`@("1M
M86=I8R`N("1S970@+B`H)R`G('@@*&QE;F=T:"@D,2D@+2!L96YG=&@H)'-E
M="DI*0H@("`@("`@('UE9SL*("`@('T*("`@("1F:"T^<')I;G0H)&QO861E
M<BD["B`@("`D+R`]('5N9&5F.PH@("`@(R!]?7T*"B`@("`C(%=R:71E(&)U
M;F1L960@;6]D=6QE<R![>WL*("`@(&EF("@D8G5N9&QE*2!["B`@("`@("`@
M<F5Q=6ER92!005(Z.DAE879Y.PH@("`@("`@(%!!4CHZ2&5A=GDZ.E]I;FET
M7V1Y;F%L;V%D97(H*3L*("`@("`@("!I;FET7VEN8R@I.PH*("`@("`@("!R
M97%U:7)E7VUO9'5L97,H*3L*"B`@("`@("`@;7D@0&EN8R`]('-O<G0@>PH@
M("`@("`@("`@("!L96YG=&@H)&(I(#P]/B!L96YG=&@H)&$I"B`@("`@("`@
M?2!G<F5P('L*("`@("`@("`@("`@(2]"4T1004XO"B`@("`@("`@?2!G<F5P
M('L*("`@("`@("`@("`@*"1B=6YD;&4@;F4@)W-I=&4G*2!O<@H@("`@("`@
M("`@("`H)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`H@
M("`@("`@("`@("`@)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]
M*3L*("`@("`@("!]($!)3D,["@H@("`@("`@(",@1FEL92!E>&ES=',@=&5S
M="!A9&1E9"!T;R!F:7@@4E0@(S0Q-SDP.@H@("`@("`@(",@1G5N;GDL(&YO
M;BUE>&ES=&EN9R!E;G1R>2!I;B!?/"XN+BYA=71O+T-O;7!R97-S+U)A=R]:
M;&EB+V%U=&]S<&QI="YI>"X*("`@("`@("`C(%1H:7,@:7,@82!B86YD+6%I
M9"!F:7@@=VET:"!N;R!D965P97(@9W)A<W`@;V8@=&AE(&ES<W5E+@H@("`@
M("`@(",@4V]M96)O9'D@<&QE87-E(&=O('1H<F]U9V@@=&AE('!A:6X@;V8@
M=6YD97)S=&%N9&EN9R!W:&%T)W,@:&%P<&5N:6YG+`H@("`@("`@(",@22!F
M86EL960N("TM(%-T969F96X*("`@("`@("!M>2`E9FEL97,["B`@("`@("`@
M+UY?/"@N*RDD+R!A;F0@+64@)#$@86YD("1F:6QE<WLD,7TK*R!F;W(@:V5Y
M<R`E.CH["B`@("`@("`@)&9I;&5S>R1??2LK(&9O<B!V86QU97,@)4E.0SL*
M"B`@("`@("`@;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X
M='T["B`@("`@("`@;7D@)7=R:71T96X["@H@("`@("`@(&9O<F5A8V@@*'-O
M<G0@:V5Y<R`E9FEL97,I('L*("`@("`@("`@("`@;7D@*"1N86UE+"`D9FEL
M92D["@H@("`@("`@("`@("!F;W)E86-H(&UY("1D:7(@*$!I;F,I('L*("`@
M("`@("`@("`@("`@(&EF("@D;F%M92`]("1005(Z.DAE879Y.CI&=6QL0V%C
M:&5[)%]]*2!["B`@("`@("`@("`@("`@("`@("`@)&9I;&4@/2`D7SL*("`@
M("`@("`@("`@("`@("`@("!L87-T.PH@("`@("`@("`@("`@("`@?0H@("`@
M("`@("`@("`@("`@96QS:68@*"]>*%Q1)&1I<EQ%7"\H+BI;7D-C72DI7%HO
M:2D@>PH@("`@("`@("`@("`@("`@("`@("@D9FEL92P@)&YA;64I(#T@*"0Q
M+"`D,BD["B`@("`@("`@("`@("`@("`@("`@;&%S=#L*("`@("`@("`@("`@
M("`@('T*("`@("`@("`@("`@("`@(&5L<VEF("AM(5XO;&]A9&5R+UM>+UTK
M+R@N*EM>0V-=*5Q:(2D@>PH@("`@("`@("`@("`@("`@("`@(&EF("AM>2`D
M<F5F(#T@)%!!4CHZ2&5A=GDZ.DUO9'5L94-A8VAE>R0Q?2D@>PH@("`@("`@
M("`@("`@("`@("`@("`@("`H)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D[
M"B`@("`@("`@("`@("`@("`@("`@("`@(&QA<W0["B`@("`@("`@("`@("`@
M("`@("`@?0H@("`@("`@("`@("`@("`@("`@(&5L<VEF("@M9B`B)&1I<B\D
M,2(I('L*("`@("`@("`@("`@("`@("`@("`@("`@*"1F:6QE+"`D;F%M92D@
M/2`H(B1D:7(O)#$B+"`D,2D["B`@("`@("`@("`@("`@("`@("`@("`@(&QA
M<W0["B`@("`@("`@("`@("`@("`@("`@?0H@("`@("`@("`@("`@("`@?0H@
M("`@("`@("`@("!]"@H@("`@("`@("`@("!N97AT('5N;&5S<R!D969I;F5D
M("1N86UE(&%N9"!N;W0@)'=R:71T96Y[)&YA;65]*RL["B`@("`@("`@("`@
M(&YE>'0@:68@(7)E9B@D9FEL92D@86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E
M>'1<120O.PH@("`@("`@("`@("!O=71S*"!J;VEN("(B+`H@("`@("`@("`@
M("`@("`@<7$H4&%C:VEN9R`B*2P@<F5F("1F:6QE(#\@)&9I;&4M/GMN86UE
M?2`Z("1F:6QE+`H@("`@("`@("`@("`@("`@<7$H(BXN+BD*("`@("`@("`@
M("`@*3L*"B`@("`@("`@("`@(&UY("1C;VYT96YT.PH@("`@("`@("`@("!I
M9B`H<F5F*"1F:6QE*2D@>PH@("`@("`@("`@("`@("`@)&-O;G1E;G0@/2`D
M9FEL92T^>V)U9GT["B`@("`@("`@("`@('T*("`@("`@("`@("`@96QS92![
M"B`@("`@("`@("`@("`@("!O<&5N($9)3$4L("<\)RP@)&9I;&4@;W(@9&EE
M(")#86XG="!O<&5N("1F:6QE.B`D(2(["B`@("`@("`@("`@("`@("!B:6YM
M;V1E*$9)3$4I.PH@("`@("`@("`@("`@("`@)&-O;G1E;G0@/2`\1DE,13X[
M"B`@("`@("`@("`@("`@("!C;&]S92!&24Q%.PH*("`@("`@("`@("`@("`@
M(%!!4CHZ1FEL=&5R.CI0;V13=')I<"T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L
M("1F:6QE*0H@("`@("`@("`@("`@("`@("`@(&EF("$D14Y6>U!!4E]615)"
M051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*"B`@("`@
M("`@("`@("`@("!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P
M<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*("`@("`@("`@("`@?0H*
M("`@("`@("`@("`@;W5T<RAQ<2A7<FET=&5N(&%S("(D;F%M92(I*3L*("`@
M("`@("`@("`@)&9H+3YP<FEN="@B1DE,12(I.PH@("`@("`@("`@("`D9F@M
M/G!R:6YT*'!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DI.PH@("`@("`@
M("`@("`D9F@M/G!R:6YT*'-P<FEN=&8H"B`@("`@("`@("`@("`@("`B)3`X
M>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@
M)&YA;64*("`@("`@("`@("`@*2D["B`@("`@("`@("`@("1F:"T^<')I;G0H
M<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI*3L*("`@("`@("`@("`@)&9H
M+3YP<FEN="@D8V]N=&5N="D["B`@("`@("`@?0H@("`@?0H@("`@(R!]?7T*
M"B`@("`C($YO=R!W<FET92!O=70@=&AE(%!!4B!A;F0@;6%G:6,@<W1R:6YG
M<R![>WL*("`@("1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@:68@)'II
M<#L*"B`@("`D8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@
M-#`["B`@("!I9B`H(21C86-H95]N86UE(&%N9"!M>2`D;71I;64@/2`H<W1A
M="@D;W5T*2E;.5TI('L*("`@("`@("!M>2`D8W1X(#T@979A;"![(')E<75I
M<F4@1&EG97-T.CI32$$[($1I9V5S=#HZ4TA!+3YN97<H,2D@?0H@("`@("`@
M("`@("!\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.E-(03$[($1I9V5S=#HZ
M4TA!,2T^;F5W('T*("`@("`@("`@("`@?'P@979A;"![(')E<75I<F4@1&EG
M97-T.CI-1#4[($1I9V5S=#HZ340U+3YN97<@?3L*"B`@("`@("`@(R!7;W)K
M87)O=6YD(&9O<B!B=6<@:6X@1&EG97-T.CI32$$@-2XS."!A;F0@-2XS.0H@
M("`@("`@(&UY("1S:&%?=F5R<VEO;B`](&5V86P@>R`D1&EG97-T.CI32$$Z
M.E9%4E-)3TX@?2!\?"`P.PH@("`@("`@(&EF("@D<VAA7W9E<G-I;VX@97$@
M)S4N,S@G(&]R("1S:&%?=F5R<VEO;B!E<2`G-2XS.2<I('L*("`@("`@("`@
M("`@)&-T>"T^861D9FEL92@D;W5T+"`B8B(I(&EF("@D8W1X*3L*("`@("`@
M("!]"B`@("`@("`@96QS92!["B`@("`@("`@("`@(&EF("@D8W1X(&%N9"!O
M<&5N*&UY("1F:"P@(CPD;W5T(BDI('L*("`@("`@("`@("`@("`@(&)I;FUO
M9&4H)&9H*3L*("`@("`@("`@("`@("`@("1C='@M/F%D9&9I;&4H)&9H*3L*
M("`@("`@("`@("`@("`@(&-L;W-E*"1F:"D["B`@("`@("`@("`@('T*("`@
M("`@("!]"@H@("`@("`@("1C86-H95]N86UE(#T@)&-T>"`_("1C='@M/FAE
M>&1I9V5S="`Z("1M=&EM93L*("`@('T*("`@("1C86-H95]N86UE("X](")<
M,"(@>"`H-#$@+2!L96YG=&@@)&-A8VAE7VYA;64I.PH@("`@)&-A8VAE7VYA
M;64@+CT@(D-!0TA%(CL*("`@("1F:"T^<')I;G0H)&-A8VAE7VYA;64I.PH@
M("`@)&9H+3YP<FEN="AP86-K*"=.)RP@)&9H+3YT96QL("T@;&5N9W1H*"1L
M;V%D97(I*2D["B`@("`D9F@M/G!R:6YT*")<;E!!4BYP;5QN(BD["B`@("`D
M9F@M/F-L;W-E.PH@("`@8VAM;V0@,#<U-2P@)&]U=#L*("`@(",@?7U]"@H@
M("`@97AI=#L*?0HC('U]?0H*(R!0<F5P87)E("1P<F]G;F%M92!I;G1O(%!!
M4B!F:6QE(&-A8VAE('M[>PI["B`@("!L87-T('5N;&5S<R!D969I;F5D("1S
M=&%R=%]P;W,["@H@("`@7V9I>%]P<F]G;F%M92@I.PH*("`@(",@3F]W(&QO
M860@=&AE(%!!4B!F:6QE(&%N9"!P=70@:70@:6YT;R!005(Z.DQI8D-A8VAE
M('M[>PH@("`@<F5Q=6ER92!005(["B`@("!005(Z.DAE879Y.CI?:6YI=%]D
M>6YA;&]A9&5R*"D["@H*("`@('L*("`@("`@("`C;&]C86P@)$E.0WLG0W=D
M+G!M)WT@/2!?7T9)3$5?7R!I9B`D7D\@;F4@)TU35VEN,S(G.PH@("`@("`@
M(')E<75I<F4@1FEL93HZ1FEN9#L*("`@("`@("!R97%U:7)E($%R8VAI=F4Z
M.EII<#L*("`@('T*("`@(&UY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL*
M("`@(&UY("1F:"`]($E/.CI&:6QE+3YN97<["B`@("`D9F@M/F9D;W!E;BAF
M:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@(B0A.B`D0"(["B`@("`D>FEP+3YR
M96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2!O<B!D:64@(B0A.B`D0"(["@H@("`@<'5S:"!`4$%2
M.CI,:6)#86-H92P@)'II<#L*("`@("1005(Z.DQI8D-A8VAE>R1P<F]G;F%M
M97T@/2`D>FEP.PH*("`@("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["B`@
M("!O=71S*'%Q*%PD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2(I
M*3L*"B`@("!I9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C('-H;W5L
M9"!B92!S970@870@=&AI<R!P;VEN="$*("`@("`@("!F;W)E86-H(&UY("1M
M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PH@("`@("`@("`@("!N97AT(&EF
M("1M96UB97(M/FES1&ER96-T;W)Y.PH@("`@("`@("`@("!M>2`D;65M8F5R
M7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*("`@("`@("`@("`@;F5X="!U
M;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["B`@("`@("`@("`@("`@("!>"B`@
M("`@("`@("`@("`@("`O/W-H;&EB+PH@("`@("`@("`@("`@("`@*#\Z)$-O
M;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B`@("`@("`@("`@("`@("`H/SHD
M0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"B`@("`@("`@("`@("`@("`H
M6UXO72LI"B`@("`@("`@("`@("`@("`D"B`@("`@("`@("`@('UX.PH@("`@
M("`@("`@("!M>2`D97AT<F%C=%]N86UE(#T@)#$["B`@("`@("`@("`@(&UY
M("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%
M35!]+"`D97AT<F%C=%]N86UE*3L*("`@("`@("`@("`@:68@*"UF("1D97-T
M7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI
M('L*("`@("`@("`@("`@("`@(&]U=',H<7$H4VMI<'!I;F<@(B1M96UB97)?
M;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B
M*2D["B`@("`@("`@("`@('T@96QS92!["B`@("`@("`@("`@("`@("!O=71S
M*'%Q*$5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B
M*2D["B`@("`@("`@("`@("`@("`D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M
M960H)&1E<W1?;F%M92D["B`@("`@("`@("`@("`@("!C:&UO9"@P-34U+"`D
M9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PH@("`@("`@("`@("!]"B`@
M("`@("`@?0H@("`@?0H@("`@(R!]?7T*?0HC('U]?0H*(R!)9B!T:&5R92=S
M(&YO(&UA:6XN<&P@=&\@<G5N+"!S:&]W('5S86=E('M[>PIU;FQE<W,@*"10
M05(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*("`@(&1I92`\/"`B+B(@=6YL
M97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@
M6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B`@("`@
M("`D,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B`@("`D14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6
M*3L*?0HC('U]?0H*<W5B($-R96%T95!A=&@@>PH@("`@;7D@*"1N86UE*2`]
M($!?.PH@("`@"B`@("!R97%U:7)E($9I;&4Z.D)A<V5N86UE.PH@("`@;7D@
M*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I
M;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PH@("`@"B`@("!R97%U:7)E($9I
M;&4Z.E!A=&@["B`@("`*("`@($9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@
M=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*
M"G-U8B!R97%U:7)E7VUO9'5L97,@>PH@("`@(VQO8V%L("1)3D-[)T-W9"YP
M;2=](#T@7U]&24Q%7U\@:68@)%Y/(&YE("=-4U=I;C,R)SL*"B`@("!R97%U
M:7)E(&QI8CL*("`@(')E<75I<F4@1'EN84QO861E<CL*("`@(')E<75I<F4@
M:6YT96=E<CL*("`@(')E<75I<F4@<W1R:6-T.PH@("`@<F5Q=6ER92!W87)N
M:6YG<SL*("`@(')E<75I<F4@=F%R<SL*("`@(')E<75I<F4@0V%R<#L*("`@
M(')E<75I<F4@0V%R<#HZ2&5A=GD["B`@("!R97%U:7)E($5R<FYO.PH@("`@
M<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["B`@("!R97%U:7)E($5X<&]R=&5R
M.PH@("`@<F5Q=6ER92!&8VYT;#L*("`@(')E<75I<F4@1FEL93HZ5&5M<#L*
M("`@(')E<75I<F4@1FEL93HZ4W!E8SL*("`@(')E<75I<F4@6%-,;V%D97([
M"B`@("!R97%U:7)E($-O;F9I9SL*("`@(')E<75I<F4@24\Z.DAA;F1L93L*
M("`@(')E<75I<F4@24\Z.D9I;&4["B`@("!R97%U:7)E($-O;7!R97-S.CI:
M;&EB.PH@("`@<F5Q=6ER92!!<F-H:79E.CI::7`["B`@("!R97%U:7)E(%!!
M4CL*("`@(')E<75I<F4@4$%2.CI(96%V>3L*("`@(')E<75I<F4@4$%2.CI$
M:7-T.PH@("`@<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["B`@("!R
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["B`@("!R97%U:7)E
M(&%T=')I8G5T97,["B`@("!E=F%L('L@<F5Q=6ER92!#=V0@?3L*("`@(&5V
M86P@>R!R97%U:7)E(%=I;C,R('T["B`@("!E=F%L('L@<F5Q=6ER92!38V%L
M87(Z.E5T:6P@?3L*("`@(&5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP
M.CI"=7)S="!].PH@("`@979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE
M9$-A<'1U<F4@?3L*("`@(&5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER
M92!097)L24\Z.G-C86QA<B!].PI]"@HC(%1H92!#('9E<G-I;VX@;V8@=&AI
M<R!C;V1E(&%P<&5A<G,@:6X@;7EL9'(O;6MT;7!D:7(N8PHC(%1H:7,@8V]D
M92!A;'-O(&QI=F5S(&EN(%!!4CHZ4V5T=7!496UP(&%S('-E=%]P87)?=&5M
M<%]E;G8A"G-U8B!?<V5T7W!A<E]T96UP('L*("`@(&EF("AD969I;F5D("1%
M3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*
M("`@("`@("`D<&%R7W1E;7`@/2`D,3L*("`@("`@("!R971U<FX["B`@("!]
M"@H@("`@9F]R96%C:"!M>2`D<&%T:"`H"B`@("`@("`@*&UA<"`D14Y6>R1?
M?2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL
M"B`@("`@("`@<7<H($,Z7%Q414U0("]T;7`@+B`I"B`@("`I('L*("`@("`@
M("!N97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@
M+7<@)'!A=&@["B`@("`@("`@;7D@)'5S97)N86UE.PH@("`@("`@(&UY("1P
M=W5I9#L*("`@("`@("`C(&1O97,@;F]T('=O<FL@979E<GEW:&5R93H*("`@
M("`@("!E=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D
M("0^.WT["@H@("`@("`@(&EF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M
M92D@*2!["B`@("`@("`@("`@("1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.
M86UE.PH@("`@("`@('T*("`@("`@("!E;'-I9B`H9&5F:6YE9"`D<'=U:60I
M('L*("`@("`@("`@("`@)'5S97)N86UE(#T@)'!W=6ED.PH@("`@("`@('T*
M("`@("`@("!E;'-E('L*("`@("`@("`@("`@)'5S97)N86UE(#T@)$5.5GM5
M4T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*("`@("`@("!]
M"B`@("`@("`@)'5S97)N86UE(#U^(',O7%<O7R]G.PH*("`@("`@("!M>2`D
M<W1M<&1I<B`]("(D<&%T:"1#;VYF:6=[7V1E;&EM?7!A<BTD=7-E<FYA;64B
M.PH@("`@("`@(&UK9&ER("1S=&UP9&ER+"`P-S4U.PH@("`@("`@(&EF("@A
M)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I;64@/2`H<W1A="@D<')O9VYA
M;64I*5LY72D@>PH@("`@("`@("`@("!O<&5N("AM>2`D9F@L("(\(BX@)'!R
M;V=N86UE*3L*("`@("`@("`@("`@<V5E:R`D9F@L("TQ."P@,CL*("`@("`@
M("`@("`@<WES<F5A9"`D9F@L(&UY("1B=68L(#8["B`@("`@("`@("`@(&EF
M("@D8G5F(&5Q(")<,$-!0TA%(BD@>PH@("`@("`@("`@("`@("`@<V5E:R`D
M9F@L("TU."P@,CL*("`@("`@("`@("`@("`@('-Y<W)E860@)&9H+"`D8G5F
M+"`T,3L*("`@("`@("`@("`@("`@("1B=68@/7X@<R]<,"\O9SL*("`@("`@
M("`@("`@("`@("1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TB
M("X@)&)U9CL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-E('L*("`@
M("`@("`@("`@("`@(&UY("1C='@@/2!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z
M.E-(03L@1&EG97-T.CI32$$M/FYE=R@Q*2!]"B`@("`@("`@("`@("`@("`@
M("`@?'P@979A;"![(')E<75I<F4@1&EG97-T.CI32$$Q.R!$:6=E<W0Z.E-(
M03$M/FYE=R!]"B`@("`@("`@("`@("`@("`@("`@?'P@979A;"![(')E<75I
M<F4@1&EG97-T.CI-1#4[($1I9V5S=#HZ340U+3YN97<@?3L*"B`@("`@("`@
M("`@("`@("`C(%=O<FMA<F]U;F0@9F]R(&)U9R!I;B!$:6=E<W0Z.E-(02`U
M+C,X(&%N9"`U+C,Y"B`@("`@("`@("`@("`@("!M>2`D<VAA7W9E<G-I;VX@
M/2!E=F%L('L@)$1I9V5S=#HZ4TA!.CI615)324].('T@?'P@,#L*("`@("`@
M("`@("`@("`@(&EF("@D<VAA7W9E<G-I;VX@97$@)S4N,S@G(&]R("1S:&%?
M=F5R<VEO;B!E<2`G-2XS.2<I('L*("`@("`@("`@("`@("`@("`@("`D8W1X
M+3YA9&1F:6QE*"1P<F]G;F%M92P@(F(B*2!I9B`H)&-T>"D["B`@("`@("`@
M("`@("`@("!]"B`@("`@("`@("`@("`@("!E;'-E('L*("`@("`@("`@("`@
M("`@("`@("!I9B`H)&-T>"!A;F0@;W!E;BAM>2`D9F@L("(\)'!R;V=N86UE
M(BDI('L*("`@("`@("`@("`@("`@("`@("`@("`@8FEN;6]D92@D9F@I.PH@
M("`@("`@("`@("`@("`@("`@("`@("`D8W1X+3YA9&1F:6QE*"1F:"D["B`@
M("`@("`@("`@("`@("`@("`@("`@(&-L;W-E*"1F:"D["B`@("`@("`@("`@
M("`@("`@("`@?0H@("`@("`@("`@("`@("`@?0H*("`@("`@("`@("`@("`@
M("1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TB("X@*"`D8W1X
M(#\@)&-T>"T^:&5X9&EG97-T(#H@)&UT:6UE("D["B`@("`@("`@("`@('T*
M("`@("`@("`@("`@8VQO<V4H)&9H*3L*("`@("`@("!]"B`@("`@("`@96QS
M92!["B`@("`@("`@("`@("1%3E9[4$%27T-,14%.?2`](#$["B`@("`@("`@
M("`@("1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UT96UP+20D(CL*("`@
M("`@("!]"@H@("`@("`@("1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["B`@
M("`@("`@;6MD:7(@)'-T;7!D:7(L(#`W-34["B`@("`@("`@;&%S=#L*("`@
M('T*"B`@("`D<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@
M)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"@IS=6(@7W1E;7!F:6QE('L*
M("`@(&UY("@D97AT+"`D8W)C*2`]($!?.PH@("`@;7D@*"1F:"P@)&9I;&5N
M86UE*3L*"B`@("`D9FEL96YA;64@/2`B)'!A<E]T96UP+R1C<F,D97AT(CL*
M"B`@("!I9B`H)$5.5GM005)?0TQ%04Y]*2!["B`@("`@("`@=6YL:6YK("1F
M:6QE;F%M92!I9B`M92`D9FEL96YA;64["B`@("`@("`@<'5S:"!`=&UP9FEL
M92P@)&9I;&5N86UE.PH@("`@?0H@("`@96QS92!["B`@("`@("`@<F5T=7)N
M("AU;F1E9BP@)&9I;&5N86UE*2!I9B`H+7(@)&9I;&5N86UE*3L*("`@('T*
M"B`@("!O<&5N("1F:"P@)SXG+"`D9FEL96YA;64@;W(@9&EE("0A.PH@("`@
M8FEN;6]D92@D9F@I.PH@("`@<F5T=7)N*"1F:"P@)&9I;&5N86UE*3L*?0H*
M(R!S86UE(&-O9&4@;&EV97,@:6X@4$%2.CI3971U<%!R;V=N86UE.CIS971?
M<')O9VYA;64*<W5B(%]S971?<')O9VYA;64@>PH@("`@:68@*&1E9FEN960@
M)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@
M+R@N*RDO*2!["B`@("`@("`@)'!R;V=N86UE(#T@)#$["B`@("!]"@H@("`@
M)'!R;V=N86UE('Q\/2`D,#L*"B`@("!I9B`H)$5.5GM005)?5$5-4'T@86YD
M(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*("`@
M("`@("`D<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R
M;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI("L@,2D["B`@("!]"@H@("`@:68@
M*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N86UE+"`D0V]N
M9FEG>U]D96QI;7TI(#X](#`I('L*("`@("`@("!I9B`H;W!E;B!M>2`D9F@L
M("<\)RP@)'!R;V=N86UE*2!["B`@("`@("`@("`@(')E='5R;B!I9B`M<R`D
M9F@["B`@("`@("`@?0H@("`@("`@(&EF("@M<R`B)'!R;V=N86UE)$-O;F9I
M9WM?97AE?2(I('L*("`@("`@("`@("`@)'!R;V=N86UE("X]("1#;VYF:6=[
M7V5X97T["B`@("`@("`@("`@(')E='5R;CL*("`@("`@("!]"B`@("!]"@H@
M("`@9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D0V]N9FEG>W!A=&A?<V5P
M?5Q%+RP@)$5.5GM0051(?2D@>PH@("`@("`@(&YE>'0@:68@97AI<W1S("1%
M3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PH@("`@
M("`@("1D:7(@/7X@<R]<421#;VYF:6=[7V1E;&EM?5Q%)"\O.PH@("`@("`@
M("@H)'!R;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D
M0V]N9FEG>U]E>&5](BDL(&QA<W0I"B`@("`@("`@("`@(&EF("US("(D9&ER
M)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I9WM?97AE?2(["B`@("`@
M("`@*"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M
M92(I+"!L87-T*0H@("`@("`@("`@("!I9B`M<R`B)&1I<B1#;VYF:6=[7V1E
M;&EM?21P<F]G;F%M92(["B`@("!]"GT*"G-U8B!?9FEX7W!R;V=N86UE('L*
M("`@("0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["B`@
M("!I9B`H:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@
M>PH@("`@("`@("1P<F]G;F%M92`]("(N)$-O;F9I9WM?9&5L:6U])'!R;V=N
M86UE(CL*("`@('T*"B`@("`C(%A86"`M(&AA8VL@=&\@;6%K92!05T0@=V]R
M:PH@("`@;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z
M.F=E=&-W9"@I"B`@("`@("`@("`@("`@("`Z("@H9&5F:6YE9"`F5VEN,S(Z
M.D=E=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["B`@("!C:&]M
M<"@D<'=D*3L*("`@("1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)$-O;F9I
M9WM?9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PH*("`@("1%3E9[
M4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0H*<W5B(%]P87)?:6YI=%]E
M;G8@>PH@("`@:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]/2`Q("D@
M>PH@("`@("`@(')E='5R;CL*("`@('T@96QS92!["B`@("`@("`@)$5.5GM0
M05)?24Y)5$E!3$E:141](#T@,CL*("`@('T*"B`@("!F;W(@*'%W*"!34$%7
M3D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@05)'0R!!4D=6
M7S`@*2`I('L*("`@("`@("!D96QE=&4@)$5.5GLG4$%27R<N)%]].PH@("`@
M?0H@("`@9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PH@
M("`@("`@("1%3E9[)U!!4E\G+B1??2`]("1%3E9[)U!!4E]'3$]"04Q?)RXD
M7WT@:68@97AI<W1S("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT["B`@("!]"@H@
M("`@;7D@)'!A<E]C;&5A;B`](")?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@
M("`@("`@("(["@H@("`@:68@*"1%3E9[4$%27U1%35!]*2!["B`@("`@("`@
M9&5L971E("1%3E9[4$%27T-,14%.?3L*("`@('T*("`@(&5L<VEF("@A97AI
M<W1S("1%3E9[4$%27T=,3T)!3%]#3$5!3GTI('L*("`@("`@("!M>2`D=F%L
M=64@/2!S=6)S='(H)'!A<E]C;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.(BDI
M.PH@("`@("`@("1%3E9[4$%27T-,14%.?2`]("0Q(&EF("1V86QU92`]?B`O
M7E!!4E]#3$5!3CTH7%,K*2\["B`@("!]"GT*"G-U8B!O=71S('L*("`@(')E
M='5R;B!I9B`D<75I970["B`@("!I9B`H)&QO9V9H*2!["B`@("`@("`@<')I
M;G0@)&QO9V9H(")`7UQN(CL*("`@('T*("`@(&5L<V4@>PH@("`@("`@('!R
M:6YT(")`7UQN(CL*("`@('T*?0H*<W5B(&EN:71?:6YC('L*("`@(')E<75I
M<F4@0V]N9FEG.PH@("`@<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D
M0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*("`@("`@("!A<F-H;&EB97AP('!R
M:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`H@("`@("`@('9E;F1O
M<F%R8VAE>'`@=F5N9&]R;&EB97AP"B`@("`I.PI]"@HC(R,C(R,C(R,C(R,C
M(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C(R,C(R,C(R,C(R,*(R!4:&4@;6%I;B!P86-K86=E(&9O<B!S8W)I<'0@
M97AE8W5T:6]N"@IP86-K86=E(&UA:6X["@IR97%U:7)E(%!!4CL*=6YS:&EF
M="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G
M<RD["@ID:64@<7$H<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D
M<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&XI"B`@("!U
M;FQE<W,@+64@)'!R;V=N86UE.PH*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET
M*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@
M:68@)$`["@I].PH*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"@I#3U)%.CIE
M>&ET*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI
M+RD["F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["@HQ.PH*(VQI;F4@
M,3`Q,0H*7U]%3D1?7PH``````````!0``````````7I2``%X$`$;#`<(D`$`
M`$P````<````>&C__\L6````0@X01`X80@X@0@XH00XP00XX1(,'A@:,!8T$
MC@./`D<.R`0#J18..$$.,$$.*$(.($(.&$(.$$(."```````)````&P```#X
M?O__[`````!9A@:#!X\"C`5.#D!(C@.-!&(*#@A'"Q0```"4````P'___Q4`
M````1`X@4`X(`"P```"L````R'___U$`````00X00X8"00X81`XP18,#`D`.
M&$$.$$$."````````"P```#<````^'___\<!````00X00PX81`X@@P.&`@.\
M`0X800X000X(`````````"0````,`0``F('__XH!````38,&A@56#L`!18X"
MC0.,!'0*#@A$"P`T````-`$```"#__^H`````$$.$$*&`D$.&$.#`T8.(&P*
M#AA##A!!#@A#"P)C#AA!#A!##@@``"P```!L`0``>(/__U`"````68T$C`6&
M!H,'3@Y@3X\"C@,"<`H."$H+`````````!0```"<`0``F(7__RT`````1`X0
M9`X(`!0```"T`0``L(7__TH`````1`X0`D$.""0```#,`0``Z(7__]8`````
M3(8#@P1,#L!`6XP"`E0*#@A)"P`````4````]`$``*"&__\%````````````
M``!,````#`(``)B&___+`0```$(.$$(.&$(.($(.*$.,!8T$C@./`D$.,$.&
M!D8..$<.T"!3@P<"2@H..$$.,$$.*$(.($(.&$(.$$(."$$+`!0```!<`@``
M&(C__R0``````````````"P```!T`@``,(C__]H`````1`X00X,"?@H."$L+
M60H."$\+>@H."$8+80X(`````!P```"D`@``X(C__\L!````1`Y``VP!"@X(
M2`L`````'````,0"``"0BO__2`````!$#A!/"@X(10MO#@@````4````Y`(`
M`,"*__\B`````$0.$%T."`!$````_`(``-B*__\,`0```$(.$$(.&$(.($.,
M!(T#C@)!#BA!#C!%@P:&!40.<`*>"@XP00XH00X@0@X80@X00@X(1@L```!,
M````1`,``*"+__^]"0```$(.$$(.&$(.($(.*$$.,$$..$.#!X8&C`6-!(X#
MCP),#L"#`@,"`PH..$$.,$$.*$(.($(.&$(.$$(."$L+`$0```"4`P``$)7_
M_Z<$````0@X01`X80@X@00XH00XP1P[`(%.#!H8%C`2-`XX"`WD""@XP00XH
M00X@0@X80@X00@X(2PL``#0```#<`P``>)G__V\`````00X00X8"00X81`X@
M@P,"50H.&$$.$$$."$$+1`X81`X000X(````%````!0$``"PF?__0`````!$
M#A!W#@@`+````"P$``#8F?__*P(```!!#A!$#C!=@P(#A`$*#A!!#@A)"P``
M````````````%````%P$``#8F___'```````````````%````'0$``#@F___
M`@``````````````)````(P$``#8F___B0````!1C`6&!E\.0$:#!X\"C@.-
M!``````````````!&P,[2/O__QL```#D8___9/O__[1Z__^T^___I'O__]S[
M___$>___]/O__R1\__\D_/__]'W__U3\__^$?___?/S__S2`__^T_/__A(+_
M_^3\__^T@O___/S__P2#__\4_?__Y(/__SS]___T@___5/W__\2%__^D_?__
M](7__[S]___4AO__[/W__Z2(__\,_O__](C__RS^__\DB?__1/[__S2*__^,
M_O__])/__]S^__^DF/__)/___Q29__]<____5)G__W3___^$F___I/___Z2;
M__^\____M)O__]3___\#`````````.C50````````@````````!H!P``````
M`!<`````````:!E````````4``````````<`````````!P`````````@&4``
M``````@`````````2``````````)`````````!@`````````%0``````````
M``````````8`````````8`)````````+`````````!@`````````!0``````
M```H#4````````H`````````.08````````$`````````&@30```````]?[_
M;P````#`%D````````$`````````!@8````````!`````````!8&````````
M`0`````````A!@````````$`````````X0`````````!`````````&4`````
M`````0`````````K!@````````P`````````T"!````````-`````````'A?
M0```````'@```````````````````/#__V\`````Q!=```````#^__]O````
M`*P80```````____;P`````"````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````_B!````````.(4```````!XA0```````+B%`
M```````^(4```````$XA0```````7B%```````!N(4```````'XA0```````
MCB%```````">(4```````*XA0```````OB%```````#.(4```````-XA0```
M````[B%```````#^(4````````XB0```````'B)````````N(D```````#XB
M0```````3B)```````!>(D```````&XB0```````?B)```````".(D``````
M`)XB0```````KB)```````"^(D```````,XB0```````WB)```````#N(D``
M`````/XB0```````#B-````````>(T```````"XC0```````/B-```````!.
M(T```````%XC0```````;B-```````!^(T```````(XC0```````GB-`````
M``"N(T```````+XC0```````SB-```````#>(T```````.XC0```````_B-`
M```````.)$```````!XD0```````+B1````````^)$```````$XD0```````
M7B1```````!N)$```````'XD0```````CB1```````">)$```````*XD0```
M````OB1```````#.)$```````-XD0```````[B1```````#^)$````````XE
M0```````'B5````````N)4```````#XE0```````3B5```````!>)4``````
M`&XE0```````?B5```````".)4```````)XE0```````KB5```````"^)4``
M`````,XE0```````WB5`````````````````````````````_____P````#1
M84```````*!;````````__________\``````````/__________````````
M``````````````!'0T,Z("A$96)I86X@-"XU+C(M-RD@-"XU+C(`1T-#.B`H
M1&5B:6%N(#0N-"XU+3$S*2`T+C0N-0````0````)````!````$=.50!G;VQD
M(#$N,3$`````+G1E>'0`+F=O=``N8V]M;65N=``N9V]T+G!L=``N<F5L82YP
M;'0`+FEN:70`+F)S<P`N9'1O<G,`+F-T;W)S`"YD>6YS='(`+F5H7V9R86UE
M7VAD<@`N:F-R`"YG;G4N=F5R<VEO;E]R`"YI;G1E<G``+G)E;&$N9'EN`"YG
M;G4N=F5R<VEO;@`N;F]T92YG;G4N9V]L9"UV97)S:6]N`"YD>6YS>6T`+F9I
M;FD`+F=N=2YH87-H`"YN;W1E+D%"22UT86<`+F5H7V9R86UE`"YN;W1E+F=N
M=2YB=6EL9"UI9``N9'EN86UI8P`N<VAS=')T86(`+G)O9&%T80`N9&%T80``
M````````````````````````````````````````````````````````````
M````````````````````````````:P````$````"```````````"0```````
M``(````````<`````````````````````0```````````````````+D````'
M`````@`````````<`D```````!P"````````(`````````````````````0`
M``````````````````#1````!P````(`````````/`)````````\`@``````
M`"0````````````````````$````````````````````H0````L````"````
M`````&`"0```````8`(```````#("@````````4````!````"``````````8
M`````````$$````#`````@`````````H#4```````"@-````````.08`````
M``````````````$```````````````````"S````!0````(`````````:!-`
M``````!H$P```````%@#````````!``````````(``````````0`````````
MKP```/;__V\"`````````,`60```````P!8````````$`0````````0`````
M````"````````````````````'T```#___]O`@````````#$%T```````,07
M````````Y@`````````$``````````(``````````@````````!<````_O__
M;P(`````````K!A```````"L&````````'``````````!0````(````$````
M````````````````<P````0````"`````````"`90```````(!D```````!(
M``````````0`````````"``````````8`````````!X````$````0@``````
M``!H&4```````&@9````````:`<````````$````#0````@`````````&```
M```````H`````0````8`````````T"!```````#0(````````!@`````````
M```````````$````````````````````(P````$````&`````````.@@0```
M````Z"``````````!0``````````````````"``````````0``````````$`
M```!````!@````````#P)4```````/`E````````B#D`````````````````
M`!````````````````````"I`````0````8`````````>%]```````!X7P``
M``````X````````````````````$````````````````````]P````$````"
M`````````*!?0```````H%\```````"A7@``````````````````(```````
M`````````````,<````!`````@````````!(OD```````$B^````````M`0`
M``````````````````@```````````````````!)`````0````(`````````
M_,)```````#\P@```````.0````````````````````$````````````````
M````Y`````8````#`````````.#30```````X,,``````````@````````4`
M````````"``````````0``````````<````!`````P````````#@U4``````
M`.#%````````"`````````````````````@````````````````````5````
M`0````,`````````Z-5```````#HQ0```````)`"```````````````````(
M````````````````````_P````$````#`````````'C80```````>,@`````
M```H````````````````````"````````````````````#H````!`````P``
M``````"@V$```````*#(````````$`````````````````````@`````````
M```````````S`````0````,`````````L-A```````"PR````````!``````
M```````````````(````````````````````5P````$````#`````````,#8
M0```````P,@````````(````````````````````"```````````````````
M`"X````(`````P````````#@V$```````.#(````````@Q``````````````
M`````"`````````````````````,`````0```#````````````````````#(
MR````````#H````````````````````!``````````$`````````B@````<`
M````````````````````````!,D````````<````````````````````!```
M`````````````````.T````#`````````````````````````"#)````````
@!0$```````````````````$`````````````````````
