package Alien::Build::Plugin::Probe::CommandLine;

use strict;
use warnings;
use Alien::Build::Plugin;
use Carp ();
use Capture::Tiny qw( capture );

# ABSTRACT: Probe for tools or commands already available
our $VERSION = '0.02'; # VERSION

has '+command' => sub { Carp::croak "@{[ __PACKAGE__ ]} requires command property" };
has 'args'       => [];
has 'secondary' => 0;
has 'match'     => undef;
has 'version'   => undef;

sub init
{
  my($self, $meta) = @_;
  
  # in core as of 5.10, but still need it for 5.8
  # apparently.
  $meta->add_requires( 'configure' => 'IPC::Cmd' => 0 );
  
  my $check = sub {
    my($build) = @_;

    unless(IPC::Cmd::can_run($self->command))
    {
      die 'Command not found ' . $self->command;
    }

    if(defined $self->match || defined $self->version)
    {
      my($out,$err,$ret) = capture {
        system( $self->command, @{ $self->args } );
      };
      die 'Command did not return a true value' if $ret;
      die 'Command output did not match' if defined $self->match && $out !~ $self->match;
      if(defined $self->version)
      {
        if($out =~ $self->version)
        {
          $build->runtime_prop->{version} = $1;
        }
      }
    }

    $build->runtime_prop->{command} = $self->command;
    'system';
  };
  
  if($self->secondary)
  {
    $meta->around_hook(
      probe => sub {
        my $orig = shift;
        my $build = shift;
        my $type = $orig->($build, @_);
        return $type unless $type eq 'system';
        $check->($build);
      },
    );
  }
  else
  {
    $meta->register_hook(
      probe => $check,
    );
  }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Alien::Build::Plugin::Probe::CommandLine - Probe for tools or commands already available

=head1 VERSION

version 0.02

=head1 AUTHOR

Graham Ollis <plicease@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Graham Ollis.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
