#!/usr/bin/perl -w

use strict;
use CORBA::IDL::parser30;
use CORBA::IDL::symbtab;
# visitors
use CORBA::IDL::repos_id;
use CORBA::JAVA::name;
use CORBA::JAVA::literal;
use CORBA::JAVA::class;
use CORBA::JAVA::xml;
use CORBA::XMLSchemas::xsdname;

my $parser = new Parser;
$parser->YYData->{verbose_error} = 1;		# 0, 1
$parser->YYData->{verbose_warning} = 1;		# 0, 1
$parser->YYData->{verbose_info} = 1;		# 0, 1
$parser->YYData->{verbose_deprecated} = 0;	# 0, 1 (concerns only version '2.4' and upper)
$parser->YYData->{symbtab} = new CORBA::IDL::Symbtab($parser);
my $cflags = '-D__idl2javaxml';
if ($Parser::IDL_version lt '3.0') {
	$cflags .= ' -D_PRE_3_0_COMPILER_';
}
if ($^O eq 'MSWin32') {
	$parser->YYData->{preprocessor} = 'cpp -C ' . $cflags;
#	$parser->YYData->{preprocessor} = 'CL /E /C /nologo ' . $cflags;	# Microsoft VC
} else {
	$parser->YYData->{preprocessor} = 'cpp -C ' . $cflags;
}
$parser->getopts("hi:p:t:vx");
if ($parser->YYData->{opt_v}) {
	print "CORBA::JAVA $CORBA::JAVA::class::VERSION\n";
	print "CORBA::IDL $CORBA::IDL::node::VERSION\n";
	print "IDL $Parser::IDL_version\n";
	print "$0\n";
	print "Perl $] on $^O\n";
	exit;
}
if ($parser->YYData->{opt_h}) {
	use Pod::Usage;
	pod2usage(-verbose => 1);
}
$parser->YYData->{forward_constructed_forbidden} = 1;
$parser->Run(@ARGV);
$parser->YYData->{symbtab}->CheckForward();
$parser->YYData->{symbtab}->CheckRepositoryID();

if (exists $parser->YYData->{nb_error}) {
	my $nb = $parser->YYData->{nb_error};
	print "$nb error(s).\n"
}
if (        $parser->YYData->{verbose_warning}
		and exists $parser->YYData->{nb_warning} ) {
	my $nb = $parser->YYData->{nb_warning};
	print "$nb warning(s).\n"
}
if (        $parser->YYData->{verbose_info}
		and exists $parser->YYData->{nb_info} ) {
	my $nb = $parser->YYData->{nb_info};
	print "$nb info(s).\n"
}
if (        $parser->YYData->{verbose_deprecated}
		and exists $parser->YYData->{nb_deprecated} ) {
	my $nb = $parser->YYData->{nb_deprecated};
	print "$nb deprecated(s).\n"
}

if (        exists $parser->YYData->{root}
		and ! exists $parser->YYData->{nb_error} ) {
	$parser->YYData->{root}->visit(new CORBA::IDL::repositoryIdVisitor($parser));
	if (        $Parser::IDL_version ge '3.0'
			and $parser->YYData->{opt_x} ) {
		$parser->YYData->{symbtab}->Export();
	}
	$parser->YYData->{root}->visit(new CORBA::JAVA::nameVisitor($parser, $parser->YYData->{opt_p}, $parser->YYData->{opt_t}));
	$parser->YYData->{root}->visit(new CORBA::JAVA::literalVisitor($parser));
	$parser->YYData->{root}->visit(new CORBA::JAVA::name2Visitor($parser));
	$parser->YYData->{root}->visit(new CORBA::JAVA::nameXmlVisitor($parser));
	$parser->YYData->{root}->visit(new CORBA::XMLSchemas::nameVisitor($parser));
	$parser->YYData->{root}->visit(new CORBA::JAVA::XMLclassVisitor($parser));
}

__END__

=head1 NAME

idl2javaxml - IDL compiler to language Java mapping & XML binding

=head1 SYNOPSIS

idl2javaxml [options] I<spec>.idl

=head1 OPTIONS

All options are forwarded to C preprocessor, except -h -i -v -x.

With the GNU C Compatible Compiler Processor, useful options are :

=over 8

=item B<-D> I<name>

=item B<-D> I<name>=I<definition>

=item B<-I> I<directory>

=item B<-I->

=item B<-nostdinc>

=back

Specific options :

=over 8

=item B<-h>

Display help.

=item B<-i> I<directory>

Specify a path for import (only for IDL version 3.0).

=item B<-p> "I<m1>=I<prefix1>;..."

Specify a list of prefix (gives full qualified Java package names).

=item B<-t> "I<m1>=I<new.name1>;..."

Specify a list of name translation (gives  full qualified Java package names).

=item B<-v>

Display version.

=item B<-x>

Enable export (only for IDL version 3.0).

=back

=head1 DESCRIPTION

B<idl2javaxml> parses the given input file (IDL) and generates the same files
as B<idl2java> and a I<class>HelperXML.java file that contains
XML marshal/demarshal methods.
The XML binding follows the CORBA to WSDL/SOAP Interworking Specification (WS-I comformant soap binding).

B<idl2javaxml> is a Perl OO application what uses the visitor design pattern.
The parser is generated by Parse::Yapp.

B<idl2javaxml> needs a B<cpp> executable.

CORBA Specifications, including IDL (Interface Language Definition) ,
the JAVA Language Mapping and CORBA to WSDL/SOAP Interworking
are available on E<lt>http://www.omg.org/E<gt>.

=head1 EXAMPLE

 // IDL
 struct StructType {
     long    field1;
     string  field2;
 };

 // Java
 public class test
 {
   public static void main (String[] args) throws Exception
   {
     XMLOutputStreamImpl os = new XMLOutputStreamImpl (new FileOutputStream ("out.xml"));
     StructType obj = new StructType (1, "toto");
     StructTypeHelperXML.write (os, obj);
     os.close ();
   }
 }

The class StructType and StructTypeHelperXML are generated by B<idl2javaxml>.

The class XMLOutputStreamImpl and XMLInputStreamImpl are an example of the run-time support;
there derive from XMLOutputStream ans XMLInputStream that define the interface.

 // XML
 <StructType><field1>1</field1><field2>toto</field2></StructType>

This XML document is valid against the Schema generated by B<idl2xsd>
(Do you really want write by hand W3C Schema ?).

 // XSD
 <xs:schema targetNamespace="http://www.omg.org/IDL-Mapped/" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:tns="http://www.omg.org/IDL-Mapped/" elementFormDefault="qualified" attributeFormDefault="unqualified">
   <xs:complexType name="StructType">
     <xs:sequence>
       <xs:element name="field1" maxOccurs="1" minOccurs="1" type="xs:int"/>
       <xs:element name="field2" maxOccurs="1" minOccurs="1" type="xs:string"/>
     </xs:sequence>
   </xs:complexType>
 </xs:schema>

The class PYXOutputStreamImpl and PYXInputStreamImpl are an example of
the run-time support for Pyxie format (see E<lt>http://www.pyxie.org/E<gt>);
there derive from XMLOutputStream and XMLInputStream.

 // PYX
 (StructType
 (field1
 -1
 )field1
 (field2
 -toto
 )field2
 )StructType

=head1 SEE ALSO

cpp, idl2html, idl2java, idl2xsd, idl2rng

=head1 COPYRIGHT

(c) 2004 Francois PERRAD, France. All rights reserved.

This program and all CORBA::JAVA modules and all Java class (run-time support)
are distributed under the terms of the Artistic Licence.

=head1 AUTHOR

Francois PERRAD, francois.perrad@gadz.org

=cut

