package Bencher::Scenario::Serializers;

our $DATE = '2016-03-26'; # DATE
our $VERSION = '0.12'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::Tiny',
            function => 'encode_json',
            code_template => 'JSON::Tiny::encode_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::Tiny',
            function => 'decode_json',
            code_template => 'JSON::Tiny::decode_json(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'PERLANCAR::JSON::Match::match_json(<data>)',
            include_by_default => 0,
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 characters/bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 characters (3072-bytes) long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-characters/bytes long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.12 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-03-26.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<JSON::PP> 2.27300

L<JSON::Tiny> 0.54

L<JSON::XS> 3.01

L<Cpanel::JSON::XS> 3.0115

L<JSON::MaybeXS> 1.003005

L<JSON::Decode::Regexp> 0.07

L<JSON::Decode::Marpa> 0.02

L<Pegex::JSON> 0.27

L<JSON::Create> 0.17

L<JSON::Parse> 0.39

L<YAML::Old> 1.07

L<YAML::Syck> 1.29

L<YAML::XS> 0.59

L<Storable> 2.53

L<Sereal> 3.005

L<Data::MessagePack> 0.48

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::Tiny::encode_json (perl_code) [json, serialize]

Code template:

 JSON::Tiny::encode_json(<data>)



=item * JSON::Tiny::decode_json (perl_code) [json, deserialize]

Code template:

 JSON::Tiny::decode_json(<data>)



=item * JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code) [json, deserialize]

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * PERLANCAR::JSON::Match::match_json (perl_code) (not included by default) [json, deserialize, cant_handle_scalar]

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code) [json, deserialize]

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * Pegex::JSON (perl_code) [json, serialize]

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Create::create_json (perl_code) [json, deserialize]

Function call template:

 JSON::Parse::parse_json(<data>)



=item * JSON::Parse::parse_json (perl_code) [yaml, serialize]

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Old::Load (perl_code) [yaml, serialize]

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::Syck::Load (perl_code) [yaml, serialize]

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::XS::Load(<data>)



=item * YAML::XS::Load (perl_code) [binary, serialize, cant_handle_scalar]

Function call template:

 Storable::freeze(<data>)



=item * Storable::freeze (perl_code) [binary, deserialize, cant_handle_scalar]

Function call template:

 Storable::thaw(<data>)



=item * Storable::thaw (perl_code) [binary, serialize]

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::encode_sereal (perl_code) [binary, deserialize]

Function call template:

 Sereal::decode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code) [binary, serialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::pack (perl_code) [binary, deserialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=item * Data::MessagePack::unpack (perl_code)

L<Data::MessagePack>::unpack



=back

=head1 BENCHMARK DATASETS

=over

=item * undef [serialize]

undef

=item * num [serialize]

A single number (-1.23)

=item * str1k [serialize]

A non-Unicode string 1024 characters/bytes long

=item * str1k [serialize, unicode]

A Unicode string 1024 characters (3072-bytes) long

=item * array_int_10 [serialize]

A 10-element array containing ints

=item * array_int_100 [serialize]

A 100-element array containing ints

=item * array_int_1000 [serialize]

A 1000-element array containing ints

=item * array_str1k_10 [serialize]

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * array_ustr1k_10 [serialize, json]

A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings

=item * hash_int_10 [serialize]

A 10-key hash {1=>0, ..., 10=>0}

=item * hash_int_100 [serialize]

A 100-key hash {1=>0, ..., 100=>0}

=item * hash_int_1000 [serialize]

A 1000-key hash {1=>0, ..., 1000=>0}

=item * json:null [deserialize]

null

=item * json:num [deserialize]

A single number (-1.23)

=item * json:str1k [deserialize]

A non-Unicode (ASCII) string 1024-characters/bytes long

=item * json:array_int_10 [deserialize]

A 10-element array containing ints

=item * json:array_int_100 [deserialize]

A 10-element array containing ints

=item * json:array_int_1000 [deserialize]

A 1000-element array containing ints

=item * json:array_str1k_10 [deserialize]

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * json:hash_int_10 [deserialize]

A 10-key hash {"1":0, ..., "10":0}

=item * json:hash_int_100 [deserialize]

A 100-key hash {"1":0, ..., "100":0}

=item * json:hash_int_1000 [deserialize]

A 1000-key hash {"1":0, ..., "1000":0}

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark serializing (C<< bencher -m Serializers --include-participant-tags serialize >>):

 +-------------------------------+-----------------+-------------+-------------+------------+---------+---------+
 | participant                   | dataset         | rate (/s)   | time (ms)   | vs_slowest | errors  | samples |
 +-------------------------------+-----------------+-------------+-------------+------------+---------+---------+
 | YAML::Old::Dump               | hash_int_1000   | 47          | 21          | 1          | 5.1e-05 | 20      |
 | YAML::Old::Dump               | array_int_1000  | 62.8        | 15.9        | 1.35       | 1.3e-05 | 20      |
 | JSON::PP::encode_json         | hash_int_1000   | 410         | 2.44        | 8.81       | 2e-06   | 20      |
 | YAML::Old::Dump               | hash_int_100    | 454         | 2.2         | 9.76       | 1.8e-06 | 20      |
 | YAML::Old::Dump               | array_int_100   | 606         | 1.65        | 13         | 9.1e-07 | 20      |
 | JSON::Tiny::encode_json       | hash_int_1000   | 737         | 1.36        | 15.8       | 6.8e-07 | 21      |
 | YAML::Syck::Dump              | hash_int_1000   | 927         | 1.08        | 19.9       | 6.9e-07 | 20      |
 | YAML::XS::Dump                | hash_int_1000   | 928         | 1.08        | 20         | 4.7e-07 | 21      |
 | JSON::PP::encode_json         | array_int_1000  | 1.03e+03    | 0.972       | 22.1       | 2.1e-07 | 21      |
 | YAML::Old::Dump               | array_ustr1k_10 | 1.36e+03    | 0.734       | 29.3       | 2e-07   | 22      |
 | JSON::Tiny::encode_json       | array_int_1000  | 1.38e+03    | 0.727       | 29.6       | 2.1e-07 | 20      |
 | JSON::Tiny::encode_json       | array_ustr1k_10 | 1.76e+03    | 0.569       | 37.8       | 2.1e-07 | 21      |
 | YAML::Syck::Dump              | array_int_1000  | 2.2e+03     | 0.46        | 47         | 6.9e-07 | 20      |
 | YAML::XS::Dump                | array_int_1000  | 2.27e+03    | 0.441       | 48.7       | 1.6e-07 | 20      |
 | JSON::PP::encode_json         | array_ustr1k_10 | 2.67e+03    | 0.374       | 57.4       | 4.9e-08 | 24      |
 | YAML::Old::Dump               | array_str1k_10  | 3.4e+03     | 0.29        | 73         | 6.4e-07 | 20      |
 | YAML::Old::Dump               | hash_int_10     | 3.65e+03    | 0.274       | 78.4       | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | hash_int_100    | 4e+03       | 0.25        | 87         | 4.2e-07 | 21      |
 | YAML::Old::Dump               | array_int_10    | 4.62e+03    | 0.217       | 99.2       | 2.1e-07 | 20      |
 | Sereal::encode_sereal         | hash_int_1000   | 4.8e+03     | 0.21        | 1e+02      | 2.1e-07 | 20      |
 | YAML::XS::Dump                | array_ustr1k_10 | 4.92e+03    | 0.203       | 106        | 1.6e-07 | 20      |
 | JSON::MaybeXS::encode_json    | array_ustr1k_10 | 5.08e+03    | 0.197       | 109        | 5.3e-08 | 20      |
 | JSON::XS::encode_json         | array_ustr1k_10 | 5.3e+03     | 0.19        | 1.1e+02    | 2.1e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | array_ustr1k_10 | 5.2e+03     | 0.19        | 1.1e+02    | 2.1e-07 | 20      |
 | Storable::freeze              | hash_int_1000   | 5.8e+03     | 0.17        | 1.2e+02    | 2.7e-07 | 20      |
 | Data::MessagePack::pack       | hash_int_1000   | 5.8e+03     | 0.17        | 1.2e+02    | 2.6e-07 | 21      |
 | JSON::XS::encode_json         | hash_int_1000   | 6.4e+03     | 0.16        | 1.4e+02    | 2.1e-07 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_1000   | 6.4e+03     | 0.16        | 1.4e+02    | 4.3e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_1000   | 6.5e+03     | 0.15        | 1.4e+02    | 4.3e-07 | 20      |
 | JSON::Create::create_json     | hash_int_1000   | 6.6e+03     | 0.15        | 1.4e+02    | 2.1e-07 | 21      |
 | JSON::Tiny::encode_json       | hash_int_100    | 7e+03       | 0.14        | 1.5e+02    | 2.7e-07 | 20      |
 | YAML::XS::Dump                | hash_int_100    | 9.76e+03    | 0.102       | 210        | 5.3e-08 | 20      |
 | YAML::Syck::Dump              | hash_int_100    | 9.5e+03     | 0.1         | 2e+02      | 2.1e-07 | 21      |
 | JSON::PP::encode_json         | array_int_100   | 1.01e+04    | 0.0994      | 216        | 2.5e-08 | 22      |
 | YAML::XS::Dump                | array_str1k_10  | 1.24e+04    | 0.0807      | 266        | 2.1e-08 | 31      |
 | JSON::Tiny::encode_json       | array_str1k_10  | 1.2e+04     | 0.08        | 2.7e+02    | 1.1e-07 | 20      |
 | JSON::Tiny::encode_json       | array_int_100   | 1.38e+04    | 0.0727      | 296        | 2.7e-08 | 20      |
 | YAML::Syck::Dump              | array_ustr1k_10 | 1.47e+04    | 0.0678      | 317        | 2.1e-08 | 33      |
 | JSON::PP::encode_json         | array_str1k_10  | 15507.5     | 0.064485    | 333.287    | 5.7e-11 | 20      |
 | JSON::Create::create_json     | array_ustr1k_10 | 2.1e+04     | 0.048       | 4.5e+02    | 5.3e-08 | 20      |
 | Storable::freeze              | array_int_1000  | 2.12e+04    | 0.0472      | 455        | 4.5e-08 | 28      |
 | YAML::Old::Dump               | str1k           | 2.1e+04     | 0.047       | 4.6e+02    | 5.2e-08 | 21      |
 | YAML::XS::Dump                | array_int_100   | 2.2e+04     | 0.046       | 4.7e+02    | 5.3e-08 | 20      |
 | YAML::Syck::Dump              | array_int_100   | 2.2e+04     | 0.045       | 4.7e+02    | 5.1e-08 | 22      |
 | YAML::Old::Dump               | num             | 2.3e+04     | 0.043       | 4.9e+02    | 5.2e-08 | 21      |
 | YAML::Old::Dump               | undef           | 2.6e+04     | 0.038       | 5.7e+02    | 4.6e-08 | 27      |
 | Cpanel::JSON::XS::encode_json | array_int_1000  | 2.83e+04    | 0.0353      | 608        | 1.2e-08 | 26      |
 | JSON::MaybeXS::encode_json    | array_int_1000  | 2.85e+04    | 0.0351      | 613        | 1.2e-08 | 25      |
 | JSON::XS::encode_json         | array_int_1000  | 28686.6     | 0.0348595   | 616.534    | 0       | 20      |
 | JSON::Create::create_json     | array_int_1000  | 3.15e+04    | 0.0318      | 676        | 1.2e-08 | 24      |
 | JSON::PP::encode_json         | hash_int_10     | 36942.5     | 0.0270691   | 793.968    | 0       | 20      |
 | YAML::Syck::Dump              | array_str1k_10  | 3.9e+04     | 0.026       | 8.3e+02    | 5e-08   | 23      |
 | Data::MessagePack::pack       | array_int_1000  | 38872       | 0.025725    | 835.44     | 4.2e-11 | 20      |
 | Sereal::encode_sereal         | array_int_1000  | 4.08e+04    | 0.0245      | 877        | 6.4e-09 | 22      |
 | JSON::Create::create_json     | array_str1k_10  | 48076.4     | 0.0208002   | 1033.26    | 0       | 20      |
 | JSON::MaybeXS::encode_json    | array_str1k_10  | 56142.2     | 0.0178119   | 1206.61    | 1.1e-11 | 20      |
 | JSON::XS::encode_json         | array_str1k_10  | 56490       | 0.0177023   | 1214.08    | 0       | 20      |
 | Cpanel::JSON::XS::encode_json | array_str1k_10  | 56509       | 0.0176963   | 1214.49    | 0       | 20      |
 | JSON::Tiny::encode_json       | hash_int_10     | 6e+04       | 0.017       | 1.3e+03    | 2.6e-08 | 21      |
 | Storable::freeze              | hash_int_100    | 6.3e+04     | 0.016       | 1.4e+03    | 2.7e-08 | 20      |
 | Data::MessagePack::pack       | hash_int_100    | 6.5e+04     | 0.015       | 1.4e+03    | 2.7e-08 | 20      |
 | Sereal::encode_sereal         | hash_int_100    | 7.2e+04     | 0.014       | 1.5e+03    | 1.1e-07 | 20      |
 | JSON::XS::encode_json         | hash_int_100    | 7.6e+04     | 0.013       | 1.6e+03    | 2e-08   | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_100    | 7.8e+04     | 0.013       | 1.7e+03    | 2.6e-08 | 21      |
 | JSON::MaybeXS::encode_json    | hash_int_100    | 7.9e+04     | 0.013       | 1.7e+03    | 2.7e-08 | 20      |
 | YAML::Syck::Dump              | hash_int_10     | 7.9e+04     | 0.013       | 1.7e+03    | 2.5e-08 | 23      |
 | JSON::PP::encode_json         | array_int_10    | 8e+04       | 0.012       | 1.7e+03    | 1.3e-08 | 20      |
 | JSON::Create::create_json     | hash_int_100    | 8.54e+04    | 0.0117      | 1.83e+03   | 9.5e-09 | 22      |
 | YAML::XS::Dump                | hash_int_10     | 9.2e+04     | 0.011       | 2e+03      | 1.7e-08 | 20      |
 | JSON::Tiny::encode_json       | array_int_10    | 1.02e+05    | 0.00978     | 2.2e+03    | 3.3e-09 | 20      |
 | YAML::XS::Dump                | str1k           | 105944      | 0.00943892  | 2276.96    | 0       | 21      |
 | JSON::Tiny::encode_json       | str1k           | 1.06e+05    | 0.00942     | 2.28e+03   | 3.3e-09 | 20      |
 | YAML::Syck::Dump              | array_int_10    | 1.3e+05     | 0.0079      | 2.7e+03    | 9.8e-09 | 21      |
 | JSON::PP::encode_json         | str1k           | 1.4207e+05  | 0.0070388   | 3053.4     | 1.1e-11 | 22      |
 | YAML::XS::Dump                | array_int_10    | 1.6e+05     | 0.0063      | 3.4e+03    | 1.3e-08 | 21      |
 | Storable::freeze              | array_int_100   | 1.7e+05     | 0.0059      | 3.6e+03    | 6.7e-09 | 20      |
 | Storable::freeze              | array_ustr1k_10 | 1.8e+05     | 0.0055      | 3.9e+03    | 6.7e-09 | 20      |
 | YAML::Syck::Dump              | str1k           | 1.83e+05    | 0.00547     | 3.93e+03   | 1.5e-09 | 24      |
 | YAML::Syck::Dump              | num             | 2.3e+05     | 0.00435     | 4.95e+03   | 1.5e-09 | 25      |
 | Sereal::encode_sereal         | array_ustr1k_10 | 2.4142e+05  | 0.0041422   | 5188.6     | 1.1e-11 | 20      |
 | Storable::freeze              | hash_int_10     | 2.53e+05    | 0.00396     | 5.43e+03   | 1.6e-09 | 22      |
 | JSON::MaybeXS::encode_json    | array_int_100   | 2.6e+05     | 0.0039      | 5.5e+03    | 5e-09   | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_100   | 2.6e+05     | 0.0038      | 5.6e+03    | 6e-09   | 25      |
 | JSON::XS::encode_json         | array_int_100   | 2.7e+05     | 0.0037      | 5.8e+03    | 6.2e-09 | 23      |
 | YAML::Syck::Dump              | undef           | 2.8e+05     | 0.0036      | 5.9e+03    | 6.7e-09 | 20      |
 | Storable::freeze              | array_str1k_10  | 2.7e+05     | 0.0036      | 5.9e+03    | 5.2e-09 | 33      |
 | JSON::Tiny::encode_json       | num             | 2.9e+05     | 0.0034      | 6.3e+03    | 6.5e-09 | 21      |
 | Data::MessagePack::pack       | array_ustr1k_10 | 3e+05       | 0.0033      | 6.5e+03    | 5.5e-09 | 29      |
 | JSON::Create::create_json     | array_int_100   | 3.4e+05     | 0.003       | 7.3e+03    | 3.3e-09 | 20      |
 | Data::MessagePack::pack       | array_int_100   | 3.59e+05    | 0.00279     | 7.71e+03   | 2.5e-09 | 20      |
 | Sereal::encode_sereal         | array_int_100   | 3.7e+05     | 0.0027      | 8e+03      | 3.3e-09 | 20      |
 | Storable::freeze              | array_int_10    | 385759      | 0.00259229  | 8290.75    | 0       | 20      |
 | JSON::PP::encode_json         | num             | 4e+05       | 0.0025      | 8.5e+03    | 3.3e-09 | 20      |
 | YAML::XS::Dump                | num             | 400463      | 0.00249711  | 8606.76    | 0       | 20      |
 | Data::MessagePack::pack       | array_str1k_10  | 4.62e+05    | 0.00216     | 9.94e+03   | 6.8e-10 | 30      |
 | JSON::Create::create_json     | str1k           | 4.78e+05    | 0.00209     | 1.03e+04   | 8.3e-10 | 20      |
 | JSON::Tiny::encode_json       | undef           | 5.13e+05    | 0.00195     | 1.1e+04    | 7e-10   | 28      |
 | Cpanel::JSON::XS::encode_json | str1k           | 541180      | 0.00184781  | 11631.1    | 0       | 34      |
 | JSON::MaybeXS::encode_json    | str1k           | 5.4278e+05  | 0.0018424   | 11665      | 1.2e-11 | 20      |
 | JSON::XS::encode_json         | str1k           | 5.43e+05    | 0.00184     | 1.17e+04   | 8.3e-10 | 20      |
 | JSON::PP::encode_json         | undef           | 560210      | 0.00178505  | 12040      | 0       | 28      |
 | Sereal::encode_sereal         | array_str1k_10  | 5.91e+05    | 0.00169     | 1.27e+04   | 7.5e-10 | 25      |
 | Data::MessagePack::pack       | hash_int_10     | 6.1e+05     | 0.0016      | 1.3e+04    | 3.3e-09 | 21      |
 | YAML::XS::Dump                | undef           | 6.5e+05     | 0.0015      | 1.4e+04    | 3.3e-09 | 20      |
 | Sereal::encode_sereal         | hash_int_10     | 6.5e+05     | 0.0015      | 1.4e+04    | 3.3e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_10     | 6.96e+05    | 0.00144     | 1.49e+04   | 1.2e-09 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_10     | 7.02e+05    | 0.00142     | 1.51e+04   | 3.9e-10 | 23      |
 | JSON::XS::encode_json         | hash_int_10     | 7.1e+05     | 0.0014      | 1.5e+04    | 1.7e-09 | 20      |
 | JSON::Create::create_json     | hash_int_10     | 7.8e+05     | 0.0013      | 1.7e+04    | 1.7e-09 | 20      |
 | Sereal::encode_sereal         | array_int_10    | 1.5e+06     | 0.00066     | 3.2e+04    | 8.3e-10 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_10    | 1.7e+06     | 0.00059     | 3.6e+04    | 6.2e-10 | 20      |
 | JSON::MaybeXS::encode_json    | array_int_10    | 1.7e+06     | 0.00059     | 3.6e+04    | 6.2e-10 | 20      |
 | JSON::XS::encode_json         | array_int_10    | 1.8e+06     | 0.00056     | 3.8e+04    | 7.2e-10 | 27      |
 | JSON::Create::create_json     | array_int_10    | 1.9e+06     | 0.00053     | 4.1e+04    | 8.3e-10 | 20      |
 | Data::MessagePack::pack       | array_int_10    | 1.91e+06    | 0.000523    | 4.11e+04   | 2.7e-10 | 25      |
 | Cpanel::JSON::XS::encode_json | num             | 2.2e+06     | 0.00045     | 4.8e+04    | 8.3e-10 | 20      |
 | JSON::MaybeXS::encode_json    | num             | 2.2e+06     | 0.00045     | 4.8e+04    | 8.3e-10 | 20      |
 | JSON::XS::encode_json         | num             | 2.25e+06    | 0.000444    | 4.84e+04   | 2.1e-10 | 20      |
 | Sereal::encode_sereal         | str1k           | 2.5e+06     | 0.0004      | 5.4e+04    | 8.3e-10 | 20      |
 | Sereal::encode_sereal         | num             | 3.05e+06    | 0.000328    | 6.55e+04   | 3.1e-10 | 20      |
 | JSON::Create::create_json     | num             | 3.19518e+06 | 0.000312971 | 68671      | 0       | 20      |
 | Sereal::encode_sereal         | undef           | 3.5e+06     | 0.00029     | 7.4e+04    | 4.5e-10 | 27      |
 | Data::MessagePack::pack       | str1k           | 3.6e+06     | 0.00028     | 7.8e+04    | 3.2e-10 | 35      |
 | Data::MessagePack::pack       | num             | 5.44e+06    | 0.000184    | 1.17e+05   | 1.4e-10 | 20      |
 | JSON::MaybeXS::encode_json    | undef           | 6.3e+06     | 0.00016     | 1.3e+05    | 4.2e-10 | 20      |
 | Data::MessagePack::pack       | undef           | 6.2e+06     | 0.00016     | 1.3e+05    | 2.6e-10 | 31      |
 | Cpanel::JSON::XS::encode_json | undef           | 6.4586e+06  | 0.000154832 | 138808     | 0       | 20      |
 | JSON::XS::encode_json         | undef           | 6.54e+06    | 0.000153    | 1.41e+05   | 1.5e-10 | 20      |
 | JSON::Create::create_json     | undef           | 7.88e+06    | 0.000127    | 1.69e+05   | 5.1e-11 | 21      |
 +-------------------------------+-----------------+-------------+-------------+------------+---------+---------+


Benchmark deserializing (C<< bencher -m Serializers --include-participant-tags deserialize >>):

 +------------------------------------+---------------------+-------------+-------------+------------+---------+---------+
 | participant                        | dataset             | rate (/s)   | time (ms)   | vs_slowest | errors  | samples |
 +------------------------------------+---------------------+-------------+-------------+------------+---------+---------+
 | Pegex::JSON                        | json:hash_int_1000  | 31.2        | 32.1        | 1          | 1.8e-05 | 20      |
 | JSON::Decode::Marpa::from_json     | json:hash_int_1000  | 32          | 31          | 1          | 9.1e-05 | 20      |
 | Pegex::JSON                        | json:array_int_1000 | 54          | 18          | 1.7        | 2.2e-05 | 23      |
 | JSON::Decode::Marpa::from_json     | json:array_int_1000 | 68          | 15          | 2.2        | 1.8e-05 | 22      |
 | JSON::Decode::Marpa::from_json     | json:array_str1k_10 | 140         | 7.17        | 4.48       | 1.8e-06 | 20      |
 | JSON::Decode::Regexp::from_json    | json:hash_int_1000  | 148         | 6.77        | 4.74       | 4.7e-06 | 20      |
 | JSON::PP::decode_json              | json:hash_int_1000  | 169         | 5.93        | 5.41       | 4.5e-06 | 20      |
 | JSON::PP::decode_json              | json:array_str1k_10 | 226         | 4.42        | 7.26       | 3.1e-06 | 20      |
 | JSON::Decode::Marpa::from_json     | json:hash_int_100   | 286         | 3.5         | 9.18       | 2.2e-06 | 20      |
 | Pegex::JSON                        | json:hash_int_100   | 291         | 3.43        | 9.35       | 2.7e-06 | 20      |
 | JSON::PP::decode_json              | json:array_int_1000 | 336         | 2.98        | 10.8       | 1.1e-06 | 20      |
 | JSON::Tiny::decode_json            | json:hash_int_1000  | 3.9e+02     | 2.6         | 12         | 3.1e-06 | 20      |
 | Pegex::JSON                        | json:array_int_100  | 5e+02       | 2           | 16         | 2.7e-06 | 20      |
 | JSON::Decode::Marpa::from_json     | json:array_int_100  | 5.5e+02     | 1.8         | 18         | 2.3e-06 | 22      |
 | JSON::Decode::Regexp::from_json    | json:array_int_1000 | 655         | 1.53        | 21         | 1.1e-06 | 22      |
 | PERLANCAR::JSON::Match::match_json | json:hash_int_1000  | 728         | 1.37        | 23.4       | 1.1e-06 | 20      |
 | JSON::Tiny::decode_json            | json:array_int_1000 | 880         | 1.14        | 28.2       | 6.8e-07 | 21      |
 | Pegex::JSON                        | json:array_str1k_10 | 1.11e+03    | 0.905       | 35.5       | 3.7e-07 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:array_int_1000 | 1.24e+03    | 0.804       | 39.9       | 2.1e-07 | 20      |
 | JSON::Decode::Marpa::from_json     | json:hash_int_10    | 1.25e+03    | 0.801       | 40.1       | 4.3e-07 | 20      |
 | JSON::Decode::Regexp::from_json    | json:hash_int_100   | 1.53e+03    | 0.652       | 49.2       | 4.8e-07 | 20      |
 | JSON::Decode::Marpa::from_json     | json:array_int_10   | 1.66e+03    | 0.603       | 53.2       | 2.1e-07 | 20      |
 | Pegex::JSON                        | json:hash_int_10    | 1.7e+03     | 0.6         | 54         | 6.4e-07 | 20      |
 | JSON::Tiny::decode_json            | json:array_str1k_10 | 1.8e+03     | 0.555       | 57.8       | 4.1e-07 | 22      |
 | JSON::PP::decode_json              | json:hash_int_100   | 1.85e+03    | 0.54        | 59.4       | 4.7e-07 | 21      |
 | Pegex::JSON                        | json:array_int_10   | 2.2e+03     | 0.454       | 70.6       | 2.1e-07 | 20      |
 | JSON::PP::decode_json              | json:str1k          | 2.25e+03    | 0.444       | 72.3       | 2.1e-07 | 20      |
 | Pegex::JSON                        | json:str1k          | 3.29e+03    | 0.304       | 106        | 2.7e-07 | 20      |
 | Pegex::JSON                        | json:null           | 3.67e+03    | 0.273       | 118        | 2.1e-07 | 20      |
 | JSON::PP::decode_json              | json:array_int_100  | 3.74e+03    | 0.267       | 120        | 2.1e-07 | 20      |
 | Pegex::JSON                        | json:num            | 3.76e+03    | 0.266       | 121        | 2.1e-07 | 20      |
 | JSON::Tiny::decode_json            | json:hash_int_100   | 3.9e+03     | 0.26        | 1.2e+02    | 4.2e-07 | 21      |
 | JSON::Decode::Regexp::from_json    | json:array_int_100  | 6475.45     | 0.154429    | 207.796    | 0       | 20      |
 | PERLANCAR::JSON::Match::match_json | json:hash_int_100   | 7.4e+03     | 0.14        | 2.4e+02    | 1.6e-07 | 20      |
 | JSON::Tiny::decode_json            | json:array_int_100  | 8.2e+03     | 0.12        | 2.6e+02    | 4.8e-07 | 20      |
 | JSON::Decode::Regexp::from_json    | json:array_str1k_10 | 8.3e+03     | 0.12        | 2.7e+02    | 2.1e-07 | 20      |
 | JSON::Parse::parse_json            | json:hash_int_1000  | 9.31e+03    | 0.107       | 299        | 5.3e-08 | 20      |
 | JSON::XS::decode_json              | json:hash_int_1000  | 9.6e+03     | 0.104       | 308        | 4.9e-08 | 24      |
 | JSON::MaybeXS::decode_json         | json:hash_int_1000  | 9671.47     | 0.103397    | 310.355    | 0       | 22      |
 | Cpanel::JSON::XS::decode_json      | json:hash_int_1000  | 9.7e+03     | 0.1         | 3.1e+02    | 2.1e-07 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:array_int_100  | 1.2e+04     | 0.082       | 3.9e+02    | 1.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json    | json:hash_int_10    | 1.5e+04     | 0.068       | 4.7e+02    | 1.1e-07 | 20      |
 | JSON::PP::decode_json              | json:hash_int_10    | 1.8e+04     | 0.056       | 5.8e+02    | 8e-08   | 20      |
 | JSON::Tiny::decode_json            | json:str1k          | 1.8e+04     | 0.0556      | 577        | 2.7e-08 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:array_str1k_10 | 2.1e+04     | 0.048       | 6.7e+02    | 5.3e-08 | 20      |
 | JSON::Tiny::decode_json            | json:hash_int_10    | 32152.9     | 0.0311014   | 1031.78    | 0       | 20      |
 | JSON::PP::decode_json              | json:array_int_10   | 3.6e+04     | 0.028       | 1.2e+03    | 3.8e-08 | 22      |
 | JSON::XS::decode_json              | json:array_int_1000 | 4.75e+04    | 0.0211      | 1.52e+03   | 5.4e-09 | 30      |
 | Cpanel::JSON::XS::decode_json      | json:array_int_1000 | 4.7e+04     | 0.021       | 1.5e+03    | 2.3e-08 | 27      |
 | JSON::MaybeXS::decode_json         | json:array_int_1000 | 4.75e+04    | 0.021       | 1.53e+03   | 6.5e-09 | 21      |
 | JSON::Parse::parse_json            | json:array_int_1000 | 4.7e+04     | 0.021       | 1.5e+03    | 2.7e-08 | 20      |
 | JSON::Decode::Regexp::from_json    | json:array_int_10   | 5.59e+04    | 0.0179      | 1.79e+03   | 5.6e-09 | 28      |
 | JSON::Tiny::decode_json            | json:array_int_10   | 6.2e+04     | 0.016       | 2e+03      | 5.3e-08 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:hash_int_10    | 6.9e+04     | 0.014       | 2.2e+03    | 3.3e-08 | 20      |
 | JSON::Parse::parse_json            | json:array_str1k_10 | 7.26e+04    | 0.0138      | 2.33e+03   | 6.5e-09 | 21      |
 | JSON::Decode::Regexp::from_json    | json:str1k          | 8e+04       | 0.013       | 2.6e+03    | 1.3e-08 | 21      |
 | JSON::XS::decode_json              | json:array_str1k_10 | 101924      | 0.00981119  | 3270.73    | 0       | 25      |
 | JSON::MaybeXS::decode_json         | json:array_str1k_10 | 101949      | 0.00980884  | 3271.51    | 0       | 22      |
 | Cpanel::JSON::XS::decode_json      | json:array_str1k_10 | 1.02e+05    | 0.00979     | 3.28e+03   | 3.3e-09 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:array_int_10   | 1.09e+05    | 0.00915     | 3.51e+03   | 3.2e-09 | 22      |
 | JSON::Parse::parse_json            | json:hash_int_100   | 1.2e+05     | 0.0081      | 4e+03      | 9.3e-09 | 23      |
 | JSON::XS::decode_json              | json:hash_int_100   | 1.28e+05    | 0.00784     | 4.09e+03   | 3.3e-09 | 20      |
 | JSON::MaybeXS::decode_json         | json:hash_int_100   | 1.28e+05    | 0.00783     | 4.1e+03    | 3e-09   | 25      |
 | Cpanel::JSON::XS::decode_json      | json:hash_int_100   | 1.3e+05     | 0.0078      | 4.1e+03    | 8.9e-09 | 25      |
 | JSON::PP::decode_json              | json:num            | 1.53e+05    | 0.00653     | 4.92e+03   | 3e-09   | 24      |
 | JSON::Tiny::decode_json            | json:num            | 1.93e+05    | 0.00519     | 6.18e+03   | 5e-09   | 20      |
 | PERLANCAR::JSON::Match::match_json | json:str1k          | 1.94e+05    | 0.00516     | 6.21e+03   | 1.5e-09 | 25      |
 | JSON::Tiny::decode_json            | json:null           | 2.46e+05    | 0.00407     | 7.88e+03   | 1.7e-09 | 20      |
 | JSON::PP::decode_json              | json:null           | 290207      | 0.00344582  | 9312.66    | 0       | 21      |
 | JSON::MaybeXS::decode_json         | json:array_int_100  | 348227      | 0.00287169  | 11174.5    | 0       | 20      |
 | JSON::XS::decode_json              | json:array_int_100  | 3.53e+05    | 0.00283     | 1.13e+04   | 8.1e-10 | 21      |
 | Cpanel::JSON::XS::decode_json      | json:array_int_100  | 3.5e+05     | 0.0028      | 1.1e+04    | 3e-09   | 25      |
 | JSON::Parse::parse_json            | json:array_int_100  | 3.73e+05    | 0.00268     | 1.2e+04    | 7.2e-10 | 27      |
 | JSON::Decode::Regexp::from_json    | json:num            | 3.81e+05    | 0.00262     | 1.22e+04   | 8.3e-10 | 20      |
 | JSON::Decode::Regexp::from_json    | json:null           | 5.53e+05    | 0.00181     | 1.77e+04   | 8.3e-10 | 20      |
 | PERLANCAR::JSON::Match::match_json | json:num            | 5.7e+05     | 0.0018      | 1.8e+04    | 2.5e-09 | 20      |
 | JSON::Parse::parse_json            | json:str1k          | 728776      | 0.00137216  | 23386.2    | 0       | 20      |
 | PERLANCAR::JSON::Match::match_json | json:null           | 8.2e+05     | 0.0012      | 2.6e+04    | 1.7e-09 | 20      |
 | JSON::XS::decode_json              | json:str1k          | 987576      | 0.00101258  | 31691      | 0       | 20      |
 | Cpanel::JSON::XS::decode_json      | json:str1k          | 1.005e+06   | 0.0009953   | 3.224e+04  | 1.1e-11 | 20      |
 | JSON::MaybeXS::decode_json         | json:str1k          | 1.01e+06    | 0.000993    | 3.23e+04   | 4e-10   | 22      |
 | JSON::MaybeXS::decode_json         | json:hash_int_10    | 1.1e+06     | 0.00088     | 3.7e+04    | 1.7e-09 | 20      |
 | Cpanel::JSON::XS::decode_json      | json:hash_int_10    | 1.15e+06    | 0.000873    | 3.67e+04   | 4.2e-10 | 20      |
 | JSON::XS::decode_json              | json:hash_int_10    | 1.17083e+06 | 0.000854094 | 37571.7    | 0       | 23      |
 | JSON::Parse::parse_json            | json:hash_int_10    | 1.22399e+06 | 0.000817003 | 39277.4    | 0       | 20      |
 | JSON::XS::decode_json              | json:num            | 1.64e+06    | 0.000608    | 5.28e+04   | 1.9e-10 | 24      |
 | JSON::MaybeXS::decode_json         | json:num            | 1.65128e+06 | 0.000605593 | 52989      | 0       | 20      |
 | Cpanel::JSON::XS::decode_json      | json:num            | 1.654e+06   | 0.0006046   | 5.308e+04  | 9.6e-12 | 20      |
 | JSON::Parse::parse_json            | json:num            | 1.77698e+06 | 0.000562752 | 57022.9    | 0       | 20      |
 | JSON::XS::decode_json              | json:array_int_10   | 2.00381e+06 | 0.000499049 | 64301.7    | 0       | 20      |
 | Cpanel::JSON::XS::decode_json      | json:array_int_10   | 2.02e+06    | 0.0004951   | 6.481e+04  | 4.7e-11 | 20      |
 | JSON::MaybeXS::decode_json         | json:array_int_10   | 2.02995e+06 | 0.000492623 | 65140.5    | 0       | 20      |
 | JSON::Parse::parse_json            | json:array_int_10   | 2.18573e+06 | 0.000457512 | 70139.6    | 0       | 20      |
 | JSON::XS::decode_json              | json:null           | 8.9992e+06  | 0.000111121 | 288782     | 0       | 23      |
 | JSON::MaybeXS::decode_json         | json:null           | 9.10213e+06 | 0.000109864 | 292085     | 0       | 20      |
 | Cpanel::JSON::XS::decode_json      | json:null           | 9.25e+06    | 0.000108    | 2.97e+05   | 1.1e-11 | 20      |
 | JSON::Parse::parse_json            | json:null           | 1.451e+07   | 6.89179e-05 | 465623     | 0       | 20      |
 +------------------------------------+---------------------+-------------+-------------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m Serializers --module-startup >>):

 +----------------------+-----------+------------------------+------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | vs_slowest | errors  | samples |
 +----------------------+-----------+------------------------+------------+---------+---------+
 | JSON::Decode::Marpa  | 81        | 78.1                   | 1          | 0.0002  | 20      |
 | YAML::XS             | 33        | 30.1                   | 2.5        | 5.1e-05 | 20      |
 | JSON::PP             | 17        | 14.1                   | 4.9        | 3.7e-05 | 20      |
 | JSON::Tiny           | 15        | 12.1                   | 5.4        | 3.1e-05 | 20      |
 | Pegex::JSON          | 15        | 12.1                   | 5.4        | 8e-05   | 21      |
 | JSON::MaybeXS        | 11        | 8.1                    | 7.2        | 3.3e-05 | 21      |
 | Storable             | 9.8       | 6.9                    | 8.3        | 3.2e-05 | 20      |
 | Sereal               | 9.5       | 6.6                    | 8.6        | 6.8e-05 | 20      |
 | YAML::Old            | 9.2       | 6.3                    | 8.8        | 1.8e-05 | 20      |
 | JSON::Parse          | 8         | 5.1                    | 1e+01      | 7.7e-05 | 20      |
 | JSON::XS             | 7.9       | 5                      | 10         | 3.9e-05 | 20      |
 | Cpanel::JSON::XS     | 7.5       | 4.6                    | 11         | 3e-05   | 21      |
 | YAML::Syck           | 7.4       | 4.5                    | 11         | 2e-05   | 21      |
 | JSON::Create         | 6         | 3.1                    | 1e+01      | 8.9e-05 | 20      |
 | JSON::Decode::Regexp | 5.4       | 2.5                    | 15         | 4e-05   | 20      |
 | Data::MessagePack    | 5.4       | 2.5                    | 15         | 2.3e-05 | 20      |
 | perl -e1 (baseline)  | 2.9       | 0                      | 28         | 4.2e-06 | 20      |
 +----------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
